/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 * 
 * @format
 */
'use strict';

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

var _objectSpread2 = _interopRequireDefault(require("@babel/runtime/helpers/objectSpread"));

var warning = require("fbjs/lib/warning");

var validateMutation = function validateMutation() {};

if (process.env.NODE_ENV !== "production") {
  validateMutation = function validateMutation(optimisticResponse, mutation, variables) {
    var operationName = mutation.operation.name;
    var context = {
      operationName: operationName,
      path: 'ROOT',
      visitedPaths: new Set(),
      variables: variables || {}
    };
    validateSelections(optimisticResponse, mutation.operation.selections, context);
    validateOptimisticResponse(optimisticResponse, context);
  };

  var validateSelections = function validateSelections(optimisticResponse, selections, context) {
    selections.forEach(function (selection) {
      return validateSelection(optimisticResponse, selection, context);
    });
  };

  var validateSelection = function validateSelection(optimisticResponse, selection, context) {
    switch (selection.kind) {
      case 'Condition':
        if (selection.passingValue === context.variables[selection.condition]) {
          validateSelections(optimisticResponse, selection.selections, context);
        }

        return;

      case 'ScalarField':
      case 'LinkedField':
        return validateField(optimisticResponse, selection, context);

      case 'InlineFragment':
        var type = selection.type;
        selection.selections.forEach(function (subselection) {
          if (optimisticResponse.__typename !== type) {
            return;
          }

          validateSelection(optimisticResponse, subselection, context);
        });
        return;

      case 'ModuleImport':
      case 'LinkedHandle':
      case 'ScalarHandle':
      case 'Defer':
      case 'Stream':
        {
          // TODO(T35864292) - Add missing validations for these types
          return;
        }

      default:
        selection;
        return;
    }
  };

  var validateField = function validateField(optimisticResponse, field, context) {
    var fieldName = field.alias || field.name;
    var path = "".concat(context.path, ".").concat(fieldName);
    context.visitedPaths.add(path);

    switch (field.kind) {
      case 'ScalarField':
        if (optimisticResponse[fieldName] === undefined) {
          process.env.NODE_ENV !== "production" ? warning(false, 'validateMutation: Expected `optimisticResponse` to match structure of server response for mutation `%s`, field %s is undefined', context.operationName, path) : void 0;
        }

        return;

      case 'LinkedField':
        var selections = field.selections;

        if (optimisticResponse[fieldName] === null) {
          return;
        }

        if (field.plural) {
          if (Array.isArray(optimisticResponse[fieldName])) {
            optimisticResponse[fieldName].forEach(function (r) {
              return validateSelections(r, selections, (0, _objectSpread2["default"])({}, context, {
                path: path
              }));
            });
            return;
          } else {
            process.env.NODE_ENV !== "production" ? warning(false, 'validateMutation: Expected `optimisticResponse` to match structure of server response for mutation `%s`, field %s is not an array', context.operationName, path) : void 0;
            return;
          }
        } else {
          if (optimisticResponse[fieldName] instanceof Object) {
            validateSelections(optimisticResponse[fieldName], selections, (0, _objectSpread2["default"])({}, context, {
              path: path
            }));
            return;
          } else {
            process.env.NODE_ENV !== "production" ? warning(false, 'validateMutation: Expected `optimisticResponse` to match structure of server response for mutation `%s`, field %s is not an object', context.operationName, path) : void 0;
            return;
          }
        }

    }
  };

  var validateOptimisticResponse = function validateOptimisticResponse(optimisticResponse, context) {
    if (Array.isArray(optimisticResponse)) {
      optimisticResponse.forEach(function (r) {
        return validateOptimisticResponse(r, context);
      });
      return;
    }

    Object.keys(optimisticResponse).forEach(function (key) {
      var value = optimisticResponse[key];
      var path = "".concat(context.path, ".").concat(key);

      if (!context.visitedPaths.has(path)) {
        process.env.NODE_ENV !== "production" ? warning(false, 'validateMutation: `optimisticResponse` for mutation `%s`, contains an unused field %s', context.operationName, path) : void 0;
        return;
      }

      if (value instanceof Object) {
        validateOptimisticResponse(value, (0, _objectSpread2["default"])({}, context, {
          path: path
        }));
      }
    });
  };
}

module.exports = validateMutation;