import { left as eitherLeft, right as eitherRight, toError, tryCatch2v as eitherTryCatch2v } from './Either';
import * as eitherT from './EitherT';
import { constant, constIdentity } from './function';
import { IO, io } from './IO';
export const URI = 'IOEither';
const T = eitherT.getEitherT2v(io);
const foldT = eitherT.fold(io);
/**
 * @since 1.6.0
 */
export class IOEither {
    constructor(value) {
        this.value = value;
    }
    /**
     * Runs the inner io
     */
    run() {
        return this.value.run();
    }
    map(f) {
        return new IOEither(T.map(this.value, f));
    }
    ap(fab) {
        return new IOEither(T.ap(fab.value, this.value));
    }
    /**
     * Flipped version of `ap`
     */
    ap_(fb) {
        return fb.ap(this);
    }
    /**
     * Combine two effectful actions, keeping only the result of the first
     */
    applyFirst(fb) {
        return fb.ap(this.map(constant));
    }
    /**
     * Combine two effectful actions, keeping only the result of the second
     */
    applySecond(fb) {
        return fb.ap(this.map(constIdentity));
    }
    chain(f) {
        return new IOEither(T.chain(this.value, a => f(a).value));
    }
    fold(left, right) {
        return foldT(left, right, this.value);
    }
    mapLeft(f) {
        return new IOEither(this.value.map(e => e.mapLeft(f)));
    }
    orElse(f) {
        return new IOEither(this.value.chain(e => e.fold(l => f(l).value, a => T.of(a))));
    }
    alt(fy) {
        return this.orElse(() => fy);
    }
    bimap(f, g) {
        return new IOEither(this.value.map(e => e.bimap(f, g)));
    }
}
const map = (fa, f) => {
    return fa.map(f);
};
const of = (a) => {
    return new IOEither(T.of(a));
};
const ap = (fab, fa) => {
    return fa.ap(fab);
};
const chain = (fa, f) => {
    return fa.chain(f);
};
const alt = (fx, fy) => {
    return fx.alt(fy);
};
const bimap = (fa, f, g) => {
    return fa.bimap(f, g);
};
/**
 * @since 1.6.0
 */
export const right = (fa) => {
    return new IOEither(fa.map(eitherRight));
};
/**
 * @since 1.6.0
 */
export const left = (fa) => {
    return new IOEither(fa.map(eitherLeft));
};
/**
 * @since 1.6.0
 */
export const fromEither = (fa) => {
    return new IOEither(io.of(fa));
};
/**
 * @since 1.6.0
 */
export const fromLeft = (l) => {
    return fromEither(eitherLeft(l));
};
/**
 * Use `tryCatch2v` instead
 *
 * @since 1.6.0
 * @deprecated
 */
export const tryCatch = (f, onerror = toError) => {
    return tryCatch2v(f, onerror);
};
/**
 * @since 1.11.0
 */
export const tryCatch2v = (f, onerror) => {
    return new IOEither(new IO(() => eitherTryCatch2v(f, onerror)));
};
const throwError = fromLeft;
/**
 * @since 1.6.0
 */
export const ioEither = {
    URI,
    bimap,
    map,
    of,
    ap,
    chain,
    alt,
    throwError,
    fromEither,
    fromOption: (o, e) => (o.isNone() ? throwError(e) : of(o.value))
};
