<div align="center">
  <a href="https://github.com/webpack/webpack">
    <img width="200" height="200" src="https://webpack.js.org/assets/icon-square-big.svg">
  </a>
</div>

[![npm][npm]][npm-url]
[![node][node]][node-url]
[![deps][deps]][deps-url]
[![tests][tests]][tests-url]
[![cover][cover]][cover-url]
[![chat][chat]][chat-url]
[![size][size]][size-url]

# terser-webpack-plugin

This plugin uses [terser](https://github.com/fabiosantoscode/terser) to minify your JavaScript.

## Requirements

This module requires a minimum of Node v6.9.0 and Webpack v4.0.0.

## Getting Started

To begin, you'll need to install `terser-webpack-plugin`:

```console
$ npm install terser-webpack-plugin --save-dev
```

Then add the plugin to your `webpack` config. For example:

**webpack.config.js**

```js
const TerserPlugin = require('terser-webpack-plugin');

module.exports = {
  //...
  optimization: {
    minimizer: [new TerserPlugin()]
  }
};
```

And run `webpack` via your preferred method.

## Options

### `test`

Type: `String|RegExp|Array<String|RegExp>`
Default: `/\.js(\?.*)?$/i`

Test to match files against.

```js
// in your webpack.config.js
new TerserPlugin({
  test: /\.js(\?.*)?$/i
})
```

### `include`

Type: `String|RegExp|Array<String|RegExp>`
Default: `undefined`

Files to include.

```js
// in your webpack.config.js
new TerserPlugin({
  include: /\/includes/
})
```

### `exclude`

Type: `String|RegExp|Array<String|RegExp>`
Default: `undefined`

Files to exclude.

```js
// in your webpack.config.js
new TerserPlugin({
  exclude: /\/excludes/
})
```

### `cache`

Type: `Boolean|String`
Default: `false`

Enable file caching. 
Default path to cache directory: `node_modules/.cache/terser-webpack-plugin`.

> ℹ️ If you use your own `minify` function please read the `minify` section for cache invalidation correctly.

#### `Boolean`

Enable/disable file caching.

```js
// in your webpack.config.js
new TerserPlugin({
  cache: true
})
```

#### `String`

Enable file caching and set path to cache directory.

**webpack.config.js**

```js
// in your webpack.config.js
new TerserPlugin({
  cache: 'path/to/cache'
})
```

### `cacheKeys`

Type: `Function<(defaultCacheKeys, file) -> Object>`
Default: `defaultCacheKeys => defaultCacheKeys`

Allows you to override default cache keys.

Default cache keys:

```js
({
  terser: require('terser/package.json').version, // terser version
  'terser-webpack-plugin': require('../package.json').version, // plugin version
  'terser-webpack-plugin-options': this.options, // plugin options
  path: compiler.outputPath ? `${compiler.outputPath}/${file}` : file, // asset path
  hash: crypto.createHash('md4').update(input).digest('hex'), // source file hash
});
```

```js
// in your webpack.config.js
new TerserPlugin({
  cache: true,
  cacheKeys: (defaultCacheKeys, file) => {
    defaultCacheKeys.myCacheKey = 'myCacheKeyValue';

    return defaultCacheKeys;
  },
})
```

### `parallel`

Type: `Boolean|Number`
Default: `false`

Use multi-process parallel running to improve the build speed. 
Default number of concurrent runs: `os.cpus().length - 1`.

> ℹ️ Parallelization can speedup your build significantly and is therefore **highly recommended**.

#### `Boolean`

Enable/disable multi-process parallel running.

```js
// in your webpack.config.js
new TerserPlugin({
  parallel: true
})
```

#### `Number`

Enable multi-process parallel running and set number of concurrent runs.

```js
// in your webpack.config.js
new TerserPlugin({
  parallel: 4
})
```

### `sourceMap`

Type: `Boolean`
Default: `false`

Use source maps to map error message locations to modules (this slows down the compilation).
If you use your own `minify` function please read the `minify` section for handling source maps correctly.

> ⚠️ **`cheap-source-map` options don't work with this plugin**.

```js
// in your webpack.config.js
new TerserPlugin({
  sourceMap: true
})
```

### `minify`

Type: `Function`
Default: `undefined`

Allows you to override default minify function. 
By default plugin uses [terser](https://github.com/fabiosantoscode/terser) package.
Useful for using and testing unpublished versions or forks.

> ⚠️ **Always use `require` inside `minify` function when `parallel` option enabled**.

```js
// in your webpack.config.js
new TerserPlugin({
  minify(file, sourceMap) {
    const extractedComments = [];

    // Custom logic for extract comments

    const { error, map, code, warnings } = require('uglify-module') // Or require('./path/to/uglify-module')
      .minify(file, {
        /* Your options for minification */
      });

    return { error, map, code, warnings, extractedComments };
  }
})
```

### `terserOptions`

Type: `Object`
Default: [default](https://github.com/fabiosantoscode/terser#minify-options)

Terser minify options.

```js
// in your webpack.config.js
new TerserPlugin({
  terserOptions: {
    ecma: undefined,
    warnings: false,
    parse: {},
    compress: {},
    mangle: true, // Note `mangle.properties` is `false` by default.
    module: false,
    output: null,
    toplevel: false,
    nameCache: null,
    ie8: false,
    keep_classnames: undefined,
    keep_fnames: false,
    safari10: false
  }
})
```

### `extractComments`

Type: `Boolean|String|RegExp|Function<(node, comment) -> Boolean|Object>|Object`
Default: `false`

Whether comments shall be extracted to a separate file, (see [details](https://github.com/webpack/webpack/commit/71933e979e51c533b432658d5e37917f9e71595a)).
By default extract only comments using `/^\**!|@preserve|@license|@cc_on/i` regexp condition and remove remaining comments.
If the original file is named `foo.js`, then the comments will be stored to `foo.js.LICENSE`.
The `terserOptions.output.comments` option specifies whether the comment will be preserved, i.e. it is possible to preserve some comments (e.g. annotations) while extracting others or even preserving comments that have been extracted.

#### `Boolean`

Enable/disable extracting comments.

```js
// in your webpack.config.js
new TerserPlugin({
  extractComments: true
})
```

#### `String`

Extract `all` or `some` (use `/^\**!|@preserve|@license|@cc_on/i` RegExp) comments.

```js
// in your webpack.config.js
new TerserPlugin({
  extractComments: 'all'
})
```

#### `RegExp`

All comments that match the given expression will be extracted to the separate file. 

```js
// in your webpack.config.js
new TerserPlugin({
  extractComments: /@extract/i
})
```

#### `Function<(node, comment) -> Boolean>`

All comments that match the given expression will be extracted to the separate file. 

```js
// in your webpack.config.js
new TerserPlugin({
  extractComments: function (astNode, comment) {
    if (/@extract/i.test(comment.value)) {
      return true;
    }
    
    return false;
  }
})
```

#### `Object`

Allow to customize condition for extract comments, specify extracted file name and banner.

```js
// in your webpack.config.js
new TerserPlugin({
  extractComments: {
    condition: /^\**!|@preserve|@license|@cc_on/i,
    filename(file) {
      return `${file}.LICENSE`;
    },
    banner(licenseFile) {
      return `License information can be found in ${licenseFile}`;
    }
  }
})
```

##### `condition`

Type: `Boolean|String|RegExp|Function<(node, comment) -> Boolean|Object>`

Condition what comments you need extract.

```js
// in your webpack.config.js
new TerserPlugin({
  extractComments: {
    condition: 'some',
    filename(file) {
     return `${file}.LICENSE`;
    },
    banner(licenseFile) {
     return `License information can be found in ${licenseFile}`;
    }
  }
})
```

##### `filename`

Type: `String|Function<(string) -> String>`
Default: `${file}.LICENSE`

The file where the extracted comments will be stored.
Default is to append the suffix `.LICENSE` to the original filename.

```js
// in your webpack.config.js
new TerserPlugin({
  extractComments: {
    condition: /^\**!|@preserve|@license|@cc_on/i,
    filename: 'extracted-comments.js',
    banner(licenseFile) {
     return `License information can be found in ${licenseFile}`;
    }
  }
})
```

##### `banner`

Type: `Boolean|String|Function<(string) -> String>`
Default: `/*! For license information please see ${commentsFile} */`

The banner text that points to the extracted file and will be added on top of the original file. 
Can be `false` (no banner), a `String`, or a `Function<(string) -> String>` that will be called with the filename where extracted comments have been stored. 
Will be wrapped into comment.

```js
// in your webpack.config.js
new TerserPlugin({
  extractComments: {
    condition: true,
    filename(file) {
     return `${file}.LICENSE`;
    },
    banner(commentsFile) {
     return `My custom banner about license information ${commentsFile}`;
    }
  }
})
```

### `warningsFilter`

Type: `Function<(warning, source) -> Boolean>`
Default: `() => true`

Allow to filter [terser](https://github.com/fabiosantoscode/terser) warnings.
Return `true` to keep the warning, `false` otherwise.

```js
// in your webpack.config.js
new TerserPlugin({
  warningsFilter: (warning, source) => {
    if (/Dropping unreachable code/i.test(warning)) {
      return true;
    }

    if (/filename\.js/i.test(source)) {
      return true;
    }

    return false;
  },
})
```

## Examples

### Cache And Parallel

Enable cache and multi-process parallel running.

```js
// in your webpack.config.js
const TerserPlugin = require('terser-webpack-plugin');

module.exports = {
  //...
  optimization: {
    minimizer: [
      new TerserPlugin({
        cache: true,
        parallel: true
      })
    ]
  }
};
```

### Preserve Comments

Extract all legal comments (i.e. `/^\**!|@preserve|@license|@cc_on/i`) and preserve `/@license/i` comments.

```js
// in your webpack.config.js
const TerserPlugin = require('terser-webpack-plugin');

module.exports = {
  //...
  optimization: {
    minimizer: [
      new TerserPlugin({
        terserOptions: {
          output: {
            comments: /@license/i
          }
        },
        extractComments: true
      })
    ]
  }
};
```

### Custom Minify Function

Override default minify function - use `uglify-js` for minification.

```js
// in your webpack.config.js
const TerserPlugin = require('terser-webpack-plugin');

module.exports = {
  //...
  optimization: {
    minimizer: [
      new TerserPlugin({
        // Uncomment lines below for cache invalidation correctly
        // cache: true,
        // cacheKeys(defaultCacheKeys) {
        //   delete defaultCacheKeys.terser;
        //
        //   return Object.assign(
        //     {},
        //     defaultCacheKeys,
        //     { 'uglify-js': require('uglify-js/package.json').version },
        //   );
        // },
        minify(file, sourceMap) {
          // https://github.com/mishoo/UglifyJS2#minify-options
          const uglifyJsOptions = {
            /* your `uglify-js` package options */
          };
      
          if (sourceMap) {
            uglifyJsOptions.sourceMap = {
              content: sourceMap,
            };
          }
      
          return require('uglify-js').minify(file, uglifyJsOptions);
        },
      })
    ]
  }
};
```

## Contributing

Please take a moment to read our contributing guidelines if you haven't yet done so.

[CONTRIBUTING](./.github/CONTRIBUTING.md)

## License

[MIT](./LICENSE)

[npm]: https://img.shields.io/npm/v/terser-webpack-plugin.svg
[npm-url]: https://npmjs.com/package/terser-webpack-plugin
[node]: https://img.shields.io/node/v/terser-webpack-plugin.svg
[node-url]: https://nodejs.org
[deps]: https://david-dm.org/webpack-contrib/terser-webpack-plugin.svg
[deps-url]: https://david-dm.org/webpack-contrib/terser-webpack-plugin
[tests]: https://img.shields.io/circleci/project/github/webpack-contrib/terser-webpack-plugin.svg
[tests-url]: https://circleci.com/gh/webpack-contrib/terser-webpack-plugin
[cover]: https://codecov.io/gh/webpack-contrib/terser-webpack-plugin/branch/master/graph/badge.svg
[cover-url]: https://codecov.io/gh/webpack-contrib/terser-webpack-plugin
[chat]: https://img.shields.io/badge/gitter-webpack%2Fwebpack-brightgreen.svg
[chat-url]: https://gitter.im/webpack/webpack
[size]: https://packagephobia.now.sh/badge?p=terser-webpack-plugin
[size-url]: https://packagephobia.now.sh/result?p=terser-webpack-plugin
