import { phantom, toString } from './function';
import { contramap as contramapOrd, getSemigroup as getOrdSemigroup } from './Ord';
import { fromEquals } from './Setoid';
export const URI = 'Tuple';
/**
 * @since 1.0.0
 */
export class Tuple {
    constructor(fst, snd) {
        this.fst = fst;
        this.snd = snd;
    }
    compose(ab) {
        return new Tuple(this.fst, ab.snd);
    }
    map(f) {
        return new Tuple(this.fst, f(this.snd));
    }
    bimap(f, g) {
        return new Tuple(f(this.fst), g(this.snd));
    }
    extract() {
        return this.snd;
    }
    extend(f) {
        return new Tuple(this.fst, f(this));
    }
    reduce(b, f) {
        return f(b, this.snd);
    }
    /** Exchange the first and second components of a tuple */
    swap() {
        return new Tuple(this.snd, this.fst);
    }
    inspect() {
        return this.toString();
    }
    toString() {
        return `new Tuple(${toString(this.fst)}, ${toString(this.snd)})`;
    }
    toTuple() {
        return [this.fst, this.snd];
    }
}
/**
 * @since 1.17.0
 */
export const getShow = (SL, SA) => {
    return {
        show: t => `new Tuple(${SL.show(t.fst)}, ${SA.show(t.snd)})`
    };
};
const fst = (fa) => {
    return fa.fst;
};
const snd = (fa) => {
    return fa.snd;
};
const compose = (bc, fa) => {
    return fa.compose(bc);
};
const map = (fa, f) => {
    return fa.map(f);
};
const bimap = (fla, f, g) => {
    return fla.bimap(f, g);
};
const extract = snd;
const extend = (fa, f) => {
    return fa.extend(f);
};
const reduce = (fa, b, f) => {
    return fa.reduce(b, f);
};
const foldMap = (M) => (fa, f) => {
    return f(fa.snd);
};
const foldr = (fa, b, f) => {
    return f(fa.snd, b);
};
/**
 * @since 1.0.0
 */
export const getSetoid = (SA, SB) => {
    return fromEquals((x, y) => SA.equals(x.fst, y.fst) && SB.equals(x.snd, y.snd));
};
/**
 * To obtain the result, the `fst`s are `compare`d, and if they are `EQ`ual, the
 * `snd`s are `compare`d.
 *
 * @since 1.0.0
 */
export const getOrd = (OL, OA) => {
    return getOrdSemigroup().concat(contramapOrd(fst, OL), contramapOrd(snd, OA));
};
/**
 * @since 1.0.0
 */
export const getSemigroup = (SL, SA) => {
    return {
        concat: (x, y) => new Tuple(SL.concat(x.fst, y.fst), SA.concat(x.snd, y.snd))
    };
};
/**
 * @since 1.0.0
 */
export const getMonoid = (ML, MA) => {
    return Object.assign({}, getSemigroup(ML, MA), { empty: new Tuple(ML.empty, MA.empty) });
};
const ap = (S) => (fab, fa) => {
    return new Tuple(S.concat(fab.fst, fa.fst), fab.snd(fa.snd));
};
/**
 * @since 1.0.0
 */
export const getApply = (S) => {
    return {
        URI,
        _L: phantom,
        map,
        ap: ap(S)
    };
};
const of = (M) => (a) => {
    return new Tuple(M.empty, a);
};
/**
 * @since 1.0.0
 */
export const getApplicative = (M) => {
    return Object.assign({}, getApply(M), { of: of(M) });
};
const chain = (S) => (fa, f) => {
    const { fst, snd } = f(fa.snd);
    return new Tuple(S.concat(fa.fst, fst), snd);
};
/**
 * @since 1.0.0
 */
export const getChain = (S) => {
    return Object.assign({}, getApply(S), { chain: chain(S) });
};
/**
 * @since 1.0.0
 */
export const getMonad = (M) => {
    return Object.assign({}, getChain(M), { of: of(M) });
};
const chainRec = (M) => (a, f) => {
    let result = f(a);
    let acc = M.empty;
    while (result.snd.isLeft()) {
        acc = M.concat(acc, result.fst);
        result = f(result.snd.value);
    }
    return new Tuple(M.concat(acc, result.fst), result.snd.value);
};
/**
 * @since 1.0.0
 */
export const getChainRec = (M) => {
    return Object.assign({}, getChain(M), { chainRec: chainRec(M) });
};
const traverse = (F) => (ta, f) => {
    return F.map(f(ta.snd), b => new Tuple(ta.fst, b));
};
const sequence = (F) => (ta) => {
    return F.map(ta.snd, b => new Tuple(ta.fst, b));
};
/**
 * @since 1.0.0
 */
export const tuple = {
    URI,
    compose,
    map,
    bimap,
    extract,
    extend,
    reduce,
    foldMap,
    foldr,
    traverse,
    sequence
};
