/**
 * @file See [Getting started with fp-ts: Semigroup](https://dev.to/gcanti/getting-started-with-fp-ts-semigroup-2mf7)
 */
import { max, min } from './Ord';
import { concat, identity } from './function';
/**
 * @since 1.0.0
 */
export const fold = (S) => (a) => (as) => {
    return as.reduce(S.concat, a);
};
/**
 * @since 1.0.0
 */
export const getFirstSemigroup = () => {
    return { concat: identity };
};
/**
 * @since 1.0.0
 */
export const getLastSemigroup = () => {
    return { concat: (_, y) => y };
};
/**
 * Given a tuple of semigroups returns a semigroup for the tuple
 *
 * @example
 * import { getTupleSemigroup, semigroupString, semigroupSum, semigroupAll } from 'fp-ts/lib/Semigroup'
 *
 * const S1 = getTupleSemigroup(semigroupString, semigroupSum)
 * assert.deepStrictEqual(S1.concat(['a', 1], ['b', 2]), ['ab', 3])
 *
 * const S2 = getTupleSemigroup(semigroupString, semigroupSum, semigroupAll)
 * assert.deepStrictEqual(S2.concat(['a', 1, true], ['b', 2, false]), ['ab', 3, false])
 *
 * @since 1.14.0
 */
export const getTupleSemigroup = (...semigroups) => {
    return {
        concat: (x, y) => semigroups.map((s, i) => s.concat(x[i], y[i]))
    };
};
/**
 * Use `getTupleSemigroup` instead
 * @since 1.0.0
 * @deprecated
 */
export const getProductSemigroup = (SA, SB) => {
    return getTupleSemigroup(SA, SB);
};
/**
 * @since 1.0.0
 */
export const getDualSemigroup = (S) => {
    return {
        concat: (x, y) => S.concat(y, x)
    };
};
/**
 * @since 1.0.0
 */
export const getFunctionSemigroup = (S) => () => {
    return {
        concat: (f, g) => a => S.concat(f(a), g(a))
    };
};
/**
 * @since 1.14.0
 */
export const getStructSemigroup = (semigroups) => {
    return {
        concat: (x, y) => {
            const r = {};
            for (const key of Object.keys(semigroups)) {
                r[key] = semigroups[key].concat(x[key], y[key]);
            }
            return r;
        }
    };
};
/**
 * Use `getStructSemigroup` instead
 * @since 1.0.0
 * @deprecated
 */
export const getRecordSemigroup = (semigroups) => {
    return getStructSemigroup(semigroups);
};
/**
 * @since 1.0.0
 */
export const getMeetSemigroup = (O) => {
    return {
        concat: min(O)
    };
};
/**
 * @since 1.0.0
 */
export const getJoinSemigroup = (O) => {
    return {
        concat: max(O)
    };
};
/**
 * Boolean semigroup under conjunction
 * @since 1.0.0
 */
export const semigroupAll = {
    concat: (x, y) => x && y
};
/**
 * Boolean semigroup under disjunction
 * @since 1.0.0
 */
export const semigroupAny = {
    concat: (x, y) => x || y
};
/**
 * Use `Monoid`'s `getArrayMonoid` instead
 * @since 1.0.0
 * @deprecated
 */
export const getArraySemigroup = () => {
    return { concat };
};
export function getDictionarySemigroup(S) {
    return {
        concat: (x, y) => {
            const r = Object.assign({}, x);
            const keys = Object.keys(y);
            const len = keys.length;
            for (let i = 0; i < len; i++) {
                const k = keys[i];
                r[k] = x.hasOwnProperty(k) ? S.concat(x[k], y[k]) : y[k];
            }
            return r;
        }
    };
}
// tslint:disable-next-line: deprecation
const semigroupAnyDictionary = getDictionarySemigroup(getLastSemigroup());
/**
 * Returns a `Semigroup` instance for objects preserving their type
 *
 * @example
 * import { getObjectSemigroup } from 'fp-ts/lib/Semigroup'
 *
 * interface Person {
 *   name: string
 *   age: number
 * }
 *
 * const S = getObjectSemigroup<Person>()
 * assert.deepStrictEqual(S.concat({ name: 'name', age: 23 }, { name: 'name', age: 24 }), { name: 'name', age: 24 })
 *
 * @since 1.4.0
 */
export const getObjectSemigroup = () => {
    return semigroupAnyDictionary;
};
/**
 * Number `Semigroup` under addition
 * @since 1.0.0
 */
export const semigroupSum = {
    concat: (x, y) => x + y
};
/**
 * Number `Semigroup` under multiplication
 * @since 1.0.0
 */
export const semigroupProduct = {
    concat: (x, y) => x * y
};
/**
 * @since 1.0.0
 */
export const semigroupString = {
    concat: (x, y) => x + y
};
/**
 * @since 1.0.0
 */
export const semigroupVoid = {
    concat: () => undefined
};
