"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const loki = require("lokijs");
class StatsDatabase {
    constructor(getISODate) {
        const db = new loki("stats-database");
        this.counters = db.addCollection("counters");
        this.customEvents = db.addCollection("customEvents");
        this.timings = db.addCollection("timing");
        this.getDate = () => getISODate();
    }
    async addCustomEvent(eventType, customEvent) {
        customEvent.date = this.getDate();
        customEvent.eventType = eventType;
        this.customEvents.insert(customEvent);
    }
    async incrementCounter(counterName) {
        const existing = await this.getUnformattedCounter(counterName);
        if (existing) {
            existing.count += 1;
            this.counters.update(existing);
        }
        else {
            this.counters.insert({ name: counterName, count: 1 });
        }
    }
    async addTiming(eventType, durationInMilliseconds, metadata = {}) {
        const timingData = { eventType, durationInMilliseconds, metadata, date: this.getDate() };
        this.timings.insert(timingData);
    }
    /** Clears all values that exist in the database.
     * returns nothing.
     */
    async clearData() {
        await this.counters.clear();
        await this.customEvents.clear();
        await this.timings.clear();
    }
    async getTimings() {
        const timings = await this.timings.find();
        timings.forEach((timing) => {
            delete timing.$loki;
            delete timing.meta;
        });
        return timings;
    }
    async getCustomEvents() {
        const events = await this.customEvents.find();
        events.forEach((event) => {
            // honey badger don't care about lokijis meta data.
            delete event.$loki;
            delete event.meta;
        });
        return events;
    }
    /** Get all counters.
     * This method strips the lokijs metadata, which external
     * callers shouldn't care about.
     * Returns something like { commits: 7, coAuthoredCommits: 8 }.
     */
    async getCounters() {
        const counters = {};
        this.counters.find().forEach((counter) => {
            counters[counter.name] = counter.count;
        });
        return counters;
    }
    /** Get a single counter.
     * Don't strip lokijs metadata, because if we want to update existing
     * items we need to pass that shizz back in.
     * Returns something like:
     * [ { name: 'coAuthoredCommits',count: 1, meta: { revision: 0, created: 1526592157642, version: 0 },'$loki': 1 } ]
     */
    async getUnformattedCounter(counterName) {
        const existing = await this.counters.find({ name: counterName });
        if (existing.length > 1) {
            // we should never get into this situation because if we are using the lokijs
            // api properly it should overwrite existing items with the same name.
            // but I've seen things (in prod) you people wouldn't believe.
            // Attack ships on fire off the shoulder of Orion.
            // Cosmic rays flipping bits and influencing the outcome of elections.
            // So throw an error just in case.
            throw new Error("multiple counters with the same name");
        }
        else if (existing.length < 1) {
            return null;
        }
        else {
            return existing[0];
        }
    }
}
exports.default = StatsDatabase;
//# sourceMappingURL=database.js.map