"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _path = _interopRequireDefault(require("path"));

var _eventKit = require("event-kit");

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _actionPipeline = require("../action-pipeline");

var _compositeGitStrategy = _interopRequireDefault(require("../composite-git-strategy"));

var _author = _interopRequireWildcard(require("./author"));

var _branch = _interopRequireDefault(require("./branch"));

var _repositoryStates = require("./repository-states");

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj["default"] = obj; return newObj; } }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const MERGE_MARKER_REGEX = /^(>|<){7} \S+$/m; // Internal option keys used to designate the desired initial state of a Repository.

const initialStateSym = Symbol('initialState');

class Repository {
  constructor(workingDirectoryPath, gitStrategy = null, options = {}) {
    this.workingDirectoryPath = workingDirectoryPath;
    this.git = gitStrategy || _compositeGitStrategy["default"].create(workingDirectoryPath);
    this.emitter = new _eventKit.Emitter();
    this.loadPromise = new Promise(resolve => {
      const sub = this.onDidChangeState(() => {
        if (!this.isLoading()) {
          resolve();
          sub.dispose();
        } else if (this.isDestroyed()) {
          sub.dispose();
        }
      });
    });
    this.pipelineManager = options.pipelineManager || (0, _actionPipeline.getNullActionPipelineManager)();
    this.transitionTo(options[initialStateSym] || _repositoryStates.Loading);
  }

  static absent(options) {
    return new Repository(null, null, _objectSpread({
      [initialStateSym]: _repositoryStates.Absent
    }, options));
  }

  static loadingGuess(options) {
    return new Repository(null, null, _objectSpread({
      [initialStateSym]: _repositoryStates.LoadingGuess
    }, options));
  }

  static absentGuess(options) {
    return new Repository(null, null, _objectSpread({
      [initialStateSym]: _repositoryStates.AbsentGuess
    }, options));
  } // State management //////////////////////////////////////////////////////////////////////////////////////////////////


  transition(currentState, StateConstructor, ...payload) {
    if (currentState !== this.state) {
      // Attempted transition from a non-active state, most likely from an asynchronous start() method.
      return Promise.resolve();
    }

    const nextState = new StateConstructor(this, ...payload);
    this.state = nextState;
    this.emitter.emit('did-change-state', {
      from: currentState,
      to: this.state
    });

    if (!this.isDestroyed()) {
      this.emitter.emit('did-update');
    }

    return this.state.start();
  }

  transitionTo(StateConstructor, ...payload) {
    return this.transition(this.state, StateConstructor, ...payload);
  }

  getLoadPromise() {
    return this.isAbsent() ? Promise.reject(new Error('An absent repository will never load')) : this.loadPromise;
  }
  /*
   * Use `callback` to request user input from all git strategies.
   */


  setPromptCallback(callback) {
    this.git.getImplementers().forEach(strategy => strategy.setPromptCallback(callback));
  } // Pipeline


  getPipeline(actionName) {
    const actionKey = this.pipelineManager.actionKeys[actionName];
    return this.pipelineManager.getPipeline(actionKey);
  }

  executePipelineAction(actionName, fn, ...args) {
    const pipeline = this.getPipeline(actionName);
    return pipeline.run(fn, this, ...args);
  } // Event subscription ////////////////////////////////////////////////////////////////////////////////////////////////


  onDidDestroy(callback) {
    return this.emitter.on('did-destroy', callback);
  }

  onDidChangeState(callback) {
    return this.emitter.on('did-change-state', callback);
  }

  onDidUpdate(callback) {
    return this.emitter.on('did-update', callback);
  }

  onPullError(callback) {
    return this.emitter.on('pull-error', callback);
  }

  didPullError() {
    return this.emitter.emit('pull-error');
  } // State-independent actions /////////////////////////////////////////////////////////////////////////////////////////
  // Actions that use direct filesystem access or otherwise don't need `this.git` to be available.


  async pathHasMergeMarkers(relativePath) {
    try {
      const contents = await _fsExtra["default"].readFile(_path["default"].join(this.getWorkingDirectoryPath(), relativePath), {
        encoding: 'utf8'
      });
      return MERGE_MARKER_REGEX.test(contents);
    } catch (e) {
      // EISDIR implies this is a submodule
      if (e.code === 'ENOENT' || e.code === 'EISDIR') {
        return false;
      } else {
        throw e;
      }
    }
  }

  async getMergeMessage() {
    try {
      const contents = await _fsExtra["default"].readFile(_path["default"].join(this.getGitDirectoryPath(), 'MERGE_MSG'), {
        encoding: 'utf8'
      });
      return contents.split(/\n/).filter(line => line.length > 0 && !line.startsWith('#')).join('\n');
    } catch (e) {
      return null;
    }
  } // State-independent accessors ///////////////////////////////////////////////////////////////////////////////////////


  getWorkingDirectoryPath() {
    return this.workingDirectoryPath;
  }

  setGitDirectoryPath(gitDirectoryPath) {
    this._gitDirectoryPath = gitDirectoryPath;
  }

  getGitDirectoryPath() {
    if (this._gitDirectoryPath) {
      return this._gitDirectoryPath;
    } else if (this.getWorkingDirectoryPath()) {
      return _path["default"].join(this.getWorkingDirectoryPath(), '.git');
    } else {
      // Absent/Loading/etc.
      return null;
    }
  }

  isInState(stateName) {
    return this.state.constructor.name === stateName;
  }

  toString() {
    return `Repository(state=${this.state.constructor.name}, workdir="${this.getWorkingDirectoryPath()}")`;
  } // Compound Getters //////////////////////////////////////////////////////////////////////////////////////////////////
  // Accessor methods for data derived from other, state-provided getters.


  async getCurrentBranch() {
    const branches = await this.getBranches();
    const head = branches.getHeadBranch();

    if (head.isPresent()) {
      return head;
    }

    const description = await this.getHeadDescription();
    return _branch["default"].createDetached(description);
  }

  async getUnstagedChanges() {
    const {
      unstagedFiles
    } = await this.getStatusBundle();
    return Object.keys(unstagedFiles).sort().map(filePath => {
      return {
        filePath,
        status: unstagedFiles[filePath]
      };
    });
  }

  async getStagedChanges() {
    const {
      stagedFiles
    } = await this.getStatusBundle();
    return Object.keys(stagedFiles).sort().map(filePath => {
      return {
        filePath,
        status: stagedFiles[filePath]
      };
    });
  }

  async getMergeConflicts() {
    const {
      mergeConflictFiles
    } = await this.getStatusBundle();
    return Object.keys(mergeConflictFiles).map(filePath => {
      return {
        filePath,
        status: mergeConflictFiles[filePath]
      };
    });
  }

  async isPartiallyStaged(fileName) {
    const {
      unstagedFiles,
      stagedFiles
    } = await this.getStatusBundle();
    const u = unstagedFiles[fileName];
    const s = stagedFiles[fileName];
    return u === 'modified' && s === 'modified' || u === 'modified' && s === 'added' || u === 'added' && s === 'deleted' || u === 'deleted' && s === 'modified';
  }

  async getRemoteForBranch(branchName) {
    const name = await this.getConfig(`branch.${branchName}.remote`);
    return (await this.getRemotes()).withName(name);
  }

  async saveDiscardHistory() {
    if (this.isDestroyed()) {
      return;
    }

    const historySha = await this.createDiscardHistoryBlob();

    if (this.isDestroyed()) {
      return;
    }

    await this.setConfig('atomGithub.historySha', historySha);
  }

  async getCommitter(options = {}) {
    const output = await this.getConfig(['--get-regexp', '^user.*'], options);
    const committer = {
      name: null,
      email: null
    }; // todo (tt, 4/2018): do we need null byte terminated output here for Windows?

    if (output) {
      output.trim().split('\n').forEach(line => {
        if (line.includes('user.email')) {
          committer.email = line.slice(11);
        } else if (line.includes('user.name')) {
          committer.name = line.slice(10);
        }
      });
    }

    return committer.name !== null && committer.email !== null ? new _author["default"](committer.email, committer.name) : _author.nullAuthor;
  } // todo (@annthurium, 3/2019): refactor GitHubTabController etc to use this method.


  async getCurrentGitHubRemote() {
    let currentRemote = null;
    const remotes = await this.getRemotes();
    const gitHubRemotes = remotes.filter(remote => remote.isGithubRepo());
    const selectedRemoteName = await this.getConfig('atomGithub.currentRemote');
    currentRemote = gitHubRemotes.withName(selectedRemoteName);

    if (!currentRemote.isPresent() && gitHubRemotes.size() === 1) {
      currentRemote = Array.from(gitHubRemotes)[0];
    } // todo: handle the case where multiple remotes are available and no chosen remote is set.


    return currentRemote;
  }

  async hasGitHubRemote(host, owner, name) {
    const remotes = await this.getRemotes();
    return remotes.matchingGitHubRepository(owner, name).length > 0;
  }

} // The methods named here will be delegated to the current State.
//
// Duplicated here rather than just using `expectedDelegates` directly so that this file is grep-friendly for answering
// the question of "what all can a Repository do exactly".


exports["default"] = Repository;
const delegates = ['isLoadingGuess', 'isAbsentGuess', 'isAbsent', 'isLoading', 'isEmpty', 'isPresent', 'isTooLarge', 'isDestroyed', 'isUndetermined', 'showGitTabInit', 'showGitTabInitInProgress', 'showGitTabLoading', 'showStatusBarTiles', 'hasDirectory', 'init', 'clone', 'destroy', 'refresh', 'observeFilesystemChange', 'updateCommitMessageAfterFileSystemChange', 'stageFiles', 'unstageFiles', 'stageFilesFromParentCommit', 'stageFileModeChange', 'stageFileSymlinkChange', 'applyPatchToIndex', 'applyPatchToWorkdir', 'commit', 'merge', 'abortMerge', 'checkoutSide', 'mergeFile', 'writeMergeConflictToIndex', 'checkout', 'checkoutPathsAtRevision', 'undoLastCommit', 'fetch', 'pull', 'push', 'setConfig', 'createBlob', 'expandBlobToFile', 'createDiscardHistoryBlob', 'updateDiscardHistory', 'storeBeforeAndAfterBlobs', 'restoreLastDiscardInTempFiles', 'popDiscardHistory', 'clearDiscardHistory', 'discardWorkDirChangesForPaths', 'getStatusBundle', 'getStatusesForChangedFiles', 'getFilePatchForPath', 'getDiffsForFilePath', 'getStagedChangesPatch', 'readFileFromIndex', 'getLastCommit', 'getCommit', 'getRecentCommits', 'isCommitPushed', 'getAuthors', 'getBranches', 'getHeadDescription', 'isMerging', 'isRebasing', 'getRemotes', 'addRemote', 'getAheadCount', 'getBehindCount', 'getConfig', 'unsetConfig', 'getBlobContents', 'hasDiscardHistory', 'getDiscardHistory', 'getLastHistorySnapshots', 'getOperationStates', 'setCommitMessage', 'getCommitMessage', 'fetchCommitMessageTemplate', 'getCache'];

for (let i = 0; i < delegates.length; i++) {
  const delegate = delegates[i];

  Repository.prototype[delegate] = function (...args) {
    return this.state[delegate](...args);
  };
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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