"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _eventKit = require("event-kit");

var _atom = require("atom");

var _path = _interopRequireDefault(require("path"));

var _eventLogger = _interopRequireDefault(require("./event-logger"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

class FileSystemChangeObserver {
  constructor(repository) {
    this.emitter = new _eventKit.Emitter();
    this.repository = repository;
    this.logger = new _eventLogger["default"]('fs watcher');
    this.started = false;
  }

  async start() {
    await this.watchRepository();
    this.started = true;
    return this;
  }

  async destroy() {
    this.started = false;
    this.emitter.dispose();
    await this.stopCurrentFileWatcher();
  }

  isStarted() {
    return this.started;
  }

  didChange(payload) {
    this.emitter.emit('did-change', payload);
  }

  didChangeWorkdirOrHead() {
    this.emitter.emit('did-change-workdir-or-head');
  }

  onDidChange(callback) {
    return this.emitter.on('did-change', callback);
  }

  onDidChangeWorkdirOrHead(callback) {
    return this.emitter.on('did-change-workdir-or-head', callback);
  }

  getRepository() {
    return this.repository;
  }

  async watchRepository() {
    const allPaths = event => {
      const ps = [event.path];

      if (event.oldPath) {
        ps.push(event.oldPath);
      }

      return ps;
    };

    const isNonGitFile = event => allPaths(event).some(eventPath => !eventPath.split(_path["default"].sep).includes('.git'));

    const isWatchedGitFile = event => allPaths(event).some(eventPath => {
      return ['config', 'index', 'HEAD', 'MERGE_HEAD'].includes(_path["default"].basename(eventPath)) || _path["default"].dirname(eventPath).includes(_path["default"].join('.git', 'refs'));
    });

    const handleEvents = events => {
      const filteredEvents = events.filter(e => isNonGitFile(e) || isWatchedGitFile(e));

      if (filteredEvents.length) {
        this.logger.showEvents(filteredEvents);
        this.didChange(filteredEvents);
        const workdirOrHeadEvent = filteredEvents.find(event => {
          return allPaths(event).every(eventPath => !['config', 'index'].includes(_path["default"].basename(eventPath)));
        });

        if (workdirOrHeadEvent) {
          this.logger.showWorkdirOrHeadEvents();
          this.didChangeWorkdirOrHead();
        }
      }
    };

    this.currentFileWatcher = await (0, _atom.watchPath)(this.repository.getWorkingDirectoryPath(), {}, handleEvents);
    this.logger.showStarted(this.repository.getWorkingDirectoryPath(), 'Atom watchPath');
  }

  stopCurrentFileWatcher() {
    if (this.currentFileWatcher) {
      this.currentFileWatcher.dispose();
      this.logger.showStopped();
    }

    return Promise.resolve();
  }

}

exports["default"] = FileSystemChangeObserver;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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