"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.nullCommit = exports["default"] = void 0;

var _patch = require("./patch");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const UNBORN = Symbol('unborn'); // Truncation elipsis styles

const WORD_ELIPSES = '...';
const NEWLINE_ELIPSES = '\n...';
const PARAGRAPH_ELIPSES = '\n\n...';

class Commit {
  static createUnborn() {
    return new Commit({
      unbornRef: UNBORN
    });
  }

  constructor({
    sha,
    authorEmail,
    authorName,
    coAuthors,
    authorDate,
    messageSubject,
    messageBody,
    unbornRef,
    patch
  }) {
    this.sha = sha;
    this.authorEmail = authorEmail;
    this.authorName = authorName;
    this.coAuthors = coAuthors || [];
    this.authorDate = authorDate;
    this.messageSubject = messageSubject;
    this.messageBody = messageBody;
    this.unbornRef = unbornRef === UNBORN;
    this.multiFileDiff = patch ? (0, _patch.buildMultiFilePatch)(patch) : (0, _patch.buildMultiFilePatch)([]);
  }

  getSha() {
    return this.sha;
  }

  getAuthorEmail() {
    return this.authorEmail;
  }

  getAuthorName() {
    return this.authorName;
  }

  getAuthorDate() {
    return this.authorDate;
  }

  getCoAuthors() {
    return this.coAuthors;
  }

  getMessageSubject() {
    return this.messageSubject;
  }

  getMessageBody() {
    return this.messageBody;
  }

  isBodyLong() {
    if (this.getMessageBody().length > this.constructor.LONG_MESSAGE_THRESHOLD) {
      return true;
    }

    if ((this.getMessageBody().match(/\r?\n/g) || []).length > this.constructor.NEWLINE_THRESHOLD) {
      return true;
    }

    return false;
  }

  getFullMessage() {
    return `${this.getMessageSubject()}\n\n${this.getMessageBody()}`.trim();
  }
  /*
   * Return the messageBody truncated to at most LONG_MESSAGE_THRESHOLD characters or NEWLINE_THRESHOLD newlines,
   * whichever comes first.
   *
   * If NEWLINE_THRESHOLD newlines are encountered before LONG_MESSAGE_THRESHOLD characters, the body will be truncated
   * at the last counted newline and elipses added.
   *
   * If a paragraph boundary is found before LONG_MESSAGE_THRESHOLD characters, the message will be truncated at the end
   * of the previous paragraph and an elipses added. If no paragraph boundary is found, but a word boundary is, the text
   * is truncated at the last word boundary and an elipsis added. If neither are found, the text is truncated hard at
   * LONG_MESSAGE_THRESHOLD - 3 characters and an elipsis is added.
   */


  abbreviatedBody() {
    if (!this.isBodyLong()) {
      return this.getMessageBody();
    }

    const {
      LONG_MESSAGE_THRESHOLD,
      NEWLINE_THRESHOLD
    } = this.constructor;
    let lastNewlineCutoff = null;
    let lastParagraphCutoff = null;
    let lastWordCutoff = null;
    const searchText = this.getMessageBody().substring(0, LONG_MESSAGE_THRESHOLD);
    const boundaryRx = /\s+/g;
    let result;
    let lineCount = 0;

    while ((result = boundaryRx.exec(searchText)) !== null) {
      const newlineCount = (result[0].match(/\r?\n/g) || []).length;
      lineCount += newlineCount;

      if (lineCount > NEWLINE_THRESHOLD) {
        lastNewlineCutoff = result.index;
        break;
      }

      if (newlineCount < 2 && result.index <= LONG_MESSAGE_THRESHOLD - WORD_ELIPSES.length) {
        lastWordCutoff = result.index;
      } else if (result.index < LONG_MESSAGE_THRESHOLD - PARAGRAPH_ELIPSES.length) {
        lastParagraphCutoff = result.index;
      }
    }

    let elipses = WORD_ELIPSES;
    let cutoffIndex = LONG_MESSAGE_THRESHOLD - WORD_ELIPSES.length;

    if (lastNewlineCutoff !== null) {
      elipses = NEWLINE_ELIPSES;
      cutoffIndex = lastNewlineCutoff;
    } else if (lastParagraphCutoff !== null) {
      elipses = PARAGRAPH_ELIPSES;
      cutoffIndex = lastParagraphCutoff;
    } else if (lastWordCutoff !== null) {
      cutoffIndex = lastWordCutoff;
    }

    return this.getMessageBody().substring(0, cutoffIndex) + elipses;
  }

  setMultiFileDiff(multiFileDiff) {
    this.multiFileDiff = multiFileDiff;
  }

  getMultiFileDiff() {
    return this.multiFileDiff;
  }

  isUnbornRef() {
    return this.unbornRef;
  }

  isPresent() {
    return true;
  }

  isEqual(other) {
    // Directly comparable properties
    const properties = ['sha', 'authorEmail', 'authorDate', 'messageSubject', 'messageBody', 'unbornRef', 'authorName'];

    for (const property of properties) {
      if (this[property] !== other[property]) {
        return false;
      }
    } // Co-author array


    if (this.coAuthors.length !== other.coAuthors.length) {
      return false;
    }

    for (let i = 0; i < this.coAuthors.length; i++) {
      const thisCoAuthor = this.coAuthors[i];
      const otherCoAuthor = other.coAuthors[i];

      if (thisCoAuthor.name !== otherCoAuthor.name || thisCoAuthor.email !== otherCoAuthor.email) {
        return false;
      }
    } // Multi-file patch


    if (!this.multiFileDiff.isEqual(other.multiFileDiff)) {
      return false;
    }

    return true;
  }

}

exports["default"] = Commit;

_defineProperty(Commit, "LONG_MESSAGE_THRESHOLD", 400);

_defineProperty(Commit, "NEWLINE_THRESHOLD", 5);

const nullCommit = {
  getSha() {
    return '';
  },

  getMessageSubject() {
    return '';
  },

  isUnbornRef() {
    return false;
  },

  isPresent() {
    return false;
  },

  isBodyLong() {
    return false;
  }

};
exports.nullCommit = nullCommit;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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