(function() {
  var BrowserWindow, CompositeDisposable, TabBarView, TabView, ipcRenderer;

  BrowserWindow = null;

  ipcRenderer = require('electron').ipcRenderer;

  CompositeDisposable = require('atom').CompositeDisposable;

  TabView = require('./tab-view');

  module.exports = TabBarView = (function() {
    function TabBarView(pane1, location1) {
      var addElementCommands, item, j, len, ref;
      this.pane = pane1;
      this.location = location1;
      this.element = document.createElement('ul');
      this.element.classList.add("list-inline");
      this.element.classList.add("tab-bar");
      this.element.classList.add("inset-panel");
      this.element.setAttribute('is', 'atom-tabs');
      this.element.setAttribute("tabindex", -1);
      this.element.setAttribute("location", this.location);
      this.tabs = [];
      this.tabsByElement = new WeakMap;
      this.subscriptions = new CompositeDisposable;
      this.paneElement = this.pane.getElement();
      this.subscriptions.add(atom.commands.add(this.paneElement, {
        'tabs:keep-pending-tab': (function(_this) {
          return function() {
            return _this.terminatePendingStates();
          };
        })(this),
        'tabs:close-tab': (function(_this) {
          return function() {
            return _this.closeTab(_this.getActiveTab());
          };
        })(this),
        'tabs:close-other-tabs': (function(_this) {
          return function() {
            return _this.closeOtherTabs(_this.getActiveTab());
          };
        })(this),
        'tabs:close-tabs-to-right': (function(_this) {
          return function() {
            return _this.closeTabsToRight(_this.getActiveTab());
          };
        })(this),
        'tabs:close-tabs-to-left': (function(_this) {
          return function() {
            return _this.closeTabsToLeft(_this.getActiveTab());
          };
        })(this),
        'tabs:close-saved-tabs': (function(_this) {
          return function() {
            return _this.closeSavedTabs();
          };
        })(this),
        'tabs:close-all-tabs': (function(_this) {
          return function(event) {
            event.stopPropagation();
            return _this.closeAllTabs();
          };
        })(this),
        'tabs:open-in-new-window': (function(_this) {
          return function() {
            return _this.openInNewWindow();
          };
        })(this)
      }));
      addElementCommands = (function(_this) {
        return function(commands) {
          var commandsWithPropagationStopped;
          commandsWithPropagationStopped = {};
          Object.keys(commands).forEach(function(name) {
            return commandsWithPropagationStopped[name] = function(event) {
              event.stopPropagation();
              return commands[name]();
            };
          });
          return _this.subscriptions.add(atom.commands.add(_this.element, commandsWithPropagationStopped));
        };
      })(this);
      addElementCommands({
        'tabs:close-tab': (function(_this) {
          return function() {
            return _this.closeTab();
          };
        })(this),
        'tabs:close-other-tabs': (function(_this) {
          return function() {
            return _this.closeOtherTabs();
          };
        })(this),
        'tabs:close-tabs-to-right': (function(_this) {
          return function() {
            return _this.closeTabsToRight();
          };
        })(this),
        'tabs:close-tabs-to-left': (function(_this) {
          return function() {
            return _this.closeTabsToLeft();
          };
        })(this),
        'tabs:close-saved-tabs': (function(_this) {
          return function() {
            return _this.closeSavedTabs();
          };
        })(this),
        'tabs:close-all-tabs': (function(_this) {
          return function() {
            return _this.closeAllTabs();
          };
        })(this),
        'tabs:split-up': (function(_this) {
          return function() {
            return _this.splitTab('splitUp');
          };
        })(this),
        'tabs:split-down': (function(_this) {
          return function() {
            return _this.splitTab('splitDown');
          };
        })(this),
        'tabs:split-left': (function(_this) {
          return function() {
            return _this.splitTab('splitLeft');
          };
        })(this),
        'tabs:split-right': (function(_this) {
          return function() {
            return _this.splitTab('splitRight');
          };
        })(this)
      });
      this.element.addEventListener("mouseenter", this.onMouseEnter.bind(this));
      this.element.addEventListener("mouseleave", this.onMouseLeave.bind(this));
      this.element.addEventListener("mousewheel", this.onMouseWheel.bind(this));
      this.element.addEventListener("dragstart", this.onDragStart.bind(this));
      this.element.addEventListener("dragend", this.onDragEnd.bind(this));
      this.element.addEventListener("dragleave", this.onDragLeave.bind(this));
      this.element.addEventListener("dragover", this.onDragOver.bind(this));
      this.element.addEventListener("drop", this.onDrop.bind(this));
      this.paneElement.addEventListener('dragenter', this.onPaneDragEnter.bind(this));
      this.paneElement.addEventListener('dragleave', this.onPaneDragLeave.bind(this));
      this.paneContainer = this.pane.getContainer();
      ref = this.pane.getItems();
      for (j = 0, len = ref.length; j < len; j++) {
        item = ref[j];
        this.addTabForItem(item);
      }
      this.subscriptions.add(this.pane.onDidDestroy((function(_this) {
        return function() {
          return _this.destroy();
        };
      })(this)));
      this.subscriptions.add(this.pane.onDidAddItem((function(_this) {
        return function(arg) {
          var index, item;
          item = arg.item, index = arg.index;
          return _this.addTabForItem(item, index);
        };
      })(this)));
      this.subscriptions.add(this.pane.onDidMoveItem((function(_this) {
        return function(arg) {
          var item, newIndex;
          item = arg.item, newIndex = arg.newIndex;
          return _this.moveItemTabToIndex(item, newIndex);
        };
      })(this)));
      this.subscriptions.add(this.pane.onDidRemoveItem((function(_this) {
        return function(arg) {
          var item;
          item = arg.item;
          return _this.removeTabForItem(item);
        };
      })(this)));
      this.subscriptions.add(this.pane.onDidChangeActiveItem((function(_this) {
        return function(item) {
          return _this.updateActiveTab();
        };
      })(this)));
      this.subscriptions.add(atom.config.observe('tabs.tabScrolling', (function(_this) {
        return function(value) {
          return _this.updateTabScrolling(value);
        };
      })(this)));
      this.subscriptions.add(atom.config.observe('tabs.tabScrollingThreshold', (function(_this) {
        return function(value) {
          return _this.updateTabScrollingThreshold(value);
        };
      })(this)));
      this.subscriptions.add(atom.config.observe('tabs.alwaysShowTabBar', (function(_this) {
        return function() {
          return _this.updateTabBarVisibility();
        };
      })(this)));
      this.updateActiveTab();
      this.element.addEventListener("mousedown", this.onMouseDown.bind(this));
      this.element.addEventListener("click", this.onClick.bind(this));
      this.element.addEventListener("auxclick", this.onClick.bind(this));
      this.element.addEventListener("dblclick", this.onDoubleClick.bind(this));
      this.onDropOnOtherWindow = this.onDropOnOtherWindow.bind(this);
      ipcRenderer.on('tab:dropped', this.onDropOnOtherWindow);
    }

    TabBarView.prototype.destroy = function() {
      ipcRenderer.removeListener('tab:dropped', this.onDropOnOtherWindow);
      this.subscriptions.dispose();
      return this.element.remove();
    };

    TabBarView.prototype.terminatePendingStates = function() {
      var j, len, ref, tab;
      ref = this.getTabs();
      for (j = 0, len = ref.length; j < len; j++) {
        tab = ref[j];
        if (typeof tab.terminatePendingState === "function") {
          tab.terminatePendingState();
        }
      }
    };

    TabBarView.prototype.addTabForItem = function(item, index) {
      var tabView;
      tabView = new TabView({
        item: item,
        pane: this.pane,
        tabs: this.tabs,
        didClickCloseIcon: (function(_this) {
          return function() {
            _this.closeTab(tabView);
          };
        })(this),
        location: this.location
      });
      if (this.isItemMovingBetweenPanes) {
        tabView.terminatePendingState();
      }
      this.tabsByElement.set(tabView.element, tabView);
      this.insertTabAtIndex(tabView, index);
      if (atom.config.get('tabs.addNewTabsAtEnd')) {
        if (!this.isItemMovingBetweenPanes) {
          return this.pane.moveItem(item, this.pane.getItems().length - 1);
        }
      }
    };

    TabBarView.prototype.moveItemTabToIndex = function(item, index) {
      var tab, tabIndex;
      tabIndex = this.tabs.findIndex(function(t) {
        return t.item === item;
      });
      if (tabIndex !== -1) {
        tab = this.tabs[tabIndex];
        tab.element.remove();
        this.tabs.splice(tabIndex, 1);
        return this.insertTabAtIndex(tab, index);
      }
    };

    TabBarView.prototype.insertTabAtIndex = function(tab, index) {
      var followingTab;
      if (index != null) {
        followingTab = this.tabs[index];
      }
      if (followingTab) {
        this.element.insertBefore(tab.element, followingTab.element);
        this.tabs.splice(index, 0, tab);
      } else {
        this.element.appendChild(tab.element);
        this.tabs.push(tab);
      }
      tab.updateTitle();
      return this.updateTabBarVisibility();
    };

    TabBarView.prototype.removeTabForItem = function(item) {
      var j, len, ref, tab, tabIndex;
      tabIndex = this.tabs.findIndex(function(t) {
        return t.item === item;
      });
      if (tabIndex !== -1) {
        tab = this.tabs[tabIndex];
        this.tabs.splice(tabIndex, 1);
        this.tabsByElement["delete"](tab);
        tab.destroy();
      }
      ref = this.getTabs();
      for (j = 0, len = ref.length; j < len; j++) {
        tab = ref[j];
        tab.updateTitle();
      }
      return this.updateTabBarVisibility();
    };

    TabBarView.prototype.updateTabBarVisibility = function() {
      if (atom.config.get('tabs.alwaysShowTabBar') || this.pane.getItems().length > 1) {
        return this.element.classList.remove('hidden');
      } else {
        return this.element.classList.add('hidden');
      }
    };

    TabBarView.prototype.getTabs = function() {
      return this.tabs.slice();
    };

    TabBarView.prototype.tabAtIndex = function(index) {
      return this.tabs[index];
    };

    TabBarView.prototype.tabForItem = function(item) {
      return this.tabs.find(function(t) {
        return t.item === item;
      });
    };

    TabBarView.prototype.setActiveTab = function(tabView) {
      var ref;
      if ((tabView != null) && tabView !== this.activeTab) {
        if ((ref = this.activeTab) != null) {
          ref.element.classList.remove('active');
        }
        this.activeTab = tabView;
        this.activeTab.element.classList.add('active');
        return this.activeTab.element.scrollIntoView(false);
      }
    };

    TabBarView.prototype.getActiveTab = function() {
      return this.tabForItem(this.pane.getActiveItem());
    };

    TabBarView.prototype.updateActiveTab = function() {
      return this.setActiveTab(this.tabForItem(this.pane.getActiveItem()));
    };

    TabBarView.prototype.closeTab = function(tab) {
      if (tab == null) {
        tab = this.rightClickedTab;
      }
      if (tab != null) {
        return this.pane.destroyItem(tab.item);
      }
    };

    TabBarView.prototype.openInNewWindow = function(tab) {
      var item, itemURI, j, len, pathsToOpen, ref;
      if (tab == null) {
        tab = this.rightClickedTab;
      }
      item = tab != null ? tab.item : void 0;
      if (item == null) {
        return;
      }
      if (typeof item.getURI === 'function') {
        itemURI = item.getURI();
      } else if (typeof item.getPath === 'function') {
        itemURI = item.getPath();
      } else if (typeof item.getUri === 'function') {
        itemURI = item.getUri();
      }
      if (itemURI == null) {
        return;
      }
      this.closeTab(tab);
      ref = this.getTabs();
      for (j = 0, len = ref.length; j < len; j++) {
        tab = ref[j];
        tab.element.style.maxWidth = '';
      }
      pathsToOpen = [atom.project.getPaths(), itemURI].reduce((function(a, b) {
        return a.concat(b);
      }), []);
      return atom.open({
        pathsToOpen: pathsToOpen,
        newWindow: true,
        devMode: atom.devMode,
        safeMode: atom.safeMode
      });
    };

    TabBarView.prototype.splitTab = function(fn) {
      var copiedItem, item, ref;
      if (item = (ref = this.rightClickedTab) != null ? ref.item : void 0) {
        if (copiedItem = typeof item.copy === "function" ? item.copy() : void 0) {
          return this.pane[fn]({
            items: [copiedItem]
          });
        }
      }
    };

    TabBarView.prototype.closeOtherTabs = function(active) {
      var j, len, results, tab, tabs;
      tabs = this.getTabs();
      if (active == null) {
        active = this.rightClickedTab;
      }
      if (active == null) {
        return;
      }
      results = [];
      for (j = 0, len = tabs.length; j < len; j++) {
        tab = tabs[j];
        if (tab !== active) {
          results.push(this.closeTab(tab));
        }
      }
      return results;
    };

    TabBarView.prototype.closeTabsToRight = function(active) {
      var i, index, j, len, results, tab, tabs;
      tabs = this.getTabs();
      if (active == null) {
        active = this.rightClickedTab;
      }
      index = tabs.indexOf(active);
      if (index === -1) {
        return;
      }
      results = [];
      for (i = j = 0, len = tabs.length; j < len; i = ++j) {
        tab = tabs[i];
        if (i > index) {
          results.push(this.closeTab(tab));
        }
      }
      return results;
    };

    TabBarView.prototype.closeTabsToLeft = function(active) {
      var i, index, j, len, results, tab, tabs;
      tabs = this.getTabs();
      if (active == null) {
        active = this.rightClickedTab;
      }
      index = tabs.indexOf(active);
      if (index === -1) {
        return;
      }
      results = [];
      for (i = j = 0, len = tabs.length; j < len; i = ++j) {
        tab = tabs[i];
        if (i < index) {
          results.push(this.closeTab(tab));
        }
      }
      return results;
    };

    TabBarView.prototype.closeSavedTabs = function() {
      var base, j, len, ref, results, tab;
      ref = this.getTabs();
      results = [];
      for (j = 0, len = ref.length; j < len; j++) {
        tab = ref[j];
        if (!(typeof (base = tab.item).isModified === "function" ? base.isModified() : void 0)) {
          results.push(this.closeTab(tab));
        } else {
          results.push(void 0);
        }
      }
      return results;
    };

    TabBarView.prototype.closeAllTabs = function() {
      var j, len, ref, results, tab;
      ref = this.getTabs();
      results = [];
      for (j = 0, len = ref.length; j < len; j++) {
        tab = ref[j];
        results.push(this.closeTab(tab));
      }
      return results;
    };

    TabBarView.prototype.getWindowId = function() {
      return this.windowId != null ? this.windowId : this.windowId = atom.getCurrentWindow().id;
    };

    TabBarView.prototype.onDragStart = function(event) {
      var item, itemURI, j, len, location, paneIndex, ref, ref1, ref2, ref3, tabIndex;
      this.draggedTab = this.tabForElement(event.target);
      if (!this.draggedTab) {
        return;
      }
      this.lastDropTargetIndex = null;
      event.dataTransfer.setData('atom-tab-event', 'true');
      this.draggedTab.element.classList.add('is-dragging');
      this.draggedTab.destroyTooltip();
      tabIndex = this.tabs.indexOf(this.draggedTab);
      event.dataTransfer.setData('sortable-index', tabIndex);
      paneIndex = this.paneContainer.getPanes().indexOf(this.pane);
      event.dataTransfer.setData('from-pane-index', paneIndex);
      event.dataTransfer.setData('from-pane-id', this.pane.id);
      event.dataTransfer.setData('from-window-id', this.getWindowId());
      item = this.pane.getItems()[this.tabs.indexOf(this.draggedTab)];
      if (item == null) {
        return;
      }
      if (typeof item.getURI === 'function') {
        itemURI = (ref = item.getURI()) != null ? ref : '';
      } else if (typeof item.getPath === 'function') {
        itemURI = (ref1 = item.getPath()) != null ? ref1 : '';
      } else if (typeof item.getUri === 'function') {
        itemURI = (ref2 = item.getUri()) != null ? ref2 : '';
      }
      if (typeof item.getAllowedLocations === 'function') {
        ref3 = item.getAllowedLocations();
        for (j = 0, len = ref3.length; j < len; j++) {
          location = ref3[j];
          event.dataTransfer.setData("allowed-location-" + location, 'true');
        }
      } else {
        event.dataTransfer.setData('allow-all-locations', 'true');
      }
      if (itemURI != null) {
        event.dataTransfer.setData('text/plain', itemURI);
        if (process.platform === 'darwin') {
          if (!this.uriHasProtocol(itemURI)) {
            itemURI = "file://" + itemURI;
          }
          event.dataTransfer.setData('text/uri-list', itemURI);
        }
        if ((typeof item.isModified === "function" ? item.isModified() : void 0) && (item.getText != null)) {
          event.dataTransfer.setData('has-unsaved-changes', 'true');
          return event.dataTransfer.setData('modified-text', item.getText());
        }
      }
    };

    TabBarView.prototype.uriHasProtocol = function(uri) {
      var error;
      try {
        return require('url').parse(uri).protocol != null;
      } catch (error1) {
        error = error1;
        return false;
      }
    };

    TabBarView.prototype.onDragLeave = function(event) {
      var j, len, ref, results, tab;
      if (!event.currentTarget.contains(event.relatedTarget)) {
        this.removePlaceholder();
        this.lastDropTargetIndex = null;
        ref = this.getTabs();
        results = [];
        for (j = 0, len = ref.length; j < len; j++) {
          tab = ref[j];
          results.push(tab.element.style.maxWidth = '');
        }
        return results;
      }
    };

    TabBarView.prototype.onDragEnd = function(event) {
      if (!this.tabForElement(event.target)) {
        return;
      }
      return this.clearDropTarget();
    };

    TabBarView.prototype.onDragOver = function(event) {
      var newDropTargetIndex, placeholder, sibling, tab, tabs;
      if (!this.isAtomTabEvent(event)) {
        return;
      }
      if (!this.itemIsAllowed(event, this.location)) {
        return;
      }
      event.preventDefault();
      event.stopPropagation();
      newDropTargetIndex = this.getDropTargetIndex(event);
      if (newDropTargetIndex == null) {
        return;
      }
      if (this.lastDropTargetIndex === newDropTargetIndex) {
        return;
      }
      this.lastDropTargetIndex = newDropTargetIndex;
      this.removeDropTargetClasses();
      tabs = this.getTabs();
      placeholder = this.getPlaceholder();
      if (placeholder == null) {
        return;
      }
      if (newDropTargetIndex < tabs.length) {
        tab = tabs[newDropTargetIndex];
        tab.element.classList.add('is-drop-target');
        return tab.element.parentElement.insertBefore(placeholder, tab.element);
      } else {
        if (tab = tabs[newDropTargetIndex - 1]) {
          tab.element.classList.add('drop-target-is-after');
          if (sibling = tab.element.nextSibling) {
            return tab.element.parentElement.insertBefore(placeholder, sibling);
          } else {
            return tab.element.parentElement.appendChild(placeholder);
          }
        }
      }
    };

    TabBarView.prototype.onDropOnOtherWindow = function(event, fromPaneId, fromItemIndex) {
      var itemToRemove;
      if (this.pane.id === fromPaneId) {
        if (itemToRemove = this.pane.getItems()[fromItemIndex]) {
          this.pane.destroyItem(itemToRemove);
        }
      }
      return this.clearDropTarget();
    };

    TabBarView.prototype.clearDropTarget = function() {
      var ref, ref1;
      if ((ref = this.draggedTab) != null) {
        ref.element.classList.remove('is-dragging');
      }
      if ((ref1 = this.draggedTab) != null) {
        ref1.updateTooltip();
      }
      this.draggedTab = null;
      this.removeDropTargetClasses();
      return this.removePlaceholder();
    };

    TabBarView.prototype.onDrop = function(event) {
      var droppedURI, fromIndex, fromPane, fromPaneId, fromPaneIndex, fromWindowId, hasUnsavedChanges, item, modifiedText, toIndex, toPane;
      if (!this.isAtomTabEvent(event)) {
        return;
      }
      event.preventDefault();
      fromWindowId = parseInt(event.dataTransfer.getData('from-window-id'));
      fromPaneId = parseInt(event.dataTransfer.getData('from-pane-id'));
      fromIndex = parseInt(event.dataTransfer.getData('sortable-index'));
      fromPaneIndex = parseInt(event.dataTransfer.getData('from-pane-index'));
      hasUnsavedChanges = event.dataTransfer.getData('has-unsaved-changes') === 'true';
      modifiedText = event.dataTransfer.getData('modified-text');
      toIndex = this.getDropTargetIndex(event);
      toPane = this.pane;
      this.clearDropTarget();
      if (!this.itemIsAllowed(event, this.location)) {
        return;
      }
      if (fromWindowId === this.getWindowId()) {
        fromPane = this.paneContainer.getPanes()[fromPaneIndex];
        if ((fromPane != null ? fromPane.id : void 0) !== fromPaneId) {
          fromPane = Array.from(document.querySelectorAll('atom-pane')).map(function(paneEl) {
            return paneEl.model;
          }).find(function(pane) {
            return pane.id === fromPaneId;
          });
        }
        item = fromPane.getItems()[fromIndex];
        if (item != null) {
          return this.moveItemBetweenPanes(fromPane, fromIndex, toPane, toIndex, item);
        }
      } else {
        droppedURI = event.dataTransfer.getData('text/plain');
        atom.workspace.open(droppedURI).then((function(_this) {
          return function(item) {
            var activeItemIndex, activePane, browserWindow;
            activePane = atom.workspace.getActivePane();
            activeItemIndex = activePane.getItems().indexOf(item);
            _this.moveItemBetweenPanes(activePane, activeItemIndex, toPane, toIndex, item);
            if (hasUnsavedChanges) {
              if (typeof item.setText === "function") {
                item.setText(modifiedText);
              }
            }
            if (!isNaN(fromWindowId)) {
              browserWindow = _this.browserWindowForId(fromWindowId);
              return browserWindow != null ? browserWindow.webContents.send('tab:dropped', fromPaneId, fromIndex) : void 0;
            }
          };
        })(this));
        return atom.focus();
      }
    };

    TabBarView.prototype.onPaneDragEnter = function(event) {
      if (!this.isAtomTabEvent(event)) {
        return;
      }
      if (!this.itemIsAllowed(event, this.location)) {
        return;
      }
      if (this.pane.getItems().length > 1 || atom.config.get('tabs.alwaysShowTabBar')) {
        return;
      }
      if (this.paneElement.contains(event.relatedTarget)) {
        return this.element.classList.remove('hidden');
      }
    };

    TabBarView.prototype.onPaneDragLeave = function(event) {
      if (!this.isAtomTabEvent(event)) {
        return;
      }
      if (!this.itemIsAllowed(event, this.location)) {
        return;
      }
      if (this.pane.getItems().length > 1 || atom.config.get('tabs.alwaysShowTabBar')) {
        return;
      }
      if (!this.paneElement.contains(event.relatedTarget)) {
        return this.element.classList.add('hidden');
      }
    };

    TabBarView.prototype.onMouseWheel = function(event) {
      if (event.shiftKey || !this.tabScrolling) {
        return;
      }
      if (this.wheelDelta == null) {
        this.wheelDelta = 0;
      }
      this.wheelDelta += event.wheelDeltaY;
      if (this.wheelDelta <= -this.tabScrollingThreshold) {
        this.wheelDelta = 0;
        return this.pane.activateNextItem();
      } else if (this.wheelDelta >= this.tabScrollingThreshold) {
        this.wheelDelta = 0;
        return this.pane.activatePreviousItem();
      }
    };

    TabBarView.prototype.onMouseDown = function(event) {
      var ref, tab;
      tab = this.tabForElement(event.target);
      if (!tab) {
        return;
      }
      if (event.which === 3 || (event.which === 1 && event.ctrlKey === true)) {
        if ((ref = this.rightClickedTab) != null) {
          ref.element.classList.remove('right-clicked');
        }
        this.rightClickedTab = tab;
        this.rightClickedTab.element.classList.add('right-clicked');
        return event.preventDefault();
      } else if (event.which === 2) {
        return event.preventDefault();
      }
    };

    TabBarView.prototype.onClick = function(event) {
      var tab;
      tab = this.tabForElement(event.target);
      if (!tab) {
        return;
      }
      event.preventDefault();
      if (event.which === 3 || (event.which === 1 && event.ctrlKey === true)) {

      } else if (event.which === 1 && !event.target.classList.contains('close-icon')) {
        this.pane.activateItem(tab.item);
        if (!this.pane.isDestroyed()) {
          return this.pane.activate();
        }
      } else if (event.which === 2) {
        return this.pane.destroyItem(tab.item);
      }
    };

    TabBarView.prototype.onDoubleClick = function(event) {
      var base, tab;
      if (tab = this.tabForElement(event.target)) {
        return typeof (base = tab.item).terminatePendingState === "function" ? base.terminatePendingState() : void 0;
      } else if (event.target === this.element) {
        atom.commands.dispatch(this.element, 'application:new-file');
        return event.preventDefault();
      }
    };

    TabBarView.prototype.updateTabScrollingThreshold = function(value) {
      return this.tabScrollingThreshold = value;
    };

    TabBarView.prototype.updateTabScrolling = function(value) {
      if (value === 'platform') {
        return this.tabScrolling = process.platform === 'linux';
      } else {
        return this.tabScrolling = value;
      }
    };

    TabBarView.prototype.browserWindowForId = function(id) {
      if (BrowserWindow == null) {
        BrowserWindow = require('electron').remote.BrowserWindow;
      }
      return BrowserWindow.fromId(id);
    };

    TabBarView.prototype.moveItemBetweenPanes = function(fromPane, fromIndex, toPane, toIndex, item) {
      try {
        if (toPane === fromPane) {
          if (fromIndex < toIndex) {
            toIndex--;
          }
          toPane.moveItem(item, toIndex);
        } else {
          this.isItemMovingBetweenPanes = true;
          fromPane.moveItemToPane(item, toPane, toIndex--);
        }
        toPane.activateItem(item);
        return toPane.activate();
      } finally {
        this.isItemMovingBetweenPanes = false;
      }
    };

    TabBarView.prototype.removeDropTargetClasses = function() {
      var dropTarget, j, k, len, len1, ref, ref1, results, workspaceElement;
      workspaceElement = atom.workspace.getElement();
      ref = workspaceElement.querySelectorAll('.tab-bar .is-drop-target');
      for (j = 0, len = ref.length; j < len; j++) {
        dropTarget = ref[j];
        dropTarget.classList.remove('is-drop-target');
      }
      ref1 = workspaceElement.querySelectorAll('.tab-bar .drop-target-is-after');
      results = [];
      for (k = 0, len1 = ref1.length; k < len1; k++) {
        dropTarget = ref1[k];
        results.push(dropTarget.classList.remove('drop-target-is-after'));
      }
      return results;
    };

    TabBarView.prototype.getDropTargetIndex = function(event) {
      var elementCenter, elementIndex, left, ref, tab, tabs, target, width;
      target = event.target;
      if (this.isPlaceholder(target)) {
        return;
      }
      tabs = this.getTabs();
      tab = this.tabForElement(target);
      if (tab == null) {
        tab = tabs[tabs.length - 1];
      }
      if (tab == null) {
        return 0;
      }
      ref = tab.element.getBoundingClientRect(), left = ref.left, width = ref.width;
      elementCenter = left + width / 2;
      elementIndex = tabs.indexOf(tab);
      if (event.pageX < elementCenter) {
        return elementIndex;
      } else {
        return elementIndex + 1;
      }
    };

    TabBarView.prototype.getPlaceholder = function() {
      if (this.placeholderEl != null) {
        return this.placeholderEl;
      }
      this.placeholderEl = document.createElement("li");
      this.placeholderEl.classList.add("placeholder");
      return this.placeholderEl;
    };

    TabBarView.prototype.removePlaceholder = function() {
      var ref;
      if ((ref = this.placeholderEl) != null) {
        ref.remove();
      }
      return this.placeholderEl = null;
    };

    TabBarView.prototype.isPlaceholder = function(element) {
      return element.classList.contains('placeholder');
    };

    TabBarView.prototype.onMouseEnter = function() {
      var j, len, ref, tab, width;
      ref = this.getTabs();
      for (j = 0, len = ref.length; j < len; j++) {
        tab = ref[j];
        width = tab.element.getBoundingClientRect().width;
        tab.element.style.maxWidth = width.toFixed(2) + 'px';
      }
    };

    TabBarView.prototype.onMouseLeave = function() {
      var j, len, ref, tab;
      ref = this.getTabs();
      for (j = 0, len = ref.length; j < len; j++) {
        tab = ref[j];
        tab.element.style.maxWidth = '';
      }
    };

    TabBarView.prototype.tabForElement = function(element) {
      var currentElement, tab;
      currentElement = element;
      while (currentElement != null) {
        if (tab = this.tabsByElement.get(currentElement)) {
          return tab;
        } else {
          currentElement = currentElement.parentElement;
        }
      }
    };

    TabBarView.prototype.isAtomTabEvent = function(event) {
      var item, j, len, ref;
      ref = event.dataTransfer.items;
      for (j = 0, len = ref.length; j < len; j++) {
        item = ref[j];
        if (item.type === 'atom-tab-event') {
          return true;
        }
      }
      return false;
    };

    TabBarView.prototype.itemIsAllowed = function(event, location) {
      var item, j, len, ref;
      ref = event.dataTransfer.items;
      for (j = 0, len = ref.length; j < len; j++) {
        item = ref[j];
        if (item.type === 'allow-all-locations' || item.type === ("allowed-location-" + location)) {
          return true;
        }
      }
      return false;
    };

    return TabBarView;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiL2J1aWxkL2F0b20vc3JjL2F0b20tMS4zOC4yL291dC9hcHAvbm9kZV9tb2R1bGVzL3RhYnMvbGliL3RhYi1iYXItdmlldy5jb2ZmZWUiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUE7QUFBQSxNQUFBOztFQUFBLGFBQUEsR0FBZ0I7O0VBQ2YsY0FBZSxPQUFBLENBQVEsVUFBUjs7RUFFZixzQkFBdUIsT0FBQSxDQUFRLE1BQVI7O0VBQ3hCLE9BQUEsR0FBVSxPQUFBLENBQVEsWUFBUjs7RUFFVixNQUFNLENBQUMsT0FBUCxHQUNNO0lBQ1Msb0JBQUMsS0FBRCxFQUFRLFNBQVI7QUFDWCxVQUFBO01BRFksSUFBQyxDQUFBLE9BQUQ7TUFBTyxJQUFDLENBQUEsV0FBRDtNQUNuQixJQUFDLENBQUEsT0FBRCxHQUFXLFFBQVEsQ0FBQyxhQUFULENBQXVCLElBQXZCO01BQ1gsSUFBQyxDQUFBLE9BQU8sQ0FBQyxTQUFTLENBQUMsR0FBbkIsQ0FBdUIsYUFBdkI7TUFDQSxJQUFDLENBQUEsT0FBTyxDQUFDLFNBQVMsQ0FBQyxHQUFuQixDQUF1QixTQUF2QjtNQUNBLElBQUMsQ0FBQSxPQUFPLENBQUMsU0FBUyxDQUFDLEdBQW5CLENBQXVCLGFBQXZCO01BQ0EsSUFBQyxDQUFBLE9BQU8sQ0FBQyxZQUFULENBQXNCLElBQXRCLEVBQTRCLFdBQTVCO01BQ0EsSUFBQyxDQUFBLE9BQU8sQ0FBQyxZQUFULENBQXNCLFVBQXRCLEVBQWtDLENBQUMsQ0FBbkM7TUFDQSxJQUFDLENBQUEsT0FBTyxDQUFDLFlBQVQsQ0FBc0IsVUFBdEIsRUFBa0MsSUFBQyxDQUFBLFFBQW5DO01BRUEsSUFBQyxDQUFBLElBQUQsR0FBUTtNQUNSLElBQUMsQ0FBQSxhQUFELEdBQWlCLElBQUk7TUFDckIsSUFBQyxDQUFBLGFBQUQsR0FBaUIsSUFBSTtNQUVyQixJQUFDLENBQUEsV0FBRCxHQUFlLElBQUMsQ0FBQSxJQUFJLENBQUMsVUFBTixDQUFBO01BRWYsSUFBQyxDQUFBLGFBQWEsQ0FBQyxHQUFmLENBQW1CLElBQUksQ0FBQyxRQUFRLENBQUMsR0FBZCxDQUFrQixJQUFDLENBQUEsV0FBbkIsRUFDakI7UUFBQSx1QkFBQSxFQUF5QixDQUFBLFNBQUEsS0FBQTtpQkFBQSxTQUFBO21CQUFHLEtBQUMsQ0FBQSxzQkFBRCxDQUFBO1VBQUg7UUFBQSxDQUFBLENBQUEsQ0FBQSxJQUFBLENBQXpCO1FBQ0EsZ0JBQUEsRUFBa0IsQ0FBQSxTQUFBLEtBQUE7aUJBQUEsU0FBQTttQkFBRyxLQUFDLENBQUEsUUFBRCxDQUFVLEtBQUMsQ0FBQSxZQUFELENBQUEsQ0FBVjtVQUFIO1FBQUEsQ0FBQSxDQUFBLENBQUEsSUFBQSxDQURsQjtRQUVBLHVCQUFBLEVBQXlCLENBQUEsU0FBQSxLQUFBO2lCQUFBLFNBQUE7bUJBQUcsS0FBQyxDQUFBLGNBQUQsQ0FBZ0IsS0FBQyxDQUFBLFlBQUQsQ0FBQSxDQUFoQjtVQUFIO1FBQUEsQ0FBQSxDQUFBLENBQUEsSUFBQSxDQUZ6QjtRQUdBLDBCQUFBLEVBQTRCLENBQUEsU0FBQSxLQUFBO2lCQUFBLFNBQUE7bUJBQUcsS0FBQyxDQUFBLGdCQUFELENBQWtCLEtBQUMsQ0FBQSxZQUFELENBQUEsQ0FBbEI7VUFBSDtRQUFBLENBQUEsQ0FBQSxDQUFBLElBQUEsQ0FINUI7UUFJQSx5QkFBQSxFQUEyQixDQUFBLFNBQUEsS0FBQTtpQkFBQSxTQUFBO21CQUFHLEtBQUMsQ0FBQSxlQUFELENBQWlCLEtBQUMsQ0FBQSxZQUFELENBQUEsQ0FBakI7VUFBSDtRQUFBLENBQUEsQ0FBQSxDQUFBLElBQUEsQ0FKM0I7UUFLQSx1QkFBQSxFQUF5QixDQUFBLFNBQUEsS0FBQTtpQkFBQSxTQUFBO21CQUFHLEtBQUMsQ0FBQSxjQUFELENBQUE7VUFBSDtRQUFBLENBQUEsQ0FBQSxDQUFBLElBQUEsQ0FMekI7UUFNQSxxQkFBQSxFQUF1QixDQUFBLFNBQUEsS0FBQTtpQkFBQSxTQUFDLEtBQUQ7WUFDckIsS0FBSyxDQUFDLGVBQU4sQ0FBQTttQkFDQSxLQUFDLENBQUEsWUFBRCxDQUFBO1VBRnFCO1FBQUEsQ0FBQSxDQUFBLENBQUEsSUFBQSxDQU52QjtRQVNBLHlCQUFBLEVBQTJCLENBQUEsU0FBQSxLQUFBO2lCQUFBLFNBQUE7bUJBQUcsS0FBQyxDQUFBLGVBQUQsQ0FBQTtVQUFIO1FBQUEsQ0FBQSxDQUFBLENBQUEsSUFBQSxDQVQzQjtPQURpQixDQUFuQjtNQVlBLGtCQUFBLEdBQXFCLENBQUEsU0FBQSxLQUFBO2VBQUEsU0FBQyxRQUFEO0FBQ25CLGNBQUE7VUFBQSw4QkFBQSxHQUFpQztVQUNqQyxNQUFNLENBQUMsSUFBUCxDQUFZLFFBQVosQ0FBcUIsQ0FBQyxPQUF0QixDQUE4QixTQUFDLElBQUQ7bUJBQzVCLDhCQUErQixDQUFBLElBQUEsQ0FBL0IsR0FBdUMsU0FBQyxLQUFEO2NBQ3JDLEtBQUssQ0FBQyxlQUFOLENBQUE7cUJBQ0EsUUFBUyxDQUFBLElBQUEsQ0FBVCxDQUFBO1lBRnFDO1VBRFgsQ0FBOUI7aUJBS0EsS0FBQyxDQUFBLGFBQWEsQ0FBQyxHQUFmLENBQW1CLElBQUksQ0FBQyxRQUFRLENBQUMsR0FBZCxDQUFrQixLQUFDLENBQUEsT0FBbkIsRUFBNEIsOEJBQTVCLENBQW5CO1FBUG1CO01BQUEsQ0FBQSxDQUFBLENBQUEsSUFBQTtNQVNyQixrQkFBQSxDQUNFO1FBQUEsZ0JBQUEsRUFBa0IsQ0FBQSxTQUFBLEtBQUE7aUJBQUEsU0FBQTttQkFBRyxLQUFDLENBQUEsUUFBRCxDQUFBO1VBQUg7UUFBQSxDQUFBLENBQUEsQ0FBQSxJQUFBLENBQWxCO1FBQ0EsdUJBQUEsRUFBeUIsQ0FBQSxTQUFBLEtBQUE7aUJBQUEsU0FBQTttQkFBRyxLQUFDLENBQUEsY0FBRCxDQUFBO1VBQUg7UUFBQSxDQUFBLENBQUEsQ0FBQSxJQUFBLENBRHpCO1FBRUEsMEJBQUEsRUFBNEIsQ0FBQSxTQUFBLEtBQUE7aUJBQUEsU0FBQTttQkFBRyxLQUFDLENBQUEsZ0JBQUQsQ0FBQTtVQUFIO1FBQUEsQ0FBQSxDQUFBLENBQUEsSUFBQSxDQUY1QjtRQUdBLHlCQUFBLEVBQTJCLENBQUEsU0FBQSxLQUFBO2lCQUFBLFNBQUE7bUJBQUcsS0FBQyxDQUFBLGVBQUQsQ0FBQTtVQUFIO1FBQUEsQ0FBQSxDQUFBLENBQUEsSUFBQSxDQUgzQjtRQUlBLHVCQUFBLEVBQXlCLENBQUEsU0FBQSxLQUFBO2lCQUFBLFNBQUE7bUJBQUcsS0FBQyxDQUFBLGNBQUQsQ0FBQTtVQUFIO1FBQUEsQ0FBQSxDQUFBLENBQUEsSUFBQSxDQUp6QjtRQUtBLHFCQUFBLEVBQXVCLENBQUEsU0FBQSxLQUFBO2lCQUFBLFNBQUE7bUJBQUcsS0FBQyxDQUFBLFlBQUQsQ0FBQTtVQUFIO1FBQUEsQ0FBQSxDQUFBLENBQUEsSUFBQSxDQUx2QjtRQU1BLGVBQUEsRUFBaUIsQ0FBQSxTQUFBLEtBQUE7aUJBQUEsU0FBQTttQkFBRyxLQUFDLENBQUEsUUFBRCxDQUFVLFNBQVY7VUFBSDtRQUFBLENBQUEsQ0FBQSxDQUFBLElBQUEsQ0FOakI7UUFPQSxpQkFBQSxFQUFtQixDQUFBLFNBQUEsS0FBQTtpQkFBQSxTQUFBO21CQUFHLEtBQUMsQ0FBQSxRQUFELENBQVUsV0FBVjtVQUFIO1FBQUEsQ0FBQSxDQUFBLENBQUEsSUFBQSxDQVBuQjtRQVFBLGlCQUFBLEVBQW1CLENBQUEsU0FBQSxLQUFBO2lCQUFBLFNBQUE7bUJBQUcsS0FBQyxDQUFBLFFBQUQsQ0FBVSxXQUFWO1VBQUg7UUFBQSxDQUFBLENBQUEsQ0FBQSxJQUFBLENBUm5CO1FBU0Esa0JBQUEsRUFBb0IsQ0FBQSxTQUFBLEtBQUE7aUJBQUEsU0FBQTttQkFBRyxLQUFDLENBQUEsUUFBRCxDQUFVLFlBQVY7VUFBSDtRQUFBLENBQUEsQ0FBQSxDQUFBLElBQUEsQ0FUcEI7T0FERjtNQVlBLElBQUMsQ0FBQSxPQUFPLENBQUMsZ0JBQVQsQ0FBMEIsWUFBMUIsRUFBd0MsSUFBQyxDQUFBLFlBQVksQ0FBQyxJQUFkLENBQW1CLElBQW5CLENBQXhDO01BQ0EsSUFBQyxDQUFBLE9BQU8sQ0FBQyxnQkFBVCxDQUEwQixZQUExQixFQUF3QyxJQUFDLENBQUEsWUFBWSxDQUFDLElBQWQsQ0FBbUIsSUFBbkIsQ0FBeEM7TUFDQSxJQUFDLENBQUEsT0FBTyxDQUFDLGdCQUFULENBQTBCLFlBQTFCLEVBQXdDLElBQUMsQ0FBQSxZQUFZLENBQUMsSUFBZCxDQUFtQixJQUFuQixDQUF4QztNQUNBLElBQUMsQ0FBQSxPQUFPLENBQUMsZ0JBQVQsQ0FBMEIsV0FBMUIsRUFBdUMsSUFBQyxDQUFBLFdBQVcsQ0FBQyxJQUFiLENBQWtCLElBQWxCLENBQXZDO01BQ0EsSUFBQyxDQUFBLE9BQU8sQ0FBQyxnQkFBVCxDQUEwQixTQUExQixFQUFxQyxJQUFDLENBQUEsU0FBUyxDQUFDLElBQVgsQ0FBZ0IsSUFBaEIsQ0FBckM7TUFDQSxJQUFDLENBQUEsT0FBTyxDQUFDLGdCQUFULENBQTBCLFdBQTFCLEVBQXVDLElBQUMsQ0FBQSxXQUFXLENBQUMsSUFBYixDQUFrQixJQUFsQixDQUF2QztNQUNBLElBQUMsQ0FBQSxPQUFPLENBQUMsZ0JBQVQsQ0FBMEIsVUFBMUIsRUFBc0MsSUFBQyxDQUFBLFVBQVUsQ0FBQyxJQUFaLENBQWlCLElBQWpCLENBQXRDO01BQ0EsSUFBQyxDQUFBLE9BQU8sQ0FBQyxnQkFBVCxDQUEwQixNQUExQixFQUFrQyxJQUFDLENBQUEsTUFBTSxDQUFDLElBQVIsQ0FBYSxJQUFiLENBQWxDO01BR0EsSUFBQyxDQUFBLFdBQVcsQ0FBQyxnQkFBYixDQUE4QixXQUE5QixFQUEyQyxJQUFDLENBQUEsZUFBZSxDQUFDLElBQWpCLENBQXNCLElBQXRCLENBQTNDO01BQ0EsSUFBQyxDQUFBLFdBQVcsQ0FBQyxnQkFBYixDQUE4QixXQUE5QixFQUEyQyxJQUFDLENBQUEsZUFBZSxDQUFDLElBQWpCLENBQXNCLElBQXRCLENBQTNDO01BRUEsSUFBQyxDQUFBLGFBQUQsR0FBaUIsSUFBQyxDQUFBLElBQUksQ0FBQyxZQUFOLENBQUE7QUFDakI7QUFBQSxXQUFBLHFDQUFBOztRQUFBLElBQUMsQ0FBQSxhQUFELENBQWUsSUFBZjtBQUFBO01BRUEsSUFBQyxDQUFBLGFBQWEsQ0FBQyxHQUFmLENBQW1CLElBQUMsQ0FBQSxJQUFJLENBQUMsWUFBTixDQUFtQixDQUFBLFNBQUEsS0FBQTtlQUFBLFNBQUE7aUJBQ3BDLEtBQUMsQ0FBQSxPQUFELENBQUE7UUFEb0M7TUFBQSxDQUFBLENBQUEsQ0FBQSxJQUFBLENBQW5CLENBQW5CO01BR0EsSUFBQyxDQUFBLGFBQWEsQ0FBQyxHQUFmLENBQW1CLElBQUMsQ0FBQSxJQUFJLENBQUMsWUFBTixDQUFtQixDQUFBLFNBQUEsS0FBQTtlQUFBLFNBQUMsR0FBRDtBQUNwQyxjQUFBO1VBRHNDLGlCQUFNO2lCQUM1QyxLQUFDLENBQUEsYUFBRCxDQUFlLElBQWYsRUFBcUIsS0FBckI7UUFEb0M7TUFBQSxDQUFBLENBQUEsQ0FBQSxJQUFBLENBQW5CLENBQW5CO01BR0EsSUFBQyxDQUFBLGFBQWEsQ0FBQyxHQUFmLENBQW1CLElBQUMsQ0FBQSxJQUFJLENBQUMsYUFBTixDQUFvQixDQUFBLFNBQUEsS0FBQTtlQUFBLFNBQUMsR0FBRDtBQUNyQyxjQUFBO1VBRHVDLGlCQUFNO2lCQUM3QyxLQUFDLENBQUEsa0JBQUQsQ0FBb0IsSUFBcEIsRUFBMEIsUUFBMUI7UUFEcUM7TUFBQSxDQUFBLENBQUEsQ0FBQSxJQUFBLENBQXBCLENBQW5CO01BR0EsSUFBQyxDQUFBLGFBQWEsQ0FBQyxHQUFmLENBQW1CLElBQUMsQ0FBQSxJQUFJLENBQUMsZUFBTixDQUFzQixDQUFBLFNBQUEsS0FBQTtlQUFBLFNBQUMsR0FBRDtBQUN2QyxjQUFBO1VBRHlDLE9BQUQ7aUJBQ3hDLEtBQUMsQ0FBQSxnQkFBRCxDQUFrQixJQUFsQjtRQUR1QztNQUFBLENBQUEsQ0FBQSxDQUFBLElBQUEsQ0FBdEIsQ0FBbkI7TUFHQSxJQUFDLENBQUEsYUFBYSxDQUFDLEdBQWYsQ0FBbUIsSUFBQyxDQUFBLElBQUksQ0FBQyxxQkFBTixDQUE0QixDQUFBLFNBQUEsS0FBQTtlQUFBLFNBQUMsSUFBRDtpQkFDN0MsS0FBQyxDQUFBLGVBQUQsQ0FBQTtRQUQ2QztNQUFBLENBQUEsQ0FBQSxDQUFBLElBQUEsQ0FBNUIsQ0FBbkI7TUFHQSxJQUFDLENBQUEsYUFBYSxDQUFDLEdBQWYsQ0FBbUIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxPQUFaLENBQW9CLG1CQUFwQixFQUF5QyxDQUFBLFNBQUEsS0FBQTtlQUFBLFNBQUMsS0FBRDtpQkFBVyxLQUFDLENBQUEsa0JBQUQsQ0FBb0IsS0FBcEI7UUFBWDtNQUFBLENBQUEsQ0FBQSxDQUFBLElBQUEsQ0FBekMsQ0FBbkI7TUFDQSxJQUFDLENBQUEsYUFBYSxDQUFDLEdBQWYsQ0FBbUIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxPQUFaLENBQW9CLDRCQUFwQixFQUFrRCxDQUFBLFNBQUEsS0FBQTtlQUFBLFNBQUMsS0FBRDtpQkFBVyxLQUFDLENBQUEsMkJBQUQsQ0FBNkIsS0FBN0I7UUFBWDtNQUFBLENBQUEsQ0FBQSxDQUFBLElBQUEsQ0FBbEQsQ0FBbkI7TUFDQSxJQUFDLENBQUEsYUFBYSxDQUFDLEdBQWYsQ0FBbUIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxPQUFaLENBQW9CLHVCQUFwQixFQUE2QyxDQUFBLFNBQUEsS0FBQTtlQUFBLFNBQUE7aUJBQUcsS0FBQyxDQUFBLHNCQUFELENBQUE7UUFBSDtNQUFBLENBQUEsQ0FBQSxDQUFBLElBQUEsQ0FBN0MsQ0FBbkI7TUFFQSxJQUFDLENBQUEsZUFBRCxDQUFBO01BRUEsSUFBQyxDQUFBLE9BQU8sQ0FBQyxnQkFBVCxDQUEwQixXQUExQixFQUF1QyxJQUFDLENBQUEsV0FBVyxDQUFDLElBQWIsQ0FBa0IsSUFBbEIsQ0FBdkM7TUFDQSxJQUFDLENBQUEsT0FBTyxDQUFDLGdCQUFULENBQTBCLE9BQTFCLEVBQW1DLElBQUMsQ0FBQSxPQUFPLENBQUMsSUFBVCxDQUFjLElBQWQsQ0FBbkM7TUFDQSxJQUFDLENBQUEsT0FBTyxDQUFDLGdCQUFULENBQTBCLFVBQTFCLEVBQXNDLElBQUMsQ0FBQSxPQUFPLENBQUMsSUFBVCxDQUFjLElBQWQsQ0FBdEM7TUFDQSxJQUFDLENBQUEsT0FBTyxDQUFDLGdCQUFULENBQTBCLFVBQTFCLEVBQXNDLElBQUMsQ0FBQSxhQUFhLENBQUMsSUFBZixDQUFvQixJQUFwQixDQUF0QztNQUVBLElBQUMsQ0FBQSxtQkFBRCxHQUF1QixJQUFDLENBQUEsbUJBQW1CLENBQUMsSUFBckIsQ0FBMEIsSUFBMUI7TUFDdkIsV0FBVyxDQUFDLEVBQVosQ0FBZSxhQUFmLEVBQThCLElBQUMsQ0FBQSxtQkFBL0I7SUEzRlc7O3lCQTZGYixPQUFBLEdBQVMsU0FBQTtNQUNQLFdBQVcsQ0FBQyxjQUFaLENBQTJCLGFBQTNCLEVBQTBDLElBQUMsQ0FBQSxtQkFBM0M7TUFDQSxJQUFDLENBQUEsYUFBYSxDQUFDLE9BQWYsQ0FBQTthQUNBLElBQUMsQ0FBQSxPQUFPLENBQUMsTUFBVCxDQUFBO0lBSE87O3lCQUtULHNCQUFBLEdBQXdCLFNBQUE7QUFDdEIsVUFBQTtBQUFBO0FBQUEsV0FBQSxxQ0FBQTs7O1VBQUEsR0FBRyxDQUFDOztBQUFKO0lBRHNCOzt5QkFJeEIsYUFBQSxHQUFlLFNBQUMsSUFBRCxFQUFPLEtBQVA7QUFDYixVQUFBO01BQUEsT0FBQSxHQUFVLElBQUksT0FBSixDQUFZO1FBQ3BCLE1BQUEsSUFEb0I7UUFFbkIsTUFBRCxJQUFDLENBQUEsSUFGbUI7UUFHbkIsTUFBRCxJQUFDLENBQUEsSUFIbUI7UUFJcEIsaUJBQUEsRUFBbUIsQ0FBQSxTQUFBLEtBQUE7aUJBQUEsU0FBQTtZQUNqQixLQUFDLENBQUEsUUFBRCxDQUFVLE9BQVY7VUFEaUI7UUFBQSxDQUFBLENBQUEsQ0FBQSxJQUFBLENBSkM7UUFPbkIsVUFBRCxJQUFDLENBQUEsUUFQbUI7T0FBWjtNQVNWLElBQW1DLElBQUMsQ0FBQSx3QkFBcEM7UUFBQSxPQUFPLENBQUMscUJBQVIsQ0FBQSxFQUFBOztNQUNBLElBQUMsQ0FBQSxhQUFhLENBQUMsR0FBZixDQUFtQixPQUFPLENBQUMsT0FBM0IsRUFBb0MsT0FBcEM7TUFDQSxJQUFDLENBQUEsZ0JBQUQsQ0FBa0IsT0FBbEIsRUFBMkIsS0FBM0I7TUFDQSxJQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBWixDQUFnQixzQkFBaEIsQ0FBSDtRQUNFLElBQUEsQ0FBeUQsSUFBQyxDQUFBLHdCQUExRDtpQkFBQSxJQUFDLENBQUEsSUFBSSxDQUFDLFFBQU4sQ0FBZSxJQUFmLEVBQXFCLElBQUMsQ0FBQSxJQUFJLENBQUMsUUFBTixDQUFBLENBQWdCLENBQUMsTUFBakIsR0FBMEIsQ0FBL0MsRUFBQTtTQURGOztJQWJhOzt5QkFnQmYsa0JBQUEsR0FBb0IsU0FBQyxJQUFELEVBQU8sS0FBUDtBQUNsQixVQUFBO01BQUEsUUFBQSxHQUFXLElBQUMsQ0FBQSxJQUFJLENBQUMsU0FBTixDQUFnQixTQUFDLENBQUQ7ZUFBTyxDQUFDLENBQUMsSUFBRixLQUFVO01BQWpCLENBQWhCO01BQ1gsSUFBRyxRQUFBLEtBQWMsQ0FBQyxDQUFsQjtRQUNFLEdBQUEsR0FBTSxJQUFDLENBQUEsSUFBSyxDQUFBLFFBQUE7UUFDWixHQUFHLENBQUMsT0FBTyxDQUFDLE1BQVosQ0FBQTtRQUNBLElBQUMsQ0FBQSxJQUFJLENBQUMsTUFBTixDQUFhLFFBQWIsRUFBdUIsQ0FBdkI7ZUFDQSxJQUFDLENBQUEsZ0JBQUQsQ0FBa0IsR0FBbEIsRUFBdUIsS0FBdkIsRUFKRjs7SUFGa0I7O3lCQVFwQixnQkFBQSxHQUFrQixTQUFDLEdBQUQsRUFBTSxLQUFOO0FBQ2hCLFVBQUE7TUFBQSxJQUErQixhQUEvQjtRQUFBLFlBQUEsR0FBZSxJQUFDLENBQUEsSUFBSyxDQUFBLEtBQUEsRUFBckI7O01BQ0EsSUFBRyxZQUFIO1FBQ0UsSUFBQyxDQUFBLE9BQU8sQ0FBQyxZQUFULENBQXNCLEdBQUcsQ0FBQyxPQUExQixFQUFtQyxZQUFZLENBQUMsT0FBaEQ7UUFDQSxJQUFDLENBQUEsSUFBSSxDQUFDLE1BQU4sQ0FBYSxLQUFiLEVBQW9CLENBQXBCLEVBQXVCLEdBQXZCLEVBRkY7T0FBQSxNQUFBO1FBSUUsSUFBQyxDQUFBLE9BQU8sQ0FBQyxXQUFULENBQXFCLEdBQUcsQ0FBQyxPQUF6QjtRQUNBLElBQUMsQ0FBQSxJQUFJLENBQUMsSUFBTixDQUFXLEdBQVgsRUFMRjs7TUFPQSxHQUFHLENBQUMsV0FBSixDQUFBO2FBQ0EsSUFBQyxDQUFBLHNCQUFELENBQUE7SUFWZ0I7O3lCQVlsQixnQkFBQSxHQUFrQixTQUFDLElBQUQ7QUFDaEIsVUFBQTtNQUFBLFFBQUEsR0FBVyxJQUFDLENBQUEsSUFBSSxDQUFDLFNBQU4sQ0FBZ0IsU0FBQyxDQUFEO2VBQU8sQ0FBQyxDQUFDLElBQUYsS0FBVTtNQUFqQixDQUFoQjtNQUNYLElBQUcsUUFBQSxLQUFjLENBQUMsQ0FBbEI7UUFDRSxHQUFBLEdBQU0sSUFBQyxDQUFBLElBQUssQ0FBQSxRQUFBO1FBQ1osSUFBQyxDQUFBLElBQUksQ0FBQyxNQUFOLENBQWEsUUFBYixFQUF1QixDQUF2QjtRQUNBLElBQUMsQ0FBQSxhQUFhLEVBQUMsTUFBRCxFQUFkLENBQXNCLEdBQXRCO1FBQ0EsR0FBRyxDQUFDLE9BQUosQ0FBQSxFQUpGOztBQUtBO0FBQUEsV0FBQSxxQ0FBQTs7UUFBQSxHQUFHLENBQUMsV0FBSixDQUFBO0FBQUE7YUFDQSxJQUFDLENBQUEsc0JBQUQsQ0FBQTtJQVJnQjs7eUJBVWxCLHNCQUFBLEdBQXdCLFNBQUE7TUFFdEIsSUFBRyxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQVosQ0FBZ0IsdUJBQWhCLENBQUEsSUFBNEMsSUFBQyxDQUFBLElBQUksQ0FBQyxRQUFOLENBQUEsQ0FBZ0IsQ0FBQyxNQUFqQixHQUEwQixDQUF6RTtlQUNFLElBQUMsQ0FBQSxPQUFPLENBQUMsU0FBUyxDQUFDLE1BQW5CLENBQTBCLFFBQTFCLEVBREY7T0FBQSxNQUFBO2VBR0UsSUFBQyxDQUFBLE9BQU8sQ0FBQyxTQUFTLENBQUMsR0FBbkIsQ0FBdUIsUUFBdkIsRUFIRjs7SUFGc0I7O3lCQU94QixPQUFBLEdBQVMsU0FBQTthQUNQLElBQUMsQ0FBQSxJQUFJLENBQUMsS0FBTixDQUFBO0lBRE87O3lCQUdULFVBQUEsR0FBWSxTQUFDLEtBQUQ7YUFDVixJQUFDLENBQUEsSUFBSyxDQUFBLEtBQUE7SUFESTs7eUJBR1osVUFBQSxHQUFZLFNBQUMsSUFBRDthQUNWLElBQUMsQ0FBQSxJQUFJLENBQUMsSUFBTixDQUFXLFNBQUMsQ0FBRDtlQUFPLENBQUMsQ0FBQyxJQUFGLEtBQVU7TUFBakIsQ0FBWDtJQURVOzt5QkFHWixZQUFBLEdBQWMsU0FBQyxPQUFEO0FBQ1osVUFBQTtNQUFBLElBQUcsaUJBQUEsSUFBYSxPQUFBLEtBQWEsSUFBQyxDQUFBLFNBQTlCOzthQUNZLENBQUUsT0FBTyxDQUFDLFNBQVMsQ0FBQyxNQUE5QixDQUFxQyxRQUFyQzs7UUFDQSxJQUFDLENBQUEsU0FBRCxHQUFhO1FBQ2IsSUFBQyxDQUFBLFNBQVMsQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLEdBQTdCLENBQWlDLFFBQWpDO2VBQ0EsSUFBQyxDQUFBLFNBQVMsQ0FBQyxPQUFPLENBQUMsY0FBbkIsQ0FBa0MsS0FBbEMsRUFKRjs7SUFEWTs7eUJBT2QsWUFBQSxHQUFjLFNBQUE7YUFDWixJQUFDLENBQUEsVUFBRCxDQUFZLElBQUMsQ0FBQSxJQUFJLENBQUMsYUFBTixDQUFBLENBQVo7SUFEWTs7eUJBR2QsZUFBQSxHQUFpQixTQUFBO2FBQ2YsSUFBQyxDQUFBLFlBQUQsQ0FBYyxJQUFDLENBQUEsVUFBRCxDQUFZLElBQUMsQ0FBQSxJQUFJLENBQUMsYUFBTixDQUFBLENBQVosQ0FBZDtJQURlOzt5QkFHakIsUUFBQSxHQUFVLFNBQUMsR0FBRDs7UUFDUixNQUFPLElBQUMsQ0FBQTs7TUFDUixJQUErQixXQUEvQjtlQUFBLElBQUMsQ0FBQSxJQUFJLENBQUMsV0FBTixDQUFrQixHQUFHLENBQUMsSUFBdEIsRUFBQTs7SUFGUTs7eUJBSVYsZUFBQSxHQUFpQixTQUFDLEdBQUQ7QUFDZixVQUFBOztRQUFBLE1BQU8sSUFBQyxDQUFBOztNQUNSLElBQUEsaUJBQU8sR0FBRyxDQUFFO01BQ1osSUFBYyxZQUFkO0FBQUEsZUFBQTs7TUFDQSxJQUFHLE9BQU8sSUFBSSxDQUFDLE1BQVosS0FBc0IsVUFBekI7UUFDRSxPQUFBLEdBQVUsSUFBSSxDQUFDLE1BQUwsQ0FBQSxFQURaO09BQUEsTUFFSyxJQUFHLE9BQU8sSUFBSSxDQUFDLE9BQVosS0FBdUIsVUFBMUI7UUFDSCxPQUFBLEdBQVUsSUFBSSxDQUFDLE9BQUwsQ0FBQSxFQURQO09BQUEsTUFFQSxJQUFHLE9BQU8sSUFBSSxDQUFDLE1BQVosS0FBc0IsVUFBekI7UUFDSCxPQUFBLEdBQVUsSUFBSSxDQUFDLE1BQUwsQ0FBQSxFQURQOztNQUVMLElBQWMsZUFBZDtBQUFBLGVBQUE7O01BQ0EsSUFBQyxDQUFBLFFBQUQsQ0FBVSxHQUFWO0FBQ0E7QUFBQSxXQUFBLHFDQUFBOztRQUFBLEdBQUcsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLFFBQWxCLEdBQTZCO0FBQTdCO01BQ0EsV0FBQSxHQUFjLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxRQUFiLENBQUEsQ0FBRCxFQUEwQixPQUExQixDQUFrQyxDQUFDLE1BQW5DLENBQTBDLENBQUMsU0FBQyxDQUFELEVBQUksQ0FBSjtlQUFVLENBQUMsQ0FBQyxNQUFGLENBQVMsQ0FBVDtNQUFWLENBQUQsQ0FBMUMsRUFBbUUsRUFBbkU7YUFDZCxJQUFJLENBQUMsSUFBTCxDQUFVO1FBQUMsV0FBQSxFQUFhLFdBQWQ7UUFBMkIsU0FBQSxFQUFXLElBQXRDO1FBQTRDLE9BQUEsRUFBUyxJQUFJLENBQUMsT0FBMUQ7UUFBbUUsUUFBQSxFQUFVLElBQUksQ0FBQyxRQUFsRjtPQUFWO0lBZGU7O3lCQWdCakIsUUFBQSxHQUFVLFNBQUMsRUFBRDtBQUNSLFVBQUE7TUFBQSxJQUFHLElBQUEsNkNBQXVCLENBQUUsYUFBNUI7UUFDRSxJQUFHLFVBQUEscUNBQWEsSUFBSSxDQUFDLGVBQXJCO2lCQUNFLElBQUMsQ0FBQSxJQUFLLENBQUEsRUFBQSxDQUFOLENBQVU7WUFBQSxLQUFBLEVBQU8sQ0FBQyxVQUFELENBQVA7V0FBVixFQURGO1NBREY7O0lBRFE7O3lCQUtWLGNBQUEsR0FBZ0IsU0FBQyxNQUFEO0FBQ2QsVUFBQTtNQUFBLElBQUEsR0FBTyxJQUFDLENBQUEsT0FBRCxDQUFBOztRQUNQLFNBQVUsSUFBQyxDQUFBOztNQUNYLElBQWMsY0FBZDtBQUFBLGVBQUE7O0FBQ0E7V0FBQSxzQ0FBQTs7WUFBbUMsR0FBQSxLQUFTO3VCQUE1QyxJQUFDLENBQUEsUUFBRCxDQUFVLEdBQVY7O0FBQUE7O0lBSmM7O3lCQU1oQixnQkFBQSxHQUFrQixTQUFDLE1BQUQ7QUFDaEIsVUFBQTtNQUFBLElBQUEsR0FBTyxJQUFDLENBQUEsT0FBRCxDQUFBOztRQUNQLFNBQVUsSUFBQyxDQUFBOztNQUNYLEtBQUEsR0FBUSxJQUFJLENBQUMsT0FBTCxDQUFhLE1BQWI7TUFDUixJQUFVLEtBQUEsS0FBUyxDQUFDLENBQXBCO0FBQUEsZUFBQTs7QUFDQTtXQUFBLDhDQUFBOztZQUFzQyxDQUFBLEdBQUk7dUJBQTFDLElBQUMsQ0FBQSxRQUFELENBQVUsR0FBVjs7QUFBQTs7SUFMZ0I7O3lCQU9sQixlQUFBLEdBQWlCLFNBQUMsTUFBRDtBQUNmLFVBQUE7TUFBQSxJQUFBLEdBQU8sSUFBQyxDQUFBLE9BQUQsQ0FBQTs7UUFDUCxTQUFVLElBQUMsQ0FBQTs7TUFDWCxLQUFBLEdBQVEsSUFBSSxDQUFDLE9BQUwsQ0FBYSxNQUFiO01BQ1IsSUFBVSxLQUFBLEtBQVMsQ0FBQyxDQUFwQjtBQUFBLGVBQUE7O0FBQ0E7V0FBQSw4Q0FBQTs7WUFBc0MsQ0FBQSxHQUFJO3VCQUExQyxJQUFDLENBQUEsUUFBRCxDQUFVLEdBQVY7O0FBQUE7O0lBTGU7O3lCQU9qQixjQUFBLEdBQWdCLFNBQUE7QUFDZCxVQUFBO0FBQUE7QUFBQTtXQUFBLHFDQUFBOztRQUNFLElBQUEsMkRBQThCLENBQUMsc0JBQS9CO3VCQUFBLElBQUMsQ0FBQSxRQUFELENBQVUsR0FBVixHQUFBO1NBQUEsTUFBQTsrQkFBQTs7QUFERjs7SUFEYzs7eUJBSWhCLFlBQUEsR0FBYyxTQUFBO0FBQ1osVUFBQTtBQUFBO0FBQUE7V0FBQSxxQ0FBQTs7cUJBQUEsSUFBQyxDQUFBLFFBQUQsQ0FBVSxHQUFWO0FBQUE7O0lBRFk7O3lCQUdkLFdBQUEsR0FBYSxTQUFBO3FDQUNYLElBQUMsQ0FBQSxXQUFELElBQUMsQ0FBQSxXQUFZLElBQUksQ0FBQyxnQkFBTCxDQUFBLENBQXVCLENBQUM7SUFEMUI7O3lCQUdiLFdBQUEsR0FBYSxTQUFDLEtBQUQ7QUFDWCxVQUFBO01BQUEsSUFBQyxDQUFBLFVBQUQsR0FBYyxJQUFDLENBQUEsYUFBRCxDQUFlLEtBQUssQ0FBQyxNQUFyQjtNQUNkLElBQUEsQ0FBYyxJQUFDLENBQUEsVUFBZjtBQUFBLGVBQUE7O01BQ0EsSUFBQyxDQUFBLG1CQUFELEdBQXVCO01BRXZCLEtBQUssQ0FBQyxZQUFZLENBQUMsT0FBbkIsQ0FBMkIsZ0JBQTNCLEVBQTZDLE1BQTdDO01BRUEsSUFBQyxDQUFBLFVBQVUsQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLEdBQTlCLENBQWtDLGFBQWxDO01BQ0EsSUFBQyxDQUFBLFVBQVUsQ0FBQyxjQUFaLENBQUE7TUFFQSxRQUFBLEdBQVcsSUFBQyxDQUFBLElBQUksQ0FBQyxPQUFOLENBQWMsSUFBQyxDQUFBLFVBQWY7TUFDWCxLQUFLLENBQUMsWUFBWSxDQUFDLE9BQW5CLENBQTJCLGdCQUEzQixFQUE2QyxRQUE3QztNQUVBLFNBQUEsR0FBWSxJQUFDLENBQUEsYUFBYSxDQUFDLFFBQWYsQ0FBQSxDQUF5QixDQUFDLE9BQTFCLENBQWtDLElBQUMsQ0FBQSxJQUFuQztNQUNaLEtBQUssQ0FBQyxZQUFZLENBQUMsT0FBbkIsQ0FBMkIsaUJBQTNCLEVBQThDLFNBQTlDO01BQ0EsS0FBSyxDQUFDLFlBQVksQ0FBQyxPQUFuQixDQUEyQixjQUEzQixFQUEyQyxJQUFDLENBQUEsSUFBSSxDQUFDLEVBQWpEO01BQ0EsS0FBSyxDQUFDLFlBQVksQ0FBQyxPQUFuQixDQUEyQixnQkFBM0IsRUFBNkMsSUFBQyxDQUFBLFdBQUQsQ0FBQSxDQUE3QztNQUVBLElBQUEsR0FBTyxJQUFDLENBQUEsSUFBSSxDQUFDLFFBQU4sQ0FBQSxDQUFpQixDQUFBLElBQUMsQ0FBQSxJQUFJLENBQUMsT0FBTixDQUFjLElBQUMsQ0FBQSxVQUFmLENBQUE7TUFDeEIsSUFBYyxZQUFkO0FBQUEsZUFBQTs7TUFFQSxJQUFHLE9BQU8sSUFBSSxDQUFDLE1BQVosS0FBc0IsVUFBekI7UUFDRSxPQUFBLHlDQUEwQixHQUQ1QjtPQUFBLE1BRUssSUFBRyxPQUFPLElBQUksQ0FBQyxPQUFaLEtBQXVCLFVBQTFCO1FBQ0gsT0FBQSw0Q0FBMkIsR0FEeEI7T0FBQSxNQUVBLElBQUcsT0FBTyxJQUFJLENBQUMsTUFBWixLQUFzQixVQUF6QjtRQUNILE9BQUEsMkNBQTBCLEdBRHZCOztNQUdMLElBQUcsT0FBTyxJQUFJLENBQUMsbUJBQVosS0FBbUMsVUFBdEM7QUFDRTtBQUFBLGFBQUEsc0NBQUE7O1VBQ0UsS0FBSyxDQUFDLFlBQVksQ0FBQyxPQUFuQixDQUEyQixtQkFBQSxHQUFvQixRQUEvQyxFQUEyRCxNQUEzRDtBQURGLFNBREY7T0FBQSxNQUFBO1FBSUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxPQUFuQixDQUEyQixxQkFBM0IsRUFBa0QsTUFBbEQsRUFKRjs7TUFNQSxJQUFHLGVBQUg7UUFDRSxLQUFLLENBQUMsWUFBWSxDQUFDLE9BQW5CLENBQTJCLFlBQTNCLEVBQXlDLE9BQXpDO1FBRUEsSUFBRyxPQUFPLENBQUMsUUFBUixLQUFvQixRQUF2QjtVQUNFLElBQUEsQ0FBcUMsSUFBQyxDQUFBLGNBQUQsQ0FBZ0IsT0FBaEIsQ0FBckM7WUFBQSxPQUFBLEdBQVUsU0FBQSxHQUFVLFFBQXBCOztVQUNBLEtBQUssQ0FBQyxZQUFZLENBQUMsT0FBbkIsQ0FBMkIsZUFBM0IsRUFBNEMsT0FBNUMsRUFGRjs7UUFJQSw2Q0FBRyxJQUFJLENBQUMsc0JBQUwsSUFBdUIsc0JBQTFCO1VBQ0UsS0FBSyxDQUFDLFlBQVksQ0FBQyxPQUFuQixDQUEyQixxQkFBM0IsRUFBa0QsTUFBbEQ7aUJBQ0EsS0FBSyxDQUFDLFlBQVksQ0FBQyxPQUFuQixDQUEyQixlQUEzQixFQUE0QyxJQUFJLENBQUMsT0FBTCxDQUFBLENBQTVDLEVBRkY7U0FQRjs7SUFsQ1c7O3lCQTZDYixjQUFBLEdBQWdCLFNBQUMsR0FBRDtBQUNkLFVBQUE7QUFBQTtlQUNFLDJDQURGO09BQUEsY0FBQTtRQUVNO2VBQ0osTUFIRjs7SUFEYzs7eUJBTWhCLFdBQUEsR0FBYSxTQUFDLEtBQUQ7QUFFWCxVQUFBO01BQUEsSUFBQSxDQUFPLEtBQUssQ0FBQyxhQUFhLENBQUMsUUFBcEIsQ0FBNkIsS0FBSyxDQUFDLGFBQW5DLENBQVA7UUFDRSxJQUFDLENBQUEsaUJBQUQsQ0FBQTtRQUNBLElBQUMsQ0FBQSxtQkFBRCxHQUF1QjtBQUN2QjtBQUFBO2FBQUEscUNBQUE7O3VCQUFBLEdBQUcsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLFFBQWxCLEdBQTZCO0FBQTdCO3VCQUhGOztJQUZXOzt5QkFPYixTQUFBLEdBQVcsU0FBQyxLQUFEO01BQ1QsSUFBQSxDQUFjLElBQUMsQ0FBQSxhQUFELENBQWUsS0FBSyxDQUFDLE1BQXJCLENBQWQ7QUFBQSxlQUFBOzthQUVBLElBQUMsQ0FBQSxlQUFELENBQUE7SUFIUzs7eUJBS1gsVUFBQSxHQUFZLFNBQUMsS0FBRDtBQUNWLFVBQUE7TUFBQSxJQUFBLENBQWMsSUFBQyxDQUFBLGNBQUQsQ0FBZ0IsS0FBaEIsQ0FBZDtBQUFBLGVBQUE7O01BQ0EsSUFBQSxDQUFjLElBQUMsQ0FBQSxhQUFELENBQWUsS0FBZixFQUFzQixJQUFDLENBQUEsUUFBdkIsQ0FBZDtBQUFBLGVBQUE7O01BRUEsS0FBSyxDQUFDLGNBQU4sQ0FBQTtNQUNBLEtBQUssQ0FBQyxlQUFOLENBQUE7TUFFQSxrQkFBQSxHQUFxQixJQUFDLENBQUEsa0JBQUQsQ0FBb0IsS0FBcEI7TUFDckIsSUFBYywwQkFBZDtBQUFBLGVBQUE7O01BQ0EsSUFBVSxJQUFDLENBQUEsbUJBQUQsS0FBd0Isa0JBQWxDO0FBQUEsZUFBQTs7TUFDQSxJQUFDLENBQUEsbUJBQUQsR0FBdUI7TUFFdkIsSUFBQyxDQUFBLHVCQUFELENBQUE7TUFFQSxJQUFBLEdBQU8sSUFBQyxDQUFBLE9BQUQsQ0FBQTtNQUNQLFdBQUEsR0FBYyxJQUFDLENBQUEsY0FBRCxDQUFBO01BQ2QsSUFBYyxtQkFBZDtBQUFBLGVBQUE7O01BRUEsSUFBRyxrQkFBQSxHQUFxQixJQUFJLENBQUMsTUFBN0I7UUFDRSxHQUFBLEdBQU0sSUFBSyxDQUFBLGtCQUFBO1FBQ1gsR0FBRyxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsR0FBdEIsQ0FBMEIsZ0JBQTFCO2VBQ0EsR0FBRyxDQUFDLE9BQU8sQ0FBQyxhQUFhLENBQUMsWUFBMUIsQ0FBdUMsV0FBdkMsRUFBb0QsR0FBRyxDQUFDLE9BQXhELEVBSEY7T0FBQSxNQUFBO1FBS0UsSUFBRyxHQUFBLEdBQU0sSUFBSyxDQUFBLGtCQUFBLEdBQXFCLENBQXJCLENBQWQ7VUFDRSxHQUFHLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxHQUF0QixDQUEwQixzQkFBMUI7VUFDQSxJQUFHLE9BQUEsR0FBVSxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQXpCO21CQUNFLEdBQUcsQ0FBQyxPQUFPLENBQUMsYUFBYSxDQUFDLFlBQTFCLENBQXVDLFdBQXZDLEVBQW9ELE9BQXBELEVBREY7V0FBQSxNQUFBO21CQUdFLEdBQUcsQ0FBQyxPQUFPLENBQUMsYUFBYSxDQUFDLFdBQTFCLENBQXNDLFdBQXRDLEVBSEY7V0FGRjtTQUxGOztJQWxCVTs7eUJBOEJaLG1CQUFBLEdBQXFCLFNBQUMsS0FBRCxFQUFRLFVBQVIsRUFBb0IsYUFBcEI7QUFDbkIsVUFBQTtNQUFBLElBQUcsSUFBQyxDQUFBLElBQUksQ0FBQyxFQUFOLEtBQVksVUFBZjtRQUNFLElBQUcsWUFBQSxHQUFlLElBQUMsQ0FBQSxJQUFJLENBQUMsUUFBTixDQUFBLENBQWlCLENBQUEsYUFBQSxDQUFuQztVQUNFLElBQUMsQ0FBQSxJQUFJLENBQUMsV0FBTixDQUFrQixZQUFsQixFQURGO1NBREY7O2FBSUEsSUFBQyxDQUFBLGVBQUQsQ0FBQTtJQUxtQjs7eUJBT3JCLGVBQUEsR0FBaUIsU0FBQTtBQUNmLFVBQUE7O1dBQVcsQ0FBRSxPQUFPLENBQUMsU0FBUyxDQUFDLE1BQS9CLENBQXNDLGFBQXRDOzs7WUFDVyxDQUFFLGFBQWIsQ0FBQTs7TUFDQSxJQUFDLENBQUEsVUFBRCxHQUFjO01BQ2QsSUFBQyxDQUFBLHVCQUFELENBQUE7YUFDQSxJQUFDLENBQUEsaUJBQUQsQ0FBQTtJQUxlOzt5QkFPakIsTUFBQSxHQUFRLFNBQUMsS0FBRDtBQUNOLFVBQUE7TUFBQSxJQUFBLENBQWMsSUFBQyxDQUFBLGNBQUQsQ0FBZ0IsS0FBaEIsQ0FBZDtBQUFBLGVBQUE7O01BRUEsS0FBSyxDQUFDLGNBQU4sQ0FBQTtNQUVBLFlBQUEsR0FBZ0IsUUFBQSxDQUFTLEtBQUssQ0FBQyxZQUFZLENBQUMsT0FBbkIsQ0FBMkIsZ0JBQTNCLENBQVQ7TUFDaEIsVUFBQSxHQUFnQixRQUFBLENBQVMsS0FBSyxDQUFDLFlBQVksQ0FBQyxPQUFuQixDQUEyQixjQUEzQixDQUFUO01BQ2hCLFNBQUEsR0FBZ0IsUUFBQSxDQUFTLEtBQUssQ0FBQyxZQUFZLENBQUMsT0FBbkIsQ0FBMkIsZ0JBQTNCLENBQVQ7TUFDaEIsYUFBQSxHQUFnQixRQUFBLENBQVMsS0FBSyxDQUFDLFlBQVksQ0FBQyxPQUFuQixDQUEyQixpQkFBM0IsQ0FBVDtNQUVoQixpQkFBQSxHQUFvQixLQUFLLENBQUMsWUFBWSxDQUFDLE9BQW5CLENBQTJCLHFCQUEzQixDQUFBLEtBQXFEO01BQ3pFLFlBQUEsR0FBZSxLQUFLLENBQUMsWUFBWSxDQUFDLE9BQW5CLENBQTJCLGVBQTNCO01BRWYsT0FBQSxHQUFVLElBQUMsQ0FBQSxrQkFBRCxDQUFvQixLQUFwQjtNQUNWLE1BQUEsR0FBUyxJQUFDLENBQUE7TUFFVixJQUFDLENBQUEsZUFBRCxDQUFBO01BRUEsSUFBQSxDQUFjLElBQUMsQ0FBQSxhQUFELENBQWUsS0FBZixFQUFzQixJQUFDLENBQUEsUUFBdkIsQ0FBZDtBQUFBLGVBQUE7O01BRUEsSUFBRyxZQUFBLEtBQWdCLElBQUMsQ0FBQSxXQUFELENBQUEsQ0FBbkI7UUFDRSxRQUFBLEdBQVcsSUFBQyxDQUFBLGFBQWEsQ0FBQyxRQUFmLENBQUEsQ0FBMEIsQ0FBQSxhQUFBO1FBQ3JDLHdCQUFHLFFBQVEsQ0FBRSxZQUFWLEtBQWtCLFVBQXJCO1VBR0UsUUFBQSxHQUFXLEtBQUssQ0FBQyxJQUFOLENBQVcsUUFBUSxDQUFDLGdCQUFULENBQTBCLFdBQTFCLENBQVgsQ0FDVCxDQUFDLEdBRFEsQ0FDSixTQUFDLE1BQUQ7bUJBQVksTUFBTSxDQUFDO1VBQW5CLENBREksQ0FFVCxDQUFDLElBRlEsQ0FFSCxTQUFDLElBQUQ7bUJBQVUsSUFBSSxDQUFDLEVBQUwsS0FBVztVQUFyQixDQUZHLEVBSGI7O1FBTUEsSUFBQSxHQUFPLFFBQVEsQ0FBQyxRQUFULENBQUEsQ0FBb0IsQ0FBQSxTQUFBO1FBQzNCLElBQXFFLFlBQXJFO2lCQUFBLElBQUMsQ0FBQSxvQkFBRCxDQUFzQixRQUF0QixFQUFnQyxTQUFoQyxFQUEyQyxNQUEzQyxFQUFtRCxPQUFuRCxFQUE0RCxJQUE1RCxFQUFBO1NBVEY7T0FBQSxNQUFBO1FBV0UsVUFBQSxHQUFhLEtBQUssQ0FBQyxZQUFZLENBQUMsT0FBbkIsQ0FBMkIsWUFBM0I7UUFDYixJQUFJLENBQUMsU0FBUyxDQUFDLElBQWYsQ0FBb0IsVUFBcEIsQ0FBK0IsQ0FBQyxJQUFoQyxDQUFxQyxDQUFBLFNBQUEsS0FBQTtpQkFBQSxTQUFDLElBQUQ7QUFHbkMsZ0JBQUE7WUFBQSxVQUFBLEdBQWEsSUFBSSxDQUFDLFNBQVMsQ0FBQyxhQUFmLENBQUE7WUFDYixlQUFBLEdBQWtCLFVBQVUsQ0FBQyxRQUFYLENBQUEsQ0FBcUIsQ0FBQyxPQUF0QixDQUE4QixJQUE5QjtZQUNsQixLQUFDLENBQUEsb0JBQUQsQ0FBc0IsVUFBdEIsRUFBa0MsZUFBbEMsRUFBbUQsTUFBbkQsRUFBMkQsT0FBM0QsRUFBb0UsSUFBcEU7WUFDQSxJQUErQixpQkFBL0I7O2dCQUFBLElBQUksQ0FBQyxRQUFTO2VBQWQ7O1lBRUEsSUFBRyxDQUFJLEtBQUEsQ0FBTSxZQUFOLENBQVA7Y0FFRSxhQUFBLEdBQWdCLEtBQUMsQ0FBQSxrQkFBRCxDQUFvQixZQUFwQjs2Q0FDaEIsYUFBYSxDQUFFLFdBQVcsQ0FBQyxJQUEzQixDQUFnQyxhQUFoQyxFQUErQyxVQUEvQyxFQUEyRCxTQUEzRCxXQUhGOztVQVJtQztRQUFBLENBQUEsQ0FBQSxDQUFBLElBQUEsQ0FBckM7ZUFhQSxJQUFJLENBQUMsS0FBTCxDQUFBLEVBekJGOztJQXBCTTs7eUJBZ0RSLGVBQUEsR0FBaUIsU0FBQyxLQUFEO01BQ2YsSUFBQSxDQUFjLElBQUMsQ0FBQSxjQUFELENBQWdCLEtBQWhCLENBQWQ7QUFBQSxlQUFBOztNQUNBLElBQUEsQ0FBYyxJQUFDLENBQUEsYUFBRCxDQUFlLEtBQWYsRUFBc0IsSUFBQyxDQUFBLFFBQXZCLENBQWQ7QUFBQSxlQUFBOztNQUNBLElBQVUsSUFBQyxDQUFBLElBQUksQ0FBQyxRQUFOLENBQUEsQ0FBZ0IsQ0FBQyxNQUFqQixHQUEwQixDQUExQixJQUErQixJQUFJLENBQUMsTUFBTSxDQUFDLEdBQVosQ0FBZ0IsdUJBQWhCLENBQXpDO0FBQUEsZUFBQTs7TUFDQSxJQUFHLElBQUMsQ0FBQSxXQUFXLENBQUMsUUFBYixDQUFzQixLQUFLLENBQUMsYUFBNUIsQ0FBSDtlQUNFLElBQUMsQ0FBQSxPQUFPLENBQUMsU0FBUyxDQUFDLE1BQW5CLENBQTBCLFFBQTFCLEVBREY7O0lBSmU7O3lCQVFqQixlQUFBLEdBQWlCLFNBQUMsS0FBRDtNQUNmLElBQUEsQ0FBYyxJQUFDLENBQUEsY0FBRCxDQUFnQixLQUFoQixDQUFkO0FBQUEsZUFBQTs7TUFDQSxJQUFBLENBQWMsSUFBQyxDQUFBLGFBQUQsQ0FBZSxLQUFmLEVBQXNCLElBQUMsQ0FBQSxRQUF2QixDQUFkO0FBQUEsZUFBQTs7TUFDQSxJQUFVLElBQUMsQ0FBQSxJQUFJLENBQUMsUUFBTixDQUFBLENBQWdCLENBQUMsTUFBakIsR0FBMEIsQ0FBMUIsSUFBK0IsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFaLENBQWdCLHVCQUFoQixDQUF6QztBQUFBLGVBQUE7O01BQ0EsSUFBQSxDQUFPLElBQUMsQ0FBQSxXQUFXLENBQUMsUUFBYixDQUFzQixLQUFLLENBQUMsYUFBNUIsQ0FBUDtlQUNFLElBQUMsQ0FBQSxPQUFPLENBQUMsU0FBUyxDQUFDLEdBQW5CLENBQXVCLFFBQXZCLEVBREY7O0lBSmU7O3lCQU9qQixZQUFBLEdBQWMsU0FBQyxLQUFEO01BQ1osSUFBVSxLQUFLLENBQUMsUUFBTixJQUFrQixDQUFJLElBQUMsQ0FBQSxZQUFqQztBQUFBLGVBQUE7OztRQUVBLElBQUMsQ0FBQSxhQUFjOztNQUNmLElBQUMsQ0FBQSxVQUFELElBQWUsS0FBSyxDQUFDO01BRXJCLElBQUcsSUFBQyxDQUFBLFVBQUQsSUFBZSxDQUFDLElBQUMsQ0FBQSxxQkFBcEI7UUFDRSxJQUFDLENBQUEsVUFBRCxHQUFjO2VBQ2QsSUFBQyxDQUFBLElBQUksQ0FBQyxnQkFBTixDQUFBLEVBRkY7T0FBQSxNQUdLLElBQUcsSUFBQyxDQUFBLFVBQUQsSUFBZSxJQUFDLENBQUEscUJBQW5CO1FBQ0gsSUFBQyxDQUFBLFVBQUQsR0FBYztlQUNkLElBQUMsQ0FBQSxJQUFJLENBQUMsb0JBQU4sQ0FBQSxFQUZHOztJQVRPOzt5QkFhZCxXQUFBLEdBQWEsU0FBQyxLQUFEO0FBQ1gsVUFBQTtNQUFBLEdBQUEsR0FBTSxJQUFDLENBQUEsYUFBRCxDQUFlLEtBQUssQ0FBQyxNQUFyQjtNQUNOLElBQUEsQ0FBYyxHQUFkO0FBQUEsZUFBQTs7TUFFQSxJQUFHLEtBQUssQ0FBQyxLQUFOLEtBQWUsQ0FBZixJQUFvQixDQUFDLEtBQUssQ0FBQyxLQUFOLEtBQWUsQ0FBZixJQUFxQixLQUFLLENBQUMsT0FBTixLQUFpQixJQUF2QyxDQUF2Qjs7YUFDa0IsQ0FBRSxPQUFPLENBQUMsU0FBUyxDQUFDLE1BQXBDLENBQTJDLGVBQTNDOztRQUNBLElBQUMsQ0FBQSxlQUFELEdBQW1CO1FBQ25CLElBQUMsQ0FBQSxlQUFlLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxHQUFuQyxDQUF1QyxlQUF2QztlQUNBLEtBQUssQ0FBQyxjQUFOLENBQUEsRUFKRjtPQUFBLE1BS0ssSUFBRyxLQUFLLENBQUMsS0FBTixLQUFlLENBQWxCO2VBR0gsS0FBSyxDQUFDLGNBQU4sQ0FBQSxFQUhHOztJQVRNOzt5QkFjYixPQUFBLEdBQVMsU0FBQyxLQUFEO0FBQ1AsVUFBQTtNQUFBLEdBQUEsR0FBTSxJQUFDLENBQUEsYUFBRCxDQUFlLEtBQUssQ0FBQyxNQUFyQjtNQUNOLElBQUEsQ0FBYyxHQUFkO0FBQUEsZUFBQTs7TUFFQSxLQUFLLENBQUMsY0FBTixDQUFBO01BQ0EsSUFBRyxLQUFLLENBQUMsS0FBTixLQUFlLENBQWYsSUFBb0IsQ0FBQyxLQUFLLENBQUMsS0FBTixLQUFlLENBQWYsSUFBcUIsS0FBSyxDQUFDLE9BQU4sS0FBaUIsSUFBdkMsQ0FBdkI7QUFBQTtPQUFBLE1BSUssSUFBRyxLQUFLLENBQUMsS0FBTixLQUFlLENBQWYsSUFBcUIsQ0FBSSxLQUFLLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxRQUF2QixDQUFnQyxZQUFoQyxDQUE1QjtRQUNILElBQUMsQ0FBQSxJQUFJLENBQUMsWUFBTixDQUFtQixHQUFHLENBQUMsSUFBdkI7UUFDQSxJQUFBLENBQXdCLElBQUMsQ0FBQSxJQUFJLENBQUMsV0FBTixDQUFBLENBQXhCO2lCQUFBLElBQUMsQ0FBQSxJQUFJLENBQUMsUUFBTixDQUFBLEVBQUE7U0FGRztPQUFBLE1BR0EsSUFBRyxLQUFLLENBQUMsS0FBTixLQUFlLENBQWxCO2VBQ0gsSUFBQyxDQUFBLElBQUksQ0FBQyxXQUFOLENBQWtCLEdBQUcsQ0FBQyxJQUF0QixFQURHOztJQVpFOzt5QkFlVCxhQUFBLEdBQWUsU0FBQyxLQUFEO0FBQ2IsVUFBQTtNQUFBLElBQUcsR0FBQSxHQUFNLElBQUMsQ0FBQSxhQUFELENBQWUsS0FBSyxDQUFDLE1BQXJCLENBQVQ7bUZBQ1UsQ0FBQyxpQ0FEWDtPQUFBLE1BRUssSUFBRyxLQUFLLENBQUMsTUFBTixLQUFnQixJQUFDLENBQUEsT0FBcEI7UUFDSCxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQWQsQ0FBdUIsSUFBQyxDQUFBLE9BQXhCLEVBQWlDLHNCQUFqQztlQUNBLEtBQUssQ0FBQyxjQUFOLENBQUEsRUFGRzs7SUFIUTs7eUJBT2YsMkJBQUEsR0FBNkIsU0FBQyxLQUFEO2FBQzNCLElBQUMsQ0FBQSxxQkFBRCxHQUF5QjtJQURFOzt5QkFHN0Isa0JBQUEsR0FBb0IsU0FBQyxLQUFEO01BQ2xCLElBQUcsS0FBQSxLQUFTLFVBQVo7ZUFDRSxJQUFDLENBQUEsWUFBRCxHQUFpQixPQUFPLENBQUMsUUFBUixLQUFvQixRQUR2QztPQUFBLE1BQUE7ZUFHRSxJQUFDLENBQUEsWUFBRCxHQUFnQixNQUhsQjs7SUFEa0I7O3lCQU1wQixrQkFBQSxHQUFvQixTQUFDLEVBQUQ7O1FBQ2xCLGdCQUFpQixPQUFBLENBQVEsVUFBUixDQUFtQixDQUFDLE1BQU0sQ0FBQzs7YUFFNUMsYUFBYSxDQUFDLE1BQWQsQ0FBcUIsRUFBckI7SUFIa0I7O3lCQUtwQixvQkFBQSxHQUFzQixTQUFDLFFBQUQsRUFBVyxTQUFYLEVBQXNCLE1BQXRCLEVBQThCLE9BQTlCLEVBQXVDLElBQXZDO0FBQ3BCO1FBQ0UsSUFBRyxNQUFBLEtBQVUsUUFBYjtVQUNFLElBQWEsU0FBQSxHQUFZLE9BQXpCO1lBQUEsT0FBQSxHQUFBOztVQUNBLE1BQU0sQ0FBQyxRQUFQLENBQWdCLElBQWhCLEVBQXNCLE9BQXRCLEVBRkY7U0FBQSxNQUFBO1VBSUUsSUFBQyxDQUFBLHdCQUFELEdBQTRCO1VBQzVCLFFBQVEsQ0FBQyxjQUFULENBQXdCLElBQXhCLEVBQThCLE1BQTlCLEVBQXNDLE9BQUEsRUFBdEMsRUFMRjs7UUFNQSxNQUFNLENBQUMsWUFBUCxDQUFvQixJQUFwQjtlQUNBLE1BQU0sQ0FBQyxRQUFQLENBQUEsRUFSRjtPQUFBO1FBVUUsSUFBQyxDQUFBLHdCQUFELEdBQTRCLE1BVjlCOztJQURvQjs7eUJBYXRCLHVCQUFBLEdBQXlCLFNBQUE7QUFDdkIsVUFBQTtNQUFBLGdCQUFBLEdBQW1CLElBQUksQ0FBQyxTQUFTLENBQUMsVUFBZixDQUFBO0FBQ25CO0FBQUEsV0FBQSxxQ0FBQTs7UUFDRSxVQUFVLENBQUMsU0FBUyxDQUFDLE1BQXJCLENBQTRCLGdCQUE1QjtBQURGO0FBR0E7QUFBQTtXQUFBLHdDQUFBOztxQkFDRSxVQUFVLENBQUMsU0FBUyxDQUFDLE1BQXJCLENBQTRCLHNCQUE1QjtBQURGOztJQUx1Qjs7eUJBUXpCLGtCQUFBLEdBQW9CLFNBQUMsS0FBRDtBQUNsQixVQUFBO01BQUEsTUFBQSxHQUFTLEtBQUssQ0FBQztNQUVmLElBQVUsSUFBQyxDQUFBLGFBQUQsQ0FBZSxNQUFmLENBQVY7QUFBQSxlQUFBOztNQUVBLElBQUEsR0FBTyxJQUFDLENBQUEsT0FBRCxDQUFBO01BQ1AsR0FBQSxHQUFNLElBQUMsQ0FBQSxhQUFELENBQWUsTUFBZjs7UUFDTixNQUFPLElBQUssQ0FBQSxJQUFJLENBQUMsTUFBTCxHQUFjLENBQWQ7O01BRVosSUFBZ0IsV0FBaEI7QUFBQSxlQUFPLEVBQVA7O01BRUEsTUFBZ0IsR0FBRyxDQUFDLE9BQU8sQ0FBQyxxQkFBWixDQUFBLENBQWhCLEVBQUMsZUFBRCxFQUFPO01BQ1AsYUFBQSxHQUFnQixJQUFBLEdBQU8sS0FBQSxHQUFRO01BQy9CLFlBQUEsR0FBZSxJQUFJLENBQUMsT0FBTCxDQUFhLEdBQWI7TUFFZixJQUFHLEtBQUssQ0FBQyxLQUFOLEdBQWMsYUFBakI7ZUFDRSxhQURGO09BQUEsTUFBQTtlQUdFLFlBQUEsR0FBZSxFQUhqQjs7SUFma0I7O3lCQW9CcEIsY0FBQSxHQUFnQixTQUFBO01BQ2QsSUFBeUIsMEJBQXpCO0FBQUEsZUFBTyxJQUFDLENBQUEsY0FBUjs7TUFFQSxJQUFDLENBQUEsYUFBRCxHQUFpQixRQUFRLENBQUMsYUFBVCxDQUF1QixJQUF2QjtNQUNqQixJQUFDLENBQUEsYUFBYSxDQUFDLFNBQVMsQ0FBQyxHQUF6QixDQUE2QixhQUE3QjthQUNBLElBQUMsQ0FBQTtJQUxhOzt5QkFPaEIsaUJBQUEsR0FBbUIsU0FBQTtBQUNqQixVQUFBOztXQUFjLENBQUUsTUFBaEIsQ0FBQTs7YUFDQSxJQUFDLENBQUEsYUFBRCxHQUFpQjtJQUZBOzt5QkFJbkIsYUFBQSxHQUFlLFNBQUMsT0FBRDthQUNiLE9BQU8sQ0FBQyxTQUFTLENBQUMsUUFBbEIsQ0FBMkIsYUFBM0I7SUFEYTs7eUJBR2YsWUFBQSxHQUFjLFNBQUE7QUFDWixVQUFBO0FBQUE7QUFBQSxXQUFBLHFDQUFBOztRQUNHLFFBQVMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxxQkFBWixDQUFBO1FBQ1YsR0FBRyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsUUFBbEIsR0FBNkIsS0FBSyxDQUFDLE9BQU4sQ0FBYyxDQUFkLENBQUEsR0FBbUI7QUFGbEQ7SUFEWTs7eUJBTWQsWUFBQSxHQUFjLFNBQUE7QUFDWixVQUFBO0FBQUE7QUFBQSxXQUFBLHFDQUFBOztRQUFBLEdBQUcsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLFFBQWxCLEdBQTZCO0FBQTdCO0lBRFk7O3lCQUlkLGFBQUEsR0FBZSxTQUFDLE9BQUQ7QUFDYixVQUFBO01BQUEsY0FBQSxHQUFpQjtBQUNqQixhQUFNLHNCQUFOO1FBQ0UsSUFBRyxHQUFBLEdBQU0sSUFBQyxDQUFBLGFBQWEsQ0FBQyxHQUFmLENBQW1CLGNBQW5CLENBQVQ7QUFDRSxpQkFBTyxJQURUO1NBQUEsTUFBQTtVQUdFLGNBQUEsR0FBaUIsY0FBYyxDQUFDLGNBSGxDOztNQURGO0lBRmE7O3lCQVFmLGNBQUEsR0FBZ0IsU0FBQyxLQUFEO0FBQ2QsVUFBQTtBQUFBO0FBQUEsV0FBQSxxQ0FBQTs7UUFDRSxJQUFHLElBQUksQ0FBQyxJQUFMLEtBQWEsZ0JBQWhCO0FBQ0UsaUJBQU8sS0FEVDs7QUFERjtBQUlBLGFBQU87SUFMTzs7eUJBT2hCLGFBQUEsR0FBZSxTQUFDLEtBQUQsRUFBUSxRQUFSO0FBQ2IsVUFBQTtBQUFBO0FBQUEsV0FBQSxxQ0FBQTs7UUFDRSxJQUFHLElBQUksQ0FBQyxJQUFMLEtBQWEscUJBQWIsSUFBc0MsSUFBSSxDQUFDLElBQUwsS0FBYSxDQUFBLG1CQUFBLEdBQW9CLFFBQXBCLENBQXREO0FBQ0UsaUJBQU8sS0FEVDs7QUFERjtBQUlBLGFBQU87SUFMTTs7Ozs7QUF6aUJqQiIsInNvdXJjZXNDb250ZW50IjpbIkJyb3dzZXJXaW5kb3cgPSBudWxsICMgRGVmZXIgcmVxdWlyZSB1bnRpbCBhY3R1YWxseSB1c2VkXG57aXBjUmVuZGVyZXJ9ID0gcmVxdWlyZSAnZWxlY3Ryb24nXG5cbntDb21wb3NpdGVEaXNwb3NhYmxlfSA9IHJlcXVpcmUgJ2F0b20nXG5UYWJWaWV3ID0gcmVxdWlyZSAnLi90YWItdmlldydcblxubW9kdWxlLmV4cG9ydHMgPVxuY2xhc3MgVGFiQmFyVmlld1xuICBjb25zdHJ1Y3RvcjogKEBwYW5lLCBAbG9jYXRpb24pIC0+XG4gICAgQGVsZW1lbnQgPSBkb2N1bWVudC5jcmVhdGVFbGVtZW50KCd1bCcpXG4gICAgQGVsZW1lbnQuY2xhc3NMaXN0LmFkZChcImxpc3QtaW5saW5lXCIpXG4gICAgQGVsZW1lbnQuY2xhc3NMaXN0LmFkZChcInRhYi1iYXJcIilcbiAgICBAZWxlbWVudC5jbGFzc0xpc3QuYWRkKFwiaW5zZXQtcGFuZWxcIilcbiAgICBAZWxlbWVudC5zZXRBdHRyaWJ1dGUoJ2lzJywgJ2F0b20tdGFicycpXG4gICAgQGVsZW1lbnQuc2V0QXR0cmlidXRlKFwidGFiaW5kZXhcIiwgLTEpXG4gICAgQGVsZW1lbnQuc2V0QXR0cmlidXRlKFwibG9jYXRpb25cIiwgQGxvY2F0aW9uKVxuXG4gICAgQHRhYnMgPSBbXVxuICAgIEB0YWJzQnlFbGVtZW50ID0gbmV3IFdlYWtNYXBcbiAgICBAc3Vic2NyaXB0aW9ucyA9IG5ldyBDb21wb3NpdGVEaXNwb3NhYmxlXG5cbiAgICBAcGFuZUVsZW1lbnQgPSBAcGFuZS5nZXRFbGVtZW50KClcblxuICAgIEBzdWJzY3JpcHRpb25zLmFkZCBhdG9tLmNvbW1hbmRzLmFkZCBAcGFuZUVsZW1lbnQsXG4gICAgICAndGFiczprZWVwLXBlbmRpbmctdGFiJzogPT4gQHRlcm1pbmF0ZVBlbmRpbmdTdGF0ZXMoKVxuICAgICAgJ3RhYnM6Y2xvc2UtdGFiJzogPT4gQGNsb3NlVGFiKEBnZXRBY3RpdmVUYWIoKSlcbiAgICAgICd0YWJzOmNsb3NlLW90aGVyLXRhYnMnOiA9PiBAY2xvc2VPdGhlclRhYnMoQGdldEFjdGl2ZVRhYigpKVxuICAgICAgJ3RhYnM6Y2xvc2UtdGFicy10by1yaWdodCc6ID0+IEBjbG9zZVRhYnNUb1JpZ2h0KEBnZXRBY3RpdmVUYWIoKSlcbiAgICAgICd0YWJzOmNsb3NlLXRhYnMtdG8tbGVmdCc6ID0+IEBjbG9zZVRhYnNUb0xlZnQoQGdldEFjdGl2ZVRhYigpKVxuICAgICAgJ3RhYnM6Y2xvc2Utc2F2ZWQtdGFicyc6ID0+IEBjbG9zZVNhdmVkVGFicygpXG4gICAgICAndGFiczpjbG9zZS1hbGwtdGFicyc6IChldmVudCkgPT5cbiAgICAgICAgZXZlbnQuc3RvcFByb3BhZ2F0aW9uKClcbiAgICAgICAgQGNsb3NlQWxsVGFicygpXG4gICAgICAndGFiczpvcGVuLWluLW5ldy13aW5kb3cnOiA9PiBAb3BlbkluTmV3V2luZG93KClcblxuICAgIGFkZEVsZW1lbnRDb21tYW5kcyA9IChjb21tYW5kcykgPT5cbiAgICAgIGNvbW1hbmRzV2l0aFByb3BhZ2F0aW9uU3RvcHBlZCA9IHt9XG4gICAgICBPYmplY3Qua2V5cyhjb21tYW5kcykuZm9yRWFjaCAobmFtZSkgLT5cbiAgICAgICAgY29tbWFuZHNXaXRoUHJvcGFnYXRpb25TdG9wcGVkW25hbWVdID0gKGV2ZW50KSAtPlxuICAgICAgICAgIGV2ZW50LnN0b3BQcm9wYWdhdGlvbigpXG4gICAgICAgICAgY29tbWFuZHNbbmFtZV0oKVxuXG4gICAgICBAc3Vic2NyaXB0aW9ucy5hZGQoYXRvbS5jb21tYW5kcy5hZGQoQGVsZW1lbnQsIGNvbW1hbmRzV2l0aFByb3BhZ2F0aW9uU3RvcHBlZCkpXG5cbiAgICBhZGRFbGVtZW50Q29tbWFuZHNcbiAgICAgICd0YWJzOmNsb3NlLXRhYic6ID0+IEBjbG9zZVRhYigpXG4gICAgICAndGFiczpjbG9zZS1vdGhlci10YWJzJzogPT4gQGNsb3NlT3RoZXJUYWJzKClcbiAgICAgICd0YWJzOmNsb3NlLXRhYnMtdG8tcmlnaHQnOiA9PiBAY2xvc2VUYWJzVG9SaWdodCgpXG4gICAgICAndGFiczpjbG9zZS10YWJzLXRvLWxlZnQnOiA9PiBAY2xvc2VUYWJzVG9MZWZ0KClcbiAgICAgICd0YWJzOmNsb3NlLXNhdmVkLXRhYnMnOiA9PiBAY2xvc2VTYXZlZFRhYnMoKVxuICAgICAgJ3RhYnM6Y2xvc2UtYWxsLXRhYnMnOiA9PiBAY2xvc2VBbGxUYWJzKClcbiAgICAgICd0YWJzOnNwbGl0LXVwJzogPT4gQHNwbGl0VGFiKCdzcGxpdFVwJylcbiAgICAgICd0YWJzOnNwbGl0LWRvd24nOiA9PiBAc3BsaXRUYWIoJ3NwbGl0RG93bicpXG4gICAgICAndGFiczpzcGxpdC1sZWZ0JzogPT4gQHNwbGl0VGFiKCdzcGxpdExlZnQnKVxuICAgICAgJ3RhYnM6c3BsaXQtcmlnaHQnOiA9PiBAc3BsaXRUYWIoJ3NwbGl0UmlnaHQnKVxuXG4gICAgQGVsZW1lbnQuYWRkRXZlbnRMaXN0ZW5lciBcIm1vdXNlZW50ZXJcIiwgQG9uTW91c2VFbnRlci5iaW5kKHRoaXMpXG4gICAgQGVsZW1lbnQuYWRkRXZlbnRMaXN0ZW5lciBcIm1vdXNlbGVhdmVcIiwgQG9uTW91c2VMZWF2ZS5iaW5kKHRoaXMpXG4gICAgQGVsZW1lbnQuYWRkRXZlbnRMaXN0ZW5lciBcIm1vdXNld2hlZWxcIiwgQG9uTW91c2VXaGVlbC5iaW5kKHRoaXMpXG4gICAgQGVsZW1lbnQuYWRkRXZlbnRMaXN0ZW5lciBcImRyYWdzdGFydFwiLCBAb25EcmFnU3RhcnQuYmluZCh0aGlzKVxuICAgIEBlbGVtZW50LmFkZEV2ZW50TGlzdGVuZXIgXCJkcmFnZW5kXCIsIEBvbkRyYWdFbmQuYmluZCh0aGlzKVxuICAgIEBlbGVtZW50LmFkZEV2ZW50TGlzdGVuZXIgXCJkcmFnbGVhdmVcIiwgQG9uRHJhZ0xlYXZlLmJpbmQodGhpcylcbiAgICBAZWxlbWVudC5hZGRFdmVudExpc3RlbmVyIFwiZHJhZ292ZXJcIiwgQG9uRHJhZ092ZXIuYmluZCh0aGlzKVxuICAgIEBlbGVtZW50LmFkZEV2ZW50TGlzdGVuZXIgXCJkcm9wXCIsIEBvbkRyb3AuYmluZCh0aGlzKVxuXG4gICAgIyBUb2dnbGUgdGhlIHRhYiBiYXIgd2hlbiBhIHRhYiBpcyBkcmFnZ2VkIG92ZXIgdGhlIHBhbmUgd2l0aCBhbHdheXNTaG93VGFiQmFyID0gZmFsc2VcbiAgICBAcGFuZUVsZW1lbnQuYWRkRXZlbnRMaXN0ZW5lciAnZHJhZ2VudGVyJywgQG9uUGFuZURyYWdFbnRlci5iaW5kKHRoaXMpXG4gICAgQHBhbmVFbGVtZW50LmFkZEV2ZW50TGlzdGVuZXIgJ2RyYWdsZWF2ZScsIEBvblBhbmVEcmFnTGVhdmUuYmluZCh0aGlzKVxuXG4gICAgQHBhbmVDb250YWluZXIgPSBAcGFuZS5nZXRDb250YWluZXIoKVxuICAgIEBhZGRUYWJGb3JJdGVtKGl0ZW0pIGZvciBpdGVtIGluIEBwYW5lLmdldEl0ZW1zKClcblxuICAgIEBzdWJzY3JpcHRpb25zLmFkZCBAcGFuZS5vbkRpZERlc3Ryb3kgPT5cbiAgICAgIEBkZXN0cm95KClcblxuICAgIEBzdWJzY3JpcHRpb25zLmFkZCBAcGFuZS5vbkRpZEFkZEl0ZW0gKHtpdGVtLCBpbmRleH0pID0+XG4gICAgICBAYWRkVGFiRm9ySXRlbShpdGVtLCBpbmRleClcblxuICAgIEBzdWJzY3JpcHRpb25zLmFkZCBAcGFuZS5vbkRpZE1vdmVJdGVtICh7aXRlbSwgbmV3SW5kZXh9KSA9PlxuICAgICAgQG1vdmVJdGVtVGFiVG9JbmRleChpdGVtLCBuZXdJbmRleClcblxuICAgIEBzdWJzY3JpcHRpb25zLmFkZCBAcGFuZS5vbkRpZFJlbW92ZUl0ZW0gKHtpdGVtfSkgPT5cbiAgICAgIEByZW1vdmVUYWJGb3JJdGVtKGl0ZW0pXG5cbiAgICBAc3Vic2NyaXB0aW9ucy5hZGQgQHBhbmUub25EaWRDaGFuZ2VBY3RpdmVJdGVtIChpdGVtKSA9PlxuICAgICAgQHVwZGF0ZUFjdGl2ZVRhYigpXG5cbiAgICBAc3Vic2NyaXB0aW9ucy5hZGQgYXRvbS5jb25maWcub2JzZXJ2ZSAndGFicy50YWJTY3JvbGxpbmcnLCAodmFsdWUpID0+IEB1cGRhdGVUYWJTY3JvbGxpbmcodmFsdWUpXG4gICAgQHN1YnNjcmlwdGlvbnMuYWRkIGF0b20uY29uZmlnLm9ic2VydmUgJ3RhYnMudGFiU2Nyb2xsaW5nVGhyZXNob2xkJywgKHZhbHVlKSA9PiBAdXBkYXRlVGFiU2Nyb2xsaW5nVGhyZXNob2xkKHZhbHVlKVxuICAgIEBzdWJzY3JpcHRpb25zLmFkZCBhdG9tLmNvbmZpZy5vYnNlcnZlICd0YWJzLmFsd2F5c1Nob3dUYWJCYXInLCA9PiBAdXBkYXRlVGFiQmFyVmlzaWJpbGl0eSgpXG5cbiAgICBAdXBkYXRlQWN0aXZlVGFiKClcblxuICAgIEBlbGVtZW50LmFkZEV2ZW50TGlzdGVuZXIgXCJtb3VzZWRvd25cIiwgQG9uTW91c2VEb3duLmJpbmQodGhpcylcbiAgICBAZWxlbWVudC5hZGRFdmVudExpc3RlbmVyIFwiY2xpY2tcIiwgQG9uQ2xpY2suYmluZCh0aGlzKVxuICAgIEBlbGVtZW50LmFkZEV2ZW50TGlzdGVuZXIgXCJhdXhjbGlja1wiLCBAb25DbGljay5iaW5kKHRoaXMpXG4gICAgQGVsZW1lbnQuYWRkRXZlbnRMaXN0ZW5lciBcImRibGNsaWNrXCIsIEBvbkRvdWJsZUNsaWNrLmJpbmQodGhpcylcblxuICAgIEBvbkRyb3BPbk90aGVyV2luZG93ID0gQG9uRHJvcE9uT3RoZXJXaW5kb3cuYmluZCh0aGlzKVxuICAgIGlwY1JlbmRlcmVyLm9uKCd0YWI6ZHJvcHBlZCcsIEBvbkRyb3BPbk90aGVyV2luZG93KVxuXG4gIGRlc3Ryb3k6IC0+XG4gICAgaXBjUmVuZGVyZXIucmVtb3ZlTGlzdGVuZXIoJ3RhYjpkcm9wcGVkJywgQG9uRHJvcE9uT3RoZXJXaW5kb3cpXG4gICAgQHN1YnNjcmlwdGlvbnMuZGlzcG9zZSgpXG4gICAgQGVsZW1lbnQucmVtb3ZlKClcblxuICB0ZXJtaW5hdGVQZW5kaW5nU3RhdGVzOiAtPlxuICAgIHRhYi50ZXJtaW5hdGVQZW5kaW5nU3RhdGU/KCkgZm9yIHRhYiBpbiBAZ2V0VGFicygpXG4gICAgcmV0dXJuXG5cbiAgYWRkVGFiRm9ySXRlbTogKGl0ZW0sIGluZGV4KSAtPlxuICAgIHRhYlZpZXcgPSBuZXcgVGFiVmlldyh7XG4gICAgICBpdGVtLFxuICAgICAgQHBhbmUsXG4gICAgICBAdGFicyxcbiAgICAgIGRpZENsaWNrQ2xvc2VJY29uOiA9PlxuICAgICAgICBAY2xvc2VUYWIodGFiVmlldylcbiAgICAgICAgcmV0dXJuXG4gICAgICBAbG9jYXRpb25cbiAgICB9KVxuICAgIHRhYlZpZXcudGVybWluYXRlUGVuZGluZ1N0YXRlKCkgaWYgQGlzSXRlbU1vdmluZ0JldHdlZW5QYW5lc1xuICAgIEB0YWJzQnlFbGVtZW50LnNldCh0YWJWaWV3LmVsZW1lbnQsIHRhYlZpZXcpXG4gICAgQGluc2VydFRhYkF0SW5kZXgodGFiVmlldywgaW5kZXgpXG4gICAgaWYgYXRvbS5jb25maWcuZ2V0KCd0YWJzLmFkZE5ld1RhYnNBdEVuZCcpXG4gICAgICBAcGFuZS5tb3ZlSXRlbShpdGVtLCBAcGFuZS5nZXRJdGVtcygpLmxlbmd0aCAtIDEpIHVubGVzcyBAaXNJdGVtTW92aW5nQmV0d2VlblBhbmVzXG5cbiAgbW92ZUl0ZW1UYWJUb0luZGV4OiAoaXRlbSwgaW5kZXgpIC0+XG4gICAgdGFiSW5kZXggPSBAdGFicy5maW5kSW5kZXgoKHQpIC0+IHQuaXRlbSBpcyBpdGVtKVxuICAgIGlmIHRhYkluZGV4IGlzbnQgLTFcbiAgICAgIHRhYiA9IEB0YWJzW3RhYkluZGV4XVxuICAgICAgdGFiLmVsZW1lbnQucmVtb3ZlKClcbiAgICAgIEB0YWJzLnNwbGljZSh0YWJJbmRleCwgMSlcbiAgICAgIEBpbnNlcnRUYWJBdEluZGV4KHRhYiwgaW5kZXgpXG5cbiAgaW5zZXJ0VGFiQXRJbmRleDogKHRhYiwgaW5kZXgpIC0+XG4gICAgZm9sbG93aW5nVGFiID0gQHRhYnNbaW5kZXhdIGlmIGluZGV4P1xuICAgIGlmIGZvbGxvd2luZ1RhYlxuICAgICAgQGVsZW1lbnQuaW5zZXJ0QmVmb3JlKHRhYi5lbGVtZW50LCBmb2xsb3dpbmdUYWIuZWxlbWVudClcbiAgICAgIEB0YWJzLnNwbGljZShpbmRleCwgMCwgdGFiKVxuICAgIGVsc2VcbiAgICAgIEBlbGVtZW50LmFwcGVuZENoaWxkKHRhYi5lbGVtZW50KVxuICAgICAgQHRhYnMucHVzaCh0YWIpXG5cbiAgICB0YWIudXBkYXRlVGl0bGUoKVxuICAgIEB1cGRhdGVUYWJCYXJWaXNpYmlsaXR5KClcblxuICByZW1vdmVUYWJGb3JJdGVtOiAoaXRlbSkgLT5cbiAgICB0YWJJbmRleCA9IEB0YWJzLmZpbmRJbmRleCgodCkgLT4gdC5pdGVtIGlzIGl0ZW0pXG4gICAgaWYgdGFiSW5kZXggaXNudCAtMVxuICAgICAgdGFiID0gQHRhYnNbdGFiSW5kZXhdXG4gICAgICBAdGFicy5zcGxpY2UodGFiSW5kZXgsIDEpXG4gICAgICBAdGFic0J5RWxlbWVudC5kZWxldGUodGFiKVxuICAgICAgdGFiLmRlc3Ryb3koKVxuICAgIHRhYi51cGRhdGVUaXRsZSgpIGZvciB0YWIgaW4gQGdldFRhYnMoKVxuICAgIEB1cGRhdGVUYWJCYXJWaXNpYmlsaXR5KClcblxuICB1cGRhdGVUYWJCYXJWaXNpYmlsaXR5OiAtPlxuICAgICMgU2hvdyB0YWIgYmFyIGlmIHRoZSBzZXR0aW5nIGlzIHRydWUgb3IgdGhlcmUgaXMgbW9yZSB0aGFuIG9uZSB0YWJcbiAgICBpZiBhdG9tLmNvbmZpZy5nZXQoJ3RhYnMuYWx3YXlzU2hvd1RhYkJhcicpIG9yIEBwYW5lLmdldEl0ZW1zKCkubGVuZ3RoID4gMVxuICAgICAgQGVsZW1lbnQuY2xhc3NMaXN0LnJlbW92ZSgnaGlkZGVuJylcbiAgICBlbHNlXG4gICAgICBAZWxlbWVudC5jbGFzc0xpc3QuYWRkKCdoaWRkZW4nKVxuXG4gIGdldFRhYnM6IC0+XG4gICAgQHRhYnMuc2xpY2UoKVxuXG4gIHRhYkF0SW5kZXg6IChpbmRleCkgLT5cbiAgICBAdGFic1tpbmRleF1cblxuICB0YWJGb3JJdGVtOiAoaXRlbSkgLT5cbiAgICBAdGFicy5maW5kKCh0KSAtPiB0Lml0ZW0gaXMgaXRlbSlcblxuICBzZXRBY3RpdmVUYWI6ICh0YWJWaWV3KSAtPlxuICAgIGlmIHRhYlZpZXc/IGFuZCB0YWJWaWV3IGlzbnQgQGFjdGl2ZVRhYlxuICAgICAgQGFjdGl2ZVRhYj8uZWxlbWVudC5jbGFzc0xpc3QucmVtb3ZlKCdhY3RpdmUnKVxuICAgICAgQGFjdGl2ZVRhYiA9IHRhYlZpZXdcbiAgICAgIEBhY3RpdmVUYWIuZWxlbWVudC5jbGFzc0xpc3QuYWRkKCdhY3RpdmUnKVxuICAgICAgQGFjdGl2ZVRhYi5lbGVtZW50LnNjcm9sbEludG9WaWV3KGZhbHNlKVxuXG4gIGdldEFjdGl2ZVRhYjogLT5cbiAgICBAdGFiRm9ySXRlbShAcGFuZS5nZXRBY3RpdmVJdGVtKCkpXG5cbiAgdXBkYXRlQWN0aXZlVGFiOiAtPlxuICAgIEBzZXRBY3RpdmVUYWIoQHRhYkZvckl0ZW0oQHBhbmUuZ2V0QWN0aXZlSXRlbSgpKSlcblxuICBjbG9zZVRhYjogKHRhYikgLT5cbiAgICB0YWIgPz0gQHJpZ2h0Q2xpY2tlZFRhYlxuICAgIEBwYW5lLmRlc3Ryb3lJdGVtKHRhYi5pdGVtKSBpZiB0YWI/XG5cbiAgb3BlbkluTmV3V2luZG93OiAodGFiKSAtPlxuICAgIHRhYiA/PSBAcmlnaHRDbGlja2VkVGFiXG4gICAgaXRlbSA9IHRhYj8uaXRlbVxuICAgIHJldHVybiB1bmxlc3MgaXRlbT9cbiAgICBpZiB0eXBlb2YgaXRlbS5nZXRVUkkgaXMgJ2Z1bmN0aW9uJ1xuICAgICAgaXRlbVVSSSA9IGl0ZW0uZ2V0VVJJKClcbiAgICBlbHNlIGlmIHR5cGVvZiBpdGVtLmdldFBhdGggaXMgJ2Z1bmN0aW9uJ1xuICAgICAgaXRlbVVSSSA9IGl0ZW0uZ2V0UGF0aCgpXG4gICAgZWxzZSBpZiB0eXBlb2YgaXRlbS5nZXRVcmkgaXMgJ2Z1bmN0aW9uJ1xuICAgICAgaXRlbVVSSSA9IGl0ZW0uZ2V0VXJpKClcbiAgICByZXR1cm4gdW5sZXNzIGl0ZW1VUkk/XG4gICAgQGNsb3NlVGFiKHRhYilcbiAgICB0YWIuZWxlbWVudC5zdHlsZS5tYXhXaWR0aCA9ICcnIGZvciB0YWIgaW4gQGdldFRhYnMoKVxuICAgIHBhdGhzVG9PcGVuID0gW2F0b20ucHJvamVjdC5nZXRQYXRocygpLCBpdGVtVVJJXS5yZWR1Y2UgKChhLCBiKSAtPiBhLmNvbmNhdChiKSksIFtdXG4gICAgYXRvbS5vcGVuKHtwYXRoc1RvT3BlbjogcGF0aHNUb09wZW4sIG5ld1dpbmRvdzogdHJ1ZSwgZGV2TW9kZTogYXRvbS5kZXZNb2RlLCBzYWZlTW9kZTogYXRvbS5zYWZlTW9kZX0pXG5cbiAgc3BsaXRUYWI6IChmbikgLT5cbiAgICBpZiBpdGVtID0gQHJpZ2h0Q2xpY2tlZFRhYj8uaXRlbVxuICAgICAgaWYgY29waWVkSXRlbSA9IGl0ZW0uY29weT8oKVxuICAgICAgICBAcGFuZVtmbl0oaXRlbXM6IFtjb3BpZWRJdGVtXSlcblxuICBjbG9zZU90aGVyVGFiczogKGFjdGl2ZSkgLT5cbiAgICB0YWJzID0gQGdldFRhYnMoKVxuICAgIGFjdGl2ZSA/PSBAcmlnaHRDbGlja2VkVGFiXG4gICAgcmV0dXJuIHVubGVzcyBhY3RpdmU/XG4gICAgQGNsb3NlVGFiIHRhYiBmb3IgdGFiIGluIHRhYnMgd2hlbiB0YWIgaXNudCBhY3RpdmVcblxuICBjbG9zZVRhYnNUb1JpZ2h0OiAoYWN0aXZlKSAtPlxuICAgIHRhYnMgPSBAZ2V0VGFicygpXG4gICAgYWN0aXZlID89IEByaWdodENsaWNrZWRUYWJcbiAgICBpbmRleCA9IHRhYnMuaW5kZXhPZihhY3RpdmUpXG4gICAgcmV0dXJuIGlmIGluZGV4IGlzIC0xXG4gICAgQGNsb3NlVGFiIHRhYiBmb3IgdGFiLCBpIGluIHRhYnMgd2hlbiBpID4gaW5kZXhcblxuICBjbG9zZVRhYnNUb0xlZnQ6IChhY3RpdmUpIC0+XG4gICAgdGFicyA9IEBnZXRUYWJzKClcbiAgICBhY3RpdmUgPz0gQHJpZ2h0Q2xpY2tlZFRhYlxuICAgIGluZGV4ID0gdGFicy5pbmRleE9mKGFjdGl2ZSlcbiAgICByZXR1cm4gaWYgaW5kZXggaXMgLTFcbiAgICBAY2xvc2VUYWIgdGFiIGZvciB0YWIsIGkgaW4gdGFicyB3aGVuIGkgPCBpbmRleFxuXG4gIGNsb3NlU2F2ZWRUYWJzOiAtPlxuICAgIGZvciB0YWIgaW4gQGdldFRhYnMoKVxuICAgICAgQGNsb3NlVGFiKHRhYikgdW5sZXNzIHRhYi5pdGVtLmlzTW9kaWZpZWQ/KClcblxuICBjbG9zZUFsbFRhYnM6IC0+XG4gICAgQGNsb3NlVGFiKHRhYikgZm9yIHRhYiBpbiBAZ2V0VGFicygpXG5cbiAgZ2V0V2luZG93SWQ6IC0+XG4gICAgQHdpbmRvd0lkID89IGF0b20uZ2V0Q3VycmVudFdpbmRvdygpLmlkXG5cbiAgb25EcmFnU3RhcnQ6IChldmVudCkgLT5cbiAgICBAZHJhZ2dlZFRhYiA9IEB0YWJGb3JFbGVtZW50KGV2ZW50LnRhcmdldClcbiAgICByZXR1cm4gdW5sZXNzIEBkcmFnZ2VkVGFiXG4gICAgQGxhc3REcm9wVGFyZ2V0SW5kZXggPSBudWxsXG5cbiAgICBldmVudC5kYXRhVHJhbnNmZXIuc2V0RGF0YSAnYXRvbS10YWItZXZlbnQnLCAndHJ1ZSdcblxuICAgIEBkcmFnZ2VkVGFiLmVsZW1lbnQuY2xhc3NMaXN0LmFkZCgnaXMtZHJhZ2dpbmcnKVxuICAgIEBkcmFnZ2VkVGFiLmRlc3Ryb3lUb29sdGlwKClcblxuICAgIHRhYkluZGV4ID0gQHRhYnMuaW5kZXhPZihAZHJhZ2dlZFRhYilcbiAgICBldmVudC5kYXRhVHJhbnNmZXIuc2V0RGF0YSAnc29ydGFibGUtaW5kZXgnLCB0YWJJbmRleFxuXG4gICAgcGFuZUluZGV4ID0gQHBhbmVDb250YWluZXIuZ2V0UGFuZXMoKS5pbmRleE9mKEBwYW5lKVxuICAgIGV2ZW50LmRhdGFUcmFuc2Zlci5zZXREYXRhICdmcm9tLXBhbmUtaW5kZXgnLCBwYW5lSW5kZXhcbiAgICBldmVudC5kYXRhVHJhbnNmZXIuc2V0RGF0YSAnZnJvbS1wYW5lLWlkJywgQHBhbmUuaWRcbiAgICBldmVudC5kYXRhVHJhbnNmZXIuc2V0RGF0YSAnZnJvbS13aW5kb3ctaWQnLCBAZ2V0V2luZG93SWQoKVxuXG4gICAgaXRlbSA9IEBwYW5lLmdldEl0ZW1zKClbQHRhYnMuaW5kZXhPZihAZHJhZ2dlZFRhYildXG4gICAgcmV0dXJuIHVubGVzcyBpdGVtP1xuXG4gICAgaWYgdHlwZW9mIGl0ZW0uZ2V0VVJJIGlzICdmdW5jdGlvbidcbiAgICAgIGl0ZW1VUkkgPSBpdGVtLmdldFVSSSgpID8gJydcbiAgICBlbHNlIGlmIHR5cGVvZiBpdGVtLmdldFBhdGggaXMgJ2Z1bmN0aW9uJ1xuICAgICAgaXRlbVVSSSA9IGl0ZW0uZ2V0UGF0aCgpID8gJydcbiAgICBlbHNlIGlmIHR5cGVvZiBpdGVtLmdldFVyaSBpcyAnZnVuY3Rpb24nXG4gICAgICBpdGVtVVJJID0gaXRlbS5nZXRVcmkoKSA/ICcnXG5cbiAgICBpZiB0eXBlb2YgaXRlbS5nZXRBbGxvd2VkTG9jYXRpb25zIGlzICdmdW5jdGlvbidcbiAgICAgIGZvciBsb2NhdGlvbiBpbiBpdGVtLmdldEFsbG93ZWRMb2NhdGlvbnMoKVxuICAgICAgICBldmVudC5kYXRhVHJhbnNmZXIuc2V0RGF0YShcImFsbG93ZWQtbG9jYXRpb24tI3tsb2NhdGlvbn1cIiwgJ3RydWUnKVxuICAgIGVsc2VcbiAgICAgIGV2ZW50LmRhdGFUcmFuc2Zlci5zZXREYXRhICdhbGxvdy1hbGwtbG9jYXRpb25zJywgJ3RydWUnXG5cbiAgICBpZiBpdGVtVVJJP1xuICAgICAgZXZlbnQuZGF0YVRyYW5zZmVyLnNldERhdGEgJ3RleHQvcGxhaW4nLCBpdGVtVVJJXG5cbiAgICAgIGlmIHByb2Nlc3MucGxhdGZvcm0gaXMgJ2RhcndpbicgIyBzZWUgIzY5XG4gICAgICAgIGl0ZW1VUkkgPSBcImZpbGU6Ly8je2l0ZW1VUkl9XCIgdW5sZXNzIEB1cmlIYXNQcm90b2NvbChpdGVtVVJJKVxuICAgICAgICBldmVudC5kYXRhVHJhbnNmZXIuc2V0RGF0YSAndGV4dC91cmktbGlzdCcsIGl0ZW1VUklcblxuICAgICAgaWYgaXRlbS5pc01vZGlmaWVkPygpIGFuZCBpdGVtLmdldFRleHQ/XG4gICAgICAgIGV2ZW50LmRhdGFUcmFuc2Zlci5zZXREYXRhICdoYXMtdW5zYXZlZC1jaGFuZ2VzJywgJ3RydWUnXG4gICAgICAgIGV2ZW50LmRhdGFUcmFuc2Zlci5zZXREYXRhICdtb2RpZmllZC10ZXh0JywgaXRlbS5nZXRUZXh0KClcblxuICB1cmlIYXNQcm90b2NvbDogKHVyaSkgLT5cbiAgICB0cnlcbiAgICAgIHJlcXVpcmUoJ3VybCcpLnBhcnNlKHVyaSkucHJvdG9jb2w/XG4gICAgY2F0Y2ggZXJyb3JcbiAgICAgIGZhbHNlXG5cbiAgb25EcmFnTGVhdmU6IChldmVudCkgLT5cbiAgICAjIERvIG5vdCBkbyBhbnl0aGluZyB1bmxlc3MgdGhlIGRyYWcgZ29lcyBvdXRzaWRlIHRoZSB0YWIgYmFyXG4gICAgdW5sZXNzIGV2ZW50LmN1cnJlbnRUYXJnZXQuY29udGFpbnMoZXZlbnQucmVsYXRlZFRhcmdldClcbiAgICAgIEByZW1vdmVQbGFjZWhvbGRlcigpXG4gICAgICBAbGFzdERyb3BUYXJnZXRJbmRleCA9IG51bGxcbiAgICAgIHRhYi5lbGVtZW50LnN0eWxlLm1heFdpZHRoID0gJycgZm9yIHRhYiBpbiBAZ2V0VGFicygpXG5cbiAgb25EcmFnRW5kOiAoZXZlbnQpIC0+XG4gICAgcmV0dXJuIHVubGVzcyBAdGFiRm9yRWxlbWVudChldmVudC50YXJnZXQpXG5cbiAgICBAY2xlYXJEcm9wVGFyZ2V0KClcblxuICBvbkRyYWdPdmVyOiAoZXZlbnQpIC0+XG4gICAgcmV0dXJuIHVubGVzcyBAaXNBdG9tVGFiRXZlbnQoZXZlbnQpXG4gICAgcmV0dXJuIHVubGVzcyBAaXRlbUlzQWxsb3dlZChldmVudCwgQGxvY2F0aW9uKVxuXG4gICAgZXZlbnQucHJldmVudERlZmF1bHQoKVxuICAgIGV2ZW50LnN0b3BQcm9wYWdhdGlvbigpXG5cbiAgICBuZXdEcm9wVGFyZ2V0SW5kZXggPSBAZ2V0RHJvcFRhcmdldEluZGV4KGV2ZW50KVxuICAgIHJldHVybiB1bmxlc3MgbmV3RHJvcFRhcmdldEluZGV4P1xuICAgIHJldHVybiBpZiBAbGFzdERyb3BUYXJnZXRJbmRleCBpcyBuZXdEcm9wVGFyZ2V0SW5kZXhcbiAgICBAbGFzdERyb3BUYXJnZXRJbmRleCA9IG5ld0Ryb3BUYXJnZXRJbmRleFxuXG4gICAgQHJlbW92ZURyb3BUYXJnZXRDbGFzc2VzKClcblxuICAgIHRhYnMgPSBAZ2V0VGFicygpXG4gICAgcGxhY2Vob2xkZXIgPSBAZ2V0UGxhY2Vob2xkZXIoKVxuICAgIHJldHVybiB1bmxlc3MgcGxhY2Vob2xkZXI/XG5cbiAgICBpZiBuZXdEcm9wVGFyZ2V0SW5kZXggPCB0YWJzLmxlbmd0aFxuICAgICAgdGFiID0gdGFic1tuZXdEcm9wVGFyZ2V0SW5kZXhdXG4gICAgICB0YWIuZWxlbWVudC5jbGFzc0xpc3QuYWRkICdpcy1kcm9wLXRhcmdldCdcbiAgICAgIHRhYi5lbGVtZW50LnBhcmVudEVsZW1lbnQuaW5zZXJ0QmVmb3JlKHBsYWNlaG9sZGVyLCB0YWIuZWxlbWVudClcbiAgICBlbHNlXG4gICAgICBpZiB0YWIgPSB0YWJzW25ld0Ryb3BUYXJnZXRJbmRleCAtIDFdXG4gICAgICAgIHRhYi5lbGVtZW50LmNsYXNzTGlzdC5hZGQgJ2Ryb3AtdGFyZ2V0LWlzLWFmdGVyJ1xuICAgICAgICBpZiBzaWJsaW5nID0gdGFiLmVsZW1lbnQubmV4dFNpYmxpbmdcbiAgICAgICAgICB0YWIuZWxlbWVudC5wYXJlbnRFbGVtZW50Lmluc2VydEJlZm9yZShwbGFjZWhvbGRlciwgc2libGluZylcbiAgICAgICAgZWxzZVxuICAgICAgICAgIHRhYi5lbGVtZW50LnBhcmVudEVsZW1lbnQuYXBwZW5kQ2hpbGQocGxhY2Vob2xkZXIpXG5cbiAgb25Ecm9wT25PdGhlcldpbmRvdzogKGV2ZW50LCBmcm9tUGFuZUlkLCBmcm9tSXRlbUluZGV4KSAtPlxuICAgIGlmIEBwYW5lLmlkIGlzIGZyb21QYW5lSWRcbiAgICAgIGlmIGl0ZW1Ub1JlbW92ZSA9IEBwYW5lLmdldEl0ZW1zKClbZnJvbUl0ZW1JbmRleF1cbiAgICAgICAgQHBhbmUuZGVzdHJveUl0ZW0oaXRlbVRvUmVtb3ZlKVxuXG4gICAgQGNsZWFyRHJvcFRhcmdldCgpXG5cbiAgY2xlYXJEcm9wVGFyZ2V0OiAtPlxuICAgIEBkcmFnZ2VkVGFiPy5lbGVtZW50LmNsYXNzTGlzdC5yZW1vdmUoJ2lzLWRyYWdnaW5nJylcbiAgICBAZHJhZ2dlZFRhYj8udXBkYXRlVG9vbHRpcCgpXG4gICAgQGRyYWdnZWRUYWIgPSBudWxsXG4gICAgQHJlbW92ZURyb3BUYXJnZXRDbGFzc2VzKClcbiAgICBAcmVtb3ZlUGxhY2Vob2xkZXIoKVxuXG4gIG9uRHJvcDogKGV2ZW50KSAtPlxuICAgIHJldHVybiB1bmxlc3MgQGlzQXRvbVRhYkV2ZW50KGV2ZW50KVxuXG4gICAgZXZlbnQucHJldmVudERlZmF1bHQoKVxuXG4gICAgZnJvbVdpbmRvd0lkICA9IHBhcnNlSW50KGV2ZW50LmRhdGFUcmFuc2Zlci5nZXREYXRhKCdmcm9tLXdpbmRvdy1pZCcpKVxuICAgIGZyb21QYW5lSWQgICAgPSBwYXJzZUludChldmVudC5kYXRhVHJhbnNmZXIuZ2V0RGF0YSgnZnJvbS1wYW5lLWlkJykpXG4gICAgZnJvbUluZGV4ICAgICA9IHBhcnNlSW50KGV2ZW50LmRhdGFUcmFuc2Zlci5nZXREYXRhKCdzb3J0YWJsZS1pbmRleCcpKVxuICAgIGZyb21QYW5lSW5kZXggPSBwYXJzZUludChldmVudC5kYXRhVHJhbnNmZXIuZ2V0RGF0YSgnZnJvbS1wYW5lLWluZGV4JykpXG5cbiAgICBoYXNVbnNhdmVkQ2hhbmdlcyA9IGV2ZW50LmRhdGFUcmFuc2Zlci5nZXREYXRhKCdoYXMtdW5zYXZlZC1jaGFuZ2VzJykgaXMgJ3RydWUnXG4gICAgbW9kaWZpZWRUZXh0ID0gZXZlbnQuZGF0YVRyYW5zZmVyLmdldERhdGEoJ21vZGlmaWVkLXRleHQnKVxuXG4gICAgdG9JbmRleCA9IEBnZXREcm9wVGFyZ2V0SW5kZXgoZXZlbnQpXG4gICAgdG9QYW5lID0gQHBhbmVcblxuICAgIEBjbGVhckRyb3BUYXJnZXQoKVxuXG4gICAgcmV0dXJuIHVubGVzcyBAaXRlbUlzQWxsb3dlZChldmVudCwgQGxvY2F0aW9uKVxuXG4gICAgaWYgZnJvbVdpbmRvd0lkIGlzIEBnZXRXaW5kb3dJZCgpXG4gICAgICBmcm9tUGFuZSA9IEBwYW5lQ29udGFpbmVyLmdldFBhbmVzKClbZnJvbVBhbmVJbmRleF1cbiAgICAgIGlmIGZyb21QYW5lPy5pZCBpc250IGZyb21QYW5lSWRcbiAgICAgICAgIyBJZiBkcmFnZ2luZyBmcm9tIGEgZGlmZmVyZW50IHBhbmUgY29udGFpbmVyLCB3ZSBoYXZlIHRvIGJlIG1vcmVcbiAgICAgICAgIyBleGhhdXN0aXZlIGluIG91ciBzZWFyY2guXG4gICAgICAgIGZyb21QYW5lID0gQXJyYXkuZnJvbSBkb2N1bWVudC5xdWVyeVNlbGVjdG9yQWxsKCdhdG9tLXBhbmUnKVxuICAgICAgICAgIC5tYXAgKHBhbmVFbCkgLT4gcGFuZUVsLm1vZGVsXG4gICAgICAgICAgLmZpbmQgKHBhbmUpIC0+IHBhbmUuaWQgaXMgZnJvbVBhbmVJZFxuICAgICAgaXRlbSA9IGZyb21QYW5lLmdldEl0ZW1zKClbZnJvbUluZGV4XVxuICAgICAgQG1vdmVJdGVtQmV0d2VlblBhbmVzKGZyb21QYW5lLCBmcm9tSW5kZXgsIHRvUGFuZSwgdG9JbmRleCwgaXRlbSkgaWYgaXRlbT9cbiAgICBlbHNlXG4gICAgICBkcm9wcGVkVVJJID0gZXZlbnQuZGF0YVRyYW5zZmVyLmdldERhdGEoJ3RleHQvcGxhaW4nKVxuICAgICAgYXRvbS53b3Jrc3BhY2Uub3Blbihkcm9wcGVkVVJJKS50aGVuIChpdGVtKSA9PlxuICAgICAgICAjIE1vdmUgdGhlIGl0ZW0gZnJvbSB0aGUgcGFuZSBpdCB3YXMgb3BlbmVkIG9uIHRvIHRoZSB0YXJnZXQgcGFuZVxuICAgICAgICAjIHdoZXJlIGl0IHdhcyBkcm9wcGVkIG9udG9cbiAgICAgICAgYWN0aXZlUGFuZSA9IGF0b20ud29ya3NwYWNlLmdldEFjdGl2ZVBhbmUoKVxuICAgICAgICBhY3RpdmVJdGVtSW5kZXggPSBhY3RpdmVQYW5lLmdldEl0ZW1zKCkuaW5kZXhPZihpdGVtKVxuICAgICAgICBAbW92ZUl0ZW1CZXR3ZWVuUGFuZXMoYWN0aXZlUGFuZSwgYWN0aXZlSXRlbUluZGV4LCB0b1BhbmUsIHRvSW5kZXgsIGl0ZW0pXG4gICAgICAgIGl0ZW0uc2V0VGV4dD8obW9kaWZpZWRUZXh0KSBpZiBoYXNVbnNhdmVkQ2hhbmdlc1xuXG4gICAgICAgIGlmIG5vdCBpc05hTihmcm9tV2luZG93SWQpXG4gICAgICAgICAgIyBMZXQgdGhlIHdpbmRvdyB3aGVyZSB0aGUgZHJhZyBzdGFydGVkIGtub3cgdGhhdCB0aGUgdGFiIHdhcyBkcm9wcGVkXG4gICAgICAgICAgYnJvd3NlcldpbmRvdyA9IEBicm93c2VyV2luZG93Rm9ySWQoZnJvbVdpbmRvd0lkKVxuICAgICAgICAgIGJyb3dzZXJXaW5kb3c/LndlYkNvbnRlbnRzLnNlbmQoJ3RhYjpkcm9wcGVkJywgZnJvbVBhbmVJZCwgZnJvbUluZGV4KVxuXG4gICAgICBhdG9tLmZvY3VzKClcblxuICAjIFNob3cgdGhlIHRhYiBiYXIgd2hlbiBhIHRhYiBpcyBiZWluZyBkcmFnZ2VkIGluIHRoaXMgcGFuZSB3aGVuIGFsd2F5c1Nob3dUYWJCYXIgPSBmYWxzZVxuICBvblBhbmVEcmFnRW50ZXI6IChldmVudCkgLT5cbiAgICByZXR1cm4gdW5sZXNzIEBpc0F0b21UYWJFdmVudChldmVudClcbiAgICByZXR1cm4gdW5sZXNzIEBpdGVtSXNBbGxvd2VkKGV2ZW50LCBAbG9jYXRpb24pXG4gICAgcmV0dXJuIGlmIEBwYW5lLmdldEl0ZW1zKCkubGVuZ3RoID4gMSBvciBhdG9tLmNvbmZpZy5nZXQoJ3RhYnMuYWx3YXlzU2hvd1RhYkJhcicpXG4gICAgaWYgQHBhbmVFbGVtZW50LmNvbnRhaW5zKGV2ZW50LnJlbGF0ZWRUYXJnZXQpXG4gICAgICBAZWxlbWVudC5jbGFzc0xpc3QucmVtb3ZlKCdoaWRkZW4nKVxuXG4gICMgSGlkZSB0aGUgdGFiIGJhciB3aGVuIHRoZSBkcmFnZ2VkIHRhYiBsZWF2ZXMgdGhpcyBwYW5lIHdoZW4gYWx3YXlzU2hvd1RhYkJhciA9IGZhbHNlXG4gIG9uUGFuZURyYWdMZWF2ZTogKGV2ZW50KSAtPlxuICAgIHJldHVybiB1bmxlc3MgQGlzQXRvbVRhYkV2ZW50KGV2ZW50KVxuICAgIHJldHVybiB1bmxlc3MgQGl0ZW1Jc0FsbG93ZWQoZXZlbnQsIEBsb2NhdGlvbilcbiAgICByZXR1cm4gaWYgQHBhbmUuZ2V0SXRlbXMoKS5sZW5ndGggPiAxIG9yIGF0b20uY29uZmlnLmdldCgndGFicy5hbHdheXNTaG93VGFiQmFyJylcbiAgICB1bmxlc3MgQHBhbmVFbGVtZW50LmNvbnRhaW5zKGV2ZW50LnJlbGF0ZWRUYXJnZXQpXG4gICAgICBAZWxlbWVudC5jbGFzc0xpc3QuYWRkKCdoaWRkZW4nKVxuXG4gIG9uTW91c2VXaGVlbDogKGV2ZW50KSAtPlxuICAgIHJldHVybiBpZiBldmVudC5zaGlmdEtleSBvciBub3QgQHRhYlNjcm9sbGluZ1xuXG4gICAgQHdoZWVsRGVsdGEgPz0gMFxuICAgIEB3aGVlbERlbHRhICs9IGV2ZW50LndoZWVsRGVsdGFZXG5cbiAgICBpZiBAd2hlZWxEZWx0YSA8PSAtQHRhYlNjcm9sbGluZ1RocmVzaG9sZFxuICAgICAgQHdoZWVsRGVsdGEgPSAwXG4gICAgICBAcGFuZS5hY3RpdmF0ZU5leHRJdGVtKClcbiAgICBlbHNlIGlmIEB3aGVlbERlbHRhID49IEB0YWJTY3JvbGxpbmdUaHJlc2hvbGRcbiAgICAgIEB3aGVlbERlbHRhID0gMFxuICAgICAgQHBhbmUuYWN0aXZhdGVQcmV2aW91c0l0ZW0oKVxuXG4gIG9uTW91c2VEb3duOiAoZXZlbnQpIC0+XG4gICAgdGFiID0gQHRhYkZvckVsZW1lbnQoZXZlbnQudGFyZ2V0KVxuICAgIHJldHVybiB1bmxlc3MgdGFiXG5cbiAgICBpZiBldmVudC53aGljaCBpcyAzIG9yIChldmVudC53aGljaCBpcyAxIGFuZCBldmVudC5jdHJsS2V5IGlzIHRydWUpXG4gICAgICBAcmlnaHRDbGlja2VkVGFiPy5lbGVtZW50LmNsYXNzTGlzdC5yZW1vdmUoJ3JpZ2h0LWNsaWNrZWQnKVxuICAgICAgQHJpZ2h0Q2xpY2tlZFRhYiA9IHRhYlxuICAgICAgQHJpZ2h0Q2xpY2tlZFRhYi5lbGVtZW50LmNsYXNzTGlzdC5hZGQoJ3JpZ2h0LWNsaWNrZWQnKVxuICAgICAgZXZlbnQucHJldmVudERlZmF1bHQoKVxuICAgIGVsc2UgaWYgZXZlbnQud2hpY2ggaXMgMlxuICAgICAgIyBUaGlzIHByZXZlbnRzIENocm9taXVtIGZyb20gYWN0aXZhdGluZyBcInNjcm9sbCBtb2RlXCIgd2hlblxuICAgICAgIyBtaWRkbGUtY2xpY2tpbmcgd2hpbGUgc29tZSB0YWJzIGFyZSBvZmYtc2NyZWVuLlxuICAgICAgZXZlbnQucHJldmVudERlZmF1bHQoKVxuXG4gIG9uQ2xpY2s6IChldmVudCkgLT5cbiAgICB0YWIgPSBAdGFiRm9yRWxlbWVudChldmVudC50YXJnZXQpXG4gICAgcmV0dXJuIHVubGVzcyB0YWJcblxuICAgIGV2ZW50LnByZXZlbnREZWZhdWx0KClcbiAgICBpZiBldmVudC53aGljaCBpcyAzIG9yIChldmVudC53aGljaCBpcyAxIGFuZCBldmVudC5jdHJsS2V5IGlzIHRydWUpXG4gICAgICAjIEJhaWwgb3V0IGVhcmx5IHdoZW4gcmVjZWl2aW5nIHRoaXMgZXZlbnQsIGJlY2F1c2Ugd2UgaGF2ZSBhbHJlYWR5XG4gICAgICAjIGhhbmRsZWQgaXQgaW4gdGhlIG1vdXNlZG93biBoYW5kbGVyLlxuICAgICAgcmV0dXJuXG4gICAgZWxzZSBpZiBldmVudC53aGljaCBpcyAxIGFuZCBub3QgZXZlbnQudGFyZ2V0LmNsYXNzTGlzdC5jb250YWlucygnY2xvc2UtaWNvbicpXG4gICAgICBAcGFuZS5hY3RpdmF0ZUl0ZW0odGFiLml0ZW0pXG4gICAgICBAcGFuZS5hY3RpdmF0ZSgpIHVubGVzcyBAcGFuZS5pc0Rlc3Ryb3llZCgpXG4gICAgZWxzZSBpZiBldmVudC53aGljaCBpcyAyXG4gICAgICBAcGFuZS5kZXN0cm95SXRlbSh0YWIuaXRlbSlcblxuICBvbkRvdWJsZUNsaWNrOiAoZXZlbnQpIC0+XG4gICAgaWYgdGFiID0gQHRhYkZvckVsZW1lbnQoZXZlbnQudGFyZ2V0KVxuICAgICAgdGFiLml0ZW0udGVybWluYXRlUGVuZGluZ1N0YXRlPygpXG4gICAgZWxzZSBpZiBldmVudC50YXJnZXQgaXMgQGVsZW1lbnRcbiAgICAgIGF0b20uY29tbWFuZHMuZGlzcGF0Y2goQGVsZW1lbnQsICdhcHBsaWNhdGlvbjpuZXctZmlsZScpXG4gICAgICBldmVudC5wcmV2ZW50RGVmYXVsdCgpXG5cbiAgdXBkYXRlVGFiU2Nyb2xsaW5nVGhyZXNob2xkOiAodmFsdWUpIC0+XG4gICAgQHRhYlNjcm9sbGluZ1RocmVzaG9sZCA9IHZhbHVlXG5cbiAgdXBkYXRlVGFiU2Nyb2xsaW5nOiAodmFsdWUpIC0+XG4gICAgaWYgdmFsdWUgaXMgJ3BsYXRmb3JtJ1xuICAgICAgQHRhYlNjcm9sbGluZyA9IChwcm9jZXNzLnBsYXRmb3JtIGlzICdsaW51eCcpXG4gICAgZWxzZVxuICAgICAgQHRhYlNjcm9sbGluZyA9IHZhbHVlXG5cbiAgYnJvd3NlcldpbmRvd0ZvcklkOiAoaWQpIC0+XG4gICAgQnJvd3NlcldpbmRvdyA/PSByZXF1aXJlKCdlbGVjdHJvbicpLnJlbW90ZS5Ccm93c2VyV2luZG93XG5cbiAgICBCcm93c2VyV2luZG93LmZyb21JZCBpZFxuXG4gIG1vdmVJdGVtQmV0d2VlblBhbmVzOiAoZnJvbVBhbmUsIGZyb21JbmRleCwgdG9QYW5lLCB0b0luZGV4LCBpdGVtKSAtPlxuICAgIHRyeVxuICAgICAgaWYgdG9QYW5lIGlzIGZyb21QYW5lXG4gICAgICAgIHRvSW5kZXgtLSBpZiBmcm9tSW5kZXggPCB0b0luZGV4XG4gICAgICAgIHRvUGFuZS5tb3ZlSXRlbShpdGVtLCB0b0luZGV4KVxuICAgICAgZWxzZVxuICAgICAgICBAaXNJdGVtTW92aW5nQmV0d2VlblBhbmVzID0gdHJ1ZVxuICAgICAgICBmcm9tUGFuZS5tb3ZlSXRlbVRvUGFuZShpdGVtLCB0b1BhbmUsIHRvSW5kZXgtLSlcbiAgICAgIHRvUGFuZS5hY3RpdmF0ZUl0ZW0oaXRlbSlcbiAgICAgIHRvUGFuZS5hY3RpdmF0ZSgpXG4gICAgZmluYWxseVxuICAgICAgQGlzSXRlbU1vdmluZ0JldHdlZW5QYW5lcyA9IGZhbHNlXG5cbiAgcmVtb3ZlRHJvcFRhcmdldENsYXNzZXM6IC0+XG4gICAgd29ya3NwYWNlRWxlbWVudCA9IGF0b20ud29ya3NwYWNlLmdldEVsZW1lbnQoKVxuICAgIGZvciBkcm9wVGFyZ2V0IGluIHdvcmtzcGFjZUVsZW1lbnQucXVlcnlTZWxlY3RvckFsbCgnLnRhYi1iYXIgLmlzLWRyb3AtdGFyZ2V0JylcbiAgICAgIGRyb3BUYXJnZXQuY2xhc3NMaXN0LnJlbW92ZSgnaXMtZHJvcC10YXJnZXQnKVxuXG4gICAgZm9yIGRyb3BUYXJnZXQgaW4gd29ya3NwYWNlRWxlbWVudC5xdWVyeVNlbGVjdG9yQWxsKCcudGFiLWJhciAuZHJvcC10YXJnZXQtaXMtYWZ0ZXInKVxuICAgICAgZHJvcFRhcmdldC5jbGFzc0xpc3QucmVtb3ZlKCdkcm9wLXRhcmdldC1pcy1hZnRlcicpXG5cbiAgZ2V0RHJvcFRhcmdldEluZGV4OiAoZXZlbnQpIC0+XG4gICAgdGFyZ2V0ID0gZXZlbnQudGFyZ2V0XG5cbiAgICByZXR1cm4gaWYgQGlzUGxhY2Vob2xkZXIodGFyZ2V0KVxuXG4gICAgdGFicyA9IEBnZXRUYWJzKClcbiAgICB0YWIgPSBAdGFiRm9yRWxlbWVudCh0YXJnZXQpXG4gICAgdGFiID89IHRhYnNbdGFicy5sZW5ndGggLSAxXVxuXG4gICAgcmV0dXJuIDAgdW5sZXNzIHRhYj9cblxuICAgIHtsZWZ0LCB3aWR0aH0gPSB0YWIuZWxlbWVudC5nZXRCb3VuZGluZ0NsaWVudFJlY3QoKVxuICAgIGVsZW1lbnRDZW50ZXIgPSBsZWZ0ICsgd2lkdGggLyAyXG4gICAgZWxlbWVudEluZGV4ID0gdGFicy5pbmRleE9mKHRhYilcblxuICAgIGlmIGV2ZW50LnBhZ2VYIDwgZWxlbWVudENlbnRlclxuICAgICAgZWxlbWVudEluZGV4XG4gICAgZWxzZVxuICAgICAgZWxlbWVudEluZGV4ICsgMVxuXG4gIGdldFBsYWNlaG9sZGVyOiAtPlxuICAgIHJldHVybiBAcGxhY2Vob2xkZXJFbCBpZiBAcGxhY2Vob2xkZXJFbD9cblxuICAgIEBwbGFjZWhvbGRlckVsID0gZG9jdW1lbnQuY3JlYXRlRWxlbWVudChcImxpXCIpXG4gICAgQHBsYWNlaG9sZGVyRWwuY2xhc3NMaXN0LmFkZChcInBsYWNlaG9sZGVyXCIpXG4gICAgQHBsYWNlaG9sZGVyRWxcblxuICByZW1vdmVQbGFjZWhvbGRlcjogLT5cbiAgICBAcGxhY2Vob2xkZXJFbD8ucmVtb3ZlKClcbiAgICBAcGxhY2Vob2xkZXJFbCA9IG51bGxcblxuICBpc1BsYWNlaG9sZGVyOiAoZWxlbWVudCkgLT5cbiAgICBlbGVtZW50LmNsYXNzTGlzdC5jb250YWlucygncGxhY2Vob2xkZXInKVxuXG4gIG9uTW91c2VFbnRlcjogLT5cbiAgICBmb3IgdGFiIGluIEBnZXRUYWJzKClcbiAgICAgIHt3aWR0aH0gPSB0YWIuZWxlbWVudC5nZXRCb3VuZGluZ0NsaWVudFJlY3QoKVxuICAgICAgdGFiLmVsZW1lbnQuc3R5bGUubWF4V2lkdGggPSB3aWR0aC50b0ZpeGVkKDIpICsgJ3B4J1xuICAgIHJldHVyblxuXG4gIG9uTW91c2VMZWF2ZTogLT5cbiAgICB0YWIuZWxlbWVudC5zdHlsZS5tYXhXaWR0aCA9ICcnIGZvciB0YWIgaW4gQGdldFRhYnMoKVxuICAgIHJldHVyblxuXG4gIHRhYkZvckVsZW1lbnQ6IChlbGVtZW50KSAtPlxuICAgIGN1cnJlbnRFbGVtZW50ID0gZWxlbWVudFxuICAgIHdoaWxlIGN1cnJlbnRFbGVtZW50P1xuICAgICAgaWYgdGFiID0gQHRhYnNCeUVsZW1lbnQuZ2V0KGN1cnJlbnRFbGVtZW50KVxuICAgICAgICByZXR1cm4gdGFiXG4gICAgICBlbHNlXG4gICAgICAgIGN1cnJlbnRFbGVtZW50ID0gY3VycmVudEVsZW1lbnQucGFyZW50RWxlbWVudFxuXG4gIGlzQXRvbVRhYkV2ZW50OiAoZXZlbnQpIC0+XG4gICAgZm9yIGl0ZW0gaW4gZXZlbnQuZGF0YVRyYW5zZmVyLml0ZW1zXG4gICAgICBpZiBpdGVtLnR5cGUgaXMgJ2F0b20tdGFiLWV2ZW50J1xuICAgICAgICByZXR1cm4gdHJ1ZVxuXG4gICAgcmV0dXJuIGZhbHNlXG5cbiAgaXRlbUlzQWxsb3dlZDogKGV2ZW50LCBsb2NhdGlvbikgLT5cbiAgICBmb3IgaXRlbSBpbiBldmVudC5kYXRhVHJhbnNmZXIuaXRlbXNcbiAgICAgIGlmIGl0ZW0udHlwZSBpcyAnYWxsb3ctYWxsLWxvY2F0aW9ucycgb3IgaXRlbS50eXBlIGlzIFwiYWxsb3dlZC1sb2NhdGlvbi0je2xvY2F0aW9ufVwiXG4gICAgICAgIHJldHVybiB0cnVlXG5cbiAgICByZXR1cm4gZmFsc2VcbiJdfQ==
