"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = exports.source = void 0;

var _yubikiri = _interopRequireDefault(require("yubikiri"));

var _eventKit = require("event-kit");

var _relayNetworkLayerManager = _interopRequireDefault(require("../relay-network-layer-manager"));

var _author = _interopRequireWildcard(require("./author"));

var _keytarStrategy = require("../shared/keytar-strategy");

var _modelObserver = _interopRequireDefault(require("./model-observer"));

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj["default"] = obj; return newObj; } }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

// This is a guess about what a reasonable value is. Can adjust if performance is poor.
const MAX_COMMITS = 5000;
const source = {
  PENDING: Symbol('pending'),
  GITLOG: Symbol('git log'),
  GITHUBAPI: Symbol('github API')
};
exports.source = source;

class GraphQLCache {
  // One hour
  constructor() {
    this.bySlug = new Map();
  }

  get(remote) {
    const slug = remote.getSlug();
    const {
      ts,
      data
    } = this.bySlug.get(slug) || {
      ts: -Infinity,
      data: {}
    };

    if (Date.now() - ts > this.constructor.MAX_AGE_MS) {
      this.bySlug["delete"](slug);
      return null;
    }

    return data;
  }

  set(remote, data) {
    this.bySlug.set(remote.getSlug(), {
      ts: Date.now(),
      data
    });
  }

}

_defineProperty(GraphQLCache, "MAX_AGE_MS", 3.6e6);

class UserStore {
  constructor({
    repository,
    login,
    config
  }) {
    this.emitter = new _eventKit.Emitter();
    this.subs = new _eventKit.CompositeDisposable(); // TODO: [ku 3/2018] Consider using Dexie (indexDB wrapper) like Desktop and persist users across sessions

    this.allUsers = new Map();
    this.excludedUsers = new Set();
    this.users = [];
    this.committer = _author.nullAuthor;
    this.last = {
      source: source.PENDING,
      repository: null,
      excludedUsers: this.excludedUsers
    };
    this.cache = new GraphQLCache();
    this.repositoryObserver = new _modelObserver["default"]({
      fetchData: r => (0, _yubikiri["default"])({
        committer: r.getCommitter(),
        authors: r.getAuthors({
          max: MAX_COMMITS
        }),
        remotes: r.getRemotes()
      }),
      didUpdate: () => this.loadUsers()
    });
    this.repositoryObserver.setActiveModel(repository);
    this.loginObserver = new _modelObserver["default"]({
      didUpdate: () => this.loadUsers()
    });
    this.loginObserver.setActiveModel(login);
    this.subs.add(config.observe('github.excludedUsers', value => {
      this.excludedUsers = new Set((value || '').split(/\s*,\s*/).filter(each => each.length > 0));
      return this.loadUsers();
    }));
  }

  dispose() {
    this.subs.dispose();
    this.emitter.dispose();
  }

  async loadUsers() {
    const data = this.repositoryObserver.getActiveModelData();

    if (!data) {
      return;
    }

    this.setCommitter(data.committer);
    const githubRemotes = Array.from(data.remotes).filter(remote => remote.isGithubRepo());

    if (githubRemotes.length > 0) {
      await this.loadUsersFromGraphQL(githubRemotes);
    } else {
      this.addUsers(data.authors, source.GITLOG);
    } // if for whatever reason, no committers can be added, fall back to
    // using git log committers as the last resort


    if (this.allUsers.size === 0) {
      this.addUsers(data.authors, source.GITLOG);
    }
  }

  loadUsersFromGraphQL(remotes) {
    return Promise.all(Array.from(remotes, remote => this.loadMentionableUsers(remote)));
  }

  async getToken(loginModel, loginAccount) {
    if (!loginModel) {
      return null;
    }

    const token = await loginModel.getToken(loginAccount);

    if (token === _keytarStrategy.UNAUTHENTICATED || token === _keytarStrategy.INSUFFICIENT) {
      return null;
    }

    return token;
  }

  async loadMentionableUsers(remote) {
    const cached = this.cache.get(remote);

    if (cached !== null) {
      this.addUsers(cached, source.GITHUBAPI);
      return;
    }

    const endpoint = remote.getEndpoint();
    const token = await this.getToken(this.loginObserver.getActiveModel(), endpoint.getLoginAccount());

    if (!token) {
      return;
    }

    const fetchQuery = _relayNetworkLayerManager["default"].getFetchQuery(endpoint, token);

    let hasMore = true;
    let cursor = null;
    const remoteUsers = [];

    while (hasMore) {
      const response = await fetchQuery({
        name: 'GetMentionableUsers',
        text: `
          query GetMentionableUsers($owner: String!, $name: String!, $first: Int!, $after: String) {
            repository(owner: $owner, name: $name) {
              mentionableUsers(first: $first, after: $after) {
                nodes {
                  login
                  email
                  name
                }
                pageInfo {
                  hasNextPage
                  endCursor
                }
              }
            }
          }
        `
      }, {
        owner: remote.getOwner(),
        name: remote.getRepo(),
        first: 100,
        after: cursor
      });

      if (response.errors && response.errors.length > 1) {
        // eslint-disable-next-line no-console
        console.error(`Error fetching mentionable users:\n${response.errors.map(e => e.message).join('\n')}`);
      }

      if (!response.data || !response.data.repository) {
        break;
      }

      const connection = response.data.repository.mentionableUsers;
      const authors = connection.nodes.map(node => {
        if (node.email === '') {
          node.email = `${node.login}@users.noreply.github.com`;
        }

        return new _author["default"](node.email, node.name, node.login);
      });
      this.addUsers(authors, source.GITHUBAPI);
      remoteUsers.push(...authors);
      cursor = connection.pageInfo.endCursor;
      hasMore = connection.pageInfo.hasNextPage;
    }

    this.cache.set(remote, remoteUsers);
  }

  addUsers(users, nextSource) {
    let changed = false;

    if (nextSource !== this.last.source || this.repositoryObserver.getActiveModel() !== this.last.repository || this.excludedUsers !== this.last.excludedUsers) {
      changed = true;
      this.allUsers.clear();
    }

    for (const author of users) {
      if (!this.allUsers.has(author.getEmail())) {
        changed = true;
      }

      this.allUsers.set(author.getEmail(), author);
    }

    if (changed) {
      this.finalize();
    }

    this.last.source = nextSource;
    this.last.repository = this.repositoryObserver.getActiveModel();
    this.last.excludedUsers = this.excludedUsers;
  }

  finalize() {
    // TODO: [ku 3/2018] consider sorting based on most recent authors or commit frequency
    const users = [];

    for (const author of this.allUsers.values()) {
      if (author.matches(this.committer)) {
        continue;
      }

      if (author.isNoReply()) {
        continue;
      }

      if (this.excludedUsers.has(author.getEmail())) {
        continue;
      }

      users.push(author);
    }

    users.sort(_author["default"].compare);
    this.users = users;
    this.didUpdate();
  }

  setRepository(repository) {
    this.repositoryObserver.setActiveModel(repository);
  }

  setLoginModel(login) {
    this.loginObserver.setActiveModel(login);
  }

  setCommitter(committer) {
    const changed = !this.committer.matches(committer);
    this.committer = committer;

    if (changed) {
      this.finalize();
    }
  }

  didUpdate() {
    this.emitter.emit('did-update', this.getUsers());
  }

  onDidUpdate(callback) {
    return this.emitter.on('did-update', callback);
  }

  getUsers() {
    return this.users;
  }

}

exports["default"] = UserStore;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbInVzZXItc3RvcmUuanMiXSwibmFtZXMiOlsiTUFYX0NPTU1JVFMiLCJzb3VyY2UiLCJQRU5ESU5HIiwiU3ltYm9sIiwiR0lUTE9HIiwiR0lUSFVCQVBJIiwiR3JhcGhRTENhY2hlIiwiY29uc3RydWN0b3IiLCJieVNsdWciLCJNYXAiLCJnZXQiLCJyZW1vdGUiLCJzbHVnIiwiZ2V0U2x1ZyIsInRzIiwiZGF0YSIsIkluZmluaXR5IiwiRGF0ZSIsIm5vdyIsIk1BWF9BR0VfTVMiLCJzZXQiLCJVc2VyU3RvcmUiLCJyZXBvc2l0b3J5IiwibG9naW4iLCJjb25maWciLCJlbWl0dGVyIiwiRW1pdHRlciIsInN1YnMiLCJDb21wb3NpdGVEaXNwb3NhYmxlIiwiYWxsVXNlcnMiLCJleGNsdWRlZFVzZXJzIiwiU2V0IiwidXNlcnMiLCJjb21taXR0ZXIiLCJudWxsQXV0aG9yIiwibGFzdCIsImNhY2hlIiwicmVwb3NpdG9yeU9ic2VydmVyIiwiTW9kZWxPYnNlcnZlciIsImZldGNoRGF0YSIsInIiLCJnZXRDb21taXR0ZXIiLCJhdXRob3JzIiwiZ2V0QXV0aG9ycyIsIm1heCIsInJlbW90ZXMiLCJnZXRSZW1vdGVzIiwiZGlkVXBkYXRlIiwibG9hZFVzZXJzIiwic2V0QWN0aXZlTW9kZWwiLCJsb2dpbk9ic2VydmVyIiwiYWRkIiwib2JzZXJ2ZSIsInZhbHVlIiwic3BsaXQiLCJmaWx0ZXIiLCJlYWNoIiwibGVuZ3RoIiwiZGlzcG9zZSIsImdldEFjdGl2ZU1vZGVsRGF0YSIsInNldENvbW1pdHRlciIsImdpdGh1YlJlbW90ZXMiLCJBcnJheSIsImZyb20iLCJpc0dpdGh1YlJlcG8iLCJsb2FkVXNlcnNGcm9tR3JhcGhRTCIsImFkZFVzZXJzIiwic2l6ZSIsIlByb21pc2UiLCJhbGwiLCJsb2FkTWVudGlvbmFibGVVc2VycyIsImdldFRva2VuIiwibG9naW5Nb2RlbCIsImxvZ2luQWNjb3VudCIsInRva2VuIiwiVU5BVVRIRU5USUNBVEVEIiwiSU5TVUZGSUNJRU5UIiwiY2FjaGVkIiwiZW5kcG9pbnQiLCJnZXRFbmRwb2ludCIsImdldEFjdGl2ZU1vZGVsIiwiZ2V0TG9naW5BY2NvdW50IiwiZmV0Y2hRdWVyeSIsIlJlbGF5TmV0d29ya0xheWVyTWFuYWdlciIsImdldEZldGNoUXVlcnkiLCJoYXNNb3JlIiwiY3Vyc29yIiwicmVtb3RlVXNlcnMiLCJyZXNwb25zZSIsIm5hbWUiLCJ0ZXh0Iiwib3duZXIiLCJnZXRPd25lciIsImdldFJlcG8iLCJmaXJzdCIsImFmdGVyIiwiZXJyb3JzIiwiY29uc29sZSIsImVycm9yIiwibWFwIiwiZSIsIm1lc3NhZ2UiLCJqb2luIiwiY29ubmVjdGlvbiIsIm1lbnRpb25hYmxlVXNlcnMiLCJub2RlcyIsIm5vZGUiLCJlbWFpbCIsIkF1dGhvciIsInB1c2giLCJwYWdlSW5mbyIsImVuZEN1cnNvciIsImhhc05leHRQYWdlIiwibmV4dFNvdXJjZSIsImNoYW5nZWQiLCJjbGVhciIsImF1dGhvciIsImhhcyIsImdldEVtYWlsIiwiZmluYWxpemUiLCJ2YWx1ZXMiLCJtYXRjaGVzIiwiaXNOb1JlcGx5Iiwic29ydCIsImNvbXBhcmUiLCJzZXRSZXBvc2l0b3J5Iiwic2V0TG9naW5Nb2RlbCIsImVtaXQiLCJnZXRVc2VycyIsIm9uRGlkVXBkYXRlIiwiY2FsbGJhY2siLCJvbiJdLCJtYXBwaW5ncyI6Ijs7Ozs7OztBQUFBOztBQUNBOztBQUVBOztBQUNBOztBQUNBOztBQUNBOzs7Ozs7OztBQUVBO0FBQ0EsTUFBTUEsV0FBVyxHQUFHLElBQXBCO0FBRU8sTUFBTUMsTUFBTSxHQUFHO0FBQ3BCQyxFQUFBQSxPQUFPLEVBQUVDLE1BQU0sQ0FBQyxTQUFELENBREs7QUFFcEJDLEVBQUFBLE1BQU0sRUFBRUQsTUFBTSxDQUFDLFNBQUQsQ0FGTTtBQUdwQkUsRUFBQUEsU0FBUyxFQUFFRixNQUFNLENBQUMsWUFBRDtBQUhHLENBQWY7OztBQU1QLE1BQU1HLFlBQU4sQ0FBbUI7QUFDakI7QUFHQUMsRUFBQUEsV0FBVyxHQUFHO0FBQ1osU0FBS0MsTUFBTCxHQUFjLElBQUlDLEdBQUosRUFBZDtBQUNEOztBQUVEQyxFQUFBQSxHQUFHLENBQUNDLE1BQUQsRUFBUztBQUNWLFVBQU1DLElBQUksR0FBR0QsTUFBTSxDQUFDRSxPQUFQLEVBQWI7QUFDQSxVQUFNO0FBQUNDLE1BQUFBLEVBQUQ7QUFBS0MsTUFBQUE7QUFBTCxRQUFhLEtBQUtQLE1BQUwsQ0FBWUUsR0FBWixDQUFnQkUsSUFBaEIsS0FBeUI7QUFDMUNFLE1BQUFBLEVBQUUsRUFBRSxDQUFDRSxRQURxQztBQUUxQ0QsTUFBQUEsSUFBSSxFQUFFO0FBRm9DLEtBQTVDOztBQUtBLFFBQUlFLElBQUksQ0FBQ0MsR0FBTCxLQUFhSixFQUFiLEdBQWtCLEtBQUtQLFdBQUwsQ0FBaUJZLFVBQXZDLEVBQW1EO0FBQ2pELFdBQUtYLE1BQUwsV0FBbUJJLElBQW5CO0FBQ0EsYUFBTyxJQUFQO0FBQ0Q7O0FBQ0QsV0FBT0csSUFBUDtBQUNEOztBQUVESyxFQUFBQSxHQUFHLENBQUNULE1BQUQsRUFBU0ksSUFBVCxFQUFlO0FBQ2hCLFNBQUtQLE1BQUwsQ0FBWVksR0FBWixDQUFnQlQsTUFBTSxDQUFDRSxPQUFQLEVBQWhCLEVBQWtDO0FBQUNDLE1BQUFBLEVBQUUsRUFBRUcsSUFBSSxDQUFDQyxHQUFMLEVBQUw7QUFBaUJILE1BQUFBO0FBQWpCLEtBQWxDO0FBQ0Q7O0FBeEJnQjs7Z0JBQWJULFksZ0JBRWdCLEs7O0FBeUJQLE1BQU1lLFNBQU4sQ0FBZ0I7QUFDN0JkLEVBQUFBLFdBQVcsQ0FBQztBQUFDZSxJQUFBQSxVQUFEO0FBQWFDLElBQUFBLEtBQWI7QUFBb0JDLElBQUFBO0FBQXBCLEdBQUQsRUFBOEI7QUFDdkMsU0FBS0MsT0FBTCxHQUFlLElBQUlDLGlCQUFKLEVBQWY7QUFDQSxTQUFLQyxJQUFMLEdBQVksSUFBSUMsNkJBQUosRUFBWixDQUZ1QyxDQUl2Qzs7QUFDQSxTQUFLQyxRQUFMLEdBQWdCLElBQUlwQixHQUFKLEVBQWhCO0FBQ0EsU0FBS3FCLGFBQUwsR0FBcUIsSUFBSUMsR0FBSixFQUFyQjtBQUNBLFNBQUtDLEtBQUwsR0FBYSxFQUFiO0FBQ0EsU0FBS0MsU0FBTCxHQUFpQkMsa0JBQWpCO0FBRUEsU0FBS0MsSUFBTCxHQUFZO0FBQ1ZsQyxNQUFBQSxNQUFNLEVBQUVBLE1BQU0sQ0FBQ0MsT0FETDtBQUVWb0IsTUFBQUEsVUFBVSxFQUFFLElBRkY7QUFHVlEsTUFBQUEsYUFBYSxFQUFFLEtBQUtBO0FBSFYsS0FBWjtBQUtBLFNBQUtNLEtBQUwsR0FBYSxJQUFJOUIsWUFBSixFQUFiO0FBRUEsU0FBSytCLGtCQUFMLEdBQTBCLElBQUlDLHlCQUFKLENBQWtCO0FBQzFDQyxNQUFBQSxTQUFTLEVBQUVDLENBQUMsSUFBSSwwQkFBUztBQUN2QlAsUUFBQUEsU0FBUyxFQUFFTyxDQUFDLENBQUNDLFlBQUYsRUFEWTtBQUV2QkMsUUFBQUEsT0FBTyxFQUFFRixDQUFDLENBQUNHLFVBQUYsQ0FBYTtBQUFDQyxVQUFBQSxHQUFHLEVBQUU1QztBQUFOLFNBQWIsQ0FGYztBQUd2QjZDLFFBQUFBLE9BQU8sRUFBRUwsQ0FBQyxDQUFDTSxVQUFGO0FBSGMsT0FBVCxDQUQwQjtBQU0xQ0MsTUFBQUEsU0FBUyxFQUFFLE1BQU0sS0FBS0MsU0FBTDtBQU55QixLQUFsQixDQUExQjtBQVFBLFNBQUtYLGtCQUFMLENBQXdCWSxjQUF4QixDQUF1QzNCLFVBQXZDO0FBRUEsU0FBSzRCLGFBQUwsR0FBcUIsSUFBSVoseUJBQUosQ0FBa0I7QUFDckNTLE1BQUFBLFNBQVMsRUFBRSxNQUFNLEtBQUtDLFNBQUw7QUFEb0IsS0FBbEIsQ0FBckI7QUFHQSxTQUFLRSxhQUFMLENBQW1CRCxjQUFuQixDQUFrQzFCLEtBQWxDO0FBRUEsU0FBS0ksSUFBTCxDQUFVd0IsR0FBVixDQUNFM0IsTUFBTSxDQUFDNEIsT0FBUCxDQUFlLHNCQUFmLEVBQXVDQyxLQUFLLElBQUk7QUFDOUMsV0FBS3ZCLGFBQUwsR0FBcUIsSUFBSUMsR0FBSixDQUNuQixDQUFDc0IsS0FBSyxJQUFJLEVBQVYsRUFBY0MsS0FBZCxDQUFvQixTQUFwQixFQUErQkMsTUFBL0IsQ0FBc0NDLElBQUksSUFBSUEsSUFBSSxDQUFDQyxNQUFMLEdBQWMsQ0FBNUQsQ0FEbUIsQ0FBckI7QUFHQSxhQUFPLEtBQUtULFNBQUwsRUFBUDtBQUNELEtBTEQsQ0FERjtBQVFEOztBQUVEVSxFQUFBQSxPQUFPLEdBQUc7QUFDUixTQUFLL0IsSUFBTCxDQUFVK0IsT0FBVjtBQUNBLFNBQUtqQyxPQUFMLENBQWFpQyxPQUFiO0FBQ0Q7O0FBRUQsUUFBTVYsU0FBTixHQUFrQjtBQUNoQixVQUFNakMsSUFBSSxHQUFHLEtBQUtzQixrQkFBTCxDQUF3QnNCLGtCQUF4QixFQUFiOztBQUVBLFFBQUksQ0FBQzVDLElBQUwsRUFBVztBQUNUO0FBQ0Q7O0FBRUQsU0FBSzZDLFlBQUwsQ0FBa0I3QyxJQUFJLENBQUNrQixTQUF2QjtBQUNBLFVBQU00QixhQUFhLEdBQUdDLEtBQUssQ0FBQ0MsSUFBTixDQUFXaEQsSUFBSSxDQUFDOEIsT0FBaEIsRUFBeUJVLE1BQXpCLENBQWdDNUMsTUFBTSxJQUFJQSxNQUFNLENBQUNxRCxZQUFQLEVBQTFDLENBQXRCOztBQUVBLFFBQUlILGFBQWEsQ0FBQ0osTUFBZCxHQUF1QixDQUEzQixFQUE4QjtBQUM1QixZQUFNLEtBQUtRLG9CQUFMLENBQTBCSixhQUExQixDQUFOO0FBQ0QsS0FGRCxNQUVPO0FBQ0wsV0FBS0ssUUFBTCxDQUFjbkQsSUFBSSxDQUFDMkIsT0FBbkIsRUFBNEJ6QyxNQUFNLENBQUNHLE1BQW5DO0FBQ0QsS0FkZSxDQWdCaEI7QUFDQTs7O0FBQ0EsUUFBSSxLQUFLeUIsUUFBTCxDQUFjc0MsSUFBZCxLQUF1QixDQUEzQixFQUE4QjtBQUM1QixXQUFLRCxRQUFMLENBQWNuRCxJQUFJLENBQUMyQixPQUFuQixFQUE0QnpDLE1BQU0sQ0FBQ0csTUFBbkM7QUFDRDtBQUNGOztBQUVENkQsRUFBQUEsb0JBQW9CLENBQUNwQixPQUFELEVBQVU7QUFDNUIsV0FBT3VCLE9BQU8sQ0FBQ0MsR0FBUixDQUNMUCxLQUFLLENBQUNDLElBQU4sQ0FBV2xCLE9BQVgsRUFBb0JsQyxNQUFNLElBQUksS0FBSzJELG9CQUFMLENBQTBCM0QsTUFBMUIsQ0FBOUIsQ0FESyxDQUFQO0FBR0Q7O0FBRUQsUUFBTTRELFFBQU4sQ0FBZUMsVUFBZixFQUEyQkMsWUFBM0IsRUFBeUM7QUFDdkMsUUFBSSxDQUFDRCxVQUFMLEVBQWlCO0FBQ2YsYUFBTyxJQUFQO0FBQ0Q7O0FBQ0QsVUFBTUUsS0FBSyxHQUFHLE1BQU1GLFVBQVUsQ0FBQ0QsUUFBWCxDQUFvQkUsWUFBcEIsQ0FBcEI7O0FBQ0EsUUFBSUMsS0FBSyxLQUFLQywrQkFBVixJQUE2QkQsS0FBSyxLQUFLRSw0QkFBM0MsRUFBeUQ7QUFDdkQsYUFBTyxJQUFQO0FBQ0Q7O0FBQ0QsV0FBT0YsS0FBUDtBQUNEOztBQUVELFFBQU1KLG9CQUFOLENBQTJCM0QsTUFBM0IsRUFBbUM7QUFDakMsVUFBTWtFLE1BQU0sR0FBRyxLQUFLekMsS0FBTCxDQUFXMUIsR0FBWCxDQUFlQyxNQUFmLENBQWY7O0FBQ0EsUUFBSWtFLE1BQU0sS0FBSyxJQUFmLEVBQXFCO0FBQ25CLFdBQUtYLFFBQUwsQ0FBY1csTUFBZCxFQUFzQjVFLE1BQU0sQ0FBQ0ksU0FBN0I7QUFDQTtBQUNEOztBQUVELFVBQU15RSxRQUFRLEdBQUduRSxNQUFNLENBQUNvRSxXQUFQLEVBQWpCO0FBQ0EsVUFBTUwsS0FBSyxHQUFHLE1BQU0sS0FBS0gsUUFBTCxDQUFjLEtBQUtyQixhQUFMLENBQW1COEIsY0FBbkIsRUFBZCxFQUFtREYsUUFBUSxDQUFDRyxlQUFULEVBQW5ELENBQXBCOztBQUNBLFFBQUksQ0FBQ1AsS0FBTCxFQUFZO0FBQ1Y7QUFDRDs7QUFFRCxVQUFNUSxVQUFVLEdBQUdDLHFDQUF5QkMsYUFBekIsQ0FBdUNOLFFBQXZDLEVBQWlESixLQUFqRCxDQUFuQjs7QUFFQSxRQUFJVyxPQUFPLEdBQUcsSUFBZDtBQUNBLFFBQUlDLE1BQU0sR0FBRyxJQUFiO0FBQ0EsVUFBTUMsV0FBVyxHQUFHLEVBQXBCOztBQUVBLFdBQU9GLE9BQVAsRUFBZ0I7QUFDZCxZQUFNRyxRQUFRLEdBQUcsTUFBTU4sVUFBVSxDQUFDO0FBQ2hDTyxRQUFBQSxJQUFJLEVBQUUscUJBRDBCO0FBRWhDQyxRQUFBQSxJQUFJLEVBQUc7Ozs7Ozs7Ozs7Ozs7Ozs7O0FBRnlCLE9BQUQsRUFtQjlCO0FBQ0RDLFFBQUFBLEtBQUssRUFBRWhGLE1BQU0sQ0FBQ2lGLFFBQVAsRUFETjtBQUVESCxRQUFBQSxJQUFJLEVBQUU5RSxNQUFNLENBQUNrRixPQUFQLEVBRkw7QUFHREMsUUFBQUEsS0FBSyxFQUFFLEdBSE47QUFJREMsUUFBQUEsS0FBSyxFQUFFVDtBQUpOLE9BbkI4QixDQUFqQzs7QUEwQkEsVUFBSUUsUUFBUSxDQUFDUSxNQUFULElBQW1CUixRQUFRLENBQUNRLE1BQVQsQ0FBZ0J2QyxNQUFoQixHQUF5QixDQUFoRCxFQUFtRDtBQUNqRDtBQUNBd0MsUUFBQUEsT0FBTyxDQUFDQyxLQUFSLENBQWUsc0NBQXFDVixRQUFRLENBQUNRLE1BQVQsQ0FBZ0JHLEdBQWhCLENBQW9CQyxDQUFDLElBQUlBLENBQUMsQ0FBQ0MsT0FBM0IsRUFBb0NDLElBQXBDLENBQXlDLElBQXpDLENBQStDLEVBQW5HO0FBQ0Q7O0FBRUQsVUFBSSxDQUFDZCxRQUFRLENBQUN6RSxJQUFWLElBQWtCLENBQUN5RSxRQUFRLENBQUN6RSxJQUFULENBQWNPLFVBQXJDLEVBQWlEO0FBQy9DO0FBQ0Q7O0FBRUQsWUFBTWlGLFVBQVUsR0FBR2YsUUFBUSxDQUFDekUsSUFBVCxDQUFjTyxVQUFkLENBQXlCa0YsZ0JBQTVDO0FBQ0EsWUFBTTlELE9BQU8sR0FBRzZELFVBQVUsQ0FBQ0UsS0FBWCxDQUFpQk4sR0FBakIsQ0FBcUJPLElBQUksSUFBSTtBQUMzQyxZQUFJQSxJQUFJLENBQUNDLEtBQUwsS0FBZSxFQUFuQixFQUF1QjtBQUNyQkQsVUFBQUEsSUFBSSxDQUFDQyxLQUFMLEdBQWMsR0FBRUQsSUFBSSxDQUFDbkYsS0FBTSwyQkFBM0I7QUFDRDs7QUFFRCxlQUFPLElBQUlxRixrQkFBSixDQUFXRixJQUFJLENBQUNDLEtBQWhCLEVBQXVCRCxJQUFJLENBQUNqQixJQUE1QixFQUFrQ2lCLElBQUksQ0FBQ25GLEtBQXZDLENBQVA7QUFDRCxPQU5lLENBQWhCO0FBT0EsV0FBSzJDLFFBQUwsQ0FBY3hCLE9BQWQsRUFBdUJ6QyxNQUFNLENBQUNJLFNBQTlCO0FBQ0FrRixNQUFBQSxXQUFXLENBQUNzQixJQUFaLENBQWlCLEdBQUduRSxPQUFwQjtBQUVBNEMsTUFBQUEsTUFBTSxHQUFHaUIsVUFBVSxDQUFDTyxRQUFYLENBQW9CQyxTQUE3QjtBQUNBMUIsTUFBQUEsT0FBTyxHQUFHa0IsVUFBVSxDQUFDTyxRQUFYLENBQW9CRSxXQUE5QjtBQUNEOztBQUVELFNBQUs1RSxLQUFMLENBQVdoQixHQUFYLENBQWVULE1BQWYsRUFBdUI0RSxXQUF2QjtBQUNEOztBQUVEckIsRUFBQUEsUUFBUSxDQUFDbEMsS0FBRCxFQUFRaUYsVUFBUixFQUFvQjtBQUMxQixRQUFJQyxPQUFPLEdBQUcsS0FBZDs7QUFFQSxRQUNFRCxVQUFVLEtBQUssS0FBSzlFLElBQUwsQ0FBVWxDLE1BQXpCLElBQ0EsS0FBS29DLGtCQUFMLENBQXdCMkMsY0FBeEIsT0FBNkMsS0FBSzdDLElBQUwsQ0FBVWIsVUFEdkQsSUFFQSxLQUFLUSxhQUFMLEtBQXVCLEtBQUtLLElBQUwsQ0FBVUwsYUFIbkMsRUFJRTtBQUNBb0YsTUFBQUEsT0FBTyxHQUFHLElBQVY7QUFDQSxXQUFLckYsUUFBTCxDQUFjc0YsS0FBZDtBQUNEOztBQUVELFNBQUssTUFBTUMsTUFBWCxJQUFxQnBGLEtBQXJCLEVBQTRCO0FBQzFCLFVBQUksQ0FBQyxLQUFLSCxRQUFMLENBQWN3RixHQUFkLENBQWtCRCxNQUFNLENBQUNFLFFBQVAsRUFBbEIsQ0FBTCxFQUEyQztBQUN6Q0osUUFBQUEsT0FBTyxHQUFHLElBQVY7QUFDRDs7QUFDRCxXQUFLckYsUUFBTCxDQUFjVCxHQUFkLENBQWtCZ0csTUFBTSxDQUFDRSxRQUFQLEVBQWxCLEVBQXFDRixNQUFyQztBQUNEOztBQUVELFFBQUlGLE9BQUosRUFBYTtBQUNYLFdBQUtLLFFBQUw7QUFDRDs7QUFDRCxTQUFLcEYsSUFBTCxDQUFVbEMsTUFBVixHQUFtQmdILFVBQW5CO0FBQ0EsU0FBSzlFLElBQUwsQ0FBVWIsVUFBVixHQUF1QixLQUFLZSxrQkFBTCxDQUF3QjJDLGNBQXhCLEVBQXZCO0FBQ0EsU0FBSzdDLElBQUwsQ0FBVUwsYUFBVixHQUEwQixLQUFLQSxhQUEvQjtBQUNEOztBQUVEeUYsRUFBQUEsUUFBUSxHQUFHO0FBQ1Q7QUFDQSxVQUFNdkYsS0FBSyxHQUFHLEVBQWQ7O0FBQ0EsU0FBSyxNQUFNb0YsTUFBWCxJQUFxQixLQUFLdkYsUUFBTCxDQUFjMkYsTUFBZCxFQUFyQixFQUE2QztBQUMzQyxVQUFJSixNQUFNLENBQUNLLE9BQVAsQ0FBZSxLQUFLeEYsU0FBcEIsQ0FBSixFQUFvQztBQUFFO0FBQVc7O0FBQ2pELFVBQUltRixNQUFNLENBQUNNLFNBQVAsRUFBSixFQUF3QjtBQUFFO0FBQVc7O0FBQ3JDLFVBQUksS0FBSzVGLGFBQUwsQ0FBbUJ1RixHQUFuQixDQUF1QkQsTUFBTSxDQUFDRSxRQUFQLEVBQXZCLENBQUosRUFBK0M7QUFBRTtBQUFXOztBQUU1RHRGLE1BQUFBLEtBQUssQ0FBQzZFLElBQU4sQ0FBV08sTUFBWDtBQUNEOztBQUNEcEYsSUFBQUEsS0FBSyxDQUFDMkYsSUFBTixDQUFXZixtQkFBT2dCLE9BQWxCO0FBQ0EsU0FBSzVGLEtBQUwsR0FBYUEsS0FBYjtBQUNBLFNBQUtlLFNBQUw7QUFDRDs7QUFFRDhFLEVBQUFBLGFBQWEsQ0FBQ3ZHLFVBQUQsRUFBYTtBQUN4QixTQUFLZSxrQkFBTCxDQUF3QlksY0FBeEIsQ0FBdUMzQixVQUF2QztBQUNEOztBQUVEd0csRUFBQUEsYUFBYSxDQUFDdkcsS0FBRCxFQUFRO0FBQ25CLFNBQUsyQixhQUFMLENBQW1CRCxjQUFuQixDQUFrQzFCLEtBQWxDO0FBQ0Q7O0FBRURxQyxFQUFBQSxZQUFZLENBQUMzQixTQUFELEVBQVk7QUFDdEIsVUFBTWlGLE9BQU8sR0FBRyxDQUFDLEtBQUtqRixTQUFMLENBQWV3RixPQUFmLENBQXVCeEYsU0FBdkIsQ0FBakI7QUFDQSxTQUFLQSxTQUFMLEdBQWlCQSxTQUFqQjs7QUFDQSxRQUFJaUYsT0FBSixFQUFhO0FBQ1gsV0FBS0ssUUFBTDtBQUNEO0FBQ0Y7O0FBRUR4RSxFQUFBQSxTQUFTLEdBQUc7QUFDVixTQUFLdEIsT0FBTCxDQUFhc0csSUFBYixDQUFrQixZQUFsQixFQUFnQyxLQUFLQyxRQUFMLEVBQWhDO0FBQ0Q7O0FBRURDLEVBQUFBLFdBQVcsQ0FBQ0MsUUFBRCxFQUFXO0FBQ3BCLFdBQU8sS0FBS3pHLE9BQUwsQ0FBYTBHLEVBQWIsQ0FBZ0IsWUFBaEIsRUFBOEJELFFBQTlCLENBQVA7QUFDRDs7QUFFREYsRUFBQUEsUUFBUSxHQUFHO0FBQ1QsV0FBTyxLQUFLaEcsS0FBWjtBQUNEOztBQXJPNEIiLCJzb3VyY2VSb290IjoiL2J1aWxkL2F0b20vc3JjL2F0b20tMS4zOC4yL291dC9hcHAvbm9kZV9tb2R1bGVzL2dpdGh1YiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB5dWJpa2lyaSBmcm9tICd5dWJpa2lyaSc7XG5pbXBvcnQge0VtaXR0ZXIsIENvbXBvc2l0ZURpc3Bvc2FibGV9IGZyb20gJ2V2ZW50LWtpdCc7XG5cbmltcG9ydCBSZWxheU5ldHdvcmtMYXllck1hbmFnZXIgZnJvbSAnLi4vcmVsYXktbmV0d29yay1sYXllci1tYW5hZ2VyJztcbmltcG9ydCBBdXRob3IsIHtudWxsQXV0aG9yfSBmcm9tICcuL2F1dGhvcic7XG5pbXBvcnQge1VOQVVUSEVOVElDQVRFRCwgSU5TVUZGSUNJRU5UfSBmcm9tICcuLi9zaGFyZWQva2V5dGFyLXN0cmF0ZWd5JztcbmltcG9ydCBNb2RlbE9ic2VydmVyIGZyb20gJy4vbW9kZWwtb2JzZXJ2ZXInO1xuXG4vLyBUaGlzIGlzIGEgZ3Vlc3MgYWJvdXQgd2hhdCBhIHJlYXNvbmFibGUgdmFsdWUgaXMuIENhbiBhZGp1c3QgaWYgcGVyZm9ybWFuY2UgaXMgcG9vci5cbmNvbnN0IE1BWF9DT01NSVRTID0gNTAwMDtcblxuZXhwb3J0IGNvbnN0IHNvdXJjZSA9IHtcbiAgUEVORElORzogU3ltYm9sKCdwZW5kaW5nJyksXG4gIEdJVExPRzogU3ltYm9sKCdnaXQgbG9nJyksXG4gIEdJVEhVQkFQSTogU3ltYm9sKCdnaXRodWIgQVBJJyksXG59O1xuXG5jbGFzcyBHcmFwaFFMQ2FjaGUge1xuICAvLyBPbmUgaG91clxuICBzdGF0aWMgTUFYX0FHRV9NUyA9IDMuNmU2XG5cbiAgY29uc3RydWN0b3IoKSB7XG4gICAgdGhpcy5ieVNsdWcgPSBuZXcgTWFwKCk7XG4gIH1cblxuICBnZXQocmVtb3RlKSB7XG4gICAgY29uc3Qgc2x1ZyA9IHJlbW90ZS5nZXRTbHVnKCk7XG4gICAgY29uc3Qge3RzLCBkYXRhfSA9IHRoaXMuYnlTbHVnLmdldChzbHVnKSB8fCB7XG4gICAgICB0czogLUluZmluaXR5LFxuICAgICAgZGF0YToge30sXG4gICAgfTtcblxuICAgIGlmIChEYXRlLm5vdygpIC0gdHMgPiB0aGlzLmNvbnN0cnVjdG9yLk1BWF9BR0VfTVMpIHtcbiAgICAgIHRoaXMuYnlTbHVnLmRlbGV0ZShzbHVnKTtcbiAgICAgIHJldHVybiBudWxsO1xuICAgIH1cbiAgICByZXR1cm4gZGF0YTtcbiAgfVxuXG4gIHNldChyZW1vdGUsIGRhdGEpIHtcbiAgICB0aGlzLmJ5U2x1Zy5zZXQocmVtb3RlLmdldFNsdWcoKSwge3RzOiBEYXRlLm5vdygpLCBkYXRhfSk7XG4gIH1cbn1cblxuZXhwb3J0IGRlZmF1bHQgY2xhc3MgVXNlclN0b3JlIHtcbiAgY29uc3RydWN0b3Ioe3JlcG9zaXRvcnksIGxvZ2luLCBjb25maWd9KSB7XG4gICAgdGhpcy5lbWl0dGVyID0gbmV3IEVtaXR0ZXIoKTtcbiAgICB0aGlzLnN1YnMgPSBuZXcgQ29tcG9zaXRlRGlzcG9zYWJsZSgpO1xuXG4gICAgLy8gVE9ETzogW2t1IDMvMjAxOF0gQ29uc2lkZXIgdXNpbmcgRGV4aWUgKGluZGV4REIgd3JhcHBlcikgbGlrZSBEZXNrdG9wIGFuZCBwZXJzaXN0IHVzZXJzIGFjcm9zcyBzZXNzaW9uc1xuICAgIHRoaXMuYWxsVXNlcnMgPSBuZXcgTWFwKCk7XG4gICAgdGhpcy5leGNsdWRlZFVzZXJzID0gbmV3IFNldCgpO1xuICAgIHRoaXMudXNlcnMgPSBbXTtcbiAgICB0aGlzLmNvbW1pdHRlciA9IG51bGxBdXRob3I7XG5cbiAgICB0aGlzLmxhc3QgPSB7XG4gICAgICBzb3VyY2U6IHNvdXJjZS5QRU5ESU5HLFxuICAgICAgcmVwb3NpdG9yeTogbnVsbCxcbiAgICAgIGV4Y2x1ZGVkVXNlcnM6IHRoaXMuZXhjbHVkZWRVc2VycyxcbiAgICB9O1xuICAgIHRoaXMuY2FjaGUgPSBuZXcgR3JhcGhRTENhY2hlKCk7XG5cbiAgICB0aGlzLnJlcG9zaXRvcnlPYnNlcnZlciA9IG5ldyBNb2RlbE9ic2VydmVyKHtcbiAgICAgIGZldGNoRGF0YTogciA9PiB5dWJpa2lyaSh7XG4gICAgICAgIGNvbW1pdHRlcjogci5nZXRDb21taXR0ZXIoKSxcbiAgICAgICAgYXV0aG9yczogci5nZXRBdXRob3JzKHttYXg6IE1BWF9DT01NSVRTfSksXG4gICAgICAgIHJlbW90ZXM6IHIuZ2V0UmVtb3RlcygpLFxuICAgICAgfSksXG4gICAgICBkaWRVcGRhdGU6ICgpID0+IHRoaXMubG9hZFVzZXJzKCksXG4gICAgfSk7XG4gICAgdGhpcy5yZXBvc2l0b3J5T2JzZXJ2ZXIuc2V0QWN0aXZlTW9kZWwocmVwb3NpdG9yeSk7XG5cbiAgICB0aGlzLmxvZ2luT2JzZXJ2ZXIgPSBuZXcgTW9kZWxPYnNlcnZlcih7XG4gICAgICBkaWRVcGRhdGU6ICgpID0+IHRoaXMubG9hZFVzZXJzKCksXG4gICAgfSk7XG4gICAgdGhpcy5sb2dpbk9ic2VydmVyLnNldEFjdGl2ZU1vZGVsKGxvZ2luKTtcblxuICAgIHRoaXMuc3Vicy5hZGQoXG4gICAgICBjb25maWcub2JzZXJ2ZSgnZ2l0aHViLmV4Y2x1ZGVkVXNlcnMnLCB2YWx1ZSA9PiB7XG4gICAgICAgIHRoaXMuZXhjbHVkZWRVc2VycyA9IG5ldyBTZXQoXG4gICAgICAgICAgKHZhbHVlIHx8ICcnKS5zcGxpdCgvXFxzKixcXHMqLykuZmlsdGVyKGVhY2ggPT4gZWFjaC5sZW5ndGggPiAwKSxcbiAgICAgICAgKTtcbiAgICAgICAgcmV0dXJuIHRoaXMubG9hZFVzZXJzKCk7XG4gICAgICB9KSxcbiAgICApO1xuICB9XG5cbiAgZGlzcG9zZSgpIHtcbiAgICB0aGlzLnN1YnMuZGlzcG9zZSgpO1xuICAgIHRoaXMuZW1pdHRlci5kaXNwb3NlKCk7XG4gIH1cblxuICBhc3luYyBsb2FkVXNlcnMoKSB7XG4gICAgY29uc3QgZGF0YSA9IHRoaXMucmVwb3NpdG9yeU9ic2VydmVyLmdldEFjdGl2ZU1vZGVsRGF0YSgpO1xuXG4gICAgaWYgKCFkYXRhKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgdGhpcy5zZXRDb21taXR0ZXIoZGF0YS5jb21taXR0ZXIpO1xuICAgIGNvbnN0IGdpdGh1YlJlbW90ZXMgPSBBcnJheS5mcm9tKGRhdGEucmVtb3RlcykuZmlsdGVyKHJlbW90ZSA9PiByZW1vdGUuaXNHaXRodWJSZXBvKCkpO1xuXG4gICAgaWYgKGdpdGh1YlJlbW90ZXMubGVuZ3RoID4gMCkge1xuICAgICAgYXdhaXQgdGhpcy5sb2FkVXNlcnNGcm9tR3JhcGhRTChnaXRodWJSZW1vdGVzKTtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhpcy5hZGRVc2VycyhkYXRhLmF1dGhvcnMsIHNvdXJjZS5HSVRMT0cpO1xuICAgIH1cblxuICAgIC8vIGlmIGZvciB3aGF0ZXZlciByZWFzb24sIG5vIGNvbW1pdHRlcnMgY2FuIGJlIGFkZGVkLCBmYWxsIGJhY2sgdG9cbiAgICAvLyB1c2luZyBnaXQgbG9nIGNvbW1pdHRlcnMgYXMgdGhlIGxhc3QgcmVzb3J0XG4gICAgaWYgKHRoaXMuYWxsVXNlcnMuc2l6ZSA9PT0gMCkge1xuICAgICAgdGhpcy5hZGRVc2VycyhkYXRhLmF1dGhvcnMsIHNvdXJjZS5HSVRMT0cpO1xuICAgIH1cbiAgfVxuXG4gIGxvYWRVc2Vyc0Zyb21HcmFwaFFMKHJlbW90ZXMpIHtcbiAgICByZXR1cm4gUHJvbWlzZS5hbGwoXG4gICAgICBBcnJheS5mcm9tKHJlbW90ZXMsIHJlbW90ZSA9PiB0aGlzLmxvYWRNZW50aW9uYWJsZVVzZXJzKHJlbW90ZSkpLFxuICAgICk7XG4gIH1cblxuICBhc3luYyBnZXRUb2tlbihsb2dpbk1vZGVsLCBsb2dpbkFjY291bnQpIHtcbiAgICBpZiAoIWxvZ2luTW9kZWwpIHtcbiAgICAgIHJldHVybiBudWxsO1xuICAgIH1cbiAgICBjb25zdCB0b2tlbiA9IGF3YWl0IGxvZ2luTW9kZWwuZ2V0VG9rZW4obG9naW5BY2NvdW50KTtcbiAgICBpZiAodG9rZW4gPT09IFVOQVVUSEVOVElDQVRFRCB8fCB0b2tlbiA9PT0gSU5TVUZGSUNJRU5UKSB7XG4gICAgICByZXR1cm4gbnVsbDtcbiAgICB9XG4gICAgcmV0dXJuIHRva2VuO1xuICB9XG5cbiAgYXN5bmMgbG9hZE1lbnRpb25hYmxlVXNlcnMocmVtb3RlKSB7XG4gICAgY29uc3QgY2FjaGVkID0gdGhpcy5jYWNoZS5nZXQocmVtb3RlKTtcbiAgICBpZiAoY2FjaGVkICE9PSBudWxsKSB7XG4gICAgICB0aGlzLmFkZFVzZXJzKGNhY2hlZCwgc291cmNlLkdJVEhVQkFQSSk7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgY29uc3QgZW5kcG9pbnQgPSByZW1vdGUuZ2V0RW5kcG9pbnQoKTtcbiAgICBjb25zdCB0b2tlbiA9IGF3YWl0IHRoaXMuZ2V0VG9rZW4odGhpcy5sb2dpbk9ic2VydmVyLmdldEFjdGl2ZU1vZGVsKCksIGVuZHBvaW50LmdldExvZ2luQWNjb3VudCgpKTtcbiAgICBpZiAoIXRva2VuKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgY29uc3QgZmV0Y2hRdWVyeSA9IFJlbGF5TmV0d29ya0xheWVyTWFuYWdlci5nZXRGZXRjaFF1ZXJ5KGVuZHBvaW50LCB0b2tlbik7XG5cbiAgICBsZXQgaGFzTW9yZSA9IHRydWU7XG4gICAgbGV0IGN1cnNvciA9IG51bGw7XG4gICAgY29uc3QgcmVtb3RlVXNlcnMgPSBbXTtcblxuICAgIHdoaWxlIChoYXNNb3JlKSB7XG4gICAgICBjb25zdCByZXNwb25zZSA9IGF3YWl0IGZldGNoUXVlcnkoe1xuICAgICAgICBuYW1lOiAnR2V0TWVudGlvbmFibGVVc2VycycsXG4gICAgICAgIHRleHQ6IGBcbiAgICAgICAgICBxdWVyeSBHZXRNZW50aW9uYWJsZVVzZXJzKCRvd25lcjogU3RyaW5nISwgJG5hbWU6IFN0cmluZyEsICRmaXJzdDogSW50ISwgJGFmdGVyOiBTdHJpbmcpIHtcbiAgICAgICAgICAgIHJlcG9zaXRvcnkob3duZXI6ICRvd25lciwgbmFtZTogJG5hbWUpIHtcbiAgICAgICAgICAgICAgbWVudGlvbmFibGVVc2VycyhmaXJzdDogJGZpcnN0LCBhZnRlcjogJGFmdGVyKSB7XG4gICAgICAgICAgICAgICAgbm9kZXMge1xuICAgICAgICAgICAgICAgICAgbG9naW5cbiAgICAgICAgICAgICAgICAgIGVtYWlsXG4gICAgICAgICAgICAgICAgICBuYW1lXG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIHBhZ2VJbmZvIHtcbiAgICAgICAgICAgICAgICAgIGhhc05leHRQYWdlXG4gICAgICAgICAgICAgICAgICBlbmRDdXJzb3JcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9XG4gICAgICAgIGAsXG4gICAgICB9LCB7XG4gICAgICAgIG93bmVyOiByZW1vdGUuZ2V0T3duZXIoKSxcbiAgICAgICAgbmFtZTogcmVtb3RlLmdldFJlcG8oKSxcbiAgICAgICAgZmlyc3Q6IDEwMCxcbiAgICAgICAgYWZ0ZXI6IGN1cnNvcixcbiAgICAgIH0pO1xuXG4gICAgICBpZiAocmVzcG9uc2UuZXJyb3JzICYmIHJlc3BvbnNlLmVycm9ycy5sZW5ndGggPiAxKSB7XG4gICAgICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1jb25zb2xlXG4gICAgICAgIGNvbnNvbGUuZXJyb3IoYEVycm9yIGZldGNoaW5nIG1lbnRpb25hYmxlIHVzZXJzOlxcbiR7cmVzcG9uc2UuZXJyb3JzLm1hcChlID0+IGUubWVzc2FnZSkuam9pbignXFxuJyl9YCk7XG4gICAgICB9XG5cbiAgICAgIGlmICghcmVzcG9uc2UuZGF0YSB8fCAhcmVzcG9uc2UuZGF0YS5yZXBvc2l0b3J5KSB7XG4gICAgICAgIGJyZWFrO1xuICAgICAgfVxuXG4gICAgICBjb25zdCBjb25uZWN0aW9uID0gcmVzcG9uc2UuZGF0YS5yZXBvc2l0b3J5Lm1lbnRpb25hYmxlVXNlcnM7XG4gICAgICBjb25zdCBhdXRob3JzID0gY29ubmVjdGlvbi5ub2Rlcy5tYXAobm9kZSA9PiB7XG4gICAgICAgIGlmIChub2RlLmVtYWlsID09PSAnJykge1xuICAgICAgICAgIG5vZGUuZW1haWwgPSBgJHtub2RlLmxvZ2lufUB1c2Vycy5ub3JlcGx5LmdpdGh1Yi5jb21gO1xuICAgICAgICB9XG5cbiAgICAgICAgcmV0dXJuIG5ldyBBdXRob3Iobm9kZS5lbWFpbCwgbm9kZS5uYW1lLCBub2RlLmxvZ2luKTtcbiAgICAgIH0pO1xuICAgICAgdGhpcy5hZGRVc2VycyhhdXRob3JzLCBzb3VyY2UuR0lUSFVCQVBJKTtcbiAgICAgIHJlbW90ZVVzZXJzLnB1c2goLi4uYXV0aG9ycyk7XG5cbiAgICAgIGN1cnNvciA9IGNvbm5lY3Rpb24ucGFnZUluZm8uZW5kQ3Vyc29yO1xuICAgICAgaGFzTW9yZSA9IGNvbm5lY3Rpb24ucGFnZUluZm8uaGFzTmV4dFBhZ2U7XG4gICAgfVxuXG4gICAgdGhpcy5jYWNoZS5zZXQocmVtb3RlLCByZW1vdGVVc2Vycyk7XG4gIH1cblxuICBhZGRVc2Vycyh1c2VycywgbmV4dFNvdXJjZSkge1xuICAgIGxldCBjaGFuZ2VkID0gZmFsc2U7XG5cbiAgICBpZiAoXG4gICAgICBuZXh0U291cmNlICE9PSB0aGlzLmxhc3Quc291cmNlIHx8XG4gICAgICB0aGlzLnJlcG9zaXRvcnlPYnNlcnZlci5nZXRBY3RpdmVNb2RlbCgpICE9PSB0aGlzLmxhc3QucmVwb3NpdG9yeSB8fFxuICAgICAgdGhpcy5leGNsdWRlZFVzZXJzICE9PSB0aGlzLmxhc3QuZXhjbHVkZWRVc2Vyc1xuICAgICkge1xuICAgICAgY2hhbmdlZCA9IHRydWU7XG4gICAgICB0aGlzLmFsbFVzZXJzLmNsZWFyKCk7XG4gICAgfVxuXG4gICAgZm9yIChjb25zdCBhdXRob3Igb2YgdXNlcnMpIHtcbiAgICAgIGlmICghdGhpcy5hbGxVc2Vycy5oYXMoYXV0aG9yLmdldEVtYWlsKCkpKSB7XG4gICAgICAgIGNoYW5nZWQgPSB0cnVlO1xuICAgICAgfVxuICAgICAgdGhpcy5hbGxVc2Vycy5zZXQoYXV0aG9yLmdldEVtYWlsKCksIGF1dGhvcik7XG4gICAgfVxuXG4gICAgaWYgKGNoYW5nZWQpIHtcbiAgICAgIHRoaXMuZmluYWxpemUoKTtcbiAgICB9XG4gICAgdGhpcy5sYXN0LnNvdXJjZSA9IG5leHRTb3VyY2U7XG4gICAgdGhpcy5sYXN0LnJlcG9zaXRvcnkgPSB0aGlzLnJlcG9zaXRvcnlPYnNlcnZlci5nZXRBY3RpdmVNb2RlbCgpO1xuICAgIHRoaXMubGFzdC5leGNsdWRlZFVzZXJzID0gdGhpcy5leGNsdWRlZFVzZXJzO1xuICB9XG5cbiAgZmluYWxpemUoKSB7XG4gICAgLy8gVE9ETzogW2t1IDMvMjAxOF0gY29uc2lkZXIgc29ydGluZyBiYXNlZCBvbiBtb3N0IHJlY2VudCBhdXRob3JzIG9yIGNvbW1pdCBmcmVxdWVuY3lcbiAgICBjb25zdCB1c2VycyA9IFtdO1xuICAgIGZvciAoY29uc3QgYXV0aG9yIG9mIHRoaXMuYWxsVXNlcnMudmFsdWVzKCkpIHtcbiAgICAgIGlmIChhdXRob3IubWF0Y2hlcyh0aGlzLmNvbW1pdHRlcikpIHsgY29udGludWU7IH1cbiAgICAgIGlmIChhdXRob3IuaXNOb1JlcGx5KCkpIHsgY29udGludWU7IH1cbiAgICAgIGlmICh0aGlzLmV4Y2x1ZGVkVXNlcnMuaGFzKGF1dGhvci5nZXRFbWFpbCgpKSkgeyBjb250aW51ZTsgfVxuXG4gICAgICB1c2Vycy5wdXNoKGF1dGhvcik7XG4gICAgfVxuICAgIHVzZXJzLnNvcnQoQXV0aG9yLmNvbXBhcmUpO1xuICAgIHRoaXMudXNlcnMgPSB1c2VycztcbiAgICB0aGlzLmRpZFVwZGF0ZSgpO1xuICB9XG5cbiAgc2V0UmVwb3NpdG9yeShyZXBvc2l0b3J5KSB7XG4gICAgdGhpcy5yZXBvc2l0b3J5T2JzZXJ2ZXIuc2V0QWN0aXZlTW9kZWwocmVwb3NpdG9yeSk7XG4gIH1cblxuICBzZXRMb2dpbk1vZGVsKGxvZ2luKSB7XG4gICAgdGhpcy5sb2dpbk9ic2VydmVyLnNldEFjdGl2ZU1vZGVsKGxvZ2luKTtcbiAgfVxuXG4gIHNldENvbW1pdHRlcihjb21taXR0ZXIpIHtcbiAgICBjb25zdCBjaGFuZ2VkID0gIXRoaXMuY29tbWl0dGVyLm1hdGNoZXMoY29tbWl0dGVyKTtcbiAgICB0aGlzLmNvbW1pdHRlciA9IGNvbW1pdHRlcjtcbiAgICBpZiAoY2hhbmdlZCkge1xuICAgICAgdGhpcy5maW5hbGl6ZSgpO1xuICAgIH1cbiAgfVxuXG4gIGRpZFVwZGF0ZSgpIHtcbiAgICB0aGlzLmVtaXR0ZXIuZW1pdCgnZGlkLXVwZGF0ZScsIHRoaXMuZ2V0VXNlcnMoKSk7XG4gIH1cblxuICBvbkRpZFVwZGF0ZShjYWxsYmFjaykge1xuICAgIHJldHVybiB0aGlzLmVtaXR0ZXIub24oJ2RpZC11cGRhdGUnLCBjYWxsYmFjayk7XG4gIH1cblxuICBnZXRVc2VycygpIHtcbiAgICByZXR1cm4gdGhpcy51c2VycztcbiAgfVxufVxuIl19