import { concat, identity, tuple } from './function';
import { none, some } from './Option';
import { getSemigroup, ordNumber, fromCompare } from './Ord';
import { getArraySetoid } from './Setoid';
export const URI = 'Array';
/**
 * @since 1.17.0
 */
export const getShow = (S) => {
    return {
        show: arr => `[${arr.map(S.show).join(', ')}]`
    };
};
/**
 *
 * @example
 * import { getMonoid } from 'fp-ts/lib/Array'
 *
 * const M = getMonoid<number>()
 * assert.deepStrictEqual(M.concat([1, 2], [3, 4]), [1, 2, 3, 4])
 *
 * @since 1.0.0
 */
export const getMonoid = () => {
    return {
        concat,
        empty
    };
};
/**
 * Derives a Setoid over the Array of a given element type from the Setoid of that type. The derived setoid defines two
 * arrays as equal if all elements of both arrays are compared equal pairwise with the given setoid `S`. In case of
 * arrays of different lengths, the result is non equality.
 *
 *
 * @example
 * import { ordString } from 'fp-ts/lib/Ord'
 * import { getSetoid } from 'fp-ts/lib/Array'
 *
 * const O = getSetoid(ordString)
 * assert.strictEqual(O.equals(['a', 'b'], ['a', 'b']), true)
 * assert.strictEqual(O.equals(['a'], []), false)
 *
 * @since 1.0.0
 */
export const getSetoid = (S) => {
    return getArraySetoid(S);
};
/**
 * Derives an `Ord` over the Array of a given element type from the `Ord` of that type. The ordering between two such
 * arrays is equal to: the first non equal comparison of each arrays elements taken pairwise in increasing order, in
 * case of equality over all the pairwise elements; the longest array is considered the greatest, if both arrays have
 * the same length, the result is equality.
 *
 *
 * @example
 * import { getOrd } from 'fp-ts/lib/Array'
 * import { ordString } from 'fp-ts/lib/Ord'
 *
 * const O = getOrd(ordString)
 * assert.strictEqual(O.compare(['b'], ['a']), 1)
 * assert.strictEqual(O.compare(['a'], ['a']), 0)
 * assert.strictEqual(O.compare(['a'], ['b']), -1)
 *
 *
 * @since 1.2.0
 */
export const getOrd = (O) => {
    return fromCompare((a, b) => {
        const aLen = a.length;
        const bLen = b.length;
        const len = Math.min(aLen, bLen);
        for (let i = 0; i < len; i++) {
            const order = O.compare(a[i], b[i]);
            if (order !== 0) {
                return order;
            }
        }
        return ordNumber.compare(aLen, bLen);
    });
};
const map = (fa, f) => {
    return fa.map(a => f(a));
};
const mapWithIndex = (fa, f) => {
    return fa.map((a, i) => f(i, a));
};
const of = (a) => {
    return [a];
};
const ap = (fab, fa) => {
    return flatten(map(fab, f => map(fa, f)));
};
const chain = (fa, f) => {
    let resLen = 0;
    const l = fa.length;
    const temp = new Array(l);
    for (let i = 0; i < l; i++) {
        const e = fa[i];
        const arr = f(e);
        resLen += arr.length;
        temp[i] = arr;
    }
    const r = Array(resLen);
    let start = 0;
    for (let i = 0; i < l; i++) {
        const arr = temp[i];
        const l = arr.length;
        for (let j = 0; j < l; j++) {
            r[j + start] = arr[j];
        }
        start += l;
    }
    return r;
};
const reduce = (fa, b, f) => {
    return reduceWithIndex(fa, b, (_, b, a) => f(b, a));
};
const foldMap = (M) => {
    const foldMapWithIndexM = foldMapWithIndex(M);
    return (fa, f) => foldMapWithIndexM(fa, (_, a) => f(a));
};
const reduceRight = (fa, b, f) => {
    return foldrWithIndex(fa, b, (_, a, b) => f(a, b));
};
const reduceWithIndex = (fa, b, f) => {
    const l = fa.length;
    let r = b;
    for (let i = 0; i < l; i++) {
        r = f(i, r, fa[i]);
    }
    return r;
};
const foldMapWithIndex = (M) => (fa, f) => {
    return fa.reduce((b, a, i) => M.concat(b, f(i, a)), M.empty);
};
const foldrWithIndex = (fa, b, f) => {
    return fa.reduceRight((b, a, i) => f(i, a, b), b);
};
export function traverse(F) {
    const traverseWithIndexF = traverseWithIndex(F);
    return (ta, f) => traverseWithIndexF(ta, (_, a) => f(a));
}
const sequence = (F) => (ta) => {
    return reduce(ta, F.of(zero()), (fas, fa) => F.ap(F.map(fas, as => (a) => snoc(as, a)), fa));
};
/**
 * An empty array
 *
 *
 * @since 1.9.0
 */
export const empty = [];
const zero = () => empty;
const alt = concat;
const unfoldr = (b, f) => {
    const ret = [];
    let bb = b;
    while (true) {
        const mt = f(bb);
        if (mt.isSome()) {
            const [a, b] = mt.value;
            ret.push(a);
            bb = b;
        }
        else {
            break;
        }
    }
    return ret;
};
/**
 * Return a list of length `n` with element `i` initialized with `f(i)`
 *
 * @example
 * import { makeBy } from 'fp-ts/lib/Array'
 *
 * const double = (n: number): number => n * 2
 * assert.deepStrictEqual(makeBy(5, double), [0, 2, 4, 6, 8])
 *
 *
 * @since 1.10.0
 */
export const makeBy = (n, f) => {
    const r = [];
    for (let i = 0; i < n; i++) {
        r.push(f(i));
    }
    return r;
};
/**
 * Create an array containing a range of integers, including both endpoints
 *
 * @example
 * import { range } from 'fp-ts/lib/Array'
 *
 * assert.deepStrictEqual(range(1, 5), [1, 2, 3, 4, 5])
 *
 *
 * @since 1.10.0
 */
export const range = (start, end) => {
    return makeBy(end - start + 1, i => start + i);
};
/**
 * Create an array containing a value repeated the specified number of times
 *
 * @example
 * import { replicate } from 'fp-ts/lib/Array'
 *
 * assert.deepStrictEqual(replicate(3, 'a'), ['a', 'a', 'a'])
 *
 *
 * @since 1.10.0
 */
export const replicate = (n, a) => {
    return makeBy(n, () => a);
};
const extend = (fa, f) => {
    return fa.map((_, i, as) => f(as.slice(i)));
};
/**
 * Removes one level of nesting
 *
 * @example
 * import { flatten } from 'fp-ts/lib/Array'
 *
 * assert.deepStrictEqual(flatten([[1], [2], [3]]), [1, 2, 3])
 *
 * @since 1.0.0
 */
export const flatten = (ffa) => {
    let rLen = 0;
    const len = ffa.length;
    for (let i = 0; i < len; i++) {
        rLen += ffa[i].length;
    }
    const r = Array(rLen);
    let start = 0;
    for (let i = 0; i < len; i++) {
        const arr = ffa[i];
        const l = arr.length;
        for (let j = 0; j < l; j++) {
            r[j + start] = arr[j];
        }
        start += l;
    }
    return r;
};
/**
 * Break an array into its first element and remaining elements
 *
 * @example
 * import { fold } from 'fp-ts/lib/Array'
 *
 * const len = <A>(as: Array<A>): number => fold(as, 0, (_, tail) => 1 + len(tail))
 * assert.strictEqual(len([1, 2, 3]), 3)
 *
 * @since 1.0.0
 */
export const fold = (as, b, cons) => {
    return isEmpty(as) ? b : cons(as[0], as.slice(1));
};
/**
 * Lazy version of `fold`
 *
 * @since 1.0.0
 */
export const foldL = (as, nil, cons) => {
    return isEmpty(as) ? nil() : cons(as[0], as.slice(1));
};
/**
 * Break an array into its initial elements and the last element
 *
 * @since 1.7.0
 * @param as
 * @param b
 * @param cons
 */
export const foldr = (as, b, cons) => {
    return isEmpty(as) ? b : cons(as.slice(0, as.length - 1), as[as.length - 1]);
};
/**
 * Lazy version of `foldr`
 *
 * @since 1.7.0
 * @param as
 * @param nil
 * @param cons
 */
export const foldrL = (as, nil, cons) => {
    return isEmpty(as) ? nil() : cons(as.slice(0, as.length - 1), as[as.length - 1]);
};
/**
 * Same as `reduce` but it carries over the intermediate steps
 *
 * ```ts
 * import { scanLeft } from 'fp-ts/lib/Array'
 *
 * assert.deepStrictEqual(scanLeft([1, 2, 3], 10, (b, a) => b - a), [ 10, 9, 7, 4 ])
 * ```
 *
 *
 * @since 1.1.0
 */
export const scanLeft = (as, b, f) => {
    const l = as.length;
    const r = new Array(l + 1);
    r[0] = b;
    for (let i = 0; i < l; i++) {
        r[i + 1] = f(r[i], as[i]);
    }
    return r;
};
/**
 * Fold an array from the right, keeping all intermediate results instead of only the final result
 *
 * @example
 * import { scanRight } from 'fp-ts/lib/Array'
 *
 * assert.deepStrictEqual(scanRight([1, 2, 3], 10, (a, b) => b - a), [ 4, 5, 7, 10 ])
 *
 *
 * @since 1.1.0
 */
export const scanRight = (as, b, f) => {
    const l = as.length;
    const r = new Array(l + 1);
    r[l] = b;
    for (let i = l - 1; i >= 0; i--) {
        r[i] = f(as[i], r[i + 1]);
    }
    return r;
};
/**
 * Test whether an array is empty
 *
 * @example
 * import { isEmpty } from 'fp-ts/lib/Array'
 *
 * assert.strictEqual(isEmpty([]), true)
 *
 * @since 1.0.0
 */
export const isEmpty = (as) => {
    return as.length === 0;
};
/**
 * Test whether an array contains a particular index
 *
 * @since 1.0.0
 */
export const isOutOfBound = (i, as) => {
    return i < 0 || i >= as.length;
};
/**
 * This function provides a safe way to read a value at a particular index from an array
 *
 * @example
 * import { lookup } from 'fp-ts/lib/Array'
 * import { some, none } from 'fp-ts/lib/Option'
 *
 * assert.deepStrictEqual(lookup(1, [1, 2, 3]), some(2))
 * assert.deepStrictEqual(lookup(3, [1, 2, 3]), none)
 *
 * @since 1.14.0
 */
export const lookup = (i, as) => {
    return isOutOfBound(i, as) ? none : some(as[i]);
};
/**
 * Use `lookup` instead
 * @since 1.0.0
 * @deprecated
 */
export const index = (i, as) => {
    return lookup(i, as);
};
/**
 * Attaches an element to the front of an array, creating a new non empty array
 *
 * @example
 * import { cons } from 'fp-ts/lib/Array'
 *
 * assert.deepStrictEqual(cons(0, [1, 2, 3]), [0, 1, 2, 3])
 *
 * @since 1.0.0
 */
export const cons = (a, as) => {
    const len = as.length;
    const r = Array(len + 1);
    for (let i = 0; i < len; i++) {
        r[i + 1] = as[i];
    }
    r[0] = a;
    return r;
};
/**
 * Append an element to the end of an array, creating a new non empty array
 *
 * @example
 * import { snoc } from 'fp-ts/lib/Array'
 *
 * assert.deepStrictEqual(snoc([1, 2, 3], 4), [1, 2, 3, 4])
 *
 * @since 1.0.0
 */
export const snoc = (as, a) => {
    const len = as.length;
    const r = Array(len + 1);
    for (let i = 0; i < len; i++) {
        r[i] = as[i];
    }
    r[len] = a;
    return r;
};
/**
 * Get the first element in an array, or `None` if the array is empty
 *
 * @example
 * import { head } from 'fp-ts/lib/Array'
 * import { some, none } from 'fp-ts/lib/Option'
 *
 * assert.deepStrictEqual(head([1, 2, 3]), some(1))
 * assert.deepStrictEqual(head([]), none)
 *
 * @since 1.0.0
 */
export const head = (as) => {
    return isEmpty(as) ? none : some(as[0]);
};
/**
 * Get the last element in an array, or `None` if the array is empty
 *
 * @example
 * import { last } from 'fp-ts/lib/Array'
 * import { some, none } from 'fp-ts/lib/Option'
 *
 * assert.deepStrictEqual(last([1, 2, 3]), some(3))
 * assert.deepStrictEqual(last([]), none)
 *
 * @since 1.0.0
 */
export const last = (as) => {
    return lookup(as.length - 1, as);
};
/**
 * Get all but the first element of an array, creating a new array, or `None` if the array is empty
 *
 * @example
 * import { tail } from 'fp-ts/lib/Array'
 * import { some, none } from 'fp-ts/lib/Option'
 *
 * assert.deepStrictEqual(tail([1, 2, 3]), some([2, 3]))
 * assert.deepStrictEqual(tail([]), none)
 *
 * @since 1.0.0
 */
export const tail = (as) => {
    return isEmpty(as) ? none : some(as.slice(1));
};
/**
 * Get all but the last element of an array, creating a new array, or `None` if the array is empty
 *
 * @example
 * import { init } from 'fp-ts/lib/Array'
 * import { some, none } from 'fp-ts/lib/Option'
 *
 * assert.deepStrictEqual(init([1, 2, 3]), some([1, 2]))
 * assert.deepStrictEqual(init([]), none)
 *
 * @since 1.0.0
 */
export const init = (as) => {
    const len = as.length;
    return len === 0 ? none : some(as.slice(0, len - 1));
};
/**
 * Keep only a number of elements from the start of an array, creating a new array.
 * `n` must be a natural number
 *
 * @example
 * import { take } from 'fp-ts/lib/Array'
 *
 * assert.deepStrictEqual(take(2, [1, 2, 3]), [1, 2])
 *
 * @since 1.0.0
 */
export const take = (n, as) => {
    return as.slice(0, n);
};
/**
 * Keep only a number of elements from the end of an array, creating a new array.
 * `n` must be a natural number
 *
 * @example
 * import { takeEnd } from 'fp-ts/lib/Array'
 *
 * assert.deepStrictEqual(takeEnd(2, [1, 2, 3, 4, 5]), [4, 5])
 *
 *
 * @since 1.10.0
 */
export const takeEnd = (n, as) => {
    return n === 0 ? empty : as.slice(-n);
};
export function takeWhile(as, predicate) {
    const i = spanIndexUncurry(as, predicate);
    const init = Array(i);
    for (let j = 0; j < i; j++) {
        init[j] = as[j];
    }
    return init;
}
const spanIndexUncurry = (as, predicate) => {
    const l = as.length;
    let i = 0;
    for (; i < l; i++) {
        if (!predicate(as[i])) {
            break;
        }
    }
    return i;
};
export function span(as, predicate) {
    const i = spanIndexUncurry(as, predicate);
    const init = Array(i);
    for (let j = 0; j < i; j++) {
        init[j] = as[j];
    }
    const l = as.length;
    const rest = Array(l - i);
    for (let j = i; j < l; j++) {
        rest[j - i] = as[j];
    }
    return { init, rest };
}
/**
 * Drop a number of elements from the start of an array, creating a new array
 *
 * @example
 * import { drop } from 'fp-ts/lib/Array'
 *
 * assert.deepStrictEqual(drop(2, [1, 2, 3]), [3])
 *
 * @since 1.0.0
 */
export const drop = (n, as) => {
    return as.slice(n, as.length);
};
/**
 * Drop a number of elements from the end of an array, creating a new array
 *
 * @example
 * import { dropEnd } from 'fp-ts/lib/Array'
 *
 * assert.deepStrictEqual(dropEnd(2, [1, 2, 3, 4, 5]), [1, 2, 3])
 *
 *
 * @since 1.10.0
 */
export const dropEnd = (n, as) => {
    return as.slice(0, as.length - n);
};
/**
 * Remove the longest initial subarray for which all element satisfy the specified predicate, creating a new array
 *
 * @example
 * import { dropWhile } from 'fp-ts/lib/Array'
 *
 * assert.deepStrictEqual(dropWhile([1, 3, 2, 4, 5], n => n % 2 === 1), [2, 4, 5])
 *
 * @since 1.0.0
 */
export const dropWhile = (as, predicate) => {
    const i = spanIndexUncurry(as, predicate);
    const l = as.length;
    const rest = Array(l - i);
    for (let j = i; j < l; j++) {
        rest[j - i] = as[j];
    }
    return rest;
};
/**
 * Find the first index for which a predicate holds
 *
 * @example
 * import { findIndex } from 'fp-ts/lib/Array'
 * import { some, none } from 'fp-ts/lib/Option'
 *
 * assert.deepStrictEqual(findIndex([1, 2, 3], x => x === 2), some(1))
 * assert.deepStrictEqual(findIndex([], x => x === 2), none)
 *
 * @since 1.0.0
 */
export const findIndex = (as, predicate) => {
    const len = as.length;
    for (let i = 0; i < len; i++) {
        if (predicate(as[i])) {
            return some(i);
        }
    }
    return none;
};
export function findFirst(as, predicate) {
    const len = as.length;
    for (let i = 0; i < len; i++) {
        if (predicate(as[i])) {
            return some(as[i]);
        }
    }
    return none;
}
/**
 * Find the first element returned by an option based selector function
 *
 * @example
 * import { findFirstMap } from 'fp-ts/lib/Array'
 * import { some, none } from 'fp-ts/lib/Option'
 *
 * interface Person {
 *   name: string
 *   age?: number
 * }
 *
 * const persons: Array<Person> = [{ name: 'John' }, { name: 'Mary', age: 45 }, { name: 'Joey', age: 28 }]
 *
 * // returns the name of the first person that has an age
 * assert.deepStrictEqual(findFirstMap(persons, p => (p.age === undefined ? none : some(p.name))), some('Mary'))
 *
 * @since 1.16.0
 */
export const findFirstMap = (arr, f) => {
    const len = arr.length;
    for (let i = 0; i < len; i++) {
        const v = f(arr[i]);
        if (v.isSome()) {
            return v;
        }
    }
    return none;
};
export function findLast(as, predicate) {
    const len = as.length;
    for (let i = len - 1; i >= 0; i--) {
        if (predicate(as[i])) {
            return some(as[i]);
        }
    }
    return none;
}
/**
 * Find the last element returned by an option based selector function
 *
 * @example
 * import { findLastMap } from 'fp-ts/lib/Array'
 * import { some, none } from 'fp-ts/lib/Option'
 *
 * interface Person {
 *   name: string
 *   age?: number
 * }
 *
 * const persons: Array<Person> = [{ name: 'John' }, { name: 'Mary', age: 45 }, { name: 'Joey', age: 28 }]
 *
 * // returns the name of the last person that has an age
 * assert.deepStrictEqual(findLastMap(persons, p => (p.age === undefined ? none : some(p.name))), some('Joey'))
 *
 * @since 1.16.0
 */
export const findLastMap = (arr, f) => {
    const len = arr.length;
    for (let i = len - 1; i >= 0; i--) {
        const v = f(arr[i]);
        if (v.isSome()) {
            return v;
        }
    }
    return none;
};
/**
 * Returns the index of the last element of the list which matches the predicate
 *
 * @example
 * import { findLastIndex } from 'fp-ts/lib/Array'
 * import { some, none } from 'fp-ts/lib/Option'
 *
 * interface X {
 *   a: number
 *   b: number
 * }
 * const xs: Array<X> = [{ a: 1, b: 0 }, { a: 1, b: 1 }]
 * assert.deepStrictEqual(findLastIndex(xs, x => x.a === 1), some(1))
 * assert.deepStrictEqual(findLastIndex(xs, x => x.a === 4), none)
 *
 *
 * @since 1.10.0
 */
export const findLastIndex = (as, predicate) => {
    const len = as.length;
    for (let i = len - 1; i >= 0; i--) {
        if (predicate(as[i])) {
            return some(i);
        }
    }
    return none;
};
/**
 * Use `array.filter` instead
 *
 * @since 1.0.0
 * @deprecated
 */
export const refine = (as, refinement) => {
    // tslint:disable-next-line: deprecation
    return filter(as, refinement);
};
/**
 * @since 1.0.0
 */
export const copy = (as) => {
    const l = as.length;
    const r = Array(l);
    for (let i = 0; i < l; i++) {
        r[i] = as[i];
    }
    return r;
};
/**
 *
 * @since 1.0.0
 */
export const unsafeInsertAt = (i, a, as) => {
    const xs = copy(as);
    xs.splice(i, 0, a);
    return xs;
};
/**
 * Insert an element at the specified index, creating a new array, or returning `None` if the index is out of bounds
 *
 * @example
 * import { insertAt } from 'fp-ts/lib/Array'
 * import { some } from 'fp-ts/lib/Option'
 *
 * assert.deepStrictEqual(insertAt(2, 5, [1, 2, 3, 4]), some([1, 2, 5, 3, 4]))
 *
 * @since 1.0.0
 */
export const insertAt = (i, a, as) => {
    return i < 0 || i > as.length ? none : some(unsafeInsertAt(i, a, as));
};
/**
 *
 * @since 1.0.0
 */
export const unsafeUpdateAt = (i, a, as) => {
    if (as[i] === a) {
        return as;
    }
    else {
        const xs = copy(as);
        xs[i] = a;
        return xs;
    }
};
/**
 * Change the element at the specified index, creating a new array, or returning `None` if the index is out of bounds
 *
 * @example
 * import { updateAt } from 'fp-ts/lib/Array'
 * import { some, none } from 'fp-ts/lib/Option'
 *
 * assert.deepStrictEqual(updateAt(1, 1, [1, 2, 3]), some([1, 1, 3]))
 * assert.deepStrictEqual(updateAt(1, 1, []), none)
 *
 * @since 1.0.0
 */
export const updateAt = (i, a, as) => {
    return isOutOfBound(i, as) ? none : some(unsafeUpdateAt(i, a, as));
};
/**
 *
 * @since 1.0.0
 */
export const unsafeDeleteAt = (i, as) => {
    const xs = copy(as);
    xs.splice(i, 1);
    return xs;
};
/**
 * Delete the element at the specified index, creating a new array, or returning `None` if the index is out of bounds
 *
 * @example
 * import { deleteAt } from 'fp-ts/lib/Array'
 * import { some, none } from 'fp-ts/lib/Option'
 *
 * assert.deepStrictEqual(deleteAt(0, [1, 2, 3]), some([2, 3]))
 * assert.deepStrictEqual(deleteAt(1, []), none)
 *
 * @since 1.0.0
 */
export const deleteAt = (i, as) => {
    return isOutOfBound(i, as) ? none : some(unsafeDeleteAt(i, as));
};
/**
 * Apply a function to the element at the specified index, creating a new array, or returning `None` if the index is out
 * of bounds
 *
 * @example
 * import { modifyAt } from 'fp-ts/lib/Array'
 * import { some, none } from 'fp-ts/lib/Option'
 *
 * const double = (x: number): number => x * 2
 * assert.deepStrictEqual(modifyAt([1, 2, 3], 1, double), some([1, 4, 3]))
 * assert.deepStrictEqual(modifyAt([], 1, double), none)
 *
 * @since 1.0.0
 */
export const modifyAt = (as, i, f) => {
    return isOutOfBound(i, as) ? none : some(unsafeUpdateAt(i, f(as[i]), as));
};
/**
 * Reverse an array, creating a new array
 *
 * @example
 * import { reverse } from 'fp-ts/lib/Array'
 *
 * assert.deepStrictEqual(reverse([1, 2, 3]), [3, 2, 1])
 *
 * @since 1.0.0
 */
export const reverse = (as) => {
    return copy(as).reverse();
};
/**
 * Extracts from an array of `Either` all the `Right` elements. All the `Right` elements are extracted in order
 *
 * @example
 * import { rights } from 'fp-ts/lib/Array'
 * import { right, left } from 'fp-ts/lib/Either'
 *
 * assert.deepStrictEqual(rights([right(1), left('foo'), right(2)]), [1, 2])
 *
 * @since 1.0.0
 */
export const rights = (as) => {
    const r = [];
    const len = as.length;
    for (let i = 0; i < len; i++) {
        const a = as[i];
        if (a.isRight()) {
            r.push(a.value);
        }
    }
    return r;
};
/**
 * Extracts from an array of `Either` all the `Left` elements. All the `Left` elements are extracted in order
 *
 * @example
 * import { lefts } from 'fp-ts/lib/Array'
 * import { left, right } from 'fp-ts/lib/Either'
 *
 * assert.deepStrictEqual(lefts([right(1), left('foo'), right(2)]), ['foo'])
 *
 * @since 1.0.0
 */
export const lefts = (as) => {
    const r = [];
    const len = as.length;
    for (let i = 0; i < len; i++) {
        const a = as[i];
        if (a.isLeft()) {
            r.push(a.value);
        }
    }
    return r;
};
/**
 * Sort the elements of an array in increasing order, creating a new array
 *
 * @example
 * import { sort } from 'fp-ts/lib/Array'
 * import { ordNumber } from 'fp-ts/lib/Ord'
 *
 * assert.deepStrictEqual(sort(ordNumber)([3, 2, 1]), [1, 2, 3])
 *
 * @since 1.0.0
 */
export const sort = (O) => (as) => {
    return copy(as).sort(O.compare);
};
/**
 * Apply a function to pairs of elements at the same index in two arrays, collecting the results in a new array. If one
 * input array is short, excess elements of the longer array are discarded.
 *
 * @example
 * import { zipWith } from 'fp-ts/lib/Array'
 *
 * assert.deepStrictEqual(zipWith([1, 2, 3], ['a', 'b', 'c', 'd'], (n, s) => s + n), ['a1', 'b2', 'c3'])
 *
 * @since 1.0.0
 */
export const zipWith = (fa, fb, f) => {
    const fc = [];
    const len = Math.min(fa.length, fb.length);
    for (let i = 0; i < len; i++) {
        fc[i] = f(fa[i], fb[i]);
    }
    return fc;
};
/**
 * Takes two arrays and returns an array of corresponding pairs. If one input array is short, excess elements of the
 * longer array are discarded
 *
 * @example
 * import { zip } from 'fp-ts/lib/Array'
 *
 * assert.deepStrictEqual(zip([1, 2, 3], ['a', 'b', 'c', 'd']), [[1, 'a'], [2, 'b'], [3, 'c']])
 *
 * @since 1.0.0
 */
export const zip = (fa, fb) => {
    return zipWith(fa, fb, tuple);
};
/**
 * The function is reverse of `zip`. Takes an array of pairs and return two corresponding arrays
 *
 * @example
 * import { unzip } from 'fp-ts/lib/Array'
 *
 * assert.deepStrictEqual(unzip([[1, 'a'], [2, 'b'], [3, 'c']]), [[1, 2, 3], ['a', 'b', 'c']])
 *
 *
 * @since 1.13.0
 */
export const unzip = (as) => {
    const fa = [];
    const fb = [];
    for (let i = 0; i < as.length; i++) {
        fa[i] = as[i][0];
        fb[i] = as[i][1];
    }
    return [fa, fb];
};
/**
 * Rotate an array to the right by `n` steps
 *
 * @example
 * import { rotate } from 'fp-ts/lib/Array'
 *
 * assert.deepStrictEqual(rotate(2, [1, 2, 3, 4, 5]), [4, 5, 1, 2, 3])
 *
 * @since 1.0.0
 */
export const rotate = (n, xs) => {
    const len = xs.length;
    if (n === 0 || len <= 1 || len === Math.abs(n)) {
        return xs;
    }
    else if (n < 0) {
        return rotate(len + n, xs);
    }
    else {
        return xs.slice(-n).concat(xs.slice(0, len - n));
    }
};
/**
 * Test if a value is a member of an array. Takes a `Setoid<A>` as a single
 * argument which returns the function to use to search for a value of type `A` in
 * an array of type `Array<A>`.
 *
 * @example
 * import { elem } from 'fp-ts/lib/Array'
 * import { setoidNumber } from 'fp-ts/lib/Setoid'
 *
 * assert.strictEqual(elem(setoidNumber)(1, [1, 2, 3]), true)
 * assert.strictEqual(elem(setoidNumber)(4, [1, 2, 3]), false)
 *
 * @since 1.14.0
 */
export const elem = (S) => (a, as) => {
    const predicate = (e) => S.equals(e, a);
    let i = 0;
    const len = as.length;
    for (; i < len; i++) {
        if (predicate(as[i])) {
            return true;
        }
    }
    return false;
};
/**
 * Use `elem` instead
 * @since 1.3.0
 * @deprecated
 */
export const member = (S) => {
    const has = elem(S);
    return (as, a) => has(a, as);
};
/**
 * Remove duplicates from an array, keeping the first occurance of an element.
 *
 * @example
 * import { uniq } from 'fp-ts/lib/Array'
 * import { setoidNumber } from 'fp-ts/lib/Setoid'
 *
 * assert.deepStrictEqual(uniq(setoidNumber)([1, 2, 1]), [1, 2])
 *
 *
 * @since 1.3.0
 */
export const uniq = (S) => {
    const elemS = elem(S);
    return as => {
        const r = [];
        const len = as.length;
        let i = 0;
        for (; i < len; i++) {
            const a = as[i];
            if (!elemS(a, r)) {
                r.push(a);
            }
        }
        return len === r.length ? as : r;
    };
};
/**
 * Sort the elements of an array in increasing order, where elements are compared using first `ords[0]`, then `ords[1]`,
 * etc...
 *
 * @example
 * import { sortBy } from 'fp-ts/lib/Array'
 * import { contramap, ordString, ordNumber } from 'fp-ts/lib/Ord'
 *
 * interface Person {
 *   name: string
 *   age: number
 * }
 * const byName = contramap((p: Person) => p.name, ordString)
 * const byAge = contramap((p: Person) => p.age, ordNumber)
 *
 * const sortByNameByAge = sortBy([byName, byAge])
 *
 * if (sortByNameByAge.isSome()) {
 *   const persons = [{ name: 'a', age: 1 }, { name: 'b', age: 3 }, { name: 'c', age: 2 }, { name: 'b', age: 2 }]
 *   assert.deepStrictEqual(sortByNameByAge.value(persons), [
 *     { name: 'a', age: 1 },
 *     { name: 'b', age: 2 },
 *     { name: 'b', age: 3 },
 *     { name: 'c', age: 2 }
 *   ])
 * }
 *
 *
 * @since 1.3.0
 */
export const sortBy = (ords) => {
    return fold(ords, none, (head, tail) => some(sortBy1(head, tail)));
};
/**
 * Non failing version of `sortBy`
 * @example
 * import { sortBy1 } from 'fp-ts/lib/Array'
 * import { contramap, ordString, ordNumber } from 'fp-ts/lib/Ord'
 *
 * interface Person {
 *   name: string
 *   age: number
 * }
 * const byName = contramap((p: Person) => p.name, ordString)
 * const byAge = contramap((p: Person) => p.age, ordNumber)
 *
 * const sortByNameByAge = sortBy1(byName, [byAge])
 *
 * const persons = [{ name: 'a', age: 1 }, { name: 'b', age: 3 }, { name: 'c', age: 2 }, { name: 'b', age: 2 }]
 * assert.deepStrictEqual(sortByNameByAge(persons), [
 *   { name: 'a', age: 1 },
 *   { name: 'b', age: 2 },
 *   { name: 'b', age: 3 },
 *   { name: 'c', age: 2 }
 * ])
 *
 *
 * @since 1.3.0
 */
export const sortBy1 = (head, tail) => {
    return sort(tail.reduce(getSemigroup().concat, head));
};
/**
 * Apply a function to each element in an array, keeping only the results which contain a value, creating a new array.
 *
 * Alias of `Filterable`'s `filterMap`
 *
 * @example
 * import { mapOption } from 'fp-ts/lib/Array'
 * import { Option, some, none } from 'fp-ts/lib/Option'
 *
 * const f = (n: number): Option<number> => (n % 2 === 0 ? none : some(n))
 * assert.deepStrictEqual(mapOption([1, 2, 3], f), [1, 3])
 *
 * @since 1.0.0
 */
export const mapOption = (as, f) => {
    return filterMapWithIndex(as, (_, a) => f(a));
};
/**
 * Filter an array of optional values, keeping only the elements which contain a value, creating a new array.
 *
 * Alias of `Compactable`'s `compact`
 *
 * @example
 * import { catOptions } from 'fp-ts/lib/Array'
 * import { some, none } from 'fp-ts/lib/Option'
 *
 * assert.deepStrictEqual(catOptions([some(1), none, some(3)]), [1, 3])
 *
 * @since 1.0.0
 */
export const catOptions = (as) => mapOption(as, identity);
/**
 * @example
 * import { array } from 'fp-ts/lib/Array'
 * import { left, right } from 'fp-ts/lib/Either'
 * import { identity } from 'fp-ts/lib/function'
 *
 * assert.deepStrictEqual(array.partitionMap([right(1), left('foo'), right(2)], identity), { left: ['foo'], right: [1, 2] })
 *
 * @since 1.0.0
 */
export const partitionMap = (fa, f) => {
    return partitionMapWithIndex(fa, (_, a) => f(a));
};
export function filter(as, predicate) {
    return as.filter(predicate);
}
export function partition(fa, p) {
    return partitionWithIndex(fa, (_, a) => p(a));
}
const compact = catOptions;
const separate = (fa) => {
    const left = [];
    const right = [];
    for (const e of fa) {
        if (e.isLeft()) {
            left.push(e.value);
        }
        else {
            right.push(e.value);
        }
    }
    return {
        left,
        right
    };
};
const filterMap = mapOption;
const wither = (F) => {
    const traverseF = traverse(F);
    return (wa, f) => F.map(traverseF(wa, f), compact);
};
const wilt = (F) => {
    const traverseF = traverse(F);
    return (wa, f) => F.map(traverseF(wa, f), separate);
};
/**
 * A useful recursion pattern for processing an array to produce a new array, often used for "chopping" up the input
 * array. Typically chop is called with some function that will consume an initial prefix of the array and produce a
 * value and the rest of the array.
 *
 * @example
 * import { Setoid, setoidNumber } from 'fp-ts/lib/Setoid'
 * import { chop, span } from 'fp-ts/lib/Array'
 *
 * const group = <A>(S: Setoid<A>) => (as: Array<A>): Array<Array<A>> => {
 *   return chop(as, as => {
 *     const { init, rest } = span(as, a => S.equals(a, as[0]))
 *     return [init, rest]
 *   })
 * }
 * assert.deepStrictEqual(group(setoidNumber)([1, 1, 2, 3, 3, 4]), [[1, 1], [2], [3, 3], [4]])
 *
 *
 * @since 1.10.0
 */
export const chop = (as, f) => {
    const result = [];
    let cs = as;
    while (cs.length > 0) {
        const [b, c] = f(cs);
        result.push(b);
        cs = c;
    }
    return result;
};
/**
 * Splits an array into two pieces, the first piece has `n` elements.
 *
 * @example
 * import { split } from 'fp-ts/lib/Array'
 *
 * assert.deepStrictEqual(split(2, [1, 2, 3, 4, 5]), [[1, 2], [3, 4, 5]])
 *
 *
 * @since 1.10.0
 */
export const split = (n, as) => {
    return [as.slice(0, n), as.slice(n)];
};
/**
 * Splits an array into length-`n` pieces. The last piece will be shorter if `n` does not evenly divide the length of
 * the array. Note that `chunksOf([], n)` is `[]`, not `[[]]`. This is intentional, and is consistent with a recursive
 * definition of `chunksOf`; it satisfies the property that
 *
 * ```ts
 * chunksOf(xs, n).concat(chunksOf(ys, n)) == chunksOf(xs.concat(ys)), n)
 * ```
 *
 * whenever `n` evenly divides the length of `xs`.
 *
 * @example
 * import { chunksOf } from 'fp-ts/lib/Array'
 *
 * assert.deepStrictEqual(chunksOf([1, 2, 3, 4, 5], 2), [[1, 2], [3, 4], [5]])
 *
 *
 * @since 1.10.0
 */
export const chunksOf = (as, n) => {
    return isOutOfBound(n - 1, as) ? [as] : chop(as, as => split(n, as));
};
export function comprehension(input, f, g) {
    const go = (scope, input) => {
        if (input.length === 0) {
            return f(...scope) ? [g(...scope)] : empty;
        }
        else {
            return chain(input[0], x => go(snoc(scope, x), input.slice(1)));
        }
    };
    return go(empty, input);
}
/**
 * Creates an array of unique values, in order, from all given arrays using a `Setoid` for equality comparisons
 *
 * @example
 * import { union } from 'fp-ts/lib/Array'
 * import { setoidNumber } from 'fp-ts/lib/Setoid'
 *
 * assert.deepStrictEqual(union(setoidNumber)([1, 2], [2, 3]), [1, 2, 3])
 *
 *
 * @since 1.12.0
 */
export const union = (S) => {
    const elemS = elem(S);
    return (xs, ys) => concat(xs, ys.filter(a => !elemS(a, xs)));
};
/**
 * Creates an array of unique values that are included in all given arrays using a `Setoid` for equality
 * comparisons. The order and references of result values are determined by the first array.
 *
 * @example
 * import { intersection } from 'fp-ts/lib/Array'
 * import { setoidNumber } from 'fp-ts/lib/Setoid'
 *
 * assert.deepStrictEqual(intersection(setoidNumber)([1, 2], [2, 3]), [2])
 *
 *
 * @since 1.12.0
 */
export const intersection = (S) => {
    const elemS = elem(S);
    return (xs, ys) => xs.filter(a => elemS(a, ys));
};
/**
 * Creates an array of array values not included in the other given array using a `Setoid` for equality
 * comparisons. The order and references of result values are determined by the first array.
 *
 * @example
 * import { difference } from 'fp-ts/lib/Array'
 * import { setoidNumber } from 'fp-ts/lib/Setoid'
 *
 * assert.deepStrictEqual(difference(setoidNumber)([1, 2], [2, 3]), [1])
 *
 *
 * @since 1.12.0
 */
export const difference = (S) => {
    const elemS = elem(S);
    return (xs, ys) => xs.filter(a => !elemS(a, ys));
};
const traverseWithIndex = (F) => (ta, f) => {
    return reduceWithIndex(ta, F.of(zero()), (i, fbs, a) => F.ap(F.map(fbs, bs => (b) => snoc(bs, b)), f(i, a)));
};
const partitionMapWithIndex = (fa, f) => {
    const left = [];
    const right = [];
    for (let i = 0; i < fa.length; i++) {
        const e = f(i, fa[i]);
        if (e.isLeft()) {
            left.push(e.value);
        }
        else {
            right.push(e.value);
        }
    }
    return {
        left,
        right
    };
};
const partitionWithIndex = (fa, p) => {
    const left = [];
    const right = [];
    for (let i = 0; i < fa.length; i++) {
        const a = fa[i];
        if (p(i, a)) {
            right.push(a);
        }
        else {
            left.push(a);
        }
    }
    return {
        left,
        right
    };
};
const filterMapWithIndex = (fa, f) => {
    const result = [];
    for (let i = 0; i < fa.length; i++) {
        const optionB = f(i, fa[i]);
        if (optionB.isSome()) {
            result.push(optionB.value);
        }
    }
    return result;
};
const filterWithIndex = (fa, p) => {
    return fa.filter((a, i) => p(i, a));
};
/**
 * @since 1.0.0
 */
export const array = {
    URI,
    map,
    mapWithIndex,
    compact,
    separate,
    filter,
    filterMap,
    partition,
    partitionMap,
    of,
    ap,
    chain,
    reduce,
    foldMap,
    foldr: reduceRight,
    unfoldr,
    traverse,
    sequence,
    zero,
    alt,
    extend,
    wither,
    wilt,
    reduceWithIndex,
    foldMapWithIndex,
    foldrWithIndex,
    traverseWithIndex,
    partitionMapWithIndex,
    partitionWithIndex,
    filterMapWithIndex,
    filterWithIndex
};
