"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildFilePatch = buildFilePatch;
exports.buildMultiFilePatch = buildMultiFilePatch;
exports.DEFAULT_OPTIONS = void 0;

var _patchBuffer = _interopRequireDefault(require("./patch-buffer"));

var _hunk = _interopRequireDefault(require("./hunk"));

var _file = _interopRequireWildcard(require("./file"));

var _patch = _interopRequireWildcard(require("./patch"));

var _region = require("./region");

var _filePatch = _interopRequireDefault(require("./file-patch"));

var _multiFilePatch = _interopRequireDefault(require("./multi-file-patch"));

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj["default"] = obj; return newObj; } }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const DEFAULT_OPTIONS = {
  // Number of lines after which we consider the diff "large"
  largeDiffThreshold: 800,
  // Map of file path (relative to repository root) to Patch render status (EXPANDED, COLLAPSED, TOO_LARGE)
  renderStatusOverrides: {},
  // Existing patch buffer to render onto
  patchBuffer: null,
  // Store off what-the-diff file patch
  preserveOriginal: false
};
exports.DEFAULT_OPTIONS = DEFAULT_OPTIONS;

function buildFilePatch(diffs, options) {
  const opts = _objectSpread({}, DEFAULT_OPTIONS, options);

  const patchBuffer = new _patchBuffer["default"]();
  let filePatch;

  if (diffs.length === 0) {
    filePatch = emptyDiffFilePatch();
  } else if (diffs.length === 1) {
    filePatch = singleDiffFilePatch(diffs[0], patchBuffer, opts);
  } else if (diffs.length === 2) {
    filePatch = dualDiffFilePatch(diffs[0], diffs[1], patchBuffer, opts);
  } else {
    throw new Error(`Unexpected number of diffs: ${diffs.length}`);
  } // Delete the trailing newline.


  patchBuffer.deleteLastNewline();
  return new _multiFilePatch["default"]({
    patchBuffer,
    filePatches: [filePatch]
  });
}

function buildMultiFilePatch(diffs, options) {
  const opts = _objectSpread({}, DEFAULT_OPTIONS, options);

  const patchBuffer = new _patchBuffer["default"]();
  const byPath = new Map();
  const actions = [];
  let index = 0;

  for (const diff of diffs) {
    const thePath = diff.oldPath || diff.newPath;

    if (diff.status === 'added' || diff.status === 'deleted') {
      // Potential paired diff. Either a symlink deletion + content addition or a symlink addition +
      // content deletion.
      const otherHalf = byPath.get(thePath);

      if (otherHalf) {
        // The second half. Complete the paired diff, or fail if they have unexpected statuses or modes.
        const [otherDiff, otherIndex] = otherHalf;

        actions[otherIndex] = function (_diff, _otherDiff) {
          return () => dualDiffFilePatch(_diff, _otherDiff, patchBuffer, opts);
        }(diff, otherDiff);

        byPath["delete"](thePath);
      } else {
        // The first half we've seen.
        byPath.set(thePath, [diff, index]);
        index++;
      }
    } else {
      actions[index] = function (_diff) {
        return () => singleDiffFilePatch(_diff, patchBuffer, opts);
      }(diff);

      index++;
    }
  } // Populate unpaired diffs that looked like they could be part of a pair, but weren't.


  for (const [unpairedDiff, originalIndex] of byPath.values()) {
    actions[originalIndex] = function (_unpairedDiff) {
      return () => singleDiffFilePatch(_unpairedDiff, patchBuffer, opts);
    }(unpairedDiff);
  }

  const filePatches = actions.map(action => action()); // Delete the final trailing newline from the last non-empty patch.

  patchBuffer.deleteLastNewline();
  return new _multiFilePatch["default"]({
    patchBuffer,
    filePatches
  });
}

function emptyDiffFilePatch() {
  return _filePatch["default"].createNull();
}

function singleDiffFilePatch(diff, patchBuffer, opts) {
  const wasSymlink = diff.oldMode === _file["default"].modes.SYMLINK;
  const isSymlink = diff.newMode === _file["default"].modes.SYMLINK;
  let oldSymlink = null;
  let newSymlink = null;

  if (wasSymlink && !isSymlink) {
    oldSymlink = diff.hunks[0].lines[0].slice(1);
  } else if (!wasSymlink && isSymlink) {
    newSymlink = diff.hunks[0].lines[0].slice(1);
  } else if (wasSymlink && isSymlink) {
    oldSymlink = diff.hunks[0].lines[0].slice(1);
    newSymlink = diff.hunks[0].lines[2].slice(1);
  }

  const oldFile = diff.oldPath !== null || diff.oldMode !== null ? new _file["default"]({
    path: diff.oldPath,
    mode: diff.oldMode,
    symlink: oldSymlink
  }) : _file.nullFile;
  const newFile = diff.newPath !== null || diff.newMode !== null ? new _file["default"]({
    path: diff.newPath,
    mode: diff.newMode,
    symlink: newSymlink
  }) : _file.nullFile;
  const renderStatusOverride = oldFile.isPresent() && opts.renderStatusOverrides[oldFile.getPath()] || newFile.isPresent() && opts.renderStatusOverrides[newFile.getPath()] || undefined;

  const renderStatus = renderStatusOverride || isDiffLarge([diff], opts) && _patch.TOO_LARGE || _patch.EXPANDED;

  if (!renderStatus.isVisible()) {
    const patchMarker = patchBuffer.markPosition(_patch["default"].layerName, patchBuffer.getBuffer().getEndPosition(), {
      invalidate: 'never',
      exclusive: false
    });
    return _filePatch["default"].createHiddenFilePatch(oldFile, newFile, patchMarker, renderStatus, () => {
      const subPatchBuffer = new _patchBuffer["default"]();
      const [hunks, nextPatchMarker] = buildHunks(diff, subPatchBuffer);
      const nextPatch = new _patch["default"]({
        status: diff.status,
        hunks,
        marker: nextPatchMarker
      });
      subPatchBuffer.deleteLastNewline();
      return {
        patch: nextPatch,
        patchBuffer: subPatchBuffer
      };
    });
  } else {
    const [hunks, patchMarker] = buildHunks(diff, patchBuffer);
    const patch = new _patch["default"]({
      status: diff.status,
      hunks,
      marker: patchMarker
    });
    const rawPatches = opts.preserveOriginal ? {
      content: diff
    } : null;
    return new _filePatch["default"](oldFile, newFile, patch, rawPatches);
  }
}

function dualDiffFilePatch(diff1, diff2, patchBuffer, opts) {
  let modeChangeDiff, contentChangeDiff;

  if (diff1.oldMode === _file["default"].modes.SYMLINK || diff1.newMode === _file["default"].modes.SYMLINK) {
    modeChangeDiff = diff1;
    contentChangeDiff = diff2;
  } else {
    modeChangeDiff = diff2;
    contentChangeDiff = diff1;
  }

  const filePath = contentChangeDiff.oldPath || contentChangeDiff.newPath;
  const symlink = modeChangeDiff.hunks[0].lines[0].slice(1);
  let status;
  let oldMode, newMode;
  let oldSymlink = null;
  let newSymlink = null;

  if (modeChangeDiff.status === 'added') {
    // contents were deleted and replaced with symlink
    status = 'deleted';
    oldMode = contentChangeDiff.oldMode;
    newMode = modeChangeDiff.newMode;
    newSymlink = symlink;
  } else if (modeChangeDiff.status === 'deleted') {
    // contents were added after symlink was deleted
    status = 'added';
    oldMode = modeChangeDiff.oldMode;
    oldSymlink = symlink;
    newMode = contentChangeDiff.newMode;
  } else {
    throw new Error(`Invalid mode change diff status: ${modeChangeDiff.status}`);
  }

  const oldFile = new _file["default"]({
    path: filePath,
    mode: oldMode,
    symlink: oldSymlink
  });
  const newFile = new _file["default"]({
    path: filePath,
    mode: newMode,
    symlink: newSymlink
  });

  const renderStatus = opts.renderStatusOverrides[filePath] || isDiffLarge([contentChangeDiff], opts) && _patch.TOO_LARGE || _patch.EXPANDED;

  if (!renderStatus.isVisible()) {
    const patchMarker = patchBuffer.markPosition(_patch["default"].layerName, patchBuffer.getBuffer().getEndPosition(), {
      invalidate: 'never',
      exclusive: false
    });
    return _filePatch["default"].createHiddenFilePatch(oldFile, newFile, patchMarker, renderStatus, () => {
      const subPatchBuffer = new _patchBuffer["default"]();
      const [hunks, nextPatchMarker] = buildHunks(contentChangeDiff, subPatchBuffer);
      const nextPatch = new _patch["default"]({
        status,
        hunks,
        marker: nextPatchMarker
      });
      subPatchBuffer.deleteLastNewline();
      return {
        patch: nextPatch,
        patchBuffer: subPatchBuffer
      };
    });
  } else {
    const [hunks, patchMarker] = buildHunks(contentChangeDiff, patchBuffer);
    const patch = new _patch["default"]({
      status,
      hunks,
      marker: patchMarker
    });
    const rawPatches = opts.preserveOriginal ? {
      content: contentChangeDiff,
      mode: modeChangeDiff
    } : null;
    return new _filePatch["default"](oldFile, newFile, patch, rawPatches);
  }
}

const CHANGEKIND = {
  '+': _region.Addition,
  '-': _region.Deletion,
  ' ': _region.Unchanged,
  '\\': _region.NoNewline
};

function buildHunks(diff, patchBuffer) {
  const inserter = patchBuffer.createInserterAtEnd().keepBefore(patchBuffer.findAllMarkers({
    endPosition: patchBuffer.getInsertionPoint()
  }));
  let patchMarker = null;
  let firstHunk = true;
  const hunks = [];
  inserter.markWhile(_patch["default"].layerName, () => {
    for (const rawHunk of diff.hunks) {
      let firstRegion = true;
      const regions = []; // Separate hunks with an unmarked newline

      if (firstHunk) {
        firstHunk = false;
      } else {
        inserter.insert('\n');
      }

      inserter.markWhile(_hunk["default"].layerName, () => {
        let firstRegionLine = true;
        let currentRegionText = '';
        let CurrentRegionKind = null;

        function finishRegion() {
          if (CurrentRegionKind === null) {
            return;
          } // Separate regions with an unmarked newline


          if (firstRegion) {
            firstRegion = false;
          } else {
            inserter.insert('\n');
          }

          inserter.insertMarked(currentRegionText, CurrentRegionKind.layerName, {
            invalidate: 'never',
            exclusive: false,
            callback: function (_regions, _CurrentRegionKind) {
              return regionMarker => {
                _regions.push(new _CurrentRegionKind(regionMarker));
              };
            }(regions, CurrentRegionKind)
          });
        }

        for (const rawLine of rawHunk.lines) {
          const NextRegionKind = CHANGEKIND[rawLine[0]];

          if (NextRegionKind === undefined) {
            throw new Error(`Unknown diff status character: "${rawLine[0]}"`);
          }

          const nextLine = rawLine.slice(1);
          let separator = '';

          if (firstRegionLine) {
            firstRegionLine = false;
          } else {
            separator = '\n';
          }

          if (NextRegionKind === CurrentRegionKind) {
            currentRegionText += separator + nextLine;
            continue;
          } else {
            finishRegion();
            CurrentRegionKind = NextRegionKind;
            currentRegionText = nextLine;
          }
        }

        finishRegion();
      }, {
        invalidate: 'never',
        exclusive: false,
        callback: function (_hunks, _rawHunk, _regions) {
          return hunkMarker => {
            _hunks.push(new _hunk["default"]({
              oldStartRow: _rawHunk.oldStartLine,
              newStartRow: _rawHunk.newStartLine,
              oldRowCount: _rawHunk.oldLineCount,
              newRowCount: _rawHunk.newLineCount,
              sectionHeading: _rawHunk.heading,
              marker: hunkMarker,
              regions: _regions
            }));
          };
        }(hunks, rawHunk, regions)
      });
    }
  }, {
    invalidate: 'never',
    exclusive: false,
    callback: marker => {
      patchMarker = marker;
    }
  }); // Separate multiple non-empty patches on the same buffer with an unmarked newline. The newline after the final
  // non-empty patch (if there is one) should be deleted before MultiFilePatch construction.

  if (diff.hunks.length > 0) {
    inserter.insert('\n');
  }

  inserter.apply();
  return [hunks, patchMarker];
}

function isDiffLarge(diffs, opts) {
  const size = diffs.reduce((diffSizeCounter, diff) => {
    return diffSizeCounter + diff.hunks.reduce((hunkSizeCounter, hunk) => {
      return hunkSizeCounter + hunk.lines.length;
    }, 0);
  }, 0);
  return size > opts.largeDiffThreshold;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImJ1aWxkZXIuanMiXSwibmFtZXMiOlsiREVGQVVMVF9PUFRJT05TIiwibGFyZ2VEaWZmVGhyZXNob2xkIiwicmVuZGVyU3RhdHVzT3ZlcnJpZGVzIiwicGF0Y2hCdWZmZXIiLCJwcmVzZXJ2ZU9yaWdpbmFsIiwiYnVpbGRGaWxlUGF0Y2giLCJkaWZmcyIsIm9wdGlvbnMiLCJvcHRzIiwiUGF0Y2hCdWZmZXIiLCJmaWxlUGF0Y2giLCJsZW5ndGgiLCJlbXB0eURpZmZGaWxlUGF0Y2giLCJzaW5nbGVEaWZmRmlsZVBhdGNoIiwiZHVhbERpZmZGaWxlUGF0Y2giLCJFcnJvciIsImRlbGV0ZUxhc3ROZXdsaW5lIiwiTXVsdGlGaWxlUGF0Y2giLCJmaWxlUGF0Y2hlcyIsImJ1aWxkTXVsdGlGaWxlUGF0Y2giLCJieVBhdGgiLCJNYXAiLCJhY3Rpb25zIiwiaW5kZXgiLCJkaWZmIiwidGhlUGF0aCIsIm9sZFBhdGgiLCJuZXdQYXRoIiwic3RhdHVzIiwib3RoZXJIYWxmIiwiZ2V0Iiwib3RoZXJEaWZmIiwib3RoZXJJbmRleCIsIl9kaWZmIiwiX290aGVyRGlmZiIsInNldCIsInVucGFpcmVkRGlmZiIsIm9yaWdpbmFsSW5kZXgiLCJ2YWx1ZXMiLCJfdW5wYWlyZWREaWZmIiwibWFwIiwiYWN0aW9uIiwiRmlsZVBhdGNoIiwiY3JlYXRlTnVsbCIsIndhc1N5bWxpbmsiLCJvbGRNb2RlIiwiRmlsZSIsIm1vZGVzIiwiU1lNTElOSyIsImlzU3ltbGluayIsIm5ld01vZGUiLCJvbGRTeW1saW5rIiwibmV3U3ltbGluayIsImh1bmtzIiwibGluZXMiLCJzbGljZSIsIm9sZEZpbGUiLCJwYXRoIiwibW9kZSIsInN5bWxpbmsiLCJudWxsRmlsZSIsIm5ld0ZpbGUiLCJyZW5kZXJTdGF0dXNPdmVycmlkZSIsImlzUHJlc2VudCIsImdldFBhdGgiLCJ1bmRlZmluZWQiLCJyZW5kZXJTdGF0dXMiLCJpc0RpZmZMYXJnZSIsIlRPT19MQVJHRSIsIkVYUEFOREVEIiwiaXNWaXNpYmxlIiwicGF0Y2hNYXJrZXIiLCJtYXJrUG9zaXRpb24iLCJQYXRjaCIsImxheWVyTmFtZSIsImdldEJ1ZmZlciIsImdldEVuZFBvc2l0aW9uIiwiaW52YWxpZGF0ZSIsImV4Y2x1c2l2ZSIsImNyZWF0ZUhpZGRlbkZpbGVQYXRjaCIsInN1YlBhdGNoQnVmZmVyIiwibmV4dFBhdGNoTWFya2VyIiwiYnVpbGRIdW5rcyIsIm5leHRQYXRjaCIsIm1hcmtlciIsInBhdGNoIiwicmF3UGF0Y2hlcyIsImNvbnRlbnQiLCJkaWZmMSIsImRpZmYyIiwibW9kZUNoYW5nZURpZmYiLCJjb250ZW50Q2hhbmdlRGlmZiIsImZpbGVQYXRoIiwiQ0hBTkdFS0lORCIsIkFkZGl0aW9uIiwiRGVsZXRpb24iLCJVbmNoYW5nZWQiLCJOb05ld2xpbmUiLCJpbnNlcnRlciIsImNyZWF0ZUluc2VydGVyQXRFbmQiLCJrZWVwQmVmb3JlIiwiZmluZEFsbE1hcmtlcnMiLCJlbmRQb3NpdGlvbiIsImdldEluc2VydGlvblBvaW50IiwiZmlyc3RIdW5rIiwibWFya1doaWxlIiwicmF3SHVuayIsImZpcnN0UmVnaW9uIiwicmVnaW9ucyIsImluc2VydCIsIkh1bmsiLCJmaXJzdFJlZ2lvbkxpbmUiLCJjdXJyZW50UmVnaW9uVGV4dCIsIkN1cnJlbnRSZWdpb25LaW5kIiwiZmluaXNoUmVnaW9uIiwiaW5zZXJ0TWFya2VkIiwiY2FsbGJhY2siLCJfcmVnaW9ucyIsIl9DdXJyZW50UmVnaW9uS2luZCIsInJlZ2lvbk1hcmtlciIsInB1c2giLCJyYXdMaW5lIiwiTmV4dFJlZ2lvbktpbmQiLCJuZXh0TGluZSIsInNlcGFyYXRvciIsIl9odW5rcyIsIl9yYXdIdW5rIiwiaHVua01hcmtlciIsIm9sZFN0YXJ0Um93Iiwib2xkU3RhcnRMaW5lIiwibmV3U3RhcnRSb3ciLCJuZXdTdGFydExpbmUiLCJvbGRSb3dDb3VudCIsIm9sZExpbmVDb3VudCIsIm5ld1Jvd0NvdW50IiwibmV3TGluZUNvdW50Iiwic2VjdGlvbkhlYWRpbmciLCJoZWFkaW5nIiwiYXBwbHkiLCJzaXplIiwicmVkdWNlIiwiZGlmZlNpemVDb3VudGVyIiwiaHVua1NpemVDb3VudGVyIiwiaHVuayJdLCJtYXBwaW5ncyI6Ijs7Ozs7Ozs7O0FBQUE7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7Ozs7Ozs7Ozs7QUFFTyxNQUFNQSxlQUFlLEdBQUc7QUFDN0I7QUFDQUMsRUFBQUEsa0JBQWtCLEVBQUUsR0FGUztBQUk3QjtBQUNBQyxFQUFBQSxxQkFBcUIsRUFBRSxFQUxNO0FBTzdCO0FBQ0FDLEVBQUFBLFdBQVcsRUFBRSxJQVJnQjtBQVU3QjtBQUNBQyxFQUFBQSxnQkFBZ0IsRUFBRTtBQVhXLENBQXhCOzs7QUFjQSxTQUFTQyxjQUFULENBQXdCQyxLQUF4QixFQUErQkMsT0FBL0IsRUFBd0M7QUFDN0MsUUFBTUMsSUFBSSxxQkFBT1IsZUFBUCxFQUEyQk8sT0FBM0IsQ0FBVjs7QUFDQSxRQUFNSixXQUFXLEdBQUcsSUFBSU0sdUJBQUosRUFBcEI7QUFFQSxNQUFJQyxTQUFKOztBQUNBLE1BQUlKLEtBQUssQ0FBQ0ssTUFBTixLQUFpQixDQUFyQixFQUF3QjtBQUN0QkQsSUFBQUEsU0FBUyxHQUFHRSxrQkFBa0IsRUFBOUI7QUFDRCxHQUZELE1BRU8sSUFBSU4sS0FBSyxDQUFDSyxNQUFOLEtBQWlCLENBQXJCLEVBQXdCO0FBQzdCRCxJQUFBQSxTQUFTLEdBQUdHLG1CQUFtQixDQUFDUCxLQUFLLENBQUMsQ0FBRCxDQUFOLEVBQVdILFdBQVgsRUFBd0JLLElBQXhCLENBQS9CO0FBQ0QsR0FGTSxNQUVBLElBQUlGLEtBQUssQ0FBQ0ssTUFBTixLQUFpQixDQUFyQixFQUF3QjtBQUM3QkQsSUFBQUEsU0FBUyxHQUFHSSxpQkFBaUIsQ0FBQ1IsS0FBSyxDQUFDLENBQUQsQ0FBTixFQUFXQSxLQUFLLENBQUMsQ0FBRCxDQUFoQixFQUFxQkgsV0FBckIsRUFBa0NLLElBQWxDLENBQTdCO0FBQ0QsR0FGTSxNQUVBO0FBQ0wsVUFBTSxJQUFJTyxLQUFKLENBQVcsK0JBQThCVCxLQUFLLENBQUNLLE1BQU8sRUFBdEQsQ0FBTjtBQUNELEdBYjRDLENBZTdDOzs7QUFDQVIsRUFBQUEsV0FBVyxDQUFDYSxpQkFBWjtBQUVBLFNBQU8sSUFBSUMsMEJBQUosQ0FBbUI7QUFBQ2QsSUFBQUEsV0FBRDtBQUFjZSxJQUFBQSxXQUFXLEVBQUUsQ0FBQ1IsU0FBRDtBQUEzQixHQUFuQixDQUFQO0FBQ0Q7O0FBRU0sU0FBU1MsbUJBQVQsQ0FBNkJiLEtBQTdCLEVBQW9DQyxPQUFwQyxFQUE2QztBQUNsRCxRQUFNQyxJQUFJLHFCQUFPUixlQUFQLEVBQTJCTyxPQUEzQixDQUFWOztBQUVBLFFBQU1KLFdBQVcsR0FBRyxJQUFJTSx1QkFBSixFQUFwQjtBQUVBLFFBQU1XLE1BQU0sR0FBRyxJQUFJQyxHQUFKLEVBQWY7QUFDQSxRQUFNQyxPQUFPLEdBQUcsRUFBaEI7QUFFQSxNQUFJQyxLQUFLLEdBQUcsQ0FBWjs7QUFDQSxPQUFLLE1BQU1DLElBQVgsSUFBbUJsQixLQUFuQixFQUEwQjtBQUN4QixVQUFNbUIsT0FBTyxHQUFHRCxJQUFJLENBQUNFLE9BQUwsSUFBZ0JGLElBQUksQ0FBQ0csT0FBckM7O0FBRUEsUUFBSUgsSUFBSSxDQUFDSSxNQUFMLEtBQWdCLE9BQWhCLElBQTJCSixJQUFJLENBQUNJLE1BQUwsS0FBZ0IsU0FBL0MsRUFBMEQ7QUFDeEQ7QUFDQTtBQUNBLFlBQU1DLFNBQVMsR0FBR1QsTUFBTSxDQUFDVSxHQUFQLENBQVdMLE9BQVgsQ0FBbEI7O0FBQ0EsVUFBSUksU0FBSixFQUFlO0FBQ2I7QUFDQSxjQUFNLENBQUNFLFNBQUQsRUFBWUMsVUFBWixJQUEwQkgsU0FBaEM7O0FBQ0FQLFFBQUFBLE9BQU8sQ0FBQ1UsVUFBRCxDQUFQLEdBQXVCLFVBQVNDLEtBQVQsRUFBZ0JDLFVBQWhCLEVBQTRCO0FBQ2pELGlCQUFPLE1BQU1wQixpQkFBaUIsQ0FBQ21CLEtBQUQsRUFBUUMsVUFBUixFQUFvQi9CLFdBQXBCLEVBQWlDSyxJQUFqQyxDQUE5QjtBQUNELFNBRnFCLENBRW5CZ0IsSUFGbUIsRUFFYk8sU0FGYSxDQUF0Qjs7QUFHQVgsUUFBQUEsTUFBTSxVQUFOLENBQWNLLE9BQWQ7QUFDRCxPQVBELE1BT087QUFDTDtBQUNBTCxRQUFBQSxNQUFNLENBQUNlLEdBQVAsQ0FBV1YsT0FBWCxFQUFvQixDQUFDRCxJQUFELEVBQU9ELEtBQVAsQ0FBcEI7QUFDQUEsUUFBQUEsS0FBSztBQUNOO0FBQ0YsS0FoQkQsTUFnQk87QUFDTEQsTUFBQUEsT0FBTyxDQUFDQyxLQUFELENBQVAsR0FBa0IsVUFBU1UsS0FBVCxFQUFnQjtBQUNoQyxlQUFPLE1BQU1wQixtQkFBbUIsQ0FBQ29CLEtBQUQsRUFBUTlCLFdBQVIsRUFBcUJLLElBQXJCLENBQWhDO0FBQ0QsT0FGZ0IsQ0FFZGdCLElBRmMsQ0FBakI7O0FBR0FELE1BQUFBLEtBQUs7QUFDTjtBQUNGLEdBbENpRCxDQW9DbEQ7OztBQUNBLE9BQUssTUFBTSxDQUFDYSxZQUFELEVBQWVDLGFBQWYsQ0FBWCxJQUE0Q2pCLE1BQU0sQ0FBQ2tCLE1BQVAsRUFBNUMsRUFBNkQ7QUFDM0RoQixJQUFBQSxPQUFPLENBQUNlLGFBQUQsQ0FBUCxHQUEwQixVQUFTRSxhQUFULEVBQXdCO0FBQ2hELGFBQU8sTUFBTTFCLG1CQUFtQixDQUFDMEIsYUFBRCxFQUFnQnBDLFdBQWhCLEVBQTZCSyxJQUE3QixDQUFoQztBQUNELEtBRndCLENBRXRCNEIsWUFGc0IsQ0FBekI7QUFHRDs7QUFFRCxRQUFNbEIsV0FBVyxHQUFHSSxPQUFPLENBQUNrQixHQUFSLENBQVlDLE1BQU0sSUFBSUEsTUFBTSxFQUE1QixDQUFwQixDQTNDa0QsQ0E2Q2xEOztBQUNBdEMsRUFBQUEsV0FBVyxDQUFDYSxpQkFBWjtBQUVBLFNBQU8sSUFBSUMsMEJBQUosQ0FBbUI7QUFBQ2QsSUFBQUEsV0FBRDtBQUFjZSxJQUFBQTtBQUFkLEdBQW5CLENBQVA7QUFDRDs7QUFFRCxTQUFTTixrQkFBVCxHQUE4QjtBQUM1QixTQUFPOEIsc0JBQVVDLFVBQVYsRUFBUDtBQUNEOztBQUVELFNBQVM5QixtQkFBVCxDQUE2QlcsSUFBN0IsRUFBbUNyQixXQUFuQyxFQUFnREssSUFBaEQsRUFBc0Q7QUFDcEQsUUFBTW9DLFVBQVUsR0FBR3BCLElBQUksQ0FBQ3FCLE9BQUwsS0FBaUJDLGlCQUFLQyxLQUFMLENBQVdDLE9BQS9DO0FBQ0EsUUFBTUMsU0FBUyxHQUFHekIsSUFBSSxDQUFDMEIsT0FBTCxLQUFpQkosaUJBQUtDLEtBQUwsQ0FBV0MsT0FBOUM7QUFFQSxNQUFJRyxVQUFVLEdBQUcsSUFBakI7QUFDQSxNQUFJQyxVQUFVLEdBQUcsSUFBakI7O0FBQ0EsTUFBSVIsVUFBVSxJQUFJLENBQUNLLFNBQW5CLEVBQThCO0FBQzVCRSxJQUFBQSxVQUFVLEdBQUczQixJQUFJLENBQUM2QixLQUFMLENBQVcsQ0FBWCxFQUFjQyxLQUFkLENBQW9CLENBQXBCLEVBQXVCQyxLQUF2QixDQUE2QixDQUE3QixDQUFiO0FBQ0QsR0FGRCxNQUVPLElBQUksQ0FBQ1gsVUFBRCxJQUFlSyxTQUFuQixFQUE4QjtBQUNuQ0csSUFBQUEsVUFBVSxHQUFHNUIsSUFBSSxDQUFDNkIsS0FBTCxDQUFXLENBQVgsRUFBY0MsS0FBZCxDQUFvQixDQUFwQixFQUF1QkMsS0FBdkIsQ0FBNkIsQ0FBN0IsQ0FBYjtBQUNELEdBRk0sTUFFQSxJQUFJWCxVQUFVLElBQUlLLFNBQWxCLEVBQTZCO0FBQ2xDRSxJQUFBQSxVQUFVLEdBQUczQixJQUFJLENBQUM2QixLQUFMLENBQVcsQ0FBWCxFQUFjQyxLQUFkLENBQW9CLENBQXBCLEVBQXVCQyxLQUF2QixDQUE2QixDQUE3QixDQUFiO0FBQ0FILElBQUFBLFVBQVUsR0FBRzVCLElBQUksQ0FBQzZCLEtBQUwsQ0FBVyxDQUFYLEVBQWNDLEtBQWQsQ0FBb0IsQ0FBcEIsRUFBdUJDLEtBQXZCLENBQTZCLENBQTdCLENBQWI7QUFDRDs7QUFFRCxRQUFNQyxPQUFPLEdBQUdoQyxJQUFJLENBQUNFLE9BQUwsS0FBaUIsSUFBakIsSUFBeUJGLElBQUksQ0FBQ3FCLE9BQUwsS0FBaUIsSUFBMUMsR0FDWixJQUFJQyxnQkFBSixDQUFTO0FBQUNXLElBQUFBLElBQUksRUFBRWpDLElBQUksQ0FBQ0UsT0FBWjtBQUFxQmdDLElBQUFBLElBQUksRUFBRWxDLElBQUksQ0FBQ3FCLE9BQWhDO0FBQXlDYyxJQUFBQSxPQUFPLEVBQUVSO0FBQWxELEdBQVQsQ0FEWSxHQUVaUyxjQUZKO0FBR0EsUUFBTUMsT0FBTyxHQUFHckMsSUFBSSxDQUFDRyxPQUFMLEtBQWlCLElBQWpCLElBQXlCSCxJQUFJLENBQUMwQixPQUFMLEtBQWlCLElBQTFDLEdBQ1osSUFBSUosZ0JBQUosQ0FBUztBQUFDVyxJQUFBQSxJQUFJLEVBQUVqQyxJQUFJLENBQUNHLE9BQVo7QUFBcUIrQixJQUFBQSxJQUFJLEVBQUVsQyxJQUFJLENBQUMwQixPQUFoQztBQUF5Q1MsSUFBQUEsT0FBTyxFQUFFUDtBQUFsRCxHQUFULENBRFksR0FFWlEsY0FGSjtBQUlBLFFBQU1FLG9CQUFvQixHQUN2Qk4sT0FBTyxDQUFDTyxTQUFSLE1BQXVCdkQsSUFBSSxDQUFDTixxQkFBTCxDQUEyQnNELE9BQU8sQ0FBQ1EsT0FBUixFQUEzQixDQUF4QixJQUNDSCxPQUFPLENBQUNFLFNBQVIsTUFBdUJ2RCxJQUFJLENBQUNOLHFCQUFMLENBQTJCMkQsT0FBTyxDQUFDRyxPQUFSLEVBQTNCLENBRHhCLElBRUFDLFNBSEY7O0FBS0EsUUFBTUMsWUFBWSxHQUFHSixvQkFBb0IsSUFDdENLLFdBQVcsQ0FBQyxDQUFDM0MsSUFBRCxDQUFELEVBQVNoQixJQUFULENBQVgsSUFBNkI0RCxnQkFEWCxJQUVuQkMsZUFGRjs7QUFJQSxNQUFJLENBQUNILFlBQVksQ0FBQ0ksU0FBYixFQUFMLEVBQStCO0FBQzdCLFVBQU1DLFdBQVcsR0FBR3BFLFdBQVcsQ0FBQ3FFLFlBQVosQ0FDbEJDLGtCQUFNQyxTQURZLEVBRWxCdkUsV0FBVyxDQUFDd0UsU0FBWixHQUF3QkMsY0FBeEIsRUFGa0IsRUFHbEI7QUFBQ0MsTUFBQUEsVUFBVSxFQUFFLE9BQWI7QUFBc0JDLE1BQUFBLFNBQVMsRUFBRTtBQUFqQyxLQUhrQixDQUFwQjtBQU1BLFdBQU9wQyxzQkFBVXFDLHFCQUFWLENBQ0x2QixPQURLLEVBQ0lLLE9BREosRUFDYVUsV0FEYixFQUMwQkwsWUFEMUIsRUFFTCxNQUFNO0FBQ0osWUFBTWMsY0FBYyxHQUFHLElBQUl2RSx1QkFBSixFQUF2QjtBQUNBLFlBQU0sQ0FBQzRDLEtBQUQsRUFBUTRCLGVBQVIsSUFBMkJDLFVBQVUsQ0FBQzFELElBQUQsRUFBT3dELGNBQVAsQ0FBM0M7QUFDQSxZQUFNRyxTQUFTLEdBQUcsSUFBSVYsaUJBQUosQ0FBVTtBQUFDN0MsUUFBQUEsTUFBTSxFQUFFSixJQUFJLENBQUNJLE1BQWQ7QUFBc0J5QixRQUFBQSxLQUF0QjtBQUE2QitCLFFBQUFBLE1BQU0sRUFBRUg7QUFBckMsT0FBVixDQUFsQjtBQUVBRCxNQUFBQSxjQUFjLENBQUNoRSxpQkFBZjtBQUNBLGFBQU87QUFBQ3FFLFFBQUFBLEtBQUssRUFBRUYsU0FBUjtBQUFtQmhGLFFBQUFBLFdBQVcsRUFBRTZFO0FBQWhDLE9BQVA7QUFDRCxLQVRJLENBQVA7QUFXRCxHQWxCRCxNQWtCTztBQUNMLFVBQU0sQ0FBQzNCLEtBQUQsRUFBUWtCLFdBQVIsSUFBdUJXLFVBQVUsQ0FBQzFELElBQUQsRUFBT3JCLFdBQVAsQ0FBdkM7QUFDQSxVQUFNa0YsS0FBSyxHQUFHLElBQUlaLGlCQUFKLENBQVU7QUFBQzdDLE1BQUFBLE1BQU0sRUFBRUosSUFBSSxDQUFDSSxNQUFkO0FBQXNCeUIsTUFBQUEsS0FBdEI7QUFBNkIrQixNQUFBQSxNQUFNLEVBQUViO0FBQXJDLEtBQVYsQ0FBZDtBQUVBLFVBQU1lLFVBQVUsR0FBRzlFLElBQUksQ0FBQ0osZ0JBQUwsR0FBd0I7QUFBQ21GLE1BQUFBLE9BQU8sRUFBRS9EO0FBQVYsS0FBeEIsR0FBMEMsSUFBN0Q7QUFDQSxXQUFPLElBQUlrQixxQkFBSixDQUFjYyxPQUFkLEVBQXVCSyxPQUF2QixFQUFnQ3dCLEtBQWhDLEVBQXVDQyxVQUF2QyxDQUFQO0FBQ0Q7QUFDRjs7QUFFRCxTQUFTeEUsaUJBQVQsQ0FBMkIwRSxLQUEzQixFQUFrQ0MsS0FBbEMsRUFBeUN0RixXQUF6QyxFQUFzREssSUFBdEQsRUFBNEQ7QUFDMUQsTUFBSWtGLGNBQUosRUFBb0JDLGlCQUFwQjs7QUFDQSxNQUFJSCxLQUFLLENBQUMzQyxPQUFOLEtBQWtCQyxpQkFBS0MsS0FBTCxDQUFXQyxPQUE3QixJQUF3Q3dDLEtBQUssQ0FBQ3RDLE9BQU4sS0FBa0JKLGlCQUFLQyxLQUFMLENBQVdDLE9BQXpFLEVBQWtGO0FBQ2hGMEMsSUFBQUEsY0FBYyxHQUFHRixLQUFqQjtBQUNBRyxJQUFBQSxpQkFBaUIsR0FBR0YsS0FBcEI7QUFDRCxHQUhELE1BR087QUFDTEMsSUFBQUEsY0FBYyxHQUFHRCxLQUFqQjtBQUNBRSxJQUFBQSxpQkFBaUIsR0FBR0gsS0FBcEI7QUFDRDs7QUFFRCxRQUFNSSxRQUFRLEdBQUdELGlCQUFpQixDQUFDakUsT0FBbEIsSUFBNkJpRSxpQkFBaUIsQ0FBQ2hFLE9BQWhFO0FBQ0EsUUFBTWdDLE9BQU8sR0FBRytCLGNBQWMsQ0FBQ3JDLEtBQWYsQ0FBcUIsQ0FBckIsRUFBd0JDLEtBQXhCLENBQThCLENBQTlCLEVBQWlDQyxLQUFqQyxDQUF1QyxDQUF2QyxDQUFoQjtBQUVBLE1BQUkzQixNQUFKO0FBQ0EsTUFBSWlCLE9BQUosRUFBYUssT0FBYjtBQUNBLE1BQUlDLFVBQVUsR0FBRyxJQUFqQjtBQUNBLE1BQUlDLFVBQVUsR0FBRyxJQUFqQjs7QUFDQSxNQUFJc0MsY0FBYyxDQUFDOUQsTUFBZixLQUEwQixPQUE5QixFQUF1QztBQUNyQztBQUNBQSxJQUFBQSxNQUFNLEdBQUcsU0FBVDtBQUNBaUIsSUFBQUEsT0FBTyxHQUFHOEMsaUJBQWlCLENBQUM5QyxPQUE1QjtBQUNBSyxJQUFBQSxPQUFPLEdBQUd3QyxjQUFjLENBQUN4QyxPQUF6QjtBQUNBRSxJQUFBQSxVQUFVLEdBQUdPLE9BQWI7QUFDRCxHQU5ELE1BTU8sSUFBSStCLGNBQWMsQ0FBQzlELE1BQWYsS0FBMEIsU0FBOUIsRUFBeUM7QUFDOUM7QUFDQUEsSUFBQUEsTUFBTSxHQUFHLE9BQVQ7QUFDQWlCLElBQUFBLE9BQU8sR0FBRzZDLGNBQWMsQ0FBQzdDLE9BQXpCO0FBQ0FNLElBQUFBLFVBQVUsR0FBR1EsT0FBYjtBQUNBVCxJQUFBQSxPQUFPLEdBQUd5QyxpQkFBaUIsQ0FBQ3pDLE9BQTVCO0FBQ0QsR0FOTSxNQU1BO0FBQ0wsVUFBTSxJQUFJbkMsS0FBSixDQUFXLG9DQUFtQzJFLGNBQWMsQ0FBQzlELE1BQU8sRUFBcEUsQ0FBTjtBQUNEOztBQUVELFFBQU00QixPQUFPLEdBQUcsSUFBSVYsZ0JBQUosQ0FBUztBQUFDVyxJQUFBQSxJQUFJLEVBQUVtQyxRQUFQO0FBQWlCbEMsSUFBQUEsSUFBSSxFQUFFYixPQUF2QjtBQUFnQ2MsSUFBQUEsT0FBTyxFQUFFUjtBQUF6QyxHQUFULENBQWhCO0FBQ0EsUUFBTVUsT0FBTyxHQUFHLElBQUlmLGdCQUFKLENBQVM7QUFBQ1csSUFBQUEsSUFBSSxFQUFFbUMsUUFBUDtBQUFpQmxDLElBQUFBLElBQUksRUFBRVIsT0FBdkI7QUFBZ0NTLElBQUFBLE9BQU8sRUFBRVA7QUFBekMsR0FBVCxDQUFoQjs7QUFFQSxRQUFNYyxZQUFZLEdBQUcxRCxJQUFJLENBQUNOLHFCQUFMLENBQTJCMEYsUUFBM0IsS0FDbEJ6QixXQUFXLENBQUMsQ0FBQ3dCLGlCQUFELENBQUQsRUFBc0JuRixJQUF0QixDQUFYLElBQTBDNEQsZ0JBRHhCLElBRW5CQyxlQUZGOztBQUlBLE1BQUksQ0FBQ0gsWUFBWSxDQUFDSSxTQUFiLEVBQUwsRUFBK0I7QUFDN0IsVUFBTUMsV0FBVyxHQUFHcEUsV0FBVyxDQUFDcUUsWUFBWixDQUNsQkMsa0JBQU1DLFNBRFksRUFFbEJ2RSxXQUFXLENBQUN3RSxTQUFaLEdBQXdCQyxjQUF4QixFQUZrQixFQUdsQjtBQUFDQyxNQUFBQSxVQUFVLEVBQUUsT0FBYjtBQUFzQkMsTUFBQUEsU0FBUyxFQUFFO0FBQWpDLEtBSGtCLENBQXBCO0FBTUEsV0FBT3BDLHNCQUFVcUMscUJBQVYsQ0FDTHZCLE9BREssRUFDSUssT0FESixFQUNhVSxXQURiLEVBQzBCTCxZQUQxQixFQUVMLE1BQU07QUFDSixZQUFNYyxjQUFjLEdBQUcsSUFBSXZFLHVCQUFKLEVBQXZCO0FBQ0EsWUFBTSxDQUFDNEMsS0FBRCxFQUFRNEIsZUFBUixJQUEyQkMsVUFBVSxDQUFDUyxpQkFBRCxFQUFvQlgsY0FBcEIsQ0FBM0M7QUFDQSxZQUFNRyxTQUFTLEdBQUcsSUFBSVYsaUJBQUosQ0FBVTtBQUFDN0MsUUFBQUEsTUFBRDtBQUFTeUIsUUFBQUEsS0FBVDtBQUFnQitCLFFBQUFBLE1BQU0sRUFBRUg7QUFBeEIsT0FBVixDQUFsQjtBQUVBRCxNQUFBQSxjQUFjLENBQUNoRSxpQkFBZjtBQUNBLGFBQU87QUFBQ3FFLFFBQUFBLEtBQUssRUFBRUYsU0FBUjtBQUFtQmhGLFFBQUFBLFdBQVcsRUFBRTZFO0FBQWhDLE9BQVA7QUFDRCxLQVRJLENBQVA7QUFXRCxHQWxCRCxNQWtCTztBQUNMLFVBQU0sQ0FBQzNCLEtBQUQsRUFBUWtCLFdBQVIsSUFBdUJXLFVBQVUsQ0FBQ1MsaUJBQUQsRUFBb0J4RixXQUFwQixDQUF2QztBQUNBLFVBQU1rRixLQUFLLEdBQUcsSUFBSVosaUJBQUosQ0FBVTtBQUFDN0MsTUFBQUEsTUFBRDtBQUFTeUIsTUFBQUEsS0FBVDtBQUFnQitCLE1BQUFBLE1BQU0sRUFBRWI7QUFBeEIsS0FBVixDQUFkO0FBRUEsVUFBTWUsVUFBVSxHQUFHOUUsSUFBSSxDQUFDSixnQkFBTCxHQUF3QjtBQUFDbUYsTUFBQUEsT0FBTyxFQUFFSSxpQkFBVjtBQUE2QmpDLE1BQUFBLElBQUksRUFBRWdDO0FBQW5DLEtBQXhCLEdBQTZFLElBQWhHO0FBQ0EsV0FBTyxJQUFJaEQscUJBQUosQ0FBY2MsT0FBZCxFQUF1QkssT0FBdkIsRUFBZ0N3QixLQUFoQyxFQUF1Q0MsVUFBdkMsQ0FBUDtBQUNEO0FBQ0Y7O0FBRUQsTUFBTU8sVUFBVSxHQUFHO0FBQ2pCLE9BQUtDLGdCQURZO0FBRWpCLE9BQUtDLGdCQUZZO0FBR2pCLE9BQUtDLGlCQUhZO0FBSWpCLFFBQU1DO0FBSlcsQ0FBbkI7O0FBT0EsU0FBU2YsVUFBVCxDQUFvQjFELElBQXBCLEVBQTBCckIsV0FBMUIsRUFBdUM7QUFDckMsUUFBTStGLFFBQVEsR0FBRy9GLFdBQVcsQ0FBQ2dHLG1CQUFaLEdBQ2RDLFVBRGMsQ0FDSGpHLFdBQVcsQ0FBQ2tHLGNBQVosQ0FBMkI7QUFBQ0MsSUFBQUEsV0FBVyxFQUFFbkcsV0FBVyxDQUFDb0csaUJBQVo7QUFBZCxHQUEzQixDQURHLENBQWpCO0FBR0EsTUFBSWhDLFdBQVcsR0FBRyxJQUFsQjtBQUNBLE1BQUlpQyxTQUFTLEdBQUcsSUFBaEI7QUFDQSxRQUFNbkQsS0FBSyxHQUFHLEVBQWQ7QUFFQTZDLEVBQUFBLFFBQVEsQ0FBQ08sU0FBVCxDQUFtQmhDLGtCQUFNQyxTQUF6QixFQUFvQyxNQUFNO0FBQ3hDLFNBQUssTUFBTWdDLE9BQVgsSUFBc0JsRixJQUFJLENBQUM2QixLQUEzQixFQUFrQztBQUNoQyxVQUFJc0QsV0FBVyxHQUFHLElBQWxCO0FBQ0EsWUFBTUMsT0FBTyxHQUFHLEVBQWhCLENBRmdDLENBSWhDOztBQUNBLFVBQUlKLFNBQUosRUFBZTtBQUNiQSxRQUFBQSxTQUFTLEdBQUcsS0FBWjtBQUNELE9BRkQsTUFFTztBQUNMTixRQUFBQSxRQUFRLENBQUNXLE1BQVQsQ0FBZ0IsSUFBaEI7QUFDRDs7QUFFRFgsTUFBQUEsUUFBUSxDQUFDTyxTQUFULENBQW1CSyxpQkFBS3BDLFNBQXhCLEVBQW1DLE1BQU07QUFDdkMsWUFBSXFDLGVBQWUsR0FBRyxJQUF0QjtBQUNBLFlBQUlDLGlCQUFpQixHQUFHLEVBQXhCO0FBQ0EsWUFBSUMsaUJBQWlCLEdBQUcsSUFBeEI7O0FBRUEsaUJBQVNDLFlBQVQsR0FBd0I7QUFDdEIsY0FBSUQsaUJBQWlCLEtBQUssSUFBMUIsRUFBZ0M7QUFDOUI7QUFDRCxXQUhxQixDQUt0Qjs7O0FBQ0EsY0FBSU4sV0FBSixFQUFpQjtBQUNmQSxZQUFBQSxXQUFXLEdBQUcsS0FBZDtBQUNELFdBRkQsTUFFTztBQUNMVCxZQUFBQSxRQUFRLENBQUNXLE1BQVQsQ0FBZ0IsSUFBaEI7QUFDRDs7QUFFRFgsVUFBQUEsUUFBUSxDQUFDaUIsWUFBVCxDQUFzQkgsaUJBQXRCLEVBQXlDQyxpQkFBaUIsQ0FBQ3ZDLFNBQTNELEVBQXNFO0FBQ3BFRyxZQUFBQSxVQUFVLEVBQUUsT0FEd0Q7QUFFcEVDLFlBQUFBLFNBQVMsRUFBRSxLQUZ5RDtBQUdwRXNDLFlBQUFBLFFBQVEsRUFBRyxVQUFTQyxRQUFULEVBQW1CQyxrQkFBbkIsRUFBdUM7QUFDaEQscUJBQU9DLFlBQVksSUFBSTtBQUFFRixnQkFBQUEsUUFBUSxDQUFDRyxJQUFULENBQWMsSUFBSUYsa0JBQUosQ0FBdUJDLFlBQXZCLENBQWQ7QUFBc0QsZUFBL0U7QUFDRCxhQUZTLENBRVBYLE9BRk8sRUFFRUssaUJBRkY7QUFIMEQsV0FBdEU7QUFPRDs7QUFFRCxhQUFLLE1BQU1RLE9BQVgsSUFBc0JmLE9BQU8sQ0FBQ3BELEtBQTlCLEVBQXFDO0FBQ25DLGdCQUFNb0UsY0FBYyxHQUFHN0IsVUFBVSxDQUFDNEIsT0FBTyxDQUFDLENBQUQsQ0FBUixDQUFqQzs7QUFDQSxjQUFJQyxjQUFjLEtBQUt6RCxTQUF2QixFQUFrQztBQUNoQyxrQkFBTSxJQUFJbEQsS0FBSixDQUFXLG1DQUFrQzBHLE9BQU8sQ0FBQyxDQUFELENBQUksR0FBeEQsQ0FBTjtBQUNEOztBQUNELGdCQUFNRSxRQUFRLEdBQUdGLE9BQU8sQ0FBQ2xFLEtBQVIsQ0FBYyxDQUFkLENBQWpCO0FBRUEsY0FBSXFFLFNBQVMsR0FBRyxFQUFoQjs7QUFDQSxjQUFJYixlQUFKLEVBQXFCO0FBQ25CQSxZQUFBQSxlQUFlLEdBQUcsS0FBbEI7QUFDRCxXQUZELE1BRU87QUFDTGEsWUFBQUEsU0FBUyxHQUFHLElBQVo7QUFDRDs7QUFFRCxjQUFJRixjQUFjLEtBQUtULGlCQUF2QixFQUEwQztBQUN4Q0QsWUFBQUEsaUJBQWlCLElBQUlZLFNBQVMsR0FBR0QsUUFBakM7QUFFQTtBQUNELFdBSkQsTUFJTztBQUNMVCxZQUFBQSxZQUFZO0FBRVpELFlBQUFBLGlCQUFpQixHQUFHUyxjQUFwQjtBQUNBVixZQUFBQSxpQkFBaUIsR0FBR1csUUFBcEI7QUFDRDtBQUNGOztBQUNEVCxRQUFBQSxZQUFZO0FBQ2IsT0FwREQsRUFvREc7QUFDRHJDLFFBQUFBLFVBQVUsRUFBRSxPQURYO0FBRURDLFFBQUFBLFNBQVMsRUFBRSxLQUZWO0FBR0RzQyxRQUFBQSxRQUFRLEVBQUcsVUFBU1MsTUFBVCxFQUFpQkMsUUFBakIsRUFBMkJULFFBQTNCLEVBQXFDO0FBQzlDLGlCQUFPVSxVQUFVLElBQUk7QUFDbkJGLFlBQUFBLE1BQU0sQ0FBQ0wsSUFBUCxDQUFZLElBQUlWLGdCQUFKLENBQVM7QUFDbkJrQixjQUFBQSxXQUFXLEVBQUVGLFFBQVEsQ0FBQ0csWUFESDtBQUVuQkMsY0FBQUEsV0FBVyxFQUFFSixRQUFRLENBQUNLLFlBRkg7QUFHbkJDLGNBQUFBLFdBQVcsRUFBRU4sUUFBUSxDQUFDTyxZQUhIO0FBSW5CQyxjQUFBQSxXQUFXLEVBQUVSLFFBQVEsQ0FBQ1MsWUFKSDtBQUtuQkMsY0FBQUEsY0FBYyxFQUFFVixRQUFRLENBQUNXLE9BTE47QUFNbkJyRCxjQUFBQSxNQUFNLEVBQUUyQyxVQU5XO0FBT25CbkIsY0FBQUEsT0FBTyxFQUFFUztBQVBVLGFBQVQsQ0FBWjtBQVNELFdBVkQ7QUFXRCxTQVpTLENBWVBoRSxLQVpPLEVBWUFxRCxPQVpBLEVBWVNFLE9BWlQ7QUFIVCxPQXBESDtBQXFFRDtBQUNGLEdBbEZELEVBa0ZHO0FBQ0QvQixJQUFBQSxVQUFVLEVBQUUsT0FEWDtBQUVEQyxJQUFBQSxTQUFTLEVBQUUsS0FGVjtBQUdEc0MsSUFBQUEsUUFBUSxFQUFFaEMsTUFBTSxJQUFJO0FBQUViLE1BQUFBLFdBQVcsR0FBR2EsTUFBZDtBQUF1QjtBQUg1QyxHQWxGSCxFQVJxQyxDQWdHckM7QUFDQTs7QUFDQSxNQUFJNUQsSUFBSSxDQUFDNkIsS0FBTCxDQUFXMUMsTUFBWCxHQUFvQixDQUF4QixFQUEyQjtBQUN6QnVGLElBQUFBLFFBQVEsQ0FBQ1csTUFBVCxDQUFnQixJQUFoQjtBQUNEOztBQUVEWCxFQUFBQSxRQUFRLENBQUN3QyxLQUFUO0FBRUEsU0FBTyxDQUFDckYsS0FBRCxFQUFRa0IsV0FBUixDQUFQO0FBQ0Q7O0FBRUQsU0FBU0osV0FBVCxDQUFxQjdELEtBQXJCLEVBQTRCRSxJQUE1QixFQUFrQztBQUNoQyxRQUFNbUksSUFBSSxHQUFHckksS0FBSyxDQUFDc0ksTUFBTixDQUFhLENBQUNDLGVBQUQsRUFBa0JySCxJQUFsQixLQUEyQjtBQUNuRCxXQUFPcUgsZUFBZSxHQUFHckgsSUFBSSxDQUFDNkIsS0FBTCxDQUFXdUYsTUFBWCxDQUFrQixDQUFDRSxlQUFELEVBQWtCQyxJQUFsQixLQUEyQjtBQUNwRSxhQUFPRCxlQUFlLEdBQUdDLElBQUksQ0FBQ3pGLEtBQUwsQ0FBVzNDLE1BQXBDO0FBQ0QsS0FGd0IsRUFFdEIsQ0FGc0IsQ0FBekI7QUFHRCxHQUpZLEVBSVYsQ0FKVSxDQUFiO0FBTUEsU0FBT2dJLElBQUksR0FBR25JLElBQUksQ0FBQ1Asa0JBQW5CO0FBQ0QiLCJzb3VyY2VSb290IjoiL2J1aWxkL2F0b20vc3JjL2F0b20tMS4zOC4yL291dC9hcHAvbm9kZV9tb2R1bGVzL2dpdGh1YiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCBQYXRjaEJ1ZmZlciBmcm9tICcuL3BhdGNoLWJ1ZmZlcic7XG5pbXBvcnQgSHVuayBmcm9tICcuL2h1bmsnO1xuaW1wb3J0IEZpbGUsIHtudWxsRmlsZX0gZnJvbSAnLi9maWxlJztcbmltcG9ydCBQYXRjaCwge1RPT19MQVJHRSwgRVhQQU5ERUR9IGZyb20gJy4vcGF0Y2gnO1xuaW1wb3J0IHtVbmNoYW5nZWQsIEFkZGl0aW9uLCBEZWxldGlvbiwgTm9OZXdsaW5lfSBmcm9tICcuL3JlZ2lvbic7XG5pbXBvcnQgRmlsZVBhdGNoIGZyb20gJy4vZmlsZS1wYXRjaCc7XG5pbXBvcnQgTXVsdGlGaWxlUGF0Y2ggZnJvbSAnLi9tdWx0aS1maWxlLXBhdGNoJztcblxuZXhwb3J0IGNvbnN0IERFRkFVTFRfT1BUSU9OUyA9IHtcbiAgLy8gTnVtYmVyIG9mIGxpbmVzIGFmdGVyIHdoaWNoIHdlIGNvbnNpZGVyIHRoZSBkaWZmIFwibGFyZ2VcIlxuICBsYXJnZURpZmZUaHJlc2hvbGQ6IDgwMCxcblxuICAvLyBNYXAgb2YgZmlsZSBwYXRoIChyZWxhdGl2ZSB0byByZXBvc2l0b3J5IHJvb3QpIHRvIFBhdGNoIHJlbmRlciBzdGF0dXMgKEVYUEFOREVELCBDT0xMQVBTRUQsIFRPT19MQVJHRSlcbiAgcmVuZGVyU3RhdHVzT3ZlcnJpZGVzOiB7fSxcblxuICAvLyBFeGlzdGluZyBwYXRjaCBidWZmZXIgdG8gcmVuZGVyIG9udG9cbiAgcGF0Y2hCdWZmZXI6IG51bGwsXG5cbiAgLy8gU3RvcmUgb2ZmIHdoYXQtdGhlLWRpZmYgZmlsZSBwYXRjaFxuICBwcmVzZXJ2ZU9yaWdpbmFsOiBmYWxzZSxcbn07XG5cbmV4cG9ydCBmdW5jdGlvbiBidWlsZEZpbGVQYXRjaChkaWZmcywgb3B0aW9ucykge1xuICBjb25zdCBvcHRzID0gey4uLkRFRkFVTFRfT1BUSU9OUywgLi4ub3B0aW9uc307XG4gIGNvbnN0IHBhdGNoQnVmZmVyID0gbmV3IFBhdGNoQnVmZmVyKCk7XG5cbiAgbGV0IGZpbGVQYXRjaDtcbiAgaWYgKGRpZmZzLmxlbmd0aCA9PT0gMCkge1xuICAgIGZpbGVQYXRjaCA9IGVtcHR5RGlmZkZpbGVQYXRjaCgpO1xuICB9IGVsc2UgaWYgKGRpZmZzLmxlbmd0aCA9PT0gMSkge1xuICAgIGZpbGVQYXRjaCA9IHNpbmdsZURpZmZGaWxlUGF0Y2goZGlmZnNbMF0sIHBhdGNoQnVmZmVyLCBvcHRzKTtcbiAgfSBlbHNlIGlmIChkaWZmcy5sZW5ndGggPT09IDIpIHtcbiAgICBmaWxlUGF0Y2ggPSBkdWFsRGlmZkZpbGVQYXRjaChkaWZmc1swXSwgZGlmZnNbMV0sIHBhdGNoQnVmZmVyLCBvcHRzKTtcbiAgfSBlbHNlIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoYFVuZXhwZWN0ZWQgbnVtYmVyIG9mIGRpZmZzOiAke2RpZmZzLmxlbmd0aH1gKTtcbiAgfVxuXG4gIC8vIERlbGV0ZSB0aGUgdHJhaWxpbmcgbmV3bGluZS5cbiAgcGF0Y2hCdWZmZXIuZGVsZXRlTGFzdE5ld2xpbmUoKTtcblxuICByZXR1cm4gbmV3IE11bHRpRmlsZVBhdGNoKHtwYXRjaEJ1ZmZlciwgZmlsZVBhdGNoZXM6IFtmaWxlUGF0Y2hdfSk7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBidWlsZE11bHRpRmlsZVBhdGNoKGRpZmZzLCBvcHRpb25zKSB7XG4gIGNvbnN0IG9wdHMgPSB7Li4uREVGQVVMVF9PUFRJT05TLCAuLi5vcHRpb25zfTtcblxuICBjb25zdCBwYXRjaEJ1ZmZlciA9IG5ldyBQYXRjaEJ1ZmZlcigpO1xuXG4gIGNvbnN0IGJ5UGF0aCA9IG5ldyBNYXAoKTtcbiAgY29uc3QgYWN0aW9ucyA9IFtdO1xuXG4gIGxldCBpbmRleCA9IDA7XG4gIGZvciAoY29uc3QgZGlmZiBvZiBkaWZmcykge1xuICAgIGNvbnN0IHRoZVBhdGggPSBkaWZmLm9sZFBhdGggfHwgZGlmZi5uZXdQYXRoO1xuXG4gICAgaWYgKGRpZmYuc3RhdHVzID09PSAnYWRkZWQnIHx8IGRpZmYuc3RhdHVzID09PSAnZGVsZXRlZCcpIHtcbiAgICAgIC8vIFBvdGVudGlhbCBwYWlyZWQgZGlmZi4gRWl0aGVyIGEgc3ltbGluayBkZWxldGlvbiArIGNvbnRlbnQgYWRkaXRpb24gb3IgYSBzeW1saW5rIGFkZGl0aW9uICtcbiAgICAgIC8vIGNvbnRlbnQgZGVsZXRpb24uXG4gICAgICBjb25zdCBvdGhlckhhbGYgPSBieVBhdGguZ2V0KHRoZVBhdGgpO1xuICAgICAgaWYgKG90aGVySGFsZikge1xuICAgICAgICAvLyBUaGUgc2Vjb25kIGhhbGYuIENvbXBsZXRlIHRoZSBwYWlyZWQgZGlmZiwgb3IgZmFpbCBpZiB0aGV5IGhhdmUgdW5leHBlY3RlZCBzdGF0dXNlcyBvciBtb2Rlcy5cbiAgICAgICAgY29uc3QgW290aGVyRGlmZiwgb3RoZXJJbmRleF0gPSBvdGhlckhhbGY7XG4gICAgICAgIGFjdGlvbnNbb3RoZXJJbmRleF0gPSAoZnVuY3Rpb24oX2RpZmYsIF9vdGhlckRpZmYpIHtcbiAgICAgICAgICByZXR1cm4gKCkgPT4gZHVhbERpZmZGaWxlUGF0Y2goX2RpZmYsIF9vdGhlckRpZmYsIHBhdGNoQnVmZmVyLCBvcHRzKTtcbiAgICAgICAgfSkoZGlmZiwgb3RoZXJEaWZmKTtcbiAgICAgICAgYnlQYXRoLmRlbGV0ZSh0aGVQYXRoKTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIC8vIFRoZSBmaXJzdCBoYWxmIHdlJ3ZlIHNlZW4uXG4gICAgICAgIGJ5UGF0aC5zZXQodGhlUGF0aCwgW2RpZmYsIGluZGV4XSk7XG4gICAgICAgIGluZGV4Kys7XG4gICAgICB9XG4gICAgfSBlbHNlIHtcbiAgICAgIGFjdGlvbnNbaW5kZXhdID0gKGZ1bmN0aW9uKF9kaWZmKSB7XG4gICAgICAgIHJldHVybiAoKSA9PiBzaW5nbGVEaWZmRmlsZVBhdGNoKF9kaWZmLCBwYXRjaEJ1ZmZlciwgb3B0cyk7XG4gICAgICB9KShkaWZmKTtcbiAgICAgIGluZGV4Kys7XG4gICAgfVxuICB9XG5cbiAgLy8gUG9wdWxhdGUgdW5wYWlyZWQgZGlmZnMgdGhhdCBsb29rZWQgbGlrZSB0aGV5IGNvdWxkIGJlIHBhcnQgb2YgYSBwYWlyLCBidXQgd2VyZW4ndC5cbiAgZm9yIChjb25zdCBbdW5wYWlyZWREaWZmLCBvcmlnaW5hbEluZGV4XSBvZiBieVBhdGgudmFsdWVzKCkpIHtcbiAgICBhY3Rpb25zW29yaWdpbmFsSW5kZXhdID0gKGZ1bmN0aW9uKF91bnBhaXJlZERpZmYpIHtcbiAgICAgIHJldHVybiAoKSA9PiBzaW5nbGVEaWZmRmlsZVBhdGNoKF91bnBhaXJlZERpZmYsIHBhdGNoQnVmZmVyLCBvcHRzKTtcbiAgICB9KSh1bnBhaXJlZERpZmYpO1xuICB9XG5cbiAgY29uc3QgZmlsZVBhdGNoZXMgPSBhY3Rpb25zLm1hcChhY3Rpb24gPT4gYWN0aW9uKCkpO1xuXG4gIC8vIERlbGV0ZSB0aGUgZmluYWwgdHJhaWxpbmcgbmV3bGluZSBmcm9tIHRoZSBsYXN0IG5vbi1lbXB0eSBwYXRjaC5cbiAgcGF0Y2hCdWZmZXIuZGVsZXRlTGFzdE5ld2xpbmUoKTtcblxuICByZXR1cm4gbmV3IE11bHRpRmlsZVBhdGNoKHtwYXRjaEJ1ZmZlciwgZmlsZVBhdGNoZXN9KTtcbn1cblxuZnVuY3Rpb24gZW1wdHlEaWZmRmlsZVBhdGNoKCkge1xuICByZXR1cm4gRmlsZVBhdGNoLmNyZWF0ZU51bGwoKTtcbn1cblxuZnVuY3Rpb24gc2luZ2xlRGlmZkZpbGVQYXRjaChkaWZmLCBwYXRjaEJ1ZmZlciwgb3B0cykge1xuICBjb25zdCB3YXNTeW1saW5rID0gZGlmZi5vbGRNb2RlID09PSBGaWxlLm1vZGVzLlNZTUxJTks7XG4gIGNvbnN0IGlzU3ltbGluayA9IGRpZmYubmV3TW9kZSA9PT0gRmlsZS5tb2Rlcy5TWU1MSU5LO1xuXG4gIGxldCBvbGRTeW1saW5rID0gbnVsbDtcbiAgbGV0IG5ld1N5bWxpbmsgPSBudWxsO1xuICBpZiAod2FzU3ltbGluayAmJiAhaXNTeW1saW5rKSB7XG4gICAgb2xkU3ltbGluayA9IGRpZmYuaHVua3NbMF0ubGluZXNbMF0uc2xpY2UoMSk7XG4gIH0gZWxzZSBpZiAoIXdhc1N5bWxpbmsgJiYgaXNTeW1saW5rKSB7XG4gICAgbmV3U3ltbGluayA9IGRpZmYuaHVua3NbMF0ubGluZXNbMF0uc2xpY2UoMSk7XG4gIH0gZWxzZSBpZiAod2FzU3ltbGluayAmJiBpc1N5bWxpbmspIHtcbiAgICBvbGRTeW1saW5rID0gZGlmZi5odW5rc1swXS5saW5lc1swXS5zbGljZSgxKTtcbiAgICBuZXdTeW1saW5rID0gZGlmZi5odW5rc1swXS5saW5lc1syXS5zbGljZSgxKTtcbiAgfVxuXG4gIGNvbnN0IG9sZEZpbGUgPSBkaWZmLm9sZFBhdGggIT09IG51bGwgfHwgZGlmZi5vbGRNb2RlICE9PSBudWxsXG4gICAgPyBuZXcgRmlsZSh7cGF0aDogZGlmZi5vbGRQYXRoLCBtb2RlOiBkaWZmLm9sZE1vZGUsIHN5bWxpbms6IG9sZFN5bWxpbmt9KVxuICAgIDogbnVsbEZpbGU7XG4gIGNvbnN0IG5ld0ZpbGUgPSBkaWZmLm5ld1BhdGggIT09IG51bGwgfHwgZGlmZi5uZXdNb2RlICE9PSBudWxsXG4gICAgPyBuZXcgRmlsZSh7cGF0aDogZGlmZi5uZXdQYXRoLCBtb2RlOiBkaWZmLm5ld01vZGUsIHN5bWxpbms6IG5ld1N5bWxpbmt9KVxuICAgIDogbnVsbEZpbGU7XG5cbiAgY29uc3QgcmVuZGVyU3RhdHVzT3ZlcnJpZGUgPVxuICAgIChvbGRGaWxlLmlzUHJlc2VudCgpICYmIG9wdHMucmVuZGVyU3RhdHVzT3ZlcnJpZGVzW29sZEZpbGUuZ2V0UGF0aCgpXSkgfHxcbiAgICAobmV3RmlsZS5pc1ByZXNlbnQoKSAmJiBvcHRzLnJlbmRlclN0YXR1c092ZXJyaWRlc1tuZXdGaWxlLmdldFBhdGgoKV0pIHx8XG4gICAgdW5kZWZpbmVkO1xuXG4gIGNvbnN0IHJlbmRlclN0YXR1cyA9IHJlbmRlclN0YXR1c092ZXJyaWRlIHx8XG4gICAgKGlzRGlmZkxhcmdlKFtkaWZmXSwgb3B0cykgJiYgVE9PX0xBUkdFKSB8fFxuICAgIEVYUEFOREVEO1xuXG4gIGlmICghcmVuZGVyU3RhdHVzLmlzVmlzaWJsZSgpKSB7XG4gICAgY29uc3QgcGF0Y2hNYXJrZXIgPSBwYXRjaEJ1ZmZlci5tYXJrUG9zaXRpb24oXG4gICAgICBQYXRjaC5sYXllck5hbWUsXG4gICAgICBwYXRjaEJ1ZmZlci5nZXRCdWZmZXIoKS5nZXRFbmRQb3NpdGlvbigpLFxuICAgICAge2ludmFsaWRhdGU6ICduZXZlcicsIGV4Y2x1c2l2ZTogZmFsc2V9LFxuICAgICk7XG5cbiAgICByZXR1cm4gRmlsZVBhdGNoLmNyZWF0ZUhpZGRlbkZpbGVQYXRjaChcbiAgICAgIG9sZEZpbGUsIG5ld0ZpbGUsIHBhdGNoTWFya2VyLCByZW5kZXJTdGF0dXMsXG4gICAgICAoKSA9PiB7XG4gICAgICAgIGNvbnN0IHN1YlBhdGNoQnVmZmVyID0gbmV3IFBhdGNoQnVmZmVyKCk7XG4gICAgICAgIGNvbnN0IFtodW5rcywgbmV4dFBhdGNoTWFya2VyXSA9IGJ1aWxkSHVua3MoZGlmZiwgc3ViUGF0Y2hCdWZmZXIpO1xuICAgICAgICBjb25zdCBuZXh0UGF0Y2ggPSBuZXcgUGF0Y2goe3N0YXR1czogZGlmZi5zdGF0dXMsIGh1bmtzLCBtYXJrZXI6IG5leHRQYXRjaE1hcmtlcn0pO1xuXG4gICAgICAgIHN1YlBhdGNoQnVmZmVyLmRlbGV0ZUxhc3ROZXdsaW5lKCk7XG4gICAgICAgIHJldHVybiB7cGF0Y2g6IG5leHRQYXRjaCwgcGF0Y2hCdWZmZXI6IHN1YlBhdGNoQnVmZmVyfTtcbiAgICAgIH0sXG4gICAgKTtcbiAgfSBlbHNlIHtcbiAgICBjb25zdCBbaHVua3MsIHBhdGNoTWFya2VyXSA9IGJ1aWxkSHVua3MoZGlmZiwgcGF0Y2hCdWZmZXIpO1xuICAgIGNvbnN0IHBhdGNoID0gbmV3IFBhdGNoKHtzdGF0dXM6IGRpZmYuc3RhdHVzLCBodW5rcywgbWFya2VyOiBwYXRjaE1hcmtlcn0pO1xuXG4gICAgY29uc3QgcmF3UGF0Y2hlcyA9IG9wdHMucHJlc2VydmVPcmlnaW5hbCA/IHtjb250ZW50OiBkaWZmfSA6IG51bGw7XG4gICAgcmV0dXJuIG5ldyBGaWxlUGF0Y2gob2xkRmlsZSwgbmV3RmlsZSwgcGF0Y2gsIHJhd1BhdGNoZXMpO1xuICB9XG59XG5cbmZ1bmN0aW9uIGR1YWxEaWZmRmlsZVBhdGNoKGRpZmYxLCBkaWZmMiwgcGF0Y2hCdWZmZXIsIG9wdHMpIHtcbiAgbGV0IG1vZGVDaGFuZ2VEaWZmLCBjb250ZW50Q2hhbmdlRGlmZjtcbiAgaWYgKGRpZmYxLm9sZE1vZGUgPT09IEZpbGUubW9kZXMuU1lNTElOSyB8fCBkaWZmMS5uZXdNb2RlID09PSBGaWxlLm1vZGVzLlNZTUxJTkspIHtcbiAgICBtb2RlQ2hhbmdlRGlmZiA9IGRpZmYxO1xuICAgIGNvbnRlbnRDaGFuZ2VEaWZmID0gZGlmZjI7XG4gIH0gZWxzZSB7XG4gICAgbW9kZUNoYW5nZURpZmYgPSBkaWZmMjtcbiAgICBjb250ZW50Q2hhbmdlRGlmZiA9IGRpZmYxO1xuICB9XG5cbiAgY29uc3QgZmlsZVBhdGggPSBjb250ZW50Q2hhbmdlRGlmZi5vbGRQYXRoIHx8IGNvbnRlbnRDaGFuZ2VEaWZmLm5ld1BhdGg7XG4gIGNvbnN0IHN5bWxpbmsgPSBtb2RlQ2hhbmdlRGlmZi5odW5rc1swXS5saW5lc1swXS5zbGljZSgxKTtcblxuICBsZXQgc3RhdHVzO1xuICBsZXQgb2xkTW9kZSwgbmV3TW9kZTtcbiAgbGV0IG9sZFN5bWxpbmsgPSBudWxsO1xuICBsZXQgbmV3U3ltbGluayA9IG51bGw7XG4gIGlmIChtb2RlQ2hhbmdlRGlmZi5zdGF0dXMgPT09ICdhZGRlZCcpIHtcbiAgICAvLyBjb250ZW50cyB3ZXJlIGRlbGV0ZWQgYW5kIHJlcGxhY2VkIHdpdGggc3ltbGlua1xuICAgIHN0YXR1cyA9ICdkZWxldGVkJztcbiAgICBvbGRNb2RlID0gY29udGVudENoYW5nZURpZmYub2xkTW9kZTtcbiAgICBuZXdNb2RlID0gbW9kZUNoYW5nZURpZmYubmV3TW9kZTtcbiAgICBuZXdTeW1saW5rID0gc3ltbGluaztcbiAgfSBlbHNlIGlmIChtb2RlQ2hhbmdlRGlmZi5zdGF0dXMgPT09ICdkZWxldGVkJykge1xuICAgIC8vIGNvbnRlbnRzIHdlcmUgYWRkZWQgYWZ0ZXIgc3ltbGluayB3YXMgZGVsZXRlZFxuICAgIHN0YXR1cyA9ICdhZGRlZCc7XG4gICAgb2xkTW9kZSA9IG1vZGVDaGFuZ2VEaWZmLm9sZE1vZGU7XG4gICAgb2xkU3ltbGluayA9IHN5bWxpbms7XG4gICAgbmV3TW9kZSA9IGNvbnRlbnRDaGFuZ2VEaWZmLm5ld01vZGU7XG4gIH0gZWxzZSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKGBJbnZhbGlkIG1vZGUgY2hhbmdlIGRpZmYgc3RhdHVzOiAke21vZGVDaGFuZ2VEaWZmLnN0YXR1c31gKTtcbiAgfVxuXG4gIGNvbnN0IG9sZEZpbGUgPSBuZXcgRmlsZSh7cGF0aDogZmlsZVBhdGgsIG1vZGU6IG9sZE1vZGUsIHN5bWxpbms6IG9sZFN5bWxpbmt9KTtcbiAgY29uc3QgbmV3RmlsZSA9IG5ldyBGaWxlKHtwYXRoOiBmaWxlUGF0aCwgbW9kZTogbmV3TW9kZSwgc3ltbGluazogbmV3U3ltbGlua30pO1xuXG4gIGNvbnN0IHJlbmRlclN0YXR1cyA9IG9wdHMucmVuZGVyU3RhdHVzT3ZlcnJpZGVzW2ZpbGVQYXRoXSB8fFxuICAgIChpc0RpZmZMYXJnZShbY29udGVudENoYW5nZURpZmZdLCBvcHRzKSAmJiBUT09fTEFSR0UpIHx8XG4gICAgRVhQQU5ERUQ7XG5cbiAgaWYgKCFyZW5kZXJTdGF0dXMuaXNWaXNpYmxlKCkpIHtcbiAgICBjb25zdCBwYXRjaE1hcmtlciA9IHBhdGNoQnVmZmVyLm1hcmtQb3NpdGlvbihcbiAgICAgIFBhdGNoLmxheWVyTmFtZSxcbiAgICAgIHBhdGNoQnVmZmVyLmdldEJ1ZmZlcigpLmdldEVuZFBvc2l0aW9uKCksXG4gICAgICB7aW52YWxpZGF0ZTogJ25ldmVyJywgZXhjbHVzaXZlOiBmYWxzZX0sXG4gICAgKTtcblxuICAgIHJldHVybiBGaWxlUGF0Y2guY3JlYXRlSGlkZGVuRmlsZVBhdGNoKFxuICAgICAgb2xkRmlsZSwgbmV3RmlsZSwgcGF0Y2hNYXJrZXIsIHJlbmRlclN0YXR1cyxcbiAgICAgICgpID0+IHtcbiAgICAgICAgY29uc3Qgc3ViUGF0Y2hCdWZmZXIgPSBuZXcgUGF0Y2hCdWZmZXIoKTtcbiAgICAgICAgY29uc3QgW2h1bmtzLCBuZXh0UGF0Y2hNYXJrZXJdID0gYnVpbGRIdW5rcyhjb250ZW50Q2hhbmdlRGlmZiwgc3ViUGF0Y2hCdWZmZXIpO1xuICAgICAgICBjb25zdCBuZXh0UGF0Y2ggPSBuZXcgUGF0Y2goe3N0YXR1cywgaHVua3MsIG1hcmtlcjogbmV4dFBhdGNoTWFya2VyfSk7XG5cbiAgICAgICAgc3ViUGF0Y2hCdWZmZXIuZGVsZXRlTGFzdE5ld2xpbmUoKTtcbiAgICAgICAgcmV0dXJuIHtwYXRjaDogbmV4dFBhdGNoLCBwYXRjaEJ1ZmZlcjogc3ViUGF0Y2hCdWZmZXJ9O1xuICAgICAgfSxcbiAgICApO1xuICB9IGVsc2Uge1xuICAgIGNvbnN0IFtodW5rcywgcGF0Y2hNYXJrZXJdID0gYnVpbGRIdW5rcyhjb250ZW50Q2hhbmdlRGlmZiwgcGF0Y2hCdWZmZXIpO1xuICAgIGNvbnN0IHBhdGNoID0gbmV3IFBhdGNoKHtzdGF0dXMsIGh1bmtzLCBtYXJrZXI6IHBhdGNoTWFya2VyfSk7XG5cbiAgICBjb25zdCByYXdQYXRjaGVzID0gb3B0cy5wcmVzZXJ2ZU9yaWdpbmFsID8ge2NvbnRlbnQ6IGNvbnRlbnRDaGFuZ2VEaWZmLCBtb2RlOiBtb2RlQ2hhbmdlRGlmZn0gOiBudWxsO1xuICAgIHJldHVybiBuZXcgRmlsZVBhdGNoKG9sZEZpbGUsIG5ld0ZpbGUsIHBhdGNoLCByYXdQYXRjaGVzKTtcbiAgfVxufVxuXG5jb25zdCBDSEFOR0VLSU5EID0ge1xuICAnKyc6IEFkZGl0aW9uLFxuICAnLSc6IERlbGV0aW9uLFxuICAnICc6IFVuY2hhbmdlZCxcbiAgJ1xcXFwnOiBOb05ld2xpbmUsXG59O1xuXG5mdW5jdGlvbiBidWlsZEh1bmtzKGRpZmYsIHBhdGNoQnVmZmVyKSB7XG4gIGNvbnN0IGluc2VydGVyID0gcGF0Y2hCdWZmZXIuY3JlYXRlSW5zZXJ0ZXJBdEVuZCgpXG4gICAgLmtlZXBCZWZvcmUocGF0Y2hCdWZmZXIuZmluZEFsbE1hcmtlcnMoe2VuZFBvc2l0aW9uOiBwYXRjaEJ1ZmZlci5nZXRJbnNlcnRpb25Qb2ludCgpfSkpO1xuXG4gIGxldCBwYXRjaE1hcmtlciA9IG51bGw7XG4gIGxldCBmaXJzdEh1bmsgPSB0cnVlO1xuICBjb25zdCBodW5rcyA9IFtdO1xuXG4gIGluc2VydGVyLm1hcmtXaGlsZShQYXRjaC5sYXllck5hbWUsICgpID0+IHtcbiAgICBmb3IgKGNvbnN0IHJhd0h1bmsgb2YgZGlmZi5odW5rcykge1xuICAgICAgbGV0IGZpcnN0UmVnaW9uID0gdHJ1ZTtcbiAgICAgIGNvbnN0IHJlZ2lvbnMgPSBbXTtcblxuICAgICAgLy8gU2VwYXJhdGUgaHVua3Mgd2l0aCBhbiB1bm1hcmtlZCBuZXdsaW5lXG4gICAgICBpZiAoZmlyc3RIdW5rKSB7XG4gICAgICAgIGZpcnN0SHVuayA9IGZhbHNlO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgaW5zZXJ0ZXIuaW5zZXJ0KCdcXG4nKTtcbiAgICAgIH1cblxuICAgICAgaW5zZXJ0ZXIubWFya1doaWxlKEh1bmsubGF5ZXJOYW1lLCAoKSA9PiB7XG4gICAgICAgIGxldCBmaXJzdFJlZ2lvbkxpbmUgPSB0cnVlO1xuICAgICAgICBsZXQgY3VycmVudFJlZ2lvblRleHQgPSAnJztcbiAgICAgICAgbGV0IEN1cnJlbnRSZWdpb25LaW5kID0gbnVsbDtcblxuICAgICAgICBmdW5jdGlvbiBmaW5pc2hSZWdpb24oKSB7XG4gICAgICAgICAgaWYgKEN1cnJlbnRSZWdpb25LaW5kID09PSBudWxsKSB7XG4gICAgICAgICAgICByZXR1cm47XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgLy8gU2VwYXJhdGUgcmVnaW9ucyB3aXRoIGFuIHVubWFya2VkIG5ld2xpbmVcbiAgICAgICAgICBpZiAoZmlyc3RSZWdpb24pIHtcbiAgICAgICAgICAgIGZpcnN0UmVnaW9uID0gZmFsc2U7XG4gICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIGluc2VydGVyLmluc2VydCgnXFxuJyk7XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgaW5zZXJ0ZXIuaW5zZXJ0TWFya2VkKGN1cnJlbnRSZWdpb25UZXh0LCBDdXJyZW50UmVnaW9uS2luZC5sYXllck5hbWUsIHtcbiAgICAgICAgICAgIGludmFsaWRhdGU6ICduZXZlcicsXG4gICAgICAgICAgICBleGNsdXNpdmU6IGZhbHNlLFxuICAgICAgICAgICAgY2FsbGJhY2s6IChmdW5jdGlvbihfcmVnaW9ucywgX0N1cnJlbnRSZWdpb25LaW5kKSB7XG4gICAgICAgICAgICAgIHJldHVybiByZWdpb25NYXJrZXIgPT4geyBfcmVnaW9ucy5wdXNoKG5ldyBfQ3VycmVudFJlZ2lvbktpbmQocmVnaW9uTWFya2VyKSk7IH07XG4gICAgICAgICAgICB9KShyZWdpb25zLCBDdXJyZW50UmVnaW9uS2luZCksXG4gICAgICAgICAgfSk7XG4gICAgICAgIH1cblxuICAgICAgICBmb3IgKGNvbnN0IHJhd0xpbmUgb2YgcmF3SHVuay5saW5lcykge1xuICAgICAgICAgIGNvbnN0IE5leHRSZWdpb25LaW5kID0gQ0hBTkdFS0lORFtyYXdMaW5lWzBdXTtcbiAgICAgICAgICBpZiAoTmV4dFJlZ2lvbktpbmQgPT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbmtub3duIGRpZmYgc3RhdHVzIGNoYXJhY3RlcjogXCIke3Jhd0xpbmVbMF19XCJgKTtcbiAgICAgICAgICB9XG4gICAgICAgICAgY29uc3QgbmV4dExpbmUgPSByYXdMaW5lLnNsaWNlKDEpO1xuXG4gICAgICAgICAgbGV0IHNlcGFyYXRvciA9ICcnO1xuICAgICAgICAgIGlmIChmaXJzdFJlZ2lvbkxpbmUpIHtcbiAgICAgICAgICAgIGZpcnN0UmVnaW9uTGluZSA9IGZhbHNlO1xuICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICBzZXBhcmF0b3IgPSAnXFxuJztcbiAgICAgICAgICB9XG5cbiAgICAgICAgICBpZiAoTmV4dFJlZ2lvbktpbmQgPT09IEN1cnJlbnRSZWdpb25LaW5kKSB7XG4gICAgICAgICAgICBjdXJyZW50UmVnaW9uVGV4dCArPSBzZXBhcmF0b3IgKyBuZXh0TGluZTtcblxuICAgICAgICAgICAgY29udGludWU7XG4gICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIGZpbmlzaFJlZ2lvbigpO1xuXG4gICAgICAgICAgICBDdXJyZW50UmVnaW9uS2luZCA9IE5leHRSZWdpb25LaW5kO1xuICAgICAgICAgICAgY3VycmVudFJlZ2lvblRleHQgPSBuZXh0TGluZTtcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgICAgZmluaXNoUmVnaW9uKCk7XG4gICAgICB9LCB7XG4gICAgICAgIGludmFsaWRhdGU6ICduZXZlcicsXG4gICAgICAgIGV4Y2x1c2l2ZTogZmFsc2UsXG4gICAgICAgIGNhbGxiYWNrOiAoZnVuY3Rpb24oX2h1bmtzLCBfcmF3SHVuaywgX3JlZ2lvbnMpIHtcbiAgICAgICAgICByZXR1cm4gaHVua01hcmtlciA9PiB7XG4gICAgICAgICAgICBfaHVua3MucHVzaChuZXcgSHVuayh7XG4gICAgICAgICAgICAgIG9sZFN0YXJ0Um93OiBfcmF3SHVuay5vbGRTdGFydExpbmUsXG4gICAgICAgICAgICAgIG5ld1N0YXJ0Um93OiBfcmF3SHVuay5uZXdTdGFydExpbmUsXG4gICAgICAgICAgICAgIG9sZFJvd0NvdW50OiBfcmF3SHVuay5vbGRMaW5lQ291bnQsXG4gICAgICAgICAgICAgIG5ld1Jvd0NvdW50OiBfcmF3SHVuay5uZXdMaW5lQ291bnQsXG4gICAgICAgICAgICAgIHNlY3Rpb25IZWFkaW5nOiBfcmF3SHVuay5oZWFkaW5nLFxuICAgICAgICAgICAgICBtYXJrZXI6IGh1bmtNYXJrZXIsXG4gICAgICAgICAgICAgIHJlZ2lvbnM6IF9yZWdpb25zLFxuICAgICAgICAgICAgfSkpO1xuICAgICAgICAgIH07XG4gICAgICAgIH0pKGh1bmtzLCByYXdIdW5rLCByZWdpb25zKSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfSwge1xuICAgIGludmFsaWRhdGU6ICduZXZlcicsXG4gICAgZXhjbHVzaXZlOiBmYWxzZSxcbiAgICBjYWxsYmFjazogbWFya2VyID0+IHsgcGF0Y2hNYXJrZXIgPSBtYXJrZXI7IH0sXG4gIH0pO1xuXG4gIC8vIFNlcGFyYXRlIG11bHRpcGxlIG5vbi1lbXB0eSBwYXRjaGVzIG9uIHRoZSBzYW1lIGJ1ZmZlciB3aXRoIGFuIHVubWFya2VkIG5ld2xpbmUuIFRoZSBuZXdsaW5lIGFmdGVyIHRoZSBmaW5hbFxuICAvLyBub24tZW1wdHkgcGF0Y2ggKGlmIHRoZXJlIGlzIG9uZSkgc2hvdWxkIGJlIGRlbGV0ZWQgYmVmb3JlIE11bHRpRmlsZVBhdGNoIGNvbnN0cnVjdGlvbi5cbiAgaWYgKGRpZmYuaHVua3MubGVuZ3RoID4gMCkge1xuICAgIGluc2VydGVyLmluc2VydCgnXFxuJyk7XG4gIH1cblxuICBpbnNlcnRlci5hcHBseSgpO1xuXG4gIHJldHVybiBbaHVua3MsIHBhdGNoTWFya2VyXTtcbn1cblxuZnVuY3Rpb24gaXNEaWZmTGFyZ2UoZGlmZnMsIG9wdHMpIHtcbiAgY29uc3Qgc2l6ZSA9IGRpZmZzLnJlZHVjZSgoZGlmZlNpemVDb3VudGVyLCBkaWZmKSA9PiB7XG4gICAgcmV0dXJuIGRpZmZTaXplQ291bnRlciArIGRpZmYuaHVua3MucmVkdWNlKChodW5rU2l6ZUNvdW50ZXIsIGh1bmspID0+IHtcbiAgICAgIHJldHVybiBodW5rU2l6ZUNvdW50ZXIgKyBodW5rLmxpbmVzLmxlbmd0aDtcbiAgICB9LCAwKTtcbiAgfSwgMCk7XG5cbiAgcmV0dXJuIHNpemUgPiBvcHRzLmxhcmdlRGlmZlRocmVzaG9sZDtcbn1cbiJdfQ==