/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 * 
 * @format
 */
'use strict';

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));

var _toConsumableArray2 = _interopRequireDefault(require("@babel/runtime/helpers/toConsumableArray"));

var invariant = require("fbjs/lib/invariant");

var queryID = 1;

/**
 * A network logger transaction is an object that you can construct, pass around
 * and add logs to, ultimately calling its commit method when you're done.
 * Different transactions can have different commit logic. One might use
 * `console.log`. Another might ping a logging endpoint. Another might add some
 * autogenerated logs before doing either of the foregoing.
 */
var RelayNetworkLoggerTransaction =
/*#__PURE__*/
function () {
  function RelayNetworkLoggerTransaction(_ref) {
    var request = _ref.request,
        variables = _ref.variables,
        cacheConfig = _ref.cacheConfig,
        uploadables = _ref.uploadables;
    (0, _defineProperty2["default"])(this, "_hasCommittedLogs", false);
    (0, _defineProperty2["default"])(this, "_logs", []);
    this._cacheConfig = cacheConfig;
    this._id = queryID++;
    this._request = request;
    this._uploadables = uploadables;
    this._variables = variables;
  }

  var _proto = RelayNetworkLoggerTransaction.prototype;

  _proto.addLog = function addLog(label) {
    for (var _len = arguments.length, values = new Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
      values[_key - 1] = arguments[_key];
    }

    this._logs.push({
      label: label,
      values: values
    });
  };

  _proto.clearLogs = function clearLogs() {
    this._logs = [];
  };

  _proto.printLogs = function printLogs(error, payload, status) {
    /* eslint-disable no-console */
    var transactionId = this.getIdentifier();
    console.groupCollapsed && console.groupCollapsed("%c".concat(transactionId), error ? 'color:red' : '');
    console.timeEnd && console.timeEnd(transactionId);
    this.getLogsToPrint(error, payload, status).forEach(function (_ref2) {
      var _console;

      var label = _ref2.label,
          values = _ref2.values;

      (_console = console).log.apply(_console, ["".concat(label, ":")].concat((0, _toConsumableArray2["default"])(values)));
    });
    console.groupEnd && console.groupEnd();
    /* eslint-enable no-console */
  };

  _proto.commitLogs = function commitLogs(error, payload, status) {
    !(this._hasCommittedLogs === false) ? process.env.NODE_ENV !== "production" ? invariant(false, "The logs for transaction #".concat(this._id, " have already been committed.")) : invariant(false) : void 0;
    this.printLogs(error, payload, status);
    this.markCommitted();
  };

  _proto.markCommitted = function markCommitted() {
    this._hasCommittedLogs = true;
  };

  _proto.flushLogs = function flushLogs(error, payload, status) {
    !(this._hasCommittedLogs === false) ? process.env.NODE_ENV !== "production" ? invariant(false, "The logs for transaction #".concat(this._id, " have already been committed.")) : invariant(false) : void 0;
    this.printLogs(error, payload, status);
    this.clearLogs();
  };

  _proto.getCacheConfig = function getCacheConfig() {
    return this._cacheConfig;
  };

  _proto.getIdentifier = function getIdentifier() {
    return "[".concat(this._id, "] Relay Modern: ").concat(this._request.name);
  };

  _proto.getLogsToPrint = function getLogsToPrint(error, payload, status) {
    return this._logs;
  };

  _proto.getRequest = function getRequest() {
    return this._request;
  };

  _proto.getUploadables = function getUploadables() {
    return this._uploadables;
  };

  _proto.getVariables = function getVariables() {
    return this._variables;
  };

  return RelayNetworkLoggerTransaction;
}();

module.exports = RelayNetworkLoggerTransaction;