(function() {
  var CharacterPattern, _;

  _ = require('underscore-plus');

  CharacterPattern = /[^\s]/;

  module.exports = {
    activate: function() {
      return this.commandDisposable = atom.commands.add('atom-text-editor', {
        'autoflow:reflow-selection': (function(_this) {
          return function(event) {
            return _this.reflowSelection(event.currentTarget.getModel());
          };
        })(this)
      });
    },
    deactivate: function() {
      var ref;
      if ((ref = this.commandDisposable) != null) {
        ref.dispose();
      }
      return this.commandDisposable = null;
    },
    reflowSelection: function(editor) {
      var range, reflowOptions, reflowedText;
      range = editor.getSelectedBufferRange();
      if (range.isEmpty()) {
        range = editor.getCurrentParagraphBufferRange();
      }
      if (range == null) {
        return;
      }
      reflowOptions = {
        wrapColumn: this.getPreferredLineLength(editor),
        tabLength: this.getTabLength(editor)
      };
      reflowedText = this.reflow(editor.getTextInRange(range), reflowOptions);
      return editor.getBuffer().setTextInRange(range, reflowedText);
    },
    reflow: function(text, arg) {
      var beginningLinesToIgnore, block, blockLines, currentLine, currentLineLength, endingLinesToIgnore, escapedLinePrefix, firstLine, i, j, latexTagEndRegex, latexTagRegex, latexTagStartRegex, leadingVerticalSpace, len, len1, linePrefix, linePrefixTabExpanded, lines, paragraphBlocks, paragraphs, ref, segment, tabLength, tabLengthInSpaces, trailingVerticalSpace, wrapColumn, wrappedLinePrefix, wrappedLines;
      wrapColumn = arg.wrapColumn, tabLength = arg.tabLength;
      paragraphs = [];
      text = text.replace(/\r\n?/g, '\n');
      leadingVerticalSpace = text.match(/^\s*\n/);
      if (leadingVerticalSpace) {
        text = text.substr(leadingVerticalSpace.length);
      } else {
        leadingVerticalSpace = '';
      }
      trailingVerticalSpace = text.match(/\n\s*$/);
      if (trailingVerticalSpace) {
        text = text.substr(0, text.length - trailingVerticalSpace.length);
      } else {
        trailingVerticalSpace = '';
      }
      paragraphBlocks = text.split(/\n\s*\n/g);
      if (tabLength) {
        tabLengthInSpaces = Array(tabLength + 1).join(' ');
      } else {
        tabLengthInSpaces = '';
      }
      for (i = 0, len = paragraphBlocks.length; i < len; i++) {
        block = paragraphBlocks[i];
        blockLines = block.split('\n');
        beginningLinesToIgnore = [];
        endingLinesToIgnore = [];
        latexTagRegex = /^\s*\\\w+(\[.*\])?\{\w+\}(\[.*\])?\s*$/g;
        latexTagStartRegex = /^\s*\\\w+\s*\{\s*$/g;
        latexTagEndRegex = /^\s*\}\s*$/g;
        while (blockLines[0].match(latexTagRegex) || blockLines[0].match(latexTagStartRegex)) {
          beginningLinesToIgnore.push(blockLines[0]);
          blockLines.shift();
        }
        while (blockLines[blockLines.length - 1].match(latexTagRegex) || blockLines[blockLines.length - 1].match(latexTagEndRegex)) {
          endingLinesToIgnore.unshift(blockLines[blockLines.length - 1]);
          blockLines.pop();
        }
        linePrefix = blockLines[0].match(/^\s*(\/\/|\/\*|;;|#'|\|\|\||--|[#%*>-])?\s*/g)[0];
        linePrefixTabExpanded = linePrefix;
        if (tabLengthInSpaces) {
          linePrefixTabExpanded = linePrefix.replace(/\t/g, tabLengthInSpaces);
        }
        if (linePrefix) {
          escapedLinePrefix = _.escapeRegExp(linePrefix);
          blockLines = blockLines.map(function(blockLine) {
            return blockLine.replace(RegExp("^" + escapedLinePrefix), '');
          });
        }
        blockLines = blockLines.map(function(blockLine) {
          return blockLine.replace(/^\s+/, '');
        });
        lines = [];
        currentLine = [];
        currentLineLength = linePrefixTabExpanded.length;
        wrappedLinePrefix = linePrefix.replace(/^(\s*)\/\*/, '$1  ').replace(/^(\s*)-(?!-)/, '$1 ');
        firstLine = true;
        ref = this.segmentText(blockLines.join(' '));
        for (j = 0, len1 = ref.length; j < len1; j++) {
          segment = ref[j];
          if (this.wrapSegment(segment, currentLineLength, wrapColumn)) {
            if (firstLine !== true) {
              if (linePrefix.search(/^\s*\/\*/) !== -1 || linePrefix.search(/^\s*-(?!-)/) !== -1) {
                linePrefix = wrappedLinePrefix;
              }
            }
            lines.push(linePrefix + currentLine.join(''));
            currentLine = [];
            currentLineLength = linePrefixTabExpanded.length;
            firstLine = false;
          }
          currentLine.push(segment);
          currentLineLength += segment.length;
        }
        lines.push(linePrefix + currentLine.join(''));
        wrappedLines = beginningLinesToIgnore.concat(lines.concat(endingLinesToIgnore));
        paragraphs.push(wrappedLines.join('\n').replace(/\s+\n/g, '\n'));
      }
      return leadingVerticalSpace + paragraphs.join('\n\n') + trailingVerticalSpace;
    },
    getTabLength: function(editor) {
      var ref;
      return (ref = atom.config.get('editor.tabLength', {
        scope: editor.getRootScopeDescriptor()
      })) != null ? ref : 2;
    },
    getPreferredLineLength: function(editor) {
      return atom.config.get('editor.preferredLineLength', {
        scope: editor.getRootScopeDescriptor()
      });
    },
    wrapSegment: function(segment, currentLineLength, wrapColumn) {
      return CharacterPattern.test(segment) && (currentLineLength + segment.length > wrapColumn) && (currentLineLength > 0 || segment.length < wrapColumn);
    },
    segmentText: function(text) {
      var match, re, segments;
      segments = [];
      re = /[\s]+|[^\s]+/g;
      while (match = re.exec(text)) {
        segments.push(match[0]);
      }
      return segments;
    }
  };

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
