"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = exports.TOO_LARGE = exports.COLLAPSED = exports.EXPANDED = void 0;

var _atom = require("atom");

var _hunk = _interopRequireDefault(require("./hunk"));

var _region = require("./region");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const EXPANDED = {
  /* istanbul ignore next */
  toString() {
    return 'RenderStatus(expanded)';
  },

  isVisible() {
    return true;
  }

};
exports.EXPANDED = EXPANDED;
const COLLAPSED = {
  /* istanbul ignore next */
  toString() {
    return 'RenderStatus(collapsed)';
  },

  isVisible() {
    return false;
  }

};
exports.COLLAPSED = COLLAPSED;
const TOO_LARGE = {
  /* istanbul ignore next */
  toString() {
    return 'RenderStatus(too-large)';
  },

  isVisible() {
    return false;
  }

};
exports.TOO_LARGE = TOO_LARGE;

class Patch {
  static createNull() {
    return new NullPatch();
  }

  static createHiddenPatch(marker, renderStatus, showFn) {
    return new HiddenPatch(marker, renderStatus, showFn);
  }

  constructor({
    status,
    hunks,
    marker
  }) {
    this.status = status;
    this.hunks = hunks;
    this.marker = marker;
    this.changedLineCount = this.getHunks().reduce((acc, hunk) => acc + hunk.changedLineCount(), 0);
  }

  getStatus() {
    return this.status;
  }

  getMarker() {
    return this.marker;
  }

  getRange() {
    return this.getMarker().getRange();
  }

  getStartRange() {
    const startPoint = this.getMarker().getRange().start;
    return _atom.Range.fromObject([startPoint, startPoint]);
  }

  getHunks() {
    return this.hunks;
  }

  getChangedLineCount() {
    return this.changedLineCount;
  }

  containsRow(row) {
    return this.marker.getRange().intersectsRow(row);
  }

  destroyMarkers() {
    this.marker.destroy();

    for (const hunk of this.hunks) {
      hunk.destroyMarkers();
    }
  }

  updateMarkers(map) {
    this.marker = map.get(this.marker) || this.marker;

    for (const hunk of this.hunks) {
      hunk.updateMarkers(map);
    }
  }

  getMaxLineNumberWidth() {
    const lastHunk = this.hunks[this.hunks.length - 1];
    return lastHunk ? lastHunk.getMaxLineNumberWidth() : 0;
  }

  clone(opts = {}) {
    return new this.constructor({
      status: opts.status !== undefined ? opts.status : this.getStatus(),
      hunks: opts.hunks !== undefined ? opts.hunks : this.getHunks(),
      marker: opts.marker !== undefined ? opts.marker : this.getMarker()
    });
  }
  /* Return the set of Markers owned by this Patch that butt up against the patch's beginning. */


  getStartingMarkers() {
    const markers = [this.marker];

    if (this.hunks.length > 0) {
      const firstHunk = this.hunks[0];
      markers.push(firstHunk.getMarker());

      if (firstHunk.getRegions().length > 0) {
        const firstRegion = firstHunk.getRegions()[0];
        markers.push(firstRegion.getMarker());
      }
    }

    return markers;
  }
  /* Return the set of Markers owned by this Patch that end at the patch's end position. */


  getEndingMarkers() {
    const markers = [this.marker];

    if (this.hunks.length > 0) {
      const lastHunk = this.hunks[this.hunks.length - 1];
      markers.push(lastHunk.getMarker());

      if (lastHunk.getRegions().length > 0) {
        const lastRegion = lastHunk.getRegions()[lastHunk.getRegions().length - 1];
        markers.push(lastRegion.getMarker());
      }
    }

    return markers;
  }

  buildStagePatchForLines(originalBuffer, nextPatchBuffer, rowSet) {
    const originalBaseOffset = this.getMarker().getRange().start.row;
    const builder = new BufferBuilder(originalBuffer, originalBaseOffset, nextPatchBuffer);
    const hunks = [];
    let newRowDelta = 0;

    for (const hunk of this.getHunks()) {
      let atLeastOneSelectedChange = false;
      let selectedDeletionRowCount = 0;
      let noNewlineRowCount = 0;

      for (const region of hunk.getRegions()) {
        for (const _ref of region.intersectRows(rowSet, true)) {
          const {
            intersection,
            gap
          } = _ref;
          region.when({
            addition: () => {
              if (gap) {
                // Unselected addition: omit from new buffer
                builder.remove(intersection);
              } else {
                // Selected addition: include in new patch
                atLeastOneSelectedChange = true;
                builder.append(intersection);
                builder.markRegion(intersection, _region.Addition);
              }
            },
            deletion: () => {
              if (gap) {
                // Unselected deletion: convert to context row
                builder.append(intersection);
                builder.markRegion(intersection, _region.Unchanged);
              } else {
                // Selected deletion: include in new patch
                atLeastOneSelectedChange = true;
                builder.append(intersection);
                builder.markRegion(intersection, _region.Deletion);
                selectedDeletionRowCount += intersection.getRowCount();
              }
            },
            unchanged: () => {
              // Untouched context line: include in new patch
              builder.append(intersection);
              builder.markRegion(intersection, _region.Unchanged);
            },
            nonewline: () => {
              builder.append(intersection);
              builder.markRegion(intersection, _region.NoNewline);
              noNewlineRowCount += intersection.getRowCount();
            }
          });
        }
      }

      if (atLeastOneSelectedChange) {
        // Hunk contains at least one selected line
        builder.markHunkRange(hunk.getRange());
        const {
          regions,
          marker
        } = builder.latestHunkWasIncluded();
        const newStartRow = hunk.getNewStartRow() + newRowDelta;
        const newRowCount = marker.getRange().getRowCount() - selectedDeletionRowCount - noNewlineRowCount;
        hunks.push(new _hunk["default"]({
          oldStartRow: hunk.getOldStartRow(),
          oldRowCount: hunk.getOldRowCount(),
          newStartRow,
          newRowCount,
          sectionHeading: hunk.getSectionHeading(),
          marker,
          regions
        }));
        newRowDelta += newRowCount - hunk.getNewRowCount();
      } else {
        newRowDelta += hunk.getOldRowCount() - hunk.getNewRowCount();
        builder.latestHunkWasDiscarded();
      }
    }

    const marker = nextPatchBuffer.markRange(this.constructor.layerName, [[0, 0], [nextPatchBuffer.getBuffer().getLastRow() - 1, Infinity]], {
      invalidate: 'never',
      exclusive: false
    });
    const wholeFile = rowSet.size === this.changedLineCount;
    const status = this.getStatus() === 'deleted' && !wholeFile ? 'modified' : this.getStatus();
    return this.clone({
      hunks,
      status,
      marker
    });
  }

  buildUnstagePatchForLines(originalBuffer, nextPatchBuffer, rowSet) {
    const originalBaseOffset = this.getMarker().getRange().start.row;
    const builder = new BufferBuilder(originalBuffer, originalBaseOffset, nextPatchBuffer);
    const hunks = [];
    let newRowDelta = 0;

    for (const hunk of this.getHunks()) {
      let atLeastOneSelectedChange = false;
      let contextRowCount = 0;
      let additionRowCount = 0;
      let deletionRowCount = 0;

      for (const region of hunk.getRegions()) {
        for (const _ref2 of region.intersectRows(rowSet, true)) {
          const {
            intersection,
            gap
          } = _ref2;
          region.when({
            addition: () => {
              if (gap) {
                // Unselected addition: become a context line.
                builder.append(intersection);
                builder.markRegion(intersection, _region.Unchanged);
                contextRowCount += intersection.getRowCount();
              } else {
                // Selected addition: become a deletion.
                atLeastOneSelectedChange = true;
                builder.append(intersection);
                builder.markRegion(intersection, _region.Deletion);
                deletionRowCount += intersection.getRowCount();
              }
            },
            deletion: () => {
              if (gap) {
                // Non-selected deletion: omit from new buffer.
                builder.remove(intersection);
              } else {
                // Selected deletion: becomes an addition
                atLeastOneSelectedChange = true;
                builder.append(intersection);
                builder.markRegion(intersection, _region.Addition);
                additionRowCount += intersection.getRowCount();
              }
            },
            unchanged: () => {
              // Untouched context line: include in new patch.
              builder.append(intersection);
              builder.markRegion(intersection, _region.Unchanged);
              contextRowCount += intersection.getRowCount();
            },
            nonewline: () => {
              // Nonewline marker: include in new patch.
              builder.append(intersection);
              builder.markRegion(intersection, _region.NoNewline);
            }
          });
        }
      }

      if (atLeastOneSelectedChange) {
        // Hunk contains at least one selected line
        builder.markHunkRange(hunk.getRange());
        const {
          marker,
          regions
        } = builder.latestHunkWasIncluded();
        hunks.push(new _hunk["default"]({
          oldStartRow: hunk.getNewStartRow(),
          oldRowCount: contextRowCount + deletionRowCount,
          newStartRow: hunk.getNewStartRow() + newRowDelta,
          newRowCount: contextRowCount + additionRowCount,
          sectionHeading: hunk.getSectionHeading(),
          marker,
          regions
        }));
      } else {
        builder.latestHunkWasDiscarded();
      } // (contextRowCount + additionRowCount) - (contextRowCount + deletionRowCount)


      newRowDelta += additionRowCount - deletionRowCount;
    }

    const wholeFile = rowSet.size === this.changedLineCount;
    let status = this.getStatus();

    if (this.getStatus() === 'added') {
      status = wholeFile ? 'deleted' : 'modified';
    } else if (this.getStatus() === 'deleted') {
      status = 'added';
    }

    const marker = nextPatchBuffer.markRange(this.constructor.layerName, [[0, 0], [nextPatchBuffer.getBuffer().getLastRow(), Infinity]], {
      invalidate: 'never',
      exclusive: false
    });
    return this.clone({
      hunks,
      status,
      marker
    });
  }

  getFirstChangeRange() {
    const firstHunk = this.getHunks()[0];

    if (!firstHunk) {
      return _atom.Range.fromObject([[0, 0], [0, 0]]);
    }

    const firstChange = firstHunk.getChanges()[0];

    if (!firstChange) {
      return _atom.Range.fromObject([[0, 0], [0, 0]]);
    }

    const firstRow = firstChange.getStartBufferRow();
    return _atom.Range.fromObject([[firstRow, 0], [firstRow, Infinity]]);
  }

  toStringIn(buffer) {
    return this.getHunks().reduce((str, hunk) => str + hunk.toStringIn(buffer), '');
  }
  /*
   * Construct a String containing internal diagnostic information.
   */

  /* istanbul ignore next */


  inspect(opts = {}) {
    const options = _objectSpread({
      indent: 0
    }, opts);

    let indentation = '';

    for (let i = 0; i < options.indent; i++) {
      indentation += ' ';
    }

    let inspectString = `${indentation}(Patch marker=${this.marker.id}`;

    if (this.marker.isDestroyed()) {
      inspectString += ' [destroyed]';
    }

    if (!this.marker.isValid()) {
      inspectString += ' [invalid]';
    }

    inspectString += '\n';

    for (const hunk of this.hunks) {
      inspectString += hunk.inspect({
        indent: options.indent + 2
      });
    }

    inspectString += `${indentation})\n`;
    return inspectString;
  }

  isPresent() {
    return true;
  }

  getRenderStatus() {
    return EXPANDED;
  }

}

exports["default"] = Patch;

_defineProperty(Patch, "layerName", 'patch');

class HiddenPatch extends Patch {
  constructor(marker, renderStatus, showFn) {
    super({
      status: null,
      hunks: [],
      marker
    });
    this.renderStatus = renderStatus;
    this.show = showFn;
  }

  getInsertionPoint() {
    return this.getRange().end;
  }

  getRenderStatus() {
    return this.renderStatus;
  }
  /*
   * Construct a String containing internal diagnostic information.
   */

  /* istanbul ignore next */


  inspect(opts = {}) {
    const options = _objectSpread({
      indent: 0
    }, opts);

    let indentation = '';

    for (let i = 0; i < options.indent; i++) {
      indentation += ' ';
    }

    return `${indentation}(HiddenPatch marker=${this.marker.id})\n`;
  }

}

class NullPatch {
  constructor() {
    const buffer = new _atom.TextBuffer();
    this.marker = buffer.markRange([[0, 0], [0, 0]]);
  }

  getStatus() {
    return null;
  }

  getMarker() {
    return this.marker;
  }

  getRange() {
    return this.getMarker().getRange();
  }

  getStartRange() {
    return _atom.Range.fromObject([[0, 0], [0, 0]]);
  }

  getHunks() {
    return [];
  }

  getChangedLineCount() {
    return 0;
  }

  containsRow() {
    return false;
  }

  getMaxLineNumberWidth() {
    return 0;
  }

  clone(opts = {}) {
    if (opts.status === undefined && opts.hunks === undefined && opts.marker === undefined && opts.renderStatus === undefined) {
      return this;
    } else {
      return new Patch({
        status: opts.status !== undefined ? opts.status : this.getStatus(),
        hunks: opts.hunks !== undefined ? opts.hunks : this.getHunks(),
        marker: opts.marker !== undefined ? opts.marker : this.getMarker(),
        renderStatus: opts.renderStatus !== undefined ? opts.renderStatus : this.getRenderStatus()
      });
    }
  }

  getStartingMarkers() {
    return [];
  }

  getEndingMarkers() {
    return [];
  }

  buildStagePatchForLines() {
    return this;
  }

  buildUnstagePatchForLines() {
    return this;
  }

  getFirstChangeRange() {
    return _atom.Range.fromObject([[0, 0], [0, 0]]);
  }

  updateMarkers() {}

  toStringIn() {
    return '';
  }
  /*
   * Construct a String containing internal diagnostic information.
   */

  /* istanbul ignore next */


  inspect(opts = {}) {
    const options = _objectSpread({
      indent: 0
    }, opts);

    let indentation = '';

    for (let i = 0; i < options.indent; i++) {
      indentation += ' ';
    }

    return `${indentation}(NullPatch)\n`;
  }

  isPresent() {
    return false;
  }

  getRenderStatus() {
    return EXPANDED;
  }

}

class BufferBuilder {
  constructor(original, originalBaseOffset, nextPatchBuffer) {
    this.originalBuffer = original;
    this.nextPatchBuffer = nextPatchBuffer; // The ranges provided to builder methods are expected to be valid within the original buffer. Account for
    // the position of the Patch within its original TextBuffer, and any existing content already on the next
    // TextBuffer.

    this.offset = this.nextPatchBuffer.getBuffer().getLastRow() - originalBaseOffset;
    this.hunkBufferText = '';
    this.hunkRowCount = 0;
    this.hunkStartOffset = this.offset;
    this.hunkRegions = [];
    this.hunkRange = null;
    this.lastOffset = 0;
  }

  append(range) {
    this.hunkBufferText += this.originalBuffer.getTextInRange(range) + '\n';
    this.hunkRowCount += range.getRowCount();
  }

  remove(range) {
    this.offset -= range.getRowCount();
  }

  markRegion(range, RegionKind) {
    const finalRange = this.offset !== 0 ? range.translate([this.offset, 0], [this.offset, 0]) : range; // Collapse consecutive ranges of the same RegionKind into one continuous region.

    const lastRegion = this.hunkRegions[this.hunkRegions.length - 1];

    if (lastRegion && lastRegion.RegionKind === RegionKind && finalRange.start.row - lastRegion.range.end.row === 1) {
      lastRegion.range.end = finalRange.end;
    } else {
      this.hunkRegions.push({
        RegionKind,
        range: finalRange
      });
    }
  }

  markHunkRange(range) {
    let finalRange = range;

    if (this.hunkStartOffset !== 0 || this.offset !== 0) {
      finalRange = finalRange.translate([this.hunkStartOffset, 0], [this.offset, 0]);
    }

    this.hunkRange = finalRange;
  }

  latestHunkWasIncluded() {
    this.nextPatchBuffer.buffer.append(this.hunkBufferText, {
      normalizeLineEndings: false
    });
    const regions = this.hunkRegions.map(({
      RegionKind,
      range
    }) => {
      const regionMarker = this.nextPatchBuffer.markRange(RegionKind.layerName, range, {
        invalidate: 'never',
        exclusive: false
      });
      return new RegionKind(regionMarker);
    });
    const marker = this.nextPatchBuffer.markRange('hunk', this.hunkRange, {
      invalidate: 'never',
      exclusive: false
    });
    this.hunkBufferText = '';
    this.hunkRowCount = 0;
    this.hunkStartOffset = this.offset;
    this.hunkRegions = [];
    this.hunkRange = null;
    return {
      regions,
      marker
    };
  }

  latestHunkWasDiscarded() {
    this.offset -= this.hunkRowCount;
    this.hunkBufferText = '';
    this.hunkRowCount = 0;
    this.hunkStartOffset = this.offset;
    this.hunkRegions = [];
    this.hunkRange = null;
    return {
      regions: [],
      marker: null
    };
  }

}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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