"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _markerTools = require("./marker-tools");

class Side {
  constructor(editor, marker, blockMarker, source, position, banner, originalText) {
    this.editor = editor;
    this.marker = marker;
    this.blockMarker = blockMarker;
    this.source = source;
    this.position = position;
    this.banner = banner;
    this.originalText = originalText;
  }

  getMarker() {
    return this.marker;
  }

  getBannerMarker() {
    return this.banner.getMarker();
  }

  getSource() {
    return this.source;
  }

  getBlockMarker() {
    return this.blockMarker;
  }

  getBlockPosition() {
    return this.position.when({
      top: () => 'before',
      middle: () => 'before',
      bottom: () => 'after'
    });
  }

  getLineCSSClass() {
    if (this.isModified() || this.isBannerModified()) {
      return 'github-ConflictModified';
    } else {
      return this.source.getCSSClass();
    }
  }

  getBannerCSSClass() {
    if (this.isModified() || this.isBannerModified()) {
      return 'github-ConflictModifiedBanner';
    } else {
      return this.source.getBannerCSSClass();
    }
  }

  getBlockCSSClasses() {
    const cxs = ['github-ConflictBlock'];
    cxs.push(this.source.getBlockCSSClass());
    cxs.push(this.position.getBlockCSSClass());

    if (this.isModified() || this.isBannerModified()) {
      cxs.push('github-ConflictModifiedBlock');
    }

    return cxs.join(' ');
  }

  getPosition() {
    return this.position;
  }

  getRange() {
    const bannerRange = this.banner.getRange();
    const bodyRange = this.marker.getBufferRange();
    return bannerRange.union(bodyRange);
  }

  includesPoint(point) {
    return this.getRange().containsPoint(point);
  }

  getText() {
    return this.editor.getTextInBufferRange(this.getMarker().getBufferRange());
  }

  isBannerModified() {
    return this.banner.isModified();
  }

  isModified() {
    return this.getText() !== this.originalText;
  }

  isEmpty() {
    return this.marker.getBufferRange().isEmpty();
  }

  revertBanner() {
    this.banner.revert();
  }

  revert() {
    const range = this.getMarker().getBufferRange();
    this.editor.setTextInBufferRange(range, this.originalText);
    this.getMarker().setBufferRange(range);
  }

  deleteBanner() {
    this.banner["delete"]();
  }

  delete() {
    (0, _markerTools.deleteMarkerIn)(this.getMarker(), this.editor);
  }

  appendText(text) {
    const insertionPoint = this.getMarker().getBufferRange().end;
    return this.editor.setTextInBufferRange([insertionPoint, insertionPoint], text);
  }

}

exports["default"] = Side;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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