"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _listSelection = _interopRequireDefault(require("./list-selection"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

const COPY = Symbol('COPY');

class CompositeListSelection {
  constructor(options) {
    if (options._copy !== COPY) {
      this.keysBySelection = new Map();
      this.selections = [];

      this.idForItem = options.idForItem || (item => item);

      this.resolveNextUpdatePromise = () => {};

      this.activeSelectionIndex = null;

      for (const [key, items] of options.listsByKey) {
        const selection = new _listSelection["default"]({
          items
        });
        this.keysBySelection.set(selection, key);
        this.selections.push(selection);

        if (this.activeSelectionIndex === null && selection.getItems().length) {
          this.activeSelectionIndex = this.selections.length - 1;
        }
      }

      if (this.activeSelectionIndex === null) {
        this.activeSelectionIndex = 0;
      }
    } else {
      this.keysBySelection = options.keysBySelection;
      this.selections = options.selections;
      this.idForItem = options.idForItem;
      this.activeSelectionIndex = options.activeSelectionIndex;
      this.resolveNextUpdatePromise = options.resolveNextUpdatePromise;
    }
  }

  copy(options = {}) {
    let selections = [];
    let keysBySelection = new Map();

    if (options.keysBySelection || options.selections) {
      if (!options.keysBySelection || !options.selections) {
        throw new Error('keysBySelection and selection must always be updated simultaneously');
      }

      selections = options.selections;
      keysBySelection = options.keysBySelection;
    } else {
      selections = this.selections;
      keysBySelection = this.keysBySelection;
    }

    return new CompositeListSelection({
      keysBySelection,
      selections,
      activeSelectionIndex: options.activeSelectionIndex !== undefined ? options.activeSelectionIndex : this.activeSelectionIndex,
      idForItem: options.idForItem || this.idForItem,
      resolveNextUpdatePromise: options.resolveNextUpdatePromise || this.resolveNextUpdatePromise,
      _copy: COPY
    });
  }

  updateLists(listsByKey) {
    let isDifferent = false;

    if (listsByKey.length === 0) {
      return this;
    }

    const newKeysBySelection = new Map();
    const newSelections = [];

    for (let i = 0; i < listsByKey.length; i++) {
      const [key, newItems] = listsByKey[i];
      let selection = this.selections[i];
      const oldItems = selection.getItems();

      if (!isDifferent) {
        isDifferent = oldItems.length !== newItems.length || oldItems.some((oldItem, j) => oldItem === newItems[j]);
      }

      const oldHeadItem = selection.getHeadItem();
      selection = selection.setItems(newItems);
      let newHeadItem = null;

      if (oldHeadItem) {
        newHeadItem = newItems.find(item => this.idForItem(item) === this.idForItem(oldHeadItem));
      }

      if (newHeadItem) {
        selection = selection.selectItem(newHeadItem);
      }

      newKeysBySelection.set(selection, key);
      newSelections.push(selection);
    }

    let updated = this.copy({
      keysBySelection: newKeysBySelection,
      selections: newSelections
    });

    if (updated.getActiveSelection().getItems().length === 0) {
      const next = updated.activateNextSelection();
      updated = next !== updated ? next : updated.activatePreviousSelection();
    }

    updated.resolveNextUpdatePromise();
    return updated;
  }

  updateActiveSelection(fn) {
    const oldSelection = this.getActiveSelection();
    const newSelection = fn(oldSelection);

    if (oldSelection === newSelection) {
      return this;
    }

    const key = this.keysBySelection.get(oldSelection);
    const newKeysBySelection = new Map(this.keysBySelection);
    newKeysBySelection["delete"](oldSelection);
    newKeysBySelection.set(newSelection, key);
    const newSelections = this.selections.slice();
    newSelections[this.activeSelectionIndex] = newSelection;
    return this.copy({
      keysBySelection: newKeysBySelection,
      selections: newSelections
    });
  }

  getNextUpdatePromise() {
    return new Promise((resolve, reject) => {
      this.resolveNextUpdatePromise = resolve;
    });
  }

  selectFirstNonEmptyList() {
    return this.copy({
      activeSelectionIndex: this.selections.findIndex(selection => selection.getItems().length > 0)
    });
  }

  getActiveListKey() {
    return this.keysBySelection.get(this.getActiveSelection());
  }

  getSelectedItems() {
    return this.getActiveSelection().getSelectedItems();
  }

  getHeadItem() {
    return this.getActiveSelection().getHeadItem();
  }

  getActiveSelection() {
    return this.selections[this.activeSelectionIndex];
  }

  activateSelection(selection) {
    const index = this.selections.indexOf(selection);

    if (index === -1) {
      throw new Error('Selection not found');
    }

    return this.copy({
      activeSelectionIndex: index
    });
  }

  activateNextSelection() {
    for (let i = this.activeSelectionIndex + 1; i < this.selections.length; i++) {
      if (this.selections[i].getItems().length > 0) {
        return this.copy({
          activeSelectionIndex: i
        });
      }
    }

    return this;
  }

  activatePreviousSelection() {
    for (let i = this.activeSelectionIndex - 1; i >= 0; i--) {
      if (this.selections[i].getItems().length > 0) {
        return this.copy({
          activeSelectionIndex: i
        });
      }
    }

    return this;
  }

  activateLastSelection() {
    for (let i = this.selections.length - 1; i >= 0; i--) {
      if (this.selections[i].getItems().length > 0) {
        return this.copy({
          activeSelectionIndex: i
        });
      }
    }

    return this;
  }

  selectItem(item, preserveTail = false) {
    const selection = this.selectionForItem(item);

    if (!selection) {
      throw new Error(`No item found: ${item}`);
    }

    let next = this;

    if (!preserveTail) {
      next = next.activateSelection(selection);
    }

    if (selection === next.getActiveSelection()) {
      next = next.updateActiveSelection(s => s.selectItem(item, preserveTail));
    }

    return next;
  }

  addOrSubtractSelection(item) {
    const selection = this.selectionForItem(item);

    if (!selection) {
      throw new Error(`No item found: ${item}`);
    }

    if (selection === this.getActiveSelection()) {
      return this.updateActiveSelection(s => s.addOrSubtractSelection(item));
    } else {
      return this.activateSelection(selection).updateActiveSelection(s => s.selectItem(item));
    }
  }

  selectAllItems() {
    return this.updateActiveSelection(s => s.selectAllItems());
  }

  selectFirstItem(preserveTail) {
    return this.updateActiveSelection(s => s.selectFirstItem(preserveTail));
  }

  selectLastItem(preserveTail) {
    return this.updateActiveSelection(s => s.selectLastItem(preserveTail));
  }

  coalesce() {
    return this.updateActiveSelection(s => s.coalesce());
  }

  selectionForItem(item) {
    return this.selections.find(selection => selection.getItems().includes(item));
  }

  listKeyForItem(item) {
    return this.keysBySelection.get(this.selectionForItem(item));
  }

  selectNextItem(preserveTail = false) {
    let next = this;

    if (!preserveTail && next.getActiveSelection().getHeadItem() === next.getActiveSelection().getLastItem()) {
      next = next.activateNextSelection();

      if (next !== this) {
        return next.updateActiveSelection(s => s.selectFirstItem());
      } else {
        return next.updateActiveSelection(s => s.selectLastItem());
      }
    } else {
      return next.updateActiveSelection(s => s.selectNextItem(preserveTail));
    }
  }

  selectPreviousItem(preserveTail = false) {
    let next = this;

    if (!preserveTail && next.getActiveSelection().getHeadItem() === next.getActiveSelection().getItems()[0]) {
      next = next.activatePreviousSelection();

      if (next !== this) {
        return next.updateActiveSelection(s => s.selectLastItem());
      } else {
        return next.updateActiveSelection(s => s.selectFirstItem());
      }
    } else {
      return next.updateActiveSelection(s => s.selectPreviousItem(preserveTail));
    }
  }

  findItem(predicate) {
    for (let i = 0; i < this.selections.length; i++) {
      const selection = this.selections[i];
      const key = this.keysBySelection.get(selection);
      const found = selection.getItems().find(item => predicate(item, key));

      if (found !== undefined) {
        return found;
      }
    }

    return null;
  }

}

exports["default"] = CompositeListSelection;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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