/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 * 
 * @format
 */
'use strict';

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

var _extends2 = _interopRequireDefault(require("@babel/runtime/helpers/extends"));

var React = require("react");

var ReactRelayContext = require("./ReactRelayContext");

var assertFragmentMap = require("./assertFragmentMap");

var invariant = require("fbjs/lib/invariant");

var mapObject = require("fbjs/lib/mapObject");

var readContext = require("./readContext");

var _require = require("./ReactRelayContainerUtils"),
    getComponentName = _require.getComponentName,
    getContainerName = _require.getContainerName;

/**
 * Creates a component class whose instances adapt to the
 * `context.relay.environment` in which they are rendered and which have the
 * necessary static methods (`getFragment()` etc) to be composed within classic
 * `Relay.Containers`.
 */
function buildReactRelayContainer(ComponentClass, fragmentSpec, createContainerWithFragments) {
  // Sanity-check user-defined fragment input
  var containerName = getContainerName(ComponentClass);
  assertFragmentMap(getComponentName(ComponentClass), fragmentSpec); // Memoize a container for the last environment instance encountered

  var environment;
  var Container;

  function ContainerConstructor(props) {
    if (Container == null || props.__relayContext.environment !== environment) {
      environment = props.__relayContext.environment;

      if (process.env.NODE_ENV !== "production") {
        var _require2 = require("relay-runtime"),
            isRelayModernEnvironment = _require2.isRelayModernEnvironment;

        if (!isRelayModernEnvironment(environment)) {
          throw new Error('RelayModernContainer: Can only use Relay Modern component ' + "".concat(containerName, " in a Relay Modern environment!\n") + 'When using Relay Modern and Relay Classic in the same ' + 'application, ensure components use Relay Compat to work in ' + 'both environments.\n' + 'See: http://facebook.github.io/relay/docs/relay-compat.html');
        }
      }

      var getFragmentFromTag = environment.unstable_internal.getFragment;

      var _fragments = mapObject(fragmentSpec, getFragmentFromTag);

      Container = createContainerWithFragments(ComponentClass, _fragments); // Attach static lifecycle to wrapper component so React can see it.

      ContainerConstructor.getDerivedStateFromProps = Container.getDerivedStateFromProps;
    } // $FlowFixMe


    return new Container(props);
  }

  function forwardRef(props, ref) {
    var context = readContext(ReactRelayContext);
    !context ? process.env.NODE_ENV !== "production" ? invariant(false, "".concat(containerName, " tried to render a context that was ") + "not valid this means that ".concat(containerName, " was rendered outside of a ") + 'query renderer.') : invariant(false) : void 0;
    return React.createElement(ContainerConstructor, (0, _extends2["default"])({}, props, {
      __relayContext: context,
      componentRef: props.componentRef || ref
    }));
  }

  forwardRef.displayName = containerName;
  var ForwardContainer = React.forwardRef(forwardRef);

  if (process.env.NODE_ENV !== "production") {
    /* $FlowFixMe(>=0.89.0 site=www,mobile,react_native_fb,oss) Suppressing
     * errors found while preparing to upgrade to 0.89.0 */
    ForwardContainer.__ComponentClass = ComponentClass; // Classic container static methods.

    /* $FlowFixMe(>=0.89.0 site=www,mobile,react_native_fb,oss) Suppressing
     * errors found while preparing to upgrade to 0.89.0 */

    ForwardContainer.getFragment = function getFragmentOnModernContainer() {
      throw new Error("RelayModernContainer: ".concat(containerName, ".getFragment() was called on ") + 'a Relay Modern component by a Relay Classic or Relay Compat ' + 'component.\n' + 'When using Relay Modern and Relay Classic in the same ' + 'application, ensure components use Relay Compat to work in ' + 'both environments.\n' + 'See: http://facebook.github.io/relay/docs/relay-compat.html');
    };
  }
  /* $FlowFixMe(>=0.89.0 site=www,mobile,react_native_fb,oss) Suppressing errors
   * found while preparing to upgrade to 0.89.0 */


  return ForwardContainer;
}

module.exports = buildReactRelayContainer;