(function() {
  var CommandLogger, FileURLRegExp, NotificationIssue, StackTraceParser, TITLE_CHAR_LIMIT, UserUtilities, fs, path,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  fs = require('fs-plus');

  path = require('path');

  StackTraceParser = require('stacktrace-parser');

  CommandLogger = require('./command-logger');

  UserUtilities = require('./user-utilities');

  TITLE_CHAR_LIMIT = 100;

  FileURLRegExp = new RegExp('file://\w*/(.*)');

  module.exports = NotificationIssue = (function() {
    function NotificationIssue(notification) {
      this.notification = notification;
      this.normalizedStackPaths = bind(this.normalizedStackPaths, this);
    }

    NotificationIssue.prototype.findSimilarIssues = function() {
      var githubHeaders, issueTitle, query, repo, repoUrl;
      repoUrl = this.getRepoUrl();
      if (repoUrl == null) {
        repoUrl = 'atom/atom';
      }
      repo = repoUrl.replace(/http(s)?:\/\/(\d+\.)?github.com\//gi, '');
      issueTitle = this.getIssueTitle();
      query = issueTitle + " repo:" + repo;
      githubHeaders = new Headers({
        accept: 'application/vnd.github.v3+json',
        contentType: "application/json"
      });
      return fetch("https://api.github.com/search/issues?q=" + (encodeURIComponent(query)) + "&sort=created", {
        headers: githubHeaders
      }).then(function(r) {
        return r != null ? r.json() : void 0;
      }).then(function(data) {
        var issue, issues, j, len, ref;
        if ((data != null ? data.items : void 0) != null) {
          issues = {};
          ref = data.items;
          for (j = 0, len = ref.length; j < len; j++) {
            issue = ref[j];
            if (issue.title.indexOf(issueTitle) > -1 && (issues[issue.state] == null)) {
              issues[issue.state] = issue;
              if ((issues.open != null) && (issues.closed != null)) {
                return issues;
              }
            }
          }
          if ((issues.open != null) || (issues.closed != null)) {
            return issues;
          }
        }
        return null;
      })["catch"](function(e) {
        return null;
      });
    };

    NotificationIssue.prototype.getIssueUrlForSystem = function() {
      return this.getIssueUrl().then(function(issueUrl) {
        return fetch("https://is.gd/create.php?format=simple", {
          method: 'POST',
          headers: {
            'Content-Type': 'application/x-www-form-urlencoded'
          },
          body: "url=" + (encodeURIComponent(issueUrl))
        }).then(function(r) {
          return r.text();
        })["catch"](function(e) {
          return null;
        });
      });
    };

    NotificationIssue.prototype.getIssueUrl = function() {
      return this.getIssueBody().then((function(_this) {
        return function(issueBody) {
          var repoUrl;
          repoUrl = _this.getRepoUrl();
          if (repoUrl == null) {
            repoUrl = 'https://github.com/atom/atom';
          }
          return repoUrl + "/issues/new?title=" + (_this.encodeURI(_this.getIssueTitle())) + "&body=" + (_this.encodeURI(issueBody));
        };
      })(this));
    };

    NotificationIssue.prototype.encodeURI = function(str) {
      return encodeURI(str).replace(/#/g, '%23').replace(/;/g, '%3B').replace(/%20/g, '+');
    };

    NotificationIssue.prototype.getIssueTitle = function() {
      var title;
      title = this.notification.getMessage();
      title = title.replace(process.env.ATOM_HOME, '$ATOM_HOME');
      if (process.platform === 'win32') {
        title = title.replace(process.env.USERPROFILE, '~');
        title = title.replace(path.sep, path.posix.sep);
      } else {
        title = title.replace(process.env.HOME, '~');
      }
      if (title.length > TITLE_CHAR_LIMIT) {
        title = title.substring(0, TITLE_CHAR_LIMIT - 3) + '...';
      }
      return title.replace(/\r?\n|\r/g, "");
    };

    NotificationIssue.prototype.getIssueBody = function() {
      return new Promise((function(_this) {
        return function(resolve, reject) {
          var nonCorePackagesPromise, systemPromise;
          if (_this.issueBody) {
            return resolve(_this.issueBody);
          }
          systemPromise = UserUtilities.getOSVersion();
          nonCorePackagesPromise = UserUtilities.getNonCorePackages();
          return Promise.all([systemPromise, nonCorePackagesPromise]).then(function(all) {
            var copyText, message, nonCorePackages, options, packageMessage, packageName, packageVersion, ref, ref1, repoUrl, rootUserStatus, systemName, systemUser;
            systemName = all[0], nonCorePackages = all[1];
            message = _this.notification.getMessage();
            options = _this.notification.getOptions();
            repoUrl = _this.getRepoUrl();
            packageName = _this.getPackageName();
            if (packageName != null) {
              packageVersion = (ref = atom.packages.getLoadedPackage(packageName)) != null ? (ref1 = ref.metadata) != null ? ref1.version : void 0 : void 0;
            }
            copyText = '';
            systemUser = process.env.USER;
            rootUserStatus = '';
            if (systemUser === 'root') {
              rootUserStatus = '**User**: root';
            }
            if ((packageName != null) && (repoUrl != null)) {
              packageMessage = "[" + packageName + "](" + repoUrl + ") package " + packageVersion;
            } else if (packageName != null) {
              packageMessage = "'" + packageName + "' package v" + packageVersion;
            } else {
              packageMessage = 'Atom Core';
            }
            _this.issueBody = "[Enter steps to reproduce:]\n\n1. ...\n2. ...\n\n**Atom**: " + (atom.getVersion()) + " " + process.arch + "\n**Electron**: " + process.versions.electron + "\n**OS**: " + systemName + "\n**Thrown From**: " + packageMessage + "\n" + rootUserStatus + "\n\n### Stack Trace\n\n" + message + "\n\n```\nAt " + options.detail + "\n\n" + (_this.normalizedStackPaths(options.stack)) + "\n```\n\n### Commands\n\n" + (CommandLogger.instance().getText()) + "\n\n### Non-Core Packages\n\n```\n" + (nonCorePackages.join('\n')) + "\n```\n\n" + copyText;
            return resolve(_this.issueBody);
          });
        };
      })(this));
    };

    NotificationIssue.prototype.normalizedStackPaths = function(stack) {
      return stack != null ? stack.replace(/(^\W+at )([\w.]{2,} [(])?(.*)(:\d+:\d+[)]?)/gm, (function(_this) {
        return function(m, p1, p2, p3, p4) {
          return p1 + (p2 || '') + _this.normalizePath(p3) + p4;
        };
      })(this)) : void 0;
    };

    NotificationIssue.prototype.normalizePath = function(path) {
      return path.replace('file:///', '').replace(/[\/]/g, '\\').replace(fs.getHomeDirectory(), '~').replace(/\\/g, '/').replace(/.*(\/(app\.asar|packages\/).*)/, '$1');
    };

    NotificationIssue.prototype.getRepoUrl = function() {
      var packageName, packagePath, ref, ref1, ref2, ref3, ref4, repo, repoUrl;
      packageName = this.getPackageName();
      if (packageName == null) {
        return;
      }
      repo = (ref = atom.packages.getLoadedPackage(packageName)) != null ? (ref1 = ref.metadata) != null ? ref1.repository : void 0 : void 0;
      repoUrl = (ref2 = repo != null ? repo.url : void 0) != null ? ref2 : repo;
      if (!repoUrl) {
        if (packagePath = atom.packages.resolvePackagePath(packageName)) {
          try {
            repo = (ref3 = JSON.parse(fs.readFileSync(path.join(packagePath, 'package.json')))) != null ? ref3.repository : void 0;
            repoUrl = (ref4 = repo != null ? repo.url : void 0) != null ? ref4 : repo;
          } catch (error) {}
        }
      }
      return repoUrl != null ? repoUrl.replace(/\.git$/, '').replace(/^git\+/, '') : void 0;
    };

    NotificationIssue.prototype.getPackageNameFromFilePath = function(filePath) {
      var packageName, ref, ref1, ref2, ref3;
      if (!filePath) {
        return;
      }
      packageName = (ref = /\/\.atom\/dev\/packages\/([^\/]+)\//.exec(filePath)) != null ? ref[1] : void 0;
      if (packageName) {
        return packageName;
      }
      packageName = (ref1 = /\\\.atom\\dev\\packages\\([^\\]+)\\/.exec(filePath)) != null ? ref1[1] : void 0;
      if (packageName) {
        return packageName;
      }
      packageName = (ref2 = /\/\.atom\/packages\/([^\/]+)\//.exec(filePath)) != null ? ref2[1] : void 0;
      if (packageName) {
        return packageName;
      }
      packageName = (ref3 = /\\\.atom\\packages\\([^\\]+)\\/.exec(filePath)) != null ? ref3[1] : void 0;
      if (packageName) {
        return packageName;
      }
    };

    NotificationIssue.prototype.getPackageName = function() {
      var file, getPackageName, i, j, options, packageName, packagePath, packagePaths, ref, stack;
      options = this.notification.getOptions();
      if (options.packageName != null) {
        return options.packageName;
      }
      if (!((options.stack != null) || (options.detail != null))) {
        return;
      }
      packagePaths = this.getPackagePathsByPackageName();
      for (packageName in packagePaths) {
        packagePath = packagePaths[packageName];
        if (packagePath.indexOf(path.join('.atom', 'dev', 'packages')) > -1 || packagePath.indexOf(path.join('.atom', 'packages')) > -1) {
          packagePaths[packageName] = fs.realpathSync(packagePath);
        }
      }
      getPackageName = (function(_this) {
        return function(filePath) {
          var isSubfolder, match, packName;
          filePath = /\((.+?):\d+|\((.+)\)|(.+)/.exec(filePath)[0];
          if (match = FileURLRegExp.exec(filePath)) {
            filePath = match[1];
          }
          filePath = path.normalize(filePath);
          if (path.isAbsolute(filePath)) {
            for (packName in packagePaths) {
              packagePath = packagePaths[packName];
              if (filePath === 'node.js') {
                continue;
              }
              isSubfolder = filePath.indexOf(path.normalize(packagePath + path.sep)) === 0;
              if (isSubfolder) {
                return packName;
              }
            }
          }
          return _this.getPackageNameFromFilePath(filePath);
        };
      })(this);
      if ((options.detail != null) && (packageName = getPackageName(options.detail))) {
        return packageName;
      }
      if (options.stack != null) {
        stack = StackTraceParser.parse(options.stack);
        for (i = j = 0, ref = stack.length; 0 <= ref ? j < ref : j > ref; i = 0 <= ref ? ++j : --j) {
          file = stack[i].file;
          if (!file) {
            return;
          }
          packageName = getPackageName(file);
          if (packageName != null) {
            return packageName;
          }
        }
      }
    };

    NotificationIssue.prototype.getPackagePathsByPackageName = function() {
      var j, len, pack, packagePathsByPackageName, ref;
      packagePathsByPackageName = {};
      ref = atom.packages.getLoadedPackages();
      for (j = 0, len = ref.length; j < len; j++) {
        pack = ref[j];
        packagePathsByPackageName[pack.name] = pack.path;
      }
      return packagePathsByPackageName;
    };

    return NotificationIssue;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
