(function() {
  var ChildProcess, Emitter, Grim, Task, _,
    slice = [].slice;

  _ = require('underscore-plus');

  ChildProcess = require('child_process');

  Emitter = require('event-kit').Emitter;

  Grim = require('grim');

  module.exports = Task = (function() {
    Task.once = function() {
      var args, task, taskPath;
      taskPath = arguments[0], args = 2 <= arguments.length ? slice.call(arguments, 1) : [];
      task = new Task(taskPath);
      task.once('task:completed', function() {
        return task.terminate();
      });
      task.start.apply(task, args);
      return task;
    };

    Task.prototype.callback = null;

    function Task(taskPath) {
      var compileCachePath, env;
      this.emitter = new Emitter;
      compileCachePath = require('./compile-cache').getCacheDirectory();
      taskPath = require.resolve(taskPath);
      env = Object.assign({}, process.env, {
        userAgent: navigator.userAgent
      });
      this.childProcess = ChildProcess.fork(require.resolve('./task-bootstrap'), [compileCachePath, taskPath], {
        env: env,
        silent: true
      });
      this.on("task:log", function() {
        return console.log.apply(console, arguments);
      });
      this.on("task:warn", function() {
        return console.warn.apply(console, arguments);
      });
      this.on("task:error", function() {
        return console.error.apply(console, arguments);
      });
      this.on("task:deprecations", function(deprecations) {
        var deprecation, i, len;
        for (i = 0, len = deprecations.length; i < len; i++) {
          deprecation = deprecations[i];
          Grim.addSerializedDeprecation(deprecation);
        }
      });
      this.on("task:completed", (function(_this) {
        return function() {
          var args;
          args = 1 <= arguments.length ? slice.call(arguments, 0) : [];
          return typeof _this.callback === "function" ? _this.callback.apply(_this, args) : void 0;
        };
      })(this));
      this.handleEvents();
    }

    Task.prototype.handleEvents = function() {
      this.childProcess.removeAllListeners('message');
      this.childProcess.on('message', (function(_this) {
        return function(arg) {
          var args, event;
          event = arg.event, args = arg.args;
          if (_this.childProcess != null) {
            return _this.emitter.emit(event, args);
          }
        };
      })(this));
      if (this.childProcess.stdout != null) {
        this.childProcess.stdout.removeAllListeners('data');
        this.childProcess.stdout.on('data', function(data) {
          return console.log(data.toString());
        });
      }
      if (this.childProcess.stderr != null) {
        this.childProcess.stderr.removeAllListeners('data');
        return this.childProcess.stderr.on('data', function(data) {
          return console.error(data.toString());
        });
      }
    };

    Task.prototype.start = function() {
      var args, callback, i;
      args = 2 <= arguments.length ? slice.call(arguments, 0, i = arguments.length - 1) : (i = 0, []), callback = arguments[i++];
      if (this.childProcess == null) {
        throw new Error('Cannot start terminated process');
      }
      this.handleEvents();
      if (_.isFunction(callback)) {
        this.callback = callback;
      } else {
        args.push(callback);
      }
      this.send({
        event: 'start',
        args: args
      });
      return void 0;
    };

    Task.prototype.send = function(message) {
      if (this.childProcess != null) {
        this.childProcess.send(message);
      } else {
        throw new Error('Cannot send message to terminated process');
      }
      return void 0;
    };

    Task.prototype.on = function(eventName, callback) {
      return this.emitter.on(eventName, function(args) {
        return callback.apply(null, args);
      });
    };

    Task.prototype.once = function(eventName, callback) {
      var disposable;
      return disposable = this.on(eventName, function() {
        var args;
        args = 1 <= arguments.length ? slice.call(arguments, 0) : [];
        disposable.dispose();
        return callback.apply(null, args);
      });
    };

    Task.prototype.terminate = function() {
      var ref, ref1;
      if (this.childProcess == null) {
        return false;
      }
      this.childProcess.removeAllListeners('message');
      if ((ref = this.childProcess.stdout) != null) {
        ref.removeAllListeners('data');
      }
      if ((ref1 = this.childProcess.stderr) != null) {
        ref1.removeAllListeners('data');
      }
      this.childProcess.kill();
      this.childProcess = null;
      return true;
    };

    Task.prototype.cancel = function() {
      var didForcefullyTerminate;
      didForcefullyTerminate = this.terminate();
      if (didForcefullyTerminate) {
        this.emitter.emit('task:cancelled');
      }
      return didForcefullyTerminate;
    };

    return Task;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
