(function() {
  var CompositeDisposable, WrapGuideElement;

  CompositeDisposable = require('atom').CompositeDisposable;

  module.exports = WrapGuideElement = (function() {
    function WrapGuideElement(editor, editorElement) {
      this.editor = editor;
      this.editorElement = editorElement;
      this.subscriptions = new CompositeDisposable();
      this.configSubscriptions = new CompositeDisposable();
      this.element = document.createElement('div');
      this.element.setAttribute('is', 'wrap-guide');
      this.element.classList.add('wrap-guide-container');
      this.attachToLines();
      this.handleEvents();
      this.updateGuide();
      this.element.updateGuide = this.updateGuide.bind(this);
      this.element.getDefaultColumn = this.getDefaultColumn.bind(this);
    }

    WrapGuideElement.prototype.attachToLines = function() {
      var scrollView;
      scrollView = this.editorElement.querySelector('.scroll-view');
      return scrollView != null ? scrollView.appendChild(this.element) : void 0;
    };

    WrapGuideElement.prototype.handleEvents = function() {
      var updateGuideCallback;
      updateGuideCallback = (function(_this) {
        return function() {
          return _this.updateGuide();
        };
      })(this);
      this.handleConfigEvents();
      this.subscriptions.add(atom.config.onDidChange('editor.fontSize', (function(_this) {
        return function() {
          return _this.editorElement.getComponent().getNextUpdatePromise().then(function() {
            return updateGuideCallback();
          });
        };
      })(this)));
      this.subscriptions.add(this.editorElement.onDidChangeScrollLeft(updateGuideCallback));
      this.subscriptions.add(this.editor.onDidChangePath(updateGuideCallback));
      this.subscriptions.add(this.editor.onDidChangeGrammar((function(_this) {
        return function() {
          _this.configSubscriptions.dispose();
          _this.handleConfigEvents();
          return updateGuideCallback();
        };
      })(this)));
      this.subscriptions.add(this.editor.onDidDestroy((function(_this) {
        return function() {
          _this.subscriptions.dispose();
          return _this.configSubscriptions.dispose();
        };
      })(this)));
      return this.subscriptions.add(this.editorElement.onDidAttach((function(_this) {
        return function() {
          _this.attachToLines();
          return updateGuideCallback();
        };
      })(this)));
    };

    WrapGuideElement.prototype.handleConfigEvents = function() {
      var uniqueAscending, updateGuideCallback, updateGuidesCallback, updatePreferredLineLengthCallback;
      uniqueAscending = require('./main').uniqueAscending;
      updatePreferredLineLengthCallback = (function(_this) {
        return function(args) {
          var columns, i;
          columns = atom.config.get('wrap-guide.columns', {
            scope: _this.editor.getRootScopeDescriptor()
          });
          if (columns.length > 0) {
            columns[columns.length - 1] = args.newValue;
            columns = uniqueAscending((function() {
              var j, len, results;
              results = [];
              for (j = 0, len = columns.length; j < len; j++) {
                i = columns[j];
                if (i <= args.newValue) {
                  results.push(i);
                }
              }
              return results;
            })());
            atom.config.set('wrap-guide.columns', columns, {
              scopeSelector: "." + (_this.editor.getGrammar().scopeName)
            });
          }
          return _this.updateGuide();
        };
      })(this);
      this.configSubscriptions.add(atom.config.onDidChange('editor.preferredLineLength', {
        scope: this.editor.getRootScopeDescriptor()
      }, updatePreferredLineLengthCallback));
      updateGuideCallback = (function(_this) {
        return function() {
          return _this.updateGuide();
        };
      })(this);
      this.configSubscriptions.add(atom.config.onDidChange('wrap-guide.enabled', {
        scope: this.editor.getRootScopeDescriptor()
      }, updateGuideCallback));
      updateGuidesCallback = (function(_this) {
        return function(args) {
          var columns;
          columns = uniqueAscending(args.newValue);
          if (columns != null ? columns.length : void 0) {
            atom.config.set('wrap-guide.columns', columns);
            atom.config.set('editor.preferredLineLength', columns[columns.length - 1], {
              scopeSelector: "." + (_this.editor.getGrammar().scopeName)
            });
            return _this.updateGuide();
          }
        };
      })(this);
      return this.configSubscriptions.add(atom.config.onDidChange('wrap-guide.columns', {
        scope: this.editor.getRootScopeDescriptor()
      }, updateGuidesCallback));
    };

    WrapGuideElement.prototype.getDefaultColumn = function() {
      return atom.config.get('editor.preferredLineLength', {
        scope: this.editor.getRootScopeDescriptor()
      });
    };

    WrapGuideElement.prototype.getGuidesColumns = function(path, scopeName) {
      var columns, ref;
      columns = (ref = atom.config.get('wrap-guide.columns', {
        scope: this.editor.getRootScopeDescriptor()
      })) != null ? ref : [];
      if (columns.length > 0) {
        return columns;
      }
      return [this.getDefaultColumn()];
    };

    WrapGuideElement.prototype.isEnabled = function() {
      var ref;
      return (ref = atom.config.get('wrap-guide.enabled', {
        scope: this.editor.getRootScopeDescriptor()
      })) != null ? ref : true;
    };

    WrapGuideElement.prototype.hide = function() {
      return this.element.style.display = 'none';
    };

    WrapGuideElement.prototype.show = function() {
      return this.element.style.display = 'block';
    };

    WrapGuideElement.prototype.updateGuide = function() {
      if (this.isEnabled()) {
        return this.updateGuides();
      } else {
        return this.hide();
      }
    };

    WrapGuideElement.prototype.updateGuides = function() {
      this.removeGuides();
      this.appendGuides();
      if (this.element.children.length) {
        return this.show();
      } else {
        return this.hide();
      }
    };

    WrapGuideElement.prototype.destroy = function() {
      this.element.remove();
      this.subscriptions.dispose();
      return this.configSubscriptions.dispose();
    };

    WrapGuideElement.prototype.removeGuides = function() {
      var results;
      results = [];
      while (this.element.firstChild) {
        results.push(this.element.removeChild(this.element.firstChild));
      }
      return results;
    };

    WrapGuideElement.prototype.appendGuides = function() {
      var column, columns, j, len, results;
      columns = this.getGuidesColumns(this.editor.getPath(), this.editor.getGrammar().scopeName);
      results = [];
      for (j = 0, len = columns.length; j < len; j++) {
        column = columns[j];
        if (!(column < 0)) {
          results.push(this.appendGuide(column));
        } else {
          results.push(void 0);
        }
      }
      return results;
    };

    WrapGuideElement.prototype.appendGuide = function(column) {
      var columnWidth, guide;
      columnWidth = this.editorElement.getDefaultCharacterWidth() * column;
      columnWidth -= this.editorElement.getScrollLeft();
      guide = document.createElement('div');
      guide.classList.add('wrap-guide');
      guide.style.left = (Math.round(columnWidth)) + "px";
      return this.element.appendChild(guide);
    };

    return WrapGuideElement;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
