(function() {
  var BufferedProcess, Client, CompositeDisposable, Emitter, PackageManager, _, createJsonParseError, createProcessError, handleProcessErrors, ref, semver,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  _ = require('underscore-plus');

  ref = require('atom'), BufferedProcess = ref.BufferedProcess, CompositeDisposable = ref.CompositeDisposable, Emitter = ref.Emitter;

  semver = require('semver');

  Client = require('./atom-io-client');

  module.exports = PackageManager = (function() {
    PackageManager.prototype.CACHE_EXPIRY = 1000 * 60 * 10;

    function PackageManager() {
      this.setProxyServersAsync = bind(this.setProxyServersAsync, this);
      this.setProxyServers = bind(this.setProxyServers, this);
      this.packagePromises = [];
      this.apmCache = {
        loadOutdated: {
          value: null,
          expiry: 0
        }
      };
      this.emitter = new Emitter;
    }

    PackageManager.prototype.getClient = function() {
      return this.client != null ? this.client : this.client = new Client(this);
    };

    PackageManager.prototype.isPackageInstalled = function(packageName) {
      if (atom.packages.isPackageLoaded(packageName)) {
        return true;
      } else {
        return atom.packages.getAvailablePackageNames().indexOf(packageName) > -1;
      }
    };

    PackageManager.prototype.packageHasSettings = function(packageName) {
      var grammar, grammars, i, len, pack, ref1, schema;
      grammars = (ref1 = atom.grammars.getGrammars()) != null ? ref1 : [];
      for (i = 0, len = grammars.length; i < len; i++) {
        grammar = grammars[i];
        if (grammar.path) {
          if (grammar.packageName === packageName) {
            return true;
          }
        }
      }
      pack = atom.packages.getLoadedPackage(packageName);
      if ((pack != null) && !atom.packages.isPackageActive(packageName)) {
        pack.activateConfig();
      }
      schema = atom.config.getSchema(packageName);
      return (schema != null) && (schema.type !== 'any');
    };

    PackageManager.prototype.setProxyServers = function(callback) {
      var session;
      session = atom.getCurrentWindow().webContents.session;
      return session.resolveProxy('http://atom.io', (function(_this) {
        return function(httpProxy) {
          _this.applyProxyToEnv('http_proxy', httpProxy);
          return session.resolveProxy('https://atom.io', function(httpsProxy) {
            _this.applyProxyToEnv('https_proxy', httpsProxy);
            return callback();
          });
        };
      })(this));
    };

    PackageManager.prototype.setProxyServersAsync = function(callback) {
      var httpProxyPromise, httpsProxyPromise;
      httpProxyPromise = atom.resolveProxy('http://atom.io').then((function(_this) {
        return function(proxy) {
          return _this.applyProxyToEnv('http_proxy', proxy);
        };
      })(this));
      httpsProxyPromise = atom.resolveProxy('https://atom.io').then((function(_this) {
        return function(proxy) {
          return _this.applyProxyToEnv('https_proxy', proxy);
        };
      })(this));
      return Promise.all([httpProxyPromise, httpsProxyPromise]).then(callback);
    };

    PackageManager.prototype.applyProxyToEnv = function(envName, proxy) {
      if (proxy != null) {
        proxy = proxy.split(' ');
        switch (proxy[0].trim().toUpperCase()) {
          case 'DIRECT':
            delete process.env[envName];
            break;
          case 'PROXY':
            process.env[envName] = 'http://' + proxy[1];
        }
      }
    };

    PackageManager.prototype.runCommand = function(args, callback) {
      var bufferedProcess, command, errorLines, exit, outputLines, stderr, stdout;
      command = atom.packages.getApmPath();
      outputLines = [];
      stdout = function(lines) {
        return outputLines.push(lines);
      };
      errorLines = [];
      stderr = function(lines) {
        return errorLines.push(lines);
      };
      exit = function(code) {
        return callback(code, outputLines.join('\n'), errorLines.join('\n'));
      };
      args.push('--no-color');
      if (atom.config.get('core.useProxySettingsWhenCallingApm')) {
        bufferedProcess = new BufferedProcess({
          command: command,
          args: args,
          stdout: stdout,
          stderr: stderr,
          exit: exit,
          autoStart: false
        });
        if (atom.resolveProxy != null) {
          this.setProxyServersAsync(function() {
            return bufferedProcess.start();
          });
        } else {
          this.setProxyServers(function() {
            return bufferedProcess.start();
          });
        }
        return bufferedProcess;
      } else {
        return new BufferedProcess({
          command: command,
          args: args,
          stdout: stdout,
          stderr: stderr,
          exit: exit
        });
      }
    };

    PackageManager.prototype.loadInstalled = function(callback) {
      var apmProcess, args, errorMessage;
      args = ['ls', '--json'];
      errorMessage = 'Fetching local packages failed.';
      apmProcess = this.runCommand(args, function(code, stdout, stderr) {
        var error, packages, parseError, ref1;
        if (code === 0) {
          try {
            packages = (ref1 = JSON.parse(stdout)) != null ? ref1 : [];
          } catch (error1) {
            parseError = error1;
            error = createJsonParseError(errorMessage, parseError, stdout);
            return callback(error);
          }
          return callback(null, packages);
        } else {
          error = new Error(errorMessage);
          error.stdout = stdout;
          error.stderr = stderr;
          return callback(error);
        }
      });
      return handleProcessErrors(apmProcess, errorMessage, callback);
    };

    PackageManager.prototype.loadFeatured = function(loadThemes, callback) {
      var apmProcess, args, errorMessage, version;
      if (!callback) {
        callback = loadThemes;
        loadThemes = false;
      }
      args = ['featured', '--json'];
      version = atom.getVersion();
      if (loadThemes) {
        args.push('--themes');
      }
      if (semver.valid(version)) {
        args.push('--compatible', version);
      }
      errorMessage = 'Fetching featured packages failed.';
      apmProcess = this.runCommand(args, function(code, stdout, stderr) {
        var error, packages, parseError, ref1;
        if (code === 0) {
          try {
            packages = (ref1 = JSON.parse(stdout)) != null ? ref1 : [];
          } catch (error1) {
            parseError = error1;
            error = createJsonParseError(errorMessage, parseError, stdout);
            return callback(error);
          }
          return callback(null, packages);
        } else {
          error = new Error(errorMessage);
          error.stdout = stdout;
          error.stderr = stderr;
          return callback(error);
        }
      });
      return handleProcessErrors(apmProcess, errorMessage, callback);
    };

    PackageManager.prototype.loadOutdated = function(clearCache, callback) {
      var apmProcess, args, errorMessage, version;
      if (clearCache) {
        this.clearOutdatedCache();
      } else if (this.apmCache.loadOutdated.value && this.apmCache.loadOutdated.expiry > Date.now()) {
        return callback(null, this.apmCache.loadOutdated.value);
      }
      args = ['outdated', '--json'];
      version = atom.getVersion();
      if (semver.valid(version)) {
        args.push('--compatible', version);
      }
      errorMessage = 'Fetching outdated packages and themes failed.';
      apmProcess = this.runCommand(args, (function(_this) {
        return function(code, stdout, stderr) {
          var error, i, len, pack, packages, parseError, ref1, updatablePackages;
          if (code === 0) {
            try {
              packages = (ref1 = JSON.parse(stdout)) != null ? ref1 : [];
            } catch (error1) {
              parseError = error1;
              error = createJsonParseError(errorMessage, parseError, stdout);
              return callback(error);
            }
            updatablePackages = (function() {
              var i, len, results;
              results = [];
              for (i = 0, len = packages.length; i < len; i++) {
                pack = packages[i];
                if (!this.getVersionPinnedPackages().includes(pack != null ? pack.name : void 0)) {
                  results.push(pack);
                }
              }
              return results;
            }).call(_this);
            _this.apmCache.loadOutdated = {
              value: updatablePackages,
              expiry: Date.now() + _this.CACHE_EXPIRY
            };
            for (i = 0, len = updatablePackages.length; i < len; i++) {
              pack = updatablePackages[i];
              _this.emitPackageEvent('update-available', pack);
            }
            return callback(null, updatablePackages);
          } else {
            error = new Error(errorMessage);
            error.stdout = stdout;
            error.stderr = stderr;
            return callback(error);
          }
        };
      })(this));
      return handleProcessErrors(apmProcess, errorMessage, callback);
    };

    PackageManager.prototype.getVersionPinnedPackages = function() {
      var ref1;
      return (ref1 = atom.config.get('core.versionPinnedPackages')) != null ? ref1 : [];
    };

    PackageManager.prototype.clearOutdatedCache = function() {
      return this.apmCache.loadOutdated = {
        value: null,
        expiry: 0
      };
    };

    PackageManager.prototype.loadPackage = function(packageName, callback) {
      var apmProcess, args, errorMessage;
      args = ['view', packageName, '--json'];
      errorMessage = "Fetching package '" + packageName + "' failed.";
      apmProcess = this.runCommand(args, function(code, stdout, stderr) {
        var error, packages, parseError, ref1;
        if (code === 0) {
          try {
            packages = (ref1 = JSON.parse(stdout)) != null ? ref1 : [];
          } catch (error1) {
            parseError = error1;
            error = createJsonParseError(errorMessage, parseError, stdout);
            return callback(error);
          }
          return callback(null, packages);
        } else {
          error = new Error(errorMessage);
          error.stdout = stdout;
          error.stderr = stderr;
          return callback(error);
        }
      });
      return handleProcessErrors(apmProcess, errorMessage, callback);
    };

    PackageManager.prototype.loadCompatiblePackageVersion = function(packageName, callback) {
      var apmProcess, args, errorMessage;
      args = ['view', packageName, '--json', '--compatible', this.normalizeVersion(atom.getVersion())];
      errorMessage = "Fetching package '" + packageName + "' failed.";
      apmProcess = this.runCommand(args, function(code, stdout, stderr) {
        var error, packages, parseError, ref1;
        if (code === 0) {
          try {
            packages = (ref1 = JSON.parse(stdout)) != null ? ref1 : [];
          } catch (error1) {
            parseError = error1;
            error = createJsonParseError(errorMessage, parseError, stdout);
            return callback(error);
          }
          return callback(null, packages);
        } else {
          error = new Error(errorMessage);
          error.stdout = stdout;
          error.stderr = stderr;
          return callback(error);
        }
      });
      return handleProcessErrors(apmProcess, errorMessage, callback);
    };

    PackageManager.prototype.getInstalled = function() {
      return new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.loadInstalled(function(error, result) {
            if (error) {
              return reject(error);
            } else {
              return resolve(result);
            }
          });
        };
      })(this));
    };

    PackageManager.prototype.getFeatured = function(loadThemes) {
      return new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.loadFeatured(!!loadThemes, function(error, result) {
            if (error) {
              return reject(error);
            } else {
              return resolve(result);
            }
          });
        };
      })(this));
    };

    PackageManager.prototype.getOutdated = function(clearCache) {
      if (clearCache == null) {
        clearCache = false;
      }
      return new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.loadOutdated(clearCache, function(error, result) {
            if (error) {
              return reject(error);
            } else {
              return resolve(result);
            }
          });
        };
      })(this));
    };

    PackageManager.prototype.getPackage = function(packageName) {
      var base;
      return (base = this.packagePromises)[packageName] != null ? base[packageName] : base[packageName] = new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.loadPackage(packageName, function(error, result) {
            if (error) {
              return reject(error);
            } else {
              return resolve(result);
            }
          });
        };
      })(this));
    };

    PackageManager.prototype.satisfiesVersion = function(version, metadata) {
      var engine, ref1, ref2;
      engine = (ref1 = (ref2 = metadata.engines) != null ? ref2.atom : void 0) != null ? ref1 : '*';
      if (!semver.validRange(engine)) {
        return false;
      }
      return semver.satisfies(version, engine);
    };

    PackageManager.prototype.normalizeVersion = function(version) {
      if (typeof version === 'string') {
        version = version.split('-')[0];
      }
      return version;
    };

    PackageManager.prototype.update = function(pack, newVersion, callback) {
      var apmInstallSource, apmProcess, args, errorMessage, exit, name, onError, theme;
      name = pack.name, theme = pack.theme, apmInstallSource = pack.apmInstallSource;
      errorMessage = newVersion ? "Updating to \u201C" + name + "@" + newVersion + "\u201D failed." : "Updating to latest sha failed.";
      onError = (function(_this) {
        return function(error) {
          error.packageInstallError = !theme;
          _this.emitPackageEvent('update-failed', pack, error);
          return typeof callback === "function" ? callback(error) : void 0;
        };
      })(this);
      if ((apmInstallSource != null ? apmInstallSource.type : void 0) === 'git') {
        args = ['install', apmInstallSource.source];
      } else {
        args = ['install', name + "@" + newVersion];
      }
      exit = (function(_this) {
        return function(code, stdout, stderr) {
          var error;
          if (code === 0) {
            _this.clearOutdatedCache();
            if (typeof callback === "function") {
              callback();
            }
            return _this.emitPackageEvent('updated', pack);
          } else {
            error = new Error(errorMessage);
            error.stdout = stdout;
            error.stderr = stderr;
            return onError(error);
          }
        };
      })(this);
      this.emitPackageEvent('updating', pack);
      apmProcess = this.runCommand(args, exit);
      return handleProcessErrors(apmProcess, errorMessage, onError);
    };

    PackageManager.prototype.unload = function(name) {
      if (atom.packages.isPackageLoaded(name)) {
        if (atom.packages.isPackageActive(name)) {
          atom.packages.deactivatePackage(name);
        }
        return atom.packages.unloadPackage(name);
      }
    };

    PackageManager.prototype.install = function(pack, callback) {
      var activateOnFailure, activateOnSuccess, apmProcess, args, errorMessage, exit, name, nameWithVersion, onError, theme, version;
      name = pack.name, version = pack.version, theme = pack.theme;
      activateOnSuccess = !theme && !atom.packages.isPackageDisabled(name);
      activateOnFailure = atom.packages.isPackageActive(name);
      nameWithVersion = version != null ? name + "@" + version : name;
      this.unload(name);
      args = ['install', nameWithVersion, '--json'];
      errorMessage = "Installing \u201C" + nameWithVersion + "\u201D failed.";
      onError = (function(_this) {
        return function(error) {
          error.packageInstallError = !theme;
          _this.emitPackageEvent('install-failed', pack, error);
          return typeof callback === "function" ? callback(error) : void 0;
        };
      })(this);
      exit = (function(_this) {
        return function(code, stdout, stderr) {
          var err, error, packageInfo;
          if (code === 0) {
            try {
              packageInfo = JSON.parse(stdout)[0];
              pack = _.extend({}, pack, packageInfo.metadata);
              name = pack.name;
            } catch (error1) {
              err = error1;
            }
            _this.clearOutdatedCache();
            if (activateOnSuccess) {
              atom.packages.activatePackage(name);
            } else {
              atom.packages.loadPackage(name);
            }
            if (typeof callback === "function") {
              callback();
            }
            return _this.emitPackageEvent('installed', pack);
          } else {
            if (activateOnFailure) {
              atom.packages.activatePackage(name);
            }
            error = new Error(errorMessage);
            error.stdout = stdout;
            error.stderr = stderr;
            return onError(error);
          }
        };
      })(this);
      this.emitPackageEvent('installing', pack);
      apmProcess = this.runCommand(args, exit);
      return handleProcessErrors(apmProcess, errorMessage, onError);
    };

    PackageManager.prototype.uninstall = function(pack, callback) {
      var apmProcess, errorMessage, name, onError;
      name = pack.name;
      if (atom.packages.isPackageActive(name)) {
        atom.packages.deactivatePackage(name);
      }
      errorMessage = "Uninstalling \u201C" + name + "\u201D failed.";
      onError = (function(_this) {
        return function(error) {
          _this.emitPackageEvent('uninstall-failed', pack, error);
          return typeof callback === "function" ? callback(error) : void 0;
        };
      })(this);
      this.emitPackageEvent('uninstalling', pack);
      apmProcess = this.runCommand(['uninstall', '--hard', name], (function(_this) {
        return function(code, stdout, stderr) {
          var error;
          if (code === 0) {
            _this.clearOutdatedCache();
            _this.unload(name);
            _this.removePackageNameFromDisabledPackages(name);
            if (typeof callback === "function") {
              callback();
            }
            return _this.emitPackageEvent('uninstalled', pack);
          } else {
            error = new Error(errorMessage);
            error.stdout = stdout;
            error.stderr = stderr;
            return onError(error);
          }
        };
      })(this));
      return handleProcessErrors(apmProcess, errorMessage, onError);
    };

    PackageManager.prototype.installAlternative = function(pack, alternativePackageName, callback) {
      var eventArg, installPromise, uninstallPromise;
      eventArg = {
        pack: pack,
        alternative: alternativePackageName
      };
      this.emitter.emit('package-installing-alternative', eventArg);
      uninstallPromise = new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.uninstall(pack, function(error) {
            if (error) {
              return reject(error);
            } else {
              return resolve();
            }
          });
        };
      })(this));
      installPromise = new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.install({
            name: alternativePackageName
          }, function(error) {
            if (error) {
              return reject(error);
            } else {
              return resolve();
            }
          });
        };
      })(this));
      return Promise.all([uninstallPromise, installPromise]).then((function(_this) {
        return function() {
          callback(null, eventArg);
          return _this.emitter.emit('package-installed-alternative', eventArg);
        };
      })(this))["catch"]((function(_this) {
        return function(error) {
          console.error(error.message, error.stack);
          callback(error, eventArg);
          eventArg.error = error;
          return _this.emitter.emit('package-install-alternative-failed', eventArg);
        };
      })(this));
    };

    PackageManager.prototype.canUpgrade = function(installedPackage, availableVersion) {
      var installedVersion;
      if (installedPackage == null) {
        return false;
      }
      installedVersion = installedPackage.metadata.version;
      if (!semver.valid(installedVersion)) {
        return false;
      }
      if (!semver.valid(availableVersion)) {
        return false;
      }
      return semver.gt(availableVersion, installedVersion);
    };

    PackageManager.prototype.getPackageTitle = function(arg) {
      var name;
      name = arg.name;
      return _.undasherize(_.uncamelcase(name));
    };

    PackageManager.prototype.getRepositoryUrl = function(arg) {
      var metadata, ref1, ref2, repoName, repoUrl, repository;
      metadata = arg.metadata;
      repository = metadata.repository;
      repoUrl = (ref1 = (ref2 = repository != null ? repository.url : void 0) != null ? ref2 : repository) != null ? ref1 : '';
      if (repoUrl.match('git@github')) {
        repoName = repoUrl.split(':')[1];
        repoUrl = "https://github.com/" + repoName;
      }
      return repoUrl.replace(/\.git$/, '').replace(/\/+$/, '').replace(/^git\+/, '');
    };

    PackageManager.prototype.getRepositoryBugUri = function(arg) {
      var bugUri, bugs, metadata, ref1, ref2;
      metadata = arg.metadata;
      bugs = metadata.bugs;
      if (typeof bugs === 'string') {
        bugUri = bugs;
      } else {
        bugUri = (ref1 = (ref2 = bugs != null ? bugs.url : void 0) != null ? ref2 : bugs != null ? bugs.email : void 0) != null ? ref1 : this.getRepositoryUrl({
          metadata: metadata
        }) + '/issues/new';
        if (bugUri.includes('@')) {
          bugUri = 'mailto:' + bugUri;
        }
      }
      return bugUri;
    };

    PackageManager.prototype.checkNativeBuildTools = function() {
      return new Promise((function(_this) {
        return function(resolve, reject) {
          var apmProcess;
          apmProcess = _this.runCommand(['install', '--check'], function(code, stdout, stderr) {
            if (code === 0) {
              return resolve();
            } else {
              return reject(new Error());
            }
          });
          return apmProcess.onWillThrowError(function(arg) {
            var error, handle;
            error = arg.error, handle = arg.handle;
            handle();
            return reject(error);
          });
        };
      })(this));
    };

    PackageManager.prototype.removePackageNameFromDisabledPackages = function(packageName) {
      return atom.config.removeAtKeyPath('core.disabledPackages', packageName);
    };

    PackageManager.prototype.emitPackageEvent = function(eventName, pack, error) {
      var ref1, ref2, theme;
      theme = (ref1 = pack.theme) != null ? ref1 : (ref2 = pack.metadata) != null ? ref2.theme : void 0;
      eventName = theme ? "theme-" + eventName : "package-" + eventName;
      return this.emitter.emit(eventName, {
        pack: pack,
        error: error
      });
    };

    PackageManager.prototype.on = function(selectors, callback) {
      var i, len, ref1, selector, subscriptions;
      subscriptions = new CompositeDisposable;
      ref1 = selectors.split(" ");
      for (i = 0, len = ref1.length; i < len; i++) {
        selector = ref1[i];
        subscriptions.add(this.emitter.on(selector, callback));
      }
      return subscriptions;
    };

    return PackageManager;

  })();

  createJsonParseError = function(message, parseError, stdout) {
    var error;
    error = new Error(message);
    error.stdout = '';
    error.stderr = parseError.message + ": " + stdout;
    return error;
  };

  createProcessError = function(message, processError) {
    var error;
    error = new Error(message);
    error.stdout = '';
    error.stderr = processError.message;
    return error;
  };

  handleProcessErrors = function(apmProcess, message, callback) {
    return apmProcess.onWillThrowError(function(arg) {
      var error, handle;
      error = arg.error, handle = arg.handle;
      handle();
      return callback(createProcessError(message, error));
    });
  };

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
