(function() {
  var AtomIoClient, fs, glob, path, remote, request;

  fs = require('fs-plus');

  path = require('path');

  remote = require('electron').remote;

  glob = require('glob');

  request = require('request');

  module.exports = AtomIoClient = (function() {
    function AtomIoClient(packageManager, baseURL) {
      this.packageManager = packageManager;
      this.baseURL = baseURL;
      if (this.baseURL == null) {
        this.baseURL = 'https://atom.io/api/';
      }
      this.expiry = 1000 * 60 * 60 * 12;
      this.createAvatarCache();
      this.expireAvatarCache();
    }

    AtomIoClient.prototype.avatar = function(login, callback) {
      return this.cachedAvatar(login, (function(_this) {
        return function(err, cached) {
          var stale;
          if (cached) {
            stale = Date.now() - parseInt(cached.split('-').pop()) > _this.expiry;
          }
          if (cached && (!stale || !_this.online())) {
            return callback(null, cached);
          } else {
            return _this.fetchAndCacheAvatar(login, callback);
          }
        };
      })(this));
    };

    AtomIoClient.prototype["package"] = function(name, callback) {
      var data, packagePath;
      packagePath = "packages/" + name;
      data = this.fetchFromCache(packagePath);
      if (data) {
        return callback(null, data);
      } else {
        return this.request(packagePath, callback);
      }
    };

    AtomIoClient.prototype.featuredPackages = function(callback) {
      var data;
      data = this.fetchFromCache('packages/featured');
      if (data) {
        return callback(null, data);
      } else {
        return this.getFeatured(false, callback);
      }
    };

    AtomIoClient.prototype.featuredThemes = function(callback) {
      var data;
      data = this.fetchFromCache('themes/featured');
      if (data) {
        return callback(null, data);
      } else {
        return this.getFeatured(true, callback);
      }
    };

    AtomIoClient.prototype.getFeatured = function(loadThemes, callback) {
      return this.packageManager.getFeatured(loadThemes).then((function(_this) {
        return function(packages) {
          var cached, key;
          key = loadThemes ? 'themes/featured' : 'packages/featured';
          cached = {
            data: packages,
            createdOn: Date.now()
          };
          localStorage.setItem(_this.cacheKeyForPath(key), JSON.stringify(cached));
          return callback(null, packages);
        };
      })(this))["catch"](function(error) {
        return callback(error, null);
      });
    };

    AtomIoClient.prototype.request = function(path, callback) {
      var options;
      options = {
        url: "" + this.baseURL + path,
        headers: {
          'User-Agent': navigator.userAgent
        },
        gzip: true
      };
      return request(options, (function(_this) {
        return function(err, res, body) {
          var cached, error;
          if (err) {
            return callback(err);
          }
          try {
            body = _this.parseJSON(body);
            delete body.versions;
            cached = {
              data: body,
              createdOn: Date.now()
            };
            localStorage.setItem(_this.cacheKeyForPath(path), JSON.stringify(cached));
            return callback(err, cached.data);
          } catch (error1) {
            error = error1;
            return callback(error);
          }
        };
      })(this));
    };

    AtomIoClient.prototype.cacheKeyForPath = function(path) {
      return "settings-view:" + path;
    };

    AtomIoClient.prototype.online = function() {
      return navigator.onLine;
    };

    AtomIoClient.prototype.fetchFromCache = function(packagePath) {
      var cached;
      cached = localStorage.getItem(this.cacheKeyForPath(packagePath));
      cached = cached ? this.parseJSON(cached) : void 0;
      if ((cached != null) && (!this.online() || Date.now() - cached.createdOn < this.expiry)) {
        return cached.data;
      } else {
        return null;
      }
    };

    AtomIoClient.prototype.createAvatarCache = function() {
      return fs.makeTree(this.getCachePath());
    };

    AtomIoClient.prototype.avatarPath = function(login) {
      return path.join(this.getCachePath(), login + "-" + (Date.now()));
    };

    AtomIoClient.prototype.cachedAvatar = function(login, callback) {
      return glob(this.avatarGlob(login), (function(_this) {
        return function(err, files) {
          var createdOn, filename, i, imagePath, len, ref;
          if (err) {
            return callback(err);
          }
          files.sort().reverse();
          for (i = 0, len = files.length; i < len; i++) {
            imagePath = files[i];
            filename = path.basename(imagePath);
            ref = filename.split('-'), createdOn = ref[ref.length - 1];
            if (Date.now() - parseInt(createdOn) < _this.expiry) {
              return callback(null, imagePath);
            }
          }
          return callback(null, null);
        };
      })(this));
    };

    AtomIoClient.prototype.avatarGlob = function(login) {
      return path.join(this.getCachePath(), login + "-*([0-9])");
    };

    AtomIoClient.prototype.fetchAndCacheAvatar = function(login, callback) {
      var imagePath, requestObject;
      if (!this.online()) {
        return callback(null, null);
      } else {
        imagePath = this.avatarPath(login);
        requestObject = {
          url: "https://avatars.githubusercontent.com/" + login,
          headers: {
            'User-Agent': navigator.userAgent
          }
        };
        return request.head(requestObject, function(error, response, body) {
          var writeStream;
          if ((error != null) || response.statusCode !== 200 || !response.headers['content-type'].startsWith('image/')) {
            return callback(error);
          } else {
            writeStream = fs.createWriteStream(imagePath);
            writeStream.on('finish', function() {
              return callback(null, imagePath);
            });
            writeStream.on('error', function(error) {
              writeStream.close();
              try {
                if (fs.existsSync(imagePath)) {
                  fs.unlinkSync(imagePath);
                }
              } catch (error1) {}
              return callback(error);
            });
            return request(requestObject).pipe(writeStream);
          }
        });
      }
    };

    AtomIoClient.prototype.expireAvatarCache = function() {
      var deleteAvatar;
      deleteAvatar = (function(_this) {
        return function(child) {
          var avatarPath;
          avatarPath = path.join(_this.getCachePath(), child);
          return fs.unlink(avatarPath, function(error) {
            if (error && error.code !== 'ENOENT') {
              return console.warn("Error deleting avatar (" + error.code + "): " + avatarPath);
            }
          });
        };
      })(this);
      return fs.readdir(this.getCachePath(), function(error, _files) {
        var children, filename, files, i, key, len, parts, results, stamp;
        if (_files == null) {
          _files = [];
        }
        files = {};
        for (i = 0, len = _files.length; i < len; i++) {
          filename = _files[i];
          parts = filename.split('-');
          stamp = parts.pop();
          key = parts.join('-');
          if (files[key] == null) {
            files[key] = [];
          }
          files[key].push(key + "-" + stamp);
        }
        results = [];
        for (key in files) {
          children = files[key];
          children.sort();
          children.pop();
          results.push(children.forEach(deleteAvatar));
        }
        return results;
      });
    };

    AtomIoClient.prototype.getCachePath = function() {
      return this.cachePath != null ? this.cachePath : this.cachePath = path.join(remote.app.getPath('userData'), 'Cache', 'settings-view');
    };

    AtomIoClient.prototype.search = function(query, options) {
      var qs;
      qs = {
        q: query
      };
      if (options.themes) {
        qs.filter = 'theme';
      } else if (options.packages) {
        qs.filter = 'package';
      }
      options = {
        url: this.baseURL + "packages/search",
        headers: {
          'User-Agent': navigator.userAgent
        },
        qs: qs,
        gzip: true
      };
      return new Promise((function(_this) {
        return function(resolve, reject) {
          return request(options, function(err, res, body) {
            var e, error;
            if (err) {
              error = new Error("Searching for \u201C" + query + "\u201D failed.");
              error.stderr = err.message;
              return reject(error);
            } else {
              try {
                body = _this.parseJSON(body);
                return resolve(body.filter(function(pkg) {
                  var ref;
                  return ((ref = pkg.releases) != null ? ref.latest : void 0) != null;
                }).map(function(arg) {
                  var downloads, metadata, readme, repository, stargazers_count;
                  readme = arg.readme, metadata = arg.metadata, downloads = arg.downloads, stargazers_count = arg.stargazers_count, repository = arg.repository;
                  return Object.assign(metadata, {
                    readme: readme,
                    downloads: downloads,
                    stargazers_count: stargazers_count,
                    repository: repository.url
                  });
                }));
              } catch (error1) {
                e = error1;
                error = new Error("Searching for \u201C" + query + "\u201D failed.");
                error.stderr = e.message + '\n' + body;
                return reject(error);
              }
            }
          });
        };
      })(this));
    };

    AtomIoClient.prototype.parseJSON = function(s) {
      return JSON.parse(s);
    };

    return AtomIoClient;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
