/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 * 
 * @format
 */
'use strict';

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

var _extends2 = _interopRequireDefault(require("@babel/runtime/helpers/extends"));

var _objectWithoutPropertiesLoose2 = _interopRequireDefault(require("@babel/runtime/helpers/objectWithoutPropertiesLoose"));

var _objectSpread2 = _interopRequireDefault(require("@babel/runtime/helpers/objectSpread"));

var _inheritsLoose2 = _interopRequireDefault(require("@babel/runtime/helpers/inheritsLoose"));

var _assertThisInitialized2 = _interopRequireDefault(require("@babel/runtime/helpers/assertThisInitialized"));

var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));

var React = require("react");

var ReactRelayContext = require("./ReactRelayContext");

var ReactRelayQueryFetcher = require("./ReactRelayQueryFetcher");

var areEqual = require("fbjs/lib/areEqual");

var buildReactRelayContainer = require("./buildReactRelayContainer");

var warning = require("fbjs/lib/warning");

var _require = require("./ReactRelayContainerProfiler"),
    profileContainer = _require.profileContainer;

var _require2 = require("./ReactRelayContainerUtils"),
    getContainerName = _require2.getContainerName;

var _require3 = require("./RelayContext"),
    assertRelayContext = _require3.assertRelayContext;

var _require4 = require("relay-runtime"),
    Observable = _require4.Observable,
    RelayFeatureFlags = _require4.RelayFeatureFlags,
    RelayProfiler = _require4.RelayProfiler,
    isScalarAndEqual = _require4.isScalarAndEqual;

/**
 * Composes a React component class, returning a new class that intercepts
 * props, resolving them with the provided fragments and subscribing for
 * updates.
 */
function createContainerWithFragments(Component, fragments, taggedNode) {
  var containerName = getContainerName(Component);

  var Container =
  /*#__PURE__*/
  function (_React$Component) {
    (0, _inheritsLoose2["default"])(Container, _React$Component);

    function Container(props) {
      var _this;

      _this = _React$Component.call(this, props) || this;
      (0, _defineProperty2["default"])((0, _assertThisInitialized2["default"])((0, _assertThisInitialized2["default"])(_this)), "_handleFragmentDataUpdate", function () {
        var profiler = RelayProfiler.profile('ReactRelayRefetchContainer.handleFragmentDataUpdate');
        var resolverFromThisUpdate = _this.state.resolver;

        _this.setState(function (updatedState) {
          // If this event belongs to the current data source, update.
          // Otherwise we should ignore it.
          if (resolverFromThisUpdate === updatedState.resolver) {
            return {
              data: updatedState.resolver.resolve()
            };
          }

          return null;
        }, profiler.stop);
      });
      (0, _defineProperty2["default"])((0, _assertThisInitialized2["default"])((0, _assertThisInitialized2["default"])(_this)), "_refetch", function (refetchVariables, renderVariables, observerOrCallback, options) {
        if (_this._isUnmounted) {
          process.env.NODE_ENV !== "production" ? warning(false, 'ReactRelayRefetchContainer: Unexpected call of `refetch` ' + 'on unmounted container `%s`. It looks like some instances ' + 'of your container still trying to refetch the data but they already ' + 'unmounted. Please make sure you clear all timers, intervals, async ' + 'calls, etc that may trigger `refetch`.', containerName) : void 0;
          return {
            dispose: function dispose() {}
          };
        }

        var _assertRelayContext = assertRelayContext(_this.props.__relayContext),
            environment = _assertRelayContext.environment,
            rootVariables = _assertRelayContext.variables;

        var fetchVariables = typeof refetchVariables === 'function' ? refetchVariables(_this._getFragmentVariables()) : refetchVariables;
        fetchVariables = (0, _objectSpread2["default"])({}, rootVariables, fetchVariables);
        var fragmentVariables = renderVariables ? RelayFeatureFlags.MERGE_FETCH_AND_FRAGMENT_VARS ? (0, _objectSpread2["default"])({}, fetchVariables, renderVariables) : (0, _objectSpread2["default"])({}, rootVariables, renderVariables) : fetchVariables;
        var cacheConfig = options ? {
          force: !!options.force
        } : undefined;
        var observer = typeof observerOrCallback === 'function' ? {
          // callback is not exectued on complete or unsubscribe
          // for backward compatibility
          next: observerOrCallback,
          error: observerOrCallback
        } : observerOrCallback || {};
        var _this$props$__relayCo = _this.props.__relayContext.environment.unstable_internal,
            createOperationDescriptor = _this$props$__relayCo.createOperationDescriptor,
            getRequest = _this$props$__relayCo.getRequest;
        var query = getRequest(taggedNode);
        var operation = createOperationDescriptor(query, fetchVariables); // TODO: T26288752 find a better way

        /* eslint-disable lint/react-state-props-mutation */

        _this.state.localVariables = fetchVariables;
        /* eslint-enable lint/react-state-props-mutation */
        // Cancel any previously running refetch.

        _this._refetchSubscription && _this._refetchSubscription.unsubscribe(); // Declare refetchSubscription before assigning it in .start(), since
        // synchronous completion may call callbacks .subscribe() returns.

        var refetchSubscription;

        if ((options === null || options === void 0 ? void 0 : options.fetchPolicy) === 'store-or-network') {
          var storeSnapshot = _this._getQueryFetcher().lookupInStore(environment, operation);

          if (storeSnapshot != null) {
            _this.state.resolver.setVariables(fragmentVariables, operation.node);

            _this.setState(function (latestState) {
              return {
                data: latestState.resolver.resolve(),
                contextForChildren: {
                  environment: _this.props.__relayContext.environment,
                  variables: fragmentVariables
                }
              };
            }, function () {
              observer.next && observer.next();
              observer.complete && observer.complete();
            });

            return {
              dispose: function dispose() {}
            };
          }
        }

        _this._getQueryFetcher().execute({
          environment: environment,
          operation: operation,
          cacheConfig: cacheConfig,
          // TODO (T26430099): Cleanup old references
          preservePreviousReferences: true
        }).mergeMap(function (response) {
          _this.state.resolver.setVariables(fragmentVariables, operation.node);

          return Observable.create(function (sink) {
            return _this.setState(function (latestState) {
              return {
                data: latestState.resolver.resolve(),
                contextForChildren: {
                  environment: _this.props.__relayContext.environment,
                  variables: fragmentVariables
                }
              };
            }, function () {
              sink.next();
              sink.complete();
            });
          });
        })["finally"](function () {
          // Finalizing a refetch should only clear this._refetchSubscription
          // if the finizing subscription is the most recent call.
          if (_this._refetchSubscription === refetchSubscription) {
            _this._refetchSubscription = null;
          }
        }).subscribe((0, _objectSpread2["default"])({}, observer, {
          start: function start(subscription) {
            _this._refetchSubscription = refetchSubscription = subscription;
            observer.start && observer.start(subscription);
          }
        }));

        return {
          dispose: function dispose() {
            refetchSubscription && refetchSubscription.unsubscribe();
          }
        };
      });
      var relayContext = assertRelayContext(props.__relayContext);
      _this._refetchSubscription = null;
      var createFragmentSpecResolver = relayContext.environment.unstable_internal.createFragmentSpecResolver; // Do not provide a subscription/callback here.
      // It is possible for this render to be interrupted or aborted,
      // In which case the subscription would cause a leak.
      // We will add the subscription in componentDidMount().

      var resolver = createFragmentSpecResolver(relayContext, containerName, fragments, props);
      _this.state = {
        data: resolver.resolve(),
        localVariables: null,
        prevProps: props,
        prevPropsContext: relayContext,
        contextForChildren: relayContext,
        relayProp: getRelayProp(relayContext.environment, _this._refetch),
        resolver: resolver
      };
      _this._isUnmounted = false;
      return _this;
    }

    var _proto = Container.prototype;

    _proto.componentDidMount = function componentDidMount() {
      this._subscribeToNewResolver();
    };

    _proto.componentDidUpdate = function componentDidUpdate(prevProps, prevState) {
      // If the environment has changed or props point to new records then
      // previously fetched data and any pending fetches no longer apply:
      // - Existing references are on the old environment.
      // - Existing references are based on old variables.
      // - Pending fetches are for the previous records.
      if (this.state.resolver !== prevState.resolver) {
        prevState.resolver.dispose();
        this._queryFetcher && this._queryFetcher.dispose();
        this._refetchSubscription && this._refetchSubscription.unsubscribe();

        this._subscribeToNewResolver();
      }
    };
    /**
     * When new props are received, read data for the new props and add it to
     * state. Props may be the same in which case previous data can be reused.
     */


    Container.getDerivedStateFromProps = function getDerivedStateFromProps(nextProps, prevState) {
      // Any props change could impact the query, so we mirror props in state.
      // This is an unusual pattern, but necessary for this container usecase.
      var prevProps = prevState.prevProps;
      var relayContext = assertRelayContext(nextProps.__relayContext);
      var _relayContext$environ = relayContext.environment.unstable_internal,
          createFragmentSpecResolver = _relayContext$environ.createFragmentSpecResolver,
          getDataIDsFromObject = _relayContext$environ.getDataIDsFromObject;
      var prevIDs = getDataIDsFromObject(fragments, prevProps);
      var nextIDs = getDataIDsFromObject(fragments, nextProps);
      var resolver = prevState.resolver; // If the environment has changed or props point to new records then
      // previously fetched data and any pending fetches no longer apply:
      // - Existing references are on the old environment.
      // - Existing references are based on old variables.
      // - Pending fetches are for the previous records.

      if (prevState.prevPropsContext.environment !== relayContext.environment || prevState.prevPropsContext.variables !== relayContext.variables || !areEqual(prevIDs, nextIDs)) {
        // Do not provide a subscription/callback here.
        // It is possible for this render to be interrupted or aborted,
        // In which case the subscription would cause a leak.
        // We will add the subscription in componentDidUpdate().
        resolver = createFragmentSpecResolver(relayContext, containerName, fragments, nextProps);
        return {
          data: resolver.resolve(),
          localVariables: null,
          prevProps: nextProps,
          prevPropsContext: relayContext,
          contextForChildren: relayContext,
          relayProp: getRelayProp(relayContext.environment, prevState.relayProp.refetch),
          resolver: resolver
        };
      } else if (!prevState.localVariables) {
        resolver.setProps(nextProps);
      }

      var data = resolver.resolve();

      if (data !== prevState.data) {
        return {
          data: data,
          prevProps: nextProps
        };
      }

      return null;
    };

    _proto.componentWillUnmount = function componentWillUnmount() {
      this._isUnmounted = true;
      this.state.resolver.dispose();
      this._queryFetcher && this._queryFetcher.dispose();
      this._refetchSubscription && this._refetchSubscription.unsubscribe();
    };

    _proto.shouldComponentUpdate = function shouldComponentUpdate(nextProps, nextState) {
      // Short-circuit if any Relay-related data has changed
      if (nextState.data !== this.state.data || nextState.relayProp !== this.state.relayProp) {
        return true;
      } // Otherwise, for convenience short-circuit if all non-Relay props
      // are scalar and equal


      var keys = Object.keys(nextProps);

      for (var ii = 0; ii < keys.length; ii++) {
        var _key = keys[ii];

        if (_key === '__relayContext') {
          if (this.state.prevPropsContext.environment !== nextState.prevPropsContext.environment || this.state.prevPropsContext.variables !== nextState.prevPropsContext.variables) {
            return true;
          }
        } else {
          if (!fragments.hasOwnProperty(_key) && !isScalarAndEqual(nextProps[_key], this.props[_key])) {
            return true;
          }
        }
      }

      return false;
    };

    _proto._subscribeToNewResolver = function _subscribeToNewResolver() {
      var _this$state = this.state,
          data = _this$state.data,
          resolver = _this$state.resolver; // Event listeners are only safe to add during the commit phase,
      // So they won't leak if render is interrupted or errors.

      resolver.setCallback(this._handleFragmentDataUpdate); // External values could change between render and commit.
      // Check for this case, even though it requires an extra store read.

      var maybeNewData = resolver.resolve();

      if (data !== maybeNewData) {
        this.setState({
          data: maybeNewData
        });
      }
    };
    /**
     * Render new data for the existing props/context.
     */


    _proto._getFragmentVariables = function _getFragmentVariables() {
      var getVariablesFromObject = this.props.__relayContext.environment.unstable_internal.getVariablesFromObject;
      return getVariablesFromObject(this.props.__relayContext.variables, fragments, this.props);
    };

    _proto._getQueryFetcher = function _getQueryFetcher() {
      if (!this._queryFetcher) {
        this._queryFetcher = new ReactRelayQueryFetcher();
      }

      return this._queryFetcher;
    };

    _proto.render = function render() {
      var _this$props = this.props,
          componentRef = _this$props.componentRef,
          __relayContext = _this$props.__relayContext,
          props = (0, _objectWithoutPropertiesLoose2["default"])(_this$props, ["componentRef", "__relayContext"]);
      var _this$state2 = this.state,
          relayProp = _this$state2.relayProp,
          contextForChildren = _this$state2.contextForChildren;
      return React.createElement(ReactRelayContext.Provider, {
        value: contextForChildren
      }, React.createElement(Component, (0, _extends2["default"])({}, props, this.state.data, {
        ref: componentRef,
        relay: relayProp
      })));
    };

    return Container;
  }(React.Component);

  (0, _defineProperty2["default"])(Container, "displayName", containerName);
  profileContainer(Container, 'ReactRelayRefetchContainer');
  return Container;
}

function getRelayProp(environment, refetch) {
  return {
    environment: environment,
    refetch: refetch
  };
}
/**
 * Wrap the basic `createContainer()` function with logic to adapt to the
 * `context.relay.environment` in which it is rendered. Specifically, the
 * extraction of the environment-specific version of fragments in the
 * `fragmentSpec` is memoized once per environment, rather than once per
 * instance of the container constructed/rendered.
 */


function createContainer(Component, fragmentSpec, taggedNode) {
  return buildReactRelayContainer(Component, fragmentSpec, function (ComponentClass, fragments) {
    return createContainerWithFragments(ComponentClass, fragments, taggedNode);
  });
}

module.exports = {
  createContainer: createContainer,
  createContainerWithFragments: createContainerWithFragments
};