"use strict";

const qs = require('querystring');

const {
  remote,
  ipcRenderer: ipc
} = require('electron');

const {
  GitProcess
} = require('dugite');

class AverageTracker {
  constructor({
    limit
  } = {
    limit: 10
  }) {
    // for now this serves a dual purpose - # of values tracked AND # discarded prior to starting avg calculation
    this.limit = limit;
    this.sum = 0;
    this.values = [];
  }

  addValue(value) {
    if (this.values.length >= this.limit) {
      const discardedValue = this.values.shift();
      this.sum -= discardedValue;
    }

    this.values.push(value);
    this.sum += value;
  }

  getAverage() {
    if (this.enoughData()) {
      return this.sum / this.limit;
    } else {
      return null;
    }
  }

  getLimit() {
    return this.limit;
  }

  enoughData() {
    return this.values.length === this.limit;
  }

}

const query = qs.parse(window.location.search.substr(1));
const sourceWebContentsId = remote.getCurrentWindow().webContents.id;
const operationCountLimit = parseInt(query.operationCountLimit, 10);
const averageTracker = new AverageTracker({
  limit: operationCountLimit
});
const childPidsById = new Map();

const destroyRenderer = () => {
  if (!managerWebContents.isDestroyed()) {
    managerWebContents.removeListener('crashed', destroyRenderer);
    managerWebContents.removeListener('destroyed', destroyRenderer);
  }

  const win = remote.BrowserWindow.fromWebContents(remote.getCurrentWebContents());

  if (win && !win.isDestroyed()) {
    win.destroy();
  }
};

const managerWebContentsId = parseInt(query.managerWebContentsId, 10);
const managerWebContents = remote.webContents.fromId(managerWebContentsId);

if (managerWebContents && !managerWebContents.isDestroyed()) {
  managerWebContents.on('crashed', destroyRenderer);
  managerWebContents.on('destroyed', destroyRenderer);

  window.onbeforeunload = () => {
    managerWebContents.removeListener('crashed', destroyRenderer);
    managerWebContents.removeListener('destroyed', destroyRenderer);
  };
}

const channelName = query.channelName;
ipc.on(channelName, (event, {
  type,
  data
}) => {
  if (type === 'git-exec') {
    const {
      args,
      workingDir,
      options,
      id
    } = data;

    if (args) {
      document.getElementById('command').textContent = `git ${args.join(' ')}`;
    }

    options.processCallback = child => {
      childPidsById.set(id, child.pid);
      child.on('error', err => {
        event.sender.sendTo(managerWebContentsId, channelName, {
          sourceWebContentsId,
          type: 'git-spawn-error',
          data: {
            id,
            err
          }
        });
      });
      child.stdin.on('error', err => {
        event.sender.sendTo(managerWebContentsId, channelName, {
          sourceWebContentsId,
          type: 'git-stdin-error',
          data: {
            id,
            stdin: options.stdin,
            err
          }
        });
      });
    };

    const spawnStart = performance.now();
    GitProcess.exec(args, workingDir, options).then(({
      stdout,
      stderr,
      exitCode
    }) => {
      const timing = {
        spawnTime: spawnEnd - spawnStart,
        execTime: performance.now() - spawnEnd
      };
      childPidsById["delete"](id);
      event.sender.sendTo(managerWebContentsId, channelName, {
        sourceWebContentsId,
        type: 'git-data',
        data: {
          id,
          average: averageTracker.getAverage(),
          results: {
            stdout,
            stderr,
            exitCode,
            timing
          }
        }
      });
    }, err => {
      const timing = {
        spawnTime: spawnEnd - spawnStart,
        execTime: performance.now() - spawnEnd
      };
      childPidsById["delete"](id);
      event.sender.sendTo(managerWebContentsId, channelName, {
        sourceWebContentsId,
        type: 'git-data',
        data: {
          id,
          average: averageTracker.getAverage(),
          results: {
            stdout: err.stdout,
            stderr: err.stderr,
            exitCode: err.code,
            signal: err.signal,
            timing
          }
        }
      });
    });
    const spawnEnd = performance.now();
    averageTracker.addValue(spawnEnd - spawnStart); // TODO: consider using this to avoid duplicate write operations upon crashing.
    // For now we won't do this to avoid clogging up ipc channel
    // event.sender.sendTo(managerWebContentsId, channelName, {sourceWebContentsId, type: 'exec-started', data: {id}});

    if (averageTracker.enoughData() && averageTracker.getAverage() > 20) {
      event.sender.sendTo(managerWebContentsId, channelName, {
        type: 'slow-spawns'
      });
    }
  } else if (type === 'git-cancel') {
    const {
      id
    } = data;
    const childPid = childPidsById.get(id);

    if (childPid !== undefined) {
      require('tree-kill')(childPid, 'SIGINT', () => {
        event.sender.sendTo(managerWebContentsId, channelName, {
          sourceWebContentsId,
          type: 'git-cancelled',
          data: {
            id,
            childPid
          }
        });
      });

      childPidsById["delete"](id);
    }
  } else {
    throw new Error(`Could not identify type ${type}`);
  }
});
ipc.sendTo(managerWebContentsId, channelName, {
  sourceWebContentsId,
  type: 'renderer-ready',
  data: {
    pid: process.pid
  }
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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