"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.incrementCounter = incrementCounter;
exports.addTiming = addTiming;
exports.addEvent = addEvent;
exports.FakeReporter = exports.reporterProxy = exports.FIVE_MINUTES_IN_MILLISECONDS = void 0;

const pjson = require('../package.json');

const FIVE_MINUTES_IN_MILLISECONDS = 1000 * 60 * 5; // this class allows us to call reporter methods
// before the reporter is actually loaded, since we don't want to
// assume that the metrics package will load before the GitHub package.

exports.FIVE_MINUTES_IN_MILLISECONDS = FIVE_MINUTES_IN_MILLISECONDS;

class ReporterProxy {
  constructor() {
    this.reporter = null;
    this.events = [];
    this.timings = [];
    this.counters = [];
    this.gitHubPackageVersion = pjson.version;
    this.timeout = null;
  } // function that is called after the reporter is actually loaded, to
  // set the reporter and send any data that have accumulated while it was loading.


  setReporter(reporter) {
    this.reporter = reporter;
    this.events.forEach(customEvent => {
      this.reporter.addCustomEvent(customEvent.eventType, customEvent.event);
    });
    this.events = [];
    this.timings.forEach(timing => {
      this.reporter.addTiming(timing.eventType, timing.durationInMilliseconds, timing.metadata);
    });
    this.timings = [];
    this.counters.forEach(counterName => {
      this.reporter.incrementCounter(counterName);
    });
    this.counters = [];
  }

  incrementCounter(counterName) {
    if (this.reporter === null) {
      this.startTimer();
      this.counters.push(counterName);
      return;
    }

    this.reporter.incrementCounter(counterName);
  }

  addTiming(eventType, durationInMilliseconds, metadata = {}) {
    if (this.reporter === null) {
      this.startTimer();
      this.timings.push({
        eventType,
        durationInMilliseconds,
        metadata
      });
      return;
    }

    this.reporter.addTiming(eventType, durationInMilliseconds, metadata);
  }

  addEvent(eventType, event) {
    if (this.reporter === null) {
      this.startTimer();
      this.events.push({
        eventType,
        event
      });
      return;
    }

    this.reporter.addCustomEvent(eventType, event);
  }

  startTimer() {
    if (this.timeout !== null) {
      return;
    } // if for some reason a user disables the metrics package, we don't want to
    // just keep accumulating events in memory until the heat death of the universe.
    // Use a no-op class, clear all queues, move on with our lives.


    this.timeout = setTimeout(FIVE_MINUTES_IN_MILLISECONDS, () => {
      if (this.reporter === null) {
        this.setReporter(new FakeReporter());
        this.events = [];
        this.timings = [];
        this.counters = [];
      }
    });
  }

}

const reporterProxy = new ReporterProxy();
exports.reporterProxy = reporterProxy;

class FakeReporter {
  addCustomEvent() {}

  addTiming() {}

  incrementCounter() {}

}

exports.FakeReporter = FakeReporter;

function incrementCounter(counterName) {
  reporterProxy.incrementCounter(counterName);
}

function addTiming(eventType, durationInMilliseconds, metadata = {}) {
  metadata.gitHubPackageVersion = reporterProxy.gitHubPackageVersion;
  reporterProxy.addTiming(eventType, durationInMilliseconds, metadata);
}

function addEvent(eventType, event) {
  event.gitHubPackageVersion = reporterProxy.gitHubPackageVersion;
  reporterProxy.addEvent(eventType, event);
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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