"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.expectRelayQuery = expectRelayQuery;
exports.clearRelayExpectations = clearRelayExpectations;
exports["default"] = void 0;

var _util = _interopRequireDefault(require("util"));

var _relayRuntime = require("relay-runtime");

var _moment = _interopRequireDefault(require("moment"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

const LODASH_ISEQUAL = 'lodash.isequal';
let isEqual = null;
const relayEnvironmentPerURL = new Map();
const tokenPerURL = new Map();
const fetchPerURL = new Map();
const responsesByQuery = new Map();

function logRatelimitApi(headers) {
  const remaining = headers.get('x-ratelimit-remaining');
  const total = headers.get('x-ratelimit-limit');
  const resets = headers.get('x-ratelimit-reset');

  const resetsIn = _moment["default"].unix(parseInt(resets, 10)).from(); // eslint-disable-next-line no-console


  console.debug(`GitHub API Rate Limit: ${remaining}/${total} — resets ${resetsIn}`);
}

function expectRelayQuery(operationPattern, response) {
  let resolve, reject;
  const handler = typeof response === 'function' ? response : () => ({
    data: response
  });
  const promise = new Promise((resolve0, reject0) => {
    resolve = resolve0;
    reject = reject0;
  });
  const existing = responsesByQuery.get(operationPattern.name) || [];
  existing.push({
    promise,
    handler,
    variables: operationPattern.variables || {},
    trace: operationPattern.trace
  });
  responsesByQuery.set(operationPattern.name, existing);

  const disable = () => responsesByQuery["delete"](operationPattern.name);

  return {
    promise,
    resolve,
    reject,
    disable
  };
}

function clearRelayExpectations() {
  responsesByQuery.clear();
}

function createFetchQuery(url) {
  if (atom.inSpecMode()) {
    return function specFetchQuery(operation, variables, _cacheConfig, _uploadables) {
      const expectations = responsesByQuery.get(operation.name) || [];
      const match = expectations.find(expectation => {
        if (isEqual === null) {
          // Lazily require lodash.isequal so we can keep it as a dev dependency.
          // Require indirectly to trick electron-link into not following this.
          isEqual = require(LODASH_ISEQUAL);
        }

        return isEqual(expectation.variables, variables);
      });

      if (!match) {
        // eslint-disable-next-line no-console
        console.log(`GraphQL query ${operation.name} was:\n  ${operation.text.replace(/\n/g, '\n  ')}\n` + _util["default"].inspect(variables));
        const e = new Error(`Unexpected GraphQL query: ${operation.name}`);
        e.rawStack = e.stack;
        throw e;
      }

      const responsePromise = match.promise.then(() => {
        return match.handler(operation);
      });

      if (match.trace) {
        // eslint-disable-next-line no-console
        console.log(`[Relay] query "${operation.name}":\n${operation.text}`);
        responsePromise.then(result => {
          // eslint-disable-next-line no-console
          console.log(`[Relay] response "${operation.name}":`, result);
        }, err => {
          // eslint-disable-next-line no-console
          console.error(`[Relay] error "${operation.name}":\n${err.stack || err}`);
          throw err;
        });
      }

      return responsePromise;
    };
  }

  return async function fetchQuery(operation, variables, _cacheConfig, _uploadables) {
    const currentToken = tokenPerURL.get(url);
    const response = await fetch(url, {
      method: 'POST',
      headers: {
        'content-type': 'application/json',
        'Authorization': `bearer ${currentToken}`,
        'Accept': 'application/vnd.github.graphql-profiling+json'
      },
      body: JSON.stringify({
        query: operation.text,
        variables
      })
    });

    try {
      atom && atom.inDevMode() && logRatelimitApi(response.headers);
    } catch (_e) {
      /* do nothing */
    }

    if (response.status !== 200) {
      const e = new Error(`GraphQL API endpoint at ${url} returned ${response.status}`);
      e.response = response;
      e.responseText = await response.text();
      e.rawStack = e.stack;
      throw e;
    }

    const payload = await response.json();

    if (payload && payload.errors && payload.errors.length > 0) {
      const e = new Error(`GraphQL API endpoint at ${url} returned an error for query ${operation.name}.`);
      e.response = response;
      e.errors = payload.errors;
      e.rawStack = e.stack;
      throw e;
    }

    return payload;
  };
}

class RelayNetworkLayerManager {
  static getEnvironmentForHost(endpoint, token) {
    const url = endpoint.getGraphQLRoot();
    let {
      environment,
      network
    } = relayEnvironmentPerURL.get(url) || {};
    tokenPerURL.set(url, token);

    if (!environment) {
      const source = new _relayRuntime.RecordSource();
      const store = new _relayRuntime.Store(source);
      network = _relayRuntime.Network.create(this.getFetchQuery(endpoint, token));
      environment = new _relayRuntime.Environment({
        network,
        store
      });
      relayEnvironmentPerURL.set(url, {
        environment,
        network
      });
    }

    return environment;
  }

  static getFetchQuery(endpoint, token) {
    const url = endpoint.getGraphQLRoot();
    tokenPerURL.set(url, token);
    let fetch = fetchPerURL.get(url);

    if (!fetch) {
      fetch = createFetchQuery(url);
      fetchPerURL.set(fetch);
    }

    return fetch;
  }

}

exports["default"] = RelayNetworkLayerManager;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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