"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _atom = require("atom");

var _util = require("util");

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const LAYER_NAMES = ['unchanged', 'addition', 'deletion', 'nonewline', 'hunk', 'patch'];

class PatchBuffer {
  constructor() {
    this.buffer = new _atom.TextBuffer();
    this.buffer.retain();
    this.layers = LAYER_NAMES.reduce((map, layerName) => {
      map[layerName] = this.buffer.addMarkerLayer();
      return map;
    }, {});
  }

  getBuffer() {
    return this.buffer;
  }

  getInsertionPoint() {
    return this.buffer.getEndPosition();
  }

  getLayer(layerName) {
    return this.layers[layerName];
  }

  findMarkers(layerName, ...args) {
    return this.layers[layerName].findMarkers(...args);
  }

  findAllMarkers(...args) {
    return LAYER_NAMES.reduce((arr, layerName) => {
      arr.push(...this.findMarkers(layerName, ...args));
      return arr;
    }, []);
  }

  markPosition(layerName, ...args) {
    return this.layers[layerName].markPosition(...args);
  }

  markRange(layerName, ...args) {
    return this.layers[layerName].markRange(...args);
  }

  clearAllLayers() {
    for (const layerName of LAYER_NAMES) {
      this.layers[layerName].clear();
    }
  }

  createInserterAt(insertionPoint) {
    return new Inserter(this, _atom.Point.fromObject(insertionPoint));
  }

  createInserterAtEnd() {
    return this.createInserterAt(this.getInsertionPoint());
  }

  createSubBuffer(rangeLike, options = {}) {
    const opts = _objectSpread({
      exclude: new Set()
    }, options);

    const range = _atom.Range.fromObject(rangeLike);

    const baseOffset = range.start.negate();
    const includedMarkersByLayer = LAYER_NAMES.reduce((map, layerName) => {
      map[layerName] = this.layers[layerName].findMarkers({
        intersectsRange: range
      }).filter(m => !opts.exclude.has(m));
      return map;
    }, {});
    const markerMap = new Map();
    const subBuffer = new PatchBuffer();
    subBuffer.getBuffer().setText(this.buffer.getTextInRange(range));

    for (const layerName of LAYER_NAMES) {
      for (const oldMarker of includedMarkersByLayer[layerName]) {
        const oldRange = oldMarker.getRange();
        const clippedStart = oldRange.start.isLessThanOrEqual(range.start) ? range.start : oldRange.start;
        const clippedEnd = oldRange.end.isGreaterThanOrEqual(range.end) ? range.end : oldRange.end; // Exclude non-empty markers that intersect *only* at the range start or end

        if (clippedStart.isEqual(clippedEnd) && !oldRange.start.isEqual(oldRange.end)) {
          continue;
        }

        const startOffset = clippedStart.row === range.start.row ? baseOffset : [baseOffset.row, 0];
        const endOffset = clippedEnd.row === range.start.row ? baseOffset : [baseOffset.row, 0];
        const newMarker = subBuffer.markRange(layerName, [clippedStart.translate(startOffset), clippedEnd.translate(endOffset)], oldMarker.getProperties());
        markerMap.set(oldMarker, newMarker);
      }
    }

    return {
      patchBuffer: subBuffer,
      markerMap
    };
  }

  extractPatchBuffer(rangeLike, options = {}) {
    const {
      patchBuffer: subBuffer,
      markerMap
    } = this.createSubBuffer(rangeLike, options);

    for (const oldMarker of markerMap.keys()) {
      oldMarker.destroy();
    }

    this.buffer.setTextInRange(rangeLike, '');
    return {
      patchBuffer: subBuffer,
      markerMap
    };
  }

  deleteLastNewline() {
    if (this.buffer.getLastLine() === '') {
      this.buffer.deleteRow(this.buffer.getLastRow());
    }

    return this;
  }

  adopt(original) {
    this.clearAllLayers();
    this.buffer.setText(original.getBuffer().getText());
    const markerMap = new Map();

    for (const layerName of LAYER_NAMES) {
      for (const originalMarker of original.getLayer(layerName).getMarkers()) {
        const newMarker = this.markRange(layerName, originalMarker.getRange(), originalMarker.getProperties());
        markerMap.set(originalMarker, newMarker);
      }
    }

    return markerMap;
  }
  /* istanbul ignore next */


  inspect(opts = {}) {
    /* istanbul ignore next */
    const options = _objectSpread({
      layerNames: LAYER_NAMES
    }, opts);

    let inspectString = '';
    const increasingMarkers = [];

    for (const layerName of options.layerNames) {
      for (const marker of this.findMarkers(layerName, {})) {
        increasingMarkers.push({
          layerName,
          point: marker.getRange().start,
          start: true,
          id: marker.id
        });
        increasingMarkers.push({
          layerName,
          point: marker.getRange().end,
          end: true,
          id: marker.id
        });
      }
    }

    increasingMarkers.sort((a, b) => {
      const cmp = a.point.compare(b.point);

      if (cmp !== 0) {
        return cmp;
      } else if (a.start && b.start) {
        return 0;
      } else if (a.start && !b.start) {
        return -1;
      } else if (!a.start && b.start) {
        return 1;
      } else {
        return 0;
      }
    });

    let inspectPoint = _atom.Point.fromObject([0, 0]);

    for (const marker of increasingMarkers) {
      if (!marker.point.isEqual(inspectPoint)) {
        inspectString += (0, _util.inspect)(this.buffer.getTextInRange([inspectPoint, marker.point])) + '\n';
      }

      if (marker.start) {
        inspectString += `  start ${marker.layerName}@${marker.id}\n`;
      } else if (marker.end) {
        inspectString += `  end ${marker.layerName}@${marker.id}\n`;
      }

      inspectPoint = marker.point;
    }

    return inspectString;
  }

}

exports["default"] = PatchBuffer;

class Inserter {
  constructor(patchBuffer, insertionPoint) {
    const clipped = patchBuffer.getBuffer().clipPosition(insertionPoint);
    this.patchBuffer = patchBuffer;
    this.startPoint = clipped.copy();
    this.insertionPoint = clipped.copy();
    this.markerBlueprints = [];
    this.markerMapCallbacks = [];
    this.markersBefore = new Set();
    this.markersAfter = new Set();
  }

  keepBefore(markers) {
    for (const marker of markers) {
      if (marker.getRange().end.isEqual(this.startPoint)) {
        this.markersBefore.add(marker);
      }
    }

    return this;
  }

  keepAfter(markers) {
    for (const marker of markers) {
      if (marker.getRange().start.isEqual(this.startPoint)) {
        this.markersAfter.add(marker);
      }
    }

    return this;
  }

  markWhile(layerName, block, markerOpts) {
    const start = this.insertionPoint.copy();
    block();
    const end = this.insertionPoint.copy();
    this.markerBlueprints.push({
      layerName,
      range: new _atom.Range(start, end),
      markerOpts
    });
    return this;
  }

  insert(text) {
    const insertedRange = this.patchBuffer.getBuffer().insert(this.insertionPoint, text);
    this.insertionPoint = insertedRange.end;
    return this;
  }

  insertMarked(text, layerName, markerOpts) {
    return this.markWhile(layerName, () => this.insert(text), markerOpts);
  }

  insertPatchBuffer(subPatchBuffer, opts) {
    const baseOffset = this.insertionPoint.copy();
    this.insert(subPatchBuffer.getBuffer().getText());
    const subMarkerMap = new Map();

    for (const layerName of LAYER_NAMES) {
      for (const oldMarker of subPatchBuffer.findMarkers(layerName, {})) {
        const startOffset = oldMarker.getRange().start.row === 0 ? baseOffset : [baseOffset.row, 0];
        const endOffset = oldMarker.getRange().end.row === 0 ? baseOffset : [baseOffset.row, 0];
        const range = oldMarker.getRange().translate(startOffset, endOffset);

        const markerOpts = _objectSpread({}, oldMarker.getProperties(), {
          callback: newMarker => {
            subMarkerMap.set(oldMarker, newMarker);
          }
        });

        this.markerBlueprints.push({
          layerName,
          range,
          markerOpts
        });
      }
    }

    this.markerMapCallbacks.push({
      markerMap: subMarkerMap,
      callback: opts.callback
    });
    return this;
  }

  apply() {
    for (const _ref of this.markerBlueprints) {
      const {
        layerName,
        range,
        markerOpts
      } = _ref;
      const callback = markerOpts.callback;
      delete markerOpts.callback;
      const marker = this.patchBuffer.markRange(layerName, range, markerOpts);

      if (callback) {
        callback(marker);
      }
    }

    for (const _ref2 of this.markerMapCallbacks) {
      const {
        markerMap,
        callback
      } = _ref2;
      callback(markerMap);
    }

    for (const beforeMarker of this.markersBefore) {
      const isEmpty = beforeMarker.getRange().isEmpty();

      if (!beforeMarker.isReversed()) {
        beforeMarker.setHeadPosition(this.startPoint);

        if (isEmpty) {
          beforeMarker.setTailPosition(this.startPoint);
        }
      } else {
        beforeMarker.setTailPosition(this.startPoint);

        if (isEmpty) {
          beforeMarker.setHeadPosition(this.startPoint);
        }
      }
    }

    for (const afterMarker of this.markersAfter) {
      const isEmpty = afterMarker.getRange().isEmpty();

      if (!afterMarker.isReversed()) {
        afterMarker.setTailPosition(this.insertionPoint);

        if (isEmpty) {
          afterMarker.setHeadPosition(this.insertionPoint);
        }
      } else {
        afterMarker.setHeadPosition(this.insertionPoint);

        if (isEmpty) {
          afterMarker.setTailPosition(this.insertionPoint);
        }
      }
    }
  }

}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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