"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _eventKit = require("event-kit");

var _file = require("./file");

var _patch = _interopRequireWildcard(require("./patch"));

var _helpers = require("../../helpers");

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj["default"] = obj; return newObj; } }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class FilePatch {
  static createNull() {
    return new this(_file.nullFile, _file.nullFile, _patch["default"].createNull());
  }

  static createHiddenFilePatch(oldFile, newFile, marker, renderStatus, showFn) {
    return new this(oldFile, newFile, _patch["default"].createHiddenPatch(marker, renderStatus, showFn));
  }

  constructor(oldFile, newFile, patch, rawPatches) {
    this.oldFile = oldFile;
    this.newFile = newFile;
    this.patch = patch;
    this.rawPatches = rawPatches;
    this.emitter = new _eventKit.Emitter();
  }

  isPresent() {
    return this.oldFile.isPresent() || this.newFile.isPresent() || this.patch.isPresent();
  }

  getRenderStatus() {
    return this.patch.getRenderStatus();
  }

  getOldFile() {
    return this.oldFile;
  }

  getNewFile() {
    return this.newFile;
  }

  getRawContentPatch() {
    if (!this.rawPatches) {
      throw new Error('FilePatch was not parsed with {perserveOriginal: true}');
    }

    return this.rawPatches.content;
  }

  getPatch() {
    return this.patch;
  }

  getMarker() {
    return this.getPatch().getMarker();
  }

  getStartRange() {
    return this.getPatch().getStartRange();
  }

  getOldPath() {
    return this.getOldFile().getPath();
  }

  getNewPath() {
    return this.getNewFile().getPath();
  }

  getOldMode() {
    return this.getOldFile().getMode();
  }

  getNewMode() {
    return this.getNewFile().getMode();
  }

  getOldSymlink() {
    return this.getOldFile().getSymlink();
  }

  getNewSymlink() {
    return this.getNewFile().getSymlink();
  }

  getFirstChangeRange() {
    return this.getPatch().getFirstChangeRange();
  }

  getMaxLineNumberWidth() {
    return this.getPatch().getMaxLineNumberWidth();
  }

  containsRow(row) {
    return this.getPatch().containsRow(row);
  }

  didChangeExecutableMode() {
    if (!this.oldFile.isPresent() || !this.newFile.isPresent()) {
      return false;
    }

    return this.oldFile.isExecutable() && !this.newFile.isExecutable() || !this.oldFile.isExecutable() && this.newFile.isExecutable();
  }

  hasSymlink() {
    return Boolean(this.getOldFile().getSymlink() || this.getNewFile().getSymlink());
  }

  hasTypechange() {
    if (!this.oldFile.isPresent() || !this.newFile.isPresent()) {
      return false;
    }

    return this.oldFile.isSymlink() && !this.newFile.isSymlink() || !this.oldFile.isSymlink() && this.newFile.isSymlink();
  }

  getPath() {
    return this.getOldPath() || this.getNewPath();
  }

  getStatus() {
    return this.getPatch().getStatus();
  }

  getHunks() {
    return this.getPatch().getHunks();
  }

  updateMarkers(map) {
    return this.patch.updateMarkers(map);
  }

  triggerCollapseIn(patchBuffer, {
    before,
    after
  }) {
    if (!this.patch.getRenderStatus().isVisible()) {
      return false;
    }

    const oldPatch = this.patch;
    const oldRange = oldPatch.getRange().copy();
    const insertionPosition = oldRange.start;
    const exclude = new Set([...before, ...after]);
    const {
      patchBuffer: subPatchBuffer,
      markerMap
    } = patchBuffer.extractPatchBuffer(oldRange, {
      exclude
    });
    oldPatch.destroyMarkers();
    oldPatch.updateMarkers(markerMap); // Delete the separating newline after the collapsing patch, if any.

    if (!oldRange.isEmpty()) {
      patchBuffer.getBuffer().deleteRow(insertionPosition.row);
    }

    const patchMarker = patchBuffer.markPosition(_patch["default"].layerName, insertionPosition, {
      invalidate: 'never',
      exclusive: true
    });
    this.patch = _patch["default"].createHiddenPatch(patchMarker, _patch.COLLAPSED, () => {
      return {
        patch: oldPatch,
        patchBuffer: subPatchBuffer
      };
    });
    this.didChangeRenderStatus();
    return true;
  }

  triggerExpandIn(patchBuffer, {
    before,
    after
  }) {
    if (this.patch.getRenderStatus().isVisible()) {
      return false;
    }

    const {
      patch: nextPatch,
      patchBuffer: subPatchBuffer
    } = this.patch.show();
    const atStart = this.patch.getInsertionPoint().isEqual([0, 0]);
    const atEnd = this.patch.getInsertionPoint().isEqual(patchBuffer.getBuffer().getEndPosition());
    const willHaveContent = !subPatchBuffer.getBuffer().isEmpty(); // The expanding patch's insertion point is just after the unmarked newline that separates adjacent visible
    // patches:
    // <p0> '\n' * <p1> '\n' <p2>
    //
    // If it's to become the first (visible) patch, its insertion point is at [0, 0]:
    // * <p0> '\n' <p1> '\n' <p2>
    //
    // If it's to become the final (visible) patch, its insertion point is at the buffer end:
    // <p0> '\n' <p1> '\n' <p2> *
    //
    // Insert a newline *before* the expanding patch if we're inserting at the buffer's end, but the buffer is non-empty
    // (so it isn't also the end of the buffer). Insert a newline *after* the expanding patch when inserting anywhere
    // but the buffer's end.

    if (willHaveContent && atEnd && !atStart) {
      const beforeNewline = [];
      const afterNewline = after.slice();

      for (const marker of before) {
        if (marker.getRange().isEmpty()) {
          afterNewline.push(marker);
        } else {
          beforeNewline.push(marker);
        }
      }

      patchBuffer.createInserterAt(this.patch.getInsertionPoint()).keepBefore(beforeNewline).keepAfter(afterNewline).insert('\n').apply();
    }

    patchBuffer.createInserterAt(this.patch.getInsertionPoint()).keepBefore(before).keepAfter(after).insertPatchBuffer(subPatchBuffer, {
      callback: map => nextPatch.updateMarkers(map)
    }).insert(!atEnd ? '\n' : '').apply();
    this.patch.destroyMarkers();
    this.patch = nextPatch;
    this.didChangeRenderStatus();
    return true;
  }

  didChangeRenderStatus() {
    return this.emitter.emit('change-render-status', this);
  }

  onDidChangeRenderStatus(callback) {
    return this.emitter.on('change-render-status', callback);
  }

  clone(opts = {}) {
    return new this.constructor(opts.oldFile !== undefined ? opts.oldFile : this.oldFile, opts.newFile !== undefined ? opts.newFile : this.newFile, opts.patch !== undefined ? opts.patch : this.patch);
  }

  getStartingMarkers() {
    return this.patch.getStartingMarkers();
  }

  getEndingMarkers() {
    return this.patch.getEndingMarkers();
  }

  buildStagePatchForLines(originalBuffer, nextPatchBuffer, selectedLineSet) {
    let newFile = this.getNewFile();

    if (this.getStatus() === 'deleted') {
      if (this.patch.getChangedLineCount() === selectedLineSet.size && Array.from(selectedLineSet, row => this.patch.containsRow(row)).every(Boolean)) {
        // Whole file deletion staged.
        newFile = _file.nullFile;
      } else {
        // Partial file deletion, which becomes a modification.
        newFile = this.getOldFile();
      }
    }

    const patch = this.patch.buildStagePatchForLines(originalBuffer, nextPatchBuffer, selectedLineSet);
    return this.clone({
      newFile,
      patch
    });
  }

  buildUnstagePatchForLines(originalBuffer, nextPatchBuffer, selectedLineSet) {
    const nonNullFile = this.getNewFile().isPresent() ? this.getNewFile() : this.getOldFile();
    let oldFile = this.getNewFile();
    let newFile = nonNullFile;

    if (this.getStatus() === 'added') {
      if (selectedLineSet.size === this.patch.getChangedLineCount() && Array.from(selectedLineSet, row => this.patch.containsRow(row)).every(Boolean)) {
        // Ensure that newFile is null if the patch is an addition because we're deleting the entire file from the
        // index. If a symlink was deleted and replaced by a non-symlink file, we don't want the symlink entry to muck
        // up the patch.
        oldFile = nonNullFile;
        newFile = _file.nullFile;
      }
    } else if (this.getStatus() === 'deleted') {
      if (selectedLineSet.size === this.patch.getChangedLineCount() && Array.from(selectedLineSet, row => this.patch.containsRow(row)).every(Boolean)) {
        oldFile = _file.nullFile;
        newFile = nonNullFile;
      }
    }

    const patch = this.patch.buildUnstagePatchForLines(originalBuffer, nextPatchBuffer, selectedLineSet);
    return this.clone({
      oldFile,
      newFile,
      patch
    });
  }

  toStringIn(buffer) {
    if (!this.isPresent()) {
      return '';
    }

    if (this.hasTypechange()) {
      const left = this.clone({
        newFile: _file.nullFile,
        patch: this.getOldSymlink() ? this.getPatch().clone({
          status: 'deleted'
        }) : this.getPatch()
      });
      const right = this.clone({
        oldFile: _file.nullFile,
        patch: this.getNewSymlink() ? this.getPatch().clone({
          status: 'added'
        }) : this.getPatch()
      });
      return left.toStringIn(buffer) + right.toStringIn(buffer);
    } else if (this.getStatus() === 'added' && this.getNewFile().isSymlink()) {
      const symlinkPath = this.getNewSymlink();
      return this.getHeaderString() + `@@ -0,0 +1 @@\n+${symlinkPath}\n\\ No newline at end of file\n`;
    } else if (this.getStatus() === 'deleted' && this.getOldFile().isSymlink()) {
      const symlinkPath = this.getOldSymlink();
      return this.getHeaderString() + `@@ -1 +0,0 @@\n-${symlinkPath}\n\\ No newline at end of file\n`;
    } else {
      return this.getHeaderString() + this.getPatch().toStringIn(buffer);
    }
  }
  /*
   * Construct a String containing diagnostic information about the internal state of this FilePatch.
   */

  /* istanbul ignore next */


  inspect(opts = {}) {
    const options = _objectSpread({
      indent: 0
    }, opts);

    let indentation = '';

    for (let i = 0; i < options.indent; i++) {
      indentation += ' ';
    }

    let inspectString = `${indentation}(FilePatch `;

    if (this.getOldPath() !== this.getNewPath()) {
      inspectString += `oldPath=${this.getOldPath()} newPath=${this.getNewPath()}`;
    } else {
      inspectString += `path=${this.getPath()}`;
    }

    inspectString += '\n';
    inspectString += this.patch.inspect({
      indent: options.indent + 2
    });
    inspectString += `${indentation})\n`;
    return inspectString;
  }

  getHeaderString() {
    const fromPath = this.getOldPath() || this.getNewPath();
    const toPath = this.getNewPath() || this.getOldPath();
    let header = `diff --git a/${(0, _helpers.toGitPathSep)(fromPath)} b/${(0, _helpers.toGitPathSep)(toPath)}`;
    header += '\n';

    if (this.getStatus() === 'added') {
      header += `new file mode ${this.getNewMode()}`;
      header += '\n';
    } else if (this.getStatus() === 'deleted') {
      header += `deleted file mode ${this.getOldMode()}`;
      header += '\n';
    }

    header += this.getOldPath() ? `--- a/${(0, _helpers.toGitPathSep)(this.getOldPath())}` : '--- /dev/null';
    header += '\n';
    header += this.getNewPath() ? `+++ b/${(0, _helpers.toGitPathSep)(this.getNewPath())}` : '+++ /dev/null';
    header += '\n';
    return header;
  }

}

exports["default"] = FilePatch;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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