"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.nullOperationStateObserver = exports["default"] = exports.FETCH = exports.PULL = exports.PUSH = void 0;

var _eventKit = require("event-kit");

const PUSH = {
  getter(o) {
    return o.isPushInProgress();
  }

};
exports.PUSH = PUSH;
const PULL = {
  getter(o) {
    return o.isPullInProgress();
  }

};
exports.PULL = PULL;
const FETCH = {
  getter(o) {
    return o.isFetchInProgress();
  }

}; // Notify subscibers when a repository completes one or more operations of interest, as observed by its OperationState
// transitioning from `true` to `false`. For exampe, use this to perform actions when a push completes.

exports.FETCH = FETCH;

class OperationStateObserver {
  constructor(repository, ...operations) {
    this.repository = repository;
    this.operations = new Set(operations);
    this.emitter = new _eventKit.Emitter();
    this.lastStates = new Map();

    for (const operation of this.operations) {
      this.lastStates.set(operation, operation.getter(this.repository.getOperationStates()));
    }

    this.sub = this.repository.onDidUpdate(this.handleUpdate.bind(this));
  }

  onDidComplete(handler) {
    return this.emitter.on('did-complete', handler);
  }

  handleUpdate() {
    let fire = false;

    for (const operation of this.operations) {
      const last = this.lastStates.get(operation);
      const current = operation.getter(this.repository.getOperationStates());

      if (last && !current) {
        fire = true;
      }

      this.lastStates.set(operation, current);
    }

    if (fire) {
      this.emitter.emit('did-complete');
    }
  }

  dispose() {
    this.emitter.dispose();
    this.sub.dispose();
  }

}

exports["default"] = OperationStateObserver;
const nullOperationStateObserver = {
  onDidComplete() {
    return new _eventKit.Disposable();
  },

  dispose() {}

};
exports.nullOperationStateObserver = nullOperationStateObserver;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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