"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _path = _interopRequireDefault(require("path"));

var _electron = require("electron");

var _react = _interopRequireWildcard(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _eventKit = require("event-kit");

var _statusBar = _interopRequireDefault(require("../atom/status-bar"));

var _panel = _interopRequireDefault(require("../atom/panel"));

var _paneItem = _interopRequireDefault(require("../atom/pane-item"));

var _cloneDialog = _interopRequireDefault(require("../views/clone-dialog"));

var _openIssueishDialog = _interopRequireDefault(require("../views/open-issueish-dialog"));

var _openCommitDialog = _interopRequireDefault(require("../views/open-commit-dialog"));

var _initDialog = _interopRequireDefault(require("../views/init-dialog"));

var _credentialDialog = _interopRequireDefault(require("../views/credential-dialog"));

var _commands = _interopRequireWildcard(require("../atom/commands"));

var _changedFileItem = _interopRequireDefault(require("../items/changed-file-item"));

var _issueishDetailItem = _interopRequireDefault(require("../items/issueish-detail-item"));

var _commitDetailItem = _interopRequireDefault(require("../items/commit-detail-item"));

var _commitPreviewItem = _interopRequireDefault(require("../items/commit-preview-item"));

var _gitTabItem = _interopRequireDefault(require("../items/git-tab-item"));

var _githubTabItem = _interopRequireDefault(require("../items/github-tab-item"));

var _reviewsItem = _interopRequireDefault(require("../items/reviews-item"));

var _commentDecorationsContainer = _interopRequireDefault(require("../containers/comment-decorations-container"));

var _statusBarTileController = _interopRequireDefault(require("./status-bar-tile-controller"));

var _repositoryConflictController = _interopRequireDefault(require("./repository-conflict-controller"));

var _gitCacheView = _interopRequireDefault(require("../views/git-cache-view"));

var _gitTimingsView = _interopRequireDefault(require("../views/git-timings-view"));

var _conflict = _interopRequireDefault(require("../models/conflicts/conflict"));

var _switchboard = _interopRequireDefault(require("../switchboard"));

var _propTypes2 = require("../prop-types");

var _helpers = require("../helpers");

var _gitShellOutStrategy = require("../git-shell-out-strategy");

var _reporterProxy = require("../reporter-proxy");

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj["default"] = obj; return newObj; } }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _extends() { _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; }; return _extends.apply(this, arguments); }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class RootController extends _react["default"].Component {
  constructor(props, context) {
    super(props, context);

    _defineProperty(this, "toggleCommitPreviewItem", () => {
      const workdir = this.props.repository.getWorkingDirectoryPath();
      return this.props.workspace.toggle(_commitPreviewItem["default"].buildURI(workdir));
    });

    _defineProperty(this, "showOpenCommitDialog", () => {
      this.setState({
        openCommitDialogActive: true
      });
    });

    _defineProperty(this, "isValidCommit", async ref => {
      try {
        await this.props.repository.getCommit(ref);
        return true;
      } catch (error) {
        if (error instanceof _gitShellOutStrategy.GitError && error.code === 128) {
          return false;
        } else {
          throw error;
        }
      }
    });

    _defineProperty(this, "acceptOpenCommit", ({
      ref
    }) => {
      const workdir = this.props.repository.getWorkingDirectoryPath();

      const uri = _commitDetailItem["default"].buildURI(workdir, ref);

      this.setState({
        openCommitDialogActive: false
      });
      this.props.workspace.open(uri).then(() => {
        (0, _reporterProxy.addEvent)('open-commit-in-pane', {
          "package": 'github',
          from: _openCommitDialog["default"].name
        });
      });
    });

    _defineProperty(this, "cancelOpenCommit", () => {
      this.setState({
        openCommitDialogActive: false
      });
    });

    _defineProperty(this, "surfaceFromFileAtPath", (filePath, stagingStatus) => {
      const gitTab = this.gitTabTracker.getComponent();
      return gitTab && gitTab.focusAndSelectStagingItem(filePath, stagingStatus);
    });

    _defineProperty(this, "surfaceToCommitPreviewButton", () => {
      const gitTab = this.gitTabTracker.getComponent();
      return gitTab && gitTab.focusAndSelectCommitPreviewButton();
    });

    _defineProperty(this, "surfaceToRecentCommit", () => {
      const gitTab = this.gitTabTracker.getComponent();
      return gitTab && gitTab.focusAndSelectRecentCommit();
    });

    _defineProperty(this, "reportMutationErrors", (friendlyMessage, err) => {
      const opts = {
        dismissable: true
      };

      if (err.errors) {
        opts.detail = err.errors.map(e => e.message).join('\n');
      } else if (err.stack) {
        opts.stack = err.stack;
      }

      this.props.notificationManager.addError(friendlyMessage, opts);
    });

    (0, _helpers.autobind)(this, 'installReactDevTools', 'clearGithubToken', 'initializeRepo', 'showOpenIssueishDialog', 'showWaterfallDiagnostics', 'showCacheDiagnostics', 'acceptClone', 'cancelClone', 'acceptInit', 'cancelInit', 'acceptOpenIssueish', 'cancelOpenIssueish', 'destroyFilePatchPaneItems', 'destroyEmptyFilePatchPaneItems', 'openCloneDialog', 'quietlySelectItem', 'viewUnstagedChangesForCurrentFile', 'viewStagedChangesForCurrentFile', 'openFiles', 'getUnsavedFiles', 'ensureNoUnsavedFiles', 'discardWorkDirChangesForPaths', 'discardLines', 'undoLastDiscard', 'refreshResolutionProgress');
    this.state = {
      cloneDialogActive: false,
      cloneDialogInProgress: false,
      initDialogActive: false,
      initDialogPath: null,
      initDialogResolve: null,
      credentialDialogQuery: null
    };
    this.gitTabTracker = new TabTracker('git', {
      uri: _gitTabItem["default"].buildURI(),
      getWorkspace: () => this.props.workspace
    });
    this.githubTabTracker = new TabTracker('github', {
      uri: _githubTabItem["default"].buildURI(),
      getWorkspace: () => this.props.workspace
    });
    this.subscription = new _eventKit.CompositeDisposable(this.props.repository.onPullError(this.gitTabTracker.ensureVisible));
    this.props.commandRegistry.onDidDispatch(event => {
      if (event.type && event.type.startsWith('github:') && event.detail && event.detail[0] && event.detail[0].contextCommand) {
        (0, _reporterProxy.addEvent)('context-menu-action', {
          "package": 'github',
          command: event.type
        });
      }
    });
  }

  componentDidMount() {
    this.openTabs();
  }

  render() {
    return _react["default"].createElement(_react.Fragment, null, this.renderCommands(), this.renderStatusBarTile(), this.renderPaneItems(), this.renderDialogs(), this.renderConflictResolver(), this.renderCommentDecorations());
  }

  renderCommands() {
    const devMode = global.atom && global.atom.inDevMode();
    return _react["default"].createElement(_commands["default"], {
      registry: this.props.commandRegistry,
      target: "atom-workspace"
    }, devMode && _react["default"].createElement(_commands.Command, {
      command: "github:install-react-dev-tools",
      callback: this.installReactDevTools
    }), _react["default"].createElement(_commands.Command, {
      command: "github:toggle-commit-preview",
      callback: this.toggleCommitPreviewItem
    }), _react["default"].createElement(_commands.Command, {
      command: "github:logout",
      callback: this.clearGithubToken
    }), _react["default"].createElement(_commands.Command, {
      command: "github:show-waterfall-diagnostics",
      callback: this.showWaterfallDiagnostics
    }), _react["default"].createElement(_commands.Command, {
      command: "github:show-cache-diagnostics",
      callback: this.showCacheDiagnostics
    }), _react["default"].createElement(_commands.Command, {
      command: "github:open-issue-or-pull-request",
      callback: this.showOpenIssueishDialog
    }), _react["default"].createElement(_commands.Command, {
      command: "github:toggle-git-tab",
      callback: this.gitTabTracker.toggle
    }), _react["default"].createElement(_commands.Command, {
      command: "github:toggle-git-tab-focus",
      callback: this.gitTabTracker.toggleFocus
    }), _react["default"].createElement(_commands.Command, {
      command: "github:toggle-github-tab",
      callback: this.githubTabTracker.toggle
    }), _react["default"].createElement(_commands.Command, {
      command: "github:toggle-github-tab-focus",
      callback: this.githubTabTracker.toggleFocus
    }), _react["default"].createElement(_commands.Command, {
      command: "github:clone",
      callback: this.openCloneDialog
    }), _react["default"].createElement(_commands.Command, {
      command: "github:open-commit",
      callback: this.showOpenCommitDialog
    }), _react["default"].createElement(_commands.Command, {
      command: "github:view-unstaged-changes-for-current-file",
      callback: this.viewUnstagedChangesForCurrentFile
    }), _react["default"].createElement(_commands.Command, {
      command: "github:view-staged-changes-for-current-file",
      callback: this.viewStagedChangesForCurrentFile
    }), _react["default"].createElement(_commands.Command, {
      command: "github:close-all-diff-views",
      callback: this.destroyFilePatchPaneItems
    }), _react["default"].createElement(_commands.Command, {
      command: "github:close-empty-diff-views",
      callback: this.destroyEmptyFilePatchPaneItems
    }));
  }

  renderStatusBarTile() {
    return _react["default"].createElement(_statusBar["default"], {
      statusBar: this.props.statusBar,
      onConsumeStatusBar: sb => this.onConsumeStatusBar(sb),
      className: "github-StatusBarTileController"
    }, _react["default"].createElement(_statusBarTileController["default"], {
      pipelineManager: this.props.pipelineManager,
      workspace: this.props.workspace,
      repository: this.props.repository,
      commandRegistry: this.props.commandRegistry,
      notificationManager: this.props.notificationManager,
      tooltips: this.props.tooltips,
      confirm: this.props.confirm,
      toggleGitTab: this.gitTabTracker.toggle,
      toggleGithubTab: this.githubTabTracker.toggle
    }));
  }

  renderDialogs() {
    return _react["default"].createElement(_react.Fragment, null, this.renderInitDialog(), this.renderCloneDialog(), this.renderCredentialDialog(), this.renderOpenIssueishDialog(), this.renderOpenCommitDialog());
  }

  renderInitDialog() {
    if (!this.state.initDialogActive) {
      return null;
    }

    return _react["default"].createElement(_panel["default"], {
      workspace: this.props.workspace,
      location: "modal"
    }, _react["default"].createElement(_initDialog["default"], {
      config: this.props.config,
      commandRegistry: this.props.commandRegistry,
      initPath: this.state.initDialogPath,
      didAccept: this.acceptInit,
      didCancel: this.cancelInit
    }));
  }

  renderCloneDialog() {
    if (!this.state.cloneDialogActive) {
      return null;
    }

    return _react["default"].createElement(_panel["default"], {
      workspace: this.props.workspace,
      location: "modal"
    }, _react["default"].createElement(_cloneDialog["default"], {
      config: this.props.config,
      commandRegistry: this.props.commandRegistry,
      didAccept: this.acceptClone,
      didCancel: this.cancelClone,
      inProgress: this.state.cloneDialogInProgress
    }));
  }

  renderOpenIssueishDialog() {
    if (!this.state.openIssueishDialogActive) {
      return null;
    }

    return _react["default"].createElement(_panel["default"], {
      workspace: this.props.workspace,
      location: "modal"
    }, _react["default"].createElement(_openIssueishDialog["default"], {
      commandRegistry: this.props.commandRegistry,
      didAccept: this.acceptOpenIssueish,
      didCancel: this.cancelOpenIssueish
    }));
  }

  renderOpenCommitDialog() {
    if (!this.state.openCommitDialogActive) {
      return null;
    }

    return _react["default"].createElement(_panel["default"], {
      workspace: this.props.workspace,
      location: "modal"
    }, _react["default"].createElement(_openCommitDialog["default"], {
      commandRegistry: this.props.commandRegistry,
      didAccept: this.acceptOpenCommit,
      didCancel: this.cancelOpenCommit,
      isValidEntry: this.isValidCommit
    }));
  }

  renderCredentialDialog() {
    if (this.state.credentialDialogQuery === null) {
      return null;
    }

    return _react["default"].createElement(_panel["default"], {
      workspace: this.props.workspace,
      location: "modal"
    }, _react["default"].createElement(_credentialDialog["default"], _extends({
      commandRegistry: this.props.commandRegistry
    }, this.state.credentialDialogQuery)));
  }

  renderCommentDecorations() {
    if (!this.props.repository) {
      return null;
    }

    return _react["default"].createElement(_commentDecorationsContainer["default"], {
      workspace: this.props.workspace,
      commands: this.props.commandRegistry,
      localRepository: this.props.repository,
      loginModel: this.props.loginModel
    });
  }

  renderConflictResolver() {
    if (!this.props.repository) {
      return null;
    }

    return _react["default"].createElement(_repositoryConflictController["default"], {
      workspace: this.props.workspace,
      config: this.props.config,
      repository: this.props.repository,
      resolutionProgress: this.props.resolutionProgress,
      refreshResolutionProgress: this.refreshResolutionProgress,
      commandRegistry: this.props.commandRegistry
    });
  }

  renderPaneItems() {
    return _react["default"].createElement(_react.Fragment, null, _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _gitTabItem["default"].uriPattern,
      className: "github-Git-root"
    }, ({
      itemHolder
    }) => _react["default"].createElement(_gitTabItem["default"], {
      ref: itemHolder.setter,
      workspace: this.props.workspace,
      commandRegistry: this.props.commandRegistry,
      notificationManager: this.props.notificationManager,
      tooltips: this.props.tooltips,
      grammars: this.props.grammars,
      project: this.props.project,
      confirm: this.props.confirm,
      config: this.props.config,
      repository: this.props.repository,
      loginModel: this.props.loginModel,
      initializeRepo: this.initializeRepo,
      resolutionProgress: this.props.resolutionProgress,
      ensureGitTab: this.gitTabTracker.ensureVisible,
      openFiles: this.openFiles,
      discardWorkDirChangesForPaths: this.discardWorkDirChangesForPaths,
      undoLastDiscard: this.undoLastDiscard,
      refreshResolutionProgress: this.refreshResolutionProgress
    })), _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _githubTabItem["default"].uriPattern,
      className: "github-GitHub-root"
    }, ({
      itemHolder
    }) => _react["default"].createElement(_githubTabItem["default"], {
      ref: itemHolder.setter,
      repository: this.props.repository,
      loginModel: this.props.loginModel,
      workspace: this.props.workspace
    })), _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _changedFileItem["default"].uriPattern
    }, ({
      itemHolder,
      params
    }) => _react["default"].createElement(_changedFileItem["default"], {
      ref: itemHolder.setter,
      workdirContextPool: this.props.workdirContextPool,
      relPath: _path["default"].join(...params.relPath),
      workingDirectory: params.workingDirectory,
      stagingStatus: params.stagingStatus,
      tooltips: this.props.tooltips,
      commands: this.props.commandRegistry,
      keymaps: this.props.keymaps,
      workspace: this.props.workspace,
      config: this.props.config,
      discardLines: this.discardLines,
      undoLastDiscard: this.undoLastDiscard,
      surfaceFileAtPath: this.surfaceFromFileAtPath
    })), _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _commitPreviewItem["default"].uriPattern,
      className: "github-CommitPreview-root"
    }, ({
      itemHolder,
      params
    }) => _react["default"].createElement(_commitPreviewItem["default"], {
      ref: itemHolder.setter,
      workdirContextPool: this.props.workdirContextPool,
      workingDirectory: params.workingDirectory,
      workspace: this.props.workspace,
      commands: this.props.commandRegistry,
      keymaps: this.props.keymaps,
      tooltips: this.props.tooltips,
      config: this.props.config,
      discardLines: this.discardLines,
      undoLastDiscard: this.undoLastDiscard,
      surfaceToCommitPreviewButton: this.surfaceToCommitPreviewButton
    })), _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _commitDetailItem["default"].uriPattern,
      className: "github-CommitDetail-root"
    }, ({
      itemHolder,
      params
    }) => _react["default"].createElement(_commitDetailItem["default"], {
      ref: itemHolder.setter,
      workdirContextPool: this.props.workdirContextPool,
      workingDirectory: params.workingDirectory,
      workspace: this.props.workspace,
      commands: this.props.commandRegistry,
      keymaps: this.props.keymaps,
      tooltips: this.props.tooltips,
      config: this.props.config,
      sha: params.sha,
      surfaceCommit: this.surfaceToRecentCommit
    })), _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _issueishDetailItem["default"].uriPattern
    }, ({
      itemHolder,
      params,
      deserialized
    }) => _react["default"].createElement(_issueishDetailItem["default"], {
      ref: itemHolder.setter,
      host: params.host,
      owner: params.owner,
      repo: params.repo,
      issueishNumber: parseInt(params.issueishNumber, 10),
      workingDirectory: params.workingDirectory,
      workdirContextPool: this.props.workdirContextPool,
      loginModel: this.props.loginModel,
      initSelectedTab: deserialized.initSelectedTab,
      workspace: this.props.workspace,
      commands: this.props.commandRegistry,
      keymaps: this.props.keymaps,
      tooltips: this.props.tooltips,
      config: this.props.config,
      reportMutationErrors: this.reportMutationErrors
    })), _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _reviewsItem["default"].uriPattern
    }, ({
      itemHolder,
      params
    }) => _react["default"].createElement(_reviewsItem["default"], {
      ref: itemHolder.setter,
      host: params.host,
      owner: params.owner,
      repo: params.repo,
      number: parseInt(params.number, 10),
      workdir: params.workdir,
      workdirContextPool: this.props.workdirContextPool,
      loginModel: this.props.loginModel,
      workspace: this.props.workspace,
      tooltips: this.props.tooltips,
      config: this.props.config,
      commands: this.props.commandRegistry,
      reportMutationErrors: this.reportMutationErrors
    })), _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _gitTimingsView["default"].uriPattern
    }, ({
      itemHolder
    }) => _react["default"].createElement(_gitTimingsView["default"], {
      ref: itemHolder.setter
    })), _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _gitCacheView["default"].uriPattern
    }, ({
      itemHolder
    }) => _react["default"].createElement(_gitCacheView["default"], {
      ref: itemHolder.setter,
      repository: this.props.repository
    })));
  }

  async openTabs() {
    if (this.props.startOpen) {
      await Promise.all([this.gitTabTracker.ensureRendered(false), this.githubTabTracker.ensureRendered(false)]);
    }

    if (this.props.startRevealed) {
      const docks = new Set([_gitTabItem["default"].buildURI(), _githubTabItem["default"].buildURI()].map(uri => this.props.workspace.paneContainerForURI(uri)).filter(container => container && typeof container.show === 'function'));

      for (const dock of docks) {
        dock.show();
      }
    }
  }

  async installReactDevTools() {
    // Prevent electron-link from attempting to descend into electron-devtools-installer, which is not available
    // when we're bundled in Atom.
    const devToolsName = 'electron-devtools-installer';

    const devTools = require(devToolsName);

    await Promise.all([this.installExtension(devTools.REACT_DEVELOPER_TOOLS.id), // relay developer tools extension id
    this.installExtension('ncedobpgnmkhcmnnkcimnobpfepidadl')]);
    this.props.notificationManager.addSuccess('🌈 Reload your window to start using the React/Relay dev tools!');
  }

  async installExtension(id) {
    const devToolsName = 'electron-devtools-installer';

    const devTools = require(devToolsName);

    const crossUnzipName = 'cross-unzip';

    const unzip = require(crossUnzipName);

    const url = 'https://clients2.google.com/service/update2/crx?' + `response=redirect&x=id%3D${id}%26uc&prodversion=32`;

    const extensionFolder = _path["default"].resolve(_electron.remote.app.getPath('userData'), `extensions/${id}`);

    const extensionFile = `${extensionFolder}.crx`;
    await _fsExtra["default"].ensureDir(_path["default"].dirname(extensionFile));
    const response = await fetch(url, {
      method: 'GET'
    });
    const body = Buffer.from((await response.arrayBuffer()));
    await _fsExtra["default"].writeFile(extensionFile, body);
    await new Promise((resolve, reject) => {
      unzip(extensionFile, extensionFolder, async err => {
        if (err && !(await _fsExtra["default"].exists(_path["default"].join(extensionFolder, 'manifest.json')))) {
          reject(err);
        }

        resolve();
      });
    });
    await _fsExtra["default"].ensureDir(extensionFolder, 0o755);
    await devTools["default"](id);
  }

  componentWillUnmount() {
    this.subscription.dispose();
  }

  componentDidUpdate() {
    this.subscription.dispose();
    this.subscription = new _eventKit.CompositeDisposable(this.props.repository.onPullError(() => this.gitTabTracker.ensureVisible()));
  }

  onConsumeStatusBar(statusBar) {
    if (statusBar.disableGitInfoTile) {
      statusBar.disableGitInfoTile();
    }
  }

  clearGithubToken() {
    return this.props.loginModel.removeToken('https://api.github.com');
  }

  initializeRepo(initDialogPath) {
    if (this.state.initDialogActive) {
      return null;
    }

    if (!initDialogPath) {
      initDialogPath = this.props.repository.getWorkingDirectoryPath();
    }

    return new Promise(resolve => {
      this.setState({
        initDialogActive: true,
        initDialogPath,
        initDialogResolve: resolve
      });
    });
  }

  showOpenIssueishDialog() {
    this.setState({
      openIssueishDialogActive: true
    });
  }

  showWaterfallDiagnostics() {
    this.props.workspace.open(_gitTimingsView["default"].buildURI());
  }

  showCacheDiagnostics() {
    this.props.workspace.open(_gitCacheView["default"].buildURI());
  }

  async acceptClone(remoteUrl, projectPath) {
    this.setState({
      cloneDialogInProgress: true
    });

    try {
      await this.props.cloneRepositoryForProjectPath(remoteUrl, projectPath);
      (0, _reporterProxy.addEvent)('clone-repo', {
        "package": 'github'
      });
    } catch (e) {
      this.props.notificationManager.addError(`Unable to clone ${remoteUrl}`, {
        detail: e.stdErr,
        dismissable: true
      });
    } finally {
      this.setState({
        cloneDialogInProgress: false,
        cloneDialogActive: false
      });
    }
  }

  cancelClone() {
    this.setState({
      cloneDialogActive: false
    });
  }

  async acceptInit(projectPath) {
    try {
      await this.props.createRepositoryForProjectPath(projectPath);

      if (this.state.initDialogResolve) {
        this.state.initDialogResolve(projectPath);
      }
    } catch (e) {
      this.props.notificationManager.addError(`Unable to initialize git repository in ${projectPath}`, {
        detail: e.stdErr,
        dismissable: true
      });
    } finally {
      this.setState({
        initDialogActive: false,
        initDialogPath: null,
        initDialogResolve: null
      });
    }
  }

  cancelInit() {
    if (this.state.initDialogResolve) {
      this.state.initDialogResolve(false);
    }

    this.setState({
      initDialogActive: false,
      initDialogPath: null,
      initDialogResolve: null
    });
  }

  acceptOpenIssueish({
    repoOwner,
    repoName,
    issueishNumber
  }) {
    const uri = _issueishDetailItem["default"].buildURI({
      host: 'github.com',
      owner: repoOwner,
      repo: repoName,
      number: issueishNumber
    });

    this.setState({
      openIssueishDialogActive: false
    });
    this.props.workspace.open(uri).then(() => {
      (0, _reporterProxy.addEvent)('open-issueish-in-pane', {
        "package": 'github',
        from: 'dialog'
      });
    });
  }

  cancelOpenIssueish() {
    this.setState({
      openIssueishDialogActive: false
    });
  }

  destroyFilePatchPaneItems() {
    (0, _helpers.destroyFilePatchPaneItems)({
      onlyStaged: false
    }, this.props.workspace);
  }

  destroyEmptyFilePatchPaneItems() {
    (0, _helpers.destroyEmptyFilePatchPaneItems)(this.props.workspace);
  }

  openCloneDialog() {
    this.setState({
      cloneDialogActive: true
    });
  }

  quietlySelectItem(filePath, stagingStatus) {
    const gitTab = this.gitTabTracker.getComponent();
    return gitTab && gitTab.quietlySelectItem(filePath, stagingStatus);
  }

  async viewChangesForCurrentFile(stagingStatus) {
    const editor = this.props.workspace.getActiveTextEditor();

    if (!editor.getPath()) {
      return;
    }

    const absFilePath = await _fsExtra["default"].realpath(editor.getPath());
    const repoPath = this.props.repository.getWorkingDirectoryPath();

    if (repoPath === null) {
      const [projectPath] = this.props.project.relativizePath(editor.getPath());
      const notification = this.props.notificationManager.addInfo("Hmm, there's nothing to compare this file to", {
        description: 'You can create a Git repository to track changes to the files in your project.',
        dismissable: true,
        buttons: [{
          className: 'btn btn-primary',
          text: 'Create a repository now',
          onDidClick: async () => {
            notification.dismiss();
            const createdPath = await this.initializeRepo(projectPath); // If the user confirmed repository creation for this project path,
            // retry the operation that got them here in the first place

            if (createdPath === projectPath) {
              this.viewChangesForCurrentFile(stagingStatus);
            }
          }
        }]
      });
      return;
    }

    if (absFilePath.startsWith(repoPath)) {
      const filePath = absFilePath.slice(repoPath.length + 1);
      this.quietlySelectItem(filePath, stagingStatus);
      const splitDirection = this.props.config.get('github.viewChangesForCurrentFileDiffPaneSplitDirection');
      const pane = this.props.workspace.getActivePane();

      if (splitDirection === 'right') {
        pane.splitRight();
      } else if (splitDirection === 'down') {
        pane.splitDown();
      }

      const lineNum = editor.getCursorBufferPosition().row + 1;
      const item = await this.props.workspace.open(_changedFileItem["default"].buildURI(filePath, repoPath, stagingStatus), {
        pending: true,
        activatePane: true,
        activateItem: true
      });
      await item.getRealItemPromise();
      await item.getFilePatchLoadedPromise();
      item.goToDiffLine(lineNum);
      item.focus();
    } else {
      throw new Error(`${absFilePath} does not belong to repo ${repoPath}`);
    }
  }

  viewUnstagedChangesForCurrentFile() {
    return this.viewChangesForCurrentFile('unstaged');
  }

  viewStagedChangesForCurrentFile() {
    return this.viewChangesForCurrentFile('staged');
  }

  openFiles(filePaths, repository = this.props.repository) {
    return Promise.all(filePaths.map(filePath => {
      const absolutePath = _path["default"].join(repository.getWorkingDirectoryPath(), filePath);

      return this.props.workspace.open(absolutePath, {
        pending: filePaths.length === 1
      });
    }));
  }

  getUnsavedFiles(filePaths, workdirPath) {
    const isModifiedByPath = new Map();
    this.props.workspace.getTextEditors().forEach(editor => {
      isModifiedByPath.set(editor.getPath(), editor.isModified());
    });
    return filePaths.filter(filePath => {
      const absFilePath = _path["default"].join(workdirPath, filePath);

      return isModifiedByPath.get(absFilePath);
    });
  }

  ensureNoUnsavedFiles(filePaths, message, workdirPath = this.props.repository.getWorkingDirectoryPath()) {
    const unsavedFiles = this.getUnsavedFiles(filePaths, workdirPath).map(filePath => `\`${filePath}\``).join('<br>');

    if (unsavedFiles.length) {
      this.props.notificationManager.addError(message, {
        description: `You have unsaved changes in:<br>${unsavedFiles}.`,
        dismissable: true
      });
      return false;
    } else {
      return true;
    }
  }

  async discardWorkDirChangesForPaths(filePaths) {
    const destructiveAction = () => {
      return this.props.repository.discardWorkDirChangesForPaths(filePaths);
    };

    return await this.props.repository.storeBeforeAndAfterBlobs(filePaths, () => this.ensureNoUnsavedFiles(filePaths, 'Cannot discard changes in selected files.'), destructiveAction);
  }

  async discardLines(multiFilePatch, lines, repository = this.props.repository) {
    // (kuychaco) For now we only support discarding rows for MultiFilePatches that contain a single file patch
    // The only way to access this method from the UI is to be in a ChangedFileItem, which only has a single file patch
    if (multiFilePatch.getFilePatches().length !== 1) {
      return Promise.resolve(null);
    }

    const filePath = multiFilePatch.getFilePatches()[0].getPath();

    const destructiveAction = async () => {
      const discardFilePatch = multiFilePatch.getUnstagePatchForLines(lines);
      await repository.applyPatchToWorkdir(discardFilePatch);
    };

    return await repository.storeBeforeAndAfterBlobs([filePath], () => this.ensureNoUnsavedFiles([filePath], 'Cannot discard lines.', repository.getWorkingDirectoryPath()), destructiveAction, filePath);
  }

  getFilePathsForLastDiscard(partialDiscardFilePath = null) {
    let lastSnapshots = this.props.repository.getLastHistorySnapshots(partialDiscardFilePath);

    if (partialDiscardFilePath) {
      lastSnapshots = lastSnapshots ? [lastSnapshots] : [];
    }

    return lastSnapshots.map(snapshot => snapshot.filePath);
  }

  async undoLastDiscard(partialDiscardFilePath = null, repository = this.props.repository) {
    const filePaths = this.getFilePathsForLastDiscard(partialDiscardFilePath);

    try {
      const results = await repository.restoreLastDiscardInTempFiles(() => this.ensureNoUnsavedFiles(filePaths, 'Cannot undo last discard.'), partialDiscardFilePath);

      if (results.length === 0) {
        return;
      }

      await this.proceedOrPromptBasedOnResults(results, partialDiscardFilePath);
    } catch (e) {
      if (e instanceof _gitShellOutStrategy.GitError && e.stdErr.match(/fatal: Not a valid object name/)) {
        this.cleanUpHistoryForFilePaths(filePaths, partialDiscardFilePath);
      } else {
        // eslint-disable-next-line no-console
        console.error(e);
      }
    }
  }

  async proceedOrPromptBasedOnResults(results, partialDiscardFilePath = null) {
    const conflicts = results.filter(({
      conflict
    }) => conflict);

    if (conflicts.length === 0) {
      await this.proceedWithLastDiscardUndo(results, partialDiscardFilePath);
    } else {
      await this.promptAboutConflicts(results, conflicts, partialDiscardFilePath);
    }
  }

  async promptAboutConflicts(results, conflicts, partialDiscardFilePath = null) {
    const conflictedFiles = conflicts.map(({
      filePath
    }) => `\t${filePath}`).join('\n');
    const choice = this.props.confirm({
      message: 'Undoing will result in conflicts...',
      detailedMessage: `for the following files:\n${conflictedFiles}\n` + 'Would you like to apply the changes with merge conflict markers, ' + 'or open the text with merge conflict markers in a new file?',
      buttons: ['Merge with conflict markers', 'Open in new file', 'Cancel']
    });

    if (choice === 0) {
      await this.proceedWithLastDiscardUndo(results, partialDiscardFilePath);
    } else if (choice === 1) {
      await this.openConflictsInNewEditors(conflicts.map(({
        resultPath
      }) => resultPath));
    }
  }

  cleanUpHistoryForFilePaths(filePaths, partialDiscardFilePath = null) {
    this.props.repository.clearDiscardHistory(partialDiscardFilePath);
    const filePathsStr = filePaths.map(filePath => `\`${filePath}\``).join('<br>');
    this.props.notificationManager.addError('Discard history has expired.', {
      description: `Cannot undo discard for<br>${filePathsStr}<br>Stale discard history has been deleted.`,
      dismissable: true
    });
  }

  async proceedWithLastDiscardUndo(results, partialDiscardFilePath = null) {
    const promises = results.map(async result => {
      const {
        filePath,
        resultPath,
        deleted,
        conflict,
        theirsSha,
        commonBaseSha,
        currentSha
      } = result;

      const absFilePath = _path["default"].join(this.props.repository.getWorkingDirectoryPath(), filePath);

      if (deleted && resultPath === null) {
        await _fsExtra["default"].remove(absFilePath);
      } else {
        await _fsExtra["default"].copy(resultPath, absFilePath);
      }

      if (conflict) {
        await this.props.repository.writeMergeConflictToIndex(filePath, commonBaseSha, currentSha, theirsSha);
      }
    });
    await Promise.all(promises);
    await this.props.repository.popDiscardHistory(partialDiscardFilePath);
  }

  async openConflictsInNewEditors(resultPaths) {
    const editorPromises = resultPaths.map(resultPath => {
      return this.props.workspace.open(resultPath);
    });
    return await Promise.all(editorPromises);
  }

  /*
   * Asynchronously count the conflict markers present in a file specified by full path.
   */
  refreshResolutionProgress(fullPath) {
    const readStream = _fsExtra["default"].createReadStream(fullPath, {
      encoding: 'utf8'
    });

    return new Promise(resolve => {
      _conflict["default"].countFromStream(readStream).then(count => {
        this.props.resolutionProgress.reportMarkerCount(fullPath, count);
      });
    });
  }
  /*
   * Display the credential entry dialog. Return a Promise that will resolve with the provided credentials on accept
   * or reject on cancel.
   */


  promptForCredentials(query) {
    return new Promise((resolve, reject) => {
      this.setState({
        credentialDialogQuery: _objectSpread({}, query, {
          onSubmit: response => this.setState({
            credentialDialogQuery: null
          }, () => resolve(response)),
          onCancel: () => this.setState({
            credentialDialogQuery: null
          }, reject)
        })
      });
    });
  }

}

exports["default"] = RootController;

_defineProperty(RootController, "propTypes", {
  workspace: _propTypes["default"].object.isRequired,
  commandRegistry: _propTypes["default"].object.isRequired,
  deserializers: _propTypes["default"].object.isRequired,
  notificationManager: _propTypes["default"].object.isRequired,
  tooltips: _propTypes["default"].object.isRequired,
  keymaps: _propTypes["default"].object.isRequired,
  grammars: _propTypes["default"].object.isRequired,
  config: _propTypes["default"].object.isRequired,
  project: _propTypes["default"].object.isRequired,
  loginModel: _propTypes["default"].object.isRequired,
  confirm: _propTypes["default"].func.isRequired,
  createRepositoryForProjectPath: _propTypes["default"].func,
  cloneRepositoryForProjectPath: _propTypes["default"].func,
  repository: _propTypes["default"].object.isRequired,
  resolutionProgress: _propTypes["default"].object.isRequired,
  statusBar: _propTypes["default"].object,
  switchboard: _propTypes["default"].instanceOf(_switchboard["default"]),
  startOpen: _propTypes["default"].bool,
  startRevealed: _propTypes["default"].bool,
  pipelineManager: _propTypes["default"].object,
  workdirContextPool: _propTypes2.WorkdirContextPoolPropType.isRequired
});

_defineProperty(RootController, "defaultProps", {
  switchboard: new _switchboard["default"](),
  startOpen: false,
  startRevealed: false
});

class TabTracker {
  constructor(name, {
    getWorkspace,
    uri
  }) {
    (0, _helpers.autobind)(this, 'toggle', 'toggleFocus', 'ensureVisible');
    this.name = name;
    this.getWorkspace = getWorkspace;
    this.uri = uri;
  }

  async toggle() {
    const focusToRestore = document.activeElement;
    let shouldRestoreFocus = false; // Rendered => the dock item is being rendered, whether or not the dock is visible or the item
    //   is visible within its dock.
    // Visible => the item is active and the dock item is active within its dock.

    const wasRendered = this.isRendered();
    const wasVisible = this.isVisible();

    if (!wasRendered || !wasVisible) {
      // Not rendered, or rendered but not an active item in a visible dock.
      await this.reveal();
      shouldRestoreFocus = true;
    } else {
      // Rendered and an active item within a visible dock.
      await this.hide();
      shouldRestoreFocus = false;
    }

    if (shouldRestoreFocus) {
      process.nextTick(() => focusToRestore.focus());
    }
  }

  async toggleFocus() {
    const hadFocus = this.hasFocus();
    await this.ensureVisible();

    if (hadFocus) {
      let workspace = this.getWorkspace();

      if (workspace.getCenter) {
        workspace = workspace.getCenter();
      }

      workspace.getActivePane().activate();
    } else {
      this.focus();
    }
  }

  async ensureVisible() {
    if (!this.isVisible()) {
      await this.reveal();
      return true;
    }

    return false;
  }

  ensureRendered() {
    return this.getWorkspace().open(this.uri, {
      searchAllPanes: true,
      activateItem: false,
      activatePane: false
    });
  }

  reveal() {
    (0, _reporterProxy.incrementCounter)(`${this.name}-tab-open`);
    return this.getWorkspace().open(this.uri, {
      searchAllPanes: true,
      activateItem: true,
      activatePane: true
    });
  }

  hide() {
    (0, _reporterProxy.incrementCounter)(`${this.name}-tab-close`);
    return this.getWorkspace().hide(this.uri);
  }

  focus() {
    this.getComponent().restoreFocus();
  }

  getItem() {
    const pane = this.getWorkspace().paneForURI(this.uri);

    if (!pane) {
      return null;
    }

    const paneItem = pane.itemForURI(this.uri);

    if (!paneItem) {
      return null;
    }

    return paneItem;
  }

  getComponent() {
    const paneItem = this.getItem();

    if (!paneItem) {
      return null;
    }

    if (typeof paneItem.getRealItem !== 'function') {
      return null;
    }

    return paneItem.getRealItem();
  }

  getDOMElement() {
    const paneItem = this.getItem();

    if (!paneItem) {
      return null;
    }

    if (typeof paneItem.getElement !== 'function') {
      return null;
    }

    return paneItem.getElement();
  }

  isRendered() {
    return !!this.getWorkspace().paneForURI(this.uri);
  }

  isVisible() {
    const workspace = this.getWorkspace();
    return workspace.getPaneContainers().filter(container => container === workspace.getCenter() || container.isVisible()).some(container => container.getPanes().some(pane => {
      const item = pane.getActiveItem();
      return item && item.getURI && item.getURI() === this.uri;
    }));
  }

  hasFocus() {
    const root = this.getDOMElement();
    return root && root.contains(document.activeElement);
  }

}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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