"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getNullActionPipelineManager = getNullActionPipelineManager;
exports["default"] = exports.ActionPipeline = void 0;

function partial(fn, ...args) {
  return function wrapped() {
    return fn(...args);
  };
}

class ActionPipeline {
  constructor(actionKey) {
    this.actionKey = actionKey;
    this.middleware = [];
  }

  run(fn, repository, ...args) {
    const pipelineFn = this.middleware.map(middleware => middleware.fn).reduceRight((acc, nextFn) => partial(nextFn, acc, repository, ...args), partial(fn, ...args));
    return pipelineFn(...args);
  }

  addMiddleware(name, fn) {
    if (!name) {
      throw new Error('Middleware must be registered with a unique middleware name');
    }

    if (this.middleware.find(middleware => middleware.name === name)) {
      throw new Error(`A middleware with the name ${name} is already registered`);
    }

    this.middleware.push({
      name,
      fn
    });
  }

}

exports.ActionPipeline = ActionPipeline;

class ActionPipelineManager {
  constructor({
    actionNames
  } = {
    actionNames: []
  }) {
    this.pipelines = new Map();
    this.actionKeys = {};
    actionNames.forEach(actionName => {
      const actionKey = Symbol(actionName);
      this.actionKeys[actionName] = actionKey;
      this.pipelines.set(actionKey, new ActionPipeline(actionKey));
    });
  }

  getPipeline(actionKey) {
    const pipeline = this.pipelines.get(actionKey);

    if (actionKey && pipeline) {
      return pipeline;
    } else {
      throw new Error(`${actionKey} is not a known action`);
    }
  }

}

exports["default"] = ActionPipelineManager;

class NullActionPipeline extends ActionPipeline {
  run(fn, repository, ...args) {
    return fn(...args);
  }

  addMiddleware() {
    throw new Error('Cannot add middleware to a null pipeline');
  }

}

const nullPipeline = new NullActionPipeline();

class NullActionPipelineManager extends ActionPipelineManager {
  getPipeline() {
    return nullPipeline;
  }

}

const nullActionPipelineManager = new NullActionPipelineManager();

function getNullActionPipelineManager() {
  return nullActionPipelineManager;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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