/**
 * <p>See <a href="http://raml.org">http://raml.org</a> for more information about RAML.</p>
 *
 * <p>This parser is at a beta state of development, as part of the API Workbench development cycle (<a href="http://apiworkbench.com">http://apiworkbench.com</a>).</p>
 *
 * <p><a href="https://github.com/raml-org/raml-js-parser-2/blob/master/documentation/GettingStarted.md">Getting Started Guide</a> describes the first steps with the parser.</p>
 *
 * <h2>Installation</h2>
 *
 * <pre><code>git clone https://github.com/raml-org/raml-js-parser-2
 *
 * cd raml-js-parser-2
 *
 * npm install
 *
 * node test/test.js  //here you should observe JSON representation of XKCD API in your console
 *
 * node test/testAsync.js  //same as above but in asynchronous mode
 * </code></pre>
 *
 * <h2>Usage</h2>
 *
 * <ul>
 * <li>For parser usage example refer to <code>test/test.js</code></li>
 * <li>For asynchrounous usage example refer to <code>test/testAsync.js</code></li>
 * </ul>
 **/
import hl = require("../../parser/highLevelAST");
import core = require("../../parser/wrapped-ast/parserCoreApi");
export interface Annotable extends core.BasicNode {
    /**
     * Most of RAML model elements may have attached annotations decribing additional meta data about this element
     **/
    annotations(): AnnotationRef[];
}
export interface ValueType extends core.AttributeNode {
    /**
     * @return JS representation of the node value
     **/
    value(): any;
}
export interface StringType extends ValueType {
    /**
     * @return String representation of the node value
     **/
    value(): string;
}
/**
 * This type currently serves both for absolute and relative urls
 **/
export interface UriTemplate extends StringType {
}
/**
 * This  type describes relative uri templates
 **/
export interface RelativeUriString extends UriTemplate {
}
/**
 * This  type describes absolute uri templates
 **/
export interface FullUriTemplateString extends UriTemplate {
}
export interface StatusCodeString extends StringType {
}
/**
 * This  type describes fixed uris
 **/
export interface FixedUriString extends StringType {
}
export interface ContentType extends StringType {
}
/**
 * [GitHub Flavored Markdown](https://help.github.com/articles/github-flavored-markdown/)
 **/
export interface MarkdownString extends StringType {
}
/**
 * Schema at this moment only two subtypes are supported (json schema and xsd)
 **/
export interface SchemaString extends StringType {
}
/**
 * This sub type of the string represents mime types
 **/
export interface MimeType extends StringType {
}
export interface AnyType extends ValueType {
}
export interface NumberType extends ValueType {
    /**
     * @return Number representation of the node value
     **/
    value(): number;
}
export interface IntegerType extends ValueType {
}
export interface NullType extends ValueType {
}
export interface TimeOnlyType extends ValueType {
}
export interface DateOnlyType extends ValueType {
}
export interface DateTimeOnlyType extends ValueType {
}
export interface DateTimeType extends ValueType {
}
export interface FileType extends ValueType {
}
export interface BooleanType extends ValueType {
    /**
     * @return Boolean representation of the node value
     **/
    value(): boolean;
}
/**
 * Elements to which this Annotation can be applied (enum)
 **/
export interface AnnotationTarget extends ValueType {
}
export interface Reference extends core.AttributeNode {
    /**
     * Returns a structured object if the reference point to one.
     **/
    structuredValue(): TypeInstance;
    /**
     * Returns name of referenced object
     **/
    name(): string;
    /**
     * @return StructuredValue object representing the node value
     **/
    value(): hl.IStructuredValue;
}
export interface TypeInstance {
    /**
     * Array of instance properties
     **/
    properties(): TypeInstanceProperty[];
    /**
     * Whether the type is scalar
     **/
    isScalar(): boolean;
    /**
     * For instances of scalar types returns scalar value
     **/
    value(): any;
    /**
     * Indicates whether the instance is array
     **/
    isArray(): boolean;
    /**
     * Returns components of array instances
     **/
    items(): TypeInstance[];
}
export interface TypeInstanceProperty {
    /**
     * Property name
     **/
    name(): string;
    /**
     * Property value
     **/
    value(): TypeInstance;
    /**
     * Array of values if property value is array
     **/
    values(): TypeInstance[];
    /**
     * Whether property has array as value
     **/
    isArray(): boolean;
}
export interface TraitRef extends Reference {
    /**
     * Returns referenced trait
     **/
    trait(): Trait;
}
export interface Operation extends Annotable {
    /**
     * An APIs resources MAY be filtered (to return a subset of results) or altered (such as transforming  a response body from JSON to XML format) by the use of query strings. If the resource or its method supports a query string, the query string MUST be defined by the queryParameters property
     **/
    queryParameters(): TypeDeclaration[];
    /**
     * Headers that allowed at this position
     **/
    headers(): TypeDeclaration[];
    /**
     * Specifies the query string needed by this method. Mutually exclusive with queryParameters.
     **/
    queryString(): TypeDeclaration;
    /**
     * Information about the expected responses to a request
     **/
    responses(): Response[];
}
export interface TypeDeclaration extends Annotable {
    /**
     * Type name for top level types. For properties and parameters -- property o parameter name, respectively. For bodies -- media type.
     **/
    name(): string;
    /**
     * The displayName attribute specifies the type display name. It is a friendly name used only for  display or documentation purposes. If displayName is not specified, it defaults to the element's key (the name of the property itself).
     **/
    displayName(): string;
    /**
     * When extending from a type you can define new facets (which can then be set to concrete values by subtypes).
     **/
    facets(): TypeDeclaration[];
    /**
     * Location of the parameter (can not be edited by user)
     **/
    location(): ModelLocation;
    /**
     * Kind of location
     **/
    locationKind(): LocationKind;
    /**
     * Provides default value for a property
     **/
    "default"(): any;
    /**
     * An example of this type instance represented as string or yaml map/sequence. This can be used, e.g., by documentation generators to generate sample values for an object of this type. Cannot be present if the examples property is present.
     **/
    example(): ExampleSpec;
    /**
     * An example of this type instance represented as string. This can be used, e.g., by documentation generators to generate sample values for an object of this type. Cannot be present if the example property is present.
     **/
    examples(): ExampleSpec[];
    /**
     * For property or parameter states if it is required.
     **/
    required(): boolean;
    /**
     * A longer, human-friendly description of the type
     **/
    description(): MarkdownString;
    xml(): XMLFacetInfo;
    /**
     * Restrictions on where annotations of this type can be applied. If this property is specified, annotations of this type may only be applied on a property corresponding to one of the target names specified as the value of this property.
     **/
    allowedTargets(): AnnotationTarget[];
    /**
     * Whether the type represents annotation
     **/
    isAnnotation(): boolean;
    /**
     * Most of RAML model elements may have attached annotations decribing additional meta data about this element
     **/
    annotations(): AnnotationRef[];
    /**
     * Returns facets fixed by the type. Value is an object with properties named after facets fixed. Value of each property is a value of the corresponding facet.
     **/
    fixedFacets(): TypeInstance;
    /**
     * Inlined supertype definition.
     **/
    structuredType(): TypeInstance;
    /**
     * For types defined in traits or resource types returns object representation of parametrized properties
     **/
    parametrizedProperties(): TypeInstance;
    /**
     * Runtime representation of type represented by this AST node
     **/
    runtimeType(): hl.ITypeDefinition;
    /**
     * validate an instance against type
     **/
    validateInstance(value: any): string[];
    /**
     * validate an instance against type
     **/
    validateInstanceWithDetailedStatuses(value: any): any;
    /**
     * A base type which the current type extends, or more generally a type expression.
     **/
    "type"(): string[];
    /**
     * A base type which the current type extends, or more generally a type expression.
     **/
    schema(): string[];
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): TypeDeclarationScalarsAnnotations;
}
export interface ModelLocation extends core.AbstractWrapperNode {
}
export interface LocationKind extends core.AbstractWrapperNode {
}
export interface ExampleSpec extends Annotable {
    /**
     * String representation of example
     **/
    value(): any;
    /**
     * By default, examples are validated against any type declaration. Set this to false to allow examples that need not validate.
     **/
    strict(): boolean;
    /**
     * Example identifier, if specified
     **/
    name(): string;
    /**
     * An alternate, human-friendly name for the example
     **/
    displayName(): string;
    /**
     * A longer, human-friendly description of the example
     **/
    description(): MarkdownString;
    /**
     * Most of RAML model elements may have attached annotations decribing additional meta data about this element
     **/
    annotations(): AnnotationRef[];
    /**
     * Returns object representation of example, if possible
     **/
    structuredValue(): TypeInstance;
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): ExampleSpecScalarsAnnotations;
}
export interface UsesDeclaration extends Annotable {
    /**
     * Name prefix (without dot) used to refer imported declarations
     **/
    key(): string;
    /**
     * Content of the schema
     **/
    value(): string;
    /**
     * Returns the root node of the AST, uses statement refers.
     **/
    ast(): Library;
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): UsesDeclarationScalarsAnnotations;
}
/**
 * UsesDeclaration scalar properties annotations accessor
 **/
export interface UsesDeclarationScalarsAnnotations {
    /**
     * UsesDeclaration.value annotations
     **/
    value(): AnnotationRef[];
}
/**
 * ExampleSpec scalar properties annotations accessor
 **/
export interface ExampleSpecScalarsAnnotations {
    /**
     * ExampleSpec.strict annotations
     **/
    strict(): AnnotationRef[];
    /**
     * ExampleSpec.displayName annotations
     **/
    displayName(): AnnotationRef[];
    /**
     * ExampleSpec.description annotations
     **/
    description(): AnnotationRef[];
}
export interface XMLFacetInfo extends Annotable {
    /**
     * If attribute is set to true, a type instance should be serialized as an XML attribute. It can only be true for scalar types.
     **/
    attribute(): boolean;
    /**
     * If wrapped is set to true, a type instance should be wrapped in its own XML element. It can not be true for scalar types and it can not be true at the same moment when attribute is true.
     **/
    wrapped(): boolean;
    /**
     * Allows to override the name of the XML element or XML attribute in it's XML representation.
     **/
    name(): string;
    /**
     * Allows to configure the name of the XML namespace.
     **/
    namespace(): string;
    /**
     * Allows to configure the prefix which will be used during serialization to XML.
     **/
    prefix(): string;
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): XMLFacetInfoScalarsAnnotations;
}
/**
 * XMLFacetInfo scalar properties annotations accessor
 **/
export interface XMLFacetInfoScalarsAnnotations {
    /**
     * XMLFacetInfo.attribute annotations
     **/
    attribute(): AnnotationRef[];
    /**
     * XMLFacetInfo.wrapped annotations
     **/
    wrapped(): AnnotationRef[];
    /**
     * XMLFacetInfo.name annotations
     **/
    name(): AnnotationRef[];
    /**
     * XMLFacetInfo.namespace annotations
     **/
    namespace(): AnnotationRef[];
    /**
     * XMLFacetInfo.prefix annotations
     **/
    prefix(): AnnotationRef[];
}
export interface ArrayTypeDeclaration extends TypeDeclaration {
    /**
     * Should items in array be unique
     **/
    uniqueItems(): boolean;
    /**
     * Minimum amount of items in array
     **/
    minItems(): number;
    /**
     * Maximum amount of items in array
     **/
    maxItems(): number;
    /**
     * Inlined component type definition
     **/
    structuredItems(): TypeInstance;
    /**
     * Anonymous type declaration defined by "items" keyword.
     * If no "items" is defined explicitly, this one is null.
     **/
    items(): string[];
    /**
     * Returns anonymous type defined by "items" keyword, or a component type if declaration can be found.
     * Does not resolve type expressions. Only returns component type declaration if it is actually defined
     * somewhere in AST.
     **/
    findComponentTypeDeclaration(): TypeDeclaration;
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): ArrayTypeDeclarationScalarsAnnotations;
}
/**
 * TypeDeclaration scalar properties annotations accessor
 **/
export interface TypeDeclarationScalarsAnnotations {
    /**
     * TypeDeclaration.displayName annotations
     **/
    displayName(): AnnotationRef[];
    /**
     * TypeDeclaration.schema annotations
     **/
    schema(): AnnotationRef[][];
    /**
     * TypeDeclaration.type annotations
     **/
    "type"(): AnnotationRef[][];
    /**
     * TypeDeclaration.location annotations
     **/
    location(): AnnotationRef[];
    /**
     * TypeDeclaration.locationKind annotations
     **/
    locationKind(): AnnotationRef[];
    /**
     * TypeDeclaration.default annotations
     **/
    "default"(): AnnotationRef[];
    /**
     * TypeDeclaration.required annotations
     **/
    required(): AnnotationRef[];
    /**
     * TypeDeclaration.description annotations
     **/
    description(): AnnotationRef[];
    /**
     * TypeDeclaration.allowedTargets annotations
     **/
    allowedTargets(): AnnotationRef[][];
    /**
     * TypeDeclaration.isAnnotation annotations
     **/
    isAnnotation(): AnnotationRef[];
}
/**
 * ArrayTypeDeclaration scalar properties annotations accessor
 **/
export interface ArrayTypeDeclarationScalarsAnnotations extends TypeDeclarationScalarsAnnotations {
    /**
     * ArrayTypeDeclaration.uniqueItems annotations
     **/
    uniqueItems(): AnnotationRef[];
    /**
     * ArrayTypeDeclaration.items annotations
     **/
    items(): AnnotationRef[][];
    /**
     * ArrayTypeDeclaration.minItems annotations
     **/
    minItems(): AnnotationRef[];
    /**
     * ArrayTypeDeclaration.maxItems annotations
     **/
    maxItems(): AnnotationRef[];
}
export interface UnionTypeDeclaration extends TypeDeclaration {
}
export interface ObjectTypeDeclaration extends TypeDeclaration {
    /**
     * The properties that instances of this type may or must have.
     **/
    properties(): TypeDeclaration[];
    /**
     * The minimum number of properties allowed for instances of this type.
     **/
    minProperties(): number;
    /**
     * The maximum number of properties allowed for instances of this type.
     **/
    maxProperties(): number;
    /**
     * A Boolean that indicates if an object instance has additional properties.
     **/
    additionalProperties(): boolean;
    /**
     * Type property name to be used as discriminator, or boolean
     **/
    discriminator(): string;
    /**
     * The value of discriminator for the type.
     **/
    discriminatorValue(): string;
    enum(): any[];
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): ObjectTypeDeclarationScalarsAnnotations;
}
/**
 * ObjectTypeDeclaration scalar properties annotations accessor
 **/
export interface ObjectTypeDeclarationScalarsAnnotations extends TypeDeclarationScalarsAnnotations {
    /**
     * ObjectTypeDeclaration.minProperties annotations
     **/
    minProperties(): AnnotationRef[];
    /**
     * ObjectTypeDeclaration.maxProperties annotations
     **/
    maxProperties(): AnnotationRef[];
    /**
     * ObjectTypeDeclaration.additionalProperties annotations
     **/
    additionalProperties(): AnnotationRef[];
    /**
     * ObjectTypeDeclaration.discriminator annotations
     **/
    discriminator(): AnnotationRef[];
    /**
     * ObjectTypeDeclaration.discriminatorValue annotations
     **/
    discriminatorValue(): AnnotationRef[];
    /**
     * ObjectTypeDeclaration.enum annotations
     **/
    enum(): AnnotationRef[][];
}
/**
 * Value must be a string
 **/
export interface StringTypeDeclaration extends TypeDeclaration {
    /**
     * Regular expression that this string should path
     **/
    pattern(): string;
    /**
     * Minimum length of the string
     **/
    minLength(): number;
    /**
     * Maximum length of the string
     **/
    maxLength(): number;
    /**
     * (Optional, applicable only for parameters of type string) The enum attribute provides an enumeration of the parameter's valid values. This MUST be an array. If the enum attribute is defined, API clients and servers MUST verify that a parameter's value matches a value in the enum array. If there is no matching value, the clients and servers MUST treat this as an error.
     **/
    enum(): string[];
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): StringTypeDeclarationScalarsAnnotations;
}
/**
 * StringTypeDeclaration scalar properties annotations accessor
 **/
export interface StringTypeDeclarationScalarsAnnotations extends TypeDeclarationScalarsAnnotations {
    /**
     * StringTypeDeclaration.pattern annotations
     **/
    pattern(): AnnotationRef[];
    /**
     * StringTypeDeclaration.minLength annotations
     **/
    minLength(): AnnotationRef[];
    /**
     * StringTypeDeclaration.maxLength annotations
     **/
    maxLength(): AnnotationRef[];
    /**
     * StringTypeDeclaration.enum annotations
     **/
    enum(): AnnotationRef[][];
}
/**
 * Value must be a boolean
 **/
export interface BooleanTypeDeclaration extends TypeDeclaration {
    enum(): boolean[];
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): BooleanTypeDeclarationScalarsAnnotations;
}
/**
 * BooleanTypeDeclaration scalar properties annotations accessor
 **/
export interface BooleanTypeDeclarationScalarsAnnotations extends TypeDeclarationScalarsAnnotations {
    /**
     * BooleanTypeDeclaration.enum annotations
     **/
    enum(): AnnotationRef[][];
}
/**
 * Value MUST be a number. Indicate floating point numbers as defined by YAML.
 **/
export interface NumberTypeDeclaration extends TypeDeclaration {
    /**
     * (Optional, applicable only for parameters of type number or integer) The minimum attribute specifies the parameter's minimum value.
     **/
    minimum(): number;
    /**
     * (Optional, applicable only for parameters of type number or integer) The maximum attribute specifies the parameter's maximum value.
     **/
    maximum(): number;
    /**
     * (Optional, applicable only for parameters of type string) The enum attribute provides an enumeration of the parameter's valid values. This MUST be an array. If the enum attribute is defined, API clients and servers MUST verify that a parameter's value matches a value in the enum array. If there is no matching value, the clients and servers MUST treat this as an error.
     **/
    enum(): number[];
    /**
     * Value format
     **/
    format(): string;
    /**
     * A numeric instance is valid against "multipleOf" if the result of the division of the instance by this keyword's value is an integer.
     **/
    multipleOf(): number;
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): NumberTypeDeclarationScalarsAnnotations;
}
/**
 * Value MUST be a integer.
 **/
export interface IntegerTypeDeclaration extends NumberTypeDeclaration {
    /**
     * Value format
     **/
    format(): string;
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): IntegerTypeDeclarationScalarsAnnotations;
}
/**
 * NumberTypeDeclaration scalar properties annotations accessor
 **/
export interface NumberTypeDeclarationScalarsAnnotations extends TypeDeclarationScalarsAnnotations {
    /**
     * NumberTypeDeclaration.minimum annotations
     **/
    minimum(): AnnotationRef[];
    /**
     * NumberTypeDeclaration.maximum annotations
     **/
    maximum(): AnnotationRef[];
    /**
     * NumberTypeDeclaration.enum annotations
     **/
    enum(): AnnotationRef[][];
    /**
     * NumberTypeDeclaration.format annotations
     **/
    format(): AnnotationRef[];
    /**
     * NumberTypeDeclaration.multipleOf annotations
     **/
    multipleOf(): AnnotationRef[];
}
/**
 * IntegerTypeDeclaration scalar properties annotations accessor
 **/
export interface IntegerTypeDeclarationScalarsAnnotations extends NumberTypeDeclarationScalarsAnnotations {
    /**
     * IntegerTypeDeclaration.format annotations
     **/
    format(): AnnotationRef[];
}
/**
 * the "full-date" notation of RFC3339, namely yyyy-mm-dd (no implications about time or timezone-offset)
 **/
export interface DateOnlyTypeDeclaration extends TypeDeclaration {
}
/**
 * the "partial-time" notation of RFC3339, namely hh:mm:ss[.ff...] (no implications about date or timezone-offset)
 **/
export interface TimeOnlyTypeDeclaration extends TypeDeclaration {
}
/**
 * combined date-only and time-only with a separator of "T", namely yyyy-mm-ddThh:mm:ss[.ff...] (no implications about timezone-offset)
 **/
export interface DateTimeOnlyTypeDeclaration extends TypeDeclaration {
}
/**
 * a timestamp, either in the "date-time" notation of RFC3339, if format is omitted or is set to rfc3339, or in the format defined in RFC2616, if format is set to rfc2616.
 **/
export interface DateTimeTypeDeclaration extends TypeDeclaration {
    /**
     * Format used for this date time rfc3339 or rfc2616
     **/
    format(): string;
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): DateTimeTypeDeclarationScalarsAnnotations;
}
/**
 * DateTimeTypeDeclaration scalar properties annotations accessor
 **/
export interface DateTimeTypeDeclarationScalarsAnnotations extends TypeDeclarationScalarsAnnotations {
    /**
     * DateTimeTypeDeclaration.format annotations
     **/
    format(): AnnotationRef[];
}
/**
 * (Applicable only to Form properties) Value is a file. Client generators SHOULD use this type to handle file uploads correctly.
 **/
export interface FileTypeDeclaration extends TypeDeclaration {
    /**
     * A list of valid content-type strings for the file. The file type * /* should be a valid value.
     **/
    fileTypes(): ContentType[];
    /**
     * The minLength attribute specifies the parameter value's minimum number of bytes.
     **/
    minLength(): number;
    /**
     * The maxLength attribute specifies the parameter value's maximum number of bytes.
     **/
    maxLength(): number;
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): FileTypeDeclarationScalarsAnnotations;
}
/**
 * FileTypeDeclaration scalar properties annotations accessor
 **/
export interface FileTypeDeclarationScalarsAnnotations extends TypeDeclarationScalarsAnnotations {
    /**
     * FileTypeDeclaration.fileTypes annotations
     **/
    fileTypes(): AnnotationRef[][];
    /**
     * FileTypeDeclaration.minLength annotations
     **/
    minLength(): AnnotationRef[];
    /**
     * FileTypeDeclaration.maxLength annotations
     **/
    maxLength(): AnnotationRef[];
}
export interface Response extends Annotable {
    /**
     * Responses MUST be a map of one or more HTTP status codes, where each status code itself is a map that describes that status code.
     **/
    code(): StatusCodeString;
    /**
     * Detailed information about any response headers returned by this method
     **/
    headers(): TypeDeclaration[];
    /**
     * The body of the response: a body declaration
     **/
    body(): TypeDeclaration[];
    /**
     * A longer, human-friendly description of the response
     **/
    description(): MarkdownString;
    /**
     * Most of RAML model elements may have attached annotations decribing additional meta data about this element
     **/
    annotations(): AnnotationRef[];
    /**
     * For responses defined in traits or resource types returns object representation of parametrized properties
     **/
    parametrizedProperties(): TypeInstance;
    /**
     * true for codes < 400 and false otherwise
     **/
    isOkRange(): boolean;
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): ResponseScalarsAnnotations;
}
/**
 * Response scalar properties annotations accessor
 **/
export interface ResponseScalarsAnnotations {
    /**
     * Response.description annotations
     **/
    description(): AnnotationRef[];
}
export interface SecuritySchemePart extends Operation {
    /**
     * Annotations to be applied to this security scheme part. Annotations are any property whose key begins with "(" and ends with ")" and whose name (the part between the beginning and ending parentheses) is a declared annotation name.
     **/
    annotations(): AnnotationRef[];
}
export interface MethodBase extends Operation {
    /**
     * Some method verbs expect the resource to be sent as a request body. For example, to create a resource, the request must include the details of the resource to create. Resources CAN have alternate representations. For example, an API might support both JSON and XML representations. A method's body is defined in the body property as a hashmap, in which the key MUST be a valid media type.
     **/
    body(): TypeDeclaration[];
    /**
     * A method can override the protocols specified in the resource or at the API root, by employing this property.
     **/
    protocols(): string[];
    /**
     * Instantiation of applyed traits
     **/
    is(): TraitRef[];
    /**
     * securityScheme may also be applied to a resource by using the securedBy key, which is equivalent to applying the securityScheme to all methods that may be declared, explicitly or implicitly, by defining the resourceTypes or traits property for that resource. To indicate that the method may be called without applying any securityScheme, the method may be annotated with the null securityScheme.
     **/
    securedBy(): SecuritySchemeRef[];
    description(): MarkdownString;
    displayName(): string;
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): MethodBaseScalarsAnnotations;
}
export interface SecuritySchemeRef extends Reference {
    /**
     * Returns the name of security scheme, this reference refers to.
     **/
    securitySchemeName(): string;
    /**
     * Returns AST node of security scheme, this reference refers to, or null.
     **/
    securityScheme(): AbstractSecurityScheme;
}
/**
 * Declares globally referable security scheme definition
 **/
export interface AbstractSecurityScheme extends Annotable {
    /**
     * Name of the security scheme
     **/
    name(): string;
    /**
     * The securitySchemes property MUST be used to specify an API's security mechanisms, including the required settings and the authentication methods that the API supports. one authentication method is allowed if the API supports them.
     **/
    "type"(): string;
    /**
     * The description MAY be used to describe a securityScheme.
     **/
    description(): MarkdownString;
    /**
     * A description of the request components related to Security that are determined by the scheme: the headers, query parameters or responses. As a best practice, even for standard security schemes, API designers SHOULD describe these properties of security schemes. Including the security scheme description completes an API documentation.
     **/
    describedBy(): SecuritySchemePart;
    /**
     * The displayName attribute specifies the security scheme display name. It is a friendly name used only for  display or documentation purposes. If displayName is not specified, it defaults to the element's key (the name of the property itself).
     **/
    displayName(): string;
    /**
     * The settings attribute MAY be used to provide security scheme-specific information. The required attributes vary depending on the type of security scheme is being declared. It describes the minimum set of properties which any processing application MUST provide and validate if it chooses to implement the security scheme. Processing applications MAY choose to recognize other properties for things such as token lifetime, preferred cryptographic algorithms, and more.
     **/
    settings(): SecuritySchemeSettings;
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): AbstractSecuritySchemeScalarsAnnotations;
}
export interface SecuritySchemeSettings extends Annotable {
}
export interface OAuth1SecuritySchemeSettings extends SecuritySchemeSettings {
    /**
     * The URI of the Temporary Credential Request endpoint as defined in RFC5849 Section 2.1
     **/
    requestTokenUri(): FixedUriString;
    /**
     * The URI of the Resource Owner Authorization endpoint as defined in RFC5849 Section 2.2
     **/
    authorizationUri(): FixedUriString;
    /**
     * The URI of the Token Request endpoint as defined in RFC5849 Section 2.3
     **/
    tokenCredentialsUri(): FixedUriString;
    /**
     * List of the signature methods used by the server. Available methods: HMAC-SHA1, RSA-SHA1, PLAINTEXT
     **/
    signatures(): string[];
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): OAuth1SecuritySchemeSettingsScalarsAnnotations;
}
/**
 * OAuth1SecuritySchemeSettings scalar properties annotations accessor
 **/
export interface OAuth1SecuritySchemeSettingsScalarsAnnotations {
    /**
     * OAuth1SecuritySchemeSettings.requestTokenUri annotations
     **/
    requestTokenUri(): AnnotationRef[];
    /**
     * OAuth1SecuritySchemeSettings.authorizationUri annotations
     **/
    authorizationUri(): AnnotationRef[];
    /**
     * OAuth1SecuritySchemeSettings.tokenCredentialsUri annotations
     **/
    tokenCredentialsUri(): AnnotationRef[];
    /**
     * OAuth1SecuritySchemeSettings.signatures annotations
     **/
    signatures(): AnnotationRef[][];
}
export interface OAuth2SecuritySchemeSettings extends SecuritySchemeSettings {
    /**
     * The URI of the Token Endpoint as defined in RFC6749 Section 3.2. Not required forby implicit grant type.
     **/
    accessTokenUri(): FixedUriString;
    /**
     * The URI of the Authorization Endpoint as defined in RFC6749 Section 3.1. Required forby authorization_code and implicit grant types.
     **/
    authorizationUri(): FixedUriString;
    /**
     * A list of the Authorization grants supported by the API as defined in RFC6749 Sections 4.1, 4.2, 4.3 and 4.4, can be any of: authorization_code, password, client_credentials, implicit, or any absolute url.
     **/
    authorizationGrants(): string[];
    /**
     * A list of scopes supported by the security scheme as defined in RFC6749 Section 3.3
     **/
    scopes(): string[];
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): OAuth2SecuritySchemeSettingsScalarsAnnotations;
}
/**
 * OAuth2SecuritySchemeSettings scalar properties annotations accessor
 **/
export interface OAuth2SecuritySchemeSettingsScalarsAnnotations {
    /**
     * OAuth2SecuritySchemeSettings.accessTokenUri annotations
     **/
    accessTokenUri(): AnnotationRef[];
    /**
     * OAuth2SecuritySchemeSettings.authorizationUri annotations
     **/
    authorizationUri(): AnnotationRef[];
    /**
     * OAuth2SecuritySchemeSettings.authorizationGrants annotations
     **/
    authorizationGrants(): AnnotationRef[][];
    /**
     * OAuth2SecuritySchemeSettings.scopes annotations
     **/
    scopes(): AnnotationRef[][];
}
/**
 * Declares globally referable security scheme definition
 **/
export interface OAuth2SecurityScheme extends AbstractSecurityScheme {
    settings(): OAuth2SecuritySchemeSettings;
}
/**
 * Declares globally referable security scheme definition
 **/
export interface OAuth1SecurityScheme extends AbstractSecurityScheme {
    settings(): OAuth1SecuritySchemeSettings;
}
/**
 * Declares globally referable security scheme definition
 **/
export interface PassThroughSecurityScheme extends AbstractSecurityScheme {
    settings(): SecuritySchemeSettings;
}
/**
 * Declares globally referable security scheme definition
 **/
export interface BasicSecurityScheme extends AbstractSecurityScheme {
}
/**
 * Declares globally referable security scheme definition
 **/
export interface DigestSecurityScheme extends AbstractSecurityScheme {
}
/**
 * Declares globally referable security scheme definition
 **/
export interface CustomSecurityScheme extends AbstractSecurityScheme {
}
/**
 * AbstractSecurityScheme scalar properties annotations accessor
 **/
export interface AbstractSecuritySchemeScalarsAnnotations {
    /**
     * AbstractSecurityScheme.type annotations
     **/
    "type"(): AnnotationRef[];
    /**
     * AbstractSecurityScheme.description annotations
     **/
    description(): AnnotationRef[];
    /**
     * AbstractSecurityScheme.displayName annotations
     **/
    displayName(): AnnotationRef[];
}
export interface Method extends MethodBase {
    /**
     * Method that can be called
     **/
    method(): string;
    /**
     * The displayName attribute specifies the method display name. It is a friendly name used only for  display or documentation purposes. If displayName is not specified, it defaults to the element's key (the name of the property itself).
     **/
    displayName(): string;
    /**
     * For types defined in resource types returns object representation of parametrized properties
     **/
    parametrizedProperties(): TypeInstance;
    /**
     * For methods of Resources returns parent resource. For methods of ResourceTypes returns null.
     **/
    parentResource(): Resource;
    /**
     * Api owning the resource as a sibling
     **/
    ownerApi(): Api;
    /**
     * For methods of Resources: `{parent Resource relative path} {methodName}`.
     * For methods of ResourceTypes: `{parent ResourceType name} {methodName}`.
     * For other methods throws Exception.
     **/
    methodId(): string;
    /**
     * Returns security schemes, resource or method is secured with. If no security schemes are set at resource or method level,
     * returns schemes defined with `securedBy` at API level.
     * @deprecated
     **/
    allSecuredBy(): SecuritySchemeRef[];
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): MethodScalarsAnnotations;
}
/**
 * MethodBase scalar properties annotations accessor
 **/
export interface MethodBaseScalarsAnnotations {
    /**
     * MethodBase.protocols annotations
     **/
    protocols(): AnnotationRef[][];
    /**
     * MethodBase.is annotations
     **/
    is(): AnnotationRef[][];
    /**
     * MethodBase.securedBy annotations
     **/
    securedBy(): AnnotationRef[][];
    /**
     * MethodBase.description annotations
     **/
    description(): AnnotationRef[];
    /**
     * MethodBase.displayName annotations
     **/
    displayName(): AnnotationRef[];
}
/**
 * Method scalar properties annotations accessor
 **/
export interface MethodScalarsAnnotations extends MethodBaseScalarsAnnotations {
    /**
     * Method.displayName annotations
     **/
    displayName(): AnnotationRef[];
}
export interface Trait extends MethodBase {
    /**
     * Name of the trait
     **/
    name(): string;
    /**
     * Instructions on how and when the trait should be used.
     **/
    usage(): string;
    /**
     * The displayName attribute specifies the trait display name. It is a friendly name used only for  display or documentation purposes. If displayName is not specified, it defaults to the element's key (the name of the property itself).
     **/
    displayName(): string;
    /**
     * Returns object representation of parametrized properties of the trait
     **/
    parametrizedProperties(): TypeInstance;
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): TraitScalarsAnnotations;
}
/**
 * Trait scalar properties annotations accessor
 **/
export interface TraitScalarsAnnotations extends MethodBaseScalarsAnnotations {
    /**
     * Trait.usage annotations
     **/
    usage(): AnnotationRef[];
    /**
     * Trait.displayName annotations
     **/
    displayName(): AnnotationRef[];
}
export interface ResourceTypeRef extends Reference {
    /**
     * Returns referenced resource type
     **/
    resourceType(): ResourceType;
}
export interface ResourceBase extends Annotable {
    /**
     * Methods that are part of this resource type definition
     **/
    methods(): Method[];
    /**
     * A list of the traits to apply to all methods declared (implicitly or explicitly) for this resource. Individual methods may override this declaration
     **/
    is(): TraitRef[];
    /**
     * The resource type which this resource inherits.
     **/
    "type"(): ResourceTypeRef;
    description(): MarkdownString;
    /**
     * The security schemes that apply to all methods declared (implicitly or explicitly) for this resource.
     **/
    securedBy(): SecuritySchemeRef[];
    /**
     * Retrieve an ordered list of all uri parameters including those which are not described in the `uriParameters` node.
     * Consider a fragment of RAML specification:
     * ```yaml
     * /resource/{objectId}/{propertyId}:
     * uriParameters:
     * objectId:
     * ```
     * Here `propertyId` uri parameter is not described in the `uriParameters` node,
     * but it is among Resource.uriParameters().
     **/
    uriParameters(): TypeDeclaration[];
    /**
     * Retrieve an ordered list of all uri parameters including those which are not described in the `uriParameters` node.
     * Consider a fragment of RAML specification:
     * ```yaml
     * /resource/{objectId}/{propertyId}:
     * uriParameters:
     * objectId:
     * ```
     * Here `propertyId` uri parameter is not described in the `uriParameters` node,
     * but it is among Resource.allUriParameters().
     * @deprecated
     **/
    allUriParameters(): TypeDeclaration[];
    /**
     * Returns security schemes, resource or method is secured with. If no security schemes are set at resource or method level,
     * returns schemes defined with `securedBy` at API level.
     * @deprecated
     **/
    allSecuredBy(): SecuritySchemeRef[];
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): ResourceBaseScalarsAnnotations;
}
export interface Resource extends ResourceBase {
    /**
     * Relative URL of this resource from the parent resource
     **/
    relativeUri(): RelativeUriString;
    /**
     * The displayName attribute specifies the resource display name. It is a friendly name used only for  display or documentation purposes. If displayName is not specified, it defaults to the element's key (the name of the property itself).
     **/
    displayName(): string;
    /**
     * A nested resource is identified as any property whose name begins with a slash ("/") and is therefore treated as a relative URI.
     **/
    resources(): Resource[];
    /**
     * A longer, human-friendly description of the resource.
     **/
    description(): MarkdownString;
    /**
     * Most of RAML model elements may have attached annotations decribing additional meta data about this element
     **/
    annotations(): AnnotationRef[];
    /**
     * Path relative to API root
     **/
    completeRelativeUri(): string;
    /**
     * baseUri of owning Api concatenated with completeRelativeUri
     **/
    absoluteUri(): string;
    /**
     * Parent resource for non top level resources
     **/
    parentResource(): Resource;
    /**
     * Get child resource by its relative path
     **/
    childResource(relPath: string): Resource;
    /**
     * Get child method by its name
     **/
    childMethod(method: string): Method[];
    /**
     * Api owning the resource as a sibling
     **/
    ownerApi(): Api;
    /**
     * Retrieve an ordered list of all absolute uri parameters. Returns a union of `Api.baseUriParameters()`
     * for `Api` owning the `Resource` and `Resource.uriParameters()`.
     **/
    absoluteUriParameters(): TypeDeclaration[];
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): ResourceScalarsAnnotations;
}
/**
 * ResourceBase scalar properties annotations accessor
 **/
export interface ResourceBaseScalarsAnnotations {
    /**
     * ResourceBase.is annotations
     **/
    is(): AnnotationRef[][];
    /**
     * ResourceBase.type annotations
     **/
    "type"(): AnnotationRef[];
    /**
     * ResourceBase.description annotations
     **/
    description(): AnnotationRef[];
    /**
     * ResourceBase.securedBy annotations
     **/
    securedBy(): AnnotationRef[][];
}
/**
 * Resource scalar properties annotations accessor
 **/
export interface ResourceScalarsAnnotations extends ResourceBaseScalarsAnnotations {
    /**
     * Resource.displayName annotations
     **/
    displayName(): AnnotationRef[];
    /**
     * Resource.description annotations
     **/
    description(): AnnotationRef[];
}
export interface ResourceType extends ResourceBase {
    /**
     * The displayName attribute specifies the resource type display name. It is a friendly name used only for  display or documentation purposes. If displayName is not specified, it defaults to the element's key (the name of the property itself).
     **/
    displayName(): string;
    /**
     * Name of the resource type
     **/
    name(): string;
    /**
     * Instructions on how and when the resource type should be used.
     **/
    usage(): string;
    /**
     * Returns object representation of parametrized properties of the resource type
     **/
    parametrizedProperties(): TypeInstance;
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): ResourceTypeScalarsAnnotations;
}
/**
 * ResourceType scalar properties annotations accessor
 **/
export interface ResourceTypeScalarsAnnotations extends ResourceBaseScalarsAnnotations {
    /**
     * ResourceType.displayName annotations
     **/
    displayName(): AnnotationRef[];
    /**
     * ResourceType.usage annotations
     **/
    usage(): AnnotationRef[];
}
/**
 * Annotations allow you to attach information to your API
 **/
export interface AnnotationRef extends Reference {
    /**
     * Returns referenced annotation
     **/
    annotation(): TypeDeclaration;
}
export interface DocumentationItem extends Annotable {
    /**
     * Title of documentation section
     **/
    title(): string;
    /**
     * Content of documentation section
     **/
    content(): MarkdownString;
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): DocumentationItemScalarsAnnotations;
}
/**
 * DocumentationItem scalar properties annotations accessor
 **/
export interface DocumentationItemScalarsAnnotations {
    /**
     * DocumentationItem.title annotations
     **/
    title(): AnnotationRef[];
    /**
     * DocumentationItem.content annotations
     **/
    content(): AnnotationRef[];
}
export interface FragmentDeclaration extends Annotable {
    uses(): UsesDeclaration[];
}
export interface LibraryBase extends FragmentDeclaration {
    /**
     * Alias for the equivalent "types" property, for compatibility with RAML 0.8. Deprecated - API definitions should use the "types" property, as the "schemas" alias for that property name may be removed in a future RAML version. The "types" property allows for XML and JSON schemas.
     **/
    schemas(): TypeDeclaration[];
    /**
     * Declarations of (data) types for use within this API
     **/
    types(): TypeDeclaration[];
    /**
     * Declarations of annotation types for use by annotations
     **/
    annotationTypes(): TypeDeclaration[];
    /**
     * Declarations of security schemes for use within this API.
     **/
    securitySchemes(): AbstractSecurityScheme[];
    /**
     * Retrieve all traits including those defined in libraries
     **/
    traits(): Trait[];
    /**
     * Retrieve all traits including those defined in libraries
     * @deprecated
     **/
    allTraits(): Trait[];
    /**
     * Retrieve all resource types including those defined in libraries
     **/
    resourceTypes(): ResourceType[];
    /**
     * Retrieve all resource types including those defined in libraries
     * @deprecated
     **/
    allResourceTypes(): ResourceType[];
}
export interface Library extends LibraryBase {
    /**
     * contains description of why library exist
     **/
    usage(): string;
    /**
     * Namespace which the library is imported under
     **/
    name(): string;
    /**
     * Equivalent Library which contains all its dependencies
     **/
    expand(): Library;
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): LibraryScalarsAnnotations;
}
/**
 * Library scalar properties annotations accessor
 **/
export interface LibraryScalarsAnnotations {
    /**
     * Library.usage annotations
     **/
    usage(): AnnotationRef[];
}
export interface Api extends LibraryBase {
    /**
     * Short plain-text label for the API
     **/
    title(): string;
    /**
     * A longer, human-friendly description of the API
     **/
    description(): MarkdownString;
    /**
     * The version of the API, e.g. 'v1'
     **/
    version(): string;
    /**
     * A URI that's to be used as the base of all the resources' URIs. Often used as the base of the URL of each resource, containing the location of the API. Can be a template URI.
     **/
    baseUri(): FullUriTemplateString;
    /**
     * The protocols supported by the API
     **/
    protocols(): string[];
    /**
     * The default media type to use for request and response bodies (payloads), e.g. "application/json"
     **/
    mediaType(): MimeType[];
    /**
     * The security schemes that apply to every resource and method in the API
     **/
    securedBy(): SecuritySchemeRef[];
    /**
     * The resources of the API, identified as relative URIs that begin with a slash (/). Every property whose key begins with a slash (/), and is either at the root of the API definition or is the child property of a resource property, is a resource property, e.g.: /users, /{groupId}, etc
     **/
    resources(): Resource[];
    /**
     * Additional overall documentation for the API
     **/
    documentation(): DocumentationItem[];
    /**
     * Most of RAML model elements may have attached annotations decribing additional meta data about this element
     **/
    annotations(): AnnotationRef[];
    /**
     * Returns RAML version. "RAML10" string is returned for RAML 1.0. "RAML08" string is returned for RAML 0.8.
     **/
    RAMLVersion(): string;
    /**
     * Equivalent API with traits and resource types expanded
     * @expLib whether to apply library expansion or not
     **/
    expand(expLib?: boolean): Api;
    /**
     * Get child resource by its relative path
     **/
    childResource(relPath: string): Resource;
    /**
     * Retrieve all resources of the Api
     **/
    allResources(): Resource[];
    /**
     * Retrieve an ordered list of all base uri parameters regardless of whether they are described in `baseUriParameters` or not
     * Consider a fragment of RAML specification:
     * ```yaml
     * version: v1
     * baseUri: https://{organization}.example.com/{version}/{service}
     * baseUriParameters:
     * service:
     * ```
     * Here `version` and `organization` are base uri parameters which are not described in the `baseUriParameters` node,
     * but they are among `Api.baseUriParameters()`.
     **/
    baseUriParameters(): TypeDeclaration[];
    /**
     * Retrieve an ordered list of all base uri parameters regardless of whether they are described in `baseUriParameters` or not
     * Consider a fragment of RAML specification:
     * ```yaml
     * version: v1
     * baseUri: https://{organization}.example.com/{version}/{service}
     * baseUriParameters:
     * service:
     * ```
     * Here `version` and `organization` are base uri parameters which are not described in the `baseUriParameters` node,
     * but they are among `Api.allBaseUriParameters()`.
     * @deprecated
     **/
    allBaseUriParameters(): TypeDeclaration[];
    /**
     * Protocols used by the API. Returns the `protocols` property value if it is specified.
     * Otherwise, returns protocol, specified in the base URI.
     * @deprecated
     **/
    allProtocols(): string[];
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): ApiScalarsAnnotations;
}
export interface Overlay extends Api {
    /**
     * contains description of why overlay exist
     **/
    usage(): string;
    /**
     * Location of a valid RAML API definition (or overlay or extension), the overlay is applied to.
     **/
    extends(): string;
    /**
     * Short plain-text label for the API
     **/
    title(): string;
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): OverlayScalarsAnnotations;
}
/**
 * Api scalar properties annotations accessor
 **/
export interface ApiScalarsAnnotations {
    /**
     * Api.title annotations
     **/
    title(): AnnotationRef[];
    /**
     * Api.description annotations
     **/
    description(): AnnotationRef[];
    /**
     * Api.version annotations
     **/
    version(): AnnotationRef[];
    /**
     * Api.baseUri annotations
     **/
    baseUri(): AnnotationRef[];
    /**
     * Api.protocols annotations
     **/
    protocols(): AnnotationRef[][];
    /**
     * Api.mediaType annotations
     **/
    mediaType(): AnnotationRef[][];
    /**
     * Api.securedBy annotations
     **/
    securedBy(): AnnotationRef[][];
}
/**
 * Overlay scalar properties annotations accessor
 **/
export interface OverlayScalarsAnnotations extends ApiScalarsAnnotations {
    /**
     * Overlay.usage annotations
     **/
    usage(): AnnotationRef[];
    /**
     * Overlay.extends annotations
     **/
    extends(): AnnotationRef[];
    /**
     * Overlay.title annotations
     **/
    title(): AnnotationRef[];
}
export interface Extension extends Api {
    /**
     * contains description of why extension exist
     **/
    usage(): string;
    /**
     * Location of a valid RAML API definition (or overlay or extension), the extension is applied to
     **/
    extends(): string;
    /**
     * Short plain-text label for the API
     **/
    title(): string;
    /**
     * Scalar properties annotations accessor
     **/
    scalarsAnnotations(): ExtensionScalarsAnnotations;
}
/**
 * Extension scalar properties annotations accessor
 **/
export interface ExtensionScalarsAnnotations extends ApiScalarsAnnotations {
    /**
     * Extension.usage annotations
     **/
    usage(): AnnotationRef[];
    /**
     * Extension.extends annotations
     **/
    extends(): AnnotationRef[];
    /**
     * Extension.title annotations
     **/
    title(): AnnotationRef[];
}
/**
 * Custom type guard for Api. Returns true if node is instance of Api. Returns false otherwise.
 * Also returns false for super interfaces of Api.
 */
export declare function isApi(node: core.AbstractWrapperNode): node is Api;
/**
 * Custom type guard for LibraryBase. Returns true if node is instance of LibraryBase. Returns false otherwise.
 * Also returns false for super interfaces of LibraryBase.
 */
export declare function isLibraryBase(node: core.AbstractWrapperNode): node is LibraryBase;
/**
 * Custom type guard for Annotable. Returns true if node is instance of Annotable. Returns false otherwise.
 * Also returns false for super interfaces of Annotable.
 */
export declare function isAnnotable(node: core.AbstractWrapperNode): node is Annotable;
/**
 * Custom type guard for AnnotationRef. Returns true if node is instance of AnnotationRef. Returns false otherwise.
 * Also returns false for super interfaces of AnnotationRef.
 */
export declare function isAnnotationRef(node: core.AbstractWrapperNode): node is AnnotationRef;
/**
 * Custom type guard for Reference. Returns true if node is instance of Reference. Returns false otherwise.
 * Also returns false for super interfaces of Reference.
 */
export declare function isReference(node: core.AbstractWrapperNode): node is Reference;
/**
 * Custom type guard for ValueType. Returns true if node is instance of ValueType. Returns false otherwise.
 * Also returns false for super interfaces of ValueType.
 */
export declare function isValueType(node: core.AbstractWrapperNode): node is ValueType;
/**
 * Custom type guard for StringType. Returns true if node is instance of StringType. Returns false otherwise.
 * Also returns false for super interfaces of StringType.
 */
export declare function isStringType(node: core.AbstractWrapperNode): node is StringType;
/**
 * Custom type guard for UriTemplate. Returns true if node is instance of UriTemplate. Returns false otherwise.
 * Also returns false for super interfaces of UriTemplate.
 */
export declare function isUriTemplate(node: core.AbstractWrapperNode): node is UriTemplate;
/**
 * Custom type guard for RelativeUriString. Returns true if node is instance of RelativeUriString. Returns false otherwise.
 * Also returns false for super interfaces of RelativeUriString.
 */
export declare function isRelativeUriString(node: core.AbstractWrapperNode): node is RelativeUriString;
/**
 * Custom type guard for FullUriTemplateString. Returns true if node is instance of FullUriTemplateString. Returns false otherwise.
 * Also returns false for super interfaces of FullUriTemplateString.
 */
export declare function isFullUriTemplateString(node: core.AbstractWrapperNode): node is FullUriTemplateString;
/**
 * Custom type guard for StatusCodeString. Returns true if node is instance of StatusCodeString. Returns false otherwise.
 * Also returns false for super interfaces of StatusCodeString.
 */
export declare function isStatusCodeString(node: core.AbstractWrapperNode): node is StatusCodeString;
/**
 * Custom type guard for FixedUriString. Returns true if node is instance of FixedUriString. Returns false otherwise.
 * Also returns false for super interfaces of FixedUriString.
 */
export declare function isFixedUriString(node: core.AbstractWrapperNode): node is FixedUriString;
/**
 * Custom type guard for ContentType. Returns true if node is instance of ContentType. Returns false otherwise.
 * Also returns false for super interfaces of ContentType.
 */
export declare function isContentType(node: core.AbstractWrapperNode): node is ContentType;
/**
 * Custom type guard for MarkdownString. Returns true if node is instance of MarkdownString. Returns false otherwise.
 * Also returns false for super interfaces of MarkdownString.
 */
export declare function isMarkdownString(node: core.AbstractWrapperNode): node is MarkdownString;
/**
 * Custom type guard for SchemaString. Returns true if node is instance of SchemaString. Returns false otherwise.
 * Also returns false for super interfaces of SchemaString.
 */
export declare function isSchemaString(node: core.AbstractWrapperNode): node is SchemaString;
/**
 * Custom type guard for MimeType. Returns true if node is instance of MimeType. Returns false otherwise.
 * Also returns false for super interfaces of MimeType.
 */
export declare function isMimeType(node: core.AbstractWrapperNode): node is MimeType;
/**
 * Custom type guard for AnyType. Returns true if node is instance of AnyType. Returns false otherwise.
 * Also returns false for super interfaces of AnyType.
 */
export declare function isAnyType(node: core.AbstractWrapperNode): node is AnyType;
/**
 * Custom type guard for NumberType. Returns true if node is instance of NumberType. Returns false otherwise.
 * Also returns false for super interfaces of NumberType.
 */
export declare function isNumberType(node: core.AbstractWrapperNode): node is NumberType;
/**
 * Custom type guard for IntegerType. Returns true if node is instance of IntegerType. Returns false otherwise.
 * Also returns false for super interfaces of IntegerType.
 */
export declare function isIntegerType(node: core.AbstractWrapperNode): node is IntegerType;
/**
 * Custom type guard for NullType. Returns true if node is instance of NullType. Returns false otherwise.
 * Also returns false for super interfaces of NullType.
 */
export declare function isNullType(node: core.AbstractWrapperNode): node is NullType;
/**
 * Custom type guard for TimeOnlyType. Returns true if node is instance of TimeOnlyType. Returns false otherwise.
 * Also returns false for super interfaces of TimeOnlyType.
 */
export declare function isTimeOnlyType(node: core.AbstractWrapperNode): node is TimeOnlyType;
/**
 * Custom type guard for DateOnlyType. Returns true if node is instance of DateOnlyType. Returns false otherwise.
 * Also returns false for super interfaces of DateOnlyType.
 */
export declare function isDateOnlyType(node: core.AbstractWrapperNode): node is DateOnlyType;
/**
 * Custom type guard for DateTimeOnlyType. Returns true if node is instance of DateTimeOnlyType. Returns false otherwise.
 * Also returns false for super interfaces of DateTimeOnlyType.
 */
export declare function isDateTimeOnlyType(node: core.AbstractWrapperNode): node is DateTimeOnlyType;
/**
 * Custom type guard for DateTimeType. Returns true if node is instance of DateTimeType. Returns false otherwise.
 * Also returns false for super interfaces of DateTimeType.
 */
export declare function isDateTimeType(node: core.AbstractWrapperNode): node is DateTimeType;
/**
 * Custom type guard for FileType. Returns true if node is instance of FileType. Returns false otherwise.
 * Also returns false for super interfaces of FileType.
 */
export declare function isFileType(node: core.AbstractWrapperNode): node is FileType;
/**
 * Custom type guard for BooleanType. Returns true if node is instance of BooleanType. Returns false otherwise.
 * Also returns false for super interfaces of BooleanType.
 */
export declare function isBooleanType(node: core.AbstractWrapperNode): node is BooleanType;
/**
 * Custom type guard for AnnotationTarget. Returns true if node is instance of AnnotationTarget. Returns false otherwise.
 * Also returns false for super interfaces of AnnotationTarget.
 */
export declare function isAnnotationTarget(node: core.AbstractWrapperNode): node is AnnotationTarget;
/**
 * Custom type guard for TraitRef. Returns true if node is instance of TraitRef. Returns false otherwise.
 * Also returns false for super interfaces of TraitRef.
 */
export declare function isTraitRef(node: core.AbstractWrapperNode): node is TraitRef;
/**
 * Custom type guard for Trait. Returns true if node is instance of Trait. Returns false otherwise.
 * Also returns false for super interfaces of Trait.
 */
export declare function isTrait(node: core.AbstractWrapperNode): node is Trait;
/**
 * Custom type guard for MethodBase. Returns true if node is instance of MethodBase. Returns false otherwise.
 * Also returns false for super interfaces of MethodBase.
 */
export declare function isMethodBase(node: core.AbstractWrapperNode): node is MethodBase;
/**
 * Custom type guard for Operation. Returns true if node is instance of Operation. Returns false otherwise.
 * Also returns false for super interfaces of Operation.
 */
export declare function isOperation(node: core.AbstractWrapperNode): node is Operation;
/**
 * Custom type guard for TypeDeclaration. Returns true if node is instance of TypeDeclaration. Returns false otherwise.
 * Also returns false for super interfaces of TypeDeclaration.
 */
export declare function isTypeDeclaration(node: core.AbstractWrapperNode): node is TypeDeclaration;
/**
 * Custom type guard for ModelLocation. Returns true if node is instance of ModelLocation. Returns false otherwise.
 * Also returns false for super interfaces of ModelLocation.
 */
export declare function isModelLocation(node: core.AbstractWrapperNode): node is ModelLocation;
/**
 * Custom type guard for LocationKind. Returns true if node is instance of LocationKind. Returns false otherwise.
 * Also returns false for super interfaces of LocationKind.
 */
export declare function isLocationKind(node: core.AbstractWrapperNode): node is LocationKind;
/**
 * Custom type guard for ExampleSpec. Returns true if node is instance of ExampleSpec. Returns false otherwise.
 * Also returns false for super interfaces of ExampleSpec.
 */
export declare function isExampleSpec(node: core.AbstractWrapperNode): node is ExampleSpec;
/**
 * Custom type guard for UsesDeclaration. Returns true if node is instance of UsesDeclaration. Returns false otherwise.
 * Also returns false for super interfaces of UsesDeclaration.
 */
export declare function isUsesDeclaration(node: core.AbstractWrapperNode): node is UsesDeclaration;
/**
 * Custom type guard for XMLFacetInfo. Returns true if node is instance of XMLFacetInfo. Returns false otherwise.
 * Also returns false for super interfaces of XMLFacetInfo.
 */
export declare function isXMLFacetInfo(node: core.AbstractWrapperNode): node is XMLFacetInfo;
/**
 * Custom type guard for ArrayTypeDeclaration. Returns true if node is instance of ArrayTypeDeclaration. Returns false otherwise.
 * Also returns false for super interfaces of ArrayTypeDeclaration.
 */
export declare function isArrayTypeDeclaration(node: core.AbstractWrapperNode): node is ArrayTypeDeclaration;
/**
 * Custom type guard for UnionTypeDeclaration. Returns true if node is instance of UnionTypeDeclaration. Returns false otherwise.
 * Also returns false for super interfaces of UnionTypeDeclaration.
 */
export declare function isUnionTypeDeclaration(node: core.AbstractWrapperNode): node is UnionTypeDeclaration;
/**
 * Custom type guard for ObjectTypeDeclaration. Returns true if node is instance of ObjectTypeDeclaration. Returns false otherwise.
 * Also returns false for super interfaces of ObjectTypeDeclaration.
 */
export declare function isObjectTypeDeclaration(node: core.AbstractWrapperNode): node is ObjectTypeDeclaration;
/**
 * Custom type guard for StringTypeDeclaration. Returns true if node is instance of StringTypeDeclaration. Returns false otherwise.
 * Also returns false for super interfaces of StringTypeDeclaration.
 */
export declare function isStringTypeDeclaration(node: core.AbstractWrapperNode): node is StringTypeDeclaration;
/**
 * Custom type guard for BooleanTypeDeclaration. Returns true if node is instance of BooleanTypeDeclaration. Returns false otherwise.
 * Also returns false for super interfaces of BooleanTypeDeclaration.
 */
export declare function isBooleanTypeDeclaration(node: core.AbstractWrapperNode): node is BooleanTypeDeclaration;
/**
 * Custom type guard for NumberTypeDeclaration. Returns true if node is instance of NumberTypeDeclaration. Returns false otherwise.
 * Also returns false for super interfaces of NumberTypeDeclaration.
 */
export declare function isNumberTypeDeclaration(node: core.AbstractWrapperNode): node is NumberTypeDeclaration;
/**
 * Custom type guard for IntegerTypeDeclaration. Returns true if node is instance of IntegerTypeDeclaration. Returns false otherwise.
 * Also returns false for super interfaces of IntegerTypeDeclaration.
 */
export declare function isIntegerTypeDeclaration(node: core.AbstractWrapperNode): node is IntegerTypeDeclaration;
/**
 * Custom type guard for DateOnlyTypeDeclaration. Returns true if node is instance of DateOnlyTypeDeclaration. Returns false otherwise.
 * Also returns false for super interfaces of DateOnlyTypeDeclaration.
 */
export declare function isDateOnlyTypeDeclaration(node: core.AbstractWrapperNode): node is DateOnlyTypeDeclaration;
/**
 * Custom type guard for TimeOnlyTypeDeclaration. Returns true if node is instance of TimeOnlyTypeDeclaration. Returns false otherwise.
 * Also returns false for super interfaces of TimeOnlyTypeDeclaration.
 */
export declare function isTimeOnlyTypeDeclaration(node: core.AbstractWrapperNode): node is TimeOnlyTypeDeclaration;
/**
 * Custom type guard for DateTimeOnlyTypeDeclaration. Returns true if node is instance of DateTimeOnlyTypeDeclaration. Returns false otherwise.
 * Also returns false for super interfaces of DateTimeOnlyTypeDeclaration.
 */
export declare function isDateTimeOnlyTypeDeclaration(node: core.AbstractWrapperNode): node is DateTimeOnlyTypeDeclaration;
/**
 * Custom type guard for DateTimeTypeDeclaration. Returns true if node is instance of DateTimeTypeDeclaration. Returns false otherwise.
 * Also returns false for super interfaces of DateTimeTypeDeclaration.
 */
export declare function isDateTimeTypeDeclaration(node: core.AbstractWrapperNode): node is DateTimeTypeDeclaration;
/**
 * Custom type guard for FileTypeDeclaration. Returns true if node is instance of FileTypeDeclaration. Returns false otherwise.
 * Also returns false for super interfaces of FileTypeDeclaration.
 */
export declare function isFileTypeDeclaration(node: core.AbstractWrapperNode): node is FileTypeDeclaration;
/**
 * Custom type guard for Response. Returns true if node is instance of Response. Returns false otherwise.
 * Also returns false for super interfaces of Response.
 */
export declare function isResponse(node: core.AbstractWrapperNode): node is Response;
/**
 * Custom type guard for SecuritySchemePart. Returns true if node is instance of SecuritySchemePart. Returns false otherwise.
 * Also returns false for super interfaces of SecuritySchemePart.
 */
export declare function isSecuritySchemePart(node: core.AbstractWrapperNode): node is SecuritySchemePart;
/**
 * Custom type guard for SecuritySchemeRef. Returns true if node is instance of SecuritySchemeRef. Returns false otherwise.
 * Also returns false for super interfaces of SecuritySchemeRef.
 */
export declare function isSecuritySchemeRef(node: core.AbstractWrapperNode): node is SecuritySchemeRef;
/**
 * Custom type guard for AbstractSecurityScheme. Returns true if node is instance of AbstractSecurityScheme. Returns false otherwise.
 * Also returns false for super interfaces of AbstractSecurityScheme.
 */
export declare function isAbstractSecurityScheme(node: core.AbstractWrapperNode): node is AbstractSecurityScheme;
/**
 * Custom type guard for SecuritySchemeSettings. Returns true if node is instance of SecuritySchemeSettings. Returns false otherwise.
 * Also returns false for super interfaces of SecuritySchemeSettings.
 */
export declare function isSecuritySchemeSettings(node: core.AbstractWrapperNode): node is SecuritySchemeSettings;
/**
 * Custom type guard for OAuth1SecuritySchemeSettings. Returns true if node is instance of OAuth1SecuritySchemeSettings. Returns false otherwise.
 * Also returns false for super interfaces of OAuth1SecuritySchemeSettings.
 */
export declare function isOAuth1SecuritySchemeSettings(node: core.AbstractWrapperNode): node is OAuth1SecuritySchemeSettings;
/**
 * Custom type guard for OAuth2SecuritySchemeSettings. Returns true if node is instance of OAuth2SecuritySchemeSettings. Returns false otherwise.
 * Also returns false for super interfaces of OAuth2SecuritySchemeSettings.
 */
export declare function isOAuth2SecuritySchemeSettings(node: core.AbstractWrapperNode): node is OAuth2SecuritySchemeSettings;
/**
 * Custom type guard for OAuth2SecurityScheme. Returns true if node is instance of OAuth2SecurityScheme. Returns false otherwise.
 * Also returns false for super interfaces of OAuth2SecurityScheme.
 */
export declare function isOAuth2SecurityScheme(node: core.AbstractWrapperNode): node is OAuth2SecurityScheme;
/**
 * Custom type guard for OAuth1SecurityScheme. Returns true if node is instance of OAuth1SecurityScheme. Returns false otherwise.
 * Also returns false for super interfaces of OAuth1SecurityScheme.
 */
export declare function isOAuth1SecurityScheme(node: core.AbstractWrapperNode): node is OAuth1SecurityScheme;
/**
 * Custom type guard for PassThroughSecurityScheme. Returns true if node is instance of PassThroughSecurityScheme. Returns false otherwise.
 * Also returns false for super interfaces of PassThroughSecurityScheme.
 */
export declare function isPassThroughSecurityScheme(node: core.AbstractWrapperNode): node is PassThroughSecurityScheme;
/**
 * Custom type guard for BasicSecurityScheme. Returns true if node is instance of BasicSecurityScheme. Returns false otherwise.
 * Also returns false for super interfaces of BasicSecurityScheme.
 */
export declare function isBasicSecurityScheme(node: core.AbstractWrapperNode): node is BasicSecurityScheme;
/**
 * Custom type guard for DigestSecurityScheme. Returns true if node is instance of DigestSecurityScheme. Returns false otherwise.
 * Also returns false for super interfaces of DigestSecurityScheme.
 */
export declare function isDigestSecurityScheme(node: core.AbstractWrapperNode): node is DigestSecurityScheme;
/**
 * Custom type guard for CustomSecurityScheme. Returns true if node is instance of CustomSecurityScheme. Returns false otherwise.
 * Also returns false for super interfaces of CustomSecurityScheme.
 */
export declare function isCustomSecurityScheme(node: core.AbstractWrapperNode): node is CustomSecurityScheme;
/**
 * Custom type guard for Method. Returns true if node is instance of Method. Returns false otherwise.
 * Also returns false for super interfaces of Method.
 */
export declare function isMethod(node: core.AbstractWrapperNode): node is Method;
/**
 * Custom type guard for ResourceTypeRef. Returns true if node is instance of ResourceTypeRef. Returns false otherwise.
 * Also returns false for super interfaces of ResourceTypeRef.
 */
export declare function isResourceTypeRef(node: core.AbstractWrapperNode): node is ResourceTypeRef;
/**
 * Custom type guard for ResourceType. Returns true if node is instance of ResourceType. Returns false otherwise.
 * Also returns false for super interfaces of ResourceType.
 */
export declare function isResourceType(node: core.AbstractWrapperNode): node is ResourceType;
/**
 * Custom type guard for ResourceBase. Returns true if node is instance of ResourceBase. Returns false otherwise.
 * Also returns false for super interfaces of ResourceBase.
 */
export declare function isResourceBase(node: core.AbstractWrapperNode): node is ResourceBase;
/**
 * Custom type guard for Resource. Returns true if node is instance of Resource. Returns false otherwise.
 * Also returns false for super interfaces of Resource.
 */
export declare function isResource(node: core.AbstractWrapperNode): node is Resource;
/**
 * Custom type guard for DocumentationItem. Returns true if node is instance of DocumentationItem. Returns false otherwise.
 * Also returns false for super interfaces of DocumentationItem.
 */
export declare function isDocumentationItem(node: core.AbstractWrapperNode): node is DocumentationItem;
/**
 * Custom type guard for Library. Returns true if node is instance of Library. Returns false otherwise.
 * Also returns false for super interfaces of Library.
 */
export declare function isLibrary(node: core.AbstractWrapperNode): node is Library;
/**
 * Custom type guard for Overlay. Returns true if node is instance of Overlay. Returns false otherwise.
 * Also returns false for super interfaces of Overlay.
 */
export declare function isOverlay(node: core.AbstractWrapperNode): node is Overlay;
/**
 * Custom type guard for Extension. Returns true if node is instance of Extension. Returns false otherwise.
 * Also returns false for super interfaces of Extension.
 */
export declare function isExtension(node: core.AbstractWrapperNode): node is Extension;
/**
 * Check if the AST node represents fragment
 */
export declare function isFragment(node: Trait | TypeDeclaration | ExampleSpec | ResourceType | DocumentationItem): boolean;
/**
 * Convert fragment representing node to FragmentDeclaration instance.
 */
export declare function asFragment(node: Trait | TypeDeclaration | ExampleSpec | ResourceType | DocumentationItem): FragmentDeclaration;
