(function() {
  var AddDialog, BufferedProcess, CompositeDisposable, CopyDialog, Directory, DirectoryView, Emitter, IgnoredNames, MoveDialog, RootDragAndDrop, TREE_VIEW_URI, TreeView, _, fs, getFullExtension, getStyleObject, nextId, path, ref, ref1, repoForPath, shell, toggleConfig,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; },
    slice = [].slice,
    indexOf = [].indexOf || function(item) { for (var i = 0, l = this.length; i < l; i++) { if (i in this && this[i] === item) return i; } return -1; };

  path = require('path');

  shell = require('electron').shell;

  _ = require('underscore-plus');

  ref = require('atom'), BufferedProcess = ref.BufferedProcess, CompositeDisposable = ref.CompositeDisposable, Emitter = ref.Emitter;

  ref1 = require("./helpers"), repoForPath = ref1.repoForPath, getStyleObject = ref1.getStyleObject, getFullExtension = ref1.getFullExtension;

  fs = require('fs-plus');

  AddDialog = require('./add-dialog');

  MoveDialog = require('./move-dialog');

  CopyDialog = require('./copy-dialog');

  IgnoredNames = null;

  Directory = require('./directory');

  DirectoryView = require('./directory-view');

  RootDragAndDrop = require('./root-drag-and-drop');

  TREE_VIEW_URI = 'atom://tree-view';

  toggleConfig = function(keyPath) {
    return atom.config.set(keyPath, !atom.config.get(keyPath));
  };

  nextId = 1;

  module.exports = TreeView = (function() {
    function TreeView(state) {
      this.onDragLeave = bind(this.onDragLeave, this);
      this.onDragEnter = bind(this.onDragEnter, this);
      this.onStylesheetsChanged = bind(this.onStylesheetsChanged, this);
      var j, len, observer, ref2, ref3, selectedPath;
      this.id = nextId++;
      this.element = document.createElement('div');
      this.element.classList.add('tool-panel', 'tree-view');
      this.element.tabIndex = -1;
      this.list = document.createElement('ol');
      this.list.classList.add('tree-view-root', 'full-menu', 'list-tree', 'has-collapsable-children', 'focusable-panel');
      this.element.appendChild(this.list);
      this.disposables = new CompositeDisposable;
      this.emitter = new Emitter;
      this.roots = [];
      this.selectedPath = null;
      this.selectOnMouseUp = null;
      this.lastFocusedEntry = null;
      this.ignoredPatterns = [];
      this.useSyncFS = false;
      this.currentlyOpening = new Map;
      this.editorsToMove = [];
      this.editorsToDestroy = [];
      this.dragEventCounts = new WeakMap;
      this.rootDragAndDrop = new RootDragAndDrop(this);
      this.handleEvents();
      process.nextTick((function(_this) {
        return function() {
          var onStylesheetsChanged;
          _this.onStylesheetsChanged();
          onStylesheetsChanged = _.debounce(_this.onStylesheetsChanged, 100);
          _this.disposables.add(atom.styles.onDidAddStyleElement(onStylesheetsChanged));
          _this.disposables.add(atom.styles.onDidRemoveStyleElement(onStylesheetsChanged));
          return _this.disposables.add(atom.styles.onDidUpdateStyleElement(onStylesheetsChanged));
        };
      })(this));
      this.updateRoots(state.directoryExpansionStates);
      if (((ref2 = state.selectedPaths) != null ? ref2.length : void 0) > 0) {
        ref3 = state.selectedPaths;
        for (j = 0, len = ref3.length; j < len; j++) {
          selectedPath = ref3[j];
          this.selectMultipleEntries(this.entryForPath(selectedPath));
        }
      } else {
        this.selectEntry(this.roots[0]);
      }
      if ((state.scrollTop != null) || (state.scrollLeft != null)) {
        observer = new IntersectionObserver((function(_this) {
          return function() {
            if (_this.isVisible()) {
              _this.element.scrollTop = state.scrollTop;
              _this.element.scrollLeft = state.scrollLeft;
              return observer.disconnect();
            }
          };
        })(this));
        observer.observe(this.element);
      }
      if (state.width > 0) {
        this.element.style.width = state.width + "px";
      }
      this.disposables.add(this.onWillMoveEntry((function(_this) {
        return function(arg) {
          var editor, editors, filePath, initialPath, k, l, len1, len2, newPath, results, results1;
          initialPath = arg.initialPath, newPath = arg.newPath;
          editors = atom.workspace.getTextEditors();
          if (fs.isDirectorySync(initialPath)) {
            initialPath += path.sep;
            results = [];
            for (k = 0, len1 = editors.length; k < len1; k++) {
              editor = editors[k];
              filePath = editor.getPath();
              if (filePath != null ? filePath.startsWith(initialPath) : void 0) {
                results.push(_this.editorsToMove.push(filePath));
              } else {
                results.push(void 0);
              }
            }
            return results;
          } else {
            results1 = [];
            for (l = 0, len2 = editors.length; l < len2; l++) {
              editor = editors[l];
              filePath = editor.getPath();
              if (filePath === initialPath) {
                results1.push(_this.editorsToMove.push(filePath));
              } else {
                results1.push(void 0);
              }
            }
            return results1;
          }
        };
      })(this)));
      this.disposables.add(this.onEntryMoved((function(_this) {
        return function(arg) {
          var editor, filePath, index, initialPath, k, len1, newPath, ref4, results;
          initialPath = arg.initialPath, newPath = arg.newPath;
          ref4 = atom.workspace.getTextEditors();
          results = [];
          for (k = 0, len1 = ref4.length; k < len1; k++) {
            editor = ref4[k];
            filePath = editor.getPath();
            index = _this.editorsToMove.indexOf(filePath);
            if (index !== -1) {
              editor.getBuffer().setPath(filePath.replace(initialPath, newPath));
              results.push(_this.editorsToMove.splice(index, 1));
            } else {
              results.push(void 0);
            }
          }
          return results;
        };
      })(this)));
      this.disposables.add(this.onMoveEntryFailed((function(_this) {
        return function(arg) {
          var index, initialPath, newPath;
          initialPath = arg.initialPath, newPath = arg.newPath;
          index = _this.editorsToMove.indexOf(initialPath);
          if (index !== -1) {
            return _this.editorsToMove.splice(index, 1);
          }
        };
      })(this)));
      this.disposables.add(this.onWillDeleteEntry((function(_this) {
        return function(arg) {
          var editor, editors, filePath, k, l, len1, len2, pathToDelete, results, results1;
          pathToDelete = arg.pathToDelete;
          editors = atom.workspace.getTextEditors();
          if (fs.isDirectorySync(pathToDelete)) {
            pathToDelete += path.sep;
            results = [];
            for (k = 0, len1 = editors.length; k < len1; k++) {
              editor = editors[k];
              filePath = editor.getPath();
              if ((filePath != null ? filePath.startsWith(pathToDelete) : void 0) && !editor.isModified()) {
                results.push(_this.editorsToDestroy.push(filePath));
              } else {
                results.push(void 0);
              }
            }
            return results;
          } else {
            results1 = [];
            for (l = 0, len2 = editors.length; l < len2; l++) {
              editor = editors[l];
              filePath = editor.getPath();
              if (filePath === pathToDelete && !editor.isModified()) {
                results1.push(_this.editorsToDestroy.push(filePath));
              } else {
                results1.push(void 0);
              }
            }
            return results1;
          }
        };
      })(this)));
      this.disposables.add(this.onEntryDeleted((function(_this) {
        return function(arg) {
          var editor, index, k, len1, pathToDelete, ref4, results;
          pathToDelete = arg.pathToDelete;
          ref4 = atom.workspace.getTextEditors();
          results = [];
          for (k = 0, len1 = ref4.length; k < len1; k++) {
            editor = ref4[k];
            index = _this.editorsToDestroy.indexOf(editor.getPath());
            if (index !== -1) {
              editor.destroy();
              results.push(_this.editorsToDestroy.splice(index, 1));
            } else {
              results.push(void 0);
            }
          }
          return results;
        };
      })(this)));
      this.disposables.add(this.onDeleteEntryFailed((function(_this) {
        return function(arg) {
          var index, pathToDelete;
          pathToDelete = arg.pathToDelete;
          index = _this.editorsToDestroy.indexOf(pathToDelete);
          if (index !== -1) {
            return _this.editorsToDestroy.splice(index, 1);
          }
        };
      })(this)));
    }

    TreeView.prototype.serialize = function() {
      return {
        directoryExpansionStates: new (function(roots) {
          var j, len, root;
          for (j = 0, len = roots.length; j < len; j++) {
            root = roots[j];
            this[root.directory.path] = root.directory.serializeExpansionState();
          }
          return this;
        })(this.roots),
        deserializer: 'TreeView',
        selectedPaths: Array.from(this.getSelectedEntries(), function(entry) {
          return entry.getPath();
        }),
        scrollLeft: this.element.scrollLeft,
        scrollTop: this.element.scrollTop,
        width: parseInt(this.element.style.width || 0)
      };
    };

    TreeView.prototype.destroy = function() {
      var j, len, ref2, root;
      ref2 = this.roots;
      for (j = 0, len = ref2.length; j < len; j++) {
        root = ref2[j];
        root.directory.destroy();
      }
      this.disposables.dispose();
      this.rootDragAndDrop.dispose();
      return this.emitter.emit('did-destroy');
    };

    TreeView.prototype.onDidDestroy = function(callback) {
      return this.emitter.on('did-destroy', callback);
    };

    TreeView.prototype.getTitle = function() {
      return "Project";
    };

    TreeView.prototype.getURI = function() {
      return TREE_VIEW_URI;
    };

    TreeView.prototype.getPreferredLocation = function() {
      if (atom.config.get('tree-view.showOnRightSide')) {
        return 'right';
      } else {
        return 'left';
      }
    };

    TreeView.prototype.getAllowedLocations = function() {
      return ["left", "right"];
    };

    TreeView.prototype.isPermanentDockItem = function() {
      return true;
    };

    TreeView.prototype.getPreferredWidth = function() {
      var result;
      this.list.style.width = 'min-content';
      result = this.list.offsetWidth;
      this.list.style.width = '';
      return result;
    };

    TreeView.prototype.onDirectoryCreated = function(callback) {
      return this.emitter.on('directory-created', callback);
    };

    TreeView.prototype.onEntryCopied = function(callback) {
      return this.emitter.on('entry-copied', callback);
    };

    TreeView.prototype.onWillDeleteEntry = function(callback) {
      return this.emitter.on('will-delete-entry', callback);
    };

    TreeView.prototype.onEntryDeleted = function(callback) {
      return this.emitter.on('entry-deleted', callback);
    };

    TreeView.prototype.onDeleteEntryFailed = function(callback) {
      return this.emitter.on('delete-entry-failed', callback);
    };

    TreeView.prototype.onWillMoveEntry = function(callback) {
      return this.emitter.on('will-move-entry', callback);
    };

    TreeView.prototype.onEntryMoved = function(callback) {
      return this.emitter.on('entry-moved', callback);
    };

    TreeView.prototype.onMoveEntryFailed = function(callback) {
      return this.emitter.on('move-entry-failed', callback);
    };

    TreeView.prototype.onFileCreated = function(callback) {
      return this.emitter.on('file-created', callback);
    };

    TreeView.prototype.handleEvents = function() {
      this.element.addEventListener('click', (function(_this) {
        return function(e) {
          if (e.target.classList.contains('entries')) {
            return;
          }
          if (!(e.shiftKey || e.metaKey || e.ctrlKey)) {
            return _this.entryClicked(e);
          }
        };
      })(this));
      this.element.addEventListener('mousedown', (function(_this) {
        return function(e) {
          return _this.onMouseDown(e);
        };
      })(this));
      this.element.addEventListener('mouseup', (function(_this) {
        return function(e) {
          return _this.onMouseUp(e);
        };
      })(this));
      this.element.addEventListener('dragstart', (function(_this) {
        return function(e) {
          return _this.onDragStart(e);
        };
      })(this));
      this.element.addEventListener('dragenter', (function(_this) {
        return function(e) {
          return _this.onDragEnter(e);
        };
      })(this));
      this.element.addEventListener('dragleave', (function(_this) {
        return function(e) {
          return _this.onDragLeave(e);
        };
      })(this));
      this.element.addEventListener('dragover', (function(_this) {
        return function(e) {
          return _this.onDragOver(e);
        };
      })(this));
      this.element.addEventListener('drop', (function(_this) {
        return function(e) {
          return _this.onDrop(e);
        };
      })(this));
      atom.commands.add(this.element, {
        'core:move-up': this.moveUp.bind(this),
        'core:move-down': this.moveDown.bind(this),
        'core:page-up': (function(_this) {
          return function() {
            return _this.pageUp();
          };
        })(this),
        'core:page-down': (function(_this) {
          return function() {
            return _this.pageDown();
          };
        })(this),
        'core:move-to-top': (function(_this) {
          return function() {
            return _this.scrollToTop();
          };
        })(this),
        'core:move-to-bottom': (function(_this) {
          return function() {
            return _this.scrollToBottom();
          };
        })(this),
        'tree-view:expand-item': (function(_this) {
          return function() {
            return _this.openSelectedEntry({
              pending: true
            }, true);
          };
        })(this),
        'tree-view:recursive-expand-directory': (function(_this) {
          return function() {
            return _this.expandDirectory(true);
          };
        })(this),
        'tree-view:collapse-directory': (function(_this) {
          return function() {
            return _this.collapseDirectory();
          };
        })(this),
        'tree-view:recursive-collapse-directory': (function(_this) {
          return function() {
            return _this.collapseDirectory(true);
          };
        })(this),
        'tree-view:collapse-all': (function(_this) {
          return function() {
            return _this.collapseDirectory(true, true);
          };
        })(this),
        'tree-view:open-selected-entry': (function(_this) {
          return function() {
            return _this.openSelectedEntry();
          };
        })(this),
        'tree-view:open-selected-entry-right': (function(_this) {
          return function() {
            return _this.openSelectedEntryRight();
          };
        })(this),
        'tree-view:open-selected-entry-left': (function(_this) {
          return function() {
            return _this.openSelectedEntryLeft();
          };
        })(this),
        'tree-view:open-selected-entry-up': (function(_this) {
          return function() {
            return _this.openSelectedEntryUp();
          };
        })(this),
        'tree-view:open-selected-entry-down': (function(_this) {
          return function() {
            return _this.openSelectedEntryDown();
          };
        })(this),
        'tree-view:move': (function(_this) {
          return function() {
            return _this.moveSelectedEntry();
          };
        })(this),
        'tree-view:copy': (function(_this) {
          return function() {
            return _this.copySelectedEntries();
          };
        })(this),
        'tree-view:cut': (function(_this) {
          return function() {
            return _this.cutSelectedEntries();
          };
        })(this),
        'tree-view:paste': (function(_this) {
          return function() {
            return _this.pasteEntries();
          };
        })(this),
        'tree-view:copy-full-path': (function(_this) {
          return function() {
            return _this.copySelectedEntryPath(false);
          };
        })(this),
        'tree-view:show-in-file-manager': (function(_this) {
          return function() {
            return _this.showSelectedEntryInFileManager();
          };
        })(this),
        'tree-view:open-in-new-window': (function(_this) {
          return function() {
            return _this.openSelectedEntryInNewWindow();
          };
        })(this),
        'tree-view:copy-project-path': (function(_this) {
          return function() {
            return _this.copySelectedEntryPath(true);
          };
        })(this),
        'tree-view:unfocus': (function(_this) {
          return function() {
            return _this.unfocus();
          };
        })(this),
        'tree-view:toggle-vcs-ignored-files': function() {
          return toggleConfig('tree-view.hideVcsIgnoredFiles');
        },
        'tree-view:toggle-ignored-names': function() {
          return toggleConfig('tree-view.hideIgnoredNames');
        },
        'tree-view:remove-project-folder': (function(_this) {
          return function(e) {
            return _this.removeProjectFolder(e);
          };
        })(this)
      });
      [0, 1, 2, 3, 4, 5, 6, 7, 8].forEach((function(_this) {
        return function(index) {
          return atom.commands.add(_this.element, "tree-view:open-selected-entry-in-pane-" + (index + 1), function() {
            return _this.openSelectedEntryInPane(index);
          });
        };
      })(this));
      this.disposables.add(atom.workspace.getCenter().onDidChangeActivePaneItem((function(_this) {
        return function() {
          _this.selectActiveFile();
          if (atom.config.get('tree-view.autoReveal')) {
            return _this.revealActiveFile({
              show: false,
              focus: false
            });
          }
        };
      })(this)));
      this.disposables.add(atom.project.onDidChangePaths((function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('tree-view.hideVcsIgnoredFiles', (function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('tree-view.hideIgnoredNames', (function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('core.ignoredNames', (function(_this) {
        return function() {
          if (atom.config.get('tree-view.hideIgnoredNames')) {
            return _this.updateRoots();
          }
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('tree-view.sortFoldersBeforeFiles', (function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
      return this.disposables.add(atom.config.onDidChange('tree-view.squashDirectoryNames', (function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
    };

    TreeView.prototype.toggle = function() {
      return atom.workspace.toggle(this);
    };

    TreeView.prototype.show = function(focus) {
      return atom.workspace.open(this, {
        searchAllPanes: true,
        activatePane: false,
        activateItem: false
      }).then((function(_this) {
        return function() {
          atom.workspace.paneContainerForURI(_this.getURI()).show();
          if (focus) {
            return _this.focus();
          }
        };
      })(this));
    };

    TreeView.prototype.hide = function() {
      return atom.workspace.hide(this);
    };

    TreeView.prototype.focus = function() {
      return this.element.focus();
    };

    TreeView.prototype.unfocus = function() {
      return atom.workspace.getCenter().activate();
    };

    TreeView.prototype.hasFocus = function() {
      return document.activeElement === this.element;
    };

    TreeView.prototype.toggleFocus = function() {
      if (this.hasFocus()) {
        return this.unfocus();
      } else {
        return this.show(true);
      }
    };

    TreeView.prototype.entryClicked = function(e) {
      var entry, isRecursive;
      if (entry = e.target.closest('.entry')) {
        isRecursive = e.altKey || false;
        this.selectEntry(entry);
        if (entry.classList.contains('directory')) {
          return entry.toggleExpansion(isRecursive);
        } else if (entry.classList.contains('file')) {
          return this.fileViewEntryClicked(e);
        }
      }
    };

    TreeView.prototype.fileViewEntryClicked = function(e) {
      var alwaysOpenExisting, detail, filePath, openPromise, ref2;
      filePath = e.target.closest('.entry').getPath();
      detail = (ref2 = e.detail) != null ? ref2 : 1;
      alwaysOpenExisting = atom.config.get('tree-view.alwaysOpenExisting');
      if (detail === 1) {
        if (atom.config.get('core.allowPendingPaneItems')) {
          openPromise = atom.workspace.open(filePath, {
            pending: true,
            activatePane: false,
            searchAllPanes: alwaysOpenExisting
          });
          this.currentlyOpening.set(filePath, openPromise);
          return openPromise.then((function(_this) {
            return function() {
              return _this.currentlyOpening["delete"](filePath);
            };
          })(this));
        }
      } else if (detail === 2) {
        return this.openAfterPromise(filePath, {
          searchAllPanes: alwaysOpenExisting
        });
      }
    };

    TreeView.prototype.openAfterPromise = function(uri, options) {
      var promise;
      if (promise = this.currentlyOpening.get(uri)) {
        return promise.then(function() {
          return atom.workspace.open(uri, options);
        });
      } else {
        return atom.workspace.open(uri, options);
      }
    };

    TreeView.prototype.updateRoots = function(expansionStates) {
      var directory, j, k, key, len, len1, oldExpansionStates, projectPath, ref2, results, root, selectedPath, selectedPaths, stats;
      if (expansionStates == null) {
        expansionStates = {};
      }
      selectedPaths = this.selectedPaths();
      oldExpansionStates = {};
      ref2 = this.roots;
      for (j = 0, len = ref2.length; j < len; j++) {
        root = ref2[j];
        oldExpansionStates[root.directory.path] = root.directory.serializeExpansionState();
        root.directory.destroy();
        root.remove();
      }
      if (IgnoredNames == null) {
        IgnoredNames = require('./ignored-names');
      }
      this.roots = (function() {
        var k, l, len1, len2, ref3, ref4, ref5, ref6, results;
        ref3 = atom.project.getPaths();
        results = [];
        for (k = 0, len1 = ref3.length; k < len1; k++) {
          projectPath = ref3[k];
          stats = fs.lstatSyncNoException(projectPath);
          if (!stats) {
            continue;
          }
          stats = _.pick.apply(_, [stats].concat(slice.call(_.keys(stats))));
          ref4 = ["atime", "birthtime", "ctime", "mtime"];
          for (l = 0, len2 = ref4.length; l < len2; l++) {
            key = ref4[l];
            stats[key] = stats[key].getTime();
          }
          directory = new Directory({
            name: path.basename(projectPath),
            fullPath: projectPath,
            symlink: false,
            isRoot: true,
            expansionState: (ref5 = (ref6 = expansionStates[projectPath]) != null ? ref6 : oldExpansionStates[projectPath]) != null ? ref5 : {
              isExpanded: true
            },
            ignoredNames: new IgnoredNames(),
            useSyncFS: this.useSyncFS,
            stats: stats
          });
          root = new DirectoryView(directory).element;
          this.list.appendChild(root);
          results.push(root);
        }
        return results;
      }).call(this);
      results = [];
      for (k = 0, len1 = selectedPaths.length; k < len1; k++) {
        selectedPath = selectedPaths[k];
        results.push(this.selectMultipleEntries(this.entryForPath(selectedPath)));
      }
      return results;
    };

    TreeView.prototype.getActivePath = function() {
      var ref2;
      return (ref2 = atom.workspace.getCenter().getActivePaneItem()) != null ? typeof ref2.getPath === "function" ? ref2.getPath() : void 0 : void 0;
    };

    TreeView.prototype.selectActiveFile = function() {
      var activeFilePath;
      activeFilePath = this.getActivePath();
      if (this.entryForPath(activeFilePath)) {
        return this.selectEntryForPath(activeFilePath);
      } else {
        return this.deselect();
      }
    };

    TreeView.prototype.revealActiveFile = function(options) {
      var focus, promise, show;
      if (options == null) {
        options = {};
      }
      if (!atom.project.getPaths().length) {
        return Promise.resolve();
      }
      show = options.show, focus = options.focus;
      if (focus == null) {
        focus = atom.config.get('tree-view.focusOnReveal');
      }
      promise = show || focus ? this.show(focus) : Promise.resolve();
      return promise.then((function(_this) {
        return function() {
          var activeFilePath, activePathComponents, currentPath, entry, j, len, pathComponent, ref2, relativePath, results, rootPath;
          if (!(activeFilePath = _this.getActivePath())) {
            return;
          }
          ref2 = atom.project.relativizePath(activeFilePath), rootPath = ref2[0], relativePath = ref2[1];
          if (rootPath == null) {
            return;
          }
          activePathComponents = relativePath.split(path.sep);
          activePathComponents.unshift(rootPath.substr(rootPath.lastIndexOf(path.sep) + 1));
          currentPath = rootPath.substr(0, rootPath.lastIndexOf(path.sep));
          results = [];
          for (j = 0, len = activePathComponents.length; j < len; j++) {
            pathComponent = activePathComponents[j];
            currentPath += path.sep + pathComponent;
            entry = _this.entryForPath(currentPath);
            if (entry.classList.contains('directory')) {
              results.push(entry.expand());
            } else {
              _this.selectEntry(entry);
              results.push(_this.scrollToEntry(entry));
            }
          }
          return results;
        };
      })(this));
    };

    TreeView.prototype.copySelectedEntryPath = function(relativePath) {
      var pathToCopy;
      if (relativePath == null) {
        relativePath = false;
      }
      if (pathToCopy = this.selectedPath) {
        if (relativePath) {
          pathToCopy = atom.project.relativize(pathToCopy);
        }
        return atom.clipboard.write(pathToCopy);
      }
    };

    TreeView.prototype.entryForPath = function(entryPath) {
      var bestMatchEntry, bestMatchLength, entry, entryLength, j, len, ref2, ref3;
      bestMatchEntry = null;
      bestMatchLength = 0;
      ref2 = this.list.querySelectorAll('.entry');
      for (j = 0, len = ref2.length; j < len; j++) {
        entry = ref2[j];
        if (entry.isPathEqual(entryPath)) {
          return entry;
        }
        entryLength = entry.getPath().length;
        if (((ref3 = entry.directory) != null ? ref3.contains(entryPath) : void 0) && entryLength > bestMatchLength) {
          bestMatchEntry = entry;
          bestMatchLength = entryLength;
        }
      }
      return bestMatchEntry;
    };

    TreeView.prototype.selectEntryForPath = function(entryPath) {
      return this.selectEntry(this.entryForPath(entryPath));
    };

    TreeView.prototype.moveDown = function(event) {
      var nextEntry, selectedEntry;
      if (event != null) {
        event.stopImmediatePropagation();
      }
      selectedEntry = this.selectedEntry();
      if (selectedEntry != null) {
        if (selectedEntry.classList.contains('directory')) {
          if (this.selectEntry(selectedEntry.entries.children[0])) {
            this.scrollToEntry(this.selectedEntry(), false);
            return;
          }
        }
        if (nextEntry = this.nextEntry(selectedEntry)) {
          this.selectEntry(nextEntry);
        }
      } else {
        this.selectEntry(this.roots[0]);
      }
      return this.scrollToEntry(this.selectedEntry(), false);
    };

    TreeView.prototype.moveUp = function(event) {
      var entries, previousEntry, selectedEntry;
      event.stopImmediatePropagation();
      selectedEntry = this.selectedEntry();
      if (selectedEntry != null) {
        if (previousEntry = this.previousEntry(selectedEntry)) {
          this.selectEntry(previousEntry);
        } else {
          this.selectEntry(selectedEntry.parentElement.closest('.directory'));
        }
      } else {
        entries = this.list.querySelectorAll('.entry');
        this.selectEntry(entries[entries.length - 1]);
      }
      return this.scrollToEntry(this.selectedEntry(), false);
    };

    TreeView.prototype.nextEntry = function(entry) {
      var currentEntry;
      currentEntry = entry;
      while (currentEntry != null) {
        if (currentEntry.nextSibling != null) {
          currentEntry = currentEntry.nextSibling;
          if (currentEntry.matches('.entry')) {
            return currentEntry;
          }
        } else {
          currentEntry = currentEntry.parentElement.closest('.directory');
        }
      }
      return null;
    };

    TreeView.prototype.previousEntry = function(entry) {
      var entries, previousEntry;
      previousEntry = entry.previousSibling;
      while ((previousEntry != null) && !previousEntry.matches('.entry')) {
        previousEntry = previousEntry.previousSibling;
      }
      if (previousEntry == null) {
        return null;
      }
      if (previousEntry.matches('.directory.expanded')) {
        entries = previousEntry.querySelectorAll('.entry');
        if (entries.length > 0) {
          return entries[entries.length - 1];
        }
      }
      return previousEntry;
    };

    TreeView.prototype.expandDirectory = function(isRecursive) {
      var directory, selectedEntry;
      if (isRecursive == null) {
        isRecursive = false;
      }
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      directory = selectedEntry.closest('.directory');
      if (isRecursive === false && directory.isExpanded) {
        if (directory.directory.getEntries().length > 0) {
          return this.moveDown();
        }
      } else {
        return directory.expand(isRecursive);
      }
    };

    TreeView.prototype.collapseDirectory = function(isRecursive, allDirectories) {
      var directory, j, len, ref2, root, selectedEntry;
      if (isRecursive == null) {
        isRecursive = false;
      }
      if (allDirectories == null) {
        allDirectories = false;
      }
      if (allDirectories) {
        ref2 = this.roots;
        for (j = 0, len = ref2.length; j < len; j++) {
          root = ref2[j];
          root.collapse(true);
        }
        return;
      }
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      if (directory = selectedEntry.closest('.expanded.directory')) {
        directory.collapse(isRecursive);
        return this.selectEntry(directory);
      }
    };

    TreeView.prototype.openSelectedEntry = function(options, expandDirectory) {
      var selectedEntry;
      if (options == null) {
        options = {};
      }
      if (expandDirectory == null) {
        expandDirectory = false;
      }
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      if (selectedEntry.classList.contains('directory')) {
        if (expandDirectory) {
          return this.expandDirectory(false);
        } else {
          return selectedEntry.toggleExpansion();
        }
      } else if (selectedEntry.classList.contains('file')) {
        if (atom.config.get('tree-view.alwaysOpenExisting')) {
          options = Object.assign({
            searchAllPanes: true
          }, options);
        }
        return this.openAfterPromise(selectedEntry.getPath(), options);
      }
    };

    TreeView.prototype.openSelectedEntrySplit = function(orientation, side) {
      var pane, selectedEntry, split;
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      pane = atom.workspace.getCenter().getActivePane();
      if (pane && selectedEntry.classList.contains('file')) {
        if (atom.workspace.getCenter().getActivePaneItem()) {
          split = pane.split(orientation, side);
          return atom.workspace.openURIInPane(selectedEntry.getPath(), split);
        } else {
          return this.openSelectedEntry(true);
        }
      }
    };

    TreeView.prototype.openSelectedEntryRight = function() {
      return this.openSelectedEntrySplit('horizontal', 'after');
    };

    TreeView.prototype.openSelectedEntryLeft = function() {
      return this.openSelectedEntrySplit('horizontal', 'before');
    };

    TreeView.prototype.openSelectedEntryUp = function() {
      return this.openSelectedEntrySplit('vertical', 'before');
    };

    TreeView.prototype.openSelectedEntryDown = function() {
      return this.openSelectedEntrySplit('vertical', 'after');
    };

    TreeView.prototype.openSelectedEntryInPane = function(index) {
      var pane, selectedEntry;
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      pane = atom.workspace.getCenter().getPanes()[index];
      if (pane && selectedEntry.classList.contains('file')) {
        return atom.workspace.openURIInPane(selectedEntry.getPath(), pane);
      }
    };

    TreeView.prototype.moveSelectedEntry = function() {
      var dialog, entry, oldPath;
      if (this.hasFocus()) {
        entry = this.selectedEntry();
        if ((entry == null) || indexOf.call(this.roots, entry) >= 0) {
          return;
        }
        oldPath = entry.getPath();
      } else {
        oldPath = this.getActivePath();
      }
      if (oldPath) {
        dialog = new MoveDialog(oldPath, {
          willMove: (function(_this) {
            return function(arg) {
              var initialPath, newPath;
              initialPath = arg.initialPath, newPath = arg.newPath;
              return _this.emitter.emit('will-move-entry', {
                initialPath: initialPath,
                newPath: newPath
              });
            };
          })(this),
          onMove: (function(_this) {
            return function(arg) {
              var initialPath, newPath;
              initialPath = arg.initialPath, newPath = arg.newPath;
              return _this.emitter.emit('entry-moved', {
                initialPath: initialPath,
                newPath: newPath
              });
            };
          })(this),
          onMoveFailed: (function(_this) {
            return function(arg) {
              var initialPath, newPath;
              initialPath = arg.initialPath, newPath = arg.newPath;
              return _this.emitter.emit('move-entry-failed', {
                initialPath: initialPath,
                newPath: newPath
              });
            };
          })(this)
        });
        return dialog.attach();
      }
    };

    TreeView.prototype.showSelectedEntryInFileManager = function() {
      var filePath, ref2;
      if (!(filePath = (ref2 = this.selectedEntry()) != null ? ref2.getPath() : void 0)) {
        return;
      }
      if (!shell.showItemInFolder(filePath)) {
        return atom.notifications.addWarning("Unable to show " + filePath + " in file manager");
      }
    };

    TreeView.prototype.showCurrentFileInFileManager = function() {
      var filePath, ref2;
      if (!(filePath = (ref2 = atom.workspace.getCenter().getActiveTextEditor()) != null ? ref2.getPath() : void 0)) {
        return;
      }
      if (!shell.showItemInFolder(filePath)) {
        return atom.notifications.addWarning("Unable to show " + filePath + " in file manager");
      }
    };

    TreeView.prototype.openSelectedEntryInNewWindow = function() {
      var pathToOpen, ref2;
      if (pathToOpen = (ref2 = this.selectedEntry()) != null ? ref2.getPath() : void 0) {
        return atom.open({
          pathsToOpen: [pathToOpen],
          newWindow: true
        });
      }
    };

    TreeView.prototype.copySelectedEntry = function() {
      var dialog, entry, oldPath;
      if (this.hasFocus()) {
        entry = this.selectedEntry();
        if (indexOf.call(this.roots, entry) >= 0) {
          return;
        }
        oldPath = entry != null ? entry.getPath() : void 0;
      } else {
        oldPath = this.getActivePath();
      }
      if (!oldPath) {
        return;
      }
      dialog = new CopyDialog(oldPath, {
        onCopy: (function(_this) {
          return function(arg) {
            var initialPath, newPath;
            initialPath = arg.initialPath, newPath = arg.newPath;
            return _this.emitter.emit('entry-copied', {
              initialPath: initialPath,
              newPath: newPath
            });
          };
        })(this)
      });
      return dialog.attach();
    };

    TreeView.prototype.removeSelectedEntries = function() {
      var activePath, j, len, ref2, ref3, root, selectedEntries, selectedPaths;
      if (this.hasFocus()) {
        selectedPaths = this.selectedPaths();
        selectedEntries = this.getSelectedEntries();
      } else if (activePath = this.getActivePath()) {
        selectedPaths = [activePath];
        selectedEntries = [this.entryForPath(activePath)];
      }
      if (!((selectedPaths != null ? selectedPaths.length : void 0) > 0)) {
        return;
      }
      ref2 = this.roots;
      for (j = 0, len = ref2.length; j < len; j++) {
        root = ref2[j];
        if (ref3 = root.getPath(), indexOf.call(selectedPaths, ref3) >= 0) {
          atom.confirm({
            message: "The root directory '" + root.directory.name + "' can't be removed.",
            buttons: ['OK']
          }, function() {});
          return;
        }
      }
      return atom.confirm({
        message: "Are you sure you want to delete the selected " + (selectedPaths.length > 1 ? 'items' : 'item') + "?",
        detailedMessage: "You are deleting:\n" + (selectedPaths.join('\n')),
        buttons: ['Move to Trash', 'Cancel']
      }, (function(_this) {
        return function(response) {
          var failedDeletions, firstSelectedEntry, k, len1, repo, selectedPath;
          if (response === 0) {
            failedDeletions = [];
            for (k = 0, len1 = selectedPaths.length; k < len1; k++) {
              selectedPath = selectedPaths[k];
              if (!fs.existsSync(selectedPath)) {
                continue;
              }
              _this.emitter.emit('will-delete-entry', {
                pathToDelete: selectedPath
              });
              if (shell.moveItemToTrash(selectedPath)) {
                _this.emitter.emit('entry-deleted', {
                  pathToDelete: selectedPath
                });
              } else {
                _this.emitter.emit('delete-entry-failed', {
                  pathToDelete: selectedPath
                });
                failedDeletions.push(selectedPath);
              }
              if (repo = repoForPath(selectedPath)) {
                repo.getPathStatus(selectedPath);
              }
            }
            if (failedDeletions.length > 0) {
              atom.notifications.addError(_this.formatTrashFailureMessage(failedDeletions), {
                description: _this.formatTrashEnabledMessage(),
                detail: "" + (failedDeletions.join('\n')),
                dismissable: true
              });
            }
            if (firstSelectedEntry = selectedEntries[0]) {
              _this.selectEntry(firstSelectedEntry.closest('.directory:not(.selected)'));
            }
            if (atom.config.get('tree-view.squashDirectoryNames')) {
              return _this.updateRoots();
            }
          }
        };
      })(this));
    };

    TreeView.prototype.formatTrashFailureMessage = function(failedDeletions) {
      var fileText;
      fileText = failedDeletions.length > 1 ? 'files' : 'file';
      return "The following " + fileText + " couldn't be moved to the trash.";
    };

    TreeView.prototype.formatTrashEnabledMessage = function() {
      switch (process.platform) {
        case 'linux':
          return 'Is `gvfs-trash` installed?';
        case 'darwin':
          return 'Is Trash enabled on the volume where the files are stored?';
        case 'win32':
          return 'Is there a Recycle Bin on the drive where the files are stored?';
      }
    };

    TreeView.prototype.copySelectedEntries = function() {
      var selectedPaths;
      selectedPaths = this.selectedPaths();
      if (!(selectedPaths && selectedPaths.length > 0)) {
        return;
      }
      window.localStorage.removeItem('tree-view:cutPath');
      return window.localStorage['tree-view:copyPath'] = JSON.stringify(selectedPaths);
    };

    TreeView.prototype.cutSelectedEntries = function() {
      var selectedPaths;
      selectedPaths = this.selectedPaths();
      if (!(selectedPaths && selectedPaths.length > 0)) {
        return;
      }
      window.localStorage.removeItem('tree-view:copyPath');
      return window.localStorage['tree-view:cutPath'] = JSON.stringify(selectedPaths);
    };

    TreeView.prototype.pasteEntries = function() {
      var basePath, catchAndShowFileErrors, copiedPaths, cutPaths, error, extension, fileCounter, filePath, initialPath, initialPathIsDirectory, initialPaths, j, len, newPath, originalNewPath, realBasePath, realInitialPath, ref2, results, selectedEntry;
      selectedEntry = this.selectedEntry();
      cutPaths = window.localStorage['tree-view:cutPath'] ? JSON.parse(window.localStorage['tree-view:cutPath']) : null;
      copiedPaths = window.localStorage['tree-view:copyPath'] ? JSON.parse(window.localStorage['tree-view:copyPath']) : null;
      initialPaths = copiedPaths || cutPaths;
      catchAndShowFileErrors = function(operation) {
        var error;
        try {
          return operation();
        } catch (error1) {
          error = error1;
          return atom.notifications.addWarning("Unable to paste paths: " + initialPaths, {
            detail: error.message
          });
        }
      };
      ref2 = initialPaths != null ? initialPaths : [];
      results = [];
      for (j = 0, len = ref2.length; j < len; j++) {
        initialPath = ref2[j];
        initialPathIsDirectory = fs.isDirectorySync(initialPath);
        if (selectedEntry && initialPath && fs.existsSync(initialPath)) {
          basePath = selectedEntry.getPath();
          if (selectedEntry.classList.contains('file')) {
            basePath = path.dirname(basePath);
          }
          newPath = path.join(basePath, path.basename(initialPath));
          realBasePath = fs.realpathSync(basePath) + path.sep;
          realInitialPath = fs.realpathSync(initialPath) + path.sep;
          if (initialPathIsDirectory && realBasePath.startsWith(realInitialPath)) {
            if (!fs.isSymbolicLinkSync(initialPath)) {
              atom.notifications.addWarning('Cannot paste a folder into itself');
              continue;
            }
          }
          if (copiedPaths) {
            fileCounter = 0;
            originalNewPath = newPath;
            while (fs.existsSync(newPath)) {
              if (initialPathIsDirectory) {
                newPath = "" + originalNewPath + fileCounter;
              } else {
                extension = getFullExtension(originalNewPath);
                filePath = path.join(path.dirname(originalNewPath), path.basename(originalNewPath, extension));
                newPath = "" + filePath + fileCounter + extension;
              }
              fileCounter += 1;
            }
            if (initialPathIsDirectory) {
              results.push(catchAndShowFileErrors((function(_this) {
                return function() {
                  fs.copySync(initialPath, newPath);
                  return _this.emitter.emit('entry-copied', {
                    initialPath: initialPath,
                    newPath: newPath
                  });
                };
              })(this)));
            } else {
              results.push(catchAndShowFileErrors((function(_this) {
                return function() {
                  fs.writeFileSync(newPath, fs.readFileSync(initialPath));
                  return _this.emitter.emit('entry-copied', {
                    initialPath: initialPath,
                    newPath: newPath
                  });
                };
              })(this)));
            }
          } else if (cutPaths) {
            try {
              this.emitter.emit('will-move-entry', {
                initialPath: initialPath,
                newPath: newPath
              });
              fs.moveSync(initialPath, newPath);
              results.push(this.emitter.emit('entry-moved', {
                initialPath: initialPath,
                newPath: newPath
              }));
            } catch (error1) {
              error = error1;
              this.emitter.emit('move-entry-failed', {
                initialPath: initialPath,
                newPath: newPath
              });
              results.push(atom.notifications.addWarning("Unable to paste paths: " + initialPaths, {
                detail: error.message
              }));
            }
          } else {
            results.push(void 0);
          }
        } else {
          results.push(void 0);
        }
      }
      return results;
    };

    TreeView.prototype.add = function(isCreatingFile) {
      var dialog, ref2, ref3, selectedEntry, selectedPath;
      selectedEntry = (ref2 = this.selectedEntry()) != null ? ref2 : this.roots[0];
      selectedPath = (ref3 = selectedEntry != null ? selectedEntry.getPath() : void 0) != null ? ref3 : '';
      dialog = new AddDialog(selectedPath, isCreatingFile);
      dialog.onDidCreateDirectory((function(_this) {
        return function(createdPath) {
          var ref4;
          if ((ref4 = _this.entryForPath(createdPath)) != null) {
            ref4.reload();
          }
          _this.selectEntryForPath(createdPath);
          if (atom.config.get('tree-view.squashDirectoryNames')) {
            _this.updateRoots();
          }
          return _this.emitter.emit('directory-created', {
            path: createdPath
          });
        };
      })(this));
      dialog.onDidCreateFile((function(_this) {
        return function(createdPath) {
          var ref4;
          if ((ref4 = _this.entryForPath(createdPath)) != null) {
            ref4.reload();
          }
          atom.workspace.open(createdPath);
          if (atom.config.get('tree-view.squashDirectoryNames')) {
            _this.updateRoots();
          }
          return _this.emitter.emit('file-created', {
            path: createdPath
          });
        };
      })(this));
      return dialog.attach();
    };

    TreeView.prototype.removeProjectFolder = function(e) {
      var pathToRemove, ref2, ref3, ref4, ref5, ref6, ref7, ref8, ref9;
      pathToRemove = (ref2 = e.target.closest(".project-root > .header")) != null ? (ref3 = ref2.querySelector(".name")) != null ? ref3.dataset.path : void 0 : void 0;
      if (pathToRemove == null) {
        pathToRemove = (ref4 = this.selectedEntry()) != null ? (ref5 = ref4.closest(".project-root")) != null ? (ref6 = ref5.querySelector(".header")) != null ? (ref7 = ref6.querySelector(".name")) != null ? ref7.dataset.path : void 0 : void 0 : void 0 : void 0;
      }
      if (this.roots.length === 1) {
        if (pathToRemove == null) {
          pathToRemove = (ref8 = this.roots[0].querySelector(".header")) != null ? (ref9 = ref8.querySelector(".name")) != null ? ref9.dataset.path : void 0 : void 0;
        }
      }
      if (pathToRemove != null) {
        return atom.project.removePath(pathToRemove);
      }
    };

    TreeView.prototype.selectedEntry = function() {
      return this.list.querySelector('.selected');
    };

    TreeView.prototype.selectEntry = function(entry) {
      var selectedEntries;
      if (entry == null) {
        return;
      }
      this.selectedPath = entry.getPath();
      this.lastFocusedEntry = entry;
      selectedEntries = this.getSelectedEntries();
      if (selectedEntries.length > 1 || selectedEntries[0] !== entry) {
        this.deselect(selectedEntries);
        entry.classList.add('selected');
      }
      return entry;
    };

    TreeView.prototype.getSelectedEntries = function() {
      return this.list.querySelectorAll('.selected');
    };

    TreeView.prototype.deselect = function(elementsToDeselect) {
      var j, len, selected;
      if (elementsToDeselect == null) {
        elementsToDeselect = this.getSelectedEntries();
      }
      for (j = 0, len = elementsToDeselect.length; j < len; j++) {
        selected = elementsToDeselect[j];
        selected.classList.remove('selected');
      }
      return void 0;
    };

    TreeView.prototype.scrollTop = function(top) {
      if (top != null) {
        return this.element.scrollTop = top;
      } else {
        return this.element.scrollTop;
      }
    };

    TreeView.prototype.scrollBottom = function(bottom) {
      if (bottom != null) {
        return this.element.scrollTop = bottom - this.element.offsetHeight;
      } else {
        return this.element.scrollTop + this.element.offsetHeight;
      }
    };

    TreeView.prototype.scrollToEntry = function(entry, center) {
      var element;
      if (center == null) {
        center = true;
      }
      element = (entry != null ? entry.classList.contains('directory') : void 0) ? entry.header : entry;
      return element != null ? element.scrollIntoViewIfNeeded(center) : void 0;
    };

    TreeView.prototype.scrollToBottom = function() {
      var lastEntry;
      if (lastEntry = _.last(this.list.querySelectorAll('.entry'))) {
        this.selectEntry(lastEntry);
        return this.scrollToEntry(lastEntry);
      }
    };

    TreeView.prototype.scrollToTop = function() {
      if (this.roots[0] != null) {
        this.selectEntry(this.roots[0]);
      }
      return this.element.scrollTop = 0;
    };

    TreeView.prototype.pageUp = function() {
      return this.element.scrollTop -= this.element.offsetHeight;
    };

    TreeView.prototype.pageDown = function() {
      return this.element.scrollTop += this.element.offsetHeight;
    };

    TreeView.prototype.moveEntry = function(initialPath, newDirectoryPath) {
      var entryName, error, newPath, realInitialPath, realNewDirectoryPath, repo;
      if (initialPath === newDirectoryPath) {
        return;
      }
      realNewDirectoryPath = fs.realpathSync(newDirectoryPath) + path.sep;
      realInitialPath = fs.realpathSync(initialPath) + path.sep;
      if (fs.isDirectorySync(initialPath) && realNewDirectoryPath.startsWith(realInitialPath)) {
        if (!fs.isSymbolicLinkSync(initialPath)) {
          atom.notifications.addWarning('Cannot move a folder into itself');
          return;
        }
      }
      entryName = path.basename(initialPath);
      newPath = path.join(newDirectoryPath, entryName);
      try {
        this.emitter.emit('will-move-entry', {
          initialPath: initialPath,
          newPath: newPath
        });
        if (!fs.existsSync(newDirectoryPath)) {
          fs.makeTreeSync(newDirectoryPath);
        }
        fs.moveSync(initialPath, newPath);
        this.emitter.emit('entry-moved', {
          initialPath: initialPath,
          newPath: newPath
        });
        if (repo = repoForPath(newPath)) {
          repo.getPathStatus(initialPath);
          return repo.getPathStatus(newPath);
        }
      } catch (error1) {
        error = error1;
        this.emitter.emit('move-entry-failed', {
          initialPath: initialPath,
          newPath: newPath
        });
        return atom.notifications.addWarning("Failed to move entry " + initialPath + " to " + newDirectoryPath, {
          detail: error.message
        });
      }
    };

    TreeView.prototype.onStylesheetsChanged = function() {
      if (!this.isVisible()) {
        return;
      }
      this.element.style.display = 'none';
      this.element.offsetWidth;
      return this.element.style.display = '';
    };

    TreeView.prototype.onMouseDown = function(e) {
      var cmdKey, entryToSelect, shiftKey;
      if (!(entryToSelect = e.target.closest('.entry'))) {
        return;
      }
      e.stopPropagation();
      cmdKey = e.metaKey || (e.ctrlKey && process.platform !== 'darwin');
      if (entryToSelect.classList.contains('selected')) {
        if (e.button === 2 || (e.ctrlKey && process.platform === 'darwin')) {
          return;
        } else {
          shiftKey = e.shiftKey;
          this.selectOnMouseUp = {
            shiftKey: shiftKey,
            cmdKey: cmdKey
          };
          return;
        }
      }
      if (e.shiftKey && cmdKey) {
        this.selectContinuousEntries(entryToSelect, false);
        return this.showMultiSelectMenuIfNecessary();
      } else if (e.shiftKey) {
        this.selectContinuousEntries(entryToSelect);
        return this.showMultiSelectMenuIfNecessary();
      } else if (cmdKey) {
        this.selectMultipleEntries(entryToSelect);
        this.lastFocusedEntry = entryToSelect;
        return this.showMultiSelectMenuIfNecessary();
      } else {
        this.selectEntry(entryToSelect);
        return this.showFullMenu();
      }
    };

    TreeView.prototype.onMouseUp = function(e) {
      var cmdKey, entryToSelect, ref2, shiftKey;
      if (this.selectOnMouseUp == null) {
        return;
      }
      ref2 = this.selectOnMouseUp, shiftKey = ref2.shiftKey, cmdKey = ref2.cmdKey;
      this.selectOnMouseUp = null;
      if (!(entryToSelect = e.target.closest('.entry'))) {
        return;
      }
      e.stopPropagation();
      if (shiftKey && cmdKey) {
        this.selectContinuousEntries(entryToSelect, false);
        return this.showMultiSelectMenuIfNecessary();
      } else if (shiftKey) {
        this.selectContinuousEntries(entryToSelect);
        return this.showMultiSelectMenuIfNecessary();
      } else if (cmdKey) {
        this.deselect([entryToSelect]);
        this.lastFocusedEntry = entryToSelect;
        return this.showMultiSelectMenuIfNecessary();
      } else {
        this.selectEntry(entryToSelect);
        return this.showFullMenu();
      }
    };

    TreeView.prototype.selectedPaths = function() {
      var entry, j, len, ref2, results;
      ref2 = this.getSelectedEntries();
      results = [];
      for (j = 0, len = ref2.length; j < len; j++) {
        entry = ref2[j];
        results.push(entry.getPath());
      }
      return results;
    };

    TreeView.prototype.selectContinuousEntries = function(entry, deselectOthers) {
      var currentSelectedEntry, element, elements, entries, entryIndex, i, j, len, parentContainer, ref2, selectedIndex;
      if (deselectOthers == null) {
        deselectOthers = true;
      }
      currentSelectedEntry = (ref2 = this.lastFocusedEntry) != null ? ref2 : this.selectedEntry();
      parentContainer = entry.parentElement;
      elements = [];
      if (parentContainer === currentSelectedEntry.parentElement) {
        entries = Array.from(parentContainer.querySelectorAll('.entry'));
        entryIndex = entries.indexOf(entry);
        selectedIndex = entries.indexOf(currentSelectedEntry);
        elements = (function() {
          var j, ref3, ref4, results;
          results = [];
          for (i = j = ref3 = entryIndex, ref4 = selectedIndex; ref3 <= ref4 ? j <= ref4 : j >= ref4; i = ref3 <= ref4 ? ++j : --j) {
            results.push(entries[i]);
          }
          return results;
        })();
        if (deselectOthers) {
          this.deselect();
        }
        for (j = 0, len = elements.length; j < len; j++) {
          element = elements[j];
          element.classList.add('selected');
        }
      }
      return elements;
    };

    TreeView.prototype.selectMultipleEntries = function(entry) {
      if (entry != null) {
        entry.classList.toggle('selected');
      }
      return entry;
    };

    TreeView.prototype.showFullMenu = function() {
      this.list.classList.remove('multi-select');
      return this.list.classList.add('full-menu');
    };

    TreeView.prototype.showMultiSelectMenu = function() {
      this.list.classList.remove('full-menu');
      return this.list.classList.add('multi-select');
    };

    TreeView.prototype.showMultiSelectMenuIfNecessary = function() {
      if (this.getSelectedEntries().length > 1) {
        return this.showMultiSelectMenu();
      } else {
        return this.showFullMenu();
      }
    };

    TreeView.prototype.multiSelectEnabled = function() {
      return this.list.classList.contains('multi-select');
    };

    TreeView.prototype.onDragEnter = function(e) {
      var entry;
      if (entry = e.target.closest('.entry.directory')) {
        if (this.rootDragAndDrop.isDragging(e)) {
          return;
        }
        e.stopPropagation();
        if (!this.dragEventCounts.get(entry)) {
          this.dragEventCounts.set(entry, 0);
        }
        if (!(this.dragEventCounts.get(entry) !== 0 || entry.classList.contains('selected'))) {
          entry.classList.add('drag-over', 'selected');
        }
        return this.dragEventCounts.set(entry, this.dragEventCounts.get(entry) + 1);
      }
    };

    TreeView.prototype.onDragLeave = function(e) {
      var entry;
      if (entry = e.target.closest('.entry.directory')) {
        if (this.rootDragAndDrop.isDragging(e)) {
          return;
        }
        e.stopPropagation();
        this.dragEventCounts.set(entry, this.dragEventCounts.get(entry) - 1);
        if (this.dragEventCounts.get(entry) === 0 && entry.classList.contains('drag-over')) {
          return entry.classList.remove('drag-over', 'selected');
        }
      }
    };

    TreeView.prototype.onDragStart = function(e) {
      var dragImage, entry, entryPath, initialPaths, j, key, len, newElement, parentSelected, ref2, ref3, target, value;
      this.dragEventCounts = new WeakMap;
      this.selectOnMouseUp = null;
      if (entry = e.target.closest('.entry')) {
        e.stopPropagation();
        if (this.rootDragAndDrop.canDragStart(e)) {
          return this.rootDragAndDrop.onDragStart(e);
        }
        dragImage = document.createElement("ol");
        dragImage.classList.add("entries", "list-tree");
        dragImage.style.position = "absolute";
        dragImage.style.top = 0;
        dragImage.style.left = 0;
        dragImage.style.willChange = "transform";
        initialPaths = [];
        ref2 = this.getSelectedEntries();
        for (j = 0, len = ref2.length; j < len; j++) {
          target = ref2[j];
          entryPath = target.querySelector(".name").dataset.path;
          parentSelected = target.parentNode.closest(".entry.selected");
          if (!parentSelected) {
            initialPaths.push(entryPath);
            newElement = target.cloneNode(true);
            if (newElement.classList.contains("directory")) {
              newElement.querySelector(".entries").remove();
            }
            ref3 = getStyleObject(target);
            for (key in ref3) {
              value = ref3[key];
              newElement.style[key] = value;
            }
            newElement.style.paddingLeft = "1em";
            newElement.style.paddingRight = "1em";
            dragImage.append(newElement);
          }
        }
        document.body.appendChild(dragImage);
        e.dataTransfer.effectAllowed = "move";
        e.dataTransfer.setDragImage(dragImage, 0, 0);
        e.dataTransfer.setData("initialPaths", initialPaths);
        return window.requestAnimationFrame(function() {
          return dragImage.remove();
        });
      }
    };

    TreeView.prototype.onDragOver = function(e) {
      var entry;
      if (entry = e.target.closest('.entry.directory')) {
        if (this.rootDragAndDrop.isDragging(e)) {
          return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (this.dragEventCounts.get(entry) > 0 && !entry.classList.contains('selected')) {
          return entry.classList.add('drag-over', 'selected');
        }
      }
    };

    TreeView.prototype.onDrop = function(e) {
      var entry, file, initialPath, initialPaths, j, k, l, len, len1, newDirectoryPath, ref2, ref3, ref4, ref5, results, results1, results2;
      this.dragEventCounts = new WeakMap;
      if (entry = e.target.closest('.entry.directory')) {
        if (this.rootDragAndDrop.isDragging(e)) {
          return;
        }
        e.preventDefault();
        e.stopPropagation();
        newDirectoryPath = (ref2 = entry.querySelector('.name')) != null ? ref2.dataset.path : void 0;
        if (!newDirectoryPath) {
          return false;
        }
        initialPaths = e.dataTransfer.getData('initialPaths');
        if (initialPaths) {
          initialPaths = initialPaths.split(',');
          if (initialPaths.includes(newDirectoryPath)) {
            return;
          }
          entry.classList.remove('drag-over', 'selected');
          results = [];
          for (j = initialPaths.length - 1; j >= 0; j += -1) {
            initialPath = initialPaths[j];
            if ((ref3 = this.entryForPath(initialPath)) != null) {
              if (typeof ref3.collapse === "function") {
                ref3.collapse();
              }
            }
            results.push(this.moveEntry(initialPath, newDirectoryPath));
          }
          return results;
        } else {
          entry.classList.remove('selected');
          ref4 = e.dataTransfer.files;
          results1 = [];
          for (k = 0, len = ref4.length; k < len; k++) {
            file = ref4[k];
            results1.push(this.moveEntry(file.path, newDirectoryPath));
          }
          return results1;
        }
      } else if (e.dataTransfer.files.length) {
        ref5 = e.dataTransfer.files;
        results2 = [];
        for (l = 0, len1 = ref5.length; l < len1; l++) {
          entry = ref5[l];
          results2.push(atom.project.addPath(entry.path));
        }
        return results2;
      }
    };

    TreeView.prototype.isVisible = function() {
      return this.element.offsetWidth !== 0 || this.element.offsetHeight !== 0;
    };

    return TreeView;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
