"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = exports.LargeRepoError = exports.GitError = void 0;

var _path = _interopRequireDefault(require("path"));

var _os = _interopRequireDefault(require("os"));

var _child_process = _interopRequireDefault(require("child_process"));

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _util = _interopRequireDefault(require("util"));

var _electron = require("electron");

var _eventKit = require("event-kit");

var _dugite = require("dugite");

var _whatTheDiff = require("what-the-diff");

var _whatTheStatus = require("what-the-status");

var _gitPromptServer = _interopRequireDefault(require("./git-prompt-server"));

var _gitTempDir = _interopRequireDefault(require("./git-temp-dir"));

var _asyncQueue = _interopRequireDefault(require("./async-queue"));

var _reporterProxy = require("./reporter-proxy");

var _helpers = require("./helpers");

var _gitTimingsView = _interopRequireDefault(require("./views/git-timings-view"));

var _file = _interopRequireDefault(require("./models/patch/file"));

var _workerManager = _interopRequireDefault(require("./worker-manager"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const MAX_STATUS_OUTPUT_LENGTH = 1024 * 1024 * 10;
let headless = null;
let execPathPromise = null;

class GitError extends Error {
  constructor(message) {
    super(message);
    this.message = message;
    this.stack = new Error().stack;
  }

}

exports.GitError = GitError;

class LargeRepoError extends Error {
  constructor(message) {
    super(message);
    this.message = message;
    this.stack = new Error().stack;
  }

} // ignored for the purposes of usage metrics tracking because they're noisy


exports.LargeRepoError = LargeRepoError;
const IGNORED_GIT_COMMANDS = ['cat-file', 'config', 'diff', 'for-each-ref', 'log', 'rev-parse', 'status'];
const DISABLE_COLOR_FLAGS = ['branch', 'diff', 'showBranch', 'status', 'ui'].reduce((acc, type) => {
  acc.unshift('-c', `color.${type}=false`);
  return acc;
}, []);
/**
 * Expand config path name per
 * https://git-scm.com/docs/git-config#git-config-pathname
 * this regex attempts to get the specified user's home directory
 * Ex: on Mac ~kuychaco/ is expanded to the specified user’s home directory (/Users/kuychaco)
 * Regex translation:
 * ^~ line starts with tilde
 * ([^\\\\/]*)[\\\\/] captures non-slash characters before first slash
 */

const EXPAND_TILDE_REGEX = new RegExp('^~([^\\\\/]*)[\\\\/]');

class GitShellOutStrategy {
  constructor(workingDir, options = {}) {
    this.workingDir = workingDir;

    if (options.queue) {
      this.commandQueue = options.queue;
    } else {
      const parallelism = options.parallelism || Math.max(3, _os["default"].cpus().length);
      this.commandQueue = new _asyncQueue["default"]({
        parallelism
      });
    }

    this.prompt = options.prompt || (query => Promise.reject());

    this.workerManager = options.workerManager;

    if (headless === null) {
      headless = !_electron.remote.getCurrentWindow().isVisible();
    }
  }
  /*
   * Provide an asynchronous callback to be used to request input from the user for git operations.
   *
   * `prompt` must be a callable that accepts a query object `{prompt, includeUsername}` and returns a Promise
   * that either resolves with a result object `{[username], password}` or rejects on cancellation.
   */


  setPromptCallback(prompt) {
    this.prompt = prompt;
  } // Execute a command and read the output using the embedded Git environment


  async exec(args, options = GitShellOutStrategy.defaultExecArgs) {
    /* eslint-disable no-console,no-control-regex */
    const {
      stdin,
      useGitPromptServer,
      useGpgWrapper,
      useGpgAtomPrompt,
      writeOperation
    } = options;
    const commandName = args[0];
    const subscriptions = new _eventKit.CompositeDisposable();
    const diagnosticsEnabled = process.env.ATOM_GITHUB_GIT_DIAGNOSTICS || atom.config.get('github.gitDiagnostics');
    const formattedArgs = `git ${args.join(' ')} in ${this.workingDir}`;

    const timingMarker = _gitTimingsView["default"].generateMarker(`git ${args.join(' ')}`);

    timingMarker.mark('queued');
    args.unshift(...DISABLE_COLOR_FLAGS);

    if (execPathPromise === null) {
      // Attempt to collect the --exec-path from a native git installation.
      execPathPromise = new Promise((resolve, reject) => {
        _child_process["default"].exec('git --exec-path', (error, stdout, stderr) => {
          /* istanbul ignore if */
          if (error) {
            // Oh well
            resolve(null);
            return;
          }

          resolve(stdout.trim());
        });
      });
    }

    const execPath = await execPathPromise;
    return this.commandQueue.push(async () => {
      timingMarker.mark('prepare');
      let gitPromptServer;
      const pathParts = [];

      if (process.env.PATH) {
        pathParts.push(process.env.PATH);
      }

      if (execPath) {
        pathParts.push(execPath);
      }

      const env = _objectSpread({}, process.env, {
        GIT_TERMINAL_PROMPT: '0',
        GIT_OPTIONAL_LOCKS: '0',
        PATH: pathParts.join(_path["default"].delimiter)
      });

      const gitTempDir = new _gitTempDir["default"]();

      if (useGpgWrapper) {
        await gitTempDir.ensure();
        args.unshift('-c', `gpg.program=${gitTempDir.getGpgWrapperSh()}`);
      }

      if (useGitPromptServer) {
        gitPromptServer = new _gitPromptServer["default"](gitTempDir);
        await gitPromptServer.start(this.prompt);
        env.ATOM_GITHUB_TMP = gitTempDir.getRootPath();
        env.ATOM_GITHUB_ASKPASS_PATH = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getAskPassJs());
        env.ATOM_GITHUB_CREDENTIAL_PATH = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getCredentialHelperJs());
        env.ATOM_GITHUB_ELECTRON_PATH = (0, _helpers.normalizeGitHelperPath)((0, _helpers.getAtomHelperPath)());
        env.ATOM_GITHUB_SOCK_PATH = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getSocketPath());
        env.ATOM_GITHUB_WORKDIR_PATH = this.workingDir;
        env.ATOM_GITHUB_DUGITE_PATH = (0, _helpers.getDugitePath)();
        env.ATOM_GITHUB_KEYTAR_STRATEGY_PATH = (0, _helpers.getSharedModulePath)('keytar-strategy'); // "ssh" won't respect SSH_ASKPASS unless:
        // (a) it's running without a tty
        // (b) DISPLAY is set to something nonempty
        // But, on a Mac, DISPLAY is unset. Ensure that it is so our SSH_ASKPASS is respected.

        if (!process.env.DISPLAY || process.env.DISPLAY.length === 0) {
          env.DISPLAY = 'atom-github-placeholder';
        }

        env.ATOM_GITHUB_ORIGINAL_PATH = process.env.PATH || '';
        env.ATOM_GITHUB_ORIGINAL_GIT_ASKPASS = process.env.GIT_ASKPASS || '';
        env.ATOM_GITHUB_ORIGINAL_SSH_ASKPASS = process.env.SSH_ASKPASS || '';
        env.ATOM_GITHUB_ORIGINAL_GIT_SSH_COMMAND = process.env.GIT_SSH_COMMAND || '';
        env.ATOM_GITHUB_SPEC_MODE = atom.inSpecMode() ? 'true' : 'false';
        env.SSH_ASKPASS = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getAskPassSh());
        env.GIT_ASKPASS = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getAskPassSh());

        if (process.platform === 'linux') {
          env.GIT_SSH_COMMAND = gitTempDir.getSshWrapperSh();
        } else {
          env.GIT_SSH_COMMAND = process.env.GIT_SSH_COMMAND;
        }

        const credentialHelperSh = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getCredentialHelperSh());
        args.unshift('-c', `credential.helper=${credentialHelperSh}`);
      }

      if (useGpgWrapper && useGitPromptServer && useGpgAtomPrompt) {
        env.ATOM_GITHUB_GPG_PROMPT = 'true';
      }
      /* istanbul ignore if */


      if (diagnosticsEnabled) {
        env.GIT_TRACE = 'true';
        env.GIT_TRACE_CURL = 'true';
      }

      let opts = {
        env
      };

      if (stdin) {
        opts.stdin = stdin;
        opts.stdinEncoding = 'utf8';
      }
      /* istanbul ignore if */


      if (process.env.PRINT_GIT_TIMES) {
        console.time(`git:${formattedArgs}`);
      }

      return new Promise(async (resolve, reject) => {
        if (options.beforeRun) {
          const newArgsOpts = await options.beforeRun({
            args,
            opts
          });
          args = newArgsOpts.args;
          opts = newArgsOpts.opts;
        }

        const {
          promise,
          cancel
        } = this.executeGitCommand(args, opts, timingMarker);
        let expectCancel = false;

        if (gitPromptServer) {
          subscriptions.add(gitPromptServer.onDidCancel(async ({
            handlerPid
          }) => {
            expectCancel = true;
            await cancel(); // On Windows, the SSH_ASKPASS handler is executed as a non-child process, so the bin\git-askpass-atom.sh
            // process does not terminate when the git process is killed.
            // Kill the handler process *after* the git process has been killed to ensure that git doesn't have a
            // chance to fall back to GIT_ASKPASS from the credential handler.

            await new Promise((resolveKill, rejectKill) => {
              require('tree-kill')(handlerPid, 'SIGTERM', err => {
                /* istanbul ignore if */
                if (err) {
                  rejectKill(err);
                } else {
                  resolveKill();
                }
              });
            });
          }));
        }

        const {
          stdout,
          stderr,
          exitCode,
          signal,
          timing
        } = await promise["catch"](err => {
          if (err.signal) {
            return {
              signal: err.signal
            };
          }

          reject(err);
          return {};
        });

        if (timing) {
          const {
            execTime,
            spawnTime,
            ipcTime
          } = timing;
          const now = performance.now();
          timingMarker.mark('nexttick', now - execTime - spawnTime - ipcTime);
          timingMarker.mark('execute', now - execTime - ipcTime);
          timingMarker.mark('ipc', now - ipcTime);
        }

        timingMarker.finalize();
        /* istanbul ignore if */

        if (process.env.PRINT_GIT_TIMES) {
          console.timeEnd(`git:${formattedArgs}`);
        }

        if (gitPromptServer) {
          gitPromptServer.terminate();
        }

        subscriptions.dispose();
        /* istanbul ignore if */

        if (diagnosticsEnabled) {
          const exposeControlCharacters = raw => {
            if (!raw) {
              return '';
            }

            return raw.replace(/\u0000/ug, '<NUL>\n').replace(/\u001F/ug, '<SEP>');
          };

          if (headless) {
            let summary = `git:${formattedArgs}\n`;

            if (exitCode !== undefined) {
              summary += `exit status: ${exitCode}\n`;
            } else if (signal) {
              summary += `exit signal: ${signal}\n`;
            }

            if (stdin && stdin.length !== 0) {
              summary += `stdin:\n${exposeControlCharacters(stdin)}\n`;
            }

            summary += 'stdout:';

            if (stdout.length === 0) {
              summary += ' <empty>\n';
            } else {
              summary += `\n${exposeControlCharacters(stdout)}\n`;
            }

            summary += 'stderr:';

            if (stderr.length === 0) {
              summary += ' <empty>\n';
            } else {
              summary += `\n${exposeControlCharacters(stderr)}\n`;
            }

            console.log(summary);
          } else {
            const headerStyle = 'font-weight: bold; color: blue;';
            console.groupCollapsed(`git:${formattedArgs}`);

            if (exitCode !== undefined) {
              console.log('%cexit status%c %d', headerStyle, 'font-weight: normal; color: black;', exitCode);
            } else if (signal) {
              console.log('%cexit signal%c %s', headerStyle, 'font-weight: normal; color: black;', signal);
            }

            console.log('%cfull arguments%c %s', headerStyle, 'font-weight: normal; color: black;', _util["default"].inspect(args, {
              breakLength: Infinity
            }));

            if (stdin && stdin.length !== 0) {
              console.log('%cstdin', headerStyle);
              console.log(exposeControlCharacters(stdin));
            }

            console.log('%cstdout', headerStyle);
            console.log(exposeControlCharacters(stdout));
            console.log('%cstderr', headerStyle);
            console.log(exposeControlCharacters(stderr));
            console.groupEnd();
          }
        }

        if (exitCode !== 0 && !expectCancel) {
          const err = new GitError(`${formattedArgs} exited with code ${exitCode}\nstdout: ${stdout}\nstderr: ${stderr}`);
          err.code = exitCode;
          err.stdErr = stderr;
          err.stdOut = stdout;
          err.command = formattedArgs;
          reject(err);
        }

        if (!IGNORED_GIT_COMMANDS.includes(commandName)) {
          (0, _reporterProxy.incrementCounter)(commandName);
        }

        resolve(stdout);
      });
    }, {
      parallel: !writeOperation
    });
    /* eslint-enable no-console,no-control-regex */
  }

  async gpgExec(args, options) {
    try {
      return await this.exec(args.slice(), _objectSpread({
        useGpgWrapper: true,
        useGpgAtomPrompt: false
      }, options));
    } catch (e) {
      if (/gpg failed/.test(e.stdErr)) {
        return await this.exec(args, _objectSpread({
          useGitPromptServer: true,
          useGpgWrapper: true,
          useGpgAtomPrompt: true
        }, options));
      } else {
        throw e;
      }
    }
  }

  executeGitCommand(args, options, marker = null) {
    if (process.env.ATOM_GITHUB_INLINE_GIT_EXEC || !_workerManager["default"].getInstance().isReady()) {
      marker && marker.mark('nexttick');
      let childPid;

      options.processCallback = child => {
        childPid = child.pid;
        /* istanbul ignore next */

        child.stdin.on('error', err => {
          throw new Error(`Error writing to stdin: git ${args.join(' ')} in ${this.workingDir}\n${options.stdin}\n${err}`);
        });
      };

      const promise = _dugite.GitProcess.exec(args, this.workingDir, options);

      marker && marker.mark('execute');
      return {
        promise,
        cancel: () => {
          /* istanbul ignore if */
          if (!childPid) {
            return Promise.resolve();
          }

          return new Promise((resolve, reject) => {
            require('tree-kill')(childPid, 'SIGTERM', err => {
              /* istanbul ignore if */
              if (err) {
                reject(err);
              } else {
                resolve();
              }
            });
          });
        }
      };
    } else {
      const workerManager = this.workerManager || _workerManager["default"].getInstance();

      return workerManager.request({
        args,
        workingDir: this.workingDir,
        options
      });
    }
  }

  async resolveDotGitDir() {
    try {
      await _fsExtra["default"].stat(this.workingDir); // fails if folder doesn't exist

      const output = await this.exec(['rev-parse', '--resolve-git-dir', _path["default"].join(this.workingDir, '.git')]);
      const dotGitDir = output.trim();
      return (0, _helpers.toNativePathSep)(dotGitDir);
    } catch (e) {
      return null;
    }
  }

  init() {
    return this.exec(['init', this.workingDir]);
  }
  /**
   * Staging/Unstaging files and patches and committing
   */


  stageFiles(paths) {
    if (paths.length === 0) {
      return Promise.resolve(null);
    }

    const args = ['add'].concat(paths.map(_helpers.toGitPathSep));
    return this.exec(args, {
      writeOperation: true
    });
  }

  async fetchCommitMessageTemplate() {
    let templatePath = await this.getConfig('commit.template');

    if (!templatePath) {
      return null;
    }

    const homeDir = _os["default"].homedir();

    templatePath = templatePath.trim().replace(EXPAND_TILDE_REGEX, (_, user) => {
      // if no user is specified, fall back to using the home directory.
      return `${user ? _path["default"].join(_path["default"].dirname(homeDir), user) : homeDir}/`;
    });
    templatePath = (0, _helpers.toNativePathSep)(templatePath);

    if (!_path["default"].isAbsolute(templatePath)) {
      templatePath = _path["default"].join(this.workingDir, templatePath);
    }

    if (!(await (0, _helpers.fileExists)(templatePath))) {
      throw new Error(`Invalid commit template path set in Git config: ${templatePath}`);
    }

    return await _fsExtra["default"].readFile(templatePath, {
      encoding: 'utf8'
    });
  }

  unstageFiles(paths, commit = 'HEAD') {
    if (paths.length === 0) {
      return Promise.resolve(null);
    }

    const args = ['reset', commit, '--'].concat(paths.map(_helpers.toGitPathSep));
    return this.exec(args, {
      writeOperation: true
    });
  }

  stageFileModeChange(filename, newMode) {
    const indexReadPromise = this.exec(['ls-files', '-s', '--', filename]);
    return this.exec(['update-index', '--cacheinfo', `${newMode},<OID_TBD>,${filename}`], {
      writeOperation: true,
      beforeRun: async function determineArgs({
        args,
        opts
      }) {
        const index = await indexReadPromise;
        const oid = index.substr(7, 40);
        return {
          opts,
          args: ['update-index', '--cacheinfo', `${newMode},${oid},${filename}`]
        };
      }
    });
  }

  stageFileSymlinkChange(filename) {
    return this.exec(['rm', '--cached', filename], {
      writeOperation: true
    });
  }

  applyPatch(patch, {
    index
  } = {}) {
    const args = ['apply', '-'];

    if (index) {
      args.splice(1, 0, '--cached');
    }

    return this.exec(args, {
      stdin: patch,
      writeOperation: true
    });
  }

  async commit(rawMessage, {
    allowEmpty,
    amend,
    coAuthors,
    verbatim
  } = {}) {
    const args = ['commit'];
    let msg; // if amending and no new message is passed, use last commit's message. Ensure that we don't
    // mangle it in the process.

    if (amend && rawMessage.length === 0) {
      const {
        unbornRef,
        messageBody,
        messageSubject
      } = await this.getHeadCommit();

      if (unbornRef) {
        msg = rawMessage;
      } else {
        msg = `${messageSubject}\n\n${messageBody}`.trim();
        verbatim = true;
      }
    } else {
      msg = rawMessage;
    } // if commit template is used, strip commented lines from commit
    // to be consistent with command line git.


    const template = await this.fetchCommitMessageTemplate();

    if (template) {
      // respecting the comment character from user settings or fall back to # as default.
      // https://git-scm.com/docs/git-config#git-config-corecommentChar
      let commentChar = await this.getConfig('core.commentChar');

      if (!commentChar) {
        commentChar = '#';
      }

      msg = msg.split('\n').filter(line => !line.startsWith(commentChar)).join('\n');
    } // Determine the cleanup mode.


    if (verbatim) {
      args.push('--cleanup=verbatim');
    } else {
      const configured = await this.getConfig('commit.cleanup');
      const mode = configured && configured !== 'default' ? configured : 'strip';
      args.push(`--cleanup=${mode}`);
    } // add co-author commit trailers if necessary


    if (coAuthors && coAuthors.length > 0) {
      msg = await this.addCoAuthorsToMessage(msg, coAuthors);
    }

    args.push('-m', msg.trim());

    if (amend) {
      args.push('--amend');
    }

    if (allowEmpty) {
      args.push('--allow-empty');
    }

    return this.gpgExec(args, {
      writeOperation: true
    });
  }

  addCoAuthorsToMessage(message, coAuthors = []) {
    const trailers = coAuthors.map(author => {
      return {
        token: 'Co-Authored-By',
        value: `${author.name} <${author.email}>`
      };
    }); // Ensure that message ends with newline for git-interpret trailers to work

    const msg = `${message.trim()}\n`;
    return trailers.length ? this.mergeTrailers(msg, trailers) : msg;
  }
  /**
   * File Status and Diffs
   */


  async getStatusBundle() {
    const args = ['status', '--porcelain=v2', '--branch', '--untracked-files=all', '--ignore-submodules=dirty', '-z'];
    const output = await this.exec(args);

    if (output.length > MAX_STATUS_OUTPUT_LENGTH) {
      throw new LargeRepoError();
    }

    const results = await (0, _whatTheStatus.parse)(output);

    for (const entryType in results) {
      if (Array.isArray(results[entryType])) {
        this.updateNativePathSepForEntries(results[entryType]);
      }
    }

    return results;
  }

  updateNativePathSepForEntries(entries) {
    entries.forEach(entry => {
      // Normally we would avoid mutating responses from other package's APIs, but we control
      // the `what-the-status` module and know there are no side effects.
      // This is a hot code path and by mutating we avoid creating new objects that will just be GC'ed
      if (entry.filePath) {
        entry.filePath = (0, _helpers.toNativePathSep)(entry.filePath);
      }

      if (entry.origFilePath) {
        entry.origFilePath = (0, _helpers.toNativePathSep)(entry.origFilePath);
      }
    });
  }

  async diffFileStatus(options = {}) {
    const args = ['diff', '--name-status', '--no-renames'];

    if (options.staged) {
      args.push('--staged');
    }

    if (options.target) {
      args.push(options.target);
    }

    const output = await this.exec(args);
    const statusMap = {
      A: 'added',
      M: 'modified',
      D: 'deleted',
      U: 'unmerged'
    };
    const fileStatuses = {};
    output && output.trim().split(_helpers.LINE_ENDING_REGEX).forEach(line => {
      const [status, rawFilePath] = line.split('\t');
      const filePath = (0, _helpers.toNativePathSep)(rawFilePath);
      fileStatuses[filePath] = statusMap[status];
    });

    if (!options.staged) {
      const untracked = await this.getUntrackedFiles();
      untracked.forEach(filePath => {
        fileStatuses[filePath] = 'added';
      });
    }

    return fileStatuses;
  }

  async getUntrackedFiles() {
    const output = await this.exec(['ls-files', '--others', '--exclude-standard']);

    if (output.trim() === '') {
      return [];
    }

    return output.trim().split(_helpers.LINE_ENDING_REGEX).map(_helpers.toNativePathSep);
  }

  async getDiffsForFilePath(filePath, {
    staged,
    baseCommit
  } = {}) {
    let args = ['diff', '--no-prefix', '--no-ext-diff', '--no-renames', '--diff-filter=u'];

    if (staged) {
      args.push('--staged');
    }

    if (baseCommit) {
      args.push(baseCommit);
    }

    args = args.concat(['--', (0, _helpers.toGitPathSep)(filePath)]);
    const output = await this.exec(args);
    let rawDiffs = [];

    if (output) {
      rawDiffs = (0, _whatTheDiff.parse)(output).filter(rawDiff => rawDiff.status !== 'unmerged');

      for (let i = 0; i < rawDiffs.length; i++) {
        const rawDiff = rawDiffs[i];

        if (rawDiff.oldPath) {
          rawDiff.oldPath = (0, _helpers.toNativePathSep)(rawDiff.oldPath);
        }

        if (rawDiff.newPath) {
          rawDiff.newPath = (0, _helpers.toNativePathSep)(rawDiff.newPath);
        }
      }
    }

    if (!staged && (await this.getUntrackedFiles()).includes(filePath)) {
      // add untracked file
      const absPath = _path["default"].join(this.workingDir, filePath);

      const executable = await (0, _helpers.isFileExecutable)(absPath);
      const symlink = await (0, _helpers.isFileSymlink)(absPath);
      const contents = await _fsExtra["default"].readFile(absPath, {
        encoding: 'utf8'
      });
      const binary = (0, _helpers.isBinary)(contents);
      let mode;
      let realpath;

      if (executable) {
        mode = _file["default"].modes.EXECUTABLE;
      } else if (symlink) {
        mode = _file["default"].modes.SYMLINK;
        realpath = await _fsExtra["default"].realpath(absPath);
      } else {
        mode = _file["default"].modes.NORMAL;
      }

      rawDiffs.push(buildAddedFilePatch(filePath, binary ? null : contents, mode, realpath));
    }

    if (rawDiffs.length > 2) {
      throw new Error(`Expected between 0 and 2 diffs for ${filePath} but got ${rawDiffs.length}`);
    }

    return rawDiffs;
  }

  async getStagedChangesPatch() {
    const output = await this.exec(['diff', '--staged', '--no-prefix', '--no-ext-diff', '--no-renames', '--diff-filter=u']);

    if (!output) {
      return [];
    }

    const diffs = (0, _whatTheDiff.parse)(output);

    for (const diff of diffs) {
      if (diff.oldPath) {
        diff.oldPath = (0, _helpers.toNativePathSep)(diff.oldPath);
      }

      if (diff.newPath) {
        diff.newPath = (0, _helpers.toNativePathSep)(diff.newPath);
      }
    }

    return diffs;
  }
  /**
   * Miscellaneous getters
   */


  async getCommit(ref) {
    const [commit] = await this.getCommits({
      max: 1,
      ref,
      includeUnborn: true
    });
    return commit;
  }

  async getHeadCommit() {
    const [headCommit] = await this.getCommits({
      max: 1,
      ref: 'HEAD',
      includeUnborn: true
    });
    return headCommit;
  }

  async getCommits(options = {}) {
    const {
      max,
      ref,
      includeUnborn,
      includePatch
    } = _objectSpread({
      max: 1,
      ref: 'HEAD',
      includeUnborn: false,
      includePatch: false
    }, options); // https://git-scm.com/docs/git-log#_pretty_formats
    // %x00 - null byte
    // %H - commit SHA
    // %ae - author email
    // %an = author full name
    // %at - timestamp, UNIX timestamp
    // %s - subject
    // %b - body


    const args = ['log', '--pretty=format:%H%x00%ae%x00%an%x00%at%x00%s%x00%b%x00', '--no-abbrev-commit', '--no-prefix', '--no-ext-diff', '--no-renames', '-z', '-n', max, ref];

    if (includePatch) {
      args.push('--patch', '-m', '--first-parent');
    }

    const output = await this.exec(args.concat('--'))["catch"](err => {
      if (/unknown revision/.test(err.stdErr) || /bad revision 'HEAD'/.test(err.stdErr)) {
        return '';
      } else {
        throw err;
      }
    });

    if (output === '') {
      return includeUnborn ? [{
        sha: '',
        message: '',
        unbornRef: true
      }] : [];
    }

    const fields = output.trim().split('\0');
    const commits = [];

    for (let i = 0; i < fields.length; i += 7) {
      const body = fields[i + 5].trim();
      let patch = [];

      if (includePatch) {
        const diffs = fields[i + 6];
        patch = (0, _whatTheDiff.parse)(diffs.trim());
      }

      const {
        message: messageBody,
        coAuthors
      } = (0, _helpers.extractCoAuthorsAndRawCommitMessage)(body);
      commits.push({
        sha: fields[i] && fields[i].trim(),
        authorEmail: fields[i + 1] && fields[i + 1].trim(),
        authorName: fields[i + 2] && fields[i + 2].trim(),
        authorDate: parseInt(fields[i + 3], 10),
        messageSubject: fields[i + 4],
        messageBody,
        coAuthors,
        unbornRef: false,
        patch
      });
    }

    return commits;
  }

  async getAuthors(options = {}) {
    const {
      max,
      ref
    } = _objectSpread({
      max: 1,
      ref: 'HEAD'
    }, options); // https://git-scm.com/docs/git-log#_pretty_formats
    // %x1F - field separator byte
    // %an - author name
    // %ae - author email
    // %cn - committer name
    // %ce - committer email
    // %(trailers:unfold,only) - the commit message trailers, separated
    //                           by newlines and unfolded (i.e. properly
    //                           formatted and one trailer per line).


    const delimiter = '1F';
    const delimiterString = String.fromCharCode(parseInt(delimiter, 16));
    const fields = ['%an', '%ae', '%cn', '%ce', '%(trailers:unfold,only)'];
    const format = fields.join(`%x${delimiter}`);

    try {
      const output = await this.exec(['log', `--format=${format}`, '-z', '-n', max, ref, '--']);
      return output.split('\0').reduce((acc, line) => {
        if (line.length === 0) {
          return acc;
        }

        const [an, ae, cn, ce, trailers] = line.split(delimiterString);
        trailers.split('\n').map(trailer => trailer.match(_helpers.CO_AUTHOR_REGEX)).filter(match => match !== null).forEach(([_, name, email]) => {
          acc[email] = name;
        });
        acc[ae] = an;
        acc[ce] = cn;
        return acc;
      }, {});
    } catch (err) {
      if (/unknown revision/.test(err.stdErr) || /bad revision 'HEAD'/.test(err.stdErr)) {
        return [];
      } else {
        throw err;
      }
    }
  }

  mergeTrailers(commitMessage, trailers) {
    const args = ['interpret-trailers'];

    for (const trailer of trailers) {
      args.push('--trailer', `${trailer.token}=${trailer.value}`);
    }

    return this.exec(args, {
      stdin: commitMessage
    });
  }

  readFileFromIndex(filePath) {
    return this.exec(['show', `:${(0, _helpers.toGitPathSep)(filePath)}`]);
  }
  /**
   * Merge
   */


  merge(branchName) {
    return this.gpgExec(['merge', branchName], {
      writeOperation: true
    });
  }

  isMerging(dotGitDir) {
    return (0, _helpers.fileExists)(_path["default"].join(dotGitDir, 'MERGE_HEAD'))["catch"](() => false);
  }

  abortMerge() {
    return this.exec(['merge', '--abort'], {
      writeOperation: true
    });
  }

  checkoutSide(side, paths) {
    if (paths.length === 0) {
      return Promise.resolve();
    }

    return this.exec(['checkout', `--${side}`, ...paths.map(_helpers.toGitPathSep)]);
  }
  /**
   * Rebase
   */


  async isRebasing(dotGitDir) {
    const results = await Promise.all([(0, _helpers.fileExists)(_path["default"].join(dotGitDir, 'rebase-merge')), (0, _helpers.fileExists)(_path["default"].join(dotGitDir, 'rebase-apply'))]);
    return results.some(r => r);
  }
  /**
   * Remote interactions
   */


  clone(remoteUrl, options = {}) {
    const args = ['clone'];

    if (options.noLocal) {
      args.push('--no-local');
    }

    if (options.bare) {
      args.push('--bare');
    }

    if (options.recursive) {
      args.push('--recursive');
    }

    args.push(remoteUrl, this.workingDir);
    return this.exec(args, {
      useGitPromptServer: true,
      writeOperation: true
    });
  }

  fetch(remoteName, branchName) {
    return this.exec(['fetch', remoteName, branchName], {
      useGitPromptServer: true,
      writeOperation: true
    });
  }

  pull(remoteName, branchName, options = {}) {
    const args = ['pull', remoteName, options.refSpec || branchName];

    if (options.ffOnly) {
      args.push('--ff-only');
    }

    return this.gpgExec(args, {
      useGitPromptServer: true,
      writeOperation: true
    });
  }

  push(remoteName, branchName, options = {}) {
    const args = ['push', remoteName || 'origin', options.refSpec || `refs/heads/${branchName}`];

    if (options.setUpstream) {
      args.push('--set-upstream');
    }

    if (options.force) {
      args.push('--force');
    }

    return this.exec(args, {
      useGitPromptServer: true,
      writeOperation: true
    });
  }
  /**
   * Undo Operations
   */


  reset(type, revision = 'HEAD') {
    const validTypes = ['soft'];

    if (!validTypes.includes(type)) {
      throw new Error(`Invalid type ${type}. Must be one of: ${validTypes.join(', ')}`);
    }

    return this.exec(['reset', `--${type}`, revision]);
  }

  deleteRef(ref) {
    return this.exec(['update-ref', '-d', ref]);
  }
  /**
   * Branches
   */


  checkout(branchName, options = {}) {
    const args = ['checkout'];

    if (options.createNew) {
      args.push('-b');
    }

    args.push(branchName);

    if (options.startPoint) {
      if (options.track) {
        args.push('--track');
      }

      args.push(options.startPoint);
    }

    return this.exec(args, {
      writeOperation: true
    });
  }

  async getBranches() {
    const format = ['%(objectname)', '%(HEAD)', '%(refname:short)', '%(upstream)', '%(upstream:remotename)', '%(upstream:remoteref)', '%(push)', '%(push:remotename)', '%(push:remoteref)'].join('%00');
    const output = await this.exec(['for-each-ref', `--format=${format}`, 'refs/heads/**']);
    return output.trim().split(_helpers.LINE_ENDING_REGEX).map(line => {
      const [sha, head, name, upstreamTrackingRef, upstreamRemoteName, upstreamRemoteRef, pushTrackingRef, pushRemoteName, pushRemoteRef] = line.split('\0');
      const branch = {
        name,
        sha,
        head: head === '*'
      };

      if (upstreamTrackingRef || upstreamRemoteName || upstreamRemoteRef) {
        branch.upstream = {
          trackingRef: upstreamTrackingRef,
          remoteName: upstreamRemoteName,
          remoteRef: upstreamRemoteRef
        };
      }

      if (branch.upstream || pushTrackingRef || pushRemoteName || pushRemoteRef) {
        branch.push = {
          trackingRef: pushTrackingRef,
          remoteName: pushRemoteName || branch.upstream && branch.upstream.remoteName,
          remoteRef: pushRemoteRef || branch.upstream && branch.upstream.remoteRef
        };
      }

      return branch;
    });
  }

  async getBranchesWithCommit(sha, option = {}) {
    const args = ['branch', '--format=%(refname)', '--contains', sha];

    if (option.showLocal && option.showRemote) {
      args.splice(1, 0, '--all');
    } else if (option.showRemote) {
      args.splice(1, 0, '--remotes');
    }

    return (await this.exec(args)).trim().split(_helpers.LINE_ENDING_REGEX);
  }

  checkoutFiles(paths, revision) {
    if (paths.length === 0) {
      return null;
    }

    const args = ['checkout'];

    if (revision) {
      args.push(revision);
    }

    return this.exec(args.concat('--', paths.map(_helpers.toGitPathSep)), {
      writeOperation: true
    });
  }

  async describeHead() {
    return (await this.exec(['describe', '--contains', '--all', '--always', 'HEAD'])).trim();
  }

  async getConfig(option, {
    local
  } = {}) {
    let output;

    try {
      let args = ['config'];

      if (local || atom.inSpecMode()) {
        args.push('--local');
      }

      args = args.concat(option);
      output = await this.exec(args);
    } catch (err) {
      if (err.code === 1) {
        // No matching config found
        return null;
      } else {
        throw err;
      }
    }

    return output.trim();
  }

  setConfig(option, value, {
    replaceAll
  } = {}) {
    let args = ['config'];

    if (replaceAll) {
      args.push('--replace-all');
    }

    args = args.concat(option, value);
    return this.exec(args, {
      writeOperation: true
    });
  }

  unsetConfig(option) {
    return this.exec(['config', '--unset', option], {
      writeOperation: true
    });
  }

  async getRemotes() {
    let output = await this.getConfig(['--get-regexp', '^remote\\..*\\.url$'], {
      local: true
    });

    if (output) {
      output = output.trim();

      if (!output.length) {
        return [];
      }

      return output.split('\n').map(line => {
        const match = line.match(/^remote\.(.*)\.url (.*)$/);
        return {
          name: match[1],
          url: match[2]
        };
      });
    } else {
      return [];
    }
  }

  addRemote(name, url) {
    return this.exec(['remote', 'add', name, url]);
  }

  async createBlob({
    filePath,
    stdin
  } = {}) {
    let output;

    if (filePath) {
      try {
        output = (await this.exec(['hash-object', '-w', filePath], {
          writeOperation: true
        })).trim();
      } catch (e) {
        if (e.stdErr && e.stdErr.match(/fatal: Cannot open .*: No such file or directory/)) {
          output = null;
        } else {
          throw e;
        }
      }
    } else if (stdin) {
      output = (await this.exec(['hash-object', '-w', '--stdin'], {
        stdin,
        writeOperation: true
      })).trim();
    } else {
      throw new Error('Must supply file path or stdin');
    }

    return output;
  }

  async expandBlobToFile(absFilePath, sha) {
    const output = await this.exec(['cat-file', '-p', sha]);
    await _fsExtra["default"].writeFile(absFilePath, output, {
      encoding: 'utf8'
    });
    return absFilePath;
  }

  async getBlobContents(sha) {
    return await this.exec(['cat-file', '-p', sha]);
  }

  async mergeFile(oursPath, commonBasePath, theirsPath, resultPath) {
    const args = ['merge-file', '-p', oursPath, commonBasePath, theirsPath, '-L', 'current', '-L', 'after discard', '-L', 'before discard'];
    let output;
    let conflict = false;

    try {
      output = await this.exec(args);
    } catch (e) {
      if (e instanceof GitError && e.code === 1) {
        output = e.stdOut;
        conflict = true;
      } else {
        throw e;
      }
    } // Interpret a relative resultPath as relative to the repository working directory for consistency with the
    // other arguments.


    const resolvedResultPath = _path["default"].resolve(this.workingDir, resultPath);

    await _fsExtra["default"].writeFile(resolvedResultPath, output, {
      encoding: 'utf8'
    });
    return {
      filePath: oursPath,
      resultPath,
      conflict
    };
  }

  async writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha) {
    const gitFilePath = (0, _helpers.toGitPathSep)(filePath);
    const fileMode = await this.getFileMode(filePath);
    let indexInfo = `0 0000000000000000000000000000000000000000\t${gitFilePath}\n`;

    if (commonBaseSha) {
      indexInfo += `${fileMode} ${commonBaseSha} 1\t${gitFilePath}\n`;
    }

    if (oursSha) {
      indexInfo += `${fileMode} ${oursSha} 2\t${gitFilePath}\n`;
    }

    if (theirsSha) {
      indexInfo += `${fileMode} ${theirsSha} 3\t${gitFilePath}\n`;
    }

    return this.exec(['update-index', '--index-info'], {
      stdin: indexInfo,
      writeOperation: true
    });
  }

  async getFileMode(filePath) {
    const output = await this.exec(['ls-files', '--stage', '--', (0, _helpers.toGitPathSep)(filePath)]);

    if (output) {
      return output.slice(0, 6);
    } else {
      const executable = await (0, _helpers.isFileExecutable)(_path["default"].join(this.workingDir, filePath));
      const symlink = await (0, _helpers.isFileSymlink)(_path["default"].join(this.workingDir, filePath));

      if (symlink) {
        return _file["default"].modes.SYMLINK;
      } else if (executable) {
        return _file["default"].modes.EXECUTABLE;
      } else {
        return _file["default"].modes.NORMAL;
      }
    }
  }

  destroy() {
    this.commandQueue.dispose();
  }

}

exports["default"] = GitShellOutStrategy;

_defineProperty(GitShellOutStrategy, "defaultExecArgs", {
  stdin: null,
  useGitPromptServer: false,
  useGpgWrapper: false,
  useGpgAtomPrompt: false,
  writeOperation: false
});

function buildAddedFilePatch(filePath, contents, mode, realpath) {
  const hunks = [];

  if (contents) {
    let noNewLine;
    let lines;

    if (mode === _file["default"].modes.SYMLINK) {
      noNewLine = false;
      lines = [`+${(0, _helpers.toGitPathSep)(realpath)}`, '\\ No newline at end of file'];
    } else {
      noNewLine = contents[contents.length - 1] !== '\n';
      lines = contents.trim().split(_helpers.LINE_ENDING_REGEX).map(line => `+${line}`);
    }

    if (noNewLine) {
      lines.push('\\ No newline at end of file');
    }

    hunks.push({
      lines,
      oldStartLine: 0,
      oldLineCount: 0,
      newStartLine: 1,
      heading: '',
      newLineCount: noNewLine ? lines.length - 1 : lines.length
    });
  }

  return {
    oldPath: null,
    newPath: (0, _helpers.toNativePathSep)(filePath),
    oldMode: null,
    newMode: mode,
    status: 'added',
    hunks
  };
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImdpdC1zaGVsbC1vdXQtc3RyYXRlZ3kuanMiXSwibmFtZXMiOlsiTUFYX1NUQVRVU19PVVRQVVRfTEVOR1RIIiwiaGVhZGxlc3MiLCJleGVjUGF0aFByb21pc2UiLCJHaXRFcnJvciIsIkVycm9yIiwiY29uc3RydWN0b3IiLCJtZXNzYWdlIiwic3RhY2siLCJMYXJnZVJlcG9FcnJvciIsIklHTk9SRURfR0lUX0NPTU1BTkRTIiwiRElTQUJMRV9DT0xPUl9GTEFHUyIsInJlZHVjZSIsImFjYyIsInR5cGUiLCJ1bnNoaWZ0IiwiRVhQQU5EX1RJTERFX1JFR0VYIiwiUmVnRXhwIiwiR2l0U2hlbGxPdXRTdHJhdGVneSIsIndvcmtpbmdEaXIiLCJvcHRpb25zIiwicXVldWUiLCJjb21tYW5kUXVldWUiLCJwYXJhbGxlbGlzbSIsIk1hdGgiLCJtYXgiLCJvcyIsImNwdXMiLCJsZW5ndGgiLCJBc3luY1F1ZXVlIiwicHJvbXB0IiwicXVlcnkiLCJQcm9taXNlIiwicmVqZWN0Iiwid29ya2VyTWFuYWdlciIsInJlbW90ZSIsImdldEN1cnJlbnRXaW5kb3ciLCJpc1Zpc2libGUiLCJzZXRQcm9tcHRDYWxsYmFjayIsImV4ZWMiLCJhcmdzIiwiZGVmYXVsdEV4ZWNBcmdzIiwic3RkaW4iLCJ1c2VHaXRQcm9tcHRTZXJ2ZXIiLCJ1c2VHcGdXcmFwcGVyIiwidXNlR3BnQXRvbVByb21wdCIsIndyaXRlT3BlcmF0aW9uIiwiY29tbWFuZE5hbWUiLCJzdWJzY3JpcHRpb25zIiwiQ29tcG9zaXRlRGlzcG9zYWJsZSIsImRpYWdub3N0aWNzRW5hYmxlZCIsInByb2Nlc3MiLCJlbnYiLCJBVE9NX0dJVEhVQl9HSVRfRElBR05PU1RJQ1MiLCJhdG9tIiwiY29uZmlnIiwiZ2V0IiwiZm9ybWF0dGVkQXJncyIsImpvaW4iLCJ0aW1pbmdNYXJrZXIiLCJHaXRUaW1pbmdzVmlldyIsImdlbmVyYXRlTWFya2VyIiwibWFyayIsInJlc29sdmUiLCJjaGlsZFByb2Nlc3MiLCJlcnJvciIsInN0ZG91dCIsInN0ZGVyciIsInRyaW0iLCJleGVjUGF0aCIsInB1c2giLCJnaXRQcm9tcHRTZXJ2ZXIiLCJwYXRoUGFydHMiLCJQQVRIIiwiR0lUX1RFUk1JTkFMX1BST01QVCIsIkdJVF9PUFRJT05BTF9MT0NLUyIsInBhdGgiLCJkZWxpbWl0ZXIiLCJnaXRUZW1wRGlyIiwiR2l0VGVtcERpciIsImVuc3VyZSIsImdldEdwZ1dyYXBwZXJTaCIsIkdpdFByb21wdFNlcnZlciIsInN0YXJ0IiwiQVRPTV9HSVRIVUJfVE1QIiwiZ2V0Um9vdFBhdGgiLCJBVE9NX0dJVEhVQl9BU0tQQVNTX1BBVEgiLCJnZXRBc2tQYXNzSnMiLCJBVE9NX0dJVEhVQl9DUkVERU5USUFMX1BBVEgiLCJnZXRDcmVkZW50aWFsSGVscGVySnMiLCJBVE9NX0dJVEhVQl9FTEVDVFJPTl9QQVRIIiwiQVRPTV9HSVRIVUJfU09DS19QQVRIIiwiZ2V0U29ja2V0UGF0aCIsIkFUT01fR0lUSFVCX1dPUktESVJfUEFUSCIsIkFUT01fR0lUSFVCX0RVR0lURV9QQVRIIiwiQVRPTV9HSVRIVUJfS0VZVEFSX1NUUkFURUdZX1BBVEgiLCJESVNQTEFZIiwiQVRPTV9HSVRIVUJfT1JJR0lOQUxfUEFUSCIsIkFUT01fR0lUSFVCX09SSUdJTkFMX0dJVF9BU0tQQVNTIiwiR0lUX0FTS1BBU1MiLCJBVE9NX0dJVEhVQl9PUklHSU5BTF9TU0hfQVNLUEFTUyIsIlNTSF9BU0tQQVNTIiwiQVRPTV9HSVRIVUJfT1JJR0lOQUxfR0lUX1NTSF9DT01NQU5EIiwiR0lUX1NTSF9DT01NQU5EIiwiQVRPTV9HSVRIVUJfU1BFQ19NT0RFIiwiaW5TcGVjTW9kZSIsImdldEFza1Bhc3NTaCIsInBsYXRmb3JtIiwiZ2V0U3NoV3JhcHBlclNoIiwiY3JlZGVudGlhbEhlbHBlclNoIiwiZ2V0Q3JlZGVudGlhbEhlbHBlclNoIiwiQVRPTV9HSVRIVUJfR1BHX1BST01QVCIsIkdJVF9UUkFDRSIsIkdJVF9UUkFDRV9DVVJMIiwib3B0cyIsInN0ZGluRW5jb2RpbmciLCJQUklOVF9HSVRfVElNRVMiLCJjb25zb2xlIiwidGltZSIsImJlZm9yZVJ1biIsIm5ld0FyZ3NPcHRzIiwicHJvbWlzZSIsImNhbmNlbCIsImV4ZWN1dGVHaXRDb21tYW5kIiwiZXhwZWN0Q2FuY2VsIiwiYWRkIiwib25EaWRDYW5jZWwiLCJoYW5kbGVyUGlkIiwicmVzb2x2ZUtpbGwiLCJyZWplY3RLaWxsIiwicmVxdWlyZSIsImVyciIsImV4aXRDb2RlIiwic2lnbmFsIiwidGltaW5nIiwiZXhlY1RpbWUiLCJzcGF3blRpbWUiLCJpcGNUaW1lIiwibm93IiwicGVyZm9ybWFuY2UiLCJmaW5hbGl6ZSIsInRpbWVFbmQiLCJ0ZXJtaW5hdGUiLCJkaXNwb3NlIiwiZXhwb3NlQ29udHJvbENoYXJhY3RlcnMiLCJyYXciLCJyZXBsYWNlIiwic3VtbWFyeSIsInVuZGVmaW5lZCIsImxvZyIsImhlYWRlclN0eWxlIiwiZ3JvdXBDb2xsYXBzZWQiLCJ1dGlsIiwiaW5zcGVjdCIsImJyZWFrTGVuZ3RoIiwiSW5maW5pdHkiLCJncm91cEVuZCIsImNvZGUiLCJzdGRFcnIiLCJzdGRPdXQiLCJjb21tYW5kIiwiaW5jbHVkZXMiLCJwYXJhbGxlbCIsImdwZ0V4ZWMiLCJzbGljZSIsImUiLCJ0ZXN0IiwibWFya2VyIiwiQVRPTV9HSVRIVUJfSU5MSU5FX0dJVF9FWEVDIiwiV29ya2VyTWFuYWdlciIsImdldEluc3RhbmNlIiwiaXNSZWFkeSIsImNoaWxkUGlkIiwicHJvY2Vzc0NhbGxiYWNrIiwiY2hpbGQiLCJwaWQiLCJvbiIsIkdpdFByb2Nlc3MiLCJyZXF1ZXN0IiwicmVzb2x2ZURvdEdpdERpciIsImZzIiwic3RhdCIsIm91dHB1dCIsImRvdEdpdERpciIsImluaXQiLCJzdGFnZUZpbGVzIiwicGF0aHMiLCJjb25jYXQiLCJtYXAiLCJ0b0dpdFBhdGhTZXAiLCJmZXRjaENvbW1pdE1lc3NhZ2VUZW1wbGF0ZSIsInRlbXBsYXRlUGF0aCIsImdldENvbmZpZyIsImhvbWVEaXIiLCJob21lZGlyIiwiXyIsInVzZXIiLCJkaXJuYW1lIiwiaXNBYnNvbHV0ZSIsInJlYWRGaWxlIiwiZW5jb2RpbmciLCJ1bnN0YWdlRmlsZXMiLCJjb21taXQiLCJzdGFnZUZpbGVNb2RlQ2hhbmdlIiwiZmlsZW5hbWUiLCJuZXdNb2RlIiwiaW5kZXhSZWFkUHJvbWlzZSIsImRldGVybWluZUFyZ3MiLCJpbmRleCIsIm9pZCIsInN1YnN0ciIsInN0YWdlRmlsZVN5bWxpbmtDaGFuZ2UiLCJhcHBseVBhdGNoIiwicGF0Y2giLCJzcGxpY2UiLCJyYXdNZXNzYWdlIiwiYWxsb3dFbXB0eSIsImFtZW5kIiwiY29BdXRob3JzIiwidmVyYmF0aW0iLCJtc2ciLCJ1bmJvcm5SZWYiLCJtZXNzYWdlQm9keSIsIm1lc3NhZ2VTdWJqZWN0IiwiZ2V0SGVhZENvbW1pdCIsInRlbXBsYXRlIiwiY29tbWVudENoYXIiLCJzcGxpdCIsImZpbHRlciIsImxpbmUiLCJzdGFydHNXaXRoIiwiY29uZmlndXJlZCIsIm1vZGUiLCJhZGRDb0F1dGhvcnNUb01lc3NhZ2UiLCJ0cmFpbGVycyIsImF1dGhvciIsInRva2VuIiwidmFsdWUiLCJuYW1lIiwiZW1haWwiLCJtZXJnZVRyYWlsZXJzIiwiZ2V0U3RhdHVzQnVuZGxlIiwicmVzdWx0cyIsImVudHJ5VHlwZSIsIkFycmF5IiwiaXNBcnJheSIsInVwZGF0ZU5hdGl2ZVBhdGhTZXBGb3JFbnRyaWVzIiwiZW50cmllcyIsImZvckVhY2giLCJlbnRyeSIsImZpbGVQYXRoIiwib3JpZ0ZpbGVQYXRoIiwiZGlmZkZpbGVTdGF0dXMiLCJzdGFnZWQiLCJ0YXJnZXQiLCJzdGF0dXNNYXAiLCJBIiwiTSIsIkQiLCJVIiwiZmlsZVN0YXR1c2VzIiwiTElORV9FTkRJTkdfUkVHRVgiLCJzdGF0dXMiLCJyYXdGaWxlUGF0aCIsInVudHJhY2tlZCIsImdldFVudHJhY2tlZEZpbGVzIiwidG9OYXRpdmVQYXRoU2VwIiwiZ2V0RGlmZnNGb3JGaWxlUGF0aCIsImJhc2VDb21taXQiLCJyYXdEaWZmcyIsInJhd0RpZmYiLCJpIiwib2xkUGF0aCIsIm5ld1BhdGgiLCJhYnNQYXRoIiwiZXhlY3V0YWJsZSIsInN5bWxpbmsiLCJjb250ZW50cyIsImJpbmFyeSIsInJlYWxwYXRoIiwiRmlsZSIsIm1vZGVzIiwiRVhFQ1VUQUJMRSIsIlNZTUxJTksiLCJOT1JNQUwiLCJidWlsZEFkZGVkRmlsZVBhdGNoIiwiZ2V0U3RhZ2VkQ2hhbmdlc1BhdGNoIiwiZGlmZnMiLCJkaWZmIiwiZ2V0Q29tbWl0IiwicmVmIiwiZ2V0Q29tbWl0cyIsImluY2x1ZGVVbmJvcm4iLCJoZWFkQ29tbWl0IiwiaW5jbHVkZVBhdGNoIiwic2hhIiwiZmllbGRzIiwiY29tbWl0cyIsImJvZHkiLCJhdXRob3JFbWFpbCIsImF1dGhvck5hbWUiLCJhdXRob3JEYXRlIiwicGFyc2VJbnQiLCJnZXRBdXRob3JzIiwiZGVsaW1pdGVyU3RyaW5nIiwiU3RyaW5nIiwiZnJvbUNoYXJDb2RlIiwiZm9ybWF0IiwiYW4iLCJhZSIsImNuIiwiY2UiLCJ0cmFpbGVyIiwibWF0Y2giLCJDT19BVVRIT1JfUkVHRVgiLCJjb21taXRNZXNzYWdlIiwicmVhZEZpbGVGcm9tSW5kZXgiLCJtZXJnZSIsImJyYW5jaE5hbWUiLCJpc01lcmdpbmciLCJhYm9ydE1lcmdlIiwiY2hlY2tvdXRTaWRlIiwic2lkZSIsImlzUmViYXNpbmciLCJhbGwiLCJzb21lIiwiciIsImNsb25lIiwicmVtb3RlVXJsIiwibm9Mb2NhbCIsImJhcmUiLCJyZWN1cnNpdmUiLCJmZXRjaCIsInJlbW90ZU5hbWUiLCJwdWxsIiwicmVmU3BlYyIsImZmT25seSIsInNldFVwc3RyZWFtIiwiZm9yY2UiLCJyZXNldCIsInJldmlzaW9uIiwidmFsaWRUeXBlcyIsImRlbGV0ZVJlZiIsImNoZWNrb3V0IiwiY3JlYXRlTmV3Iiwic3RhcnRQb2ludCIsInRyYWNrIiwiZ2V0QnJhbmNoZXMiLCJoZWFkIiwidXBzdHJlYW1UcmFja2luZ1JlZiIsInVwc3RyZWFtUmVtb3RlTmFtZSIsInVwc3RyZWFtUmVtb3RlUmVmIiwicHVzaFRyYWNraW5nUmVmIiwicHVzaFJlbW90ZU5hbWUiLCJwdXNoUmVtb3RlUmVmIiwiYnJhbmNoIiwidXBzdHJlYW0iLCJ0cmFja2luZ1JlZiIsInJlbW90ZVJlZiIsImdldEJyYW5jaGVzV2l0aENvbW1pdCIsIm9wdGlvbiIsInNob3dMb2NhbCIsInNob3dSZW1vdGUiLCJjaGVja291dEZpbGVzIiwiZGVzY3JpYmVIZWFkIiwibG9jYWwiLCJzZXRDb25maWciLCJyZXBsYWNlQWxsIiwidW5zZXRDb25maWciLCJnZXRSZW1vdGVzIiwidXJsIiwiYWRkUmVtb3RlIiwiY3JlYXRlQmxvYiIsImV4cGFuZEJsb2JUb0ZpbGUiLCJhYnNGaWxlUGF0aCIsIndyaXRlRmlsZSIsImdldEJsb2JDb250ZW50cyIsIm1lcmdlRmlsZSIsIm91cnNQYXRoIiwiY29tbW9uQmFzZVBhdGgiLCJ0aGVpcnNQYXRoIiwicmVzdWx0UGF0aCIsImNvbmZsaWN0IiwicmVzb2x2ZWRSZXN1bHRQYXRoIiwid3JpdGVNZXJnZUNvbmZsaWN0VG9JbmRleCIsImNvbW1vbkJhc2VTaGEiLCJvdXJzU2hhIiwidGhlaXJzU2hhIiwiZ2l0RmlsZVBhdGgiLCJmaWxlTW9kZSIsImdldEZpbGVNb2RlIiwiaW5kZXhJbmZvIiwiZGVzdHJveSIsImh1bmtzIiwibm9OZXdMaW5lIiwibGluZXMiLCJvbGRTdGFydExpbmUiLCJvbGRMaW5lQ291bnQiLCJuZXdTdGFydExpbmUiLCJoZWFkaW5nIiwibmV3TGluZUNvdW50Iiwib2xkTW9kZSJdLCJtYXBwaW5ncyI6Ijs7Ozs7OztBQUFBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUVBOztBQUNBOztBQUNBOztBQUNBOztBQUVBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUtBOztBQUNBOztBQUNBOzs7Ozs7OztBQUVBLE1BQU1BLHdCQUF3QixHQUFHLE9BQU8sSUFBUCxHQUFjLEVBQS9DO0FBRUEsSUFBSUMsUUFBUSxHQUFHLElBQWY7QUFDQSxJQUFJQyxlQUFlLEdBQUcsSUFBdEI7O0FBRU8sTUFBTUMsUUFBTixTQUF1QkMsS0FBdkIsQ0FBNkI7QUFDbENDLEVBQUFBLFdBQVcsQ0FBQ0MsT0FBRCxFQUFVO0FBQ25CLFVBQU1BLE9BQU47QUFDQSxTQUFLQSxPQUFMLEdBQWVBLE9BQWY7QUFDQSxTQUFLQyxLQUFMLEdBQWEsSUFBSUgsS0FBSixHQUFZRyxLQUF6QjtBQUNEOztBQUxpQzs7OztBQVE3QixNQUFNQyxjQUFOLFNBQTZCSixLQUE3QixDQUFtQztBQUN4Q0MsRUFBQUEsV0FBVyxDQUFDQyxPQUFELEVBQVU7QUFDbkIsVUFBTUEsT0FBTjtBQUNBLFNBQUtBLE9BQUwsR0FBZUEsT0FBZjtBQUNBLFNBQUtDLEtBQUwsR0FBYSxJQUFJSCxLQUFKLEdBQVlHLEtBQXpCO0FBQ0Q7O0FBTHVDLEMsQ0FRMUM7Ozs7QUFDQSxNQUFNRSxvQkFBb0IsR0FBRyxDQUFDLFVBQUQsRUFBYSxRQUFiLEVBQXVCLE1BQXZCLEVBQStCLGNBQS9CLEVBQStDLEtBQS9DLEVBQXNELFdBQXRELEVBQW1FLFFBQW5FLENBQTdCO0FBRUEsTUFBTUMsbUJBQW1CLEdBQUcsQ0FDMUIsUUFEMEIsRUFDaEIsTUFEZ0IsRUFDUixZQURRLEVBQ00sUUFETixFQUNnQixJQURoQixFQUUxQkMsTUFGMEIsQ0FFbkIsQ0FBQ0MsR0FBRCxFQUFNQyxJQUFOLEtBQWU7QUFDdEJELEVBQUFBLEdBQUcsQ0FBQ0UsT0FBSixDQUFZLElBQVosRUFBbUIsU0FBUUQsSUFBSyxRQUFoQztBQUNBLFNBQU9ELEdBQVA7QUFDRCxDQUwyQixFQUt6QixFQUx5QixDQUE1QjtBQU9BOzs7Ozs7Ozs7O0FBU0EsTUFBTUcsa0JBQWtCLEdBQUcsSUFBSUMsTUFBSixDQUFXLHNCQUFYLENBQTNCOztBQUVlLE1BQU1DLG1CQUFOLENBQTBCO0FBU3ZDWixFQUFBQSxXQUFXLENBQUNhLFVBQUQsRUFBYUMsT0FBTyxHQUFHLEVBQXZCLEVBQTJCO0FBQ3BDLFNBQUtELFVBQUwsR0FBa0JBLFVBQWxCOztBQUNBLFFBQUlDLE9BQU8sQ0FBQ0MsS0FBWixFQUFtQjtBQUNqQixXQUFLQyxZQUFMLEdBQW9CRixPQUFPLENBQUNDLEtBQTVCO0FBQ0QsS0FGRCxNQUVPO0FBQ0wsWUFBTUUsV0FBVyxHQUFHSCxPQUFPLENBQUNHLFdBQVIsSUFBdUJDLElBQUksQ0FBQ0MsR0FBTCxDQUFTLENBQVQsRUFBWUMsZUFBR0MsSUFBSCxHQUFVQyxNQUF0QixDQUEzQztBQUNBLFdBQUtOLFlBQUwsR0FBb0IsSUFBSU8sc0JBQUosQ0FBZTtBQUFDTixRQUFBQTtBQUFELE9BQWYsQ0FBcEI7QUFDRDs7QUFFRCxTQUFLTyxNQUFMLEdBQWNWLE9BQU8sQ0FBQ1UsTUFBUixLQUFtQkMsS0FBSyxJQUFJQyxPQUFPLENBQUNDLE1BQVIsRUFBNUIsQ0FBZDs7QUFDQSxTQUFLQyxhQUFMLEdBQXFCZCxPQUFPLENBQUNjLGFBQTdCOztBQUVBLFFBQUloQyxRQUFRLEtBQUssSUFBakIsRUFBdUI7QUFDckJBLE1BQUFBLFFBQVEsR0FBRyxDQUFDaUMsaUJBQU9DLGdCQUFQLEdBQTBCQyxTQUExQixFQUFaO0FBQ0Q7QUFDRjtBQUVEOzs7Ozs7OztBQU1BQyxFQUFBQSxpQkFBaUIsQ0FBQ1IsTUFBRCxFQUFTO0FBQ3hCLFNBQUtBLE1BQUwsR0FBY0EsTUFBZDtBQUNELEdBbENzQyxDQW9DdkM7OztBQUNBLFFBQU1TLElBQU4sQ0FBV0MsSUFBWCxFQUFpQnBCLE9BQU8sR0FBR0YsbUJBQW1CLENBQUN1QixlQUEvQyxFQUFnRTtBQUM5RDtBQUNBLFVBQU07QUFBQ0MsTUFBQUEsS0FBRDtBQUFRQyxNQUFBQSxrQkFBUjtBQUE0QkMsTUFBQUEsYUFBNUI7QUFBMkNDLE1BQUFBLGdCQUEzQztBQUE2REMsTUFBQUE7QUFBN0QsUUFBK0UxQixPQUFyRjtBQUNBLFVBQU0yQixXQUFXLEdBQUdQLElBQUksQ0FBQyxDQUFELENBQXhCO0FBQ0EsVUFBTVEsYUFBYSxHQUFHLElBQUlDLDZCQUFKLEVBQXRCO0FBQ0EsVUFBTUMsa0JBQWtCLEdBQUdDLE9BQU8sQ0FBQ0MsR0FBUixDQUFZQywyQkFBWixJQUEyQ0MsSUFBSSxDQUFDQyxNQUFMLENBQVlDLEdBQVosQ0FBZ0IsdUJBQWhCLENBQXRFO0FBRUEsVUFBTUMsYUFBYSxHQUFJLE9BQU1qQixJQUFJLENBQUNrQixJQUFMLENBQVUsR0FBVixDQUFlLE9BQU0sS0FBS3ZDLFVBQVcsRUFBbEU7O0FBQ0EsVUFBTXdDLFlBQVksR0FBR0MsMkJBQWVDLGNBQWYsQ0FBK0IsT0FBTXJCLElBQUksQ0FBQ2tCLElBQUwsQ0FBVSxHQUFWLENBQWUsRUFBcEQsQ0FBckI7O0FBQ0FDLElBQUFBLFlBQVksQ0FBQ0csSUFBYixDQUFrQixRQUFsQjtBQUVBdEIsSUFBQUEsSUFBSSxDQUFDekIsT0FBTCxDQUFhLEdBQUdKLG1CQUFoQjs7QUFFQSxRQUFJUixlQUFlLEtBQUssSUFBeEIsRUFBOEI7QUFDNUI7QUFDQUEsTUFBQUEsZUFBZSxHQUFHLElBQUk2QixPQUFKLENBQVksQ0FBQytCLE9BQUQsRUFBVTlCLE1BQVYsS0FBcUI7QUFDakQrQixrQ0FBYXpCLElBQWIsQ0FBa0IsaUJBQWxCLEVBQXFDLENBQUMwQixLQUFELEVBQVFDLE1BQVIsRUFBZ0JDLE1BQWhCLEtBQTJCO0FBQzlEO0FBQ0EsY0FBSUYsS0FBSixFQUFXO0FBQ1Q7QUFDQUYsWUFBQUEsT0FBTyxDQUFDLElBQUQsQ0FBUDtBQUNBO0FBQ0Q7O0FBRURBLFVBQUFBLE9BQU8sQ0FBQ0csTUFBTSxDQUFDRSxJQUFQLEVBQUQsQ0FBUDtBQUNELFNBVEQ7QUFVRCxPQVhpQixDQUFsQjtBQVlEOztBQUNELFVBQU1DLFFBQVEsR0FBRyxNQUFNbEUsZUFBdkI7QUFFQSxXQUFPLEtBQUttQixZQUFMLENBQWtCZ0QsSUFBbEIsQ0FBdUIsWUFBWTtBQUN4Q1gsTUFBQUEsWUFBWSxDQUFDRyxJQUFiLENBQWtCLFNBQWxCO0FBQ0EsVUFBSVMsZUFBSjtBQUVBLFlBQU1DLFNBQVMsR0FBRyxFQUFsQjs7QUFDQSxVQUFJckIsT0FBTyxDQUFDQyxHQUFSLENBQVlxQixJQUFoQixFQUFzQjtBQUNwQkQsUUFBQUEsU0FBUyxDQUFDRixJQUFWLENBQWVuQixPQUFPLENBQUNDLEdBQVIsQ0FBWXFCLElBQTNCO0FBQ0Q7O0FBQ0QsVUFBSUosUUFBSixFQUFjO0FBQ1pHLFFBQUFBLFNBQVMsQ0FBQ0YsSUFBVixDQUFlRCxRQUFmO0FBQ0Q7O0FBRUQsWUFBTWpCLEdBQUcscUJBQ0pELE9BQU8sQ0FBQ0MsR0FESjtBQUVQc0IsUUFBQUEsbUJBQW1CLEVBQUUsR0FGZDtBQUdQQyxRQUFBQSxrQkFBa0IsRUFBRSxHQUhiO0FBSVBGLFFBQUFBLElBQUksRUFBRUQsU0FBUyxDQUFDZCxJQUFWLENBQWVrQixpQkFBS0MsU0FBcEI7QUFKQyxRQUFUOztBQU9BLFlBQU1DLFVBQVUsR0FBRyxJQUFJQyxzQkFBSixFQUFuQjs7QUFFQSxVQUFJbkMsYUFBSixFQUFtQjtBQUNqQixjQUFNa0MsVUFBVSxDQUFDRSxNQUFYLEVBQU47QUFDQXhDLFFBQUFBLElBQUksQ0FBQ3pCLE9BQUwsQ0FBYSxJQUFiLEVBQW9CLGVBQWMrRCxVQUFVLENBQUNHLGVBQVgsRUFBNkIsRUFBL0Q7QUFDRDs7QUFFRCxVQUFJdEMsa0JBQUosRUFBd0I7QUFDdEI0QixRQUFBQSxlQUFlLEdBQUcsSUFBSVcsMkJBQUosQ0FBb0JKLFVBQXBCLENBQWxCO0FBQ0EsY0FBTVAsZUFBZSxDQUFDWSxLQUFoQixDQUFzQixLQUFLckQsTUFBM0IsQ0FBTjtBQUVBc0IsUUFBQUEsR0FBRyxDQUFDZ0MsZUFBSixHQUFzQk4sVUFBVSxDQUFDTyxXQUFYLEVBQXRCO0FBQ0FqQyxRQUFBQSxHQUFHLENBQUNrQyx3QkFBSixHQUErQixxQ0FBdUJSLFVBQVUsQ0FBQ1MsWUFBWCxFQUF2QixDQUEvQjtBQUNBbkMsUUFBQUEsR0FBRyxDQUFDb0MsMkJBQUosR0FBa0MscUNBQXVCVixVQUFVLENBQUNXLHFCQUFYLEVBQXZCLENBQWxDO0FBQ0FyQyxRQUFBQSxHQUFHLENBQUNzQyx5QkFBSixHQUFnQyxxQ0FBdUIsaUNBQXZCLENBQWhDO0FBQ0F0QyxRQUFBQSxHQUFHLENBQUN1QyxxQkFBSixHQUE0QixxQ0FBdUJiLFVBQVUsQ0FBQ2MsYUFBWCxFQUF2QixDQUE1QjtBQUVBeEMsUUFBQUEsR0FBRyxDQUFDeUMsd0JBQUosR0FBK0IsS0FBSzFFLFVBQXBDO0FBQ0FpQyxRQUFBQSxHQUFHLENBQUMwQyx1QkFBSixHQUE4Qiw2QkFBOUI7QUFDQTFDLFFBQUFBLEdBQUcsQ0FBQzJDLGdDQUFKLEdBQXVDLGtDQUFvQixpQkFBcEIsQ0FBdkMsQ0Fac0IsQ0FjdEI7QUFDQTtBQUNBO0FBQ0E7O0FBQ0EsWUFBSSxDQUFDNUMsT0FBTyxDQUFDQyxHQUFSLENBQVk0QyxPQUFiLElBQXdCN0MsT0FBTyxDQUFDQyxHQUFSLENBQVk0QyxPQUFaLENBQW9CcEUsTUFBcEIsS0FBK0IsQ0FBM0QsRUFBOEQ7QUFDNUR3QixVQUFBQSxHQUFHLENBQUM0QyxPQUFKLEdBQWMseUJBQWQ7QUFDRDs7QUFFRDVDLFFBQUFBLEdBQUcsQ0FBQzZDLHlCQUFKLEdBQWdDOUMsT0FBTyxDQUFDQyxHQUFSLENBQVlxQixJQUFaLElBQW9CLEVBQXBEO0FBQ0FyQixRQUFBQSxHQUFHLENBQUM4QyxnQ0FBSixHQUF1Qy9DLE9BQU8sQ0FBQ0MsR0FBUixDQUFZK0MsV0FBWixJQUEyQixFQUFsRTtBQUNBL0MsUUFBQUEsR0FBRyxDQUFDZ0QsZ0NBQUosR0FBdUNqRCxPQUFPLENBQUNDLEdBQVIsQ0FBWWlELFdBQVosSUFBMkIsRUFBbEU7QUFDQWpELFFBQUFBLEdBQUcsQ0FBQ2tELG9DQUFKLEdBQTJDbkQsT0FBTyxDQUFDQyxHQUFSLENBQVltRCxlQUFaLElBQStCLEVBQTFFO0FBQ0FuRCxRQUFBQSxHQUFHLENBQUNvRCxxQkFBSixHQUE0QmxELElBQUksQ0FBQ21ELFVBQUwsS0FBb0IsTUFBcEIsR0FBNkIsT0FBekQ7QUFFQXJELFFBQUFBLEdBQUcsQ0FBQ2lELFdBQUosR0FBa0IscUNBQXVCdkIsVUFBVSxDQUFDNEIsWUFBWCxFQUF2QixDQUFsQjtBQUNBdEQsUUFBQUEsR0FBRyxDQUFDK0MsV0FBSixHQUFrQixxQ0FBdUJyQixVQUFVLENBQUM0QixZQUFYLEVBQXZCLENBQWxCOztBQUVBLFlBQUl2RCxPQUFPLENBQUN3RCxRQUFSLEtBQXFCLE9BQXpCLEVBQWtDO0FBQ2hDdkQsVUFBQUEsR0FBRyxDQUFDbUQsZUFBSixHQUFzQnpCLFVBQVUsQ0FBQzhCLGVBQVgsRUFBdEI7QUFDRCxTQUZELE1BRU87QUFDTHhELFVBQUFBLEdBQUcsQ0FBQ21ELGVBQUosR0FBc0JwRCxPQUFPLENBQUNDLEdBQVIsQ0FBWW1ELGVBQWxDO0FBQ0Q7O0FBRUQsY0FBTU0sa0JBQWtCLEdBQUcscUNBQXVCL0IsVUFBVSxDQUFDZ0MscUJBQVgsRUFBdkIsQ0FBM0I7QUFDQXRFLFFBQUFBLElBQUksQ0FBQ3pCLE9BQUwsQ0FBYSxJQUFiLEVBQW9CLHFCQUFvQjhGLGtCQUFtQixFQUEzRDtBQUNEOztBQUVELFVBQUlqRSxhQUFhLElBQUlELGtCQUFqQixJQUF1Q0UsZ0JBQTNDLEVBQTZEO0FBQzNETyxRQUFBQSxHQUFHLENBQUMyRCxzQkFBSixHQUE2QixNQUE3QjtBQUNEO0FBRUQ7OztBQUNBLFVBQUk3RCxrQkFBSixFQUF3QjtBQUN0QkUsUUFBQUEsR0FBRyxDQUFDNEQsU0FBSixHQUFnQixNQUFoQjtBQUNBNUQsUUFBQUEsR0FBRyxDQUFDNkQsY0FBSixHQUFxQixNQUFyQjtBQUNEOztBQUVELFVBQUlDLElBQUksR0FBRztBQUFDOUQsUUFBQUE7QUFBRCxPQUFYOztBQUVBLFVBQUlWLEtBQUosRUFBVztBQUNUd0UsUUFBQUEsSUFBSSxDQUFDeEUsS0FBTCxHQUFhQSxLQUFiO0FBQ0F3RSxRQUFBQSxJQUFJLENBQUNDLGFBQUwsR0FBcUIsTUFBckI7QUFDRDtBQUVEOzs7QUFDQSxVQUFJaEUsT0FBTyxDQUFDQyxHQUFSLENBQVlnRSxlQUFoQixFQUFpQztBQUMvQkMsUUFBQUEsT0FBTyxDQUFDQyxJQUFSLENBQWMsT0FBTTdELGFBQWMsRUFBbEM7QUFDRDs7QUFFRCxhQUFPLElBQUl6QixPQUFKLENBQVksT0FBTytCLE9BQVAsRUFBZ0I5QixNQUFoQixLQUEyQjtBQUM1QyxZQUFJYixPQUFPLENBQUNtRyxTQUFaLEVBQXVCO0FBQ3JCLGdCQUFNQyxXQUFXLEdBQUcsTUFBTXBHLE9BQU8sQ0FBQ21HLFNBQVIsQ0FBa0I7QUFBQy9FLFlBQUFBLElBQUQ7QUFBTzBFLFlBQUFBO0FBQVAsV0FBbEIsQ0FBMUI7QUFDQTFFLFVBQUFBLElBQUksR0FBR2dGLFdBQVcsQ0FBQ2hGLElBQW5CO0FBQ0EwRSxVQUFBQSxJQUFJLEdBQUdNLFdBQVcsQ0FBQ04sSUFBbkI7QUFDRDs7QUFDRCxjQUFNO0FBQUNPLFVBQUFBLE9BQUQ7QUFBVUMsVUFBQUE7QUFBVixZQUFvQixLQUFLQyxpQkFBTCxDQUF1Qm5GLElBQXZCLEVBQTZCMEUsSUFBN0IsRUFBbUN2RCxZQUFuQyxDQUExQjtBQUNBLFlBQUlpRSxZQUFZLEdBQUcsS0FBbkI7O0FBQ0EsWUFBSXJELGVBQUosRUFBcUI7QUFDbkJ2QixVQUFBQSxhQUFhLENBQUM2RSxHQUFkLENBQWtCdEQsZUFBZSxDQUFDdUQsV0FBaEIsQ0FBNEIsT0FBTztBQUFDQyxZQUFBQTtBQUFELFdBQVAsS0FBd0I7QUFDcEVILFlBQUFBLFlBQVksR0FBRyxJQUFmO0FBQ0Esa0JBQU1GLE1BQU0sRUFBWixDQUZvRSxDQUlwRTtBQUNBO0FBQ0E7QUFDQTs7QUFDQSxrQkFBTSxJQUFJMUYsT0FBSixDQUFZLENBQUNnRyxXQUFELEVBQWNDLFVBQWQsS0FBNkI7QUFDN0NDLGNBQUFBLE9BQU8sQ0FBQyxXQUFELENBQVAsQ0FBcUJILFVBQXJCLEVBQWlDLFNBQWpDLEVBQTRDSSxHQUFHLElBQUk7QUFDakQ7QUFDQSxvQkFBSUEsR0FBSixFQUFTO0FBQUVGLGtCQUFBQSxVQUFVLENBQUNFLEdBQUQsQ0FBVjtBQUFrQixpQkFBN0IsTUFBbUM7QUFBRUgsa0JBQUFBLFdBQVc7QUFBSztBQUN0RCxlQUhEO0FBSUQsYUFMSyxDQUFOO0FBTUQsV0FkaUIsQ0FBbEI7QUFlRDs7QUFFRCxjQUFNO0FBQUM5RCxVQUFBQSxNQUFEO0FBQVNDLFVBQUFBLE1BQVQ7QUFBaUJpRSxVQUFBQSxRQUFqQjtBQUEyQkMsVUFBQUEsTUFBM0I7QUFBbUNDLFVBQUFBO0FBQW5DLFlBQTZDLE1BQU1iLE9BQU8sU0FBUCxDQUFjVSxHQUFHLElBQUk7QUFDNUUsY0FBSUEsR0FBRyxDQUFDRSxNQUFSLEVBQWdCO0FBQ2QsbUJBQU87QUFBQ0EsY0FBQUEsTUFBTSxFQUFFRixHQUFHLENBQUNFO0FBQWIsYUFBUDtBQUNEOztBQUNEcEcsVUFBQUEsTUFBTSxDQUFDa0csR0FBRCxDQUFOO0FBQ0EsaUJBQU8sRUFBUDtBQUNELFNBTndELENBQXpEOztBQVFBLFlBQUlHLE1BQUosRUFBWTtBQUNWLGdCQUFNO0FBQUNDLFlBQUFBLFFBQUQ7QUFBV0MsWUFBQUEsU0FBWDtBQUFzQkMsWUFBQUE7QUFBdEIsY0FBaUNILE1BQXZDO0FBQ0EsZ0JBQU1JLEdBQUcsR0FBR0MsV0FBVyxDQUFDRCxHQUFaLEVBQVo7QUFDQS9FLFVBQUFBLFlBQVksQ0FBQ0csSUFBYixDQUFrQixVQUFsQixFQUE4QjRFLEdBQUcsR0FBR0gsUUFBTixHQUFpQkMsU0FBakIsR0FBNkJDLE9BQTNEO0FBQ0E5RSxVQUFBQSxZQUFZLENBQUNHLElBQWIsQ0FBa0IsU0FBbEIsRUFBNkI0RSxHQUFHLEdBQUdILFFBQU4sR0FBaUJFLE9BQTlDO0FBQ0E5RSxVQUFBQSxZQUFZLENBQUNHLElBQWIsQ0FBa0IsS0FBbEIsRUFBeUI0RSxHQUFHLEdBQUdELE9BQS9CO0FBQ0Q7O0FBQ0Q5RSxRQUFBQSxZQUFZLENBQUNpRixRQUFiO0FBRUE7O0FBQ0EsWUFBSXpGLE9BQU8sQ0FBQ0MsR0FBUixDQUFZZ0UsZUFBaEIsRUFBaUM7QUFDL0JDLFVBQUFBLE9BQU8sQ0FBQ3dCLE9BQVIsQ0FBaUIsT0FBTXBGLGFBQWMsRUFBckM7QUFDRDs7QUFFRCxZQUFJYyxlQUFKLEVBQXFCO0FBQ25CQSxVQUFBQSxlQUFlLENBQUN1RSxTQUFoQjtBQUNEOztBQUNEOUYsUUFBQUEsYUFBYSxDQUFDK0YsT0FBZDtBQUVBOztBQUNBLFlBQUk3RixrQkFBSixFQUF3QjtBQUN0QixnQkFBTThGLHVCQUF1QixHQUFHQyxHQUFHLElBQUk7QUFDckMsZ0JBQUksQ0FBQ0EsR0FBTCxFQUFVO0FBQUUscUJBQU8sRUFBUDtBQUFZOztBQUV4QixtQkFBT0EsR0FBRyxDQUNQQyxPQURJLENBQ0ksVUFESixFQUNnQixTQURoQixFQUVKQSxPQUZJLENBRUksVUFGSixFQUVnQixPQUZoQixDQUFQO0FBR0QsV0FORDs7QUFRQSxjQUFJaEosUUFBSixFQUFjO0FBQ1osZ0JBQUlpSixPQUFPLEdBQUksT0FBTTFGLGFBQWMsSUFBbkM7O0FBQ0EsZ0JBQUkyRSxRQUFRLEtBQUtnQixTQUFqQixFQUE0QjtBQUMxQkQsY0FBQUEsT0FBTyxJQUFLLGdCQUFlZixRQUFTLElBQXBDO0FBQ0QsYUFGRCxNQUVPLElBQUlDLE1BQUosRUFBWTtBQUNqQmMsY0FBQUEsT0FBTyxJQUFLLGdCQUFlZCxNQUFPLElBQWxDO0FBQ0Q7O0FBQ0QsZ0JBQUkzRixLQUFLLElBQUlBLEtBQUssQ0FBQ2QsTUFBTixLQUFpQixDQUE5QixFQUFpQztBQUMvQnVILGNBQUFBLE9BQU8sSUFBSyxXQUFVSCx1QkFBdUIsQ0FBQ3RHLEtBQUQsQ0FBUSxJQUFyRDtBQUNEOztBQUNEeUcsWUFBQUEsT0FBTyxJQUFJLFNBQVg7O0FBQ0EsZ0JBQUlqRixNQUFNLENBQUN0QyxNQUFQLEtBQWtCLENBQXRCLEVBQXlCO0FBQ3ZCdUgsY0FBQUEsT0FBTyxJQUFJLFlBQVg7QUFDRCxhQUZELE1BRU87QUFDTEEsY0FBQUEsT0FBTyxJQUFLLEtBQUlILHVCQUF1QixDQUFDOUUsTUFBRCxDQUFTLElBQWhEO0FBQ0Q7O0FBQ0RpRixZQUFBQSxPQUFPLElBQUksU0FBWDs7QUFDQSxnQkFBSWhGLE1BQU0sQ0FBQ3ZDLE1BQVAsS0FBa0IsQ0FBdEIsRUFBeUI7QUFDdkJ1SCxjQUFBQSxPQUFPLElBQUksWUFBWDtBQUNELGFBRkQsTUFFTztBQUNMQSxjQUFBQSxPQUFPLElBQUssS0FBSUgsdUJBQXVCLENBQUM3RSxNQUFELENBQVMsSUFBaEQ7QUFDRDs7QUFFRGtELFlBQUFBLE9BQU8sQ0FBQ2dDLEdBQVIsQ0FBWUYsT0FBWjtBQUNELFdBeEJELE1Bd0JPO0FBQ0wsa0JBQU1HLFdBQVcsR0FBRyxpQ0FBcEI7QUFFQWpDLFlBQUFBLE9BQU8sQ0FBQ2tDLGNBQVIsQ0FBd0IsT0FBTTlGLGFBQWMsRUFBNUM7O0FBQ0EsZ0JBQUkyRSxRQUFRLEtBQUtnQixTQUFqQixFQUE0QjtBQUMxQi9CLGNBQUFBLE9BQU8sQ0FBQ2dDLEdBQVIsQ0FBWSxvQkFBWixFQUFrQ0MsV0FBbEMsRUFBK0Msb0NBQS9DLEVBQXFGbEIsUUFBckY7QUFDRCxhQUZELE1BRU8sSUFBSUMsTUFBSixFQUFZO0FBQ2pCaEIsY0FBQUEsT0FBTyxDQUFDZ0MsR0FBUixDQUFZLG9CQUFaLEVBQWtDQyxXQUFsQyxFQUErQyxvQ0FBL0MsRUFBcUZqQixNQUFyRjtBQUNEOztBQUNEaEIsWUFBQUEsT0FBTyxDQUFDZ0MsR0FBUixDQUNFLHVCQURGLEVBRUVDLFdBRkYsRUFFZSxvQ0FGZixFQUdFRSxpQkFBS0MsT0FBTCxDQUFhakgsSUFBYixFQUFtQjtBQUFDa0gsY0FBQUEsV0FBVyxFQUFFQztBQUFkLGFBQW5CLENBSEY7O0FBS0EsZ0JBQUlqSCxLQUFLLElBQUlBLEtBQUssQ0FBQ2QsTUFBTixLQUFpQixDQUE5QixFQUFpQztBQUMvQnlGLGNBQUFBLE9BQU8sQ0FBQ2dDLEdBQVIsQ0FBWSxTQUFaLEVBQXVCQyxXQUF2QjtBQUNBakMsY0FBQUEsT0FBTyxDQUFDZ0MsR0FBUixDQUFZTCx1QkFBdUIsQ0FBQ3RHLEtBQUQsQ0FBbkM7QUFDRDs7QUFDRDJFLFlBQUFBLE9BQU8sQ0FBQ2dDLEdBQVIsQ0FBWSxVQUFaLEVBQXdCQyxXQUF4QjtBQUNBakMsWUFBQUEsT0FBTyxDQUFDZ0MsR0FBUixDQUFZTCx1QkFBdUIsQ0FBQzlFLE1BQUQsQ0FBbkM7QUFDQW1ELFlBQUFBLE9BQU8sQ0FBQ2dDLEdBQVIsQ0FBWSxVQUFaLEVBQXdCQyxXQUF4QjtBQUNBakMsWUFBQUEsT0FBTyxDQUFDZ0MsR0FBUixDQUFZTCx1QkFBdUIsQ0FBQzdFLE1BQUQsQ0FBbkM7QUFDQWtELFlBQUFBLE9BQU8sQ0FBQ3VDLFFBQVI7QUFDRDtBQUNGOztBQUVELFlBQUl4QixRQUFRLEtBQUssQ0FBYixJQUFrQixDQUFDUixZQUF2QixFQUFxQztBQUNuQyxnQkFBTU8sR0FBRyxHQUFHLElBQUkvSCxRQUFKLENBQ1QsR0FBRXFELGFBQWMscUJBQW9CMkUsUUFBUyxhQUFZbEUsTUFBTyxhQUFZQyxNQUFPLEVBRDFFLENBQVo7QUFHQWdFLFVBQUFBLEdBQUcsQ0FBQzBCLElBQUosR0FBV3pCLFFBQVg7QUFDQUQsVUFBQUEsR0FBRyxDQUFDMkIsTUFBSixHQUFhM0YsTUFBYjtBQUNBZ0UsVUFBQUEsR0FBRyxDQUFDNEIsTUFBSixHQUFhN0YsTUFBYjtBQUNBaUUsVUFBQUEsR0FBRyxDQUFDNkIsT0FBSixHQUFjdkcsYUFBZDtBQUNBeEIsVUFBQUEsTUFBTSxDQUFDa0csR0FBRCxDQUFOO0FBQ0Q7O0FBRUQsWUFBSSxDQUFDekgsb0JBQW9CLENBQUN1SixRQUFyQixDQUE4QmxILFdBQTlCLENBQUwsRUFBaUQ7QUFDL0MsK0NBQWlCQSxXQUFqQjtBQUNEOztBQUNEZ0IsUUFBQUEsT0FBTyxDQUFDRyxNQUFELENBQVA7QUFDRCxPQWhJTSxDQUFQO0FBaUlELEtBMU5NLEVBME5KO0FBQUNnRyxNQUFBQSxRQUFRLEVBQUUsQ0FBQ3BIO0FBQVosS0ExTkksQ0FBUDtBQTJOQTtBQUNEOztBQUVELFFBQU1xSCxPQUFOLENBQWMzSCxJQUFkLEVBQW9CcEIsT0FBcEIsRUFBNkI7QUFDM0IsUUFBSTtBQUNGLGFBQU8sTUFBTSxLQUFLbUIsSUFBTCxDQUFVQyxJQUFJLENBQUM0SCxLQUFMLEVBQVY7QUFDWHhILFFBQUFBLGFBQWEsRUFBRSxJQURKO0FBRVhDLFFBQUFBLGdCQUFnQixFQUFFO0FBRlAsU0FHUnpCLE9BSFEsRUFBYjtBQUtELEtBTkQsQ0FNRSxPQUFPaUosQ0FBUCxFQUFVO0FBQ1YsVUFBSSxhQUFhQyxJQUFiLENBQWtCRCxDQUFDLENBQUNQLE1BQXBCLENBQUosRUFBaUM7QUFDL0IsZUFBTyxNQUFNLEtBQUt2SCxJQUFMLENBQVVDLElBQVY7QUFDWEcsVUFBQUEsa0JBQWtCLEVBQUUsSUFEVDtBQUVYQyxVQUFBQSxhQUFhLEVBQUUsSUFGSjtBQUdYQyxVQUFBQSxnQkFBZ0IsRUFBRTtBQUhQLFdBSVJ6QixPQUpRLEVBQWI7QUFNRCxPQVBELE1BT087QUFDTCxjQUFNaUosQ0FBTjtBQUNEO0FBQ0Y7QUFDRjs7QUFFRDFDLEVBQUFBLGlCQUFpQixDQUFDbkYsSUFBRCxFQUFPcEIsT0FBUCxFQUFnQm1KLE1BQU0sR0FBRyxJQUF6QixFQUErQjtBQUM5QyxRQUFJcEgsT0FBTyxDQUFDQyxHQUFSLENBQVlvSCwyQkFBWixJQUEyQyxDQUFDQywwQkFBY0MsV0FBZCxHQUE0QkMsT0FBNUIsRUFBaEQsRUFBdUY7QUFDckZKLE1BQUFBLE1BQU0sSUFBSUEsTUFBTSxDQUFDekcsSUFBUCxDQUFZLFVBQVosQ0FBVjtBQUVBLFVBQUk4RyxRQUFKOztBQUNBeEosTUFBQUEsT0FBTyxDQUFDeUosZUFBUixHQUEwQkMsS0FBSyxJQUFJO0FBQ2pDRixRQUFBQSxRQUFRLEdBQUdFLEtBQUssQ0FBQ0MsR0FBakI7QUFFQTs7QUFDQUQsUUFBQUEsS0FBSyxDQUFDcEksS0FBTixDQUFZc0ksRUFBWixDQUFlLE9BQWYsRUFBd0I3QyxHQUFHLElBQUk7QUFDN0IsZ0JBQU0sSUFBSTlILEtBQUosQ0FDSCwrQkFBOEJtQyxJQUFJLENBQUNrQixJQUFMLENBQVUsR0FBVixDQUFlLE9BQU0sS0FBS3ZDLFVBQVcsS0FBSUMsT0FBTyxDQUFDc0IsS0FBTSxLQUFJeUYsR0FBSSxFQUQxRixDQUFOO0FBRUQsU0FIRDtBQUlELE9BUkQ7O0FBVUEsWUFBTVYsT0FBTyxHQUFHd0QsbUJBQVcxSSxJQUFYLENBQWdCQyxJQUFoQixFQUFzQixLQUFLckIsVUFBM0IsRUFBdUNDLE9BQXZDLENBQWhCOztBQUNBbUosTUFBQUEsTUFBTSxJQUFJQSxNQUFNLENBQUN6RyxJQUFQLENBQVksU0FBWixDQUFWO0FBQ0EsYUFBTztBQUNMMkQsUUFBQUEsT0FESztBQUVMQyxRQUFBQSxNQUFNLEVBQUUsTUFBTTtBQUNaO0FBQ0EsY0FBSSxDQUFDa0QsUUFBTCxFQUFlO0FBQ2IsbUJBQU81SSxPQUFPLENBQUMrQixPQUFSLEVBQVA7QUFDRDs7QUFFRCxpQkFBTyxJQUFJL0IsT0FBSixDQUFZLENBQUMrQixPQUFELEVBQVU5QixNQUFWLEtBQXFCO0FBQ3RDaUcsWUFBQUEsT0FBTyxDQUFDLFdBQUQsQ0FBUCxDQUFxQjBDLFFBQXJCLEVBQStCLFNBQS9CLEVBQTBDekMsR0FBRyxJQUFJO0FBQy9DO0FBQ0Esa0JBQUlBLEdBQUosRUFBUztBQUFFbEcsZ0JBQUFBLE1BQU0sQ0FBQ2tHLEdBQUQsQ0FBTjtBQUFjLGVBQXpCLE1BQStCO0FBQUVwRSxnQkFBQUEsT0FBTztBQUFLO0FBQzlDLGFBSEQ7QUFJRCxXQUxNLENBQVA7QUFNRDtBQWRJLE9BQVA7QUFnQkQsS0FoQ0QsTUFnQ087QUFDTCxZQUFNN0IsYUFBYSxHQUFHLEtBQUtBLGFBQUwsSUFBc0J1SSwwQkFBY0MsV0FBZCxFQUE1Qzs7QUFDQSxhQUFPeEksYUFBYSxDQUFDZ0osT0FBZCxDQUFzQjtBQUMzQjFJLFFBQUFBLElBRDJCO0FBRTNCckIsUUFBQUEsVUFBVSxFQUFFLEtBQUtBLFVBRlU7QUFHM0JDLFFBQUFBO0FBSDJCLE9BQXRCLENBQVA7QUFLRDtBQUNGOztBQUVELFFBQU0rSixnQkFBTixHQUF5QjtBQUN2QixRQUFJO0FBQ0YsWUFBTUMsb0JBQUdDLElBQUgsQ0FBUSxLQUFLbEssVUFBYixDQUFOLENBREUsQ0FDOEI7O0FBQ2hDLFlBQU1tSyxNQUFNLEdBQUcsTUFBTSxLQUFLL0ksSUFBTCxDQUFVLENBQUMsV0FBRCxFQUFjLG1CQUFkLEVBQW1DcUMsaUJBQUtsQixJQUFMLENBQVUsS0FBS3ZDLFVBQWYsRUFBMkIsTUFBM0IsQ0FBbkMsQ0FBVixDQUFyQjtBQUNBLFlBQU1vSyxTQUFTLEdBQUdELE1BQU0sQ0FBQ2xILElBQVAsRUFBbEI7QUFDQSxhQUFPLDhCQUFnQm1ILFNBQWhCLENBQVA7QUFDRCxLQUxELENBS0UsT0FBT2xCLENBQVAsRUFBVTtBQUNWLGFBQU8sSUFBUDtBQUNEO0FBQ0Y7O0FBRURtQixFQUFBQSxJQUFJLEdBQUc7QUFDTCxXQUFPLEtBQUtqSixJQUFMLENBQVUsQ0FBQyxNQUFELEVBQVMsS0FBS3BCLFVBQWQsQ0FBVixDQUFQO0FBQ0Q7QUFFRDs7Ozs7QUFHQXNLLEVBQUFBLFVBQVUsQ0FBQ0MsS0FBRCxFQUFRO0FBQ2hCLFFBQUlBLEtBQUssQ0FBQzlKLE1BQU4sS0FBaUIsQ0FBckIsRUFBd0I7QUFBRSxhQUFPSSxPQUFPLENBQUMrQixPQUFSLENBQWdCLElBQWhCLENBQVA7QUFBK0I7O0FBQ3pELFVBQU12QixJQUFJLEdBQUcsQ0FBQyxLQUFELEVBQVFtSixNQUFSLENBQWVELEtBQUssQ0FBQ0UsR0FBTixDQUFVQyxxQkFBVixDQUFmLENBQWI7QUFDQSxXQUFPLEtBQUt0SixJQUFMLENBQVVDLElBQVYsRUFBZ0I7QUFBQ00sTUFBQUEsY0FBYyxFQUFFO0FBQWpCLEtBQWhCLENBQVA7QUFDRDs7QUFFRCxRQUFNZ0osMEJBQU4sR0FBbUM7QUFDakMsUUFBSUMsWUFBWSxHQUFHLE1BQU0sS0FBS0MsU0FBTCxDQUFlLGlCQUFmLENBQXpCOztBQUNBLFFBQUksQ0FBQ0QsWUFBTCxFQUFtQjtBQUNqQixhQUFPLElBQVA7QUFDRDs7QUFFRCxVQUFNRSxPQUFPLEdBQUd2SyxlQUFHd0ssT0FBSCxFQUFoQjs7QUFFQUgsSUFBQUEsWUFBWSxHQUFHQSxZQUFZLENBQUMzSCxJQUFiLEdBQW9COEUsT0FBcEIsQ0FBNEJsSSxrQkFBNUIsRUFBZ0QsQ0FBQ21MLENBQUQsRUFBSUMsSUFBSixLQUFhO0FBQzFFO0FBQ0EsYUFBUSxHQUFFQSxJQUFJLEdBQUd4SCxpQkFBS2xCLElBQUwsQ0FBVWtCLGlCQUFLeUgsT0FBTCxDQUFhSixPQUFiLENBQVYsRUFBaUNHLElBQWpDLENBQUgsR0FBNENILE9BQVEsR0FBbEU7QUFDRCxLQUhjLENBQWY7QUFJQUYsSUFBQUEsWUFBWSxHQUFHLDhCQUFnQkEsWUFBaEIsQ0FBZjs7QUFFQSxRQUFJLENBQUNuSCxpQkFBSzBILFVBQUwsQ0FBZ0JQLFlBQWhCLENBQUwsRUFBb0M7QUFDbENBLE1BQUFBLFlBQVksR0FBR25ILGlCQUFLbEIsSUFBTCxDQUFVLEtBQUt2QyxVQUFmLEVBQTJCNEssWUFBM0IsQ0FBZjtBQUNEOztBQUVELFFBQUksRUFBQyxNQUFNLHlCQUFXQSxZQUFYLENBQVAsQ0FBSixFQUFxQztBQUNuQyxZQUFNLElBQUkxTCxLQUFKLENBQVcsbURBQWtEMEwsWUFBYSxFQUExRSxDQUFOO0FBQ0Q7O0FBQ0QsV0FBTyxNQUFNWCxvQkFBR21CLFFBQUgsQ0FBWVIsWUFBWixFQUEwQjtBQUFDUyxNQUFBQSxRQUFRLEVBQUU7QUFBWCxLQUExQixDQUFiO0FBQ0Q7O0FBRURDLEVBQUFBLFlBQVksQ0FBQ2YsS0FBRCxFQUFRZ0IsTUFBTSxHQUFHLE1BQWpCLEVBQXlCO0FBQ25DLFFBQUloQixLQUFLLENBQUM5SixNQUFOLEtBQWlCLENBQXJCLEVBQXdCO0FBQUUsYUFBT0ksT0FBTyxDQUFDK0IsT0FBUixDQUFnQixJQUFoQixDQUFQO0FBQStCOztBQUN6RCxVQUFNdkIsSUFBSSxHQUFHLENBQUMsT0FBRCxFQUFVa0ssTUFBVixFQUFrQixJQUFsQixFQUF3QmYsTUFBeEIsQ0FBK0JELEtBQUssQ0FBQ0UsR0FBTixDQUFVQyxxQkFBVixDQUEvQixDQUFiO0FBQ0EsV0FBTyxLQUFLdEosSUFBTCxDQUFVQyxJQUFWLEVBQWdCO0FBQUNNLE1BQUFBLGNBQWMsRUFBRTtBQUFqQixLQUFoQixDQUFQO0FBQ0Q7O0FBRUQ2SixFQUFBQSxtQkFBbUIsQ0FBQ0MsUUFBRCxFQUFXQyxPQUFYLEVBQW9CO0FBQ3JDLFVBQU1DLGdCQUFnQixHQUFHLEtBQUt2SyxJQUFMLENBQVUsQ0FBQyxVQUFELEVBQWEsSUFBYixFQUFtQixJQUFuQixFQUF5QnFLLFFBQXpCLENBQVYsQ0FBekI7QUFDQSxXQUFPLEtBQUtySyxJQUFMLENBQVUsQ0FBQyxjQUFELEVBQWlCLGFBQWpCLEVBQWlDLEdBQUVzSyxPQUFRLGNBQWFELFFBQVMsRUFBakUsQ0FBVixFQUErRTtBQUNwRjlKLE1BQUFBLGNBQWMsRUFBRSxJQURvRTtBQUVwRnlFLE1BQUFBLFNBQVMsRUFBRSxlQUFld0YsYUFBZixDQUE2QjtBQUFDdkssUUFBQUEsSUFBRDtBQUFPMEUsUUFBQUE7QUFBUCxPQUE3QixFQUEyQztBQUNwRCxjQUFNOEYsS0FBSyxHQUFHLE1BQU1GLGdCQUFwQjtBQUNBLGNBQU1HLEdBQUcsR0FBR0QsS0FBSyxDQUFDRSxNQUFOLENBQWEsQ0FBYixFQUFnQixFQUFoQixDQUFaO0FBQ0EsZUFBTztBQUNMaEcsVUFBQUEsSUFESztBQUVMMUUsVUFBQUEsSUFBSSxFQUFFLENBQUMsY0FBRCxFQUFpQixhQUFqQixFQUFpQyxHQUFFcUssT0FBUSxJQUFHSSxHQUFJLElBQUdMLFFBQVMsRUFBOUQ7QUFGRCxTQUFQO0FBSUQ7QUFUbUYsS0FBL0UsQ0FBUDtBQVdEOztBQUVETyxFQUFBQSxzQkFBc0IsQ0FBQ1AsUUFBRCxFQUFXO0FBQy9CLFdBQU8sS0FBS3JLLElBQUwsQ0FBVSxDQUFDLElBQUQsRUFBTyxVQUFQLEVBQW1CcUssUUFBbkIsQ0FBVixFQUF3QztBQUFDOUosTUFBQUEsY0FBYyxFQUFFO0FBQWpCLEtBQXhDLENBQVA7QUFDRDs7QUFFRHNLLEVBQUFBLFVBQVUsQ0FBQ0MsS0FBRCxFQUFRO0FBQUNMLElBQUFBO0FBQUQsTUFBVSxFQUFsQixFQUFzQjtBQUM5QixVQUFNeEssSUFBSSxHQUFHLENBQUMsT0FBRCxFQUFVLEdBQVYsQ0FBYjs7QUFDQSxRQUFJd0ssS0FBSixFQUFXO0FBQUV4SyxNQUFBQSxJQUFJLENBQUM4SyxNQUFMLENBQVksQ0FBWixFQUFlLENBQWYsRUFBa0IsVUFBbEI7QUFBZ0M7O0FBQzdDLFdBQU8sS0FBSy9LLElBQUwsQ0FBVUMsSUFBVixFQUFnQjtBQUFDRSxNQUFBQSxLQUFLLEVBQUUySyxLQUFSO0FBQWV2SyxNQUFBQSxjQUFjLEVBQUU7QUFBL0IsS0FBaEIsQ0FBUDtBQUNEOztBQUVELFFBQU00SixNQUFOLENBQWFhLFVBQWIsRUFBeUI7QUFBQ0MsSUFBQUEsVUFBRDtBQUFhQyxJQUFBQSxLQUFiO0FBQW9CQyxJQUFBQSxTQUFwQjtBQUErQkMsSUFBQUE7QUFBL0IsTUFBMkMsRUFBcEUsRUFBd0U7QUFDdEUsVUFBTW5MLElBQUksR0FBRyxDQUFDLFFBQUQsQ0FBYjtBQUNBLFFBQUlvTCxHQUFKLENBRnNFLENBSXRFO0FBQ0E7O0FBQ0EsUUFBSUgsS0FBSyxJQUFJRixVQUFVLENBQUMzTCxNQUFYLEtBQXNCLENBQW5DLEVBQXNDO0FBQ3BDLFlBQU07QUFBQ2lNLFFBQUFBLFNBQUQ7QUFBWUMsUUFBQUEsV0FBWjtBQUF5QkMsUUFBQUE7QUFBekIsVUFBMkMsTUFBTSxLQUFLQyxhQUFMLEVBQXZEOztBQUNBLFVBQUlILFNBQUosRUFBZTtBQUNiRCxRQUFBQSxHQUFHLEdBQUdMLFVBQU47QUFDRCxPQUZELE1BRU87QUFDTEssUUFBQUEsR0FBRyxHQUFJLEdBQUVHLGNBQWUsT0FBTUQsV0FBWSxFQUFwQyxDQUFzQzFKLElBQXRDLEVBQU47QUFDQXVKLFFBQUFBLFFBQVEsR0FBRyxJQUFYO0FBQ0Q7QUFDRixLQVJELE1BUU87QUFDTEMsTUFBQUEsR0FBRyxHQUFHTCxVQUFOO0FBQ0QsS0FoQnFFLENBa0J0RTtBQUNBOzs7QUFDQSxVQUFNVSxRQUFRLEdBQUcsTUFBTSxLQUFLbkMsMEJBQUwsRUFBdkI7O0FBQ0EsUUFBSW1DLFFBQUosRUFBYztBQUVaO0FBQ0E7QUFDQSxVQUFJQyxXQUFXLEdBQUcsTUFBTSxLQUFLbEMsU0FBTCxDQUFlLGtCQUFmLENBQXhCOztBQUNBLFVBQUksQ0FBQ2tDLFdBQUwsRUFBa0I7QUFDaEJBLFFBQUFBLFdBQVcsR0FBRyxHQUFkO0FBQ0Q7O0FBQ0ROLE1BQUFBLEdBQUcsR0FBR0EsR0FBRyxDQUFDTyxLQUFKLENBQVUsSUFBVixFQUFnQkMsTUFBaEIsQ0FBdUJDLElBQUksSUFBSSxDQUFDQSxJQUFJLENBQUNDLFVBQUwsQ0FBZ0JKLFdBQWhCLENBQWhDLEVBQThEeEssSUFBOUQsQ0FBbUUsSUFBbkUsQ0FBTjtBQUNELEtBOUJxRSxDQWdDdEU7OztBQUNBLFFBQUlpSyxRQUFKLEVBQWM7QUFDWm5MLE1BQUFBLElBQUksQ0FBQzhCLElBQUwsQ0FBVSxvQkFBVjtBQUNELEtBRkQsTUFFTztBQUNMLFlBQU1pSyxVQUFVLEdBQUcsTUFBTSxLQUFLdkMsU0FBTCxDQUFlLGdCQUFmLENBQXpCO0FBQ0EsWUFBTXdDLElBQUksR0FBSUQsVUFBVSxJQUFJQSxVQUFVLEtBQUssU0FBOUIsR0FBMkNBLFVBQTNDLEdBQXdELE9BQXJFO0FBQ0EvTCxNQUFBQSxJQUFJLENBQUM4QixJQUFMLENBQVcsYUFBWWtLLElBQUssRUFBNUI7QUFDRCxLQXZDcUUsQ0F5Q3RFOzs7QUFDQSxRQUFJZCxTQUFTLElBQUlBLFNBQVMsQ0FBQzlMLE1BQVYsR0FBbUIsQ0FBcEMsRUFBdUM7QUFDckNnTSxNQUFBQSxHQUFHLEdBQUcsTUFBTSxLQUFLYSxxQkFBTCxDQUEyQmIsR0FBM0IsRUFBZ0NGLFNBQWhDLENBQVo7QUFDRDs7QUFFRGxMLElBQUFBLElBQUksQ0FBQzhCLElBQUwsQ0FBVSxJQUFWLEVBQWdCc0osR0FBRyxDQUFDeEosSUFBSixFQUFoQjs7QUFFQSxRQUFJcUosS0FBSixFQUFXO0FBQUVqTCxNQUFBQSxJQUFJLENBQUM4QixJQUFMLENBQVUsU0FBVjtBQUF1Qjs7QUFDcEMsUUFBSWtKLFVBQUosRUFBZ0I7QUFBRWhMLE1BQUFBLElBQUksQ0FBQzhCLElBQUwsQ0FBVSxlQUFWO0FBQTZCOztBQUMvQyxXQUFPLEtBQUs2RixPQUFMLENBQWEzSCxJQUFiLEVBQW1CO0FBQUNNLE1BQUFBLGNBQWMsRUFBRTtBQUFqQixLQUFuQixDQUFQO0FBQ0Q7O0FBRUQyTCxFQUFBQSxxQkFBcUIsQ0FBQ2xPLE9BQUQsRUFBVW1OLFNBQVMsR0FBRyxFQUF0QixFQUEwQjtBQUM3QyxVQUFNZ0IsUUFBUSxHQUFHaEIsU0FBUyxDQUFDOUIsR0FBVixDQUFjK0MsTUFBTSxJQUFJO0FBQ3ZDLGFBQU87QUFDTEMsUUFBQUEsS0FBSyxFQUFFLGdCQURGO0FBRUxDLFFBQUFBLEtBQUssRUFBRyxHQUFFRixNQUFNLENBQUNHLElBQUssS0FBSUgsTUFBTSxDQUFDSSxLQUFNO0FBRmxDLE9BQVA7QUFJRCxLQUxnQixDQUFqQixDQUQ2QyxDQVE3Qzs7QUFDQSxVQUFNbkIsR0FBRyxHQUFJLEdBQUVyTixPQUFPLENBQUM2RCxJQUFSLEVBQWUsSUFBOUI7QUFFQSxXQUFPc0ssUUFBUSxDQUFDOU0sTUFBVCxHQUFrQixLQUFLb04sYUFBTCxDQUFtQnBCLEdBQW5CLEVBQXdCYyxRQUF4QixDQUFsQixHQUFzRGQsR0FBN0Q7QUFDRDtBQUVEOzs7OztBQUdBLFFBQU1xQixlQUFOLEdBQXdCO0FBQ3RCLFVBQU16TSxJQUFJLEdBQUcsQ0FBQyxRQUFELEVBQVcsZ0JBQVgsRUFBNkIsVUFBN0IsRUFBeUMsdUJBQXpDLEVBQWtFLDJCQUFsRSxFQUErRixJQUEvRixDQUFiO0FBQ0EsVUFBTThJLE1BQU0sR0FBRyxNQUFNLEtBQUsvSSxJQUFMLENBQVVDLElBQVYsQ0FBckI7O0FBQ0EsUUFBSThJLE1BQU0sQ0FBQzFKLE1BQVAsR0FBZ0IzQix3QkFBcEIsRUFBOEM7QUFDNUMsWUFBTSxJQUFJUSxjQUFKLEVBQU47QUFDRDs7QUFFRCxVQUFNeU8sT0FBTyxHQUFHLE1BQU0sMEJBQVk1RCxNQUFaLENBQXRCOztBQUVBLFNBQUssTUFBTTZELFNBQVgsSUFBd0JELE9BQXhCLEVBQWlDO0FBQy9CLFVBQUlFLEtBQUssQ0FBQ0MsT0FBTixDQUFjSCxPQUFPLENBQUNDLFNBQUQsQ0FBckIsQ0FBSixFQUF1QztBQUNyQyxhQUFLRyw2QkFBTCxDQUFtQ0osT0FBTyxDQUFDQyxTQUFELENBQTFDO0FBQ0Q7QUFDRjs7QUFFRCxXQUFPRCxPQUFQO0FBQ0Q7O0FBRURJLEVBQUFBLDZCQUE2QixDQUFDQyxPQUFELEVBQVU7QUFDckNBLElBQUFBLE9BQU8sQ0FBQ0MsT0FBUixDQUFnQkMsS0FBSyxJQUFJO0FBQ3ZCO0FBQ0E7QUFDQTtBQUNBLFVBQUlBLEtBQUssQ0FBQ0MsUUFBVixFQUFvQjtBQUNsQkQsUUFBQUEsS0FBSyxDQUFDQyxRQUFOLEdBQWlCLDhCQUFnQkQsS0FBSyxDQUFDQyxRQUF0QixDQUFqQjtBQUNEOztBQUNELFVBQUlELEtBQUssQ0FBQ0UsWUFBVixFQUF3QjtBQUN0QkYsUUFBQUEsS0FBSyxDQUFDRSxZQUFOLEdBQXFCLDhCQUFnQkYsS0FBSyxDQUFDRSxZQUF0QixDQUFyQjtBQUNEO0FBQ0YsS0FWRDtBQVdEOztBQUVELFFBQU1DLGNBQU4sQ0FBcUJ4TyxPQUFPLEdBQUcsRUFBL0IsRUFBbUM7QUFDakMsVUFBTW9CLElBQUksR0FBRyxDQUFDLE1BQUQsRUFBUyxlQUFULEVBQTBCLGNBQTFCLENBQWI7O0FBQ0EsUUFBSXBCLE9BQU8sQ0FBQ3lPLE1BQVosRUFBb0I7QUFBRXJOLE1BQUFBLElBQUksQ0FBQzhCLElBQUwsQ0FBVSxVQUFWO0FBQXdCOztBQUM5QyxRQUFJbEQsT0FBTyxDQUFDME8sTUFBWixFQUFvQjtBQUFFdE4sTUFBQUEsSUFBSSxDQUFDOEIsSUFBTCxDQUFVbEQsT0FBTyxDQUFDME8sTUFBbEI7QUFBNEI7O0FBQ2xELFVBQU14RSxNQUFNLEdBQUcsTUFBTSxLQUFLL0ksSUFBTCxDQUFVQyxJQUFWLENBQXJCO0FBRUEsVUFBTXVOLFNBQVMsR0FBRztBQUNoQkMsTUFBQUEsQ0FBQyxFQUFFLE9BRGE7QUFFaEJDLE1BQUFBLENBQUMsRUFBRSxVQUZhO0FBR2hCQyxNQUFBQSxDQUFDLEVBQUUsU0FIYTtBQUloQkMsTUFBQUEsQ0FBQyxFQUFFO0FBSmEsS0FBbEI7QUFPQSxVQUFNQyxZQUFZLEdBQUcsRUFBckI7QUFDQTlFLElBQUFBLE1BQU0sSUFBSUEsTUFBTSxDQUFDbEgsSUFBUCxHQUFjK0osS0FBZCxDQUFvQmtDLDBCQUFwQixFQUF1Q2IsT0FBdkMsQ0FBK0NuQixJQUFJLElBQUk7QUFDL0QsWUFBTSxDQUFDaUMsTUFBRCxFQUFTQyxXQUFULElBQXdCbEMsSUFBSSxDQUFDRixLQUFMLENBQVcsSUFBWCxDQUE5QjtBQUNBLFlBQU11QixRQUFRLEdBQUcsOEJBQWdCYSxXQUFoQixDQUFqQjtBQUNBSCxNQUFBQSxZQUFZLENBQUNWLFFBQUQsQ0FBWixHQUF5QkssU0FBUyxDQUFDTyxNQUFELENBQWxDO0FBQ0QsS0FKUyxDQUFWOztBQUtBLFFBQUksQ0FBQ2xQLE9BQU8sQ0FBQ3lPLE1BQWIsRUFBcUI7QUFDbkIsWUFBTVcsU0FBUyxHQUFHLE1BQU0sS0FBS0MsaUJBQUwsRUFBeEI7QUFDQUQsTUFBQUEsU0FBUyxDQUFDaEIsT0FBVixDQUFrQkUsUUFBUSxJQUFJO0FBQUVVLFFBQUFBLFlBQVksQ0FBQ1YsUUFBRCxDQUFaLEdBQXlCLE9BQXpCO0FBQW1DLE9BQW5FO0FBQ0Q7O0FBQ0QsV0FBT1UsWUFBUDtBQUNEOztBQUVELFFBQU1LLGlCQUFOLEdBQTBCO0FBQ3hCLFVBQU1uRixNQUFNLEdBQUcsTUFBTSxLQUFLL0ksSUFBTCxDQUFVLENBQUMsVUFBRCxFQUFhLFVBQWIsRUFBeUIsb0JBQXpCLENBQVYsQ0FBckI7O0FBQ0EsUUFBSStJLE1BQU0sQ0FBQ2xILElBQVAsT0FBa0IsRUFBdEIsRUFBMEI7QUFBRSxhQUFPLEVBQVA7QUFBWTs7QUFDeEMsV0FBT2tILE1BQU0sQ0FBQ2xILElBQVAsR0FBYytKLEtBQWQsQ0FBb0JrQywwQkFBcEIsRUFBdUN6RSxHQUF2QyxDQUEyQzhFLHdCQUEzQyxDQUFQO0FBQ0Q7O0FBRUQsUUFBTUMsbUJBQU4sQ0FBMEJqQixRQUExQixFQUFvQztBQUFDRyxJQUFBQSxNQUFEO0FBQVNlLElBQUFBO0FBQVQsTUFBdUIsRUFBM0QsRUFBK0Q7QUFDN0QsUUFBSXBPLElBQUksR0FBRyxDQUFDLE1BQUQsRUFBUyxhQUFULEVBQXdCLGVBQXhCLEVBQXlDLGNBQXpDLEVBQXlELGlCQUF6RCxDQUFYOztBQUNBLFFBQUlxTixNQUFKLEVBQVk7QUFBRXJOLE1BQUFBLElBQUksQ0FBQzhCLElBQUwsQ0FBVSxVQUFWO0FBQXdCOztBQUN0QyxRQUFJc00sVUFBSixFQUFnQjtBQUFFcE8sTUFBQUEsSUFBSSxDQUFDOEIsSUFBTCxDQUFVc00sVUFBVjtBQUF3Qjs7QUFDMUNwTyxJQUFBQSxJQUFJLEdBQUdBLElBQUksQ0FBQ21KLE1BQUwsQ0FBWSxDQUFDLElBQUQsRUFBTywyQkFBYStELFFBQWIsQ0FBUCxDQUFaLENBQVA7QUFDQSxVQUFNcEUsTUFBTSxHQUFHLE1BQU0sS0FBSy9JLElBQUwsQ0FBVUMsSUFBVixDQUFyQjtBQUVBLFFBQUlxTyxRQUFRLEdBQUcsRUFBZjs7QUFDQSxRQUFJdkYsTUFBSixFQUFZO0FBQ1Z1RixNQUFBQSxRQUFRLEdBQUcsd0JBQVV2RixNQUFWLEVBQ1I4QyxNQURRLENBQ0QwQyxPQUFPLElBQUlBLE9BQU8sQ0FBQ1IsTUFBUixLQUFtQixVQUQ3QixDQUFYOztBQUdBLFdBQUssSUFBSVMsQ0FBQyxHQUFHLENBQWIsRUFBZ0JBLENBQUMsR0FBR0YsUUFBUSxDQUFDalAsTUFBN0IsRUFBcUNtUCxDQUFDLEVBQXRDLEVBQTBDO0FBQ3hDLGNBQU1ELE9BQU8sR0FBR0QsUUFBUSxDQUFDRSxDQUFELENBQXhCOztBQUNBLFlBQUlELE9BQU8sQ0FBQ0UsT0FBWixFQUFxQjtBQUNuQkYsVUFBQUEsT0FBTyxDQUFDRSxPQUFSLEdBQWtCLDhCQUFnQkYsT0FBTyxDQUFDRSxPQUF4QixDQUFsQjtBQUNEOztBQUNELFlBQUlGLE9BQU8sQ0FBQ0csT0FBWixFQUFxQjtBQUNuQkgsVUFBQUEsT0FBTyxDQUFDRyxPQUFSLEdBQWtCLDhCQUFnQkgsT0FBTyxDQUFDRyxPQUF4QixDQUFsQjtBQUNEO0FBQ0Y7QUFDRjs7QUFFRCxRQUFJLENBQUNwQixNQUFELElBQVcsQ0FBQyxNQUFNLEtBQUtZLGlCQUFMLEVBQVAsRUFBaUN4RyxRQUFqQyxDQUEwQ3lGLFFBQTFDLENBQWYsRUFBb0U7QUFDbEU7QUFDQSxZQUFNd0IsT0FBTyxHQUFHdE0saUJBQUtsQixJQUFMLENBQVUsS0FBS3ZDLFVBQWYsRUFBMkJ1TyxRQUEzQixDQUFoQjs7QUFDQSxZQUFNeUIsVUFBVSxHQUFHLE1BQU0sK0JBQWlCRCxPQUFqQixDQUF6QjtBQUNBLFlBQU1FLE9BQU8sR0FBRyxNQUFNLDRCQUFjRixPQUFkLENBQXRCO0FBQ0EsWUFBTUcsUUFBUSxHQUFHLE1BQU1qRyxvQkFBR21CLFFBQUgsQ0FBWTJFLE9BQVosRUFBcUI7QUFBQzFFLFFBQUFBLFFBQVEsRUFBRTtBQUFYLE9BQXJCLENBQXZCO0FBQ0EsWUFBTThFLE1BQU0sR0FBRyx1QkFBU0QsUUFBVCxDQUFmO0FBQ0EsVUFBSTdDLElBQUo7QUFDQSxVQUFJK0MsUUFBSjs7QUFDQSxVQUFJSixVQUFKLEVBQWdCO0FBQ2QzQyxRQUFBQSxJQUFJLEdBQUdnRCxpQkFBS0MsS0FBTCxDQUFXQyxVQUFsQjtBQUNELE9BRkQsTUFFTyxJQUFJTixPQUFKLEVBQWE7QUFDbEI1QyxRQUFBQSxJQUFJLEdBQUdnRCxpQkFBS0MsS0FBTCxDQUFXRSxPQUFsQjtBQUNBSixRQUFBQSxRQUFRLEdBQUcsTUFBTW5HLG9CQUFHbUcsUUFBSCxDQUFZTCxPQUFaLENBQWpCO0FBQ0QsT0FITSxNQUdBO0FBQ0wxQyxRQUFBQSxJQUFJLEdBQUdnRCxpQkFBS0MsS0FBTCxDQUFXRyxNQUFsQjtBQUNEOztBQUVEZixNQUFBQSxRQUFRLENBQUN2TSxJQUFULENBQWN1TixtQkFBbUIsQ0FBQ25DLFFBQUQsRUFBVzRCLE1BQU0sR0FBRyxJQUFILEdBQVVELFFBQTNCLEVBQXFDN0MsSUFBckMsRUFBMkMrQyxRQUEzQyxDQUFqQztBQUNEOztBQUNELFFBQUlWLFFBQVEsQ0FBQ2pQLE1BQVQsR0FBa0IsQ0FBdEIsRUFBeUI7QUFDdkIsWUFBTSxJQUFJdkIsS0FBSixDQUFXLHNDQUFxQ3FQLFFBQVMsWUFBV21CLFFBQVEsQ0FBQ2pQLE1BQU8sRUFBcEYsQ0FBTjtBQUNEOztBQUNELFdBQU9pUCxRQUFQO0FBQ0Q7O0FBRUQsUUFBTWlCLHFCQUFOLEdBQThCO0FBQzVCLFVBQU14RyxNQUFNLEdBQUcsTUFBTSxLQUFLL0ksSUFBTCxDQUFVLENBQzdCLE1BRDZCLEVBQ3JCLFVBRHFCLEVBQ1QsYUFEUyxFQUNNLGVBRE4sRUFDdUIsY0FEdkIsRUFDdUMsaUJBRHZDLENBQVYsQ0FBckI7O0FBSUEsUUFBSSxDQUFDK0ksTUFBTCxFQUFhO0FBQ1gsYUFBTyxFQUFQO0FBQ0Q7O0FBRUQsVUFBTXlHLEtBQUssR0FBRyx3QkFBVXpHLE1BQVYsQ0FBZDs7QUFDQSxTQUFLLE1BQU0wRyxJQUFYLElBQW1CRCxLQUFuQixFQUEwQjtBQUN4QixVQUFJQyxJQUFJLENBQUNoQixPQUFULEVBQWtCO0FBQUVnQixRQUFBQSxJQUFJLENBQUNoQixPQUFMLEdBQWUsOEJBQWdCZ0IsSUFBSSxDQUFDaEIsT0FBckIsQ0FBZjtBQUErQzs7QUFDbkUsVUFBSWdCLElBQUksQ0FBQ2YsT0FBVCxFQUFrQjtBQUFFZSxRQUFBQSxJQUFJLENBQUNmLE9BQUwsR0FBZSw4QkFBZ0JlLElBQUksQ0FBQ2YsT0FBckIsQ0FBZjtBQUErQztBQUNwRTs7QUFDRCxXQUFPYyxLQUFQO0FBQ0Q7QUFFRDs7Ozs7QUFHQSxRQUFNRSxTQUFOLENBQWdCQyxHQUFoQixFQUFxQjtBQUNuQixVQUFNLENBQUN4RixNQUFELElBQVcsTUFBTSxLQUFLeUYsVUFBTCxDQUFnQjtBQUFDMVEsTUFBQUEsR0FBRyxFQUFFLENBQU47QUFBU3lRLE1BQUFBLEdBQVQ7QUFBY0UsTUFBQUEsYUFBYSxFQUFFO0FBQTdCLEtBQWhCLENBQXZCO0FBQ0EsV0FBTzFGLE1BQVA7QUFDRDs7QUFFRCxRQUFNc0IsYUFBTixHQUFzQjtBQUNwQixVQUFNLENBQUNxRSxVQUFELElBQWUsTUFBTSxLQUFLRixVQUFMLENBQWdCO0FBQUMxUSxNQUFBQSxHQUFHLEVBQUUsQ0FBTjtBQUFTeVEsTUFBQUEsR0FBRyxFQUFFLE1BQWQ7QUFBc0JFLE1BQUFBLGFBQWEsRUFBRTtBQUFyQyxLQUFoQixDQUEzQjtBQUNBLFdBQU9DLFVBQVA7QUFDRDs7QUFFRCxRQUFNRixVQUFOLENBQWlCL1EsT0FBTyxHQUFHLEVBQTNCLEVBQStCO0FBQzdCLFVBQU07QUFBQ0ssTUFBQUEsR0FBRDtBQUFNeVEsTUFBQUEsR0FBTjtBQUFXRSxNQUFBQSxhQUFYO0FBQTBCRSxNQUFBQTtBQUExQjtBQUNKN1EsTUFBQUEsR0FBRyxFQUFFLENBREQ7QUFFSnlRLE1BQUFBLEdBQUcsRUFBRSxNQUZEO0FBR0pFLE1BQUFBLGFBQWEsRUFBRSxLQUhYO0FBSUpFLE1BQUFBLFlBQVksRUFBRTtBQUpWLE9BS0RsUixPQUxDLENBQU4sQ0FENkIsQ0FTN0I7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7O0FBQ0EsVUFBTW9CLElBQUksR0FBRyxDQUNYLEtBRFcsRUFFWCx5REFGVyxFQUdYLG9CQUhXLEVBSVgsYUFKVyxFQUtYLGVBTFcsRUFNWCxjQU5XLEVBT1gsSUFQVyxFQVFYLElBUlcsRUFTWGYsR0FUVyxFQVVYeVEsR0FWVyxDQUFiOztBQWFBLFFBQUlJLFlBQUosRUFBa0I7QUFDaEI5UCxNQUFBQSxJQUFJLENBQUM4QixJQUFMLENBQVUsU0FBVixFQUFxQixJQUFyQixFQUEyQixnQkFBM0I7QUFDRDs7QUFFRCxVQUFNZ0gsTUFBTSxHQUFHLE1BQU0sS0FBSy9JLElBQUwsQ0FBVUMsSUFBSSxDQUFDbUosTUFBTCxDQUFZLElBQVosQ0FBVixXQUFtQ3hELEdBQUcsSUFBSTtBQUM3RCxVQUFJLG1CQUFtQm1DLElBQW5CLENBQXdCbkMsR0FBRyxDQUFDMkIsTUFBNUIsS0FBdUMsc0JBQXNCUSxJQUF0QixDQUEyQm5DLEdBQUcsQ0FBQzJCLE1BQS9CLENBQTNDLEVBQW1GO0FBQ2pGLGVBQU8sRUFBUDtBQUNELE9BRkQsTUFFTztBQUNMLGNBQU0zQixHQUFOO0FBQ0Q7QUFDRixLQU5vQixDQUFyQjs7QUFRQSxRQUFJbUQsTUFBTSxLQUFLLEVBQWYsRUFBbUI7QUFDakIsYUFBTzhHLGFBQWEsR0FBRyxDQUFDO0FBQUNHLFFBQUFBLEdBQUcsRUFBRSxFQUFOO0FBQVVoUyxRQUFBQSxPQUFPLEVBQUUsRUFBbkI7QUFBdUJzTixRQUFBQSxTQUFTLEVBQUU7QUFBbEMsT0FBRCxDQUFILEdBQStDLEVBQW5FO0FBQ0Q7O0FBRUQsVUFBTTJFLE1BQU0sR0FBR2xILE1BQU0sQ0FBQ2xILElBQVAsR0FBYytKLEtBQWQsQ0FBb0IsSUFBcEIsQ0FBZjtBQUVBLFVBQU1zRSxPQUFPLEdBQUcsRUFBaEI7O0FBQ0EsU0FBSyxJQUFJMUIsQ0FBQyxHQUFHLENBQWIsRUFBZ0JBLENBQUMsR0FBR3lCLE1BQU0sQ0FBQzVRLE1BQTNCLEVBQW1DbVAsQ0FBQyxJQUFJLENBQXhDLEVBQTJDO0FBQ3pDLFlBQU0yQixJQUFJLEdBQUdGLE1BQU0sQ0FBQ3pCLENBQUMsR0FBRyxDQUFMLENBQU4sQ0FBYzNNLElBQWQsRUFBYjtBQUNBLFVBQUlpSixLQUFLLEdBQUcsRUFBWjs7QUFDQSxVQUFJaUYsWUFBSixFQUFrQjtBQUNoQixjQUFNUCxLQUFLLEdBQUdTLE1BQU0sQ0FBQ3pCLENBQUMsR0FBRyxDQUFMLENBQXBCO0FBQ0ExRCxRQUFBQSxLQUFLLEdBQUcsd0JBQVUwRSxLQUFLLENBQUMzTixJQUFOLEVBQVYsQ0FBUjtBQUNEOztBQUVELFlBQU07QUFBQzdELFFBQUFBLE9BQU8sRUFBRXVOLFdBQVY7QUFBdUJKLFFBQUFBO0FBQXZCLFVBQW9DLGtEQUFvQ2dGLElBQXBDLENBQTFDO0FBRUFELE1BQUFBLE9BQU8sQ0FBQ25PLElBQVIsQ0FBYTtBQUNYaU8sUUFBQUEsR0FBRyxFQUFFQyxNQUFNLENBQUN6QixDQUFELENBQU4sSUFBYXlCLE1BQU0sQ0FBQ3pCLENBQUQsQ0FBTixDQUFVM00sSUFBVixFQURQO0FBRVh1TyxRQUFBQSxXQUFXLEVBQUVILE1BQU0sQ0FBQ3pCLENBQUMsR0FBRyxDQUFMLENBQU4sSUFBaUJ5QixNQUFNLENBQUN6QixDQUFDLEdBQUcsQ0FBTCxDQUFOLENBQWMzTSxJQUFkLEVBRm5CO0FBR1h3TyxRQUFBQSxVQUFVLEVBQUVKLE1BQU0sQ0FBQ3pCLENBQUMsR0FBRyxDQUFMLENBQU4sSUFBaUJ5QixNQUFNLENBQUN6QixDQUFDLEdBQUcsQ0FBTCxDQUFOLENBQWMzTSxJQUFkLEVBSGxCO0FBSVh5TyxRQUFBQSxVQUFVLEVBQUVDLFFBQVEsQ0FBQ04sTUFBTSxDQUFDekIsQ0FBQyxHQUFHLENBQUwsQ0FBUCxFQUFnQixFQUFoQixDQUpUO0FBS1hoRCxRQUFBQSxjQUFjLEVBQUV5RSxNQUFNLENBQUN6QixDQUFDLEdBQUcsQ0FBTCxDQUxYO0FBTVhqRCxRQUFBQSxXQU5XO0FBT1hKLFFBQUFBLFNBUFc7QUFRWEcsUUFBQUEsU0FBUyxFQUFFLEtBUkE7QUFTWFIsUUFBQUE7QUFUVyxPQUFiO0FBV0Q7O0FBQ0QsV0FBT29GLE9BQVA7QUFDRDs7QUFFRCxRQUFNTSxVQUFOLENBQWlCM1IsT0FBTyxHQUFHLEVBQTNCLEVBQStCO0FBQzdCLFVBQU07QUFBQ0ssTUFBQUEsR0FBRDtBQUFNeVEsTUFBQUE7QUFBTjtBQUFjelEsTUFBQUEsR0FBRyxFQUFFLENBQW5CO0FBQXNCeVEsTUFBQUEsR0FBRyxFQUFFO0FBQTNCLE9BQXNDOVEsT0FBdEMsQ0FBTixDQUQ2QixDQUc3QjtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7OztBQUVBLFVBQU15RCxTQUFTLEdBQUcsSUFBbEI7QUFDQSxVQUFNbU8sZUFBZSxHQUFHQyxNQUFNLENBQUNDLFlBQVAsQ0FBb0JKLFFBQVEsQ0FBQ2pPLFNBQUQsRUFBWSxFQUFaLENBQTVCLENBQXhCO0FBQ0EsVUFBTTJOLE1BQU0sR0FBRyxDQUFDLEtBQUQsRUFBUSxLQUFSLEVBQWUsS0FBZixFQUFzQixLQUF0QixFQUE2Qix5QkFBN0IsQ0FBZjtBQUNBLFVBQU1XLE1BQU0sR0FBR1gsTUFBTSxDQUFDOU8sSUFBUCxDQUFhLEtBQUltQixTQUFVLEVBQTNCLENBQWY7O0FBRUEsUUFBSTtBQUNGLFlBQU15RyxNQUFNLEdBQUcsTUFBTSxLQUFLL0ksSUFBTCxDQUFVLENBQzdCLEtBRDZCLEVBQ3JCLFlBQVc0USxNQUFPLEVBREcsRUFDQSxJQURBLEVBQ00sSUFETixFQUNZMVIsR0FEWixFQUNpQnlRLEdBRGpCLEVBQ3NCLElBRHRCLENBQVYsQ0FBckI7QUFJQSxhQUFPNUcsTUFBTSxDQUFDNkMsS0FBUCxDQUFhLElBQWIsRUFDSnZOLE1BREksQ0FDRyxDQUFDQyxHQUFELEVBQU13TixJQUFOLEtBQWU7QUFDckIsWUFBSUEsSUFBSSxDQUFDek0sTUFBTCxLQUFnQixDQUFwQixFQUF1QjtBQUFFLGlCQUFPZixHQUFQO0FBQWE7O0FBRXRDLGNBQU0sQ0FBQ3VTLEVBQUQsRUFBS0MsRUFBTCxFQUFTQyxFQUFULEVBQWFDLEVBQWIsRUFBaUI3RSxRQUFqQixJQUE2QkwsSUFBSSxDQUFDRixLQUFMLENBQVc2RSxlQUFYLENBQW5DO0FBQ0F0RSxRQUFBQSxRQUFRLENBQ0xQLEtBREgsQ0FDUyxJQURULEVBRUd2QyxHQUZILENBRU80SCxPQUFPLElBQUlBLE9BQU8sQ0FBQ0MsS0FBUixDQUFjQyx3QkFBZCxDQUZsQixFQUdHdEYsTUFISCxDQUdVcUYsS0FBSyxJQUFJQSxLQUFLLEtBQUssSUFIN0IsRUFJR2pFLE9BSkgsQ0FJVyxDQUFDLENBQUNyRCxDQUFELEVBQUkyQyxJQUFKLEVBQVVDLEtBQVYsQ0FBRCxLQUFzQjtBQUFFbE8sVUFBQUEsR0FBRyxDQUFDa08sS0FBRCxDQUFILEdBQWFELElBQWI7QUFBb0IsU0FKdkQ7QUFNQWpPLFFBQUFBLEdBQUcsQ0FBQ3dTLEVBQUQsQ0FBSCxHQUFVRCxFQUFWO0FBQ0F2UyxRQUFBQSxHQUFHLENBQUMwUyxFQUFELENBQUgsR0FBVUQsRUFBVjtBQUVBLGVBQU96UyxHQUFQO0FBQ0QsT0FmSSxFQWVGLEVBZkUsQ0FBUDtBQWdCRCxLQXJCRCxDQXFCRSxPQUFPc0gsR0FBUCxFQUFZO0FBQ1osVUFBSSxtQkFBbUJtQyxJQUFuQixDQUF3Qm5DLEdBQUcsQ0FBQzJCLE1BQTVCLEtBQXVDLHNCQUFzQlEsSUFBdEIsQ0FBMkJuQyxHQUFHLENBQUMyQixNQUEvQixDQUEzQyxFQUFtRjtBQUNqRixlQUFPLEVBQVA7QUFDRCxPQUZELE1BRU87QUFDTCxjQUFNM0IsR0FBTjtBQUNEO0FBQ0Y7QUFDRjs7QUFFRDZHLEVBQUFBLGFBQWEsQ0FBQzJFLGFBQUQsRUFBZ0JqRixRQUFoQixFQUEwQjtBQUNyQyxVQUFNbE0sSUFBSSxHQUFHLENBQUMsb0JBQUQsQ0FBYjs7QUFDQSxTQUFLLE1BQU1nUixPQUFYLElBQXNCOUUsUUFBdEIsRUFBZ0M7QUFDOUJsTSxNQUFBQSxJQUFJLENBQUM4QixJQUFMLENBQVUsV0FBVixFQUF3QixHQUFFa1AsT0FBTyxDQUFDNUUsS0FBTSxJQUFHNEUsT0FBTyxDQUFDM0UsS0FBTSxFQUF6RDtBQUNEOztBQUNELFdBQU8sS0FBS3RNLElBQUwsQ0FBVUMsSUFBVixFQUFnQjtBQUFDRSxNQUFBQSxLQUFLLEVBQUVpUjtBQUFSLEtBQWhCLENBQVA7QUFDRDs7QUFFREMsRUFBQUEsaUJBQWlCLENBQUNsRSxRQUFELEVBQVc7QUFDMUIsV0FBTyxLQUFLbk4sSUFBTCxDQUFVLENBQUMsTUFBRCxFQUFVLElBQUcsMkJBQWFtTixRQUFiLENBQXVCLEVBQXBDLENBQVYsQ0FBUDtBQUNEO0FBRUQ7Ozs7O0FBR0FtRSxFQUFBQSxLQUFLLENBQUNDLFVBQUQsRUFBYTtBQUNoQixXQUFPLEtBQUszSixPQUFMLENBQWEsQ0FBQyxPQUFELEVBQVUySixVQUFWLENBQWIsRUFBb0M7QUFBQ2hSLE1BQUFBLGNBQWMsRUFBRTtBQUFqQixLQUFwQyxDQUFQO0FBQ0Q7O0FBRURpUixFQUFBQSxTQUFTLENBQUN4SSxTQUFELEVBQVk7QUFDbkIsV0FBTyx5QkFBVzNHLGlCQUFLbEIsSUFBTCxDQUFVNkgsU0FBVixFQUFxQixZQUFyQixDQUFYLFdBQXFELE1BQU0sS0FBM0QsQ0FBUDtBQUNEOztBQUVEeUksRUFBQUEsVUFBVSxHQUFHO0FBQ1gsV0FBTyxLQUFLelIsSUFBTCxDQUFVLENBQUMsT0FBRCxFQUFVLFNBQVYsQ0FBVixFQUFnQztBQUFDTyxNQUFBQSxjQUFjLEVBQUU7QUFBakIsS0FBaEMsQ0FBUDtBQUNEOztBQUVEbVIsRUFBQUEsWUFBWSxDQUFDQyxJQUFELEVBQU94SSxLQUFQLEVBQWM7QUFDeEIsUUFBSUEsS0FBSyxDQUFDOUosTUFBTixLQUFpQixDQUFyQixFQUF3QjtBQUN0QixhQUFPSSxPQUFPLENBQUMrQixPQUFSLEVBQVA7QUFDRDs7QUFFRCxXQUFPLEtBQUt4QixJQUFMLENBQVUsQ0FBQyxVQUFELEVBQWMsS0FBSTJSLElBQUssRUFBdkIsRUFBMEIsR0FBR3hJLEtBQUssQ0FBQ0UsR0FBTixDQUFVQyxxQkFBVixDQUE3QixDQUFWLENBQVA7QUFDRDtBQUVEOzs7OztBQUdBLFFBQU1zSSxVQUFOLENBQWlCNUksU0FBakIsRUFBNEI7QUFDMUIsVUFBTTJELE9BQU8sR0FBRyxNQUFNbE4sT0FBTyxDQUFDb1MsR0FBUixDQUFZLENBQ2hDLHlCQUFXeFAsaUJBQUtsQixJQUFMLENBQVU2SCxTQUFWLEVBQXFCLGNBQXJCLENBQVgsQ0FEZ0MsRUFFaEMseUJBQVczRyxpQkFBS2xCLElBQUwsQ0FBVTZILFNBQVYsRUFBcUIsY0FBckIsQ0FBWCxDQUZnQyxDQUFaLENBQXRCO0FBSUEsV0FBTzJELE9BQU8sQ0FBQ21GLElBQVIsQ0FBYUMsQ0FBQyxJQUFJQSxDQUFsQixDQUFQO0FBQ0Q7QUFFRDs7Ozs7QUFHQUMsRUFBQUEsS0FBSyxDQUFDQyxTQUFELEVBQVlwVCxPQUFPLEdBQUcsRUFBdEIsRUFBMEI7QUFDN0IsVUFBTW9CLElBQUksR0FBRyxDQUFDLE9BQUQsQ0FBYjs7QUFDQSxRQUFJcEIsT0FBTyxDQUFDcVQsT0FBWixFQUFxQjtBQUFFalMsTUFBQUEsSUFBSSxDQUFDOEIsSUFBTCxDQUFVLFlBQVY7QUFBMEI7O0FBQ2pELFFBQUlsRCxPQUFPLENBQUNzVCxJQUFaLEVBQWtCO0FBQUVsUyxNQUFBQSxJQUFJLENBQUM4QixJQUFMLENBQVUsUUFBVjtBQUFzQjs7QUFDMUMsUUFBSWxELE9BQU8sQ0FBQ3VULFNBQVosRUFBdUI7QUFBRW5TLE1BQUFBLElBQUksQ0FBQzhCLElBQUwsQ0FBVSxhQUFWO0FBQTJCOztBQUNwRDlCLElBQUFBLElBQUksQ0FBQzhCLElBQUwsQ0FBVWtRLFNBQVYsRUFBcUIsS0FBS3JULFVBQTFCO0FBRUEsV0FBTyxLQUFLb0IsSUFBTCxDQUFVQyxJQUFWLEVBQWdCO0FBQUNHLE1BQUFBLGtCQUFrQixFQUFFLElBQXJCO0FBQTJCRyxNQUFBQSxjQUFjLEVBQUU7QUFBM0MsS0FBaEIsQ0FBUDtBQUNEOztBQUVEOFIsRUFBQUEsS0FBSyxDQUFDQyxVQUFELEVBQWFmLFVBQWIsRUFBeUI7QUFDNUIsV0FBTyxLQUFLdlIsSUFBTCxDQUFVLENBQUMsT0FBRCxFQUFVc1MsVUFBVixFQUFzQmYsVUFBdEIsQ0FBVixFQUE2QztBQUFDblIsTUFBQUEsa0JBQWtCLEVBQUUsSUFBckI7QUFBMkJHLE1BQUFBLGNBQWMsRUFBRTtBQUEzQyxLQUE3QyxDQUFQO0FBQ0Q7O0FBRURnUyxFQUFBQSxJQUFJLENBQUNELFVBQUQsRUFBYWYsVUFBYixFQUF5QjFTLE9BQU8sR0FBRyxFQUFuQyxFQUF1QztBQUN6QyxVQUFNb0IsSUFBSSxHQUFHLENBQUMsTUFBRCxFQUFTcVMsVUFBVCxFQUFxQnpULE9BQU8sQ0FBQzJULE9BQVIsSUFBbUJqQixVQUF4QyxDQUFiOztBQUNBLFFBQUkxUyxPQUFPLENBQUM0VCxNQUFaLEVBQW9CO0FBQ2xCeFMsTUFBQUEsSUFBSSxDQUFDOEIsSUFBTCxDQUFVLFdBQVY7QUFDRDs7QUFDRCxXQUFPLEtBQUs2RixPQUFMLENBQWEzSCxJQUFiLEVBQW1CO0FBQUNHLE1BQUFBLGtCQUFrQixFQUFFLElBQXJCO0FBQTJCRyxNQUFBQSxjQUFjLEVBQUU7QUFBM0MsS0FBbkIsQ0FBUDtBQUNEOztBQUVEd0IsRUFBQUEsSUFBSSxDQUFDdVEsVUFBRCxFQUFhZixVQUFiLEVBQXlCMVMsT0FBTyxHQUFHLEVBQW5DLEVBQXVDO0FBQ3pDLFVBQU1vQixJQUFJLEdBQUcsQ0FBQyxNQUFELEVBQVNxUyxVQUFVLElBQUksUUFBdkIsRUFBaUN6VCxPQUFPLENBQUMyVCxPQUFSLElBQW9CLGNBQWFqQixVQUFXLEVBQTdFLENBQWI7O0FBQ0EsUUFBSTFTLE9BQU8sQ0FBQzZULFdBQVosRUFBeUI7QUFBRXpTLE1BQUFBLElBQUksQ0FBQzhCLElBQUwsQ0FBVSxnQkFBVjtBQUE4Qjs7QUFDekQsUUFBSWxELE9BQU8sQ0FBQzhULEtBQVosRUFBbUI7QUFBRTFTLE1BQUFBLElBQUksQ0FBQzhCLElBQUwsQ0FBVSxTQUFWO0FBQXVCOztBQUM1QyxXQUFPLEtBQUsvQixJQUFMLENBQVVDLElBQVYsRUFBZ0I7QUFBQ0csTUFBQUEsa0JBQWtCLEVBQUUsSUFBckI7QUFBMkJHLE1BQUFBLGNBQWMsRUFBRTtBQUEzQyxLQUFoQixDQUFQO0FBQ0Q7QUFFRDs7Ozs7QUFHQXFTLEVBQUFBLEtBQUssQ0FBQ3JVLElBQUQsRUFBT3NVLFFBQVEsR0FBRyxNQUFsQixFQUEwQjtBQUM3QixVQUFNQyxVQUFVLEdBQUcsQ0FBQyxNQUFELENBQW5COztBQUNBLFFBQUksQ0FBQ0EsVUFBVSxDQUFDcEwsUUFBWCxDQUFvQm5KLElBQXBCLENBQUwsRUFBZ0M7QUFDOUIsWUFBTSxJQUFJVCxLQUFKLENBQVcsZ0JBQWVTLElBQUsscUJBQW9CdVUsVUFBVSxDQUFDM1IsSUFBWCxDQUFnQixJQUFoQixDQUFzQixFQUF6RSxDQUFOO0FBQ0Q7O0FBQ0QsV0FBTyxLQUFLbkIsSUFBTCxDQUFVLENBQUMsT0FBRCxFQUFXLEtBQUl6QixJQUFLLEVBQXBCLEVBQXVCc1UsUUFBdkIsQ0FBVixDQUFQO0FBQ0Q7O0FBRURFLEVBQUFBLFNBQVMsQ0FBQ3BELEdBQUQsRUFBTTtBQUNiLFdBQU8sS0FBSzNQLElBQUwsQ0FBVSxDQUFDLFlBQUQsRUFBZSxJQUFmLEVBQXFCMlAsR0FBckIsQ0FBVixDQUFQO0FBQ0Q7QUFFRDs7Ozs7QUFHQXFELEVBQUFBLFFBQVEsQ0FBQ3pCLFVBQUQsRUFBYTFTLE9BQU8sR0FBRyxFQUF2QixFQUEyQjtBQUNqQyxVQUFNb0IsSUFBSSxHQUFHLENBQUMsVUFBRCxDQUFiOztBQUNBLFFBQUlwQixPQUFPLENBQUNvVSxTQUFaLEVBQXVCO0FBQ3JCaFQsTUFBQUEsSUFBSSxDQUFDOEIsSUFBTCxDQUFVLElBQVY7QUFDRDs7QUFDRDlCLElBQUFBLElBQUksQ0FBQzhCLElBQUwsQ0FBVXdQLFVBQVY7O0FBQ0EsUUFBSTFTLE9BQU8sQ0FBQ3FVLFVBQVosRUFBd0I7QUFDdEIsVUFBSXJVLE9BQU8sQ0FBQ3NVLEtBQVosRUFBbUI7QUFBRWxULFFBQUFBLElBQUksQ0FBQzhCLElBQUwsQ0FBVSxTQUFWO0FBQXVCOztBQUM1QzlCLE1BQUFBLElBQUksQ0FBQzhCLElBQUwsQ0FBVWxELE9BQU8sQ0FBQ3FVLFVBQWxCO0FBQ0Q7O0FBRUQsV0FBTyxLQUFLbFQsSUFBTCxDQUFVQyxJQUFWLEVBQWdCO0FBQUNNLE1BQUFBLGNBQWMsRUFBRTtBQUFqQixLQUFoQixDQUFQO0FBQ0Q7O0FBRUQsUUFBTTZTLFdBQU4sR0FBb0I7QUFDbEIsVUFBTXhDLE1BQU0sR0FBRyxDQUNiLGVBRGEsRUFDSSxTQURKLEVBQ2Usa0JBRGYsRUFFYixhQUZhLEVBRUUsd0JBRkYsRUFFNEIsdUJBRjVCLEVBR2IsU0FIYSxFQUdGLG9CQUhFLEVBR29CLG1CQUhwQixFQUlielAsSUFKYSxDQUlSLEtBSlEsQ0FBZjtBQU1BLFVBQU00SCxNQUFNLEdBQUcsTUFBTSxLQUFLL0ksSUFBTCxDQUFVLENBQUMsY0FBRCxFQUFrQixZQUFXNFEsTUFBTyxFQUFwQyxFQUF1QyxlQUF2QyxDQUFWLENBQXJCO0FBQ0EsV0FBTzdILE1BQU0sQ0FBQ2xILElBQVAsR0FBYytKLEtBQWQsQ0FBb0JrQywwQkFBcEIsRUFBdUN6RSxHQUF2QyxDQUEyQ3lDLElBQUksSUFBSTtBQUN4RCxZQUFNLENBQ0prRSxHQURJLEVBQ0NxRCxJQURELEVBQ085RyxJQURQLEVBRUorRyxtQkFGSSxFQUVpQkMsa0JBRmpCLEVBRXFDQyxpQkFGckMsRUFHSkMsZUFISSxFQUdhQyxjQUhiLEVBRzZCQyxhQUg3QixJQUlGN0gsSUFBSSxDQUFDRixLQUFMLENBQVcsSUFBWCxDQUpKO0FBTUEsWUFBTWdJLE1BQU0sR0FBRztBQUFDckgsUUFBQUEsSUFBRDtBQUFPeUQsUUFBQUEsR0FBUDtBQUFZcUQsUUFBQUEsSUFBSSxFQUFFQSxJQUFJLEtBQUs7QUFBM0IsT0FBZjs7QUFDQSxVQUFJQyxtQkFBbUIsSUFBSUMsa0JBQXZCLElBQTZDQyxpQkFBakQsRUFBb0U7QUFDbEVJLFFBQUFBLE1BQU0sQ0FBQ0MsUUFBUCxHQUFrQjtBQUNoQkMsVUFBQUEsV0FBVyxFQUFFUixtQkFERztBQUVoQmhCLFVBQUFBLFVBQVUsRUFBRWlCLGtCQUZJO0FBR2hCUSxVQUFBQSxTQUFTLEVBQUVQO0FBSEssU0FBbEI7QUFLRDs7QUFDRCxVQUFJSSxNQUFNLENBQUNDLFFBQVAsSUFBbUJKLGVBQW5CLElBQXNDQyxjQUF0QyxJQUF3REMsYUFBNUQsRUFBMkU7QUFDekVDLFFBQUFBLE1BQU0sQ0FBQzdSLElBQVAsR0FBYztBQUNaK1IsVUFBQUEsV0FBVyxFQUFFTCxlQUREO0FBRVpuQixVQUFBQSxVQUFVLEVBQUVvQixjQUFjLElBQUtFLE1BQU0sQ0FBQ0MsUUFBUCxJQUFtQkQsTUFBTSxDQUFDQyxRQUFQLENBQWdCdkIsVUFGdEQ7QUFHWnlCLFVBQUFBLFNBQVMsRUFBRUosYUFBYSxJQUFLQyxNQUFNLENBQUNDLFFBQVAsSUFBbUJELE1BQU0sQ0FBQ0MsUUFBUCxDQUFnQkU7QUFIcEQsU0FBZDtBQUtEOztBQUNELGFBQU9ILE1BQVA7QUFDRCxLQXZCTSxDQUFQO0FBd0JEOztBQUVELFFBQU1JLHFCQUFOLENBQTRCaEUsR0FBNUIsRUFBaUNpRSxNQUFNLEdBQUcsRUFBMUMsRUFBOEM7QUFDNUMsVUFBTWhVLElBQUksR0FBRyxDQUFDLFFBQUQsRUFBVyxxQkFBWCxFQUFrQyxZQUFsQyxFQUFnRCtQLEdBQWhELENBQWI7O0FBQ0EsUUFBSWlFLE1BQU0sQ0FBQ0MsU0FBUCxJQUFvQkQsTUFBTSxDQUFDRSxVQUEvQixFQUEyQztBQUN6Q2xVLE1BQUFBLElBQUksQ0FBQzhLLE1BQUwsQ0FBWSxDQUFaLEVBQWUsQ0FBZixFQUFrQixPQUFsQjtBQUNELEtBRkQsTUFFTyxJQUFJa0osTUFBTSxDQUFDRSxVQUFYLEVBQXVCO0FBQzVCbFUsTUFBQUEsSUFBSSxDQUFDOEssTUFBTCxDQUFZLENBQVosRUFBZSxDQUFmLEVBQWtCLFdBQWxCO0FBQ0Q7O0FBQ0QsV0FBTyxDQUFDLE1BQU0sS0FBSy9LLElBQUwsQ0FBVUMsSUFBVixDQUFQLEVBQXdCNEIsSUFBeEIsR0FBK0IrSixLQUEvQixDQUFxQ2tDLDBCQUFyQyxDQUFQO0FBQ0Q7O0FBRURzRyxFQUFBQSxhQUFhLENBQUNqTCxLQUFELEVBQVEwSixRQUFSLEVBQWtCO0FBQzdCLFFBQUkxSixLQUFLLENBQUM5SixNQUFOLEtBQWlCLENBQXJCLEVBQXdCO0FBQUUsYUFBTyxJQUFQO0FBQWM7O0FBQ3hDLFVBQU1ZLElBQUksR0FBRyxDQUFDLFVBQUQsQ0FBYjs7QUFDQSxRQUFJNFMsUUFBSixFQUFjO0FBQUU1UyxNQUFBQSxJQUFJLENBQUM4QixJQUFMLENBQVU4USxRQUFWO0FBQXNCOztBQUN0QyxXQUFPLEtBQUs3UyxJQUFMLENBQVVDLElBQUksQ0FBQ21KLE1BQUwsQ0FBWSxJQUFaLEVBQWtCRCxLQUFLLENBQUNFLEdBQU4sQ0FBVUMscUJBQVYsQ0FBbEIsQ0FBVixFQUFzRDtBQUFDL0ksTUFBQUEsY0FBYyxFQUFFO0FBQWpCLEtBQXRELENBQVA7QUFDRDs7QUFFRCxRQUFNOFQsWUFBTixHQUFxQjtBQUNuQixXQUFPLENBQUMsTUFBTSxLQUFLclUsSUFBTCxDQUFVLENBQUMsVUFBRCxFQUFhLFlBQWIsRUFBMkIsT0FBM0IsRUFBb0MsVUFBcEMsRUFBZ0QsTUFBaEQsQ0FBVixDQUFQLEVBQTJFNkIsSUFBM0UsRUFBUDtBQUNEOztBQUVELFFBQU00SCxTQUFOLENBQWdCd0ssTUFBaEIsRUFBd0I7QUFBQ0ssSUFBQUE7QUFBRCxNQUFVLEVBQWxDLEVBQXNDO0FBQ3BDLFFBQUl2TCxNQUFKOztBQUNBLFFBQUk7QUFDRixVQUFJOUksSUFBSSxHQUFHLENBQUMsUUFBRCxDQUFYOztBQUNBLFVBQUlxVSxLQUFLLElBQUl2VCxJQUFJLENBQUNtRCxVQUFMLEVBQWIsRUFBZ0M7QUFBRWpFLFFBQUFBLElBQUksQ0FBQzhCLElBQUwsQ0FBVSxTQUFWO0FBQXVCOztBQUN6RDlCLE1BQUFBLElBQUksR0FBR0EsSUFBSSxDQUFDbUosTUFBTCxDQUFZNkssTUFBWixDQUFQO0FBQ0FsTCxNQUFBQSxNQUFNLEdBQUcsTUFBTSxLQUFLL0ksSUFBTCxDQUFVQyxJQUFWLENBQWY7QUFDRCxLQUxELENBS0UsT0FBTzJGLEdBQVAsRUFBWTtBQUNaLFVBQUlBLEdBQUcsQ0FBQzBCLElBQUosS0FBYSxDQUFqQixFQUFvQjtBQUNsQjtBQUNBLGVBQU8sSUFBUDtBQUNELE9BSEQsTUFHTztBQUNMLGNBQU0xQixHQUFOO0FBQ0Q7QUFDRjs7QUFFRCxXQUFPbUQsTUFBTSxDQUFDbEgsSUFBUCxFQUFQO0FBQ0Q7O0FBRUQwUyxFQUFBQSxTQUFTLENBQUNOLE1BQUQsRUFBUzNILEtBQVQsRUFBZ0I7QUFBQ2tJLElBQUFBO0FBQUQsTUFBZSxFQUEvQixFQUFtQztBQUMxQyxRQUFJdlUsSUFBSSxHQUFHLENBQUMsUUFBRCxDQUFYOztBQUNBLFFBQUl1VSxVQUFKLEVBQWdCO0FBQUV2VSxNQUFBQSxJQUFJLENBQUM4QixJQUFMLENBQVUsZUFBVjtBQUE2Qjs7QUFDL0M5QixJQUFBQSxJQUFJLEdBQUdBLElBQUksQ0FBQ21KLE1BQUwsQ0FBWTZLLE1BQVosRUFBb0IzSCxLQUFwQixDQUFQO0FBQ0EsV0FBTyxLQUFLdE0sSUFBTCxDQUFVQyxJQUFWLEVBQWdCO0FBQUNNLE1BQUFBLGNBQWMsRUFBRTtBQUFqQixLQUFoQixDQUFQO0FBQ0Q7O0FBRURrVSxFQUFBQSxXQUFXLENBQUNSLE1BQUQsRUFBUztBQUNsQixXQUFPLEtBQUtqVSxJQUFMLENBQVUsQ0FBQyxRQUFELEVBQVcsU0FBWCxFQUFzQmlVLE1BQXRCLENBQVYsRUFBeUM7QUFBQzFULE1BQUFBLGNBQWMsRUFBRTtBQUFqQixLQUF6QyxDQUFQO0FBQ0Q7O0FBRUQsUUFBTW1VLFVBQU4sR0FBbUI7QUFDakIsUUFBSTNMLE1BQU0sR0FBRyxNQUFNLEtBQUtVLFNBQUwsQ0FBZSxDQUFDLGNBQUQsRUFBaUIscUJBQWpCLENBQWYsRUFBd0Q7QUFBQzZLLE1BQUFBLEtBQUssRUFBRTtBQUFSLEtBQXhELENBQW5COztBQUNBLFFBQUl2TCxNQUFKLEVBQVk7QUFDVkEsTUFBQUEsTUFBTSxHQUFHQSxNQUFNLENBQUNsSCxJQUFQLEVBQVQ7O0FBQ0EsVUFBSSxDQUFDa0gsTUFBTSxDQUFDMUosTUFBWixFQUFvQjtBQUFFLGVBQU8sRUFBUDtBQUFZOztBQUNsQyxhQUFPMEosTUFBTSxDQUFDNkMsS0FBUCxDQUFhLElBQWIsRUFBbUJ2QyxHQUFuQixDQUF1QnlDLElBQUksSUFBSTtBQUNwQyxjQUFNb0YsS0FBSyxHQUFHcEYsSUFBSSxDQUFDb0YsS0FBTCxDQUFXLDBCQUFYLENBQWQ7QUFDQSxlQUFPO0FBQ0wzRSxVQUFBQSxJQUFJLEVBQUUyRSxLQUFLLENBQUMsQ0FBRCxDQUROO0FBRUx5RCxVQUFBQSxHQUFHLEVBQUV6RCxLQUFLLENBQUMsQ0FBRDtBQUZMLFNBQVA7QUFJRCxPQU5NLENBQVA7QUFPRCxLQVZELE1BVU87QUFDTCxhQUFPLEVBQVA7QUFDRDtBQUNGOztBQUVEMEQsRUFBQUEsU0FBUyxDQUFDckksSUFBRCxFQUFPb0ksR0FBUCxFQUFZO0FBQ25CLFdBQU8sS0FBSzNVLElBQUwsQ0FBVSxDQUFDLFFBQUQsRUFBVyxLQUFYLEVBQWtCdU0sSUFBbEIsRUFBd0JvSSxHQUF4QixDQUFWLENBQVA7QUFDRDs7QUFFRCxRQUFNRSxVQUFOLENBQWlCO0FBQUMxSCxJQUFBQSxRQUFEO0FBQVdoTixJQUFBQTtBQUFYLE1BQW9CLEVBQXJDLEVBQXlDO0FBQ3ZDLFFBQUk0SSxNQUFKOztBQUNBLFFBQUlvRSxRQUFKLEVBQWM7QUFDWixVQUFJO0FBQ0ZwRSxRQUFBQSxNQUFNLEdBQUcsQ0FBQyxNQUFNLEtBQUsvSSxJQUFMLENBQVUsQ0FBQyxhQUFELEVBQWdCLElBQWhCLEVBQXNCbU4sUUFBdEIsQ0FBVixFQUEyQztBQUFDNU0sVUFBQUEsY0FBYyxFQUFFO0FBQWpCLFNBQTNDLENBQVAsRUFBMkVzQixJQUEzRSxFQUFUO0FBQ0QsT0FGRCxDQUVFLE9BQU9pRyxDQUFQLEVBQVU7QUFDVixZQUFJQSxDQUFDLENBQUNQLE1BQUYsSUFBWU8sQ0FBQyxDQUFDUCxNQUFGLENBQVMySixLQUFULENBQWUsa0RBQWYsQ0FBaEIsRUFBb0Y7QUFDbEZuSSxVQUFBQSxNQUFNLEdBQUcsSUFBVDtBQUNELFNBRkQsTUFFTztBQUNMLGdCQUFNakIsQ0FBTjtBQUNEO0FBQ0Y7QUFDRixLQVZELE1BVU8sSUFBSTNILEtBQUosRUFBVztBQUNoQjRJLE1BQUFBLE1BQU0sR0FBRyxDQUFDLE1BQU0sS0FBSy9JLElBQUwsQ0FBVSxDQUFDLGFBQUQsRUFBZ0IsSUFBaEIsRUFBc0IsU0FBdEIsQ0FBVixFQUE0QztBQUFDRyxRQUFBQSxLQUFEO0FBQVFJLFFBQUFBLGNBQWMsRUFBRTtBQUF4QixPQUE1QyxDQUFQLEVBQW1Gc0IsSUFBbkYsRUFBVDtBQUNELEtBRk0sTUFFQTtBQUNMLFlBQU0sSUFBSS9ELEtBQUosQ0FBVSxnQ0FBVixDQUFOO0FBQ0Q7O0FBQ0QsV0FBT2lMLE1BQVA7QUFDRDs7QUFFRCxRQUFNK0wsZ0JBQU4sQ0FBdUJDLFdBQXZCLEVBQW9DL0UsR0FBcEMsRUFBeUM7QUFDdkMsVUFBTWpILE1BQU0sR0FBRyxNQUFNLEtBQUsvSSxJQUFMLENBQVUsQ0FBQyxVQUFELEVBQWEsSUFBYixFQUFtQmdRLEdBQW5CLENBQVYsQ0FBckI7QUFDQSxVQUFNbkgsb0JBQUdtTSxTQUFILENBQWFELFdBQWIsRUFBMEJoTSxNQUExQixFQUFrQztBQUFDa0IsTUFBQUEsUUFBUSxFQUFFO0FBQVgsS0FBbEMsQ0FBTjtBQUNBLFdBQU84SyxXQUFQO0FBQ0Q7O0FBRUQsUUFBTUUsZUFBTixDQUFzQmpGLEdBQXRCLEVBQTJCO0FBQ3pCLFdBQU8sTUFBTSxLQUFLaFEsSUFBTCxDQUFVLENBQUMsVUFBRCxFQUFhLElBQWIsRUFBbUJnUSxHQUFuQixDQUFWLENBQWI7QUFDRDs7QUFFRCxRQUFNa0YsU0FBTixDQUFnQkMsUUFBaEIsRUFBMEJDLGNBQTFCLEVBQTBDQyxVQUExQyxFQUFzREMsVUFBdEQsRUFBa0U7QUFDaEUsVUFBTXJWLElBQUksR0FBRyxDQUNYLFlBRFcsRUFDRyxJQURILEVBQ1NrVixRQURULEVBQ21CQyxjQURuQixFQUNtQ0MsVUFEbkMsRUFFWCxJQUZXLEVBRUwsU0FGSyxFQUVNLElBRk4sRUFFWSxlQUZaLEVBRTZCLElBRjdCLEVBRW1DLGdCQUZuQyxDQUFiO0FBSUEsUUFBSXRNLE1BQUo7QUFDQSxRQUFJd00sUUFBUSxHQUFHLEtBQWY7O0FBQ0EsUUFBSTtBQUNGeE0sTUFBQUEsTUFBTSxHQUFHLE1BQU0sS0FBSy9JLElBQUwsQ0FBVUMsSUFBVixDQUFmO0FBQ0QsS0FGRCxDQUVFLE9BQU82SCxDQUFQLEVBQVU7QUFDVixVQUFJQSxDQUFDLFlBQVlqSyxRQUFiLElBQXlCaUssQ0FBQyxDQUFDUixJQUFGLEtBQVcsQ0FBeEMsRUFBMkM7QUFDekN5QixRQUFBQSxNQUFNLEdBQUdqQixDQUFDLENBQUNOLE1BQVg7QUFDQStOLFFBQUFBLFFBQVEsR0FBRyxJQUFYO0FBQ0QsT0FIRCxNQUdPO0FBQ0wsY0FBTXpOLENBQU47QUFDRDtBQUNGLEtBaEIrRCxDQWtCaEU7QUFDQTs7O0FBQ0EsVUFBTTBOLGtCQUFrQixHQUFHblQsaUJBQUtiLE9BQUwsQ0FBYSxLQUFLNUMsVUFBbEIsRUFBOEIwVyxVQUE5QixDQUEzQjs7QUFDQSxVQUFNek0sb0JBQUdtTSxTQUFILENBQWFRLGtCQUFiLEVBQWlDek0sTUFBakMsRUFBeUM7QUFBQ2tCLE1BQUFBLFFBQVEsRUFBRTtBQUFYLEtBQXpDLENBQU47QUFFQSxXQUFPO0FBQUNrRCxNQUFBQSxRQUFRLEVBQUVnSSxRQUFYO0FBQXFCRyxNQUFBQSxVQUFyQjtBQUFpQ0MsTUFBQUE7QUFBakMsS0FBUDtBQUNEOztBQUVELFFBQU1FLHlCQUFOLENBQWdDdEksUUFBaEMsRUFBMEN1SSxhQUExQyxFQUF5REMsT0FBekQsRUFBa0VDLFNBQWxFLEVBQTZFO0FBQzNFLFVBQU1DLFdBQVcsR0FBRywyQkFBYTFJLFFBQWIsQ0FBcEI7QUFDQSxVQUFNMkksUUFBUSxHQUFHLE1BQU0sS0FBS0MsV0FBTCxDQUFpQjVJLFFBQWpCLENBQXZCO0FBQ0EsUUFBSTZJLFNBQVMsR0FBSSwrQ0FBOENILFdBQVksSUFBM0U7O0FBQ0EsUUFBSUgsYUFBSixFQUFtQjtBQUFFTSxNQUFBQSxTQUFTLElBQUssR0FBRUYsUUFBUyxJQUFHSixhQUFjLE9BQU1HLFdBQVksSUFBNUQ7QUFBa0U7O0FBQ3ZGLFFBQUlGLE9BQUosRUFBYTtBQUFFSyxNQUFBQSxTQUFTLElBQUssR0FBRUYsUUFBUyxJQUFHSCxPQUFRLE9BQU1FLFdBQVksSUFBdEQ7QUFBNEQ7O0FBQzNFLFFBQUlELFNBQUosRUFBZTtBQUFFSSxNQUFBQSxTQUFTLElBQUssR0FBRUYsUUFBUyxJQUFHRixTQUFVLE9BQU1DLFdBQVksSUFBeEQ7QUFBOEQ7O0FBQy9FLFdBQU8sS0FBSzdWLElBQUwsQ0FBVSxDQUFDLGNBQUQsRUFBaUIsY0FBakIsQ0FBVixFQUE0QztBQUFDRyxNQUFBQSxLQUFLLEVBQUU2VixTQUFSO0FBQW1CelYsTUFBQUEsY0FBYyxFQUFFO0FBQW5DLEtBQTVDLENBQVA7QUFDRDs7QUFFRCxRQUFNd1YsV0FBTixDQUFrQjVJLFFBQWxCLEVBQTRCO0FBQzFCLFVBQU1wRSxNQUFNLEdBQUcsTUFBTSxLQUFLL0ksSUFBTCxDQUFVLENBQUMsVUFBRCxFQUFhLFNBQWIsRUFBd0IsSUFBeEIsRUFBOEIsMkJBQWFtTixRQUFiLENBQTlCLENBQVYsQ0FBckI7O0FBQ0EsUUFBSXBFLE1BQUosRUFBWTtBQUNWLGFBQU9BLE1BQU0sQ0FBQ2xCLEtBQVAsQ0FBYSxDQUFiLEVBQWdCLENBQWhCLENBQVA7QUFDRCxLQUZELE1BRU87QUFDTCxZQUFNK0csVUFBVSxHQUFHLE1BQU0sK0JBQWlCdk0saUJBQUtsQixJQUFMLENBQVUsS0FBS3ZDLFVBQWYsRUFBMkJ1TyxRQUEzQixDQUFqQixDQUF6QjtBQUNBLFlBQU0wQixPQUFPLEdBQUcsTUFBTSw0QkFBY3hNLGlCQUFLbEIsSUFBTCxDQUFVLEtBQUt2QyxVQUFmLEVBQTJCdU8sUUFBM0IsQ0FBZCxDQUF0Qjs7QUFDQSxVQUFJMEIsT0FBSixFQUFhO0FBQ1gsZUFBT0ksaUJBQUtDLEtBQUwsQ0FBV0UsT0FBbEI7QUFDRCxPQUZELE1BRU8sSUFBSVIsVUFBSixFQUFnQjtBQUNyQixlQUFPSyxpQkFBS0MsS0FBTCxDQUFXQyxVQUFsQjtBQUNELE9BRk0sTUFFQTtBQUNMLGVBQU9GLGlCQUFLQyxLQUFMLENBQVdHLE1BQWxCO0FBQ0Q7QUFDRjtBQUNGOztBQUVENEcsRUFBQUEsT0FBTyxHQUFHO0FBQ1IsU0FBS2xYLFlBQUwsQ0FBa0J5SCxPQUFsQjtBQUNEOztBQTVpQ3NDOzs7O2dCQUFwQjdILG1CLHFCQUNNO0FBQ3ZCd0IsRUFBQUEsS0FBSyxFQUFFLElBRGdCO0FBRXZCQyxFQUFBQSxrQkFBa0IsRUFBRSxLQUZHO0FBR3ZCQyxFQUFBQSxhQUFhLEVBQUUsS0FIUTtBQUl2QkMsRUFBQUEsZ0JBQWdCLEVBQUUsS0FKSztBQUt2QkMsRUFBQUEsY0FBYyxFQUFFO0FBTE8sQzs7QUE4aUMzQixTQUFTK08sbUJBQVQsQ0FBNkJuQyxRQUE3QixFQUF1QzJCLFFBQXZDLEVBQWlEN0MsSUFBakQsRUFBdUQrQyxRQUF2RCxFQUFpRTtBQUMvRCxRQUFNa0gsS0FBSyxHQUFHLEVBQWQ7O0FBQ0EsTUFBSXBILFFBQUosRUFBYztBQUNaLFFBQUlxSCxTQUFKO0FBQ0EsUUFBSUMsS0FBSjs7QUFDQSxRQUFJbkssSUFBSSxLQUFLZ0QsaUJBQUtDLEtBQUwsQ0FBV0UsT0FBeEIsRUFBaUM7QUFDL0IrRyxNQUFBQSxTQUFTLEdBQUcsS0FBWjtBQUNBQyxNQUFBQSxLQUFLLEdBQUcsQ0FBRSxJQUFHLDJCQUFhcEgsUUFBYixDQUF1QixFQUE1QixFQUErQiw4QkFBL0IsQ0FBUjtBQUNELEtBSEQsTUFHTztBQUNMbUgsTUFBQUEsU0FBUyxHQUFHckgsUUFBUSxDQUFDQSxRQUFRLENBQUN6UCxNQUFULEdBQWtCLENBQW5CLENBQVIsS0FBa0MsSUFBOUM7QUFDQStXLE1BQUFBLEtBQUssR0FBR3RILFFBQVEsQ0FBQ2pOLElBQVQsR0FBZ0IrSixLQUFoQixDQUFzQmtDLDBCQUF0QixFQUF5Q3pFLEdBQXpDLENBQTZDeUMsSUFBSSxJQUFLLElBQUdBLElBQUssRUFBOUQsQ0FBUjtBQUNEOztBQUNELFFBQUlxSyxTQUFKLEVBQWU7QUFBRUMsTUFBQUEsS0FBSyxDQUFDclUsSUFBTixDQUFXLDhCQUFYO0FBQTZDOztBQUM5RG1VLElBQUFBLEtBQUssQ0FBQ25VLElBQU4sQ0FBVztBQUNUcVUsTUFBQUEsS0FEUztBQUVUQyxNQUFBQSxZQUFZLEVBQUUsQ0FGTDtBQUdUQyxNQUFBQSxZQUFZLEVBQUUsQ0FITDtBQUlUQyxNQUFBQSxZQUFZLEVBQUUsQ0FKTDtBQUtUQyxNQUFBQSxPQUFPLEVBQUUsRUFMQTtBQU1UQyxNQUFBQSxZQUFZLEVBQUVOLFNBQVMsR0FBR0MsS0FBSyxDQUFDL1csTUFBTixHQUFlLENBQWxCLEdBQXNCK1csS0FBSyxDQUFDL1c7QUFOMUMsS0FBWDtBQVFEOztBQUNELFNBQU87QUFDTG9QLElBQUFBLE9BQU8sRUFBRSxJQURKO0FBRUxDLElBQUFBLE9BQU8sRUFBRSw4QkFBZ0J2QixRQUFoQixDQUZKO0FBR0x1SixJQUFBQSxPQUFPLEVBQUUsSUFISjtBQUlMcE0sSUFBQUEsT0FBTyxFQUFFMkIsSUFKSjtBQUtMOEIsSUFBQUEsTUFBTSxFQUFFLE9BTEg7QUFNTG1JLElBQUFBO0FBTkssR0FBUDtBQVFEIiwic291cmNlUm9vdCI6Ii9idWlsZC9hdG9tL3NyYy9hdG9tLTEuMzcuMC9vdXQvYXBwL25vZGVfbW9kdWxlcy9naXRodWIiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCBvcyBmcm9tICdvcyc7XG5pbXBvcnQgY2hpbGRQcm9jZXNzIGZyb20gJ2NoaWxkX3Byb2Nlc3MnO1xuaW1wb3J0IGZzIGZyb20gJ2ZzLWV4dHJhJztcbmltcG9ydCB1dGlsIGZyb20gJ3V0aWwnO1xuaW1wb3J0IHtyZW1vdGV9IGZyb20gJ2VsZWN0cm9uJztcblxuaW1wb3J0IHtDb21wb3NpdGVEaXNwb3NhYmxlfSBmcm9tICdldmVudC1raXQnO1xuaW1wb3J0IHtHaXRQcm9jZXNzfSBmcm9tICdkdWdpdGUnO1xuaW1wb3J0IHtwYXJzZSBhcyBwYXJzZURpZmZ9IGZyb20gJ3doYXQtdGhlLWRpZmYnO1xuaW1wb3J0IHtwYXJzZSBhcyBwYXJzZVN0YXR1c30gZnJvbSAnd2hhdC10aGUtc3RhdHVzJztcblxuaW1wb3J0IEdpdFByb21wdFNlcnZlciBmcm9tICcuL2dpdC1wcm9tcHQtc2VydmVyJztcbmltcG9ydCBHaXRUZW1wRGlyIGZyb20gJy4vZ2l0LXRlbXAtZGlyJztcbmltcG9ydCBBc3luY1F1ZXVlIGZyb20gJy4vYXN5bmMtcXVldWUnO1xuaW1wb3J0IHtpbmNyZW1lbnRDb3VudGVyfSBmcm9tICcuL3JlcG9ydGVyLXByb3h5JztcbmltcG9ydCB7XG4gIGdldER1Z2l0ZVBhdGgsIGdldFNoYXJlZE1vZHVsZVBhdGgsIGdldEF0b21IZWxwZXJQYXRoLFxuICBleHRyYWN0Q29BdXRob3JzQW5kUmF3Q29tbWl0TWVzc2FnZSwgZmlsZUV4aXN0cywgaXNGaWxlRXhlY3V0YWJsZSwgaXNGaWxlU3ltbGluaywgaXNCaW5hcnksXG4gIG5vcm1hbGl6ZUdpdEhlbHBlclBhdGgsIHRvTmF0aXZlUGF0aFNlcCwgdG9HaXRQYXRoU2VwLCBMSU5FX0VORElOR19SRUdFWCwgQ09fQVVUSE9SX1JFR0VYLFxufSBmcm9tICcuL2hlbHBlcnMnO1xuaW1wb3J0IEdpdFRpbWluZ3NWaWV3IGZyb20gJy4vdmlld3MvZ2l0LXRpbWluZ3Mtdmlldyc7XG5pbXBvcnQgRmlsZSBmcm9tICcuL21vZGVscy9wYXRjaC9maWxlJztcbmltcG9ydCBXb3JrZXJNYW5hZ2VyIGZyb20gJy4vd29ya2VyLW1hbmFnZXInO1xuXG5jb25zdCBNQVhfU1RBVFVTX09VVFBVVF9MRU5HVEggPSAxMDI0ICogMTAyNCAqIDEwO1xuXG5sZXQgaGVhZGxlc3MgPSBudWxsO1xubGV0IGV4ZWNQYXRoUHJvbWlzZSA9IG51bGw7XG5cbmV4cG9ydCBjbGFzcyBHaXRFcnJvciBleHRlbmRzIEVycm9yIHtcbiAgY29uc3RydWN0b3IobWVzc2FnZSkge1xuICAgIHN1cGVyKG1lc3NhZ2UpO1xuICAgIHRoaXMubWVzc2FnZSA9IG1lc3NhZ2U7XG4gICAgdGhpcy5zdGFjayA9IG5ldyBFcnJvcigpLnN0YWNrO1xuICB9XG59XG5cbmV4cG9ydCBjbGFzcyBMYXJnZVJlcG9FcnJvciBleHRlbmRzIEVycm9yIHtcbiAgY29uc3RydWN0b3IobWVzc2FnZSkge1xuICAgIHN1cGVyKG1lc3NhZ2UpO1xuICAgIHRoaXMubWVzc2FnZSA9IG1lc3NhZ2U7XG4gICAgdGhpcy5zdGFjayA9IG5ldyBFcnJvcigpLnN0YWNrO1xuICB9XG59XG5cbi8vIGlnbm9yZWQgZm9yIHRoZSBwdXJwb3NlcyBvZiB1c2FnZSBtZXRyaWNzIHRyYWNraW5nIGJlY2F1c2UgdGhleSdyZSBub2lzeVxuY29uc3QgSUdOT1JFRF9HSVRfQ09NTUFORFMgPSBbJ2NhdC1maWxlJywgJ2NvbmZpZycsICdkaWZmJywgJ2Zvci1lYWNoLXJlZicsICdsb2cnLCAncmV2LXBhcnNlJywgJ3N0YXR1cyddO1xuXG5jb25zdCBESVNBQkxFX0NPTE9SX0ZMQUdTID0gW1xuICAnYnJhbmNoJywgJ2RpZmYnLCAnc2hvd0JyYW5jaCcsICdzdGF0dXMnLCAndWknLFxuXS5yZWR1Y2UoKGFjYywgdHlwZSkgPT4ge1xuICBhY2MudW5zaGlmdCgnLWMnLCBgY29sb3IuJHt0eXBlfT1mYWxzZWApO1xuICByZXR1cm4gYWNjO1xufSwgW10pO1xuXG4vKipcbiAqIEV4cGFuZCBjb25maWcgcGF0aCBuYW1lIHBlclxuICogaHR0cHM6Ly9naXQtc2NtLmNvbS9kb2NzL2dpdC1jb25maWcjZ2l0LWNvbmZpZy1wYXRobmFtZVxuICogdGhpcyByZWdleCBhdHRlbXB0cyB0byBnZXQgdGhlIHNwZWNpZmllZCB1c2VyJ3MgaG9tZSBkaXJlY3RvcnlcbiAqIEV4OiBvbiBNYWMgfmt1eWNoYWNvLyBpcyBleHBhbmRlZCB0byB0aGUgc3BlY2lmaWVkIHVzZXLigJlzIGhvbWUgZGlyZWN0b3J5ICgvVXNlcnMva3V5Y2hhY28pXG4gKiBSZWdleCB0cmFuc2xhdGlvbjpcbiAqIF5+IGxpbmUgc3RhcnRzIHdpdGggdGlsZGVcbiAqIChbXlxcXFxcXFxcL10qKVtcXFxcXFxcXC9dIGNhcHR1cmVzIG5vbi1zbGFzaCBjaGFyYWN0ZXJzIGJlZm9yZSBmaXJzdCBzbGFzaFxuICovXG5jb25zdCBFWFBBTkRfVElMREVfUkVHRVggPSBuZXcgUmVnRXhwKCdefihbXlxcXFxcXFxcL10qKVtcXFxcXFxcXC9dJyk7XG5cbmV4cG9ydCBkZWZhdWx0IGNsYXNzIEdpdFNoZWxsT3V0U3RyYXRlZ3kge1xuICBzdGF0aWMgZGVmYXVsdEV4ZWNBcmdzID0ge1xuICAgIHN0ZGluOiBudWxsLFxuICAgIHVzZUdpdFByb21wdFNlcnZlcjogZmFsc2UsXG4gICAgdXNlR3BnV3JhcHBlcjogZmFsc2UsXG4gICAgdXNlR3BnQXRvbVByb21wdDogZmFsc2UsXG4gICAgd3JpdGVPcGVyYXRpb246IGZhbHNlLFxuICB9XG5cbiAgY29uc3RydWN0b3Iod29ya2luZ0Rpciwgb3B0aW9ucyA9IHt9KSB7XG4gICAgdGhpcy53b3JraW5nRGlyID0gd29ya2luZ0RpcjtcbiAgICBpZiAob3B0aW9ucy5xdWV1ZSkge1xuICAgICAgdGhpcy5jb21tYW5kUXVldWUgPSBvcHRpb25zLnF1ZXVlO1xuICAgIH0gZWxzZSB7XG4gICAgICBjb25zdCBwYXJhbGxlbGlzbSA9IG9wdGlvbnMucGFyYWxsZWxpc20gfHwgTWF0aC5tYXgoMywgb3MuY3B1cygpLmxlbmd0aCk7XG4gICAgICB0aGlzLmNvbW1hbmRRdWV1ZSA9IG5ldyBBc3luY1F1ZXVlKHtwYXJhbGxlbGlzbX0pO1xuICAgIH1cblxuICAgIHRoaXMucHJvbXB0ID0gb3B0aW9ucy5wcm9tcHQgfHwgKHF1ZXJ5ID0+IFByb21pc2UucmVqZWN0KCkpO1xuICAgIHRoaXMud29ya2VyTWFuYWdlciA9IG9wdGlvbnMud29ya2VyTWFuYWdlcjtcblxuICAgIGlmIChoZWFkbGVzcyA9PT0gbnVsbCkge1xuICAgICAgaGVhZGxlc3MgPSAhcmVtb3RlLmdldEN1cnJlbnRXaW5kb3coKS5pc1Zpc2libGUoKTtcbiAgICB9XG4gIH1cblxuICAvKlxuICAgKiBQcm92aWRlIGFuIGFzeW5jaHJvbm91cyBjYWxsYmFjayB0byBiZSB1c2VkIHRvIHJlcXVlc3QgaW5wdXQgZnJvbSB0aGUgdXNlciBmb3IgZ2l0IG9wZXJhdGlvbnMuXG4gICAqXG4gICAqIGBwcm9tcHRgIG11c3QgYmUgYSBjYWxsYWJsZSB0aGF0IGFjY2VwdHMgYSBxdWVyeSBvYmplY3QgYHtwcm9tcHQsIGluY2x1ZGVVc2VybmFtZX1gIGFuZCByZXR1cm5zIGEgUHJvbWlzZVxuICAgKiB0aGF0IGVpdGhlciByZXNvbHZlcyB3aXRoIGEgcmVzdWx0IG9iamVjdCBge1t1c2VybmFtZV0sIHBhc3N3b3JkfWAgb3IgcmVqZWN0cyBvbiBjYW5jZWxsYXRpb24uXG4gICAqL1xuICBzZXRQcm9tcHRDYWxsYmFjayhwcm9tcHQpIHtcbiAgICB0aGlzLnByb21wdCA9IHByb21wdDtcbiAgfVxuXG4gIC8vIEV4ZWN1dGUgYSBjb21tYW5kIGFuZCByZWFkIHRoZSBvdXRwdXQgdXNpbmcgdGhlIGVtYmVkZGVkIEdpdCBlbnZpcm9ubWVudFxuICBhc3luYyBleGVjKGFyZ3MsIG9wdGlvbnMgPSBHaXRTaGVsbE91dFN0cmF0ZWd5LmRlZmF1bHRFeGVjQXJncykge1xuICAgIC8qIGVzbGludC1kaXNhYmxlIG5vLWNvbnNvbGUsbm8tY29udHJvbC1yZWdleCAqL1xuICAgIGNvbnN0IHtzdGRpbiwgdXNlR2l0UHJvbXB0U2VydmVyLCB1c2VHcGdXcmFwcGVyLCB1c2VHcGdBdG9tUHJvbXB0LCB3cml0ZU9wZXJhdGlvbn0gPSBvcHRpb25zO1xuICAgIGNvbnN0IGNvbW1hbmROYW1lID0gYXJnc1swXTtcbiAgICBjb25zdCBzdWJzY3JpcHRpb25zID0gbmV3IENvbXBvc2l0ZURpc3Bvc2FibGUoKTtcbiAgICBjb25zdCBkaWFnbm9zdGljc0VuYWJsZWQgPSBwcm9jZXNzLmVudi5BVE9NX0dJVEhVQl9HSVRfRElBR05PU1RJQ1MgfHwgYXRvbS5jb25maWcuZ2V0KCdnaXRodWIuZ2l0RGlhZ25vc3RpY3MnKTtcblxuICAgIGNvbnN0IGZvcm1hdHRlZEFyZ3MgPSBgZ2l0ICR7YXJncy5qb2luKCcgJyl9IGluICR7dGhpcy53b3JraW5nRGlyfWA7XG4gICAgY29uc3QgdGltaW5nTWFya2VyID0gR2l0VGltaW5nc1ZpZXcuZ2VuZXJhdGVNYXJrZXIoYGdpdCAke2FyZ3Muam9pbignICcpfWApO1xuICAgIHRpbWluZ01hcmtlci5tYXJrKCdxdWV1ZWQnKTtcblxuICAgIGFyZ3MudW5zaGlmdCguLi5ESVNBQkxFX0NPTE9SX0ZMQUdTKTtcblxuICAgIGlmIChleGVjUGF0aFByb21pc2UgPT09IG51bGwpIHtcbiAgICAgIC8vIEF0dGVtcHQgdG8gY29sbGVjdCB0aGUgLS1leGVjLXBhdGggZnJvbSBhIG5hdGl2ZSBnaXQgaW5zdGFsbGF0aW9uLlxuICAgICAgZXhlY1BhdGhQcm9taXNlID0gbmV3IFByb21pc2UoKHJlc29sdmUsIHJlamVjdCkgPT4ge1xuICAgICAgICBjaGlsZFByb2Nlc3MuZXhlYygnZ2l0IC0tZXhlYy1wYXRoJywgKGVycm9yLCBzdGRvdXQsIHN0ZGVycikgPT4ge1xuICAgICAgICAgIC8qIGlzdGFuYnVsIGlnbm9yZSBpZiAqL1xuICAgICAgICAgIGlmIChlcnJvcikge1xuICAgICAgICAgICAgLy8gT2ggd2VsbFxuICAgICAgICAgICAgcmVzb2x2ZShudWxsKTtcbiAgICAgICAgICAgIHJldHVybjtcbiAgICAgICAgICB9XG5cbiAgICAgICAgICByZXNvbHZlKHN0ZG91dC50cmltKCkpO1xuICAgICAgICB9KTtcbiAgICAgIH0pO1xuICAgIH1cbiAgICBjb25zdCBleGVjUGF0aCA9IGF3YWl0IGV4ZWNQYXRoUHJvbWlzZTtcblxuICAgIHJldHVybiB0aGlzLmNvbW1hbmRRdWV1ZS5wdXNoKGFzeW5jICgpID0+IHtcbiAgICAgIHRpbWluZ01hcmtlci5tYXJrKCdwcmVwYXJlJyk7XG4gICAgICBsZXQgZ2l0UHJvbXB0U2VydmVyO1xuXG4gICAgICBjb25zdCBwYXRoUGFydHMgPSBbXTtcbiAgICAgIGlmIChwcm9jZXNzLmVudi5QQVRIKSB7XG4gICAgICAgIHBhdGhQYXJ0cy5wdXNoKHByb2Nlc3MuZW52LlBBVEgpO1xuICAgICAgfVxuICAgICAgaWYgKGV4ZWNQYXRoKSB7XG4gICAgICAgIHBhdGhQYXJ0cy5wdXNoKGV4ZWNQYXRoKTtcbiAgICAgIH1cblxuICAgICAgY29uc3QgZW52ID0ge1xuICAgICAgICAuLi5wcm9jZXNzLmVudixcbiAgICAgICAgR0lUX1RFUk1JTkFMX1BST01QVDogJzAnLFxuICAgICAgICBHSVRfT1BUSU9OQUxfTE9DS1M6ICcwJyxcbiAgICAgICAgUEFUSDogcGF0aFBhcnRzLmpvaW4ocGF0aC5kZWxpbWl0ZXIpLFxuICAgICAgfTtcblxuICAgICAgY29uc3QgZ2l0VGVtcERpciA9IG5ldyBHaXRUZW1wRGlyKCk7XG5cbiAgICAgIGlmICh1c2VHcGdXcmFwcGVyKSB7XG4gICAgICAgIGF3YWl0IGdpdFRlbXBEaXIuZW5zdXJlKCk7XG4gICAgICAgIGFyZ3MudW5zaGlmdCgnLWMnLCBgZ3BnLnByb2dyYW09JHtnaXRUZW1wRGlyLmdldEdwZ1dyYXBwZXJTaCgpfWApO1xuICAgICAgfVxuXG4gICAgICBpZiAodXNlR2l0UHJvbXB0U2VydmVyKSB7XG4gICAgICAgIGdpdFByb21wdFNlcnZlciA9IG5ldyBHaXRQcm9tcHRTZXJ2ZXIoZ2l0VGVtcERpcik7XG4gICAgICAgIGF3YWl0IGdpdFByb21wdFNlcnZlci5zdGFydCh0aGlzLnByb21wdCk7XG5cbiAgICAgICAgZW52LkFUT01fR0lUSFVCX1RNUCA9IGdpdFRlbXBEaXIuZ2V0Um9vdFBhdGgoKTtcbiAgICAgICAgZW52LkFUT01fR0lUSFVCX0FTS1BBU1NfUEFUSCA9IG5vcm1hbGl6ZUdpdEhlbHBlclBhdGgoZ2l0VGVtcERpci5nZXRBc2tQYXNzSnMoKSk7XG4gICAgICAgIGVudi5BVE9NX0dJVEhVQl9DUkVERU5USUFMX1BBVEggPSBub3JtYWxpemVHaXRIZWxwZXJQYXRoKGdpdFRlbXBEaXIuZ2V0Q3JlZGVudGlhbEhlbHBlckpzKCkpO1xuICAgICAgICBlbnYuQVRPTV9HSVRIVUJfRUxFQ1RST05fUEFUSCA9IG5vcm1hbGl6ZUdpdEhlbHBlclBhdGgoZ2V0QXRvbUhlbHBlclBhdGgoKSk7XG4gICAgICAgIGVudi5BVE9NX0dJVEhVQl9TT0NLX1BBVEggPSBub3JtYWxpemVHaXRIZWxwZXJQYXRoKGdpdFRlbXBEaXIuZ2V0U29ja2V0UGF0aCgpKTtcblxuICAgICAgICBlbnYuQVRPTV9HSVRIVUJfV09SS0RJUl9QQVRIID0gdGhpcy53b3JraW5nRGlyO1xuICAgICAgICBlbnYuQVRPTV9HSVRIVUJfRFVHSVRFX1BBVEggPSBnZXREdWdpdGVQYXRoKCk7XG4gICAgICAgIGVudi5BVE9NX0dJVEhVQl9LRVlUQVJfU1RSQVRFR1lfUEFUSCA9IGdldFNoYXJlZE1vZHVsZVBhdGgoJ2tleXRhci1zdHJhdGVneScpO1xuXG4gICAgICAgIC8vIFwic3NoXCIgd29uJ3QgcmVzcGVjdCBTU0hfQVNLUEFTUyB1bmxlc3M6XG4gICAgICAgIC8vIChhKSBpdCdzIHJ1bm5pbmcgd2l0aG91dCBhIHR0eVxuICAgICAgICAvLyAoYikgRElTUExBWSBpcyBzZXQgdG8gc29tZXRoaW5nIG5vbmVtcHR5XG4gICAgICAgIC8vIEJ1dCwgb24gYSBNYWMsIERJU1BMQVkgaXMgdW5zZXQuIEVuc3VyZSB0aGF0IGl0IGlzIHNvIG91ciBTU0hfQVNLUEFTUyBpcyByZXNwZWN0ZWQuXG4gICAgICAgIGlmICghcHJvY2Vzcy5lbnYuRElTUExBWSB8fCBwcm9jZXNzLmVudi5ESVNQTEFZLmxlbmd0aCA9PT0gMCkge1xuICAgICAgICAgIGVudi5ESVNQTEFZID0gJ2F0b20tZ2l0aHViLXBsYWNlaG9sZGVyJztcbiAgICAgICAgfVxuXG4gICAgICAgIGVudi5BVE9NX0dJVEhVQl9PUklHSU5BTF9QQVRIID0gcHJvY2Vzcy5lbnYuUEFUSCB8fCAnJztcbiAgICAgICAgZW52LkFUT01fR0lUSFVCX09SSUdJTkFMX0dJVF9BU0tQQVNTID0gcHJvY2Vzcy5lbnYuR0lUX0FTS1BBU1MgfHwgJyc7XG4gICAgICAgIGVudi5BVE9NX0dJVEhVQl9PUklHSU5BTF9TU0hfQVNLUEFTUyA9IHByb2Nlc3MuZW52LlNTSF9BU0tQQVNTIHx8ICcnO1xuICAgICAgICBlbnYuQVRPTV9HSVRIVUJfT1JJR0lOQUxfR0lUX1NTSF9DT01NQU5EID0gcHJvY2Vzcy5lbnYuR0lUX1NTSF9DT01NQU5EIHx8ICcnO1xuICAgICAgICBlbnYuQVRPTV9HSVRIVUJfU1BFQ19NT0RFID0gYXRvbS5pblNwZWNNb2RlKCkgPyAndHJ1ZScgOiAnZmFsc2UnO1xuXG4gICAgICAgIGVudi5TU0hfQVNLUEFTUyA9IG5vcm1hbGl6ZUdpdEhlbHBlclBhdGgoZ2l0VGVtcERpci5nZXRBc2tQYXNzU2goKSk7XG4gICAgICAgIGVudi5HSVRfQVNLUEFTUyA9IG5vcm1hbGl6ZUdpdEhlbHBlclBhdGgoZ2l0VGVtcERpci5nZXRBc2tQYXNzU2goKSk7XG5cbiAgICAgICAgaWYgKHByb2Nlc3MucGxhdGZvcm0gPT09ICdsaW51eCcpIHtcbiAgICAgICAgICBlbnYuR0lUX1NTSF9DT01NQU5EID0gZ2l0VGVtcERpci5nZXRTc2hXcmFwcGVyU2goKTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICBlbnYuR0lUX1NTSF9DT01NQU5EID0gcHJvY2Vzcy5lbnYuR0lUX1NTSF9DT01NQU5EO1xuICAgICAgICB9XG5cbiAgICAgICAgY29uc3QgY3JlZGVudGlhbEhlbHBlclNoID0gbm9ybWFsaXplR2l0SGVscGVyUGF0aChnaXRUZW1wRGlyLmdldENyZWRlbnRpYWxIZWxwZXJTaCgpKTtcbiAgICAgICAgYXJncy51bnNoaWZ0KCctYycsIGBjcmVkZW50aWFsLmhlbHBlcj0ke2NyZWRlbnRpYWxIZWxwZXJTaH1gKTtcbiAgICAgIH1cblxuICAgICAgaWYgKHVzZUdwZ1dyYXBwZXIgJiYgdXNlR2l0UHJvbXB0U2VydmVyICYmIHVzZUdwZ0F0b21Qcm9tcHQpIHtcbiAgICAgICAgZW52LkFUT01fR0lUSFVCX0dQR19QUk9NUFQgPSAndHJ1ZSc7XG4gICAgICB9XG5cbiAgICAgIC8qIGlzdGFuYnVsIGlnbm9yZSBpZiAqL1xuICAgICAgaWYgKGRpYWdub3N0aWNzRW5hYmxlZCkge1xuICAgICAgICBlbnYuR0lUX1RSQUNFID0gJ3RydWUnO1xuICAgICAgICBlbnYuR0lUX1RSQUNFX0NVUkwgPSAndHJ1ZSc7XG4gICAgICB9XG5cbiAgICAgIGxldCBvcHRzID0ge2Vudn07XG5cbiAgICAgIGlmIChzdGRpbikge1xuICAgICAgICBvcHRzLnN0ZGluID0gc3RkaW47XG4gICAgICAgIG9wdHMuc3RkaW5FbmNvZGluZyA9ICd1dGY4JztcbiAgICAgIH1cblxuICAgICAgLyogaXN0YW5idWwgaWdub3JlIGlmICovXG4gICAgICBpZiAocHJvY2Vzcy5lbnYuUFJJTlRfR0lUX1RJTUVTKSB7XG4gICAgICAgIGNvbnNvbGUudGltZShgZ2l0OiR7Zm9ybWF0dGVkQXJnc31gKTtcbiAgICAgIH1cblxuICAgICAgcmV0dXJuIG5ldyBQcm9taXNlKGFzeW5jIChyZXNvbHZlLCByZWplY3QpID0+IHtcbiAgICAgICAgaWYgKG9wdGlvbnMuYmVmb3JlUnVuKSB7XG4gICAgICAgICAgY29uc3QgbmV3QXJnc09wdHMgPSBhd2FpdCBvcHRpb25zLmJlZm9yZVJ1bih7YXJncywgb3B0c30pO1xuICAgICAgICAgIGFyZ3MgPSBuZXdBcmdzT3B0cy5hcmdzO1xuICAgICAgICAgIG9wdHMgPSBuZXdBcmdzT3B0cy5vcHRzO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IHtwcm9taXNlLCBjYW5jZWx9ID0gdGhpcy5leGVjdXRlR2l0Q29tbWFuZChhcmdzLCBvcHRzLCB0aW1pbmdNYXJrZXIpO1xuICAgICAgICBsZXQgZXhwZWN0Q2FuY2VsID0gZmFsc2U7XG4gICAgICAgIGlmIChnaXRQcm9tcHRTZXJ2ZXIpIHtcbiAgICAgICAgICBzdWJzY3JpcHRpb25zLmFkZChnaXRQcm9tcHRTZXJ2ZXIub25EaWRDYW5jZWwoYXN5bmMgKHtoYW5kbGVyUGlkfSkgPT4ge1xuICAgICAgICAgICAgZXhwZWN0Q2FuY2VsID0gdHJ1ZTtcbiAgICAgICAgICAgIGF3YWl0IGNhbmNlbCgpO1xuXG4gICAgICAgICAgICAvLyBPbiBXaW5kb3dzLCB0aGUgU1NIX0FTS1BBU1MgaGFuZGxlciBpcyBleGVjdXRlZCBhcyBhIG5vbi1jaGlsZCBwcm9jZXNzLCBzbyB0aGUgYmluXFxnaXQtYXNrcGFzcy1hdG9tLnNoXG4gICAgICAgICAgICAvLyBwcm9jZXNzIGRvZXMgbm90IHRlcm1pbmF0ZSB3aGVuIHRoZSBnaXQgcHJvY2VzcyBpcyBraWxsZWQuXG4gICAgICAgICAgICAvLyBLaWxsIHRoZSBoYW5kbGVyIHByb2Nlc3MgKmFmdGVyKiB0aGUgZ2l0IHByb2Nlc3MgaGFzIGJlZW4ga2lsbGVkIHRvIGVuc3VyZSB0aGF0IGdpdCBkb2Vzbid0IGhhdmUgYVxuICAgICAgICAgICAgLy8gY2hhbmNlIHRvIGZhbGwgYmFjayB0byBHSVRfQVNLUEFTUyBmcm9tIHRoZSBjcmVkZW50aWFsIGhhbmRsZXIuXG4gICAgICAgICAgICBhd2FpdCBuZXcgUHJvbWlzZSgocmVzb2x2ZUtpbGwsIHJlamVjdEtpbGwpID0+IHtcbiAgICAgICAgICAgICAgcmVxdWlyZSgndHJlZS1raWxsJykoaGFuZGxlclBpZCwgJ1NJR1RFUk0nLCBlcnIgPT4ge1xuICAgICAgICAgICAgICAgIC8qIGlzdGFuYnVsIGlnbm9yZSBpZiAqL1xuICAgICAgICAgICAgICAgIGlmIChlcnIpIHsgcmVqZWN0S2lsbChlcnIpOyB9IGVsc2UgeyByZXNvbHZlS2lsbCgpOyB9XG4gICAgICAgICAgICAgIH0pO1xuICAgICAgICAgICAgfSk7XG4gICAgICAgICAgfSkpO1xuICAgICAgICB9XG5cbiAgICAgICAgY29uc3Qge3N0ZG91dCwgc3RkZXJyLCBleGl0Q29kZSwgc2lnbmFsLCB0aW1pbmd9ID0gYXdhaXQgcHJvbWlzZS5jYXRjaChlcnIgPT4ge1xuICAgICAgICAgIGlmIChlcnIuc2lnbmFsKSB7XG4gICAgICAgICAgICByZXR1cm4ge3NpZ25hbDogZXJyLnNpZ25hbH07XG4gICAgICAgICAgfVxuICAgICAgICAgIHJlamVjdChlcnIpO1xuICAgICAgICAgIHJldHVybiB7fTtcbiAgICAgICAgfSk7XG5cbiAgICAgICAgaWYgKHRpbWluZykge1xuICAgICAgICAgIGNvbnN0IHtleGVjVGltZSwgc3Bhd25UaW1lLCBpcGNUaW1lfSA9IHRpbWluZztcbiAgICAgICAgICBjb25zdCBub3cgPSBwZXJmb3JtYW5jZS5ub3coKTtcbiAgICAgICAgICB0aW1pbmdNYXJrZXIubWFyaygnbmV4dHRpY2snLCBub3cgLSBleGVjVGltZSAtIHNwYXduVGltZSAtIGlwY1RpbWUpO1xuICAgICAgICAgIHRpbWluZ01hcmtlci5tYXJrKCdleGVjdXRlJywgbm93IC0gZXhlY1RpbWUgLSBpcGNUaW1lKTtcbiAgICAgICAgICB0aW1pbmdNYXJrZXIubWFyaygnaXBjJywgbm93IC0gaXBjVGltZSk7XG4gICAgICAgIH1cbiAgICAgICAgdGltaW5nTWFya2VyLmZpbmFsaXplKCk7XG5cbiAgICAgICAgLyogaXN0YW5idWwgaWdub3JlIGlmICovXG4gICAgICAgIGlmIChwcm9jZXNzLmVudi5QUklOVF9HSVRfVElNRVMpIHtcbiAgICAgICAgICBjb25zb2xlLnRpbWVFbmQoYGdpdDoke2Zvcm1hdHRlZEFyZ3N9YCk7XG4gICAgICAgIH1cblxuICAgICAgICBpZiAoZ2l0UHJvbXB0U2VydmVyKSB7XG4gICAgICAgICAgZ2l0UHJvbXB0U2VydmVyLnRlcm1pbmF0ZSgpO1xuICAgICAgICB9XG4gICAgICAgIHN1YnNjcmlwdGlvbnMuZGlzcG9zZSgpO1xuXG4gICAgICAgIC8qIGlzdGFuYnVsIGlnbm9yZSBpZiAqL1xuICAgICAgICBpZiAoZGlhZ25vc3RpY3NFbmFibGVkKSB7XG4gICAgICAgICAgY29uc3QgZXhwb3NlQ29udHJvbENoYXJhY3RlcnMgPSByYXcgPT4ge1xuICAgICAgICAgICAgaWYgKCFyYXcpIHsgcmV0dXJuICcnOyB9XG5cbiAgICAgICAgICAgIHJldHVybiByYXdcbiAgICAgICAgICAgICAgLnJlcGxhY2UoL1xcdTAwMDAvdWcsICc8TlVMPlxcbicpXG4gICAgICAgICAgICAgIC5yZXBsYWNlKC9cXHUwMDFGL3VnLCAnPFNFUD4nKTtcbiAgICAgICAgICB9O1xuXG4gICAgICAgICAgaWYgKGhlYWRsZXNzKSB7XG4gICAgICAgICAgICBsZXQgc3VtbWFyeSA9IGBnaXQ6JHtmb3JtYXR0ZWRBcmdzfVxcbmA7XG4gICAgICAgICAgICBpZiAoZXhpdENvZGUgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgICAgICBzdW1tYXJ5ICs9IGBleGl0IHN0YXR1czogJHtleGl0Q29kZX1cXG5gO1xuICAgICAgICAgICAgfSBlbHNlIGlmIChzaWduYWwpIHtcbiAgICAgICAgICAgICAgc3VtbWFyeSArPSBgZXhpdCBzaWduYWw6ICR7c2lnbmFsfVxcbmA7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBpZiAoc3RkaW4gJiYgc3RkaW4ubGVuZ3RoICE9PSAwKSB7XG4gICAgICAgICAgICAgIHN1bW1hcnkgKz0gYHN0ZGluOlxcbiR7ZXhwb3NlQ29udHJvbENoYXJhY3RlcnMoc3RkaW4pfVxcbmA7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBzdW1tYXJ5ICs9ICdzdGRvdXQ6JztcbiAgICAgICAgICAgIGlmIChzdGRvdXQubGVuZ3RoID09PSAwKSB7XG4gICAgICAgICAgICAgIHN1bW1hcnkgKz0gJyA8ZW1wdHk+XFxuJztcbiAgICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICAgIHN1bW1hcnkgKz0gYFxcbiR7ZXhwb3NlQ29udHJvbENoYXJhY3RlcnMoc3Rkb3V0KX1cXG5gO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgc3VtbWFyeSArPSAnc3RkZXJyOic7XG4gICAgICAgICAgICBpZiAoc3RkZXJyLmxlbmd0aCA9PT0gMCkge1xuICAgICAgICAgICAgICBzdW1tYXJ5ICs9ICcgPGVtcHR5Plxcbic7XG4gICAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgICBzdW1tYXJ5ICs9IGBcXG4ke2V4cG9zZUNvbnRyb2xDaGFyYWN0ZXJzKHN0ZGVycil9XFxuYDtcbiAgICAgICAgICAgIH1cblxuICAgICAgICAgICAgY29uc29sZS5sb2coc3VtbWFyeSk7XG4gICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIGNvbnN0IGhlYWRlclN0eWxlID0gJ2ZvbnQtd2VpZ2h0OiBib2xkOyBjb2xvcjogYmx1ZTsnO1xuXG4gICAgICAgICAgICBjb25zb2xlLmdyb3VwQ29sbGFwc2VkKGBnaXQ6JHtmb3JtYXR0ZWRBcmdzfWApO1xuICAgICAgICAgICAgaWYgKGV4aXRDb2RlICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICAgICAgY29uc29sZS5sb2coJyVjZXhpdCBzdGF0dXMlYyAlZCcsIGhlYWRlclN0eWxlLCAnZm9udC13ZWlnaHQ6IG5vcm1hbDsgY29sb3I6IGJsYWNrOycsIGV4aXRDb2RlKTtcbiAgICAgICAgICAgIH0gZWxzZSBpZiAoc2lnbmFsKSB7XG4gICAgICAgICAgICAgIGNvbnNvbGUubG9nKCclY2V4aXQgc2lnbmFsJWMgJXMnLCBoZWFkZXJTdHlsZSwgJ2ZvbnQtd2VpZ2h0OiBub3JtYWw7IGNvbG9yOiBibGFjazsnLCBzaWduYWwpO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgY29uc29sZS5sb2coXG4gICAgICAgICAgICAgICclY2Z1bGwgYXJndW1lbnRzJWMgJXMnLFxuICAgICAgICAgICAgICBoZWFkZXJTdHlsZSwgJ2ZvbnQtd2VpZ2h0OiBub3JtYWw7IGNvbG9yOiBibGFjazsnLFxuICAgICAgICAgICAgICB1dGlsLmluc3BlY3QoYXJncywge2JyZWFrTGVuZ3RoOiBJbmZpbml0eX0pLFxuICAgICAgICAgICAgKTtcbiAgICAgICAgICAgIGlmIChzdGRpbiAmJiBzdGRpbi5sZW5ndGggIT09IDApIHtcbiAgICAgICAgICAgICAgY29uc29sZS5sb2coJyVjc3RkaW4nLCBoZWFkZXJTdHlsZSk7XG4gICAgICAgICAgICAgIGNvbnNvbGUubG9nKGV4cG9zZUNvbnRyb2xDaGFyYWN0ZXJzKHN0ZGluKSk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBjb25zb2xlLmxvZygnJWNzdGRvdXQnLCBoZWFkZXJTdHlsZSk7XG4gICAgICAgICAgICBjb25zb2xlLmxvZyhleHBvc2VDb250cm9sQ2hhcmFjdGVycyhzdGRvdXQpKTtcbiAgICAgICAgICAgIGNvbnNvbGUubG9nKCclY3N0ZGVycicsIGhlYWRlclN0eWxlKTtcbiAgICAgICAgICAgIGNvbnNvbGUubG9nKGV4cG9zZUNvbnRyb2xDaGFyYWN0ZXJzKHN0ZGVycikpO1xuICAgICAgICAgICAgY29uc29sZS5ncm91cEVuZCgpO1xuICAgICAgICAgIH1cbiAgICAgICAgfVxuXG4gICAgICAgIGlmIChleGl0Q29kZSAhPT0gMCAmJiAhZXhwZWN0Q2FuY2VsKSB7XG4gICAgICAgICAgY29uc3QgZXJyID0gbmV3IEdpdEVycm9yKFxuICAgICAgICAgICAgYCR7Zm9ybWF0dGVkQXJnc30gZXhpdGVkIHdpdGggY29kZSAke2V4aXRDb2RlfVxcbnN0ZG91dDogJHtzdGRvdXR9XFxuc3RkZXJyOiAke3N0ZGVycn1gLFxuICAgICAgICAgICk7XG4gICAgICAgICAgZXJyLmNvZGUgPSBleGl0Q29kZTtcbiAgICAgICAgICBlcnIuc3RkRXJyID0gc3RkZXJyO1xuICAgICAgICAgIGVyci5zdGRPdXQgPSBzdGRvdXQ7XG4gICAgICAgICAgZXJyLmNvbW1hbmQgPSBmb3JtYXR0ZWRBcmdzO1xuICAgICAgICAgIHJlamVjdChlcnIpO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKCFJR05PUkVEX0dJVF9DT01NQU5EUy5pbmNsdWRlcyhjb21tYW5kTmFtZSkpIHtcbiAgICAgICAgICBpbmNyZW1lbnRDb3VudGVyKGNvbW1hbmROYW1lKTtcbiAgICAgICAgfVxuICAgICAgICByZXNvbHZlKHN0ZG91dCk7XG4gICAgICB9KTtcbiAgICB9LCB7cGFyYWxsZWw6ICF3cml0ZU9wZXJhdGlvbn0pO1xuICAgIC8qIGVzbGludC1lbmFibGUgbm8tY29uc29sZSxuby1jb250cm9sLXJlZ2V4ICovXG4gIH1cblxuICBhc3luYyBncGdFeGVjKGFyZ3MsIG9wdGlvbnMpIHtcbiAgICB0cnkge1xuICAgICAgcmV0dXJuIGF3YWl0IHRoaXMuZXhlYyhhcmdzLnNsaWNlKCksIHtcbiAgICAgICAgdXNlR3BnV3JhcHBlcjogdHJ1ZSxcbiAgICAgICAgdXNlR3BnQXRvbVByb21wdDogZmFsc2UsXG4gICAgICAgIC4uLm9wdGlvbnMsXG4gICAgICB9KTtcbiAgICB9IGNhdGNoIChlKSB7XG4gICAgICBpZiAoL2dwZyBmYWlsZWQvLnRlc3QoZS5zdGRFcnIpKSB7XG4gICAgICAgIHJldHVybiBhd2FpdCB0aGlzLmV4ZWMoYXJncywge1xuICAgICAgICAgIHVzZUdpdFByb21wdFNlcnZlcjogdHJ1ZSxcbiAgICAgICAgICB1c2VHcGdXcmFwcGVyOiB0cnVlLFxuICAgICAgICAgIHVzZUdwZ0F0b21Qcm9tcHQ6IHRydWUsXG4gICAgICAgICAgLi4ub3B0aW9ucyxcbiAgICAgICAgfSk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICB0aHJvdyBlO1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIGV4ZWN1dGVHaXRDb21tYW5kKGFyZ3MsIG9wdGlvbnMsIG1hcmtlciA9IG51bGwpIHtcbiAgICBpZiAocHJvY2Vzcy5lbnYuQVRPTV9HSVRIVUJfSU5MSU5FX0dJVF9FWEVDIHx8ICFXb3JrZXJNYW5hZ2VyLmdldEluc3RhbmNlKCkuaXNSZWFkeSgpKSB7XG4gICAgICBtYXJrZXIgJiYgbWFya2VyLm1hcmsoJ25leHR0aWNrJyk7XG5cbiAgICAgIGxldCBjaGlsZFBpZDtcbiAgICAgIG9wdGlvbnMucHJvY2Vzc0NhbGxiYWNrID0gY2hpbGQgPT4ge1xuICAgICAgICBjaGlsZFBpZCA9IGNoaWxkLnBpZDtcblxuICAgICAgICAvKiBpc3RhbmJ1bCBpZ25vcmUgbmV4dCAqL1xuICAgICAgICBjaGlsZC5zdGRpbi5vbignZXJyb3InLCBlcnIgPT4ge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICAgIGBFcnJvciB3cml0aW5nIHRvIHN0ZGluOiBnaXQgJHthcmdzLmpvaW4oJyAnKX0gaW4gJHt0aGlzLndvcmtpbmdEaXJ9XFxuJHtvcHRpb25zLnN0ZGlufVxcbiR7ZXJyfWApO1xuICAgICAgICB9KTtcbiAgICAgIH07XG5cbiAgICAgIGNvbnN0IHByb21pc2UgPSBHaXRQcm9jZXNzLmV4ZWMoYXJncywgdGhpcy53b3JraW5nRGlyLCBvcHRpb25zKTtcbiAgICAgIG1hcmtlciAmJiBtYXJrZXIubWFyaygnZXhlY3V0ZScpO1xuICAgICAgcmV0dXJuIHtcbiAgICAgICAgcHJvbWlzZSxcbiAgICAgICAgY2FuY2VsOiAoKSA9PiB7XG4gICAgICAgICAgLyogaXN0YW5idWwgaWdub3JlIGlmICovXG4gICAgICAgICAgaWYgKCFjaGlsZFBpZCkge1xuICAgICAgICAgICAgcmV0dXJuIFByb21pc2UucmVzb2x2ZSgpO1xuICAgICAgICAgIH1cblxuICAgICAgICAgIHJldHVybiBuZXcgUHJvbWlzZSgocmVzb2x2ZSwgcmVqZWN0KSA9PiB7XG4gICAgICAgICAgICByZXF1aXJlKCd0cmVlLWtpbGwnKShjaGlsZFBpZCwgJ1NJR1RFUk0nLCBlcnIgPT4ge1xuICAgICAgICAgICAgICAvKiBpc3RhbmJ1bCBpZ25vcmUgaWYgKi9cbiAgICAgICAgICAgICAgaWYgKGVycikgeyByZWplY3QoZXJyKTsgfSBlbHNlIHsgcmVzb2x2ZSgpOyB9XG4gICAgICAgICAgICB9KTtcbiAgICAgICAgICB9KTtcbiAgICAgICAgfSxcbiAgICAgIH07XG4gICAgfSBlbHNlIHtcbiAgICAgIGNvbnN0IHdvcmtlck1hbmFnZXIgPSB0aGlzLndvcmtlck1hbmFnZXIgfHwgV29ya2VyTWFuYWdlci5nZXRJbnN0YW5jZSgpO1xuICAgICAgcmV0dXJuIHdvcmtlck1hbmFnZXIucmVxdWVzdCh7XG4gICAgICAgIGFyZ3MsXG4gICAgICAgIHdvcmtpbmdEaXI6IHRoaXMud29ya2luZ0RpcixcbiAgICAgICAgb3B0aW9ucyxcbiAgICAgIH0pO1xuICAgIH1cbiAgfVxuXG4gIGFzeW5jIHJlc29sdmVEb3RHaXREaXIoKSB7XG4gICAgdHJ5IHtcbiAgICAgIGF3YWl0IGZzLnN0YXQodGhpcy53b3JraW5nRGlyKTsgLy8gZmFpbHMgaWYgZm9sZGVyIGRvZXNuJ3QgZXhpc3RcbiAgICAgIGNvbnN0IG91dHB1dCA9IGF3YWl0IHRoaXMuZXhlYyhbJ3Jldi1wYXJzZScsICctLXJlc29sdmUtZ2l0LWRpcicsIHBhdGguam9pbih0aGlzLndvcmtpbmdEaXIsICcuZ2l0JyldKTtcbiAgICAgIGNvbnN0IGRvdEdpdERpciA9IG91dHB1dC50cmltKCk7XG4gICAgICByZXR1cm4gdG9OYXRpdmVQYXRoU2VwKGRvdEdpdERpcik7XG4gICAgfSBjYXRjaCAoZSkge1xuICAgICAgcmV0dXJuIG51bGw7XG4gICAgfVxuICB9XG5cbiAgaW5pdCgpIHtcbiAgICByZXR1cm4gdGhpcy5leGVjKFsnaW5pdCcsIHRoaXMud29ya2luZ0Rpcl0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0YWdpbmcvVW5zdGFnaW5nIGZpbGVzIGFuZCBwYXRjaGVzIGFuZCBjb21taXR0aW5nXG4gICAqL1xuICBzdGFnZUZpbGVzKHBhdGhzKSB7XG4gICAgaWYgKHBhdGhzLmxlbmd0aCA9PT0gMCkgeyByZXR1cm4gUHJvbWlzZS5yZXNvbHZlKG51bGwpOyB9XG4gICAgY29uc3QgYXJncyA9IFsnYWRkJ10uY29uY2F0KHBhdGhzLm1hcCh0b0dpdFBhdGhTZXApKTtcbiAgICByZXR1cm4gdGhpcy5leGVjKGFyZ3MsIHt3cml0ZU9wZXJhdGlvbjogdHJ1ZX0pO1xuICB9XG5cbiAgYXN5bmMgZmV0Y2hDb21taXRNZXNzYWdlVGVtcGxhdGUoKSB7XG4gICAgbGV0IHRlbXBsYXRlUGF0aCA9IGF3YWl0IHRoaXMuZ2V0Q29uZmlnKCdjb21taXQudGVtcGxhdGUnKTtcbiAgICBpZiAoIXRlbXBsYXRlUGF0aCkge1xuICAgICAgcmV0dXJuIG51bGw7XG4gICAgfVxuXG4gICAgY29uc3QgaG9tZURpciA9IG9zLmhvbWVkaXIoKTtcblxuICAgIHRlbXBsYXRlUGF0aCA9IHRlbXBsYXRlUGF0aC50cmltKCkucmVwbGFjZShFWFBBTkRfVElMREVfUkVHRVgsIChfLCB1c2VyKSA9PiB7XG4gICAgICAvLyBpZiBubyB1c2VyIGlzIHNwZWNpZmllZCwgZmFsbCBiYWNrIHRvIHVzaW5nIHRoZSBob21lIGRpcmVjdG9yeS5cbiAgICAgIHJldHVybiBgJHt1c2VyID8gcGF0aC5qb2luKHBhdGguZGlybmFtZShob21lRGlyKSwgdXNlcikgOiBob21lRGlyfS9gO1xuICAgIH0pO1xuICAgIHRlbXBsYXRlUGF0aCA9IHRvTmF0aXZlUGF0aFNlcCh0ZW1wbGF0ZVBhdGgpO1xuXG4gICAgaWYgKCFwYXRoLmlzQWJzb2x1dGUodGVtcGxhdGVQYXRoKSkge1xuICAgICAgdGVtcGxhdGVQYXRoID0gcGF0aC5qb2luKHRoaXMud29ya2luZ0RpciwgdGVtcGxhdGVQYXRoKTtcbiAgICB9XG5cbiAgICBpZiAoIWF3YWl0IGZpbGVFeGlzdHModGVtcGxhdGVQYXRoKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBJbnZhbGlkIGNvbW1pdCB0ZW1wbGF0ZSBwYXRoIHNldCBpbiBHaXQgY29uZmlnOiAke3RlbXBsYXRlUGF0aH1gKTtcbiAgICB9XG4gICAgcmV0dXJuIGF3YWl0IGZzLnJlYWRGaWxlKHRlbXBsYXRlUGF0aCwge2VuY29kaW5nOiAndXRmOCd9KTtcbiAgfVxuXG4gIHVuc3RhZ2VGaWxlcyhwYXRocywgY29tbWl0ID0gJ0hFQUQnKSB7XG4gICAgaWYgKHBhdGhzLmxlbmd0aCA9PT0gMCkgeyByZXR1cm4gUHJvbWlzZS5yZXNvbHZlKG51bGwpOyB9XG4gICAgY29uc3QgYXJncyA9IFsncmVzZXQnLCBjb21taXQsICctLSddLmNvbmNhdChwYXRocy5tYXAodG9HaXRQYXRoU2VwKSk7XG4gICAgcmV0dXJuIHRoaXMuZXhlYyhhcmdzLCB7d3JpdGVPcGVyYXRpb246IHRydWV9KTtcbiAgfVxuXG4gIHN0YWdlRmlsZU1vZGVDaGFuZ2UoZmlsZW5hbWUsIG5ld01vZGUpIHtcbiAgICBjb25zdCBpbmRleFJlYWRQcm9taXNlID0gdGhpcy5leGVjKFsnbHMtZmlsZXMnLCAnLXMnLCAnLS0nLCBmaWxlbmFtZV0pO1xuICAgIHJldHVybiB0aGlzLmV4ZWMoWyd1cGRhdGUtaW5kZXgnLCAnLS1jYWNoZWluZm8nLCBgJHtuZXdNb2RlfSw8T0lEX1RCRD4sJHtmaWxlbmFtZX1gXSwge1xuICAgICAgd3JpdGVPcGVyYXRpb246IHRydWUsXG4gICAgICBiZWZvcmVSdW46IGFzeW5jIGZ1bmN0aW9uIGRldGVybWluZUFyZ3Moe2FyZ3MsIG9wdHN9KSB7XG4gICAgICAgIGNvbnN0IGluZGV4ID0gYXdhaXQgaW5kZXhSZWFkUHJvbWlzZTtcbiAgICAgICAgY29uc3Qgb2lkID0gaW5kZXguc3Vic3RyKDcsIDQwKTtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICBvcHRzLFxuICAgICAgICAgIGFyZ3M6IFsndXBkYXRlLWluZGV4JywgJy0tY2FjaGVpbmZvJywgYCR7bmV3TW9kZX0sJHtvaWR9LCR7ZmlsZW5hbWV9YF0sXG4gICAgICAgIH07XG4gICAgICB9LFxuICAgIH0pO1xuICB9XG5cbiAgc3RhZ2VGaWxlU3ltbGlua0NoYW5nZShmaWxlbmFtZSkge1xuICAgIHJldHVybiB0aGlzLmV4ZWMoWydybScsICctLWNhY2hlZCcsIGZpbGVuYW1lXSwge3dyaXRlT3BlcmF0aW9uOiB0cnVlfSk7XG4gIH1cblxuICBhcHBseVBhdGNoKHBhdGNoLCB7aW5kZXh9ID0ge30pIHtcbiAgICBjb25zdCBhcmdzID0gWydhcHBseScsICctJ107XG4gICAgaWYgKGluZGV4KSB7IGFyZ3Muc3BsaWNlKDEsIDAsICctLWNhY2hlZCcpOyB9XG4gICAgcmV0dXJuIHRoaXMuZXhlYyhhcmdzLCB7c3RkaW46IHBhdGNoLCB3cml0ZU9wZXJhdGlvbjogdHJ1ZX0pO1xuICB9XG5cbiAgYXN5bmMgY29tbWl0KHJhd01lc3NhZ2UsIHthbGxvd0VtcHR5LCBhbWVuZCwgY29BdXRob3JzLCB2ZXJiYXRpbX0gPSB7fSkge1xuICAgIGNvbnN0IGFyZ3MgPSBbJ2NvbW1pdCddO1xuICAgIGxldCBtc2c7XG5cbiAgICAvLyBpZiBhbWVuZGluZyBhbmQgbm8gbmV3IG1lc3NhZ2UgaXMgcGFzc2VkLCB1c2UgbGFzdCBjb21taXQncyBtZXNzYWdlLiBFbnN1cmUgdGhhdCB3ZSBkb24ndFxuICAgIC8vIG1hbmdsZSBpdCBpbiB0aGUgcHJvY2Vzcy5cbiAgICBpZiAoYW1lbmQgJiYgcmF3TWVzc2FnZS5sZW5ndGggPT09IDApIHtcbiAgICAgIGNvbnN0IHt1bmJvcm5SZWYsIG1lc3NhZ2VCb2R5LCBtZXNzYWdlU3ViamVjdH0gPSBhd2FpdCB0aGlzLmdldEhlYWRDb21taXQoKTtcbiAgICAgIGlmICh1bmJvcm5SZWYpIHtcbiAgICAgICAgbXNnID0gcmF3TWVzc2FnZTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIG1zZyA9IGAke21lc3NhZ2VTdWJqZWN0fVxcblxcbiR7bWVzc2FnZUJvZHl9YC50cmltKCk7XG4gICAgICAgIHZlcmJhdGltID0gdHJ1ZTtcbiAgICAgIH1cbiAgICB9IGVsc2Uge1xuICAgICAgbXNnID0gcmF3TWVzc2FnZTtcbiAgICB9XG5cbiAgICAvLyBpZiBjb21taXQgdGVtcGxhdGUgaXMgdXNlZCwgc3RyaXAgY29tbWVudGVkIGxpbmVzIGZyb20gY29tbWl0XG4gICAgLy8gdG8gYmUgY29uc2lzdGVudCB3aXRoIGNvbW1hbmQgbGluZSBnaXQuXG4gICAgY29uc3QgdGVtcGxhdGUgPSBhd2FpdCB0aGlzLmZldGNoQ29tbWl0TWVzc2FnZVRlbXBsYXRlKCk7XG4gICAgaWYgKHRlbXBsYXRlKSB7XG5cbiAgICAgIC8vIHJlc3BlY3RpbmcgdGhlIGNvbW1lbnQgY2hhcmFjdGVyIGZyb20gdXNlciBzZXR0aW5ncyBvciBmYWxsIGJhY2sgdG8gIyBhcyBkZWZhdWx0LlxuICAgICAgLy8gaHR0cHM6Ly9naXQtc2NtLmNvbS9kb2NzL2dpdC1jb25maWcjZ2l0LWNvbmZpZy1jb3JlY29tbWVudENoYXJcbiAgICAgIGxldCBjb21tZW50Q2hhciA9IGF3YWl0IHRoaXMuZ2V0Q29uZmlnKCdjb3JlLmNvbW1lbnRDaGFyJyk7XG4gICAgICBpZiAoIWNvbW1lbnRDaGFyKSB7XG4gICAgICAgIGNvbW1lbnRDaGFyID0gJyMnO1xuICAgICAgfVxuICAgICAgbXNnID0gbXNnLnNwbGl0KCdcXG4nKS5maWx0ZXIobGluZSA9PiAhbGluZS5zdGFydHNXaXRoKGNvbW1lbnRDaGFyKSkuam9pbignXFxuJyk7XG4gICAgfVxuXG4gICAgLy8gRGV0ZXJtaW5lIHRoZSBjbGVhbnVwIG1vZGUuXG4gICAgaWYgKHZlcmJhdGltKSB7XG4gICAgICBhcmdzLnB1c2goJy0tY2xlYW51cD12ZXJiYXRpbScpO1xuICAgIH0gZWxzZSB7XG4gICAgICBjb25zdCBjb25maWd1cmVkID0gYXdhaXQgdGhpcy5nZXRDb25maWcoJ2NvbW1pdC5jbGVhbnVwJyk7XG4gICAgICBjb25zdCBtb2RlID0gKGNvbmZpZ3VyZWQgJiYgY29uZmlndXJlZCAhPT0gJ2RlZmF1bHQnKSA/IGNvbmZpZ3VyZWQgOiAnc3RyaXAnO1xuICAgICAgYXJncy5wdXNoKGAtLWNsZWFudXA9JHttb2RlfWApO1xuICAgIH1cblxuICAgIC8vIGFkZCBjby1hdXRob3IgY29tbWl0IHRyYWlsZXJzIGlmIG5lY2Vzc2FyeVxuICAgIGlmIChjb0F1dGhvcnMgJiYgY29BdXRob3JzLmxlbmd0aCA+IDApIHtcbiAgICAgIG1zZyA9IGF3YWl0IHRoaXMuYWRkQ29BdXRob3JzVG9NZXNzYWdlKG1zZywgY29BdXRob3JzKTtcbiAgICB9XG5cbiAgICBhcmdzLnB1c2goJy1tJywgbXNnLnRyaW0oKSk7XG5cbiAgICBpZiAoYW1lbmQpIHsgYXJncy5wdXNoKCctLWFtZW5kJyk7IH1cbiAgICBpZiAoYWxsb3dFbXB0eSkgeyBhcmdzLnB1c2goJy0tYWxsb3ctZW1wdHknKTsgfVxuICAgIHJldHVybiB0aGlzLmdwZ0V4ZWMoYXJncywge3dyaXRlT3BlcmF0aW9uOiB0cnVlfSk7XG4gIH1cblxuICBhZGRDb0F1dGhvcnNUb01lc3NhZ2UobWVzc2FnZSwgY29BdXRob3JzID0gW10pIHtcbiAgICBjb25zdCB0cmFpbGVycyA9IGNvQXV0aG9ycy5tYXAoYXV0aG9yID0+IHtcbiAgICAgIHJldHVybiB7XG4gICAgICAgIHRva2VuOiAnQ28tQXV0aG9yZWQtQnknLFxuICAgICAgICB2YWx1ZTogYCR7YXV0aG9yLm5hbWV9IDwke2F1dGhvci5lbWFpbH0+YCxcbiAgICAgIH07XG4gICAgfSk7XG5cbiAgICAvLyBFbnN1cmUgdGhhdCBtZXNzYWdlIGVuZHMgd2l0aCBuZXdsaW5lIGZvciBnaXQtaW50ZXJwcmV0IHRyYWlsZXJzIHRvIHdvcmtcbiAgICBjb25zdCBtc2cgPSBgJHttZXNzYWdlLnRyaW0oKX1cXG5gO1xuXG4gICAgcmV0dXJuIHRyYWlsZXJzLmxlbmd0aCA/IHRoaXMubWVyZ2VUcmFpbGVycyhtc2csIHRyYWlsZXJzKSA6IG1zZztcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWxlIFN0YXR1cyBhbmQgRGlmZnNcbiAgICovXG4gIGFzeW5jIGdldFN0YXR1c0J1bmRsZSgpIHtcbiAgICBjb25zdCBhcmdzID0gWydzdGF0dXMnLCAnLS1wb3JjZWxhaW49djInLCAnLS1icmFuY2gnLCAnLS11bnRyYWNrZWQtZmlsZXM9YWxsJywgJy0taWdub3JlLXN1Ym1vZHVsZXM9ZGlydHknLCAnLXonXTtcbiAgICBjb25zdCBvdXRwdXQgPSBhd2FpdCB0aGlzLmV4ZWMoYXJncyk7XG4gICAgaWYgKG91dHB1dC5sZW5ndGggPiBNQVhfU1RBVFVTX09VVFBVVF9MRU5HVEgpIHtcbiAgICAgIHRocm93IG5ldyBMYXJnZVJlcG9FcnJvcigpO1xuICAgIH1cblxuICAgIGNvbnN0IHJlc3VsdHMgPSBhd2FpdCBwYXJzZVN0YXR1cyhvdXRwdXQpO1xuXG4gICAgZm9yIChjb25zdCBlbnRyeVR5cGUgaW4gcmVzdWx0cykge1xuICAgICAgaWYgKEFycmF5LmlzQXJyYXkocmVzdWx0c1tlbnRyeVR5cGVdKSkge1xuICAgICAgICB0aGlzLnVwZGF0ZU5hdGl2ZVBhdGhTZXBGb3JFbnRyaWVzKHJlc3VsdHNbZW50cnlUeXBlXSk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgcmV0dXJuIHJlc3VsdHM7XG4gIH1cblxuICB1cGRhdGVOYXRpdmVQYXRoU2VwRm9yRW50cmllcyhlbnRyaWVzKSB7XG4gICAgZW50cmllcy5mb3JFYWNoKGVudHJ5ID0+IHtcbiAgICAgIC8vIE5vcm1hbGx5IHdlIHdvdWxkIGF2b2lkIG11dGF0aW5nIHJlc3BvbnNlcyBmcm9tIG90aGVyIHBhY2thZ2UncyBBUElzLCBidXQgd2UgY29udHJvbFxuICAgICAgLy8gdGhlIGB3aGF0LXRoZS1zdGF0dXNgIG1vZHVsZSBhbmQga25vdyB0aGVyZSBhcmUgbm8gc2lkZSBlZmZlY3RzLlxuICAgICAgLy8gVGhpcyBpcyBhIGhvdCBjb2RlIHBhdGggYW5kIGJ5IG11dGF0aW5nIHdlIGF2b2lkIGNyZWF0aW5nIG5ldyBvYmplY3RzIHRoYXQgd2lsbCBqdXN0IGJlIEdDJ2VkXG4gICAgICBpZiAoZW50cnkuZmlsZVBhdGgpIHtcbiAgICAgICAgZW50cnkuZmlsZVBhdGggPSB0b05hdGl2ZVBhdGhTZXAoZW50cnkuZmlsZVBhdGgpO1xuICAgICAgfVxuICAgICAgaWYgKGVudHJ5Lm9yaWdGaWxlUGF0aCkge1xuICAgICAgICBlbnRyeS5vcmlnRmlsZVBhdGggPSB0b05hdGl2ZVBhdGhTZXAoZW50cnkub3JpZ0ZpbGVQYXRoKTtcbiAgICAgIH1cbiAgICB9KTtcbiAgfVxuXG4gIGFzeW5jIGRpZmZGaWxlU3RhdHVzKG9wdGlvbnMgPSB7fSkge1xuICAgIGNvbnN0IGFyZ3MgPSBbJ2RpZmYnLCAnLS1uYW1lLXN0YXR1cycsICctLW5vLXJlbmFtZXMnXTtcbiAgICBpZiAob3B0aW9ucy5zdGFnZWQpIHsgYXJncy5wdXNoKCctLXN0YWdlZCcpOyB9XG4gICAgaWYgKG9wdGlvbnMudGFyZ2V0KSB7IGFyZ3MucHVzaChvcHRpb25zLnRhcmdldCk7IH1cbiAgICBjb25zdCBvdXRwdXQgPSBhd2FpdCB0aGlzLmV4ZWMoYXJncyk7XG5cbiAgICBjb25zdCBzdGF0dXNNYXAgPSB7XG4gICAgICBBOiAnYWRkZWQnLFxuICAgICAgTTogJ21vZGlmaWVkJyxcbiAgICAgIEQ6ICdkZWxldGVkJyxcbiAgICAgIFU6ICd1bm1lcmdlZCcsXG4gICAgfTtcblxuICAgIGNvbnN0IGZpbGVTdGF0dXNlcyA9IHt9O1xuICAgIG91dHB1dCAmJiBvdXRwdXQudHJpbSgpLnNwbGl0KExJTkVfRU5ESU5HX1JFR0VYKS5mb3JFYWNoKGxpbmUgPT4ge1xuICAgICAgY29uc3QgW3N0YXR1cywgcmF3RmlsZVBhdGhdID0gbGluZS5zcGxpdCgnXFx0Jyk7XG4gICAgICBjb25zdCBmaWxlUGF0aCA9IHRvTmF0aXZlUGF0aFNlcChyYXdGaWxlUGF0aCk7XG4gICAgICBmaWxlU3RhdHVzZXNbZmlsZVBhdGhdID0gc3RhdHVzTWFwW3N0YXR1c107XG4gICAgfSk7XG4gICAgaWYgKCFvcHRpb25zLnN0YWdlZCkge1xuICAgICAgY29uc3QgdW50cmFja2VkID0gYXdhaXQgdGhpcy5nZXRVbnRyYWNrZWRGaWxlcygpO1xuICAgICAgdW50cmFja2VkLmZvckVhY2goZmlsZVBhdGggPT4geyBmaWxlU3RhdHVzZXNbZmlsZVBhdGhdID0gJ2FkZGVkJzsgfSk7XG4gICAgfVxuICAgIHJldHVybiBmaWxlU3RhdHVzZXM7XG4gIH1cblxuICBhc3luYyBnZXRVbnRyYWNrZWRGaWxlcygpIHtcbiAgICBjb25zdCBvdXRwdXQgPSBhd2FpdCB0aGlzLmV4ZWMoWydscy1maWxlcycsICctLW90aGVycycsICctLWV4Y2x1ZGUtc3RhbmRhcmQnXSk7XG4gICAgaWYgKG91dHB1dC50cmltKCkgPT09ICcnKSB7IHJldHVybiBbXTsgfVxuICAgIHJldHVybiBvdXRwdXQudHJpbSgpLnNwbGl0KExJTkVfRU5ESU5HX1JFR0VYKS5tYXAodG9OYXRpdmVQYXRoU2VwKTtcbiAgfVxuXG4gIGFzeW5jIGdldERpZmZzRm9yRmlsZVBhdGgoZmlsZVBhdGgsIHtzdGFnZWQsIGJhc2VDb21taXR9ID0ge30pIHtcbiAgICBsZXQgYXJncyA9IFsnZGlmZicsICctLW5vLXByZWZpeCcsICctLW5vLWV4dC1kaWZmJywgJy0tbm8tcmVuYW1lcycsICctLWRpZmYtZmlsdGVyPXUnXTtcbiAgICBpZiAoc3RhZ2VkKSB7IGFyZ3MucHVzaCgnLS1zdGFnZWQnKTsgfVxuICAgIGlmIChiYXNlQ29tbWl0KSB7IGFyZ3MucHVzaChiYXNlQ29tbWl0KTsgfVxuICAgIGFyZ3MgPSBhcmdzLmNvbmNhdChbJy0tJywgdG9HaXRQYXRoU2VwKGZpbGVQYXRoKV0pO1xuICAgIGNvbnN0IG91dHB1dCA9IGF3YWl0IHRoaXMuZXhlYyhhcmdzKTtcblxuICAgIGxldCByYXdEaWZmcyA9IFtdO1xuICAgIGlmIChvdXRwdXQpIHtcbiAgICAgIHJhd0RpZmZzID0gcGFyc2VEaWZmKG91dHB1dClcbiAgICAgICAgLmZpbHRlcihyYXdEaWZmID0+IHJhd0RpZmYuc3RhdHVzICE9PSAndW5tZXJnZWQnKTtcblxuICAgICAgZm9yIChsZXQgaSA9IDA7IGkgPCByYXdEaWZmcy5sZW5ndGg7IGkrKykge1xuICAgICAgICBjb25zdCByYXdEaWZmID0gcmF3RGlmZnNbaV07XG4gICAgICAgIGlmIChyYXdEaWZmLm9sZFBhdGgpIHtcbiAgICAgICAgICByYXdEaWZmLm9sZFBhdGggPSB0b05hdGl2ZVBhdGhTZXAocmF3RGlmZi5vbGRQYXRoKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAocmF3RGlmZi5uZXdQYXRoKSB7XG4gICAgICAgICAgcmF3RGlmZi5uZXdQYXRoID0gdG9OYXRpdmVQYXRoU2VwKHJhd0RpZmYubmV3UGF0aCk7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG5cbiAgICBpZiAoIXN0YWdlZCAmJiAoYXdhaXQgdGhpcy5nZXRVbnRyYWNrZWRGaWxlcygpKS5pbmNsdWRlcyhmaWxlUGF0aCkpIHtcbiAgICAgIC8vIGFkZCB1bnRyYWNrZWQgZmlsZVxuICAgICAgY29uc3QgYWJzUGF0aCA9IHBhdGguam9pbih0aGlzLndvcmtpbmdEaXIsIGZpbGVQYXRoKTtcbiAgICAgIGNvbnN0IGV4ZWN1dGFibGUgPSBhd2FpdCBpc0ZpbGVFeGVjdXRhYmxlKGFic1BhdGgpO1xuICAgICAgY29uc3Qgc3ltbGluayA9IGF3YWl0IGlzRmlsZVN5bWxpbmsoYWJzUGF0aCk7XG4gICAgICBjb25zdCBjb250ZW50cyA9IGF3YWl0IGZzLnJlYWRGaWxlKGFic1BhdGgsIHtlbmNvZGluZzogJ3V0ZjgnfSk7XG4gICAgICBjb25zdCBiaW5hcnkgPSBpc0JpbmFyeShjb250ZW50cyk7XG4gICAgICBsZXQgbW9kZTtcbiAgICAgIGxldCByZWFscGF0aDtcbiAgICAgIGlmIChleGVjdXRhYmxlKSB7XG4gICAgICAgIG1vZGUgPSBGaWxlLm1vZGVzLkVYRUNVVEFCTEU7XG4gICAgICB9IGVsc2UgaWYgKHN5bWxpbmspIHtcbiAgICAgICAgbW9kZSA9IEZpbGUubW9kZXMuU1lNTElOSztcbiAgICAgICAgcmVhbHBhdGggPSBhd2FpdCBmcy5yZWFscGF0aChhYnNQYXRoKTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIG1vZGUgPSBGaWxlLm1vZGVzLk5PUk1BTDtcbiAgICAgIH1cblxuICAgICAgcmF3RGlmZnMucHVzaChidWlsZEFkZGVkRmlsZVBhdGNoKGZpbGVQYXRoLCBiaW5hcnkgPyBudWxsIDogY29udGVudHMsIG1vZGUsIHJlYWxwYXRoKSk7XG4gICAgfVxuICAgIGlmIChyYXdEaWZmcy5sZW5ndGggPiAyKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEV4cGVjdGVkIGJldHdlZW4gMCBhbmQgMiBkaWZmcyBmb3IgJHtmaWxlUGF0aH0gYnV0IGdvdCAke3Jhd0RpZmZzLmxlbmd0aH1gKTtcbiAgICB9XG4gICAgcmV0dXJuIHJhd0RpZmZzO1xuICB9XG5cbiAgYXN5bmMgZ2V0U3RhZ2VkQ2hhbmdlc1BhdGNoKCkge1xuICAgIGNvbnN0IG91dHB1dCA9IGF3YWl0IHRoaXMuZXhlYyhbXG4gICAgICAnZGlmZicsICctLXN0YWdlZCcsICctLW5vLXByZWZpeCcsICctLW5vLWV4dC1kaWZmJywgJy0tbm8tcmVuYW1lcycsICctLWRpZmYtZmlsdGVyPXUnLFxuICAgIF0pO1xuXG4gICAgaWYgKCFvdXRwdXQpIHtcbiAgICAgIHJldHVybiBbXTtcbiAgICB9XG5cbiAgICBjb25zdCBkaWZmcyA9IHBhcnNlRGlmZihvdXRwdXQpO1xuICAgIGZvciAoY29uc3QgZGlmZiBvZiBkaWZmcykge1xuICAgICAgaWYgKGRpZmYub2xkUGF0aCkgeyBkaWZmLm9sZFBhdGggPSB0b05hdGl2ZVBhdGhTZXAoZGlmZi5vbGRQYXRoKTsgfVxuICAgICAgaWYgKGRpZmYubmV3UGF0aCkgeyBkaWZmLm5ld1BhdGggPSB0b05hdGl2ZVBhdGhTZXAoZGlmZi5uZXdQYXRoKTsgfVxuICAgIH1cbiAgICByZXR1cm4gZGlmZnM7XG4gIH1cblxuICAvKipcbiAgICogTWlzY2VsbGFuZW91cyBnZXR0ZXJzXG4gICAqL1xuICBhc3luYyBnZXRDb21taXQocmVmKSB7XG4gICAgY29uc3QgW2NvbW1pdF0gPSBhd2FpdCB0aGlzLmdldENvbW1pdHMoe21heDogMSwgcmVmLCBpbmNsdWRlVW5ib3JuOiB0cnVlfSk7XG4gICAgcmV0dXJuIGNvbW1pdDtcbiAgfVxuXG4gIGFzeW5jIGdldEhlYWRDb21taXQoKSB7XG4gICAgY29uc3QgW2hlYWRDb21taXRdID0gYXdhaXQgdGhpcy5nZXRDb21taXRzKHttYXg6IDEsIHJlZjogJ0hFQUQnLCBpbmNsdWRlVW5ib3JuOiB0cnVlfSk7XG4gICAgcmV0dXJuIGhlYWRDb21taXQ7XG4gIH1cblxuICBhc3luYyBnZXRDb21taXRzKG9wdGlvbnMgPSB7fSkge1xuICAgIGNvbnN0IHttYXgsIHJlZiwgaW5jbHVkZVVuYm9ybiwgaW5jbHVkZVBhdGNofSA9IHtcbiAgICAgIG1heDogMSxcbiAgICAgIHJlZjogJ0hFQUQnLFxuICAgICAgaW5jbHVkZVVuYm9ybjogZmFsc2UsXG4gICAgICBpbmNsdWRlUGF0Y2g6IGZhbHNlLFxuICAgICAgLi4ub3B0aW9ucyxcbiAgICB9O1xuXG4gICAgLy8gaHR0cHM6Ly9naXQtc2NtLmNvbS9kb2NzL2dpdC1sb2cjX3ByZXR0eV9mb3JtYXRzXG4gICAgLy8gJXgwMCAtIG51bGwgYnl0ZVxuICAgIC8vICVIIC0gY29tbWl0IFNIQVxuICAgIC8vICVhZSAtIGF1dGhvciBlbWFpbFxuICAgIC8vICVhbiA9IGF1dGhvciBmdWxsIG5hbWVcbiAgICAvLyAlYXQgLSB0aW1lc3RhbXAsIFVOSVggdGltZXN0YW1wXG4gICAgLy8gJXMgLSBzdWJqZWN0XG4gICAgLy8gJWIgLSBib2R5XG4gICAgY29uc3QgYXJncyA9IFtcbiAgICAgICdsb2cnLFxuICAgICAgJy0tcHJldHR5PWZvcm1hdDolSCV4MDAlYWUleDAwJWFuJXgwMCVhdCV4MDAlcyV4MDAlYiV4MDAnLFxuICAgICAgJy0tbm8tYWJicmV2LWNvbW1pdCcsXG4gICAgICAnLS1uby1wcmVmaXgnLFxuICAgICAgJy0tbm8tZXh0LWRpZmYnLFxuICAgICAgJy0tbm8tcmVuYW1lcycsXG4gICAgICAnLXonLFxuICAgICAgJy1uJyxcbiAgICAgIG1heCxcbiAgICAgIHJlZixcbiAgICBdO1xuXG4gICAgaWYgKGluY2x1ZGVQYXRjaCkge1xuICAgICAgYXJncy5wdXNoKCctLXBhdGNoJywgJy1tJywgJy0tZmlyc3QtcGFyZW50Jyk7XG4gICAgfVxuXG4gICAgY29uc3Qgb3V0cHV0ID0gYXdhaXQgdGhpcy5leGVjKGFyZ3MuY29uY2F0KCctLScpKS5jYXRjaChlcnIgPT4ge1xuICAgICAgaWYgKC91bmtub3duIHJldmlzaW9uLy50ZXN0KGVyci5zdGRFcnIpIHx8IC9iYWQgcmV2aXNpb24gJ0hFQUQnLy50ZXN0KGVyci5zdGRFcnIpKSB7XG4gICAgICAgIHJldHVybiAnJztcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHRocm93IGVycjtcbiAgICAgIH1cbiAgICB9KTtcblxuICAgIGlmIChvdXRwdXQgPT09ICcnKSB7XG4gICAgICByZXR1cm4gaW5jbHVkZVVuYm9ybiA/IFt7c2hhOiAnJywgbWVzc2FnZTogJycsIHVuYm9yblJlZjogdHJ1ZX1dIDogW107XG4gICAgfVxuXG4gICAgY29uc3QgZmllbGRzID0gb3V0cHV0LnRyaW0oKS5zcGxpdCgnXFwwJyk7XG5cbiAgICBjb25zdCBjb21taXRzID0gW107XG4gICAgZm9yIChsZXQgaSA9IDA7IGkgPCBmaWVsZHMubGVuZ3RoOyBpICs9IDcpIHtcbiAgICAgIGNvbnN0IGJvZHkgPSBmaWVsZHNbaSArIDVdLnRyaW0oKTtcbiAgICAgIGxldCBwYXRjaCA9IFtdO1xuICAgICAgaWYgKGluY2x1ZGVQYXRjaCkge1xuICAgICAgICBjb25zdCBkaWZmcyA9IGZpZWxkc1tpICsgNl07XG4gICAgICAgIHBhdGNoID0gcGFyc2VEaWZmKGRpZmZzLnRyaW0oKSk7XG4gICAgICB9XG5cbiAgICAgIGNvbnN0IHttZXNzYWdlOiBtZXNzYWdlQm9keSwgY29BdXRob3JzfSA9IGV4dHJhY3RDb0F1dGhvcnNBbmRSYXdDb21taXRNZXNzYWdlKGJvZHkpO1xuXG4gICAgICBjb21taXRzLnB1c2goe1xuICAgICAgICBzaGE6IGZpZWxkc1tpXSAmJiBmaWVsZHNbaV0udHJpbSgpLFxuICAgICAgICBhdXRob3JFbWFpbDogZmllbGRzW2kgKyAxXSAmJiBmaWVsZHNbaSArIDFdLnRyaW0oKSxcbiAgICAgICAgYXV0aG9yTmFtZTogZmllbGRzW2kgKyAyXSAmJiBmaWVsZHNbaSArIDJdLnRyaW0oKSxcbiAgICAgICAgYXV0aG9yRGF0ZTogcGFyc2VJbnQoZmllbGRzW2kgKyAzXSwgMTApLFxuICAgICAgICBtZXNzYWdlU3ViamVjdDogZmllbGRzW2kgKyA0XSxcbiAgICAgICAgbWVzc2FnZUJvZHksXG4gICAgICAgIGNvQXV0aG9ycyxcbiAgICAgICAgdW5ib3JuUmVmOiBmYWxzZSxcbiAgICAgICAgcGF0Y2gsXG4gICAgICB9KTtcbiAgICB9XG4gICAgcmV0dXJuIGNvbW1pdHM7XG4gIH1cblxuICBhc3luYyBnZXRBdXRob3JzKG9wdGlvbnMgPSB7fSkge1xuICAgIGNvbnN0IHttYXgsIHJlZn0gPSB7bWF4OiAxLCByZWY6ICdIRUFEJywgLi4ub3B0aW9uc307XG5cbiAgICAvLyBodHRwczovL2dpdC1zY20uY29tL2RvY3MvZ2l0LWxvZyNfcHJldHR5X2Zvcm1hdHNcbiAgICAvLyAleDFGIC0gZmllbGQgc2VwYXJhdG9yIGJ5dGVcbiAgICAvLyAlYW4gLSBhdXRob3IgbmFtZVxuICAgIC8vICVhZSAtIGF1dGhvciBlbWFpbFxuICAgIC8vICVjbiAtIGNvbW1pdHRlciBuYW1lXG4gICAgLy8gJWNlIC0gY29tbWl0dGVyIGVtYWlsXG4gICAgLy8gJSh0cmFpbGVyczp1bmZvbGQsb25seSkgLSB0aGUgY29tbWl0IG1lc3NhZ2UgdHJhaWxlcnMsIHNlcGFyYXRlZFxuICAgIC8vICAgICAgICAgICAgICAgICAgICAgICAgICAgYnkgbmV3bGluZXMgYW5kIHVuZm9sZGVkIChpLmUuIHByb3Blcmx5XG4gICAgLy8gICAgICAgICAgICAgICAgICAgICAgICAgICBmb3JtYXR0ZWQgYW5kIG9uZSB0cmFpbGVyIHBlciBsaW5lKS5cblxuICAgIGNvbnN0IGRlbGltaXRlciA9ICcxRic7XG4gICAgY29uc3QgZGVsaW1pdGVyU3RyaW5nID0gU3RyaW5nLmZyb21DaGFyQ29kZShwYXJzZUludChkZWxpbWl0ZXIsIDE2KSk7XG4gICAgY29uc3QgZmllbGRzID0gWyclYW4nLCAnJWFlJywgJyVjbicsICclY2UnLCAnJSh0cmFpbGVyczp1bmZvbGQsb25seSknXTtcbiAgICBjb25zdCBmb3JtYXQgPSBmaWVsZHMuam9pbihgJXgke2RlbGltaXRlcn1gKTtcblxuICAgIHRyeSB7XG4gICAgICBjb25zdCBvdXRwdXQgPSBhd2FpdCB0aGlzLmV4ZWMoW1xuICAgICAgICAnbG9nJywgYC0tZm9ybWF0PSR7Zm9ybWF0fWAsICcteicsICctbicsIG1heCwgcmVmLCAnLS0nLFxuICAgICAgXSk7XG5cbiAgICAgIHJldHVybiBvdXRwdXQuc3BsaXQoJ1xcMCcpXG4gICAgICAgIC5yZWR1Y2UoKGFjYywgbGluZSkgPT4ge1xuICAgICAgICAgIGlmIChsaW5lLmxlbmd0aCA9PT0gMCkgeyByZXR1cm4gYWNjOyB9XG5cbiAgICAgICAgICBjb25zdCBbYW4sIGFlLCBjbiwgY2UsIHRyYWlsZXJzXSA9IGxpbmUuc3BsaXQoZGVsaW1pdGVyU3RyaW5nKTtcbiAgICAgICAgICB0cmFpbGVyc1xuICAgICAgICAgICAgLnNwbGl0KCdcXG4nKVxuICAgICAgICAgICAgLm1hcCh0cmFpbGVyID0+IHRyYWlsZXIubWF0Y2goQ09fQVVUSE9SX1JFR0VYKSlcbiAgICAgICAgICAgIC5maWx0ZXIobWF0Y2ggPT4gbWF0Y2ggIT09IG51bGwpXG4gICAgICAgICAgICAuZm9yRWFjaCgoW18sIG5hbWUsIGVtYWlsXSkgPT4geyBhY2NbZW1haWxdID0gbmFtZTsgfSk7XG5cbiAgICAgICAgICBhY2NbYWVdID0gYW47XG4gICAgICAgICAgYWNjW2NlXSA9IGNuO1xuXG4gICAgICAgICAgcmV0dXJuIGFjYztcbiAgICAgICAgfSwge30pO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgaWYgKC91bmtub3duIHJldmlzaW9uLy50ZXN0KGVyci5zdGRFcnIpIHx8IC9iYWQgcmV2aXNpb24gJ0hFQUQnLy50ZXN0KGVyci5zdGRFcnIpKSB7XG4gICAgICAgIHJldHVybiBbXTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHRocm93IGVycjtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICBtZXJnZVRyYWlsZXJzKGNvbW1pdE1lc3NhZ2UsIHRyYWlsZXJzKSB7XG4gICAgY29uc3QgYXJncyA9IFsnaW50ZXJwcmV0LXRyYWlsZXJzJ107XG4gICAgZm9yIChjb25zdCB0cmFpbGVyIG9mIHRyYWlsZXJzKSB7XG4gICAgICBhcmdzLnB1c2goJy0tdHJhaWxlcicsIGAke3RyYWlsZXIudG9rZW59PSR7dHJhaWxlci52YWx1ZX1gKTtcbiAgICB9XG4gICAgcmV0dXJuIHRoaXMuZXhlYyhhcmdzLCB7c3RkaW46IGNvbW1pdE1lc3NhZ2V9KTtcbiAgfVxuXG4gIHJlYWRGaWxlRnJvbUluZGV4KGZpbGVQYXRoKSB7XG4gICAgcmV0dXJuIHRoaXMuZXhlYyhbJ3Nob3cnLCBgOiR7dG9HaXRQYXRoU2VwKGZpbGVQYXRoKX1gXSk7XG4gIH1cblxuICAvKipcbiAgICogTWVyZ2VcbiAgICovXG4gIG1lcmdlKGJyYW5jaE5hbWUpIHtcbiAgICByZXR1cm4gdGhpcy5ncGdFeGVjKFsnbWVyZ2UnLCBicmFuY2hOYW1lXSwge3dyaXRlT3BlcmF0aW9uOiB0cnVlfSk7XG4gIH1cblxuICBpc01lcmdpbmcoZG90R2l0RGlyKSB7XG4gICAgcmV0dXJuIGZpbGVFeGlzdHMocGF0aC5qb2luKGRvdEdpdERpciwgJ01FUkdFX0hFQUQnKSkuY2F0Y2goKCkgPT4gZmFsc2UpO1xuICB9XG5cbiAgYWJvcnRNZXJnZSgpIHtcbiAgICByZXR1cm4gdGhpcy5leGVjKFsnbWVyZ2UnLCAnLS1hYm9ydCddLCB7d3JpdGVPcGVyYXRpb246IHRydWV9KTtcbiAgfVxuXG4gIGNoZWNrb3V0U2lkZShzaWRlLCBwYXRocykge1xuICAgIGlmIChwYXRocy5sZW5ndGggPT09IDApIHtcbiAgICAgIHJldHVybiBQcm9taXNlLnJlc29sdmUoKTtcbiAgICB9XG5cbiAgICByZXR1cm4gdGhpcy5leGVjKFsnY2hlY2tvdXQnLCBgLS0ke3NpZGV9YCwgLi4ucGF0aHMubWFwKHRvR2l0UGF0aFNlcCldKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZWJhc2VcbiAgICovXG4gIGFzeW5jIGlzUmViYXNpbmcoZG90R2l0RGlyKSB7XG4gICAgY29uc3QgcmVzdWx0cyA9IGF3YWl0IFByb21pc2UuYWxsKFtcbiAgICAgIGZpbGVFeGlzdHMocGF0aC5qb2luKGRvdEdpdERpciwgJ3JlYmFzZS1tZXJnZScpKSxcbiAgICAgIGZpbGVFeGlzdHMocGF0aC5qb2luKGRvdEdpdERpciwgJ3JlYmFzZS1hcHBseScpKSxcbiAgICBdKTtcbiAgICByZXR1cm4gcmVzdWx0cy5zb21lKHIgPT4gcik7XG4gIH1cblxuICAvKipcbiAgICogUmVtb3RlIGludGVyYWN0aW9uc1xuICAgKi9cbiAgY2xvbmUocmVtb3RlVXJsLCBvcHRpb25zID0ge30pIHtcbiAgICBjb25zdCBhcmdzID0gWydjbG9uZSddO1xuICAgIGlmIChvcHRpb25zLm5vTG9jYWwpIHsgYXJncy5wdXNoKCctLW5vLWxvY2FsJyk7IH1cbiAgICBpZiAob3B0aW9ucy5iYXJlKSB7IGFyZ3MucHVzaCgnLS1iYXJlJyk7IH1cbiAgICBpZiAob3B0aW9ucy5yZWN1cnNpdmUpIHsgYXJncy5wdXNoKCctLXJlY3Vyc2l2ZScpOyB9XG4gICAgYXJncy5wdXNoKHJlbW90ZVVybCwgdGhpcy53b3JraW5nRGlyKTtcblxuICAgIHJldHVybiB0aGlzLmV4ZWMoYXJncywge3VzZUdpdFByb21wdFNlcnZlcjogdHJ1ZSwgd3JpdGVPcGVyYXRpb246IHRydWV9KTtcbiAgfVxuXG4gIGZldGNoKHJlbW90ZU5hbWUsIGJyYW5jaE5hbWUpIHtcbiAgICByZXR1cm4gdGhpcy5leGVjKFsnZmV0Y2gnLCByZW1vdGVOYW1lLCBicmFuY2hOYW1lXSwge3VzZUdpdFByb21wdFNlcnZlcjogdHJ1ZSwgd3JpdGVPcGVyYXRpb246IHRydWV9KTtcbiAgfVxuXG4gIHB1bGwocmVtb3RlTmFtZSwgYnJhbmNoTmFtZSwgb3B0aW9ucyA9IHt9KSB7XG4gICAgY29uc3QgYXJncyA9IFsncHVsbCcsIHJlbW90ZU5hbWUsIG9wdGlvbnMucmVmU3BlYyB8fCBicmFuY2hOYW1lXTtcbiAgICBpZiAob3B0aW9ucy5mZk9ubHkpIHtcbiAgICAgIGFyZ3MucHVzaCgnLS1mZi1vbmx5Jyk7XG4gICAgfVxuICAgIHJldHVybiB0aGlzLmdwZ0V4ZWMoYXJncywge3VzZUdpdFByb21wdFNlcnZlcjogdHJ1ZSwgd3JpdGVPcGVyYXRpb246IHRydWV9KTtcbiAgfVxuXG4gIHB1c2gocmVtb3RlTmFtZSwgYnJhbmNoTmFtZSwgb3B0aW9ucyA9IHt9KSB7XG4gICAgY29uc3QgYXJncyA9IFsncHVzaCcsIHJlbW90ZU5hbWUgfHwgJ29yaWdpbicsIG9wdGlvbnMucmVmU3BlYyB8fCBgcmVmcy9oZWFkcy8ke2JyYW5jaE5hbWV9YF07XG4gICAgaWYgKG9wdGlvbnMuc2V0VXBzdHJlYW0pIHsgYXJncy5wdXNoKCctLXNldC11cHN0cmVhbScpOyB9XG4gICAgaWYgKG9wdGlvbnMuZm9yY2UpIHsgYXJncy5wdXNoKCctLWZvcmNlJyk7IH1cbiAgICByZXR1cm4gdGhpcy5leGVjKGFyZ3MsIHt1c2VHaXRQcm9tcHRTZXJ2ZXI6IHRydWUsIHdyaXRlT3BlcmF0aW9uOiB0cnVlfSk7XG4gIH1cblxuICAvKipcbiAgICogVW5kbyBPcGVyYXRpb25zXG4gICAqL1xuICByZXNldCh0eXBlLCByZXZpc2lvbiA9ICdIRUFEJykge1xuICAgIGNvbnN0IHZhbGlkVHlwZXMgPSBbJ3NvZnQnXTtcbiAgICBpZiAoIXZhbGlkVHlwZXMuaW5jbHVkZXModHlwZSkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgSW52YWxpZCB0eXBlICR7dHlwZX0uIE11c3QgYmUgb25lIG9mOiAke3ZhbGlkVHlwZXMuam9pbignLCAnKX1gKTtcbiAgICB9XG4gICAgcmV0dXJuIHRoaXMuZXhlYyhbJ3Jlc2V0JywgYC0tJHt0eXBlfWAsIHJldmlzaW9uXSk7XG4gIH1cblxuICBkZWxldGVSZWYocmVmKSB7XG4gICAgcmV0dXJuIHRoaXMuZXhlYyhbJ3VwZGF0ZS1yZWYnLCAnLWQnLCByZWZdKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBCcmFuY2hlc1xuICAgKi9cbiAgY2hlY2tvdXQoYnJhbmNoTmFtZSwgb3B0aW9ucyA9IHt9KSB7XG4gICAgY29uc3QgYXJncyA9IFsnY2hlY2tvdXQnXTtcbiAgICBpZiAob3B0aW9ucy5jcmVhdGVOZXcpIHtcbiAgICAgIGFyZ3MucHVzaCgnLWInKTtcbiAgICB9XG4gICAgYXJncy5wdXNoKGJyYW5jaE5hbWUpO1xuICAgIGlmIChvcHRpb25zLnN0YXJ0UG9pbnQpIHtcbiAgICAgIGlmIChvcHRpb25zLnRyYWNrKSB7IGFyZ3MucHVzaCgnLS10cmFjaycpOyB9XG4gICAgICBhcmdzLnB1c2gob3B0aW9ucy5zdGFydFBvaW50KTtcbiAgICB9XG5cbiAgICByZXR1cm4gdGhpcy5leGVjKGFyZ3MsIHt3cml0ZU9wZXJhdGlvbjogdHJ1ZX0pO1xuICB9XG5cbiAgYXN5bmMgZ2V0QnJhbmNoZXMoKSB7XG4gICAgY29uc3QgZm9ybWF0ID0gW1xuICAgICAgJyUob2JqZWN0bmFtZSknLCAnJShIRUFEKScsICclKHJlZm5hbWU6c2hvcnQpJyxcbiAgICAgICclKHVwc3RyZWFtKScsICclKHVwc3RyZWFtOnJlbW90ZW5hbWUpJywgJyUodXBzdHJlYW06cmVtb3RlcmVmKScsXG4gICAgICAnJShwdXNoKScsICclKHB1c2g6cmVtb3RlbmFtZSknLCAnJShwdXNoOnJlbW90ZXJlZiknLFxuICAgIF0uam9pbignJTAwJyk7XG5cbiAgICBjb25zdCBvdXRwdXQgPSBhd2FpdCB0aGlzLmV4ZWMoWydmb3ItZWFjaC1yZWYnLCBgLS1mb3JtYXQ9JHtmb3JtYXR9YCwgJ3JlZnMvaGVhZHMvKionXSk7XG4gICAgcmV0dXJuIG91dHB1dC50cmltKCkuc3BsaXQoTElORV9FTkRJTkdfUkVHRVgpLm1hcChsaW5lID0+IHtcbiAgICAgIGNvbnN0IFtcbiAgICAgICAgc2hhLCBoZWFkLCBuYW1lLFxuICAgICAgICB1cHN0cmVhbVRyYWNraW5nUmVmLCB1cHN0cmVhbVJlbW90ZU5hbWUsIHVwc3RyZWFtUmVtb3RlUmVmLFxuICAgICAgICBwdXNoVHJhY2tpbmdSZWYsIHB1c2hSZW1vdGVOYW1lLCBwdXNoUmVtb3RlUmVmLFxuICAgICAgXSA9IGxpbmUuc3BsaXQoJ1xcMCcpO1xuXG4gICAgICBjb25zdCBicmFuY2ggPSB7bmFtZSwgc2hhLCBoZWFkOiBoZWFkID09PSAnKid9O1xuICAgICAgaWYgKHVwc3RyZWFtVHJhY2tpbmdSZWYgfHwgdXBzdHJlYW1SZW1vdGVOYW1lIHx8IHVwc3RyZWFtUmVtb3RlUmVmKSB7XG4gICAgICAgIGJyYW5jaC51cHN0cmVhbSA9IHtcbiAgICAgICAgICB0cmFja2luZ1JlZjogdXBzdHJlYW1UcmFja2luZ1JlZixcbiAgICAgICAgICByZW1vdGVOYW1lOiB1cHN0cmVhbVJlbW90ZU5hbWUsXG4gICAgICAgICAgcmVtb3RlUmVmOiB1cHN0cmVhbVJlbW90ZVJlZixcbiAgICAgICAgfTtcbiAgICAgIH1cbiAgICAgIGlmIChicmFuY2gudXBzdHJlYW0gfHwgcHVzaFRyYWNraW5nUmVmIHx8IHB1c2hSZW1vdGVOYW1lIHx8IHB1c2hSZW1vdGVSZWYpIHtcbiAgICAgICAgYnJhbmNoLnB1c2ggPSB7XG4gICAgICAgICAgdHJhY2tpbmdSZWY6IHB1c2hUcmFja2luZ1JlZixcbiAgICAgICAgICByZW1vdGVOYW1lOiBwdXNoUmVtb3RlTmFtZSB8fCAoYnJhbmNoLnVwc3RyZWFtICYmIGJyYW5jaC51cHN0cmVhbS5yZW1vdGVOYW1lKSxcbiAgICAgICAgICByZW1vdGVSZWY6IHB1c2hSZW1vdGVSZWYgfHwgKGJyYW5jaC51cHN0cmVhbSAmJiBicmFuY2gudXBzdHJlYW0ucmVtb3RlUmVmKSxcbiAgICAgICAgfTtcbiAgICAgIH1cbiAgICAgIHJldHVybiBicmFuY2g7XG4gICAgfSk7XG4gIH1cblxuICBhc3luYyBnZXRCcmFuY2hlc1dpdGhDb21taXQoc2hhLCBvcHRpb24gPSB7fSkge1xuICAgIGNvbnN0IGFyZ3MgPSBbJ2JyYW5jaCcsICctLWZvcm1hdD0lKHJlZm5hbWUpJywgJy0tY29udGFpbnMnLCBzaGFdO1xuICAgIGlmIChvcHRpb24uc2hvd0xvY2FsICYmIG9wdGlvbi5zaG93UmVtb3RlKSB7XG4gICAgICBhcmdzLnNwbGljZSgxLCAwLCAnLS1hbGwnKTtcbiAgICB9IGVsc2UgaWYgKG9wdGlvbi5zaG93UmVtb3RlKSB7XG4gICAgICBhcmdzLnNwbGljZSgxLCAwLCAnLS1yZW1vdGVzJyk7XG4gICAgfVxuICAgIHJldHVybiAoYXdhaXQgdGhpcy5leGVjKGFyZ3MpKS50cmltKCkuc3BsaXQoTElORV9FTkRJTkdfUkVHRVgpO1xuICB9XG5cbiAgY2hlY2tvdXRGaWxlcyhwYXRocywgcmV2aXNpb24pIHtcbiAgICBpZiAocGF0aHMubGVuZ3RoID09PSAwKSB7IHJldHVybiBudWxsOyB9XG4gICAgY29uc3QgYXJncyA9IFsnY2hlY2tvdXQnXTtcbiAgICBpZiAocmV2aXNpb24pIHsgYXJncy5wdXNoKHJldmlzaW9uKTsgfVxuICAgIHJldHVybiB0aGlzLmV4ZWMoYXJncy5jb25jYXQoJy0tJywgcGF0aHMubWFwKHRvR2l0UGF0aFNlcCkpLCB7d3JpdGVPcGVyYXRpb246IHRydWV9KTtcbiAgfVxuXG4gIGFzeW5jIGRlc2NyaWJlSGVhZCgpIHtcbiAgICByZXR1cm4gKGF3YWl0IHRoaXMuZXhlYyhbJ2Rlc2NyaWJlJywgJy0tY29udGFpbnMnLCAnLS1hbGwnLCAnLS1hbHdheXMnLCAnSEVBRCddKSkudHJpbSgpO1xuICB9XG5cbiAgYXN5bmMgZ2V0Q29uZmlnKG9wdGlvbiwge2xvY2FsfSA9IHt9KSB7XG4gICAgbGV0IG91dHB1dDtcbiAgICB0cnkge1xuICAgICAgbGV0IGFyZ3MgPSBbJ2NvbmZpZyddO1xuICAgICAgaWYgKGxvY2FsIHx8IGF0b20uaW5TcGVjTW9kZSgpKSB7IGFyZ3MucHVzaCgnLS1sb2NhbCcpOyB9XG4gICAgICBhcmdzID0gYXJncy5jb25jYXQob3B0aW9uKTtcbiAgICAgIG91dHB1dCA9IGF3YWl0IHRoaXMuZXhlYyhhcmdzKTtcbiAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgIGlmIChlcnIuY29kZSA9PT0gMSkge1xuICAgICAgICAvLyBObyBtYXRjaGluZyBjb25maWcgZm91bmRcbiAgICAgICAgcmV0dXJuIG51bGw7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICB0aHJvdyBlcnI7XG4gICAgICB9XG4gICAgfVxuXG4gICAgcmV0dXJuIG91dHB1dC50cmltKCk7XG4gIH1cblxuICBzZXRDb25maWcob3B0aW9uLCB2YWx1ZSwge3JlcGxhY2VBbGx9ID0ge30pIHtcbiAgICBsZXQgYXJncyA9IFsnY29uZmlnJ107XG4gICAgaWYgKHJlcGxhY2VBbGwpIHsgYXJncy5wdXNoKCctLXJlcGxhY2UtYWxsJyk7IH1cbiAgICBhcmdzID0gYXJncy5jb25jYXQob3B0aW9uLCB2YWx1ZSk7XG4gICAgcmV0dXJuIHRoaXMuZXhlYyhhcmdzLCB7d3JpdGVPcGVyYXRpb246IHRydWV9KTtcbiAgfVxuXG4gIHVuc2V0Q29uZmlnKG9wdGlvbikge1xuICAgIHJldHVybiB0aGlzLmV4ZWMoWydjb25maWcnLCAnLS11bnNldCcsIG9wdGlvbl0sIHt3cml0ZU9wZXJhdGlvbjogdHJ1ZX0pO1xuICB9XG5cbiAgYXN5bmMgZ2V0UmVtb3RlcygpIHtcbiAgICBsZXQgb3V0cHV0ID0gYXdhaXQgdGhpcy5nZXRDb25maWcoWyctLWdldC1yZWdleHAnLCAnXnJlbW90ZVxcXFwuLipcXFxcLnVybCQnXSwge2xvY2FsOiB0cnVlfSk7XG4gICAgaWYgKG91dHB1dCkge1xuICAgICAgb3V0cHV0ID0gb3V0cHV0LnRyaW0oKTtcbiAgICAgIGlmICghb3V0cHV0Lmxlbmd0aCkgeyByZXR1cm4gW107IH1cbiAgICAgIHJldHVybiBvdXRwdXQuc3BsaXQoJ1xcbicpLm1hcChsaW5lID0+IHtcbiAgICAgICAgY29uc3QgbWF0Y2ggPSBsaW5lLm1hdGNoKC9ecmVtb3RlXFwuKC4qKVxcLnVybCAoLiopJC8pO1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgIG5hbWU6IG1hdGNoWzFdLFxuICAgICAgICAgIHVybDogbWF0Y2hbMl0sXG4gICAgICAgIH07XG4gICAgICB9KTtcbiAgICB9IGVsc2Uge1xuICAgICAgcmV0dXJuIFtdO1xuICAgIH1cbiAgfVxuXG4gIGFkZFJlbW90ZShuYW1lLCB1cmwpIHtcbiAgICByZXR1cm4gdGhpcy5leGVjKFsncmVtb3RlJywgJ2FkZCcsIG5hbWUsIHVybF0pO1xuICB9XG5cbiAgYXN5bmMgY3JlYXRlQmxvYih7ZmlsZVBhdGgsIHN0ZGlufSA9IHt9KSB7XG4gICAgbGV0IG91dHB1dDtcbiAgICBpZiAoZmlsZVBhdGgpIHtcbiAgICAgIHRyeSB7XG4gICAgICAgIG91dHB1dCA9IChhd2FpdCB0aGlzLmV4ZWMoWydoYXNoLW9iamVjdCcsICctdycsIGZpbGVQYXRoXSwge3dyaXRlT3BlcmF0aW9uOiB0cnVlfSkpLnRyaW0oKTtcbiAgICAgIH0gY2F0Y2ggKGUpIHtcbiAgICAgICAgaWYgKGUuc3RkRXJyICYmIGUuc3RkRXJyLm1hdGNoKC9mYXRhbDogQ2Fubm90IG9wZW4gLio6IE5vIHN1Y2ggZmlsZSBvciBkaXJlY3RvcnkvKSkge1xuICAgICAgICAgIG91dHB1dCA9IG51bGw7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgdGhyb3cgZTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0gZWxzZSBpZiAoc3RkaW4pIHtcbiAgICAgIG91dHB1dCA9IChhd2FpdCB0aGlzLmV4ZWMoWydoYXNoLW9iamVjdCcsICctdycsICctLXN0ZGluJ10sIHtzdGRpbiwgd3JpdGVPcGVyYXRpb246IHRydWV9KSkudHJpbSgpO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ011c3Qgc3VwcGx5IGZpbGUgcGF0aCBvciBzdGRpbicpO1xuICAgIH1cbiAgICByZXR1cm4gb3V0cHV0O1xuICB9XG5cbiAgYXN5bmMgZXhwYW5kQmxvYlRvRmlsZShhYnNGaWxlUGF0aCwgc2hhKSB7XG4gICAgY29uc3Qgb3V0cHV0ID0gYXdhaXQgdGhpcy5leGVjKFsnY2F0LWZpbGUnLCAnLXAnLCBzaGFdKTtcbiAgICBhd2FpdCBmcy53cml0ZUZpbGUoYWJzRmlsZVBhdGgsIG91dHB1dCwge2VuY29kaW5nOiAndXRmOCd9KTtcbiAgICByZXR1cm4gYWJzRmlsZVBhdGg7XG4gIH1cblxuICBhc3luYyBnZXRCbG9iQ29udGVudHMoc2hhKSB7XG4gICAgcmV0dXJuIGF3YWl0IHRoaXMuZXhlYyhbJ2NhdC1maWxlJywgJy1wJywgc2hhXSk7XG4gIH1cblxuICBhc3luYyBtZXJnZUZpbGUob3Vyc1BhdGgsIGNvbW1vbkJhc2VQYXRoLCB0aGVpcnNQYXRoLCByZXN1bHRQYXRoKSB7XG4gICAgY29uc3QgYXJncyA9IFtcbiAgICAgICdtZXJnZS1maWxlJywgJy1wJywgb3Vyc1BhdGgsIGNvbW1vbkJhc2VQYXRoLCB0aGVpcnNQYXRoLFxuICAgICAgJy1MJywgJ2N1cnJlbnQnLCAnLUwnLCAnYWZ0ZXIgZGlzY2FyZCcsICctTCcsICdiZWZvcmUgZGlzY2FyZCcsXG4gICAgXTtcbiAgICBsZXQgb3V0cHV0O1xuICAgIGxldCBjb25mbGljdCA9IGZhbHNlO1xuICAgIHRyeSB7XG4gICAgICBvdXRwdXQgPSBhd2FpdCB0aGlzLmV4ZWMoYXJncyk7XG4gICAgfSBjYXRjaCAoZSkge1xuICAgICAgaWYgKGUgaW5zdGFuY2VvZiBHaXRFcnJvciAmJiBlLmNvZGUgPT09IDEpIHtcbiAgICAgICAgb3V0cHV0ID0gZS5zdGRPdXQ7XG4gICAgICAgIGNvbmZsaWN0ID0gdHJ1ZTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHRocm93IGU7XG4gICAgICB9XG4gICAgfVxuXG4gICAgLy8gSW50ZXJwcmV0IGEgcmVsYXRpdmUgcmVzdWx0UGF0aCBhcyByZWxhdGl2ZSB0byB0aGUgcmVwb3NpdG9yeSB3b3JraW5nIGRpcmVjdG9yeSBmb3IgY29uc2lzdGVuY3kgd2l0aCB0aGVcbiAgICAvLyBvdGhlciBhcmd1bWVudHMuXG4gICAgY29uc3QgcmVzb2x2ZWRSZXN1bHRQYXRoID0gcGF0aC5yZXNvbHZlKHRoaXMud29ya2luZ0RpciwgcmVzdWx0UGF0aCk7XG4gICAgYXdhaXQgZnMud3JpdGVGaWxlKHJlc29sdmVkUmVzdWx0UGF0aCwgb3V0cHV0LCB7ZW5jb2Rpbmc6ICd1dGY4J30pO1xuXG4gICAgcmV0dXJuIHtmaWxlUGF0aDogb3Vyc1BhdGgsIHJlc3VsdFBhdGgsIGNvbmZsaWN0fTtcbiAgfVxuXG4gIGFzeW5jIHdyaXRlTWVyZ2VDb25mbGljdFRvSW5kZXgoZmlsZVBhdGgsIGNvbW1vbkJhc2VTaGEsIG91cnNTaGEsIHRoZWlyc1NoYSkge1xuICAgIGNvbnN0IGdpdEZpbGVQYXRoID0gdG9HaXRQYXRoU2VwKGZpbGVQYXRoKTtcbiAgICBjb25zdCBmaWxlTW9kZSA9IGF3YWl0IHRoaXMuZ2V0RmlsZU1vZGUoZmlsZVBhdGgpO1xuICAgIGxldCBpbmRleEluZm8gPSBgMCAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwXFx0JHtnaXRGaWxlUGF0aH1cXG5gO1xuICAgIGlmIChjb21tb25CYXNlU2hhKSB7IGluZGV4SW5mbyArPSBgJHtmaWxlTW9kZX0gJHtjb21tb25CYXNlU2hhfSAxXFx0JHtnaXRGaWxlUGF0aH1cXG5gOyB9XG4gICAgaWYgKG91cnNTaGEpIHsgaW5kZXhJbmZvICs9IGAke2ZpbGVNb2RlfSAke291cnNTaGF9IDJcXHQke2dpdEZpbGVQYXRofVxcbmA7IH1cbiAgICBpZiAodGhlaXJzU2hhKSB7IGluZGV4SW5mbyArPSBgJHtmaWxlTW9kZX0gJHt0aGVpcnNTaGF9IDNcXHQke2dpdEZpbGVQYXRofVxcbmA7IH1cbiAgICByZXR1cm4gdGhpcy5leGVjKFsndXBkYXRlLWluZGV4JywgJy0taW5kZXgtaW5mbyddLCB7c3RkaW46IGluZGV4SW5mbywgd3JpdGVPcGVyYXRpb246IHRydWV9KTtcbiAgfVxuXG4gIGFzeW5jIGdldEZpbGVNb2RlKGZpbGVQYXRoKSB7XG4gICAgY29uc3Qgb3V0cHV0ID0gYXdhaXQgdGhpcy5leGVjKFsnbHMtZmlsZXMnLCAnLS1zdGFnZScsICctLScsIHRvR2l0UGF0aFNlcChmaWxlUGF0aCldKTtcbiAgICBpZiAob3V0cHV0KSB7XG4gICAgICByZXR1cm4gb3V0cHV0LnNsaWNlKDAsIDYpO1xuICAgIH0gZWxzZSB7XG4gICAgICBjb25zdCBleGVjdXRhYmxlID0gYXdhaXQgaXNGaWxlRXhlY3V0YWJsZShwYXRoLmpvaW4odGhpcy53b3JraW5nRGlyLCBmaWxlUGF0aCkpO1xuICAgICAgY29uc3Qgc3ltbGluayA9IGF3YWl0IGlzRmlsZVN5bWxpbmsocGF0aC5qb2luKHRoaXMud29ya2luZ0RpciwgZmlsZVBhdGgpKTtcbiAgICAgIGlmIChzeW1saW5rKSB7XG4gICAgICAgIHJldHVybiBGaWxlLm1vZGVzLlNZTUxJTks7XG4gICAgICB9IGVsc2UgaWYgKGV4ZWN1dGFibGUpIHtcbiAgICAgICAgcmV0dXJuIEZpbGUubW9kZXMuRVhFQ1VUQUJMRTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHJldHVybiBGaWxlLm1vZGVzLk5PUk1BTDtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICBkZXN0cm95KCkge1xuICAgIHRoaXMuY29tbWFuZFF1ZXVlLmRpc3Bvc2UoKTtcbiAgfVxufVxuXG5mdW5jdGlvbiBidWlsZEFkZGVkRmlsZVBhdGNoKGZpbGVQYXRoLCBjb250ZW50cywgbW9kZSwgcmVhbHBhdGgpIHtcbiAgY29uc3QgaHVua3MgPSBbXTtcbiAgaWYgKGNvbnRlbnRzKSB7XG4gICAgbGV0IG5vTmV3TGluZTtcbiAgICBsZXQgbGluZXM7XG4gICAgaWYgKG1vZGUgPT09IEZpbGUubW9kZXMuU1lNTElOSykge1xuICAgICAgbm9OZXdMaW5lID0gZmFsc2U7XG4gICAgICBsaW5lcyA9IFtgKyR7dG9HaXRQYXRoU2VwKHJlYWxwYXRoKX1gLCAnXFxcXCBObyBuZXdsaW5lIGF0IGVuZCBvZiBmaWxlJ107XG4gICAgfSBlbHNlIHtcbiAgICAgIG5vTmV3TGluZSA9IGNvbnRlbnRzW2NvbnRlbnRzLmxlbmd0aCAtIDFdICE9PSAnXFxuJztcbiAgICAgIGxpbmVzID0gY29udGVudHMudHJpbSgpLnNwbGl0KExJTkVfRU5ESU5HX1JFR0VYKS5tYXAobGluZSA9PiBgKyR7bGluZX1gKTtcbiAgICB9XG4gICAgaWYgKG5vTmV3TGluZSkgeyBsaW5lcy5wdXNoKCdcXFxcIE5vIG5ld2xpbmUgYXQgZW5kIG9mIGZpbGUnKTsgfVxuICAgIGh1bmtzLnB1c2goe1xuICAgICAgbGluZXMsXG4gICAgICBvbGRTdGFydExpbmU6IDAsXG4gICAgICBvbGRMaW5lQ291bnQ6IDAsXG4gICAgICBuZXdTdGFydExpbmU6IDEsXG4gICAgICBoZWFkaW5nOiAnJyxcbiAgICAgIG5ld0xpbmVDb3VudDogbm9OZXdMaW5lID8gbGluZXMubGVuZ3RoIC0gMSA6IGxpbmVzLmxlbmd0aCxcbiAgICB9KTtcbiAgfVxuICByZXR1cm4ge1xuICAgIG9sZFBhdGg6IG51bGwsXG4gICAgbmV3UGF0aDogdG9OYXRpdmVQYXRoU2VwKGZpbGVQYXRoKSxcbiAgICBvbGRNb2RlOiBudWxsLFxuICAgIG5ld01vZGU6IG1vZGUsXG4gICAgc3RhdHVzOiAnYWRkZWQnLFxuICAgIGh1bmtzLFxuICB9O1xufVxuIl19