(function() {
  var AtomIoClient, fs, glob, path, remote, request;

  fs = require('fs-plus');

  path = require('path');

  remote = require('electron').remote;

  glob = require('glob');

  request = require('request');

  module.exports = AtomIoClient = (function() {
    function AtomIoClient(packageManager, baseURL) {
      this.packageManager = packageManager;
      this.baseURL = baseURL;
      if (this.baseURL == null) {
        this.baseURL = 'https://atom.io/api/';
      }
      this.expiry = 1000 * 60 * 60 * 12;
      this.createAvatarCache();
      this.expireAvatarCache();
    }

    AtomIoClient.prototype.avatar = function(login, callback) {
      return this.cachedAvatar(login, (function(_this) {
        return function(err, cached) {
          var stale;
          if (cached) {
            stale = Date.now() - parseInt(cached.split('-').pop()) > _this.expiry;
          }
          if (cached && (!stale || !_this.online())) {
            return callback(null, cached);
          } else {
            return _this.fetchAndCacheAvatar(login, callback);
          }
        };
      })(this));
    };

    AtomIoClient.prototype["package"] = function(name, callback) {
      var data, packagePath;
      packagePath = "packages/" + name;
      data = this.fetchFromCache(packagePath);
      if (data) {
        return callback(null, data);
      } else {
        return this.request(packagePath, callback);
      }
    };

    AtomIoClient.prototype.featuredPackages = function(callback) {
      var data;
      data = this.fetchFromCache('packages/featured');
      if (data) {
        return callback(null, data);
      } else {
        return this.getFeatured(false, callback);
      }
    };

    AtomIoClient.prototype.featuredThemes = function(callback) {
      var data;
      data = this.fetchFromCache('themes/featured');
      if (data) {
        return callback(null, data);
      } else {
        return this.getFeatured(true, callback);
      }
    };

    AtomIoClient.prototype.getFeatured = function(loadThemes, callback) {
      return this.packageManager.getFeatured(loadThemes).then((function(_this) {
        return function(packages) {
          var cached, key;
          key = loadThemes ? 'themes/featured' : 'packages/featured';
          cached = {
            data: packages,
            createdOn: Date.now()
          };
          localStorage.setItem(_this.cacheKeyForPath(key), JSON.stringify(cached));
          return callback(null, packages);
        };
      })(this))["catch"](function(error) {
        return callback(error, null);
      });
    };

    AtomIoClient.prototype.request = function(path, callback) {
      var options;
      options = {
        url: "" + this.baseURL + path,
        headers: {
          'User-Agent': navigator.userAgent
        },
        gzip: true
      };
      return request(options, (function(_this) {
        return function(err, res, body) {
          var cached, error;
          if (err) {
            return callback(err);
          }
          try {
            body = JSON.parse(body);
            delete body.versions;
            cached = {
              data: body,
              createdOn: Date.now()
            };
            localStorage.setItem(_this.cacheKeyForPath(path), JSON.stringify(cached));
            return callback(err, cached.data);
          } catch (error1) {
            error = error1;
            return callback(error);
          }
        };
      })(this));
    };

    AtomIoClient.prototype.cacheKeyForPath = function(path) {
      return "settings-view:" + path;
    };

    AtomIoClient.prototype.online = function() {
      return navigator.onLine;
    };

    AtomIoClient.prototype.fetchFromCache = function(packagePath) {
      var cached;
      cached = localStorage.getItem(this.cacheKeyForPath(packagePath));
      cached = cached ? JSON.parse(cached) : void 0;
      if ((cached != null) && (!this.online() || Date.now() - cached.createdOn < this.expiry)) {
        return cached.data;
      } else {
        return null;
      }
    };

    AtomIoClient.prototype.createAvatarCache = function() {
      return fs.makeTree(this.getCachePath());
    };

    AtomIoClient.prototype.avatarPath = function(login) {
      return path.join(this.getCachePath(), login + "-" + (Date.now()));
    };

    AtomIoClient.prototype.cachedAvatar = function(login, callback) {
      return glob(this.avatarGlob(login), (function(_this) {
        return function(err, files) {
          var createdOn, filename, i, imagePath, len, ref;
          if (err) {
            return callback(err);
          }
          files.sort().reverse();
          for (i = 0, len = files.length; i < len; i++) {
            imagePath = files[i];
            filename = path.basename(imagePath);
            ref = filename.split('-'), createdOn = ref[ref.length - 1];
            if (Date.now() - parseInt(createdOn) < _this.expiry) {
              return callback(null, imagePath);
            }
          }
          return callback(null, null);
        };
      })(this));
    };

    AtomIoClient.prototype.avatarGlob = function(login) {
      return path.join(this.getCachePath(), login + "-*([0-9])");
    };

    AtomIoClient.prototype.fetchAndCacheAvatar = function(login, callback) {
      var imagePath, requestObject;
      if (!this.online()) {
        return callback(null, null);
      } else {
        imagePath = this.avatarPath(login);
        requestObject = {
          url: "https://avatars.githubusercontent.com/" + login,
          headers: {
            'User-Agent': navigator.userAgent
          }
        };
        return request.head(requestObject, function(error, response, body) {
          var writeStream;
          if ((error != null) || response.statusCode !== 200 || !response.headers['content-type'].startsWith('image/')) {
            return callback(error);
          } else {
            writeStream = fs.createWriteStream(imagePath);
            writeStream.on('finish', function() {
              return callback(null, imagePath);
            });
            writeStream.on('error', function(error) {
              writeStream.close();
              try {
                if (fs.existsSync(imagePath)) {
                  fs.unlinkSync(imagePath);
                }
              } catch (error1) {}
              return callback(error);
            });
            return request(requestObject).pipe(writeStream);
          }
        });
      }
    };

    AtomIoClient.prototype.expireAvatarCache = function() {
      var deleteAvatar;
      deleteAvatar = (function(_this) {
        return function(child) {
          var avatarPath;
          avatarPath = path.join(_this.getCachePath(), child);
          return fs.unlink(avatarPath, function(error) {
            if (error && error.code !== 'ENOENT') {
              return console.warn("Error deleting avatar (" + error.code + "): " + avatarPath);
            }
          });
        };
      })(this);
      return fs.readdir(this.getCachePath(), function(error, _files) {
        var children, filename, files, i, key, len, parts, results, stamp;
        if (_files == null) {
          _files = [];
        }
        files = {};
        for (i = 0, len = _files.length; i < len; i++) {
          filename = _files[i];
          parts = filename.split('-');
          stamp = parts.pop();
          key = parts.join('-');
          if (files[key] == null) {
            files[key] = [];
          }
          files[key].push(key + "-" + stamp);
        }
        results = [];
        for (key in files) {
          children = files[key];
          children.sort();
          children.pop();
          results.push(children.forEach(deleteAvatar));
        }
        return results;
      });
    };

    AtomIoClient.prototype.getCachePath = function() {
      return this.cachePath != null ? this.cachePath : this.cachePath = path.join(remote.app.getPath('userData'), 'Cache', 'settings-view');
    };

    AtomIoClient.prototype.search = function(query, options) {
      var qs;
      qs = {
        q: query
      };
      if (options.themes) {
        qs.filter = 'theme';
      } else if (options.packages) {
        qs.filter = 'package';
      }
      options = {
        url: this.baseURL + "packages/search",
        headers: {
          'User-Agent': navigator.userAgent
        },
        qs: qs,
        gzip: true
      };
      return new Promise(function(resolve, reject) {
        return request(options, function(err, res, body) {
          var e, error;
          if (err) {
            error = new Error("Searching for \u201C" + query + "\u201D failed.");
            error.stderr = err.message;
            return reject(error);
          } else {
            try {
              body = JSON.parse(body);
              return resolve(body.filter(function(pkg) {
                var ref;
                return ((ref = pkg.releases) != null ? ref.latest : void 0) != null;
              }).map(function(arg) {
                var downloads, metadata, readme, repository, stargazers_count;
                readme = arg.readme, metadata = arg.metadata, downloads = arg.downloads, stargazers_count = arg.stargazers_count, repository = arg.repository;
                return Object.assign(metadata, {
                  readme: readme,
                  downloads: downloads,
                  stargazers_count: stargazers_count,
                  repository: repository.url
                });
              }));
            } catch (error1) {
              e = error1;
              error = new Error("Searching for \u201C" + query + "\u201D failed.");
              error.stderr = e.message + '\n' + body;
              return reject(error);
            }
          }
        });
      });
    };

    return AtomIoClient;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
