"use strict";
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (Object.hasOwnProperty.call(mod, k)) result[k] = mod[k];
    result["default"] = mod;
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const path = __importStar(require("path"));
const fs_1 = require("fs");
const electron_1 = require("electron");
const electron_log_1 = __importDefault(require("electron-log"));
const electron_updater_1 = require("electron-updater");
const electron_dl_1 = __importDefault(require("electron-dl"));
const electron_context_menu_1 = __importDefault(require("electron-context-menu"));
const electron_is_dev_1 = __importDefault(require("electron-is-dev"));
const electron_debug_1 = __importDefault(require("electron-debug"));
const electron_util_1 = require("electron-util");
const facebook_locales_1 = require("facebook-locales");
const menu_1 = __importDefault(require("./menu"));
const config_1 = __importDefault(require("./config"));
const tray_1 = __importDefault(require("./tray"));
const util_1 = require("./util");
const emoji_1 = require("./emoji");
const ensure_online_1 = __importDefault(require("./ensure-online"));
require("./touch-bar"); // eslint-disable-line import/no-unassigned-import
electron_1.ipcMain.setMaxListeners(100);
electron_debug_1.default({
    enabled: true,
    showDevTools: false
});
electron_dl_1.default();
electron_context_menu_1.default();
const domain = config_1.default.get('useWorkChat') ? 'facebook.com' : 'messenger.com';
electron_1.app.setAppUserModelId('com.sindresorhus.caprine');
// Disables broken color space correction in Chromium.
// You can see differing background color on the login screen.
// https://github.com/electron/electron/issues/9671
electron_1.app.commandLine.appendSwitch('disable-color-correct-rendering');
if (!config_1.default.get('hardwareAcceleration')) {
    electron_1.app.disableHardwareAcceleration();
}
if (!electron_is_dev_1.default) {
    electron_log_1.default.transports.file.level = 'info';
    electron_updater_1.autoUpdater.logger = electron_log_1.default;
    const FOUR_HOURS = 1000 * 60 * 60 * 4;
    setInterval(() => {
        electron_updater_1.autoUpdater.checkForUpdates();
    }, FOUR_HOURS);
    electron_updater_1.autoUpdater.checkForUpdates();
}
let mainWindow;
let isQuitting = false;
let prevMessageCount = 0;
let dockMenu;
if (!electron_1.app.requestSingleInstanceLock()) {
    electron_1.app.quit();
}
electron_1.app.on('second-instance', () => {
    if (mainWindow) {
        if (mainWindow.isMinimized()) {
            mainWindow.restore();
        }
        mainWindow.show();
    }
});
function updateBadge(conversations) {
    // Ignore `Sindre messaged you` blinking
    if (!Array.isArray(conversations)) {
        return;
    }
    const messageCount = conversations.filter(({ unread }) => unread).length;
    if (electron_util_1.is.macos || electron_util_1.is.linux) {
        if (config_1.default.get('showUnreadBadge')) {
            electron_1.app.setBadgeCount(messageCount);
        }
        if (electron_util_1.is.macos && config_1.default.get('bounceDockOnMessage') && prevMessageCount !== messageCount) {
            electron_1.app.dock.bounce('informational');
            prevMessageCount = messageCount;
        }
    }
    if (electron_util_1.is.linux || electron_util_1.is.windows) {
        if (config_1.default.get('showUnreadBadge')) {
            tray_1.default.setBadge(messageCount > 0);
        }
        if (config_1.default.get('flashWindowOnMessage')) {
            mainWindow.flashFrame(messageCount !== 0);
        }
    }
    if (electron_util_1.is.windows) {
        if (config_1.default.get('showUnreadBadge')) {
            if (messageCount === 0) {
                mainWindow.setOverlayIcon(null, '');
            }
            else {
                // Delegate drawing of overlay icon to renderer process
                mainWindow.webContents.send('render-overlay-icon', messageCount);
            }
        }
    }
}
electron_1.ipcMain.on('update-overlay-icon', (_event, data, text) => {
    const img = electron_1.nativeImage.createFromDataURL(data);
    mainWindow.setOverlayIcon(img, text);
});
function enableHiresResources() {
    const scaleFactor = Math.max(...electron_1.screen.getAllDisplays().map(display => display.scaleFactor));
    if (scaleFactor === 1) {
        return;
    }
    const filter = { urls: [`*://*.${domain}/`] };
    electron_1.session.defaultSession.webRequest.onBeforeSendHeaders(filter, (details, callback) => {
        let cookie = details.requestHeaders.Cookie;
        if (cookie && details.method === 'GET') {
            if (/(; )?dpr=\d/.test(cookie)) {
                cookie = cookie.replace(/dpr=\d/, `dpr=${scaleFactor}`);
            }
            else {
                cookie = `${cookie}; dpr=${scaleFactor}`;
            }
            details.requestHeaders.Cookie = cookie;
        }
        callback({
            cancel: false,
            requestHeaders: details.requestHeaders
        });
    });
}
function initRequestsFiltering() {
    const filter = {
        urls: [
            `*://*.${domain}/*typ.php*`,
            `*://*.${domain}/*change_read_status.php*`,
            `*://*.${domain}/*delivery_receipts*`,
            `*://*.${domain}/*unread_threads*`,
            '*://*.fbcdn.net/images/emoji.php/v9/*',
            '*://*.facebook.com/images/emoji.php/v9/*' // Emoji
        ]
    };
    electron_1.session.defaultSession.webRequest.onBeforeRequest(filter, async ({ url }, callback) => {
        if (url.includes('emoji.php')) {
            callback(await emoji_1.process(url));
        }
        else if (url.includes('typ.php')) {
            callback({ cancel: config_1.default.get('block.typingIndicator') });
        }
        else if (url.includes('change_read_status.php')) {
            callback({ cancel: config_1.default.get('block.chatSeen') });
        }
        else if (url.includes('delivery_receipts') || url.includes('unread_threads')) {
            callback({ cancel: config_1.default.get('block.deliveryReceipt') });
        }
    });
}
function setUserLocale() {
    const userLocale = facebook_locales_1.bestFacebookLocaleFor(electron_1.app.getLocale().replace('-', '_'));
    const cookie = {
        url: 'https://www.messenger.com/',
        name: 'locale',
        value: userLocale
    };
    electron_1.session.defaultSession.cookies.set(cookie, () => { });
}
function setNotificationsMute(status) {
    const label = 'Mute Notifications';
    const muteMenuItem = electron_1.Menu.getApplicationMenu().getMenuItemById('mute-notifications');
    config_1.default.set('notificationsMuted', status);
    muteMenuItem.checked = status;
    if (electron_util_1.is.macos) {
        const item = dockMenu.items.find(x => x.label === label);
        item.checked = status;
    }
}
function createMainWindow() {
    const lastWindowState = config_1.default.get('lastWindowState');
    const isDarkMode = config_1.default.get('darkMode');
    // Messenger or Work Chat
    const mainURL = config_1.default.get('useWorkChat')
        ? 'https://work.facebook.com/chat'
        : 'https://www.messenger.com/login/';
    const win = new electron_1.BrowserWindow({
        title: electron_1.app.getName(),
        show: false,
        x: lastWindowState.x,
        y: lastWindowState.y,
        width: lastWindowState.width,
        height: lastWindowState.height,
        icon: electron_util_1.is.linux ? path.join(__dirname, '..', 'static', 'Icon.png') : undefined,
        minWidth: 400,
        minHeight: 200,
        alwaysOnTop: config_1.default.get('alwaysOnTop'),
        titleBarStyle: 'hiddenInset',
        autoHideMenuBar: config_1.default.get('autoHideMenuBar'),
        darkTheme: isDarkMode,
        webPreferences: {
            preload: path.join(__dirname, 'browser.js'),
            nodeIntegration: false,
            contextIsolation: true,
            plugins: true
        }
    });
    setUserLocale();
    initRequestsFiltering();
    electron_util_1.darkMode.onChange(() => {
        win.webContents.send('set-dark-mode');
    });
    if (electron_util_1.is.macos) {
        win.setSheetOffset(40);
    }
    win.loadURL(mainURL);
    win.on('close', e => {
        if (config_1.default.get('quitOnWindowClose')) {
            electron_1.app.quit();
            return;
        }
        if (!isQuitting) {
            e.preventDefault();
            // Workaround for electron/electron#10023
            win.blur();
            win.hide();
        }
    });
    win.on('focus', () => {
        if (config_1.default.get('flashWindowOnMessage')) {
            // This is a security in the case where messageCount is not reset by page title update
            win.flashFrame(false);
        }
    });
    return win;
}
(async () => {
    await Promise.all([ensure_online_1.default(), electron_1.app.whenReady()]);
    const trackingUrlPrefix = `https://l.${domain}/l.php`;
    await menu_1.default();
    mainWindow = createMainWindow();
    tray_1.default.create(mainWindow);
    if (electron_util_1.is.macos) {
        const firstItem = {
            label: 'Mute Notifications',
            type: 'checkbox',
            checked: config_1.default.get('notificationsMuted'),
            click() {
                mainWindow.webContents.send('toggle-mute-notifications');
            }
        };
        dockMenu = electron_1.Menu.buildFromTemplate([firstItem]);
        electron_1.app.dock.setMenu(dockMenu);
        electron_1.ipcMain.on('conversations', (_event, conversations) => {
            if (conversations.length === 0) {
                return;
            }
            const items = conversations.map(({ label, icon }, index) => {
                return {
                    label: `${label}`,
                    icon: electron_1.nativeImage.createFromDataURL(icon),
                    click: () => {
                        mainWindow.show();
                        util_1.sendAction('jump-to-conversation', index + 1);
                    }
                };
            });
            electron_1.app.dock.setMenu(electron_1.Menu.buildFromTemplate([firstItem, { type: 'separator' }, ...items]));
        });
    }
    // Update badge on conversations change
    electron_1.ipcMain.on('conversations', (_event, conversations) => {
        updateBadge(conversations);
    });
    enableHiresResources();
    const { webContents } = mainWindow;
    webContents.on('dom-ready', async () => {
        await menu_1.default();
        webContents.insertCSS(fs_1.readFileSync(path.join(__dirname, '..', 'css', 'browser.css'), 'utf8'));
        webContents.insertCSS(fs_1.readFileSync(path.join(__dirname, '..', 'css', 'dark-mode.css'), 'utf8'));
        webContents.insertCSS(fs_1.readFileSync(path.join(__dirname, '..', 'css', 'vibrancy.css'), 'utf8'));
        webContents.insertCSS(fs_1.readFileSync(path.join(__dirname, '..', 'css', 'code-blocks.css'), 'utf8'));
        if (config_1.default.get('useWorkChat')) {
            webContents.insertCSS(fs_1.readFileSync(path.join(__dirname, '..', 'css', 'workchat.css'), 'utf8'));
        }
        if (fs_1.existsSync(path.join(electron_1.app.getPath('userData'), 'custom.css'))) {
            webContents.insertCSS(fs_1.readFileSync(path.join(electron_1.app.getPath('userData'), 'custom.css'), 'utf8'));
        }
        if (config_1.default.get('launchMinimized') || electron_1.app.getLoginItemSettings().wasOpenedAsHidden) {
            mainWindow.hide();
        }
        else {
            mainWindow.show();
        }
        webContents.send('toggle-mute-notifications', config_1.default.get('notificationsMuted'));
        webContents.send('toggle-message-buttons', config_1.default.get('showMessageButtons'));
        webContents.executeJavaScript(fs_1.readFileSync(path.join(__dirname, 'notifications-isolated.js'), 'utf8'));
    });
    webContents.on('new-window', (event, url, frameName, _disposition, options) => {
        event.preventDefault();
        if (url === 'about:blank') {
            if (frameName !== 'about:blank') {
                // Voice/video call popup
                options.show = true;
                options.titleBarStyle = 'default';
                options.webPreferences.nodeIntegration = false;
                options.webPreferences.preload = path.join(__dirname, 'browser-call.js');
                event.newGuest = new electron_1.BrowserWindow(options);
            }
        }
        else {
            if (url.startsWith(trackingUrlPrefix)) {
                url = new URL(url).searchParams.get('u');
            }
            electron_1.shell.openExternal(url);
        }
    });
    webContents.on('will-navigate', (event, url) => {
        const isMessengerDotCom = (url) => {
            const { hostname } = new URL(url);
            return hostname.endsWith('.messenger.com');
        };
        const isTwoFactorAuth = (url) => {
            const twoFactorAuthURL = 'https://www.facebook.com/checkpoint/start';
            return url.startsWith(twoFactorAuthURL);
        };
        const isWorkChat = (url) => {
            const { hostname, pathname } = new URL(url);
            if (hostname === 'work.facebook.com') {
                return true;
            }
            if (
            // Example: https://company-name.facebook.com/login
            hostname.endsWith('.facebook.com') &&
                (pathname.startsWith('/login') || pathname.startsWith('/chat'))) {
                return true;
            }
            if (hostname === 'login.microsoftonline.com') {
                return true;
            }
            return false;
        };
        if (isMessengerDotCom(url) || isTwoFactorAuth(url) || isWorkChat(url)) {
            return;
        }
        event.preventDefault();
        electron_1.shell.openExternal(url);
    });
})();
if (electron_util_1.is.macos) {
    electron_1.ipcMain.on('set-vibrancy', () => {
        mainWindow.setVibrancy('sidebar');
        if (config_1.default.get('followSystemAppearance')) {
            electron_1.systemPreferences.setAppLevelAppearance(electron_1.systemPreferences.isDarkMode() ? 'dark' : 'light');
        }
        else {
            electron_1.systemPreferences.setAppLevelAppearance(config_1.default.get('darkMode') ? 'dark' : 'light');
        }
    });
}
electron_1.ipcMain.on('mute-notifications-toggled', (_event, status) => {
    setNotificationsMute(status);
});
electron_1.app.on('activate', () => {
    if (mainWindow) {
        mainWindow.show();
    }
});
electron_1.app.on('before-quit', () => {
    isQuitting = true;
    config_1.default.set('lastWindowState', mainWindow.getNormalBounds());
});
const notifications = new Map();
electron_1.ipcMain.on('notification', (_event, { id, title, body, icon, silent }) => {
    const notification = new electron_1.Notification({
        title,
        body,
        hasReply: true,
        icon: electron_1.nativeImage.createFromDataURL(icon),
        silent
    });
    notifications.set(id, notification);
    notification.on('click', () => {
        mainWindow.show();
        util_1.sendAction('notification-callback', { callbackName: 'onclick', id });
        notifications.delete(id);
    });
    notification.on('reply', (_event, reply) => {
        // We use onclick event used by messenger to go to the right convo
        util_1.sendBackgroundAction('notification-reply-callback', { callbackName: 'onclick', id, reply });
        notifications.delete(id);
    });
    notification.on('close', () => {
        util_1.sendAction('notification-callback', { callbackName: 'onclose', id });
        notifications.delete(id);
    });
    notification.show();
});
//# sourceMappingURL=index.js.map