"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Operation = exports.RendererProcess = exports.Worker = exports["default"] = void 0;

var _path = _interopRequireDefault(require("path"));

var _querystring = _interopRequireDefault(require("querystring"));

var _electron = require("electron");

var _eventKit = require("event-kit");

var _helpers = require("./helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const {
  BrowserWindow
} = _electron.remote;

class WorkerManager {
  static getInstance() {
    if (!this.instance) {
      this.instance = new WorkerManager();
    }

    return this.instance;
  }

  static reset(force) {
    if (this.instance) {
      this.instance.destroy(force);
    }

    this.instance = null;
  }

  constructor() {
    (0, _helpers.autobind)(this, 'onDestroyed', 'onCrashed', 'onSick');
    this.workers = new Set();
    this.activeWorker = null;
    this.createNewWorker();
  }

  isReady() {
    return this.activeWorker.isReady();
  }

  request(data) {
    if (this.destroyed) {
      throw new Error('Worker is destroyed');
    }

    let operation;
    const requestPromise = new Promise((resolve, reject) => {
      operation = new Operation(data, resolve, reject);
      return this.activeWorker.executeOperation(operation);
    });
    operation.setPromise(requestPromise);
    return {
      cancel: () => this.activeWorker.cancelOperation(operation),
      promise: requestPromise
    };
  }

  createNewWorker({
    operationCountLimit
  } = {
    operationCountLimit: 10
  }) {
    if (this.destroyed) {
      return;
    }

    this.activeWorker = new Worker({
      operationCountLimit,
      onDestroyed: this.onDestroyed,
      onCrashed: this.onCrashed,
      onSick: this.onSick
    });
    this.workers.add(this.activeWorker);
  }

  onDestroyed(destroyedWorker) {
    this.workers["delete"](destroyedWorker);
  }

  onCrashed(crashedWorker) {
    if (crashedWorker === this.getActiveWorker()) {
      this.createNewWorker({
        operationCountLimit: crashedWorker.getOperationCountLimit()
      });
    }

    crashedWorker.getRemainingOperations().forEach(operation => this.activeWorker.executeOperation(operation));
  }

  onSick(sickWorker) {
    if (!atom.inSpecMode()) {
      // eslint-disable-next-line no-console
      console.warn(`Sick worker detected.
        operationCountLimit: ${sickWorker.getOperationCountLimit()},
        completed operation count: ${sickWorker.getCompletedOperationCount()}`);
    }

    const operationCountLimit = this.calculateNewOperationCountLimit(sickWorker);
    return this.createNewWorker({
      operationCountLimit
    });
  }

  calculateNewOperationCountLimit(lastWorker) {
    let operationCountLimit = 10;

    if (lastWorker.getOperationCountLimit() >= lastWorker.getCompletedOperationCount()) {
      operationCountLimit = Math.min(lastWorker.getOperationCountLimit() * 2, 100);
    }

    return operationCountLimit;
  }

  getActiveWorker() {
    return this.activeWorker;
  }

  getReadyPromise() {
    return this.activeWorker.getReadyPromise();
  }

  destroy(force) {
    this.destroyed = true;
    this.workers.forEach(worker => worker.destroy(force));
  }

}

exports["default"] = WorkerManager;

_defineProperty(WorkerManager, "instance", null);

class Worker {
  constructor({
    operationCountLimit,
    onSick,
    onCrashed,
    onDestroyed
  }) {
    (0, _helpers.autobind)(this, 'handleDataReceived', 'onOperationComplete', 'handleCancelled', 'handleExecStarted', 'handleSpawnError', 'handleStdinError', 'handleSick', 'handleCrashed');
    this.operationCountLimit = operationCountLimit;
    this.onSick = onSick;
    this.onCrashed = onCrashed;
    this.onDestroyed = onDestroyed;
    this.operationsById = new Map();
    this.completedOperationCount = 0;
    this.sick = false;
    this.rendererProcess = new RendererProcess({
      loadUrl: this.getLoadUrl(operationCountLimit),
      onData: this.handleDataReceived,
      onCancelled: this.handleCancelled,
      onExecStarted: this.handleExecStarted,
      onSpawnError: this.handleSpawnError,
      onStdinError: this.handleStdinError,
      onSick: this.handleSick,
      onCrashed: this.handleCrashed,
      onDestroyed: this.destroy
    });
  }

  isReady() {
    return this.rendererProcess.isReady();
  }

  getLoadUrl(operationCountLimit) {
    const htmlPath = _path["default"].join((0, _helpers.getPackageRoot)(), 'lib', 'renderer.html');

    const rendererJsPath = _path["default"].join((0, _helpers.getPackageRoot)(), 'lib', 'worker.js');

    const qs = _querystring["default"].stringify({
      js: rendererJsPath,
      managerWebContentsId: this.getWebContentsId(),
      operationCountLimit,
      channelName: Worker.channelName
    });

    return `file://${htmlPath}?${qs}`;
  }

  getWebContentsId() {
    return _electron.remote.getCurrentWebContents().id;
  }

  executeOperation(operation) {
    this.operationsById.set(operation.id, operation);
    operation.onComplete(this.onOperationComplete);
    return this.rendererProcess.executeOperation(operation);
  }

  cancelOperation(operation) {
    return this.rendererProcess.cancelOperation(operation);
  }

  handleDataReceived({
    id,
    results
  }) {
    const operation = this.operationsById.get(id);
    operation.complete(results, data => {
      const {
        timing
      } = data;
      const totalInternalTime = timing.execTime + timing.spawnTime;
      const ipcTime = operation.getExecutionTime() - totalInternalTime;
      data.timing.ipcTime = ipcTime;
      return data;
    });
  }

  onOperationComplete(operation) {
    this.completedOperationCount++;
    this.operationsById["delete"](operation.id);

    if (this.sick && this.operationsById.size === 0) {
      this.destroy();
    }
  }

  handleCancelled({
    id
  }) {
    const operation = this.operationsById.get(id);

    if (operation) {
      // handleDataReceived() can be received before handleCancelled()
      operation.wasCancelled();
    }
  }

  handleExecStarted({
    id
  }) {
    const operation = this.operationsById.get(id);
    operation.setInProgress();
  }

  handleSpawnError({
    id,
    err
  }) {
    const operation = this.operationsById.get(id);
    operation.error(err);
  }

  handleStdinError({
    id,
    stdin,
    err
  }) {
    const operation = this.operationsById.get(id);
    operation.error(err);
  }

  handleSick() {
    this.sick = true;
    this.onSick(this);
  }

  handleCrashed() {
    this.onCrashed(this);
    this.destroy();
  }

  getOperationCountLimit() {
    return this.operationCountLimit;
  }

  getCompletedOperationCount() {
    return this.completedOperationCount;
  }

  getRemainingOperations() {
    return Array.from(this.operationsById.values());
  }

  getPid() {
    return this.rendererProcess.getPid();
  }

  getReadyPromise() {
    return this.rendererProcess.getReadyPromise();
  }

  async destroy(force) {
    this.onDestroyed(this);

    if (this.operationsById.size > 0 && !force) {
      const remainingOperationPromises = this.getRemainingOperations().map(operation => operation.getPromise()["catch"](() => null));
      await Promise.all(remainingOperationPromises);
    }

    this.rendererProcess.destroy();
  }

}
/*
Sends operations to renderer processes
*/


exports.Worker = Worker;

_defineProperty(Worker, "channelName", 'github:renderer-ipc');

class RendererProcess {
  constructor({
    loadUrl,
    onDestroyed,
    onCrashed,
    onSick,
    onData,
    onCancelled,
    onSpawnError,
    onStdinError,
    onExecStarted
  }) {
    (0, _helpers.autobind)(this, 'handleDestroy');
    this.onDestroyed = onDestroyed;
    this.onCrashed = onCrashed;
    this.onSick = onSick;
    this.onData = onData;
    this.onCancelled = onCancelled;
    this.onSpawnError = onSpawnError;
    this.onStdinError = onStdinError;
    this.onExecStarted = onExecStarted;
    this.win = new BrowserWindow({
      show: !!process.env.ATOM_GITHUB_SHOW_RENDERER_WINDOW
    });
    this.webContents = this.win.webContents; // this.webContents.openDevTools();

    this.emitter = new _eventKit.Emitter();
    this.subscriptions = new _eventKit.CompositeDisposable();
    this.registerListeners();
    this.win.loadURL(loadUrl);
    this.win.webContents.on('crashed', this.handleDestroy);
    this.win.webContents.on('destroyed', this.handleDestroy);
    this.subscriptions.add(new _eventKit.Disposable(() => {
      if (!this.win.isDestroyed()) {
        this.win.webContents.removeListener('crashed', this.handleDestroy);
        this.win.webContents.removeListener('destroyed', this.handleDestroy);
        this.win.destroy();
      }
    }), this.emitter);
    this.ready = false;
    this.readyPromise = new Promise(resolve => {
      this.resolveReady = resolve;
    });
  }

  isReady() {
    return this.ready;
  }

  handleDestroy(...args) {
    this.destroy();
    this.onCrashed(...args);
  }

  registerListeners() {
    const handleMessages = (event, {
      sourceWebContentsId,
      type,
      data
    }) => {
      if (sourceWebContentsId === this.win.webContents.id) {
        this.emitter.emit(type, data);
      }
    };

    _electron.ipcRenderer.on(Worker.channelName, handleMessages);

    this.emitter.on('renderer-ready', ({
      pid
    }) => {
      this.pid = pid;
      this.ready = true;
      this.resolveReady();
    });
    this.emitter.on('git-data', this.onData);
    this.emitter.on('git-cancelled', this.onCancelled);
    this.emitter.on('git-spawn-error', this.onSpawnError);
    this.emitter.on('git-stdin-error', this.onStdinError);
    this.emitter.on('slow-spawns', this.onSick); // not currently used to avoid clogging up ipc channel
    // keeping it around as it's potentially useful for avoiding duplicate write operations upon renderer crashing

    this.emitter.on('exec-started', this.onExecStarted);
    this.subscriptions.add(new _eventKit.Disposable(() => _electron.ipcRenderer.removeListener(Worker.channelName, handleMessages)));
  }

  executeOperation(operation) {
    return operation.execute(payload => {
      if (this.destroyed) {
        return null;
      }

      return this.webContents.send(Worker.channelName, {
        type: 'git-exec',
        data: payload
      });
    });
  }

  cancelOperation(operation) {
    return operation.cancel(payload => {
      if (this.destroyed) {
        return null;
      }

      return this.webContents.send(Worker.channelName, {
        type: 'git-cancel',
        data: payload
      });
    });
  }

  getPid() {
    return this.pid;
  }

  getReadyPromise() {
    return this.readyPromise;
  }

  destroy() {
    this.destroyed = true;
    this.subscriptions.dispose();
  }

}

exports.RendererProcess = RendererProcess;

class Operation {
  constructor(data, resolve, reject) {
    this.id = Operation.id++;
    this.data = data;
    this.resolve = resolve;
    this.reject = reject;
    this.promise = null;

    this.cancellationResolve = () => {};

    this.startTime = null;
    this.endTime = null;
    this.status = Operation.status.PENDING;
    this.results = null;
    this.emitter = new _eventKit.Emitter();
  }

  onComplete(cb) {
    return this.emitter.on('complete', cb);
  }

  setPromise(promise) {
    this.promise = promise;
  }

  getPromise() {
    return this.promise;
  }

  setInProgress() {
    // after exec has been called but before results a received
    this.status = Operation.status.INPROGRESS;
  }

  getExecutionTime() {
    if (!this.startTime || !this.endTime) {
      return NaN;
    } else {
      return this.endTime - this.startTime;
    }
  }

  complete(results, mutate = data => data) {
    this.endTime = performance.now();
    this.results = results;
    this.resolve(mutate(results));
    this.cancellationResolve();
    this.status = Operation.status.COMPLETE;
    this.emitter.emit('complete', this);
    this.emitter.dispose();
  }

  wasCancelled() {
    this.status = Operation.status.CANCELLED;
    this.cancellationResolve();
  }

  error(results) {
    this.endTime = performance.now();
    const err = new Error(results.message, results.fileName, results.lineNumber);
    err.stack = results.stack;
    this.reject(err);
  }

  execute(execFn) {
    this.startTime = performance.now();
    return execFn(_objectSpread({}, this.data, {
      id: this.id
    }));
  }

  cancel(execFn) {
    return new Promise(resolve => {
      this.status = Operation.status.CANCELLING;
      this.cancellationResolve = resolve;
      execFn({
        id: this.id
      });
    });
  }

}

exports.Operation = Operation;

_defineProperty(Operation, "status", {
  PENDING: Symbol('pending'),
  INPROGRESS: Symbol('in-progress'),
  COMPLETE: Symbol('complete'),
  CANCELLING: Symbol('cancelling'),
  CANCELLED: Symbol('canceled')
});

_defineProperty(Operation, "id", 0);
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndvcmtlci1tYW5hZ2VyLmpzIl0sIm5hbWVzIjpbIkJyb3dzZXJXaW5kb3ciLCJyZW1vdGUiLCJXb3JrZXJNYW5hZ2VyIiwiZ2V0SW5zdGFuY2UiLCJpbnN0YW5jZSIsInJlc2V0IiwiZm9yY2UiLCJkZXN0cm95IiwiY29uc3RydWN0b3IiLCJ3b3JrZXJzIiwiU2V0IiwiYWN0aXZlV29ya2VyIiwiY3JlYXRlTmV3V29ya2VyIiwiaXNSZWFkeSIsInJlcXVlc3QiLCJkYXRhIiwiZGVzdHJveWVkIiwiRXJyb3IiLCJvcGVyYXRpb24iLCJyZXF1ZXN0UHJvbWlzZSIsIlByb21pc2UiLCJyZXNvbHZlIiwicmVqZWN0IiwiT3BlcmF0aW9uIiwiZXhlY3V0ZU9wZXJhdGlvbiIsInNldFByb21pc2UiLCJjYW5jZWwiLCJjYW5jZWxPcGVyYXRpb24iLCJwcm9taXNlIiwib3BlcmF0aW9uQ291bnRMaW1pdCIsIldvcmtlciIsIm9uRGVzdHJveWVkIiwib25DcmFzaGVkIiwib25TaWNrIiwiYWRkIiwiZGVzdHJveWVkV29ya2VyIiwiY3Jhc2hlZFdvcmtlciIsImdldEFjdGl2ZVdvcmtlciIsImdldE9wZXJhdGlvbkNvdW50TGltaXQiLCJnZXRSZW1haW5pbmdPcGVyYXRpb25zIiwiZm9yRWFjaCIsInNpY2tXb3JrZXIiLCJhdG9tIiwiaW5TcGVjTW9kZSIsImNvbnNvbGUiLCJ3YXJuIiwiZ2V0Q29tcGxldGVkT3BlcmF0aW9uQ291bnQiLCJjYWxjdWxhdGVOZXdPcGVyYXRpb25Db3VudExpbWl0IiwibGFzdFdvcmtlciIsIk1hdGgiLCJtaW4iLCJnZXRSZWFkeVByb21pc2UiLCJ3b3JrZXIiLCJvcGVyYXRpb25zQnlJZCIsIk1hcCIsImNvbXBsZXRlZE9wZXJhdGlvbkNvdW50Iiwic2ljayIsInJlbmRlcmVyUHJvY2VzcyIsIlJlbmRlcmVyUHJvY2VzcyIsImxvYWRVcmwiLCJnZXRMb2FkVXJsIiwib25EYXRhIiwiaGFuZGxlRGF0YVJlY2VpdmVkIiwib25DYW5jZWxsZWQiLCJoYW5kbGVDYW5jZWxsZWQiLCJvbkV4ZWNTdGFydGVkIiwiaGFuZGxlRXhlY1N0YXJ0ZWQiLCJvblNwYXduRXJyb3IiLCJoYW5kbGVTcGF3bkVycm9yIiwib25TdGRpbkVycm9yIiwiaGFuZGxlU3RkaW5FcnJvciIsImhhbmRsZVNpY2siLCJoYW5kbGVDcmFzaGVkIiwiaHRtbFBhdGgiLCJwYXRoIiwiam9pbiIsInJlbmRlcmVySnNQYXRoIiwicXMiLCJxdWVyeXN0cmluZyIsInN0cmluZ2lmeSIsImpzIiwibWFuYWdlcldlYkNvbnRlbnRzSWQiLCJnZXRXZWJDb250ZW50c0lkIiwiY2hhbm5lbE5hbWUiLCJnZXRDdXJyZW50V2ViQ29udGVudHMiLCJpZCIsInNldCIsIm9uQ29tcGxldGUiLCJvbk9wZXJhdGlvbkNvbXBsZXRlIiwicmVzdWx0cyIsImdldCIsImNvbXBsZXRlIiwidGltaW5nIiwidG90YWxJbnRlcm5hbFRpbWUiLCJleGVjVGltZSIsInNwYXduVGltZSIsImlwY1RpbWUiLCJnZXRFeGVjdXRpb25UaW1lIiwic2l6ZSIsIndhc0NhbmNlbGxlZCIsInNldEluUHJvZ3Jlc3MiLCJlcnIiLCJlcnJvciIsInN0ZGluIiwiQXJyYXkiLCJmcm9tIiwidmFsdWVzIiwiZ2V0UGlkIiwicmVtYWluaW5nT3BlcmF0aW9uUHJvbWlzZXMiLCJtYXAiLCJnZXRQcm9taXNlIiwiYWxsIiwid2luIiwic2hvdyIsInByb2Nlc3MiLCJlbnYiLCJBVE9NX0dJVEhVQl9TSE9XX1JFTkRFUkVSX1dJTkRPVyIsIndlYkNvbnRlbnRzIiwiZW1pdHRlciIsIkVtaXR0ZXIiLCJzdWJzY3JpcHRpb25zIiwiQ29tcG9zaXRlRGlzcG9zYWJsZSIsInJlZ2lzdGVyTGlzdGVuZXJzIiwibG9hZFVSTCIsIm9uIiwiaGFuZGxlRGVzdHJveSIsIkRpc3Bvc2FibGUiLCJpc0Rlc3Ryb3llZCIsInJlbW92ZUxpc3RlbmVyIiwicmVhZHkiLCJyZWFkeVByb21pc2UiLCJyZXNvbHZlUmVhZHkiLCJhcmdzIiwiaGFuZGxlTWVzc2FnZXMiLCJldmVudCIsInNvdXJjZVdlYkNvbnRlbnRzSWQiLCJ0eXBlIiwiZW1pdCIsImlwYyIsInBpZCIsImV4ZWN1dGUiLCJwYXlsb2FkIiwic2VuZCIsImRpc3Bvc2UiLCJjYW5jZWxsYXRpb25SZXNvbHZlIiwic3RhcnRUaW1lIiwiZW5kVGltZSIsInN0YXR1cyIsIlBFTkRJTkciLCJjYiIsIklOUFJPR1JFU1MiLCJOYU4iLCJtdXRhdGUiLCJwZXJmb3JtYW5jZSIsIm5vdyIsIkNPTVBMRVRFIiwiQ0FOQ0VMTEVEIiwibWVzc2FnZSIsImZpbGVOYW1lIiwibGluZU51bWJlciIsInN0YWNrIiwiZXhlY0ZuIiwiQ0FOQ0VMTElORyIsIlN5bWJvbCJdLCJtYXBwaW5ncyI6Ijs7Ozs7OztBQUFBOztBQUNBOztBQUVBOztBQUVBOztBQUVBOzs7Ozs7OztBQUhBLE1BQU07QUFBQ0EsRUFBQUE7QUFBRCxJQUFrQkMsZ0JBQXhCOztBQUtlLE1BQU1DLGFBQU4sQ0FBb0I7QUFHakMsU0FBT0MsV0FBUCxHQUFxQjtBQUNuQixRQUFJLENBQUMsS0FBS0MsUUFBVixFQUFvQjtBQUNsQixXQUFLQSxRQUFMLEdBQWdCLElBQUlGLGFBQUosRUFBaEI7QUFDRDs7QUFDRCxXQUFPLEtBQUtFLFFBQVo7QUFDRDs7QUFFRCxTQUFPQyxLQUFQLENBQWFDLEtBQWIsRUFBb0I7QUFDbEIsUUFBSSxLQUFLRixRQUFULEVBQW1CO0FBQUUsV0FBS0EsUUFBTCxDQUFjRyxPQUFkLENBQXNCRCxLQUF0QjtBQUErQjs7QUFDcEQsU0FBS0YsUUFBTCxHQUFnQixJQUFoQjtBQUNEOztBQUVESSxFQUFBQSxXQUFXLEdBQUc7QUFDWiwyQkFBUyxJQUFULEVBQWUsYUFBZixFQUE4QixXQUE5QixFQUEyQyxRQUEzQztBQUVBLFNBQUtDLE9BQUwsR0FBZSxJQUFJQyxHQUFKLEVBQWY7QUFDQSxTQUFLQyxZQUFMLEdBQW9CLElBQXBCO0FBQ0EsU0FBS0MsZUFBTDtBQUNEOztBQUVEQyxFQUFBQSxPQUFPLEdBQUc7QUFDUixXQUFPLEtBQUtGLFlBQUwsQ0FBa0JFLE9BQWxCLEVBQVA7QUFDRDs7QUFFREMsRUFBQUEsT0FBTyxDQUFDQyxJQUFELEVBQU87QUFDWixRQUFJLEtBQUtDLFNBQVQsRUFBb0I7QUFBRSxZQUFNLElBQUlDLEtBQUosQ0FBVSxxQkFBVixDQUFOO0FBQXlDOztBQUMvRCxRQUFJQyxTQUFKO0FBQ0EsVUFBTUMsY0FBYyxHQUFHLElBQUlDLE9BQUosQ0FBWSxDQUFDQyxPQUFELEVBQVVDLE1BQVYsS0FBcUI7QUFDdERKLE1BQUFBLFNBQVMsR0FBRyxJQUFJSyxTQUFKLENBQWNSLElBQWQsRUFBb0JNLE9BQXBCLEVBQTZCQyxNQUE3QixDQUFaO0FBQ0EsYUFBTyxLQUFLWCxZQUFMLENBQWtCYSxnQkFBbEIsQ0FBbUNOLFNBQW5DLENBQVA7QUFDRCxLQUhzQixDQUF2QjtBQUlBQSxJQUFBQSxTQUFTLENBQUNPLFVBQVYsQ0FBcUJOLGNBQXJCO0FBQ0EsV0FBTztBQUNMTyxNQUFBQSxNQUFNLEVBQUUsTUFBTSxLQUFLZixZQUFMLENBQWtCZ0IsZUFBbEIsQ0FBa0NULFNBQWxDLENBRFQ7QUFFTFUsTUFBQUEsT0FBTyxFQUFFVDtBQUZKLEtBQVA7QUFJRDs7QUFFRFAsRUFBQUEsZUFBZSxDQUFDO0FBQUNpQixJQUFBQTtBQUFELE1BQXdCO0FBQUNBLElBQUFBLG1CQUFtQixFQUFFO0FBQXRCLEdBQXpCLEVBQW9EO0FBQ2pFLFFBQUksS0FBS2IsU0FBVCxFQUFvQjtBQUFFO0FBQVM7O0FBQy9CLFNBQUtMLFlBQUwsR0FBb0IsSUFBSW1CLE1BQUosQ0FBVztBQUM3QkQsTUFBQUEsbUJBRDZCO0FBRTdCRSxNQUFBQSxXQUFXLEVBQUUsS0FBS0EsV0FGVztBQUc3QkMsTUFBQUEsU0FBUyxFQUFFLEtBQUtBLFNBSGE7QUFJN0JDLE1BQUFBLE1BQU0sRUFBRSxLQUFLQTtBQUpnQixLQUFYLENBQXBCO0FBTUEsU0FBS3hCLE9BQUwsQ0FBYXlCLEdBQWIsQ0FBaUIsS0FBS3ZCLFlBQXRCO0FBQ0Q7O0FBRURvQixFQUFBQSxXQUFXLENBQUNJLGVBQUQsRUFBa0I7QUFDM0IsU0FBSzFCLE9BQUwsV0FBb0IwQixlQUFwQjtBQUNEOztBQUVESCxFQUFBQSxTQUFTLENBQUNJLGFBQUQsRUFBZ0I7QUFDdkIsUUFBSUEsYUFBYSxLQUFLLEtBQUtDLGVBQUwsRUFBdEIsRUFBOEM7QUFDNUMsV0FBS3pCLGVBQUwsQ0FBcUI7QUFBQ2lCLFFBQUFBLG1CQUFtQixFQUFFTyxhQUFhLENBQUNFLHNCQUFkO0FBQXRCLE9BQXJCO0FBQ0Q7O0FBQ0RGLElBQUFBLGFBQWEsQ0FBQ0csc0JBQWQsR0FBdUNDLE9BQXZDLENBQStDdEIsU0FBUyxJQUFJLEtBQUtQLFlBQUwsQ0FBa0JhLGdCQUFsQixDQUFtQ04sU0FBbkMsQ0FBNUQ7QUFDRDs7QUFFRGUsRUFBQUEsTUFBTSxDQUFDUSxVQUFELEVBQWE7QUFDakIsUUFBSSxDQUFDQyxJQUFJLENBQUNDLFVBQUwsRUFBTCxFQUF3QjtBQUN0QjtBQUNBQyxNQUFBQSxPQUFPLENBQUNDLElBQVIsQ0FBYzsrQkFDV0osVUFBVSxDQUFDSCxzQkFBWCxFQUFvQztxQ0FDOUJHLFVBQVUsQ0FBQ0ssMEJBQVgsRUFBd0MsRUFGdkU7QUFHRDs7QUFDRCxVQUFNakIsbUJBQW1CLEdBQUcsS0FBS2tCLCtCQUFMLENBQXFDTixVQUFyQyxDQUE1QjtBQUNBLFdBQU8sS0FBSzdCLGVBQUwsQ0FBcUI7QUFBQ2lCLE1BQUFBO0FBQUQsS0FBckIsQ0FBUDtBQUNEOztBQUVEa0IsRUFBQUEsK0JBQStCLENBQUNDLFVBQUQsRUFBYTtBQUMxQyxRQUFJbkIsbUJBQW1CLEdBQUcsRUFBMUI7O0FBQ0EsUUFBSW1CLFVBQVUsQ0FBQ1Ysc0JBQVgsTUFBdUNVLFVBQVUsQ0FBQ0YsMEJBQVgsRUFBM0MsRUFBb0Y7QUFDbEZqQixNQUFBQSxtQkFBbUIsR0FBR29CLElBQUksQ0FBQ0MsR0FBTCxDQUFTRixVQUFVLENBQUNWLHNCQUFYLEtBQXNDLENBQS9DLEVBQWtELEdBQWxELENBQXRCO0FBQ0Q7O0FBQ0QsV0FBT1QsbUJBQVA7QUFDRDs7QUFFRFEsRUFBQUEsZUFBZSxHQUFHO0FBQ2hCLFdBQU8sS0FBSzFCLFlBQVo7QUFDRDs7QUFFRHdDLEVBQUFBLGVBQWUsR0FBRztBQUNoQixXQUFPLEtBQUt4QyxZQUFMLENBQWtCd0MsZUFBbEIsRUFBUDtBQUNEOztBQUVENUMsRUFBQUEsT0FBTyxDQUFDRCxLQUFELEVBQVE7QUFDYixTQUFLVSxTQUFMLEdBQWlCLElBQWpCO0FBQ0EsU0FBS1AsT0FBTCxDQUFhK0IsT0FBYixDQUFxQlksTUFBTSxJQUFJQSxNQUFNLENBQUM3QyxPQUFQLENBQWVELEtBQWYsQ0FBL0I7QUFDRDs7QUE3RmdDOzs7O2dCQUFkSixhLGNBQ0QsSTs7QUFnR2IsTUFBTTRCLE1BQU4sQ0FBYTtBQUdsQnRCLEVBQUFBLFdBQVcsQ0FBQztBQUFDcUIsSUFBQUEsbUJBQUQ7QUFBc0JJLElBQUFBLE1BQXRCO0FBQThCRCxJQUFBQSxTQUE5QjtBQUF5Q0QsSUFBQUE7QUFBekMsR0FBRCxFQUF3RDtBQUNqRSwyQkFDRSxJQURGLEVBRUUsb0JBRkYsRUFFd0IscUJBRnhCLEVBRStDLGlCQUYvQyxFQUVrRSxtQkFGbEUsRUFFdUYsa0JBRnZGLEVBR0Usa0JBSEYsRUFHc0IsWUFIdEIsRUFHb0MsZUFIcEM7QUFNQSxTQUFLRixtQkFBTCxHQUEyQkEsbUJBQTNCO0FBQ0EsU0FBS0ksTUFBTCxHQUFjQSxNQUFkO0FBQ0EsU0FBS0QsU0FBTCxHQUFpQkEsU0FBakI7QUFDQSxTQUFLRCxXQUFMLEdBQW1CQSxXQUFuQjtBQUVBLFNBQUtzQixjQUFMLEdBQXNCLElBQUlDLEdBQUosRUFBdEI7QUFDQSxTQUFLQyx1QkFBTCxHQUErQixDQUEvQjtBQUNBLFNBQUtDLElBQUwsR0FBWSxLQUFaO0FBRUEsU0FBS0MsZUFBTCxHQUF1QixJQUFJQyxlQUFKLENBQW9CO0FBQ3pDQyxNQUFBQSxPQUFPLEVBQUUsS0FBS0MsVUFBTCxDQUFnQi9CLG1CQUFoQixDQURnQztBQUV6Q2dDLE1BQUFBLE1BQU0sRUFBRSxLQUFLQyxrQkFGNEI7QUFHekNDLE1BQUFBLFdBQVcsRUFBRSxLQUFLQyxlQUh1QjtBQUl6Q0MsTUFBQUEsYUFBYSxFQUFFLEtBQUtDLGlCQUpxQjtBQUt6Q0MsTUFBQUEsWUFBWSxFQUFFLEtBQUtDLGdCQUxzQjtBQU16Q0MsTUFBQUEsWUFBWSxFQUFFLEtBQUtDLGdCQU5zQjtBQU96Q3JDLE1BQUFBLE1BQU0sRUFBRSxLQUFLc0MsVUFQNEI7QUFRekN2QyxNQUFBQSxTQUFTLEVBQUUsS0FBS3dDLGFBUnlCO0FBU3pDekMsTUFBQUEsV0FBVyxFQUFFLEtBQUt4QjtBQVR1QixLQUFwQixDQUF2QjtBQVdEOztBQUVETSxFQUFBQSxPQUFPLEdBQUc7QUFDUixXQUFPLEtBQUs0QyxlQUFMLENBQXFCNUMsT0FBckIsRUFBUDtBQUNEOztBQUVEK0MsRUFBQUEsVUFBVSxDQUFDL0IsbUJBQUQsRUFBc0I7QUFDOUIsVUFBTTRDLFFBQVEsR0FBR0MsaUJBQUtDLElBQUwsQ0FBVSw4QkFBVixFQUE0QixLQUE1QixFQUFtQyxlQUFuQyxDQUFqQjs7QUFDQSxVQUFNQyxjQUFjLEdBQUdGLGlCQUFLQyxJQUFMLENBQVUsOEJBQVYsRUFBNEIsS0FBNUIsRUFBbUMsV0FBbkMsQ0FBdkI7O0FBQ0EsVUFBTUUsRUFBRSxHQUFHQyx3QkFBWUMsU0FBWixDQUFzQjtBQUMvQkMsTUFBQUEsRUFBRSxFQUFFSixjQUQyQjtBQUUvQkssTUFBQUEsb0JBQW9CLEVBQUUsS0FBS0MsZ0JBQUwsRUFGUztBQUcvQnJELE1BQUFBLG1CQUgrQjtBQUkvQnNELE1BQUFBLFdBQVcsRUFBRXJELE1BQU0sQ0FBQ3FEO0FBSlcsS0FBdEIsQ0FBWDs7QUFNQSxXQUFRLFVBQVNWLFFBQVMsSUFBR0ksRUFBRyxFQUFoQztBQUNEOztBQUVESyxFQUFBQSxnQkFBZ0IsR0FBRztBQUNqQixXQUFPakYsaUJBQU9tRixxQkFBUCxHQUErQkMsRUFBdEM7QUFDRDs7QUFFRDdELEVBQUFBLGdCQUFnQixDQUFDTixTQUFELEVBQVk7QUFDMUIsU0FBS21DLGNBQUwsQ0FBb0JpQyxHQUFwQixDQUF3QnBFLFNBQVMsQ0FBQ21FLEVBQWxDLEVBQXNDbkUsU0FBdEM7QUFDQUEsSUFBQUEsU0FBUyxDQUFDcUUsVUFBVixDQUFxQixLQUFLQyxtQkFBMUI7QUFDQSxXQUFPLEtBQUsvQixlQUFMLENBQXFCakMsZ0JBQXJCLENBQXNDTixTQUF0QyxDQUFQO0FBQ0Q7O0FBRURTLEVBQUFBLGVBQWUsQ0FBQ1QsU0FBRCxFQUFZO0FBQ3pCLFdBQU8sS0FBS3VDLGVBQUwsQ0FBcUI5QixlQUFyQixDQUFxQ1QsU0FBckMsQ0FBUDtBQUNEOztBQUVENEMsRUFBQUEsa0JBQWtCLENBQUM7QUFBQ3VCLElBQUFBLEVBQUQ7QUFBS0ksSUFBQUE7QUFBTCxHQUFELEVBQWdCO0FBQ2hDLFVBQU12RSxTQUFTLEdBQUcsS0FBS21DLGNBQUwsQ0FBb0JxQyxHQUFwQixDQUF3QkwsRUFBeEIsQ0FBbEI7QUFDQW5FLElBQUFBLFNBQVMsQ0FBQ3lFLFFBQVYsQ0FBbUJGLE9BQW5CLEVBQTRCMUUsSUFBSSxJQUFJO0FBQ2xDLFlBQU07QUFBQzZFLFFBQUFBO0FBQUQsVUFBVzdFLElBQWpCO0FBQ0EsWUFBTThFLGlCQUFpQixHQUFHRCxNQUFNLENBQUNFLFFBQVAsR0FBa0JGLE1BQU0sQ0FBQ0csU0FBbkQ7QUFDQSxZQUFNQyxPQUFPLEdBQUc5RSxTQUFTLENBQUMrRSxnQkFBVixLQUErQkosaUJBQS9DO0FBQ0E5RSxNQUFBQSxJQUFJLENBQUM2RSxNQUFMLENBQVlJLE9BQVosR0FBc0JBLE9BQXRCO0FBQ0EsYUFBT2pGLElBQVA7QUFDRCxLQU5EO0FBT0Q7O0FBRUR5RSxFQUFBQSxtQkFBbUIsQ0FBQ3RFLFNBQUQsRUFBWTtBQUM3QixTQUFLcUMsdUJBQUw7QUFDQSxTQUFLRixjQUFMLFdBQTJCbkMsU0FBUyxDQUFDbUUsRUFBckM7O0FBRUEsUUFBSSxLQUFLN0IsSUFBTCxJQUFhLEtBQUtILGNBQUwsQ0FBb0I2QyxJQUFwQixLQUE2QixDQUE5QyxFQUFpRDtBQUMvQyxXQUFLM0YsT0FBTDtBQUNEO0FBQ0Y7O0FBRUR5RCxFQUFBQSxlQUFlLENBQUM7QUFBQ3FCLElBQUFBO0FBQUQsR0FBRCxFQUFPO0FBQ3BCLFVBQU1uRSxTQUFTLEdBQUcsS0FBS21DLGNBQUwsQ0FBb0JxQyxHQUFwQixDQUF3QkwsRUFBeEIsQ0FBbEI7O0FBQ0EsUUFBSW5FLFNBQUosRUFBZTtBQUNiO0FBQ0FBLE1BQUFBLFNBQVMsQ0FBQ2lGLFlBQVY7QUFDRDtBQUNGOztBQUVEakMsRUFBQUEsaUJBQWlCLENBQUM7QUFBQ21CLElBQUFBO0FBQUQsR0FBRCxFQUFPO0FBQ3RCLFVBQU1uRSxTQUFTLEdBQUcsS0FBS21DLGNBQUwsQ0FBb0JxQyxHQUFwQixDQUF3QkwsRUFBeEIsQ0FBbEI7QUFDQW5FLElBQUFBLFNBQVMsQ0FBQ2tGLGFBQVY7QUFDRDs7QUFFRGhDLEVBQUFBLGdCQUFnQixDQUFDO0FBQUNpQixJQUFBQSxFQUFEO0FBQUtnQixJQUFBQTtBQUFMLEdBQUQsRUFBWTtBQUMxQixVQUFNbkYsU0FBUyxHQUFHLEtBQUttQyxjQUFMLENBQW9CcUMsR0FBcEIsQ0FBd0JMLEVBQXhCLENBQWxCO0FBQ0FuRSxJQUFBQSxTQUFTLENBQUNvRixLQUFWLENBQWdCRCxHQUFoQjtBQUNEOztBQUVEL0IsRUFBQUEsZ0JBQWdCLENBQUM7QUFBQ2UsSUFBQUEsRUFBRDtBQUFLa0IsSUFBQUEsS0FBTDtBQUFZRixJQUFBQTtBQUFaLEdBQUQsRUFBbUI7QUFDakMsVUFBTW5GLFNBQVMsR0FBRyxLQUFLbUMsY0FBTCxDQUFvQnFDLEdBQXBCLENBQXdCTCxFQUF4QixDQUFsQjtBQUNBbkUsSUFBQUEsU0FBUyxDQUFDb0YsS0FBVixDQUFnQkQsR0FBaEI7QUFDRDs7QUFFRDlCLEVBQUFBLFVBQVUsR0FBRztBQUNYLFNBQUtmLElBQUwsR0FBWSxJQUFaO0FBQ0EsU0FBS3ZCLE1BQUwsQ0FBWSxJQUFaO0FBQ0Q7O0FBRUR1QyxFQUFBQSxhQUFhLEdBQUc7QUFDZCxTQUFLeEMsU0FBTCxDQUFlLElBQWY7QUFDQSxTQUFLekIsT0FBTDtBQUNEOztBQUVEK0IsRUFBQUEsc0JBQXNCLEdBQUc7QUFDdkIsV0FBTyxLQUFLVCxtQkFBWjtBQUNEOztBQUVEaUIsRUFBQUEsMEJBQTBCLEdBQUc7QUFDM0IsV0FBTyxLQUFLUyx1QkFBWjtBQUNEOztBQUVEaEIsRUFBQUEsc0JBQXNCLEdBQUc7QUFDdkIsV0FBT2lFLEtBQUssQ0FBQ0MsSUFBTixDQUFXLEtBQUtwRCxjQUFMLENBQW9CcUQsTUFBcEIsRUFBWCxDQUFQO0FBQ0Q7O0FBRURDLEVBQUFBLE1BQU0sR0FBRztBQUNQLFdBQU8sS0FBS2xELGVBQUwsQ0FBcUJrRCxNQUFyQixFQUFQO0FBQ0Q7O0FBRUR4RCxFQUFBQSxlQUFlLEdBQUc7QUFDaEIsV0FBTyxLQUFLTSxlQUFMLENBQXFCTixlQUFyQixFQUFQO0FBQ0Q7O0FBRUQsUUFBTTVDLE9BQU4sQ0FBY0QsS0FBZCxFQUFxQjtBQUNuQixTQUFLeUIsV0FBTCxDQUFpQixJQUFqQjs7QUFDQSxRQUFJLEtBQUtzQixjQUFMLENBQW9CNkMsSUFBcEIsR0FBMkIsQ0FBM0IsSUFBZ0MsQ0FBQzVGLEtBQXJDLEVBQTRDO0FBQzFDLFlBQU1zRywwQkFBMEIsR0FBRyxLQUFLckUsc0JBQUwsR0FDaENzRSxHQURnQyxDQUM1QjNGLFNBQVMsSUFBSUEsU0FBUyxDQUFDNEYsVUFBVixZQUE2QixNQUFNLElBQW5DLENBRGUsQ0FBbkM7QUFFQSxZQUFNMUYsT0FBTyxDQUFDMkYsR0FBUixDQUFZSCwwQkFBWixDQUFOO0FBQ0Q7O0FBQ0QsU0FBS25ELGVBQUwsQ0FBcUJsRCxPQUFyQjtBQUNEOztBQS9JaUI7QUFtSnBCOzs7Ozs7O2dCQW5KYXVCLE0saUJBQ1UscUI7O0FBcUpoQixNQUFNNEIsZUFBTixDQUFzQjtBQUMzQmxELEVBQUFBLFdBQVcsQ0FBQztBQUFDbUQsSUFBQUEsT0FBRDtBQUNWNUIsSUFBQUEsV0FEVTtBQUNHQyxJQUFBQSxTQURIO0FBQ2NDLElBQUFBLE1BRGQ7QUFDc0I0QixJQUFBQSxNQUR0QjtBQUM4QkUsSUFBQUEsV0FEOUI7QUFDMkNJLElBQUFBLFlBRDNDO0FBQ3lERSxJQUFBQSxZQUR6RDtBQUN1RUosSUFBQUE7QUFEdkUsR0FBRCxFQUN3RjtBQUNqRywyQkFBUyxJQUFULEVBQWUsZUFBZjtBQUNBLFNBQUtsQyxXQUFMLEdBQW1CQSxXQUFuQjtBQUNBLFNBQUtDLFNBQUwsR0FBaUJBLFNBQWpCO0FBQ0EsU0FBS0MsTUFBTCxHQUFjQSxNQUFkO0FBQ0EsU0FBSzRCLE1BQUwsR0FBY0EsTUFBZDtBQUNBLFNBQUtFLFdBQUwsR0FBbUJBLFdBQW5CO0FBQ0EsU0FBS0ksWUFBTCxHQUFvQkEsWUFBcEI7QUFDQSxTQUFLRSxZQUFMLEdBQW9CQSxZQUFwQjtBQUNBLFNBQUtKLGFBQUwsR0FBcUJBLGFBQXJCO0FBRUEsU0FBSytDLEdBQUwsR0FBVyxJQUFJaEgsYUFBSixDQUFrQjtBQUFDaUgsTUFBQUEsSUFBSSxFQUFFLENBQUMsQ0FBQ0MsT0FBTyxDQUFDQyxHQUFSLENBQVlDO0FBQXJCLEtBQWxCLENBQVg7QUFDQSxTQUFLQyxXQUFMLEdBQW1CLEtBQUtMLEdBQUwsQ0FBU0ssV0FBNUIsQ0FaaUcsQ0Fhakc7O0FBRUEsU0FBS0MsT0FBTCxHQUFlLElBQUlDLGlCQUFKLEVBQWY7QUFDQSxTQUFLQyxhQUFMLEdBQXFCLElBQUlDLDZCQUFKLEVBQXJCO0FBQ0EsU0FBS0MsaUJBQUw7QUFFQSxTQUFLVixHQUFMLENBQVNXLE9BQVQsQ0FBaUJoRSxPQUFqQjtBQUNBLFNBQUtxRCxHQUFMLENBQVNLLFdBQVQsQ0FBcUJPLEVBQXJCLENBQXdCLFNBQXhCLEVBQW1DLEtBQUtDLGFBQXhDO0FBQ0EsU0FBS2IsR0FBTCxDQUFTSyxXQUFULENBQXFCTyxFQUFyQixDQUF3QixXQUF4QixFQUFxQyxLQUFLQyxhQUExQztBQUNBLFNBQUtMLGFBQUwsQ0FBbUJ0RixHQUFuQixDQUNFLElBQUk0RixvQkFBSixDQUFlLE1BQU07QUFDbkIsVUFBSSxDQUFDLEtBQUtkLEdBQUwsQ0FBU2UsV0FBVCxFQUFMLEVBQTZCO0FBQzNCLGFBQUtmLEdBQUwsQ0FBU0ssV0FBVCxDQUFxQlcsY0FBckIsQ0FBb0MsU0FBcEMsRUFBK0MsS0FBS0gsYUFBcEQ7QUFDQSxhQUFLYixHQUFMLENBQVNLLFdBQVQsQ0FBcUJXLGNBQXJCLENBQW9DLFdBQXBDLEVBQWlELEtBQUtILGFBQXREO0FBQ0EsYUFBS2IsR0FBTCxDQUFTekcsT0FBVDtBQUNEO0FBQ0YsS0FORCxDQURGLEVBUUUsS0FBSytHLE9BUlA7QUFXQSxTQUFLVyxLQUFMLEdBQWEsS0FBYjtBQUNBLFNBQUtDLFlBQUwsR0FBb0IsSUFBSTlHLE9BQUosQ0FBWUMsT0FBTyxJQUFJO0FBQUUsV0FBSzhHLFlBQUwsR0FBb0I5RyxPQUFwQjtBQUE4QixLQUF2RCxDQUFwQjtBQUNEOztBQUVEUixFQUFBQSxPQUFPLEdBQUc7QUFDUixXQUFPLEtBQUtvSCxLQUFaO0FBQ0Q7O0FBRURKLEVBQUFBLGFBQWEsQ0FBQyxHQUFHTyxJQUFKLEVBQVU7QUFDckIsU0FBSzdILE9BQUw7QUFDQSxTQUFLeUIsU0FBTCxDQUFlLEdBQUdvRyxJQUFsQjtBQUNEOztBQUVEVixFQUFBQSxpQkFBaUIsR0FBRztBQUNsQixVQUFNVyxjQUFjLEdBQUcsQ0FBQ0MsS0FBRCxFQUFRO0FBQUNDLE1BQUFBLG1CQUFEO0FBQXNCQyxNQUFBQSxJQUF0QjtBQUE0QnpILE1BQUFBO0FBQTVCLEtBQVIsS0FBOEM7QUFDbkUsVUFBSXdILG1CQUFtQixLQUFLLEtBQUt2QixHQUFMLENBQVNLLFdBQVQsQ0FBcUJoQyxFQUFqRCxFQUFxRDtBQUNuRCxhQUFLaUMsT0FBTCxDQUFhbUIsSUFBYixDQUFrQkQsSUFBbEIsRUFBd0J6SCxJQUF4QjtBQUNEO0FBQ0YsS0FKRDs7QUFNQTJILDBCQUFJZCxFQUFKLENBQU85RixNQUFNLENBQUNxRCxXQUFkLEVBQTJCa0QsY0FBM0I7O0FBQ0EsU0FBS2YsT0FBTCxDQUFhTSxFQUFiLENBQWdCLGdCQUFoQixFQUFrQyxDQUFDO0FBQUNlLE1BQUFBO0FBQUQsS0FBRCxLQUFXO0FBQzNDLFdBQUtBLEdBQUwsR0FBV0EsR0FBWDtBQUNBLFdBQUtWLEtBQUwsR0FBYSxJQUFiO0FBQ0EsV0FBS0UsWUFBTDtBQUNELEtBSkQ7QUFLQSxTQUFLYixPQUFMLENBQWFNLEVBQWIsQ0FBZ0IsVUFBaEIsRUFBNEIsS0FBSy9ELE1BQWpDO0FBQ0EsU0FBS3lELE9BQUwsQ0FBYU0sRUFBYixDQUFnQixlQUFoQixFQUFpQyxLQUFLN0QsV0FBdEM7QUFDQSxTQUFLdUQsT0FBTCxDQUFhTSxFQUFiLENBQWdCLGlCQUFoQixFQUFtQyxLQUFLekQsWUFBeEM7QUFDQSxTQUFLbUQsT0FBTCxDQUFhTSxFQUFiLENBQWdCLGlCQUFoQixFQUFtQyxLQUFLdkQsWUFBeEM7QUFDQSxTQUFLaUQsT0FBTCxDQUFhTSxFQUFiLENBQWdCLGFBQWhCLEVBQStCLEtBQUszRixNQUFwQyxFQWpCa0IsQ0FtQmxCO0FBQ0E7O0FBQ0EsU0FBS3FGLE9BQUwsQ0FBYU0sRUFBYixDQUFnQixjQUFoQixFQUFnQyxLQUFLM0QsYUFBckM7QUFFQSxTQUFLdUQsYUFBTCxDQUFtQnRGLEdBQW5CLENBQ0UsSUFBSTRGLG9CQUFKLENBQWUsTUFBTVksc0JBQUlWLGNBQUosQ0FBbUJsRyxNQUFNLENBQUNxRCxXQUExQixFQUF1Q2tELGNBQXZDLENBQXJCLENBREY7QUFHRDs7QUFFRDdHLEVBQUFBLGdCQUFnQixDQUFDTixTQUFELEVBQVk7QUFDMUIsV0FBT0EsU0FBUyxDQUFDMEgsT0FBVixDQUFrQkMsT0FBTyxJQUFJO0FBQ2xDLFVBQUksS0FBSzdILFNBQVQsRUFBb0I7QUFBRSxlQUFPLElBQVA7QUFBYzs7QUFDcEMsYUFBTyxLQUFLcUcsV0FBTCxDQUFpQnlCLElBQWpCLENBQXNCaEgsTUFBTSxDQUFDcUQsV0FBN0IsRUFBMEM7QUFDL0NxRCxRQUFBQSxJQUFJLEVBQUUsVUFEeUM7QUFFL0N6SCxRQUFBQSxJQUFJLEVBQUU4SDtBQUZ5QyxPQUExQyxDQUFQO0FBSUQsS0FOTSxDQUFQO0FBT0Q7O0FBRURsSCxFQUFBQSxlQUFlLENBQUNULFNBQUQsRUFBWTtBQUN6QixXQUFPQSxTQUFTLENBQUNRLE1BQVYsQ0FBaUJtSCxPQUFPLElBQUk7QUFDakMsVUFBSSxLQUFLN0gsU0FBVCxFQUFvQjtBQUFFLGVBQU8sSUFBUDtBQUFjOztBQUNwQyxhQUFPLEtBQUtxRyxXQUFMLENBQWlCeUIsSUFBakIsQ0FBc0JoSCxNQUFNLENBQUNxRCxXQUE3QixFQUEwQztBQUMvQ3FELFFBQUFBLElBQUksRUFBRSxZQUR5QztBQUUvQ3pILFFBQUFBLElBQUksRUFBRThIO0FBRnlDLE9BQTFDLENBQVA7QUFJRCxLQU5NLENBQVA7QUFPRDs7QUFFRGxDLEVBQUFBLE1BQU0sR0FBRztBQUNQLFdBQU8sS0FBS2dDLEdBQVo7QUFDRDs7QUFFRHhGLEVBQUFBLGVBQWUsR0FBRztBQUNoQixXQUFPLEtBQUsrRSxZQUFaO0FBQ0Q7O0FBRUQzSCxFQUFBQSxPQUFPLEdBQUc7QUFDUixTQUFLUyxTQUFMLEdBQWlCLElBQWpCO0FBQ0EsU0FBS3dHLGFBQUwsQ0FBbUJ1QixPQUFuQjtBQUNEOztBQTNHMEI7Ozs7QUErR3RCLE1BQU14SCxTQUFOLENBQWdCO0FBV3JCZixFQUFBQSxXQUFXLENBQUNPLElBQUQsRUFBT00sT0FBUCxFQUFnQkMsTUFBaEIsRUFBd0I7QUFDakMsU0FBSytELEVBQUwsR0FBVTlELFNBQVMsQ0FBQzhELEVBQVYsRUFBVjtBQUNBLFNBQUt0RSxJQUFMLEdBQVlBLElBQVo7QUFDQSxTQUFLTSxPQUFMLEdBQWVBLE9BQWY7QUFDQSxTQUFLQyxNQUFMLEdBQWNBLE1BQWQ7QUFDQSxTQUFLTSxPQUFMLEdBQWUsSUFBZjs7QUFDQSxTQUFLb0gsbUJBQUwsR0FBMkIsTUFBTSxDQUFFLENBQW5DOztBQUNBLFNBQUtDLFNBQUwsR0FBaUIsSUFBakI7QUFDQSxTQUFLQyxPQUFMLEdBQWUsSUFBZjtBQUNBLFNBQUtDLE1BQUwsR0FBYzVILFNBQVMsQ0FBQzRILE1BQVYsQ0FBaUJDLE9BQS9CO0FBQ0EsU0FBSzNELE9BQUwsR0FBZSxJQUFmO0FBQ0EsU0FBSzZCLE9BQUwsR0FBZSxJQUFJQyxpQkFBSixFQUFmO0FBQ0Q7O0FBRURoQyxFQUFBQSxVQUFVLENBQUM4RCxFQUFELEVBQUs7QUFDYixXQUFPLEtBQUsvQixPQUFMLENBQWFNLEVBQWIsQ0FBZ0IsVUFBaEIsRUFBNEJ5QixFQUE1QixDQUFQO0FBQ0Q7O0FBRUQ1SCxFQUFBQSxVQUFVLENBQUNHLE9BQUQsRUFBVTtBQUNsQixTQUFLQSxPQUFMLEdBQWVBLE9BQWY7QUFDRDs7QUFFRGtGLEVBQUFBLFVBQVUsR0FBRztBQUNYLFdBQU8sS0FBS2xGLE9BQVo7QUFDRDs7QUFFRHdFLEVBQUFBLGFBQWEsR0FBRztBQUNkO0FBQ0EsU0FBSytDLE1BQUwsR0FBYzVILFNBQVMsQ0FBQzRILE1BQVYsQ0FBaUJHLFVBQS9CO0FBQ0Q7O0FBRURyRCxFQUFBQSxnQkFBZ0IsR0FBRztBQUNqQixRQUFJLENBQUMsS0FBS2dELFNBQU4sSUFBbUIsQ0FBQyxLQUFLQyxPQUE3QixFQUFzQztBQUNwQyxhQUFPSyxHQUFQO0FBQ0QsS0FGRCxNQUVPO0FBQ0wsYUFBTyxLQUFLTCxPQUFMLEdBQWUsS0FBS0QsU0FBM0I7QUFDRDtBQUNGOztBQUVEdEQsRUFBQUEsUUFBUSxDQUFDRixPQUFELEVBQVUrRCxNQUFNLEdBQUd6SSxJQUFJLElBQUlBLElBQTNCLEVBQWlDO0FBQ3ZDLFNBQUttSSxPQUFMLEdBQWVPLFdBQVcsQ0FBQ0MsR0FBWixFQUFmO0FBQ0EsU0FBS2pFLE9BQUwsR0FBZUEsT0FBZjtBQUNBLFNBQUtwRSxPQUFMLENBQWFtSSxNQUFNLENBQUMvRCxPQUFELENBQW5CO0FBQ0EsU0FBS3VELG1CQUFMO0FBQ0EsU0FBS0csTUFBTCxHQUFjNUgsU0FBUyxDQUFDNEgsTUFBVixDQUFpQlEsUUFBL0I7QUFDQSxTQUFLckMsT0FBTCxDQUFhbUIsSUFBYixDQUFrQixVQUFsQixFQUE4QixJQUE5QjtBQUNBLFNBQUtuQixPQUFMLENBQWF5QixPQUFiO0FBQ0Q7O0FBRUQ1QyxFQUFBQSxZQUFZLEdBQUc7QUFDYixTQUFLZ0QsTUFBTCxHQUFjNUgsU0FBUyxDQUFDNEgsTUFBVixDQUFpQlMsU0FBL0I7QUFDQSxTQUFLWixtQkFBTDtBQUNEOztBQUVEMUMsRUFBQUEsS0FBSyxDQUFDYixPQUFELEVBQVU7QUFDYixTQUFLeUQsT0FBTCxHQUFlTyxXQUFXLENBQUNDLEdBQVosRUFBZjtBQUNBLFVBQU1yRCxHQUFHLEdBQUcsSUFBSXBGLEtBQUosQ0FBVXdFLE9BQU8sQ0FBQ29FLE9BQWxCLEVBQTJCcEUsT0FBTyxDQUFDcUUsUUFBbkMsRUFBNkNyRSxPQUFPLENBQUNzRSxVQUFyRCxDQUFaO0FBQ0ExRCxJQUFBQSxHQUFHLENBQUMyRCxLQUFKLEdBQVl2RSxPQUFPLENBQUN1RSxLQUFwQjtBQUNBLFNBQUsxSSxNQUFMLENBQVkrRSxHQUFaO0FBQ0Q7O0FBRUR1QyxFQUFBQSxPQUFPLENBQUNxQixNQUFELEVBQVM7QUFDZCxTQUFLaEIsU0FBTCxHQUFpQlEsV0FBVyxDQUFDQyxHQUFaLEVBQWpCO0FBQ0EsV0FBT08sTUFBTSxtQkFBSyxLQUFLbEosSUFBVjtBQUFnQnNFLE1BQUFBLEVBQUUsRUFBRSxLQUFLQTtBQUF6QixPQUFiO0FBQ0Q7O0FBRUQzRCxFQUFBQSxNQUFNLENBQUN1SSxNQUFELEVBQVM7QUFDYixXQUFPLElBQUk3SSxPQUFKLENBQVlDLE9BQU8sSUFBSTtBQUM1QixXQUFLOEgsTUFBTCxHQUFjNUgsU0FBUyxDQUFDNEgsTUFBVixDQUFpQmUsVUFBL0I7QUFDQSxXQUFLbEIsbUJBQUwsR0FBMkIzSCxPQUEzQjtBQUNBNEksTUFBQUEsTUFBTSxDQUFDO0FBQUM1RSxRQUFBQSxFQUFFLEVBQUUsS0FBS0E7QUFBVixPQUFELENBQU47QUFDRCxLQUpNLENBQVA7QUFLRDs7QUFuRm9COzs7O2dCQUFWOUQsUyxZQUNLO0FBQ2Q2SCxFQUFBQSxPQUFPLEVBQUVlLE1BQU0sQ0FBQyxTQUFELENBREQ7QUFFZGIsRUFBQUEsVUFBVSxFQUFFYSxNQUFNLENBQUMsYUFBRCxDQUZKO0FBR2RSLEVBQUFBLFFBQVEsRUFBRVEsTUFBTSxDQUFDLFVBQUQsQ0FIRjtBQUlkRCxFQUFBQSxVQUFVLEVBQUVDLE1BQU0sQ0FBQyxZQUFELENBSko7QUFLZFAsRUFBQUEsU0FBUyxFQUFFTyxNQUFNLENBQUMsVUFBRDtBQUxILEM7O2dCQURMNUksUyxRQVNDLEMiLCJzb3VyY2VSb290IjoiL2J1aWxkL2F0b20vc3JjL2F0b20tMS4zNy4wL291dC9hcHAvbm9kZV9tb2R1bGVzL2dpdGh1YiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0IHF1ZXJ5c3RyaW5nIGZyb20gJ3F1ZXJ5c3RyaW5nJztcblxuaW1wb3J0IHtyZW1vdGUsIGlwY1JlbmRlcmVyIGFzIGlwY30gZnJvbSAnZWxlY3Ryb24nO1xuY29uc3Qge0Jyb3dzZXJXaW5kb3d9ID0gcmVtb3RlO1xuaW1wb3J0IHtFbWl0dGVyLCBEaXNwb3NhYmxlLCBDb21wb3NpdGVEaXNwb3NhYmxlfSBmcm9tICdldmVudC1raXQnO1xuXG5pbXBvcnQge2dldFBhY2thZ2VSb290LCBhdXRvYmluZH0gZnJvbSAnLi9oZWxwZXJzJztcblxuZXhwb3J0IGRlZmF1bHQgY2xhc3MgV29ya2VyTWFuYWdlciB7XG4gIHN0YXRpYyBpbnN0YW5jZSA9IG51bGw7XG5cbiAgc3RhdGljIGdldEluc3RhbmNlKCkge1xuICAgIGlmICghdGhpcy5pbnN0YW5jZSkge1xuICAgICAgdGhpcy5pbnN0YW5jZSA9IG5ldyBXb3JrZXJNYW5hZ2VyKCk7XG4gICAgfVxuICAgIHJldHVybiB0aGlzLmluc3RhbmNlO1xuICB9XG5cbiAgc3RhdGljIHJlc2V0KGZvcmNlKSB7XG4gICAgaWYgKHRoaXMuaW5zdGFuY2UpIHsgdGhpcy5pbnN0YW5jZS5kZXN0cm95KGZvcmNlKTsgfVxuICAgIHRoaXMuaW5zdGFuY2UgPSBudWxsO1xuICB9XG5cbiAgY29uc3RydWN0b3IoKSB7XG4gICAgYXV0b2JpbmQodGhpcywgJ29uRGVzdHJveWVkJywgJ29uQ3Jhc2hlZCcsICdvblNpY2snKTtcblxuICAgIHRoaXMud29ya2VycyA9IG5ldyBTZXQoKTtcbiAgICB0aGlzLmFjdGl2ZVdvcmtlciA9IG51bGw7XG4gICAgdGhpcy5jcmVhdGVOZXdXb3JrZXIoKTtcbiAgfVxuXG4gIGlzUmVhZHkoKSB7XG4gICAgcmV0dXJuIHRoaXMuYWN0aXZlV29ya2VyLmlzUmVhZHkoKTtcbiAgfVxuXG4gIHJlcXVlc3QoZGF0YSkge1xuICAgIGlmICh0aGlzLmRlc3Ryb3llZCkgeyB0aHJvdyBuZXcgRXJyb3IoJ1dvcmtlciBpcyBkZXN0cm95ZWQnKTsgfVxuICAgIGxldCBvcGVyYXRpb247XG4gICAgY29uc3QgcmVxdWVzdFByb21pc2UgPSBuZXcgUHJvbWlzZSgocmVzb2x2ZSwgcmVqZWN0KSA9PiB7XG4gICAgICBvcGVyYXRpb24gPSBuZXcgT3BlcmF0aW9uKGRhdGEsIHJlc29sdmUsIHJlamVjdCk7XG4gICAgICByZXR1cm4gdGhpcy5hY3RpdmVXb3JrZXIuZXhlY3V0ZU9wZXJhdGlvbihvcGVyYXRpb24pO1xuICAgIH0pO1xuICAgIG9wZXJhdGlvbi5zZXRQcm9taXNlKHJlcXVlc3RQcm9taXNlKTtcbiAgICByZXR1cm4ge1xuICAgICAgY2FuY2VsOiAoKSA9PiB0aGlzLmFjdGl2ZVdvcmtlci5jYW5jZWxPcGVyYXRpb24ob3BlcmF0aW9uKSxcbiAgICAgIHByb21pc2U6IHJlcXVlc3RQcm9taXNlLFxuICAgIH07XG4gIH1cblxuICBjcmVhdGVOZXdXb3JrZXIoe29wZXJhdGlvbkNvdW50TGltaXR9ID0ge29wZXJhdGlvbkNvdW50TGltaXQ6IDEwfSkge1xuICAgIGlmICh0aGlzLmRlc3Ryb3llZCkgeyByZXR1cm47IH1cbiAgICB0aGlzLmFjdGl2ZVdvcmtlciA9IG5ldyBXb3JrZXIoe1xuICAgICAgb3BlcmF0aW9uQ291bnRMaW1pdCxcbiAgICAgIG9uRGVzdHJveWVkOiB0aGlzLm9uRGVzdHJveWVkLFxuICAgICAgb25DcmFzaGVkOiB0aGlzLm9uQ3Jhc2hlZCxcbiAgICAgIG9uU2ljazogdGhpcy5vblNpY2ssXG4gICAgfSk7XG4gICAgdGhpcy53b3JrZXJzLmFkZCh0aGlzLmFjdGl2ZVdvcmtlcik7XG4gIH1cblxuICBvbkRlc3Ryb3llZChkZXN0cm95ZWRXb3JrZXIpIHtcbiAgICB0aGlzLndvcmtlcnMuZGVsZXRlKGRlc3Ryb3llZFdvcmtlcik7XG4gIH1cblxuICBvbkNyYXNoZWQoY3Jhc2hlZFdvcmtlcikge1xuICAgIGlmIChjcmFzaGVkV29ya2VyID09PSB0aGlzLmdldEFjdGl2ZVdvcmtlcigpKSB7XG4gICAgICB0aGlzLmNyZWF0ZU5ld1dvcmtlcih7b3BlcmF0aW9uQ291bnRMaW1pdDogY3Jhc2hlZFdvcmtlci5nZXRPcGVyYXRpb25Db3VudExpbWl0KCl9KTtcbiAgICB9XG4gICAgY3Jhc2hlZFdvcmtlci5nZXRSZW1haW5pbmdPcGVyYXRpb25zKCkuZm9yRWFjaChvcGVyYXRpb24gPT4gdGhpcy5hY3RpdmVXb3JrZXIuZXhlY3V0ZU9wZXJhdGlvbihvcGVyYXRpb24pKTtcbiAgfVxuXG4gIG9uU2ljayhzaWNrV29ya2VyKSB7XG4gICAgaWYgKCFhdG9tLmluU3BlY01vZGUoKSkge1xuICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLWNvbnNvbGVcbiAgICAgIGNvbnNvbGUud2FybihgU2ljayB3b3JrZXIgZGV0ZWN0ZWQuXG4gICAgICAgIG9wZXJhdGlvbkNvdW50TGltaXQ6ICR7c2lja1dvcmtlci5nZXRPcGVyYXRpb25Db3VudExpbWl0KCl9LFxuICAgICAgICBjb21wbGV0ZWQgb3BlcmF0aW9uIGNvdW50OiAke3NpY2tXb3JrZXIuZ2V0Q29tcGxldGVkT3BlcmF0aW9uQ291bnQoKX1gKTtcbiAgICB9XG4gICAgY29uc3Qgb3BlcmF0aW9uQ291bnRMaW1pdCA9IHRoaXMuY2FsY3VsYXRlTmV3T3BlcmF0aW9uQ291bnRMaW1pdChzaWNrV29ya2VyKTtcbiAgICByZXR1cm4gdGhpcy5jcmVhdGVOZXdXb3JrZXIoe29wZXJhdGlvbkNvdW50TGltaXR9KTtcbiAgfVxuXG4gIGNhbGN1bGF0ZU5ld09wZXJhdGlvbkNvdW50TGltaXQobGFzdFdvcmtlcikge1xuICAgIGxldCBvcGVyYXRpb25Db3VudExpbWl0ID0gMTA7XG4gICAgaWYgKGxhc3RXb3JrZXIuZ2V0T3BlcmF0aW9uQ291bnRMaW1pdCgpID49IGxhc3RXb3JrZXIuZ2V0Q29tcGxldGVkT3BlcmF0aW9uQ291bnQoKSkge1xuICAgICAgb3BlcmF0aW9uQ291bnRMaW1pdCA9IE1hdGgubWluKGxhc3RXb3JrZXIuZ2V0T3BlcmF0aW9uQ291bnRMaW1pdCgpICogMiwgMTAwKTtcbiAgICB9XG4gICAgcmV0dXJuIG9wZXJhdGlvbkNvdW50TGltaXQ7XG4gIH1cblxuICBnZXRBY3RpdmVXb3JrZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMuYWN0aXZlV29ya2VyO1xuICB9XG5cbiAgZ2V0UmVhZHlQcm9taXNlKCkge1xuICAgIHJldHVybiB0aGlzLmFjdGl2ZVdvcmtlci5nZXRSZWFkeVByb21pc2UoKTtcbiAgfVxuXG4gIGRlc3Ryb3koZm9yY2UpIHtcbiAgICB0aGlzLmRlc3Ryb3llZCA9IHRydWU7XG4gICAgdGhpcy53b3JrZXJzLmZvckVhY2god29ya2VyID0+IHdvcmtlci5kZXN0cm95KGZvcmNlKSk7XG4gIH1cbn1cblxuXG5leHBvcnQgY2xhc3MgV29ya2VyIHtcbiAgc3RhdGljIGNoYW5uZWxOYW1lID0gJ2dpdGh1YjpyZW5kZXJlci1pcGMnO1xuXG4gIGNvbnN0cnVjdG9yKHtvcGVyYXRpb25Db3VudExpbWl0LCBvblNpY2ssIG9uQ3Jhc2hlZCwgb25EZXN0cm95ZWR9KSB7XG4gICAgYXV0b2JpbmQoXG4gICAgICB0aGlzLFxuICAgICAgJ2hhbmRsZURhdGFSZWNlaXZlZCcsICdvbk9wZXJhdGlvbkNvbXBsZXRlJywgJ2hhbmRsZUNhbmNlbGxlZCcsICdoYW5kbGVFeGVjU3RhcnRlZCcsICdoYW5kbGVTcGF3bkVycm9yJyxcbiAgICAgICdoYW5kbGVTdGRpbkVycm9yJywgJ2hhbmRsZVNpY2snLCAnaGFuZGxlQ3Jhc2hlZCcsXG4gICAgKTtcblxuICAgIHRoaXMub3BlcmF0aW9uQ291bnRMaW1pdCA9IG9wZXJhdGlvbkNvdW50TGltaXQ7XG4gICAgdGhpcy5vblNpY2sgPSBvblNpY2s7XG4gICAgdGhpcy5vbkNyYXNoZWQgPSBvbkNyYXNoZWQ7XG4gICAgdGhpcy5vbkRlc3Ryb3llZCA9IG9uRGVzdHJveWVkO1xuXG4gICAgdGhpcy5vcGVyYXRpb25zQnlJZCA9IG5ldyBNYXAoKTtcbiAgICB0aGlzLmNvbXBsZXRlZE9wZXJhdGlvbkNvdW50ID0gMDtcbiAgICB0aGlzLnNpY2sgPSBmYWxzZTtcblxuICAgIHRoaXMucmVuZGVyZXJQcm9jZXNzID0gbmV3IFJlbmRlcmVyUHJvY2Vzcyh7XG4gICAgICBsb2FkVXJsOiB0aGlzLmdldExvYWRVcmwob3BlcmF0aW9uQ291bnRMaW1pdCksXG4gICAgICBvbkRhdGE6IHRoaXMuaGFuZGxlRGF0YVJlY2VpdmVkLFxuICAgICAgb25DYW5jZWxsZWQ6IHRoaXMuaGFuZGxlQ2FuY2VsbGVkLFxuICAgICAgb25FeGVjU3RhcnRlZDogdGhpcy5oYW5kbGVFeGVjU3RhcnRlZCxcbiAgICAgIG9uU3Bhd25FcnJvcjogdGhpcy5oYW5kbGVTcGF3bkVycm9yLFxuICAgICAgb25TdGRpbkVycm9yOiB0aGlzLmhhbmRsZVN0ZGluRXJyb3IsXG4gICAgICBvblNpY2s6IHRoaXMuaGFuZGxlU2ljayxcbiAgICAgIG9uQ3Jhc2hlZDogdGhpcy5oYW5kbGVDcmFzaGVkLFxuICAgICAgb25EZXN0cm95ZWQ6IHRoaXMuZGVzdHJveSxcbiAgICB9KTtcbiAgfVxuXG4gIGlzUmVhZHkoKSB7XG4gICAgcmV0dXJuIHRoaXMucmVuZGVyZXJQcm9jZXNzLmlzUmVhZHkoKTtcbiAgfVxuXG4gIGdldExvYWRVcmwob3BlcmF0aW9uQ291bnRMaW1pdCkge1xuICAgIGNvbnN0IGh0bWxQYXRoID0gcGF0aC5qb2luKGdldFBhY2thZ2VSb290KCksICdsaWInLCAncmVuZGVyZXIuaHRtbCcpO1xuICAgIGNvbnN0IHJlbmRlcmVySnNQYXRoID0gcGF0aC5qb2luKGdldFBhY2thZ2VSb290KCksICdsaWInLCAnd29ya2VyLmpzJyk7XG4gICAgY29uc3QgcXMgPSBxdWVyeXN0cmluZy5zdHJpbmdpZnkoe1xuICAgICAganM6IHJlbmRlcmVySnNQYXRoLFxuICAgICAgbWFuYWdlcldlYkNvbnRlbnRzSWQ6IHRoaXMuZ2V0V2ViQ29udGVudHNJZCgpLFxuICAgICAgb3BlcmF0aW9uQ291bnRMaW1pdCxcbiAgICAgIGNoYW5uZWxOYW1lOiBXb3JrZXIuY2hhbm5lbE5hbWUsXG4gICAgfSk7XG4gICAgcmV0dXJuIGBmaWxlOi8vJHtodG1sUGF0aH0/JHtxc31gO1xuICB9XG5cbiAgZ2V0V2ViQ29udGVudHNJZCgpIHtcbiAgICByZXR1cm4gcmVtb3RlLmdldEN1cnJlbnRXZWJDb250ZW50cygpLmlkO1xuICB9XG5cbiAgZXhlY3V0ZU9wZXJhdGlvbihvcGVyYXRpb24pIHtcbiAgICB0aGlzLm9wZXJhdGlvbnNCeUlkLnNldChvcGVyYXRpb24uaWQsIG9wZXJhdGlvbik7XG4gICAgb3BlcmF0aW9uLm9uQ29tcGxldGUodGhpcy5vbk9wZXJhdGlvbkNvbXBsZXRlKTtcbiAgICByZXR1cm4gdGhpcy5yZW5kZXJlclByb2Nlc3MuZXhlY3V0ZU9wZXJhdGlvbihvcGVyYXRpb24pO1xuICB9XG5cbiAgY2FuY2VsT3BlcmF0aW9uKG9wZXJhdGlvbikge1xuICAgIHJldHVybiB0aGlzLnJlbmRlcmVyUHJvY2Vzcy5jYW5jZWxPcGVyYXRpb24ob3BlcmF0aW9uKTtcbiAgfVxuXG4gIGhhbmRsZURhdGFSZWNlaXZlZCh7aWQsIHJlc3VsdHN9KSB7XG4gICAgY29uc3Qgb3BlcmF0aW9uID0gdGhpcy5vcGVyYXRpb25zQnlJZC5nZXQoaWQpO1xuICAgIG9wZXJhdGlvbi5jb21wbGV0ZShyZXN1bHRzLCBkYXRhID0+IHtcbiAgICAgIGNvbnN0IHt0aW1pbmd9ID0gZGF0YTtcbiAgICAgIGNvbnN0IHRvdGFsSW50ZXJuYWxUaW1lID0gdGltaW5nLmV4ZWNUaW1lICsgdGltaW5nLnNwYXduVGltZTtcbiAgICAgIGNvbnN0IGlwY1RpbWUgPSBvcGVyYXRpb24uZ2V0RXhlY3V0aW9uVGltZSgpIC0gdG90YWxJbnRlcm5hbFRpbWU7XG4gICAgICBkYXRhLnRpbWluZy5pcGNUaW1lID0gaXBjVGltZTtcbiAgICAgIHJldHVybiBkYXRhO1xuICAgIH0pO1xuICB9XG5cbiAgb25PcGVyYXRpb25Db21wbGV0ZShvcGVyYXRpb24pIHtcbiAgICB0aGlzLmNvbXBsZXRlZE9wZXJhdGlvbkNvdW50Kys7XG4gICAgdGhpcy5vcGVyYXRpb25zQnlJZC5kZWxldGUob3BlcmF0aW9uLmlkKTtcblxuICAgIGlmICh0aGlzLnNpY2sgJiYgdGhpcy5vcGVyYXRpb25zQnlJZC5zaXplID09PSAwKSB7XG4gICAgICB0aGlzLmRlc3Ryb3koKTtcbiAgICB9XG4gIH1cblxuICBoYW5kbGVDYW5jZWxsZWQoe2lkfSkge1xuICAgIGNvbnN0IG9wZXJhdGlvbiA9IHRoaXMub3BlcmF0aW9uc0J5SWQuZ2V0KGlkKTtcbiAgICBpZiAob3BlcmF0aW9uKSB7XG4gICAgICAvLyBoYW5kbGVEYXRhUmVjZWl2ZWQoKSBjYW4gYmUgcmVjZWl2ZWQgYmVmb3JlIGhhbmRsZUNhbmNlbGxlZCgpXG4gICAgICBvcGVyYXRpb24ud2FzQ2FuY2VsbGVkKCk7XG4gICAgfVxuICB9XG5cbiAgaGFuZGxlRXhlY1N0YXJ0ZWQoe2lkfSkge1xuICAgIGNvbnN0IG9wZXJhdGlvbiA9IHRoaXMub3BlcmF0aW9uc0J5SWQuZ2V0KGlkKTtcbiAgICBvcGVyYXRpb24uc2V0SW5Qcm9ncmVzcygpO1xuICB9XG5cbiAgaGFuZGxlU3Bhd25FcnJvcih7aWQsIGVycn0pIHtcbiAgICBjb25zdCBvcGVyYXRpb24gPSB0aGlzLm9wZXJhdGlvbnNCeUlkLmdldChpZCk7XG4gICAgb3BlcmF0aW9uLmVycm9yKGVycik7XG4gIH1cblxuICBoYW5kbGVTdGRpbkVycm9yKHtpZCwgc3RkaW4sIGVycn0pIHtcbiAgICBjb25zdCBvcGVyYXRpb24gPSB0aGlzLm9wZXJhdGlvbnNCeUlkLmdldChpZCk7XG4gICAgb3BlcmF0aW9uLmVycm9yKGVycik7XG4gIH1cblxuICBoYW5kbGVTaWNrKCkge1xuICAgIHRoaXMuc2ljayA9IHRydWU7XG4gICAgdGhpcy5vblNpY2sodGhpcyk7XG4gIH1cblxuICBoYW5kbGVDcmFzaGVkKCkge1xuICAgIHRoaXMub25DcmFzaGVkKHRoaXMpO1xuICAgIHRoaXMuZGVzdHJveSgpO1xuICB9XG5cbiAgZ2V0T3BlcmF0aW9uQ291bnRMaW1pdCgpIHtcbiAgICByZXR1cm4gdGhpcy5vcGVyYXRpb25Db3VudExpbWl0O1xuICB9XG5cbiAgZ2V0Q29tcGxldGVkT3BlcmF0aW9uQ291bnQoKSB7XG4gICAgcmV0dXJuIHRoaXMuY29tcGxldGVkT3BlcmF0aW9uQ291bnQ7XG4gIH1cblxuICBnZXRSZW1haW5pbmdPcGVyYXRpb25zKCkge1xuICAgIHJldHVybiBBcnJheS5mcm9tKHRoaXMub3BlcmF0aW9uc0J5SWQudmFsdWVzKCkpO1xuICB9XG5cbiAgZ2V0UGlkKCkge1xuICAgIHJldHVybiB0aGlzLnJlbmRlcmVyUHJvY2Vzcy5nZXRQaWQoKTtcbiAgfVxuXG4gIGdldFJlYWR5UHJvbWlzZSgpIHtcbiAgICByZXR1cm4gdGhpcy5yZW5kZXJlclByb2Nlc3MuZ2V0UmVhZHlQcm9taXNlKCk7XG4gIH1cblxuICBhc3luYyBkZXN0cm95KGZvcmNlKSB7XG4gICAgdGhpcy5vbkRlc3Ryb3llZCh0aGlzKTtcbiAgICBpZiAodGhpcy5vcGVyYXRpb25zQnlJZC5zaXplID4gMCAmJiAhZm9yY2UpIHtcbiAgICAgIGNvbnN0IHJlbWFpbmluZ09wZXJhdGlvblByb21pc2VzID0gdGhpcy5nZXRSZW1haW5pbmdPcGVyYXRpb25zKClcbiAgICAgICAgLm1hcChvcGVyYXRpb24gPT4gb3BlcmF0aW9uLmdldFByb21pc2UoKS5jYXRjaCgoKSA9PiBudWxsKSk7XG4gICAgICBhd2FpdCBQcm9taXNlLmFsbChyZW1haW5pbmdPcGVyYXRpb25Qcm9taXNlcyk7XG4gICAgfVxuICAgIHRoaXMucmVuZGVyZXJQcm9jZXNzLmRlc3Ryb3koKTtcbiAgfVxufVxuXG5cbi8qXG5TZW5kcyBvcGVyYXRpb25zIHRvIHJlbmRlcmVyIHByb2Nlc3Nlc1xuKi9cbmV4cG9ydCBjbGFzcyBSZW5kZXJlclByb2Nlc3Mge1xuICBjb25zdHJ1Y3Rvcih7bG9hZFVybCxcbiAgICBvbkRlc3Ryb3llZCwgb25DcmFzaGVkLCBvblNpY2ssIG9uRGF0YSwgb25DYW5jZWxsZWQsIG9uU3Bhd25FcnJvciwgb25TdGRpbkVycm9yLCBvbkV4ZWNTdGFydGVkfSkge1xuICAgIGF1dG9iaW5kKHRoaXMsICdoYW5kbGVEZXN0cm95Jyk7XG4gICAgdGhpcy5vbkRlc3Ryb3llZCA9IG9uRGVzdHJveWVkO1xuICAgIHRoaXMub25DcmFzaGVkID0gb25DcmFzaGVkO1xuICAgIHRoaXMub25TaWNrID0gb25TaWNrO1xuICAgIHRoaXMub25EYXRhID0gb25EYXRhO1xuICAgIHRoaXMub25DYW5jZWxsZWQgPSBvbkNhbmNlbGxlZDtcbiAgICB0aGlzLm9uU3Bhd25FcnJvciA9IG9uU3Bhd25FcnJvcjtcbiAgICB0aGlzLm9uU3RkaW5FcnJvciA9IG9uU3RkaW5FcnJvcjtcbiAgICB0aGlzLm9uRXhlY1N0YXJ0ZWQgPSBvbkV4ZWNTdGFydGVkO1xuXG4gICAgdGhpcy53aW4gPSBuZXcgQnJvd3NlcldpbmRvdyh7c2hvdzogISFwcm9jZXNzLmVudi5BVE9NX0dJVEhVQl9TSE9XX1JFTkRFUkVSX1dJTkRPV30pO1xuICAgIHRoaXMud2ViQ29udGVudHMgPSB0aGlzLndpbi53ZWJDb250ZW50cztcbiAgICAvLyB0aGlzLndlYkNvbnRlbnRzLm9wZW5EZXZUb29scygpO1xuXG4gICAgdGhpcy5lbWl0dGVyID0gbmV3IEVtaXR0ZXIoKTtcbiAgICB0aGlzLnN1YnNjcmlwdGlvbnMgPSBuZXcgQ29tcG9zaXRlRGlzcG9zYWJsZSgpO1xuICAgIHRoaXMucmVnaXN0ZXJMaXN0ZW5lcnMoKTtcblxuICAgIHRoaXMud2luLmxvYWRVUkwobG9hZFVybCk7XG4gICAgdGhpcy53aW4ud2ViQ29udGVudHMub24oJ2NyYXNoZWQnLCB0aGlzLmhhbmRsZURlc3Ryb3kpO1xuICAgIHRoaXMud2luLndlYkNvbnRlbnRzLm9uKCdkZXN0cm95ZWQnLCB0aGlzLmhhbmRsZURlc3Ryb3kpO1xuICAgIHRoaXMuc3Vic2NyaXB0aW9ucy5hZGQoXG4gICAgICBuZXcgRGlzcG9zYWJsZSgoKSA9PiB7XG4gICAgICAgIGlmICghdGhpcy53aW4uaXNEZXN0cm95ZWQoKSkge1xuICAgICAgICAgIHRoaXMud2luLndlYkNvbnRlbnRzLnJlbW92ZUxpc3RlbmVyKCdjcmFzaGVkJywgdGhpcy5oYW5kbGVEZXN0cm95KTtcbiAgICAgICAgICB0aGlzLndpbi53ZWJDb250ZW50cy5yZW1vdmVMaXN0ZW5lcignZGVzdHJveWVkJywgdGhpcy5oYW5kbGVEZXN0cm95KTtcbiAgICAgICAgICB0aGlzLndpbi5kZXN0cm95KCk7XG4gICAgICAgIH1cbiAgICAgIH0pLFxuICAgICAgdGhpcy5lbWl0dGVyLFxuICAgICk7XG5cbiAgICB0aGlzLnJlYWR5ID0gZmFsc2U7XG4gICAgdGhpcy5yZWFkeVByb21pc2UgPSBuZXcgUHJvbWlzZShyZXNvbHZlID0+IHsgdGhpcy5yZXNvbHZlUmVhZHkgPSByZXNvbHZlOyB9KTtcbiAgfVxuXG4gIGlzUmVhZHkoKSB7XG4gICAgcmV0dXJuIHRoaXMucmVhZHk7XG4gIH1cblxuICBoYW5kbGVEZXN0cm95KC4uLmFyZ3MpIHtcbiAgICB0aGlzLmRlc3Ryb3koKTtcbiAgICB0aGlzLm9uQ3Jhc2hlZCguLi5hcmdzKTtcbiAgfVxuXG4gIHJlZ2lzdGVyTGlzdGVuZXJzKCkge1xuICAgIGNvbnN0IGhhbmRsZU1lc3NhZ2VzID0gKGV2ZW50LCB7c291cmNlV2ViQ29udGVudHNJZCwgdHlwZSwgZGF0YX0pID0+IHtcbiAgICAgIGlmIChzb3VyY2VXZWJDb250ZW50c0lkID09PSB0aGlzLndpbi53ZWJDb250ZW50cy5pZCkge1xuICAgICAgICB0aGlzLmVtaXR0ZXIuZW1pdCh0eXBlLCBkYXRhKTtcbiAgICAgIH1cbiAgICB9O1xuXG4gICAgaXBjLm9uKFdvcmtlci5jaGFubmVsTmFtZSwgaGFuZGxlTWVzc2FnZXMpO1xuICAgIHRoaXMuZW1pdHRlci5vbigncmVuZGVyZXItcmVhZHknLCAoe3BpZH0pID0+IHtcbiAgICAgIHRoaXMucGlkID0gcGlkO1xuICAgICAgdGhpcy5yZWFkeSA9IHRydWU7XG4gICAgICB0aGlzLnJlc29sdmVSZWFkeSgpO1xuICAgIH0pO1xuICAgIHRoaXMuZW1pdHRlci5vbignZ2l0LWRhdGEnLCB0aGlzLm9uRGF0YSk7XG4gICAgdGhpcy5lbWl0dGVyLm9uKCdnaXQtY2FuY2VsbGVkJywgdGhpcy5vbkNhbmNlbGxlZCk7XG4gICAgdGhpcy5lbWl0dGVyLm9uKCdnaXQtc3Bhd24tZXJyb3InLCB0aGlzLm9uU3Bhd25FcnJvcik7XG4gICAgdGhpcy5lbWl0dGVyLm9uKCdnaXQtc3RkaW4tZXJyb3InLCB0aGlzLm9uU3RkaW5FcnJvcik7XG4gICAgdGhpcy5lbWl0dGVyLm9uKCdzbG93LXNwYXducycsIHRoaXMub25TaWNrKTtcblxuICAgIC8vIG5vdCBjdXJyZW50bHkgdXNlZCB0byBhdm9pZCBjbG9nZ2luZyB1cCBpcGMgY2hhbm5lbFxuICAgIC8vIGtlZXBpbmcgaXQgYXJvdW5kIGFzIGl0J3MgcG90ZW50aWFsbHkgdXNlZnVsIGZvciBhdm9pZGluZyBkdXBsaWNhdGUgd3JpdGUgb3BlcmF0aW9ucyB1cG9uIHJlbmRlcmVyIGNyYXNoaW5nXG4gICAgdGhpcy5lbWl0dGVyLm9uKCdleGVjLXN0YXJ0ZWQnLCB0aGlzLm9uRXhlY1N0YXJ0ZWQpO1xuXG4gICAgdGhpcy5zdWJzY3JpcHRpb25zLmFkZChcbiAgICAgIG5ldyBEaXNwb3NhYmxlKCgpID0+IGlwYy5yZW1vdmVMaXN0ZW5lcihXb3JrZXIuY2hhbm5lbE5hbWUsIGhhbmRsZU1lc3NhZ2VzKSksXG4gICAgKTtcbiAgfVxuXG4gIGV4ZWN1dGVPcGVyYXRpb24ob3BlcmF0aW9uKSB7XG4gICAgcmV0dXJuIG9wZXJhdGlvbi5leGVjdXRlKHBheWxvYWQgPT4ge1xuICAgICAgaWYgKHRoaXMuZGVzdHJveWVkKSB7IHJldHVybiBudWxsOyB9XG4gICAgICByZXR1cm4gdGhpcy53ZWJDb250ZW50cy5zZW5kKFdvcmtlci5jaGFubmVsTmFtZSwge1xuICAgICAgICB0eXBlOiAnZ2l0LWV4ZWMnLFxuICAgICAgICBkYXRhOiBwYXlsb2FkLFxuICAgICAgfSk7XG4gICAgfSk7XG4gIH1cblxuICBjYW5jZWxPcGVyYXRpb24ob3BlcmF0aW9uKSB7XG4gICAgcmV0dXJuIG9wZXJhdGlvbi5jYW5jZWwocGF5bG9hZCA9PiB7XG4gICAgICBpZiAodGhpcy5kZXN0cm95ZWQpIHsgcmV0dXJuIG51bGw7IH1cbiAgICAgIHJldHVybiB0aGlzLndlYkNvbnRlbnRzLnNlbmQoV29ya2VyLmNoYW5uZWxOYW1lLCB7XG4gICAgICAgIHR5cGU6ICdnaXQtY2FuY2VsJyxcbiAgICAgICAgZGF0YTogcGF5bG9hZCxcbiAgICAgIH0pO1xuICAgIH0pO1xuICB9XG5cbiAgZ2V0UGlkKCkge1xuICAgIHJldHVybiB0aGlzLnBpZDtcbiAgfVxuXG4gIGdldFJlYWR5UHJvbWlzZSgpIHtcbiAgICByZXR1cm4gdGhpcy5yZWFkeVByb21pc2U7XG4gIH1cblxuICBkZXN0cm95KCkge1xuICAgIHRoaXMuZGVzdHJveWVkID0gdHJ1ZTtcbiAgICB0aGlzLnN1YnNjcmlwdGlvbnMuZGlzcG9zZSgpO1xuICB9XG59XG5cblxuZXhwb3J0IGNsYXNzIE9wZXJhdGlvbiB7XG4gIHN0YXRpYyBzdGF0dXMgPSB7XG4gICAgUEVORElORzogU3ltYm9sKCdwZW5kaW5nJyksXG4gICAgSU5QUk9HUkVTUzogU3ltYm9sKCdpbi1wcm9ncmVzcycpLFxuICAgIENPTVBMRVRFOiBTeW1ib2woJ2NvbXBsZXRlJyksXG4gICAgQ0FOQ0VMTElORzogU3ltYm9sKCdjYW5jZWxsaW5nJyksXG4gICAgQ0FOQ0VMTEVEOiBTeW1ib2woJ2NhbmNlbGVkJyksXG4gIH1cblxuICBzdGF0aWMgaWQgPSAwO1xuXG4gIGNvbnN0cnVjdG9yKGRhdGEsIHJlc29sdmUsIHJlamVjdCkge1xuICAgIHRoaXMuaWQgPSBPcGVyYXRpb24uaWQrKztcbiAgICB0aGlzLmRhdGEgPSBkYXRhO1xuICAgIHRoaXMucmVzb2x2ZSA9IHJlc29sdmU7XG4gICAgdGhpcy5yZWplY3QgPSByZWplY3Q7XG4gICAgdGhpcy5wcm9taXNlID0gbnVsbDtcbiAgICB0aGlzLmNhbmNlbGxhdGlvblJlc29sdmUgPSAoKSA9PiB7fTtcbiAgICB0aGlzLnN0YXJ0VGltZSA9IG51bGw7XG4gICAgdGhpcy5lbmRUaW1lID0gbnVsbDtcbiAgICB0aGlzLnN0YXR1cyA9IE9wZXJhdGlvbi5zdGF0dXMuUEVORElORztcbiAgICB0aGlzLnJlc3VsdHMgPSBudWxsO1xuICAgIHRoaXMuZW1pdHRlciA9IG5ldyBFbWl0dGVyKCk7XG4gIH1cblxuICBvbkNvbXBsZXRlKGNiKSB7XG4gICAgcmV0dXJuIHRoaXMuZW1pdHRlci5vbignY29tcGxldGUnLCBjYik7XG4gIH1cblxuICBzZXRQcm9taXNlKHByb21pc2UpIHtcbiAgICB0aGlzLnByb21pc2UgPSBwcm9taXNlO1xuICB9XG5cbiAgZ2V0UHJvbWlzZSgpIHtcbiAgICByZXR1cm4gdGhpcy5wcm9taXNlO1xuICB9XG5cbiAgc2V0SW5Qcm9ncmVzcygpIHtcbiAgICAvLyBhZnRlciBleGVjIGhhcyBiZWVuIGNhbGxlZCBidXQgYmVmb3JlIHJlc3VsdHMgYSByZWNlaXZlZFxuICAgIHRoaXMuc3RhdHVzID0gT3BlcmF0aW9uLnN0YXR1cy5JTlBST0dSRVNTO1xuICB9XG5cbiAgZ2V0RXhlY3V0aW9uVGltZSgpIHtcbiAgICBpZiAoIXRoaXMuc3RhcnRUaW1lIHx8ICF0aGlzLmVuZFRpbWUpIHtcbiAgICAgIHJldHVybiBOYU47XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiB0aGlzLmVuZFRpbWUgLSB0aGlzLnN0YXJ0VGltZTtcbiAgICB9XG4gIH1cblxuICBjb21wbGV0ZShyZXN1bHRzLCBtdXRhdGUgPSBkYXRhID0+IGRhdGEpIHtcbiAgICB0aGlzLmVuZFRpbWUgPSBwZXJmb3JtYW5jZS5ub3coKTtcbiAgICB0aGlzLnJlc3VsdHMgPSByZXN1bHRzO1xuICAgIHRoaXMucmVzb2x2ZShtdXRhdGUocmVzdWx0cykpO1xuICAgIHRoaXMuY2FuY2VsbGF0aW9uUmVzb2x2ZSgpO1xuICAgIHRoaXMuc3RhdHVzID0gT3BlcmF0aW9uLnN0YXR1cy5DT01QTEVURTtcbiAgICB0aGlzLmVtaXR0ZXIuZW1pdCgnY29tcGxldGUnLCB0aGlzKTtcbiAgICB0aGlzLmVtaXR0ZXIuZGlzcG9zZSgpO1xuICB9XG5cbiAgd2FzQ2FuY2VsbGVkKCkge1xuICAgIHRoaXMuc3RhdHVzID0gT3BlcmF0aW9uLnN0YXR1cy5DQU5DRUxMRUQ7XG4gICAgdGhpcy5jYW5jZWxsYXRpb25SZXNvbHZlKCk7XG4gIH1cblxuICBlcnJvcihyZXN1bHRzKSB7XG4gICAgdGhpcy5lbmRUaW1lID0gcGVyZm9ybWFuY2Uubm93KCk7XG4gICAgY29uc3QgZXJyID0gbmV3IEVycm9yKHJlc3VsdHMubWVzc2FnZSwgcmVzdWx0cy5maWxlTmFtZSwgcmVzdWx0cy5saW5lTnVtYmVyKTtcbiAgICBlcnIuc3RhY2sgPSByZXN1bHRzLnN0YWNrO1xuICAgIHRoaXMucmVqZWN0KGVycik7XG4gIH1cblxuICBleGVjdXRlKGV4ZWNGbikge1xuICAgIHRoaXMuc3RhcnRUaW1lID0gcGVyZm9ybWFuY2Uubm93KCk7XG4gICAgcmV0dXJuIGV4ZWNGbih7Li4udGhpcy5kYXRhLCBpZDogdGhpcy5pZH0pO1xuICB9XG5cbiAgY2FuY2VsKGV4ZWNGbikge1xuICAgIHJldHVybiBuZXcgUHJvbWlzZShyZXNvbHZlID0+IHtcbiAgICAgIHRoaXMuc3RhdHVzID0gT3BlcmF0aW9uLnN0YXR1cy5DQU5DRUxMSU5HO1xuICAgICAgdGhpcy5jYW5jZWxsYXRpb25SZXNvbHZlID0gcmVzb2x2ZTtcbiAgICAgIGV4ZWNGbih7aWQ6IHRoaXMuaWR9KTtcbiAgICB9KTtcbiAgfVxufVxuIl19