"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.category = category;
exports["default"] = exports.BarePrStatusesView = exports.stateToIconAndStyle = void 0;

var _react = _interopRequireDefault(require("react"));

var _reactRelay = require("react-relay");

var _propTypes = _interopRequireDefault(require("prop-types"));

var _helpers = require("../helpers");

var _prStatusContextView = _interopRequireDefault(require("./pr-status-context-view"));

var _octicon = _interopRequireDefault(require("../atom/octicon"));

var _statusDonutChart = _interopRequireDefault(require("./status-donut-chart"));

var _periodicRefresher = _interopRequireDefault(require("../periodic-refresher"));

var _propTypes2 = require("../prop-types");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const stateToIconAndStyle = {
  EXPECTED: {
    category: 'PENDING',
    icon: 'primitive-dot',
    style: 'github-PrStatuses--warning'
  },
  PENDING: {
    category: 'PENDING',
    icon: 'primitive-dot',
    style: 'github-PrStatuses--warning'
  },
  SUCCESS: {
    category: 'SUCCESS',
    icon: 'check',
    style: 'github-PrStatuses--success'
  },
  ERROR: {
    category: 'FAILURE',
    icon: 'alert',
    style: 'github-PrStatuses--error'
  },
  FAILURE: {
    category: 'FAILURE',
    icon: 'x',
    style: 'github-PrStatuses--error'
  }
};
exports.stateToIconAndStyle = stateToIconAndStyle;

function category(state) {
  const info = stateToIconAndStyle[state];

  if (!info) {
    throw new Error(`Unknown state ${state}`);
  }

  return info.category;
}

class BarePrStatusesView extends _react["default"].Component {
  constructor(props) {
    super(props);
    (0, _helpers.autobind)(this, 'refresh');
  }

  componentDidMount() {
    this.refresher = new _periodicRefresher["default"](this.constructor, {
      interval: () => {
        if (this.isPendingResults()) {
          return this.constructor.PENDING_REFRESH_TIMEOUT;
        } else {
          return this.constructor.SUCCESS_REFRESH_TIMEOUT;
        }
      },
      getCurrentId: () => this.props.pullRequest.id,
      refresh: this.refresh,
      minimumIntervalPerId: this.constructor.MINIMUM_REFRESH_INTERVAL
    });
    this.refresher.start();
  }

  componentWillUnmount() {
    this.refresher.destroy();
  }

  refresh() {
    this.props.relay.refetch({
      id: this.props.pullRequest.id
    }, null, null, {
      force: true
    });
  }

  render() {
    const headCommit = this.props.pullRequest.recentCommits.edges[0].node.commit;

    if (!headCommit.status) {
      return null;
    }

    const {
      state,
      contexts
    } = headCommit.status;

    if (this.props.displayType === 'check') {
      const {
        icon,
        style
      } = stateToIconAndStyle[state];
      return _react["default"].createElement(_octicon["default"], {
        icon: icon,
        className: style
      });
    } else if (this.props.displayType === 'full') {
      return _react["default"].createElement("div", {
        className: "github-PrStatuses"
      }, _react["default"].createElement("div", {
        className: "github-PrStatuses-header"
      }, _react["default"].createElement("div", {
        className: "github-PrStatuses-donut-chart"
      }, this.renderDonutChart(headCommit.status)), _react["default"].createElement("div", {
        className: "github-PrStatuses-summary"
      }, this.summarySentence(headCommit.status))), _react["default"].createElement("ul", {
        className: "github-PrStatuses-list"
      }, contexts.map(context => _react["default"].createElement(_prStatusContextView["default"], {
        key: context.id,
        context: context
      }))));
    } else {
      throw new Error(`Invalid \`displayType\` prop value: ${this.props.displayType}`);
    }
  }

  isPendingResults() {
    const headCommit = this.props.pullRequest.recentCommits.edges[0].node.commit;

    if (!headCommit.status) {
      return false;
    }

    const {
      contexts
    } = headCommit.status;
    return contexts.some(c => category(c.state) === 'PENDING');
  }

  renderDonutChart(status) {
    const {
      contexts
    } = status;
    const pendingLen = contexts.filter(c => category(c.state) === 'PENDING').length;
    const failedLen = contexts.filter(c => category(c.state) === 'FAILURE').length;
    const succeededLen = contexts.filter(c => category(c.state) === 'SUCCESS').length;
    return _react["default"].createElement(_statusDonutChart["default"], {
      pending: pendingLen,
      failure: failedLen,
      success: succeededLen
    });
  }

  summarySentence(status) {
    if (this.isAllSucceeded(status)) {
      return 'All checks succeeded';
    } else if (this.isAllFailed(status)) {
      return 'All checks failed';
    } else {
      const {
        contexts
      } = status;
      const noun = contexts.length === 1 ? 'check' : 'checks';
      const parts = [];
      const pending = contexts.filter(c => category(c.state) === 'PENDING');
      const failing = contexts.filter(c => category(c.state) === 'FAILURE');
      const succeeded = contexts.filter(c => category(c.state) === 'SUCCESS');

      if (pending.length) {
        parts.push(`${pending.length} pending`);
      }

      if (failing.length) {
        parts.push(`${failing.length} failing`);
      }

      if (succeeded.length) {
        parts.push(`${succeeded.length} successful`);
      }

      return (0, _helpers.toSentence)(parts) + ` ${noun}`;
    }
  }

  isAllSucceeded(status) {
    return category(status.state) === 'SUCCESS';
  }

  isAllFailed(status) {
    return status.contexts.every(c => category(c.state) === 'FAILURE');
  }

}

exports.BarePrStatusesView = BarePrStatusesView;

_defineProperty(BarePrStatusesView, "propTypes", {
  relay: _propTypes["default"].shape({
    refetch: _propTypes["default"].func.isRequired
  }).isRequired,
  displayType: _propTypes["default"].oneOf(['check', 'full']),
  pullRequest: _propTypes["default"].shape({
    id: _propTypes["default"].string.isRequired,
    recentCommits: (0, _propTypes2.RelayConnectionPropType)(_propTypes["default"].shape({
      commit: _propTypes["default"].shape({
        status: _propTypes["default"].shape({
          state: _propTypes["default"].string.isRequired,
          contexts: _propTypes["default"].arrayOf(_propTypes["default"].shape({
            id: _propTypes["default"].string.isRequired
          }).isRequired).isRequired
        })
      }).isRequired
    }).isRequired).isRequired
  }).isRequired
});

_defineProperty(BarePrStatusesView, "defaultProps", {
  displayType: 'full'
});

_defineProperty(BarePrStatusesView, "lastRefreshPerPr", new Map());

_defineProperty(BarePrStatusesView, "SUCCESS_REFRESH_TIMEOUT", 3 * 60 * 1000);

_defineProperty(BarePrStatusesView, "PENDING_REFRESH_TIMEOUT", 30 * 1000);

_defineProperty(BarePrStatusesView, "MINIMUM_REFRESH_INTERVAL", 15 * 1000);

var _default = (0, _reactRelay.createRefetchContainer)(BarePrStatusesView, {
  pullRequest: function () {
    const node = require("./__generated__/prStatusesView_pullRequest.graphql");

    if (node.hash && node.hash !== "96a5a2c7004e7526967de3b7b551a54b") {
      console.error("The definition of 'prStatusesView_pullRequest' appears to have changed. Run `relay-compiler` to update the generated files to receive the expected data.");
    }

    return require("./__generated__/prStatusesView_pullRequest.graphql");
  }
}, function () {
  const node = require("./__generated__/prStatusesViewRefetchQuery.graphql");

  if (node.hash && node.hash !== "a7600333b3bc426d899c4d0183095a1f") {
    console.error("The definition of 'prStatusesViewRefetchQuery' appears to have changed. Run `relay-compiler` to update the generated files to receive the expected data.");
  }

  return require("./__generated__/prStatusesViewRefetchQuery.graphql");
});

exports["default"] = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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