"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _eventKit = require("event-kit");

var _path = _interopRequireDefault(require("path"));

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _react = _interopRequireDefault(require("react"));

var _reactDom = _interopRequireDefault(require("react-dom"));

var _helpers = require("./helpers");

var _workdirCache = _interopRequireDefault(require("./models/workdir-cache"));

var _workdirContext = _interopRequireDefault(require("./models/workdir-context"));

var _workdirContextPool = _interopRequireDefault(require("./models/workdir-context-pool"));

var _repository = _interopRequireDefault(require("./models/repository"));

var _styleCalculator = _interopRequireDefault(require("./models/style-calculator"));

var _githubLoginModel = _interopRequireDefault(require("./models/github-login-model"));

var _rootController = _interopRequireDefault(require("./controllers/root-controller"));

var _stubItem = _interopRequireDefault(require("./items/stub-item"));

var _switchboard = _interopRequireDefault(require("./switchboard"));

var _yardstick = _interopRequireDefault(require("./yardstick"));

var _gitTimingsView = _interopRequireDefault(require("./views/git-timings-view"));

var _contextMenuInterceptor = _interopRequireDefault(require("./context-menu-interceptor"));

var _asyncQueue = _interopRequireDefault(require("./async-queue"));

var _workerManager = _interopRequireDefault(require("./worker-manager"));

var _getRepoPipelineManager = _interopRequireDefault(require("./get-repo-pipeline-manager"));

var _reporterProxy = require("./reporter-proxy");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const defaultState = {
  newProject: true
};

class GithubPackage {
  constructor({
    workspace,
    project,
    commandRegistry,
    notificationManager,
    tooltips,
    styles,
    grammars,
    keymaps,
    config,
    deserializers,
    confirm,
    getLoadSettings,
    configDirPath,
    renderFn,
    loginModel
  }) {
    (0, _helpers.autobind)(this, 'consumeStatusBar', 'createGitTimingsView', 'createIssueishPaneItemStub', 'createDockItemStub', 'createFilePatchControllerStub', 'destroyGitTabItem', 'destroyGithubTabItem', 'createRepositoryForProjectPath', 'cloneRepositoryForProjectPath', 'getRepositoryForWorkdir', 'scheduleActiveContextUpdate');
    this.workspace = workspace;
    this.project = project;
    this.commandRegistry = commandRegistry;
    this.deserializers = deserializers;
    this.notificationManager = notificationManager;
    this.tooltips = tooltips;
    this.config = config;
    this.styles = styles;
    this.grammars = grammars;
    this.keymaps = keymaps;
    this.configPath = _path["default"].join(configDirPath, 'github.cson');
    this.styleCalculator = new _styleCalculator["default"](this.styles, this.config);
    this.confirm = confirm;
    this.startOpen = false;
    this.activated = false;
    const criteria = {
      projectPathCount: this.project.getPaths().length,
      initPathCount: (getLoadSettings().initialPaths || []).length
    };
    this.pipelineManager = (0, _getRepoPipelineManager["default"])({
      confirm,
      notificationManager,
      workspace
    });
    this.activeContextQueue = new _asyncQueue["default"]();
    this.guessedContext = _workdirContext["default"].guess(criteria, this.pipelineManager);
    this.activeContext = this.guessedContext;
    this.workdirCache = new _workdirCache["default"]();
    this.contextPool = new _workdirContextPool["default"]({
      window,
      workspace,
      promptCallback: query => this.controller.promptForCredentials(query),
      pipelineManager: this.pipelineManager
    });
    this.switchboard = new _switchboard["default"]();
    this.loginModel = loginModel || new _githubLoginModel["default"]();

    this.renderFn = renderFn || ((component, node, callback) => {
      return _reactDom["default"].render(component, node, callback);
    }); // Handle events from all resident contexts.


    this.subscriptions = new _eventKit.CompositeDisposable(this.contextPool.onDidChangeWorkdirOrHead(context => {
      this.refreshAtomGitRepository(context.getWorkingDirectory());
    }), this.contextPool.onDidUpdateRepository(context => {
      this.switchboard.didUpdateRepository(context.getRepository());
    }), this.contextPool.onDidDestroyRepository(context => {
      if (context === this.activeContext) {
        this.setActiveContext(_workdirContext["default"].absent({
          pipelineManager: this.pipelineManager
        }));
      }
    }), _contextMenuInterceptor["default"]);
    this.setupYardstick();
  }

  setupYardstick() {
    const stagingSeries = ['stageLine', 'stageHunk', 'unstageLine', 'unstageHunk'];
    this.subscriptions.add( // Staging and unstaging operations
    this.switchboard.onDidBeginStageOperation(payload => {
      if (payload.stage && payload.line) {
        _yardstick["default"].begin('stageLine');
      } else if (payload.stage && payload.hunk) {
        _yardstick["default"].begin('stageHunk');
      } else if (payload.stage && payload.file) {
        _yardstick["default"].begin('stageFile');
      } else if (payload.stage && payload.mode) {
        _yardstick["default"].begin('stageMode');
      } else if (payload.stage && payload.symlink) {
        _yardstick["default"].begin('stageSymlink');
      } else if (payload.unstage && payload.line) {
        _yardstick["default"].begin('unstageLine');
      } else if (payload.unstage && payload.hunk) {
        _yardstick["default"].begin('unstageHunk');
      } else if (payload.unstage && payload.file) {
        _yardstick["default"].begin('unstageFile');
      } else if (payload.unstage && payload.mode) {
        _yardstick["default"].begin('unstageMode');
      } else if (payload.unstage && payload.symlink) {
        _yardstick["default"].begin('unstageSymlink');
      }
    }), this.switchboard.onDidUpdateRepository(() => {
      _yardstick["default"].mark(stagingSeries, 'update-repository');
    }), this.switchboard.onDidFinishRender(context => {
      if (context === 'RootController.showFilePatchForPath') {
        _yardstick["default"].finish(stagingSeries);
      }
    }), // Active context changes
    this.switchboard.onDidScheduleActiveContextUpdate(() => {
      _yardstick["default"].begin('activeContextChange');
    }), this.switchboard.onDidBeginActiveContextUpdate(() => {
      _yardstick["default"].mark('activeContextChange', 'queue-wait');
    }), this.switchboard.onDidFinishContextChangeRender(() => {
      _yardstick["default"].mark('activeContextChange', 'render');
    }), this.switchboard.onDidFinishActiveContextUpdate(() => {
      _yardstick["default"].finish('activeContextChange');
    }));
  }

  async activate(state = {}) {
    this.savedState = _objectSpread({}, defaultState, state);
    const firstRun = !(await (0, _helpers.fileExists)(this.configPath));
    const newProject = this.savedState.firstRun !== undefined ? this.savedState.firstRun : this.savedState.newProject;
    this.startOpen = firstRun || newProject;
    this.startRevealed = firstRun && !this.config.get('welcome.showOnStartup');

    if (firstRun) {
      await _fsExtra["default"].writeFile(this.configPath, '# Store non-visible GitHub package state.\n', {
        encoding: 'utf8'
      });
    }

    const hasSelectedFiles = event => {
      return !!event.target.closest('.github-FilePatchListView').querySelector('.is-selected');
    };

    this.subscriptions.add(this.project.onDidChangePaths(this.scheduleActiveContextUpdate), this.workspace.getCenter().onDidChangeActivePaneItem(this.scheduleActiveContextUpdate), this.styleCalculator.startWatching('github-package-styles', ['editor.fontSize', 'editor.fontFamily', 'editor.lineHeight', 'editor.tabLength'], config => `
          .github-HunkView-line {
            font-family: ${config.get('editor.fontFamily')};
            line-height: ${config.get('editor.lineHeight')};
            tab-size: ${config.get('editor.tabLength')}
          }
        `), atom.contextMenu.add({
      '.github-UnstagedChanges .github-FilePatchListView': [{
        label: 'Stage',
        command: 'core:confirm',
        shouldDisplay: hasSelectedFiles
      }, {
        type: 'separator',
        shouldDisplay: hasSelectedFiles
      }, {
        label: 'Discard Changes',
        command: 'github:discard-changes-in-selected-files',
        shouldDisplay: hasSelectedFiles
      }],
      '.github-StagedChanges .github-FilePatchListView': [{
        label: 'Unstage',
        command: 'core:confirm',
        shouldDisplay: hasSelectedFiles
      }],
      '.github-MergeConflictPaths .github-FilePatchListView': [{
        label: 'Stage',
        command: 'core:confirm',
        shouldDisplay: hasSelectedFiles
      }, {
        type: 'separator',
        shouldDisplay: hasSelectedFiles
      }, {
        label: 'Resolve File As Ours',
        command: 'github:resolve-file-as-ours',
        shouldDisplay: hasSelectedFiles
      }, {
        label: 'Resolve File As Theirs',
        command: 'github:resolve-file-as-theirs',
        shouldDisplay: hasSelectedFiles
      }]
    }));
    this.activated = true;
    this.scheduleActiveContextUpdate(this.savedState);
    this.rerender();
  }

  serialize() {
    const activeRepository = this.getActiveRepository();
    const activeRepositoryPath = activeRepository ? activeRepository.getWorkingDirectoryPath() : null;
    return {
      activeRepositoryPath,
      newProject: false
    };
  }

  rerender(callback) {
    if (this.workspace.isDestroyed()) {
      return;
    }

    if (!this.activated) {
      return;
    }

    if (!this.element) {
      this.element = document.createElement('div');
      this.subscriptions.add(new _eventKit.Disposable(() => {
        _reactDom["default"].unmountComponentAtNode(this.element);

        delete this.element;
      }));
    }

    this.renderFn(_react["default"].createElement(_rootController["default"], {
      ref: c => {
        this.controller = c;
      },
      workspace: this.workspace,
      deserializers: this.deserializers,
      commandRegistry: this.commandRegistry,
      notificationManager: this.notificationManager,
      tooltips: this.tooltips,
      grammars: this.grammars,
      keymaps: this.keymaps,
      config: this.config,
      project: this.project,
      confirm: this.confirm,
      workdirContextPool: this.contextPool,
      loginModel: this.loginModel,
      repository: this.getActiveRepository(),
      resolutionProgress: this.getActiveResolutionProgress(),
      statusBar: this.statusBar,
      createRepositoryForProjectPath: this.createRepositoryForProjectPath,
      cloneRepositoryForProjectPath: this.cloneRepositoryForProjectPath,
      switchboard: this.switchboard,
      startOpen: this.startOpen,
      startRevealed: this.startRevealed,
      removeFilePatchItem: this.removeFilePatchItem
    }), this.element, callback);
  }

  async deactivate() {
    this.subscriptions.dispose();
    this.contextPool.clear();

    _workerManager["default"].reset(false);

    if (this.guessedContext) {
      this.guessedContext.destroy();
      this.guessedContext = null;
    }

    await _yardstick["default"].flush();
  }

  consumeStatusBar(statusBar) {
    this.statusBar = statusBar;
    this.rerender();
  }

  consumeReporter(reporter) {
    _reporterProxy.reporterProxy.setReporter(reporter);
  }

  createGitTimingsView() {
    return _stubItem["default"].create('git-timings-view', {
      title: 'GitHub Package Timings View'
    }, _gitTimingsView["default"].buildURI());
  }

  createIssueishPaneItemStub({
    uri,
    selectedTab
  }) {
    return _stubItem["default"].create('issueish-detail-item', {
      title: 'Issueish',
      initSelectedTab: selectedTab
    }, uri);
  }

  createDockItemStub({
    uri
  }) {
    let item;

    switch (uri) {
      // always return an empty stub
      // but only set it as the active item for a tab type
      // if it doesn't already exist
      case 'atom-github://dock-item/git':
        item = this.createGitStub(uri);
        this.gitTabStubItem = this.gitTabStubItem || item;
        break;

      case 'atom-github://dock-item/github':
        item = this.createGitHubStub(uri);
        this.githubTabStubItem = this.githubTabStubItem || item;
        break;

      default:
        throw new Error(`Invalid DockItem stub URI: ${uri}`);
    }

    if (this.controller) {
      this.rerender();
    }

    return item;
  }

  createGitStub(uri) {
    return _stubItem["default"].create('git', {
      title: 'Git'
    }, uri);
  }

  createGitHubStub(uri) {
    return _stubItem["default"].create('github', {
      title: 'GitHub'
    }, uri);
  }

  createFilePatchControllerStub({
    uri
  } = {}) {
    const item = _stubItem["default"].create('git-file-patch-controller', {
      title: 'Diff'
    }, uri);

    if (this.controller) {
      this.rerender();
    }

    return item;
  }

  createCommitPreviewStub({
    uri
  }) {
    const item = _stubItem["default"].create('git-commit-preview', {
      title: 'Commit preview'
    }, uri);

    if (this.controller) {
      this.rerender();
    }

    return item;
  }

  createCommitDetailStub({
    uri
  }) {
    const item = _stubItem["default"].create('git-commit-detail', {
      title: 'Commit'
    }, uri);

    if (this.controller) {
      this.rerender();
    }

    return item;
  }

  createReviewsStub({
    uri
  }) {
    const item = _stubItem["default"].create('github-reviews', {
      title: 'Reviews'
    }, uri);

    if (this.controller) {
      this.rerender();
    }

    return item;
  }

  destroyGitTabItem() {
    if (this.gitTabStubItem) {
      this.gitTabStubItem.destroy();
      this.gitTabStubItem = null;

      if (this.controller) {
        this.rerender();
      }
    }
  }

  destroyGithubTabItem() {
    if (this.githubTabStubItem) {
      this.githubTabStubItem.destroy();
      this.githubTabStubItem = null;

      if (this.controller) {
        this.rerender();
      }
    }
  }

  async createRepositoryForProjectPath(projectPath) {
    await _fsExtra["default"].mkdirs(projectPath);
    const repository = this.contextPool.add(projectPath).getRepository();
    await repository.init();
    this.workdirCache.invalidate();

    if (!this.project.contains(projectPath)) {
      this.project.addPath(projectPath);
    }

    await this.scheduleActiveContextUpdate();
  }

  async cloneRepositoryForProjectPath(remoteUrl, projectPath) {
    const context = this.contextPool.getContext(projectPath);
    let repository;

    if (context.isPresent()) {
      repository = context.getRepository();
      await repository.clone(remoteUrl);
      repository.destroy();
    } else {
      repository = new _repository["default"](projectPath, null, {
        pipelineManager: this.pipelineManager
      });
      await repository.clone(remoteUrl);
    }

    this.workdirCache.invalidate();
    this.project.addPath(projectPath);
    await this.scheduleActiveContextUpdate();
  }

  getRepositoryForWorkdir(projectPath) {
    const loadingGuessRepo = _repository["default"].loadingGuess({
      pipelineManager: this.pipelineManager
    });

    return this.guessedContext ? loadingGuessRepo : this.contextPool.getContext(projectPath).getRepository();
  }

  getActiveWorkdir() {
    return this.activeContext.getWorkingDirectory();
  }

  getActiveRepository() {
    return this.activeContext.getRepository();
  }

  getActiveResolutionProgress() {
    return this.activeContext.getResolutionProgress();
  }

  getContextPool() {
    return this.contextPool;
  }

  getSwitchboard() {
    return this.switchboard;
  }

  async scheduleActiveContextUpdate(savedState = {}) {
    this.switchboard.didScheduleActiveContextUpdate();
    await this.activeContextQueue.push(this.updateActiveContext.bind(this, savedState), {
      parallel: false
    });
  }
  /**
   * Derive the git working directory context that should be used for the package's git operations based on the current
   * state of the Atom workspace. In priority, this prefers:
   *
   * - A git working directory that contains the active pane item in the workspace's center.
   * - A git working directory corresponding to a single Project.
   * - When initially activating the package, the working directory that was active when the package was last
   *   serialized.
   * - The current context, unchanged, which may be a `NullWorkdirContext`.
   *
   * First updates the pool of resident contexts to match all git working directories that correspond to open
   * projects and pane items.
   */


  async getNextContext(savedState) {
    const workdirs = new Set((await Promise.all(this.project.getPaths().map(async projectPath => {
      const workdir = await this.workdirCache.find(projectPath);
      return workdir || projectPath;
    }))));

    const fromPaneItem = async maybeItem => {
      const itemPath = pathForPaneItem(maybeItem);

      if (!itemPath) {
        return {};
      }

      const itemWorkdir = await this.workdirCache.find(itemPath);

      if (itemWorkdir && !this.project.contains(itemPath)) {
        workdirs.add(itemWorkdir);
      }

      return {
        itemPath,
        itemWorkdir
      };
    };

    const active = await fromPaneItem(this.workspace.getCenter().getActivePaneItem());
    this.contextPool.set(workdirs, savedState);

    if (active.itemPath) {
      // Prefer an active item
      return this.contextPool.getContext(active.itemWorkdir || active.itemPath);
    }

    if (this.project.getPaths().length === 1) {
      // Single project
      const projectPath = this.project.getPaths()[0];
      const activeWorkingDir = await this.workdirCache.find(projectPath);
      return this.contextPool.getContext(activeWorkingDir || projectPath);
    }

    if (this.project.getPaths().length === 0 && !this.activeContext.getRepository().isUndetermined()) {
      // No projects. Revert to the absent context unless we've guessed that more projects are on the way.
      return _workdirContext["default"].absent({
        pipelineManager: this.pipelineManager
      });
    } // Restore models from saved state. Will return a NullWorkdirContext if this path is not presently
    // resident in the pool.


    const savedWorkingDir = savedState.activeRepositoryPath;

    if (savedWorkingDir) {
      return this.contextPool.getContext(savedWorkingDir);
    }

    return this.activeContext;
  }

  setActiveContext(nextActiveContext) {
    if (nextActiveContext !== this.activeContext) {
      if (this.activeContext === this.guessedContext) {
        this.guessedContext.destroy();
        this.guessedContext = null;
      }

      this.activeContext = nextActiveContext;
      this.rerender(() => {
        this.switchboard.didFinishContextChangeRender();
        this.switchboard.didFinishActiveContextUpdate();
      });
    } else {
      this.switchboard.didFinishActiveContextUpdate();
    }
  }

  async updateActiveContext(savedState = {}) {
    if (this.workspace.isDestroyed()) {
      return;
    }

    this.switchboard.didBeginActiveContextUpdate();
    const nextActiveContext = await this.getNextContext(savedState);
    this.setActiveContext(nextActiveContext);
  }

  refreshAtomGitRepository(workdir) {
    const atomGitRepo = this.project.getRepositories().find(repo => {
      return repo && _path["default"].normalize(repo.getWorkingDirectory()) === workdir;
    });
    return atomGitRepo ? atomGitRepo.refreshStatus() : Promise.resolve();
  }

}

exports["default"] = GithubPackage;

function pathForPaneItem(paneItem) {
  if (!paneItem) {
    return null;
  } // Likely GitHub package provided pane item


  if (typeof paneItem.getWorkingDirectory === 'function') {
    return paneItem.getWorkingDirectory();
  } // TextEditor-like


  if (typeof paneItem.getPath === 'function') {
    return paneItem.getPath();
  } // Oh well


  return null;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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