/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.ssl;

import java.nio.file.Path;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.crypto.Cipher;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.TrustManagerFactory;
import org.elasticsearch.common.ssl.DefaultJdkTrustConfig;
import org.elasticsearch.common.ssl.EmptyKeyConfig;
import org.elasticsearch.common.ssl.KeyStoreUtil;
import org.elasticsearch.common.ssl.PemKeyConfig;
import org.elasticsearch.common.ssl.PemTrustConfig;
import org.elasticsearch.common.ssl.SslClientAuthenticationMode;
import org.elasticsearch.common.ssl.SslConfigException;
import org.elasticsearch.common.ssl.SslConfiguration;
import org.elasticsearch.common.ssl.SslKeyConfig;
import org.elasticsearch.common.ssl.SslTrustConfig;
import org.elasticsearch.common.ssl.SslVerificationMode;
import org.elasticsearch.common.ssl.StoreKeyConfig;
import org.elasticsearch.common.ssl.StoreTrustConfig;
import org.elasticsearch.common.ssl.TrustEverythingConfig;

public abstract class SslConfigurationLoader {
    static final List<String> DEFAULT_PROTOCOLS = Collections.unmodifiableList(SslConfiguration.ORDERED_PROTOCOL_ALGORITHM_MAP.containsKey("TLSv1.3") ? Arrays.asList("TLSv1.3", "TLSv1.2", "TLSv1.1") : Arrays.asList("TLSv1.2", "TLSv1.1"));
    static final List<String> DEFAULT_CIPHERS = SslConfigurationLoader.loadDefaultCiphers();
    private static final char[] EMPTY_PASSWORD = new char[0];
    private final String settingPrefix;
    private SslTrustConfig defaultTrustConfig;
    private SslKeyConfig defaultKeyConfig;
    private SslVerificationMode defaultVerificationMode;
    private SslClientAuthenticationMode defaultClientAuth;
    private List<String> defaultCiphers;
    private List<String> defaultProtocols;

    public SslConfigurationLoader(String settingPrefix) {
        String string = this.settingPrefix = settingPrefix == null ? "" : settingPrefix;
        if (!this.settingPrefix.isEmpty() && !this.settingPrefix.endsWith(".")) {
            throw new IllegalArgumentException("Setting prefix [" + settingPrefix + "] must be blank or end in '.'");
        }
        this.defaultTrustConfig = new DefaultJdkTrustConfig();
        this.defaultKeyConfig = EmptyKeyConfig.INSTANCE;
        this.defaultVerificationMode = SslVerificationMode.FULL;
        this.defaultClientAuth = SslClientAuthenticationMode.OPTIONAL;
        this.defaultProtocols = DEFAULT_PROTOCOLS;
        this.defaultCiphers = DEFAULT_CIPHERS;
    }

    public void setDefaultTrustConfig(SslTrustConfig defaultTrustConfig) {
        this.defaultTrustConfig = defaultTrustConfig;
    }

    public void setDefaultKeyConfig(SslKeyConfig defaultKeyConfig) {
        this.defaultKeyConfig = defaultKeyConfig;
    }

    public void setDefaultVerificationMode(SslVerificationMode defaultVerificationMode) {
        this.defaultVerificationMode = defaultVerificationMode;
    }

    public void setDefaultClientAuth(SslClientAuthenticationMode defaultClientAuth) {
        this.defaultClientAuth = defaultClientAuth;
    }

    public void setDefaultCiphers(List<String> defaultCiphers) {
        this.defaultCiphers = defaultCiphers;
    }

    public void setDefaultProtocols(List<String> defaultProtocols) {
        this.defaultProtocols = defaultProtocols;
    }

    protected abstract String getSettingAsString(String var1) throws Exception;

    protected abstract char[] getSecureSetting(String var1) throws Exception;

    protected abstract List<String> getSettingAsList(String var1) throws Exception;

    public SslConfiguration load(Path basePath) {
        Objects.requireNonNull(basePath, "Base Path cannot be null");
        List<String> protocols = this.resolveListSetting("supported_protocols", Function.identity(), this.defaultProtocols);
        List<String> ciphers = this.resolveListSetting("cipher_suites", Function.identity(), this.defaultCiphers);
        SslVerificationMode verificationMode = this.resolveSetting("verification_mode", SslVerificationMode::parse, this.defaultVerificationMode);
        SslClientAuthenticationMode clientAuth = this.resolveSetting("client_authentication", SslClientAuthenticationMode::parse, this.defaultClientAuth);
        SslTrustConfig trustConfig = this.buildTrustConfig(basePath, verificationMode);
        SslKeyConfig keyConfig = this.buildKeyConfig(basePath);
        if (protocols == null || protocols.isEmpty()) {
            throw new SslConfigException("no protocols configured in [" + this.settingPrefix + "supported_protocols" + "]");
        }
        if (ciphers == null || ciphers.isEmpty()) {
            throw new SslConfigException("no cipher suites configured in [" + this.settingPrefix + "cipher_suites" + "]");
        }
        return new SslConfiguration(trustConfig, keyConfig, verificationMode, clientAuth, ciphers, protocols);
    }

    private SslTrustConfig buildTrustConfig(Path basePath, SslVerificationMode verificationMode) {
        List<Path> certificateAuthorities = this.resolveListSetting("certificate_authorities", basePath::resolve, null);
        Path trustStorePath = this.resolveSetting("truststore.path", basePath::resolve, null);
        if (certificateAuthorities != null && trustStorePath != null) {
            throw new SslConfigException("cannot specify both [" + this.settingPrefix + "certificate_authorities" + "] and [" + this.settingPrefix + "truststore.path" + "]");
        }
        if (!verificationMode.isCertificateVerificationEnabled()) {
            return TrustEverythingConfig.TRUST_EVERYTHING;
        }
        if (certificateAuthorities != null) {
            return new PemTrustConfig(certificateAuthorities);
        }
        if (trustStorePath != null) {
            char[] password = this.resolvePasswordSetting("truststore.secure_password", "truststore.password");
            String storeType = this.resolveSetting("truststore.type", Function.identity(), KeyStoreUtil.inferKeyStoreType(trustStorePath));
            String algorithm = this.resolveSetting("truststore.algorithm", Function.identity(), TrustManagerFactory.getDefaultAlgorithm());
            return new StoreTrustConfig(trustStorePath, password, storeType, algorithm);
        }
        return this.defaultTrustConfig;
    }

    private SslKeyConfig buildKeyConfig(Path basePath) {
        Path certificatePath = this.resolveSetting("certificate", basePath::resolve, null);
        Path keyPath = this.resolveSetting("key", basePath::resolve, null);
        Path keyStorePath = this.resolveSetting("keystore.path", basePath::resolve, null);
        if (certificatePath != null && keyStorePath != null) {
            throw new SslConfigException("cannot specify both [" + this.settingPrefix + "certificate" + "] and [" + this.settingPrefix + "keystore.path" + "]");
        }
        if (certificatePath != null || keyPath != null) {
            if (keyPath == null) {
                throw new SslConfigException("cannot specify [" + this.settingPrefix + "certificate" + "] without also setting [" + this.settingPrefix + "key" + "]");
            }
            if (certificatePath == null) {
                throw new SslConfigException("cannot specify [" + this.settingPrefix + "keystore.path" + "] without also setting [" + this.settingPrefix + "certificate" + "]");
            }
            char[] password = this.resolvePasswordSetting("secure_key_passphrase", "key_passphrase");
            return new PemKeyConfig(certificatePath, keyPath, password);
        }
        if (keyStorePath != null) {
            char[] storePassword = this.resolvePasswordSetting("keystore.secure_password", "keystore.password");
            char[] keyPassword = this.resolvePasswordSetting("keystore.secure_key_password", "keystore.key_password");
            if (keyPassword.length == 0) {
                keyPassword = storePassword;
            }
            String storeType = this.resolveSetting("keystore.type", Function.identity(), KeyStoreUtil.inferKeyStoreType(keyStorePath));
            String algorithm = this.resolveSetting("keystore.algorithm", Function.identity(), KeyManagerFactory.getDefaultAlgorithm());
            return new StoreKeyConfig(keyStorePath, storePassword, storeType, keyPassword, algorithm);
        }
        return this.defaultKeyConfig;
    }

    private char[] resolvePasswordSetting(String secureSettingKey, String legacySettingKey) {
        char[] securePassword = this.resolveSecureSetting(secureSettingKey, null);
        String legacyPassword = this.resolveSetting(legacySettingKey, Function.identity(), null);
        if (securePassword == null) {
            if (legacyPassword == null) {
                return EMPTY_PASSWORD;
            }
            return legacyPassword.toCharArray();
        }
        if (legacyPassword != null) {
            throw new SslConfigException("cannot specify both [" + this.settingPrefix + secureSettingKey + "] and [" + this.settingPrefix + legacySettingKey + "]");
        }
        return securePassword;
    }

    private <V> V resolveSetting(String key, Function<String, V> parser, V defaultValue) {
        try {
            String setting = this.getSettingAsString(this.settingPrefix + key);
            if (setting == null || setting.isEmpty()) {
                return defaultValue;
            }
            return parser.apply(setting);
        }
        catch (RuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            throw new SslConfigException("cannot retrieve setting [" + this.settingPrefix + key + "]", e);
        }
    }

    private char[] resolveSecureSetting(String key, char[] defaultValue) {
        try {
            char[] setting = this.getSecureSetting(this.settingPrefix + key);
            if (setting == null || setting.length == 0) {
                return defaultValue;
            }
            return setting;
        }
        catch (RuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            throw new SslConfigException("cannot retrieve secure setting [" + this.settingPrefix + key + "]", e);
        }
    }

    private <V> List<V> resolveListSetting(String key, Function<String, V> parser, List<V> defaultValue) {
        try {
            List<String> list = this.getSettingAsList(this.settingPrefix + key);
            if (list == null || list.isEmpty()) {
                return defaultValue;
            }
            return list.stream().map(parser).collect(Collectors.toList());
        }
        catch (RuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            throw new SslConfigException("cannot retrieve setting [" + this.settingPrefix + key + "]", e);
        }
    }

    private static List<String> loadDefaultCiphers() {
        List<String> ciphers128 = Arrays.asList("TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256", "TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256", "TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA", "TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA", "TLS_RSA_WITH_AES_128_CBC_SHA256", "TLS_RSA_WITH_AES_128_CBC_SHA");
        List<String> ciphers256 = Arrays.asList("TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384", "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384", "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA", "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA", "TLS_RSA_WITH_AES_256_CBC_SHA256", "TLS_RSA_WITH_AES_256_CBC_SHA");
        if (SslConfigurationLoader.has256BitAES()) {
            ArrayList<String> ciphers = new ArrayList<String>(ciphers256.size() + ciphers128.size());
            ciphers.addAll(ciphers256);
            ciphers.addAll(ciphers128);
            return ciphers;
        }
        return ciphers128;
    }

    private static boolean has256BitAES() {
        try {
            return Cipher.getMaxAllowedKeyLength("AES") > 128;
        }
        catch (NoSuchAlgorithmException e) {
            return false;
        }
    }
}

