"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _path = _interopRequireDefault(require("path"));

var _os = _interopRequireDefault(require("os"));

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _mkdirp = _interopRequireDefault(require("mkdirp"));

var _discardHistoryStores = require("./discard-history-stores");

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const emptyFilePath = _path["default"].join(_os["default"].tmpdir(), 'empty-file.txt');

const emptyFilePromise = _fsExtra["default"].writeFile(emptyFilePath, '');

class DiscardHistory {
  constructor(createBlob, expandBlobToFile, mergeFile, workdirPath, {
    maxHistoryLength
  } = {}) {
    this.createBlob = createBlob;
    this.expandBlobToFile = expandBlobToFile;
    this.mergeFile = mergeFile;
    this.workdirPath = workdirPath;
    this.partialFileHistory = new _discardHistoryStores.PartialFileDiscardHistory(maxHistoryLength);
    this.wholeFileHistory = new _discardHistoryStores.WholeFileDiscardHistory(maxHistoryLength);
  }

  getLastSnapshots(partialDiscardFilePath = null) {
    if (partialDiscardFilePath) {
      return this.partialFileHistory.getLastSnapshotsForPath(partialDiscardFilePath);
    } else {
      return this.wholeFileHistory.getLastSnapshots();
    }
  }

  getHistory(partialDiscardFilePath = null) {
    if (partialDiscardFilePath) {
      return this.partialFileHistory.getHistoryForPath(partialDiscardFilePath);
    } else {
      return this.wholeFileHistory.getHistory();
    }
  }

  hasHistory(partialDiscardFilePath = null) {
    const history = this.getHistory(partialDiscardFilePath);
    return history.length > 0;
  }

  popHistory(partialDiscardFilePath = null) {
    if (partialDiscardFilePath) {
      return this.partialFileHistory.popHistoryForPath(partialDiscardFilePath);
    } else {
      return this.wholeFileHistory.popHistory();
    }
  }

  clearHistory(partialDiscardFilePath = null) {
    if (partialDiscardFilePath) {
      this.partialFileHistory.clearHistoryForPath(partialDiscardFilePath);
    } else {
      this.wholeFileHistory.clearHistory();
    }
  }

  updateHistory(history) {
    this.partialFileHistory.setHistory(history.partialFileHistory || {});
    this.wholeFileHistory.setHistory(history.wholeFileHistory || []);
  }

  async createHistoryBlob() {
    const histories = {
      wholeFileHistory: this.wholeFileHistory.getHistory(),
      partialFileHistory: this.partialFileHistory.getHistory()
    };
    const historySha = await this.createBlob({
      stdin: JSON.stringify(histories)
    });
    return historySha;
  }

  async storeBeforeAndAfterBlobs(filePaths, isSafe, destructiveAction, partialDiscardFilePath = null) {
    if (partialDiscardFilePath) {
      return await this.storeBlobsForPartialFileHistory(partialDiscardFilePath, isSafe, destructiveAction);
    } else {
      return await this.storeBlobsForWholeFileHistory(filePaths, isSafe, destructiveAction);
    }
  }

  async storeBlobsForPartialFileHistory(filePath, isSafe, destructiveAction) {
    const beforeSha = await this.createBlob({
      filePath
    });
    const isNotSafe = !(await isSafe());

    if (isNotSafe) {
      return null;
    }

    await destructiveAction();
    const afterSha = await this.createBlob({
      filePath
    });
    const snapshots = {
      beforeSha,
      afterSha
    };
    this.partialFileHistory.addHistory(filePath, snapshots);
    return snapshots;
  }

  async storeBlobsForWholeFileHistory(filePaths, isSafe, destructiveAction) {
    const snapshotsByPath = {};
    const beforePromises = filePaths.map(async filePath => {
      snapshotsByPath[filePath] = {
        beforeSha: await this.createBlob({
          filePath
        })
      };
    });
    await Promise.all(beforePromises);
    const isNotSafe = !(await isSafe());

    if (isNotSafe) {
      return null;
    }

    await destructiveAction();
    const afterPromises = filePaths.map(async filePath => {
      snapshotsByPath[filePath].afterSha = await this.createBlob({
        filePath
      });
    });
    await Promise.all(afterPromises);
    this.wholeFileHistory.addHistory(snapshotsByPath);
    return snapshotsByPath;
  }

  async restoreLastDiscardInTempFiles(isSafe, partialDiscardFilePath = null) {
    let lastDiscardSnapshots = this.getLastSnapshots(partialDiscardFilePath);

    if (partialDiscardFilePath) {
      lastDiscardSnapshots = lastDiscardSnapshots ? [lastDiscardSnapshots] : [];
    }

    const tempFolderPaths = await this.expandBlobsToFilesInTempFolder(lastDiscardSnapshots);

    if (!isSafe()) {
      return [];
    }

    return await this.mergeFiles(tempFolderPaths);
  }

  async expandBlobsToFilesInTempFolder(snapshots) {
    const tempFolderPath = await (0, _helpers.getTempDir)({
      prefix: 'github-discard-history-'
    });
    const pathPromises = snapshots.map(async ({
      filePath,
      beforeSha,
      afterSha
    }) => {
      const dir = _path["default"].dirname(_path["default"].join(tempFolderPath, filePath));

      await (0, _mkdirp["default"])(dir);
      const theirsPath = !beforeSha ? null : await this.expandBlobToFile(_path["default"].join(tempFolderPath, `${filePath}-before-discard`), beforeSha);
      const commonBasePath = !afterSha ? null : await this.expandBlobToFile(_path["default"].join(tempFolderPath, `${filePath}-after-discard`), afterSha);

      const resultPath = _path["default"].join(tempFolderPath, `~${_path["default"].basename(filePath)}-merge-result`);

      return {
        filePath,
        commonBasePath,
        theirsPath,
        resultPath,
        theirsSha: beforeSha,
        commonBaseSha: afterSha
      };
    });
    return await Promise.all(pathPromises);
  }

  async mergeFiles(filePaths) {
    const mergeFilePromises = filePaths.map(async (filePathInfo, i) => {
      const {
        filePath,
        commonBasePath,
        theirsPath,
        resultPath,
        theirsSha,
        commonBaseSha
      } = filePathInfo;
      const currentSha = await this.createBlob({
        filePath
      });
      let mergeResult;

      if (theirsPath && commonBasePath) {
        mergeResult = await this.mergeFile(filePath, commonBasePath, theirsPath, resultPath);
      } else if (!theirsPath && commonBasePath) {
        // deleted file
        const oursSha = await this.createBlob({
          filePath
        });

        if (oursSha === commonBaseSha) {
          // no changes since discard, mark file to be deleted
          mergeResult = {
            filePath,
            resultPath: null,
            deleted: true,
            conflict: false
          };
        } else {
          // changes since discard result in conflict
          await _fsExtra["default"].copy(_path["default"].join(this.workdirPath, filePath), resultPath);
          mergeResult = {
            filePath,
            resultPath,
            conflict: true
          };
        }
      } else if (theirsPath && !commonBasePath) {
        // added file
        const fileDoesExist = await (0, _helpers.fileExists)(_path["default"].join(this.workdirPath, filePath));

        if (!fileDoesExist) {
          await _fsExtra["default"].copy(theirsPath, resultPath);
          mergeResult = {
            filePath,
            resultPath,
            conflict: false
          };
        } else {
          await emptyFilePromise;
          mergeResult = await this.mergeFile(filePath, emptyFilePath, theirsPath, resultPath);
        }
      } else {
        throw new Error('One of the following must be defined - theirsPath:' + `${theirsPath} or commonBasePath: ${commonBasePath}`);
      }

      return _objectSpread({}, mergeResult, {
        theirsSha,
        commonBaseSha,
        currentSha
      });
    });
    return await Promise.all(mergeFilePromises);
  }

}

exports["default"] = DiscardHistory;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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