"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _eventKit = require("event-kit");

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _compareSets = _interopRequireDefault(require("compare-sets"));

var _commands = _interopRequireWildcard(require("../atom/commands"));

var _conflict = _interopRequireDefault(require("../models/conflicts/conflict"));

var _conflictController = _interopRequireDefault(require("./conflict-controller"));

var _source = require("../models/conflicts/source");

var _helpers = require("../helpers");

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj["default"] = obj; return newObj; } }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/**
 * Render a `ConflictController` for each conflict marker within an open TextEditor.
 */
class EditorConflictController extends _react["default"].Component {
  constructor(props, context) {
    super(props, context);
    (0, _helpers.autobind)(this, 'resolveAsCurrent', 'revertConflictModifications', 'dismissCurrent'); // this.layer = props.editor.addMarkerLayer({
    //   maintainHistory: true,
    //   persistent: false,
    // });

    this.layer = props.editor.getDefaultMarkerLayer();
    this.state = {
      conflicts: new Set(_conflict["default"].allFromEditor(props.editor, this.layer, props.isRebase))
    };
    this.subscriptions = new _eventKit.CompositeDisposable();
    this.updateMarkerCount();
  }

  componentDidMount() {
    const buffer = this.props.editor.getBuffer();
    this.subscriptions.add(this.props.editor.onDidDestroy(() => this.props.refreshResolutionProgress(this.props.editor.getPath())), buffer.onDidReload(() => this.reparseConflicts()));
  }

  render() {
    this.updateMarkerCount();
    return _react["default"].createElement("div", null, this.state.conflicts.size > 0 && _react["default"].createElement(_commands["default"], {
      registry: this.props.commandRegistry,
      target: "atom-text-editor"
    }, _react["default"].createElement(_commands.Command, {
      command: "github:resolve-as-ours",
      callback: this.getResolverUsing([_source.OURS])
    }), _react["default"].createElement(_commands.Command, {
      command: "github:resolve-as-theirs",
      callback: this.getResolverUsing([_source.THEIRS])
    }), _react["default"].createElement(_commands.Command, {
      command: "github:resolve-as-base",
      callback: this.getResolverUsing([_source.BASE])
    }), _react["default"].createElement(_commands.Command, {
      command: "github:resolve-as-ours-then-theirs",
      callback: this.getResolverUsing([_source.OURS, _source.THEIRS])
    }), _react["default"].createElement(_commands.Command, {
      command: "github:resolve-as-theirs-then-ours",
      callback: this.getResolverUsing([_source.THEIRS, _source.OURS])
    }), _react["default"].createElement(_commands.Command, {
      command: "github:resolve-as-current",
      callback: this.resolveAsCurrent
    }), _react["default"].createElement(_commands.Command, {
      command: "github:revert-conflict-modifications",
      callback: this.revertConflictModifications
    }), _react["default"].createElement(_commands.Command, {
      command: "github:dismiss-conflict",
      callback: this.dismissCurrent
    })), Array.from(this.state.conflicts, c => _react["default"].createElement(_conflictController["default"], {
      key: c.getKey(),
      editor: this.props.editor,
      conflict: c,
      resolveAsSequence: sources => this.resolveAsSequence(c, sources),
      dismiss: () => this.dismissConflicts([c])
    })));
  }

  componentWillUnmount() {
    // this.layer.destroy();
    this.subscriptions.dispose();
  }
  /*
   * Return an Array containing `Conflict` objects whose marked regions include any cursor position in the current
   * `TextEditor` and the `Sides` that contain a cursor within each.
   *
   * This method is written to have linear complexity with respect to the number of cursors and the number of
   * conflicts, to gracefully handle files with large numbers of both.
   */


  getCurrentConflicts() {
    const cursorPositions = this.props.editor.getCursorBufferPositions();
    cursorPositions.sort((a, b) => a.compare(b));
    const cursorIterator = cursorPositions[Symbol.iterator]();
    const conflictIterator = this.state.conflicts.keys();
    let currentCursor = cursorIterator.next();
    let currentConflict = conflictIterator.next();
    const activeConflicts = [];

    while (!currentCursor.done && !currentConflict.done) {
      // Advance currentCursor to the first cursor beyond the earliest conflict.
      const earliestConflictPosition = currentConflict.value.getRange().start;

      while (!currentCursor.done && currentCursor.value.isLessThan(earliestConflictPosition)) {
        currentCursor = cursorIterator.next();
      } // Advance currentConflict until the first conflict that begins at a position after the current cursor.
      // Compare each to the current cursor, and add it to activeConflicts if it contains it.


      while (!currentConflict.done && !currentCursor.done && currentConflict.value.getRange().start.isLessThan(currentCursor.value)) {
        if (currentConflict.value.includesPoint(currentCursor.value)) {
          // Hit; determine which sides of this conflict contain cursors.
          const conflict = currentConflict.value;
          const endPosition = conflict.getRange().end;
          const sides = new Set();

          while (!currentCursor.done && currentCursor.value.isLessThan(endPosition)) {
            const side = conflict.getSideContaining(currentCursor.value);

            if (side) {
              sides.add(side);
            }

            currentCursor = cursorIterator.next();
          }

          activeConflicts.push({
            conflict,
            sides
          });
        }

        currentConflict = conflictIterator.next();
      }
    }

    return activeConflicts;
  }

  getResolverUsing(sequence) {
    return () => {
      this.getCurrentConflicts().forEach(match => this.resolveAsSequence(match.conflict, sequence));
    };
  }

  resolveAsCurrent() {
    this.getCurrentConflicts().forEach(match => {
      if (match.sides.size === 1) {
        const side = match.sides.keys().next().value;
        this.resolveAs(match.conflict, side.getSource());
      }
    });
  }

  revertConflictModifications() {
    this.getCurrentConflicts().forEach(match => {
      match.sides.forEach(side => {
        side.isModified() && side.revert();
        side.isBannerModified() && side.revertBanner();
      });
    });
  }

  dismissCurrent() {
    this.dismissConflicts(this.getCurrentConflicts().map(match => match.conflict));
  }

  dismissConflicts(conflicts) {
    this.setState(prevState => {
      const {
        added
      } = (0, _compareSets["default"])(new Set(conflicts), prevState.conflicts);
      return {
        conflicts: added
      };
    });
  }

  resolveAsSequence(conflict, sources) {
    const [firstSide, ...restOfSides] = sources.map(source => conflict.getSide(source)).filter(side => side);
    const textToAppend = restOfSides.map(side => side.getText()).join('');
    this.props.editor.transact(() => {
      // Append text from all but the first Side to the first Side. Adjust the following DisplayMarker so that only that
      // Side's marker includes the appended text, not the next one.
      const appendedRange = firstSide.appendText(textToAppend);
      const nextMarker = conflict.markerAfter(firstSide.getPosition());

      if (nextMarker) {
        nextMarker.setTailBufferPosition(appendedRange.end);
      }

      this.innerResolveAs(conflict, sources[0]);
    });
  }

  resolveAs(conflict, source) {
    this.props.editor.transact(() => {
      this.innerResolveAs(conflict, source);
    });
  }

  innerResolveAs(conflict, source) {
    conflict.resolveAs(source);
    const chosenSide = conflict.getChosenSide();

    if (!chosenSide.isBannerModified()) {
      chosenSide.deleteBanner();
    }

    const separator = conflict.getSeparator();

    if (!separator.isModified()) {
      separator["delete"]();
    }

    conflict.getUnchosenSides().forEach(side => {
      side.deleteBanner();
      side["delete"]();
    });
    this.updateMarkerCount();
  }

  reparseConflicts() {
    const newConflicts = new Set(_conflict["default"].allFromEditor(this.props.editor, this.layer, this.props.isRebase));
    this.setState({
      conflicts: newConflicts
    });
  }

  updateMarkerCount() {
    this.props.resolutionProgress.reportMarkerCount(this.props.editor.getPath(), Array.from(this.state.conflicts, c => !c.isResolved()).filter(b => b).length);
  }

}

exports["default"] = EditorConflictController;

_defineProperty(EditorConflictController, "propTypes", {
  editor: _propTypes["default"].object.isRequired,
  commandRegistry: _propTypes["default"].object.isRequired,
  resolutionProgress: _propTypes["default"].object.isRequired,
  isRebase: _propTypes["default"].bool.isRequired,
  refreshResolutionProgress: _propTypes["default"].func.isRequired
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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