"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _util = require("util");

var _observeModel = _interopRequireDefault(require("./observe-model"));

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const sortOrders = {
  'by key': (a, b) => a.key.localeCompare(b.key),
  'oldest first': (a, b) => b.age - a.age,
  'newest first': (a, b) => a.age - b.age,
  'most hits': (a, b) => b.hits - a.hits,
  'fewest hits': (a, b) => a.hits - b.hits
};

class GitCacheView extends _react["default"].Component {
  static buildURI() {
    return this.uriPattern;
  }

  constructor(props, context) {
    super(props, context);
    (0, _helpers.autobind)(this, 'fetchRepositoryData', 'fetchCacheData', 'renderCache', 'didSelectItem', 'clearCache');
    this.state = {
      order: 'by key'
    };
  }

  getURI() {
    return 'atom-github://debug/cache';
  }

  getTitle() {
    return 'GitHub Package Cache View';
  }

  serialize() {
    return null;
  }

  fetchRepositoryData(repository) {
    return repository.getCache();
  }

  fetchCacheData(cache) {
    const cached = {};
    const promises = [];
    const now = performance.now();

    for (const [key, value] of cache) {
      cached[key] = {
        hits: value.hits,
        age: now - value.createdAt
      };
      promises.push(value.promise.then(payload => (0, _util.inspect)(payload, {
        depth: 3,
        breakLength: 30
      }), err => `${err.message}\n${err.stack}`).then(resolved => {
        cached[key].value = resolved;
      }));
    }

    return Promise.all(promises).then(() => cached);
  }

  render() {
    return _react["default"].createElement(_observeModel["default"], {
      model: this.props.repository,
      fetchData: this.fetchRepositoryData
    }, cache => _react["default"].createElement(_observeModel["default"], {
      model: cache,
      fetchData: this.fetchCacheData
    }, this.renderCache));
  }

  renderCache(contents) {
    const rows = Object.keys(contents || {}).map(key => {
      return {
        key,
        age: contents[key].age,
        hits: contents[key].hits,
        content: contents[key].value
      };
    });
    rows.sort(sortOrders[this.state.order]);
    const orders = Object.keys(sortOrders);
    return _react["default"].createElement("div", {
      className: "github-CacheView"
    }, _react["default"].createElement("header", null, _react["default"].createElement("h1", null, "Cache contents"), _react["default"].createElement("p", null, _react["default"].createElement("span", {
      className: "badge"
    }, rows.length), " cached items")), _react["default"].createElement("main", null, _react["default"].createElement("p", {
      className: "github-CacheView-Controls"
    }, _react["default"].createElement("span", {
      className: "github-CacheView-Order"
    }, "order", _react["default"].createElement("select", {
      className: "input-select",
      onChange: this.didSelectItem,
      value: this.state.order
    }, orders.map(order => {
      return _react["default"].createElement("option", {
        key: order,
        value: order
      }, order);
    }))), _react["default"].createElement("span", {
      className: "github-CacheView-Clear"
    }, _react["default"].createElement("button", {
      className: "btn icon icon-trashcan",
      onClick: this.clearCache
    }, "Clear"))), _react["default"].createElement("table", null, _react["default"].createElement("thead", null, _react["default"].createElement("tr", null, _react["default"].createElement("td", {
      className: "github-CacheView-Key"
    }, "key"), _react["default"].createElement("td", {
      className: "github-CacheView-Age"
    }, "age"), _react["default"].createElement("td", {
      className: "github-CacheView-Hits"
    }, "hits"), _react["default"].createElement("td", {
      className: "github-CacheView-Content"
    }, "content"))), _react["default"].createElement("tbody", null, rows.map(row => _react["default"].createElement("tr", {
      key: row.key,
      className: "github-CacheView-Row"
    }, _react["default"].createElement("td", {
      className: "github-CacheView-Key"
    }, _react["default"].createElement("button", {
      className: "btn",
      onClick: () => this.didClickKey(row.key)
    }, row.key)), _react["default"].createElement("td", {
      className: "github-CacheView-Age"
    }, this.formatAge(row.age)), _react["default"].createElement("td", {
      className: "github-CacheView-Hits"
    }, row.hits), _react["default"].createElement("td", {
      className: "github-CacheView-Content"
    }, _react["default"].createElement("code", null, row.content))))))));
  }

  formatAge(ageMs) {
    let remaining = ageMs;
    const parts = [];

    if (remaining > 3600000) {
      const hours = Math.floor(remaining / 3600000);
      parts.push(`${hours}h`);
      remaining -= 3600000 * hours;
    }

    if (remaining > 60000) {
      const minutes = Math.floor(remaining / 60000);
      parts.push(`${minutes}m`);
      remaining -= 60000 * minutes;
    }

    if (remaining > 1000) {
      const seconds = Math.floor(remaining / 1000);
      parts.push(`${seconds}s`);
      remaining -= 1000 * seconds;
    }

    parts.push(`${Math.floor(remaining)}ms`);
    return parts.slice(parts.length - 2).join(' ');
  }

  didSelectItem(event) {
    this.setState({
      order: event.target.value
    });
  }

  didClickKey(key) {
    const cache = this.props.repository.getCache();

    if (!cache) {
      return;
    }

    cache.removePrimary(key);
  }

  clearCache() {
    const cache = this.props.repository.getCache();

    if (!cache) {
      return;
    }

    cache.clear();
  }

}

exports["default"] = GitCacheView;

_defineProperty(GitCacheView, "uriPattern", 'atom-github://debug/cache');

_defineProperty(GitCacheView, "propTypes", {
  repository: _propTypes["default"].object.isRequired
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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