"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = exports.FOCUS = void 0;

var _path = _interopRequireDefault(require("path"));

var _eventKit = require("event-kit");

var _atom = require("atom");

var _eventLogger = _interopRequireDefault(require("./event-logger"));

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

const FOCUS = Symbol('focus');
exports.FOCUS = FOCUS;

class WorkspaceChangeObserver {
  constructor(window, workspace, repository) {
    (0, _helpers.autobind)(this, 'observeTextEditor');
    this.window = window;
    this.repository = repository;
    this.workspace = workspace;
    this.observedBuffers = new WeakSet();
    this.emitter = new _eventKit.Emitter();
    this.disposables = new _eventKit.CompositeDisposable();
    this.logger = new _eventLogger["default"]('workspace watcher');
    this.started = false;
  }

  async start() {
    const focusHandler = event => {
      if (this.repository) {
        this.logger.showFocusEvent();
        this.didChange([{
          special: FOCUS
        }]);
      }
    };

    this.window.addEventListener('focus', focusHandler);
    this.disposables.add(this.workspace.observeTextEditors(this.observeTextEditor), new _eventKit.Disposable(() => this.window.removeEventListener('focus', focusHandler)));
    await this.watchActiveRepositoryGitDirectory();
    this.started = true;
    return this;
  }

  async destroy() {
    this.started = false;
    this.observedBuffers = new WeakSet();
    this.emitter.dispose();
    this.disposables.dispose();
    await this.stopCurrentFileWatcher();
  }

  isStarted() {
    return this.started;
  }

  didChange(payload) {
    this.emitter.emit('did-change', payload);
  }

  didChangeWorkdirOrHead() {
    this.emitter.emit('did-change-workdir-or-head');
  }

  onDidChange(callback) {
    return this.emitter.on('did-change', callback);
  }

  onDidChangeWorkdirOrHead(callback) {
    return this.emitter.on('did-change-workdir-or-head', callback);
  }

  getRepository() {
    return this.repository;
  }

  async watchActiveRepositoryGitDirectory() {
    const repository = this.getRepository();
    const gitDirectoryPath = repository.getGitDirectoryPath();
    const basenamesOfInterest = ['config', 'index', 'HEAD', 'MERGE_HEAD'];
    const workdirOrHeadBasenames = ['config', 'index'];

    const eventPaths = event => {
      const ps = [event.path];

      if (event.oldPath) {
        ps.push(event.oldPath);
      }

      return ps;
    };

    const acceptEvent = event => {
      return eventPaths(event).some(eventPath => {
        return basenamesOfInterest.includes(_path["default"].basename(eventPath)) || _path["default"].dirname(eventPath).includes(_path["default"].join('.git', 'refs'));
      });
    };

    const isWorkdirOrHeadEvent = event => {
      return eventPaths(event).some(eventPath => workdirOrHeadBasenames.includes(_path["default"].basename(eventPath)));
    };

    this.currentFileWatcher = await (0, _atom.watchPath)(gitDirectoryPath, {}, events => {
      const filteredEvents = events.filter(acceptEvent);

      if (filteredEvents.length) {
        this.logger.showEvents(filteredEvents);
        this.didChange(filteredEvents);

        if (filteredEvents.some(isWorkdirOrHeadEvent)) {
          this.logger.showWorkdirOrHeadEvents();
          this.didChangeWorkdirOrHead();
        }
      }
    });
    this.currentFileWatcher.onDidError(error => {
      const workingDirectory = repository.getWorkingDirectoryPath(); // eslint-disable-next-line no-console

      console.warn(`Error in WorkspaceChangeObserver in ${workingDirectory}:`, error);
      this.stopCurrentFileWatcher();
    });
    this.logger.showStarted(gitDirectoryPath, 'workspace emulated');
  }

  stopCurrentFileWatcher() {
    if (this.currentFileWatcher) {
      this.currentFileWatcher.dispose();
      this.currentFileWatcher = null;
      this.logger.showStopped();
    }

    return Promise.resolve();
  }

  activeRepositoryContainsPath(filePath) {
    const repository = this.getRepository();

    if (filePath && repository) {
      return filePath.indexOf(repository.getWorkingDirectoryPath()) !== -1;
    } else {
      return false;
    }
  }

  observeTextEditor(editor) {
    const buffer = editor.getBuffer();

    if (!this.observedBuffers.has(buffer)) {
      let lastPath = buffer.getPath();

      const didChange = () => {
        const currentPath = buffer.getPath();
        const events = currentPath === lastPath ? [{
          action: 'modified',
          path: currentPath
        }] : [{
          action: 'renamed',
          path: currentPath,
          oldPath: lastPath
        }];
        lastPath = currentPath;
        this.logger.showEvents(events);
        this.didChange(events);
      };

      this.observedBuffers.add(buffer);
      const disposables = new _eventKit.CompositeDisposable(buffer.onDidSave(() => {
        if (this.activeRepositoryContainsPath(buffer.getPath())) {
          didChange();
        }
      }), buffer.onDidReload(() => {
        if (this.activeRepositoryContainsPath(buffer.getPath())) {
          didChange();
        }
      }), buffer.onDidDestroy(() => {
        if (this.activeRepositoryContainsPath(buffer.getPath())) {
          didChange();
        }

        disposables.dispose();
      }));
      this.disposables.add(disposables);
    }
  }

}

exports["default"] = WorkspaceChangeObserver;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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