"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.autobind = autobind;
exports.extractProps = extractProps;
exports.unusedProps = unusedProps;
exports.getPackageRoot = getPackageRoot;
exports.getAtomHelperPath = getAtomHelperPath;
exports.getDugitePath = getDugitePath;
exports.getSharedModulePath = getSharedModulePath;
exports.isBinary = isBinary;
exports.firstImplementer = firstImplementer;
exports.isValidWorkdir = isValidWorkdir;
exports.fileExists = fileExists;
exports.getTempDir = getTempDir;
exports.isFileExecutable = isFileExecutable;
exports.isFileSymlink = isFileSymlink;
exports.shortenSha = shortenSha;
exports.normalizeGitHelperPath = normalizeGitHelperPath;
exports.toNativePathSep = toNativePathSep;
exports.toGitPathSep = toGitPathSep;
exports.filePathEndsWith = filePathEndsWith;
exports.toSentence = toSentence;
exports.pushAtKey = pushAtKey;
exports.getCommitMessagePath = getCommitMessagePath;
exports.getCommitMessageEditors = getCommitMessageEditors;
exports.getFilePatchPaneItems = getFilePatchPaneItems;
exports.destroyFilePatchPaneItems = destroyFilePatchPaneItems;
exports.destroyEmptyFilePatchPaneItems = destroyEmptyFilePatchPaneItems;
exports.extractCoAuthorsAndRawCommitMessage = extractCoAuthorsAndRawCommitMessage;
exports.createItem = createItem;
exports.equalSets = equalSets;
exports.blankLabel = blankLabel;
exports.GHOST_USER = exports.reactionTypeToEmoji = exports.NBSP_CHARACTER = exports.classNameForStatus = exports.PAGINATION_WAIT_TIME_MS = exports.PAGE_SIZE = exports.CO_AUTHOR_REGEX = exports.LINE_ENDING_REGEX = void 0;

var _path = _interopRequireDefault(require("path"));

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _os = _interopRequireDefault(require("os"));

var _temp = _interopRequireDefault(require("temp"));

var _refHolder = _interopRequireDefault(require("./models/ref-holder"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const LINE_ENDING_REGEX = /\r?\n/;
exports.LINE_ENDING_REGEX = LINE_ENDING_REGEX;
const CO_AUTHOR_REGEX = /^co-authored-by. (.+?) <(.+?)>$/i;
exports.CO_AUTHOR_REGEX = CO_AUTHOR_REGEX;
const PAGE_SIZE = 50;
exports.PAGE_SIZE = PAGE_SIZE;
const PAGINATION_WAIT_TIME_MS = 100;
exports.PAGINATION_WAIT_TIME_MS = PAGINATION_WAIT_TIME_MS;

function autobind(self, ...methods) {
  for (const method of methods) {
    if (typeof self[method] !== 'function') {
      throw new Error(`Unable to autobind method ${method}`);
    }

    self[method] = self[method].bind(self);
  }
} // Extract a subset of props chosen from a propTypes object from a component's props to pass to a different API.
//
// Usage:
//
// ```js
// const apiProps = {
//   zero: PropTypes.number.isRequired,
//   one: PropTypes.string,
//   two: PropTypes.object,
// };
//
// class Component extends React.Component {
//   static propTypes = {
//     ...apiProps,
//     extra: PropTypes.func,
//   }
//
//   action() {
//     const options = extractProps(this.props, apiProps);
//     // options contains zero, one, and two, but not extra
//   }
// }
// ```


function extractProps(props, propTypes, nameMap = {}) {
  return Object.keys(propTypes).reduce((opts, propName) => {
    if (props[propName] !== undefined) {
      const destPropName = nameMap[propName] || propName;
      opts[destPropName] = props[propName];
    }

    return opts;
  }, {});
} // The opposite of extractProps. Return a subset of props that do *not* appear in a component's prop types.


function unusedProps(props, propTypes) {
  return Object.keys(props).reduce((opts, propName) => {
    if (propTypes[propName] === undefined) {
      opts[propName] = props[propName];
    }

    return opts;
  }, {});
}

function getPackageRoot() {
  const {
    resourcePath
  } = atom.getLoadSettings();
  const currentFileWasRequiredFromSnapshot = !_path["default"].isAbsolute(__dirname);

  if (currentFileWasRequiredFromSnapshot) {
    return _path["default"].join(resourcePath, 'node_modules', 'github');
  } else {
    const packageRoot = _path["default"].resolve(__dirname, '..');

    if (_path["default"].extname(resourcePath) === '.asar') {
      if (packageRoot.indexOf(resourcePath) === 0) {
        return _path["default"].join(`${resourcePath}.unpacked`, 'node_modules', 'github');
      }
    }

    return packageRoot;
  }
}

function getAtomAppName() {
  const match = atom.getVersion().match(/-([A-Za-z]+)(\d+|-)/);

  if (match) {
    const channel = match[1];
    return `Atom ${channel.charAt(0).toUpperCase() + channel.slice(1)} Helper`;
  }

  return 'Atom Helper';
}

function getAtomHelperPath() {
  if (process.platform === 'darwin') {
    const appName = getAtomAppName();
    return _path["default"].resolve(process.resourcesPath, '..', 'Frameworks', `${appName}.app`, 'Contents', 'MacOS', appName);
  } else {
    return process.execPath;
  }
}

let DUGITE_PATH;

function getDugitePath() {
  if (!DUGITE_PATH) {
    DUGITE_PATH = require.resolve('dugite');

    if (!_path["default"].isAbsolute(DUGITE_PATH)) {
      // Assume we're snapshotted
      const {
        resourcePath
      } = atom.getLoadSettings();

      if (_path["default"].extname(resourcePath) === '.asar') {
        DUGITE_PATH = _path["default"].join(`${resourcePath}.unpacked`, 'node_modules', 'dugite');
      } else {
        DUGITE_PATH = _path["default"].join(resourcePath, 'node_modules', 'dugite');
      }
    }
  }

  return DUGITE_PATH;
}

const SHARED_MODULE_PATHS = new Map();

function getSharedModulePath(relPath) {
  let modulePath = SHARED_MODULE_PATHS.get(relPath);

  if (!modulePath) {
    modulePath = require.resolve(_path["default"].join(__dirname, 'shared', relPath));

    if (!_path["default"].isAbsolute(modulePath)) {
      // Assume we're snapshotted
      const {
        resourcePath
      } = atom.getLoadSettings();
      modulePath = _path["default"].join(resourcePath, modulePath);
    }

    SHARED_MODULE_PATHS.set(relPath, modulePath);
  }

  return modulePath;
}

function isBinary(data) {
  for (let i = 0; i < 50; i++) {
    const code = data.charCodeAt(i); // Char code 65533 is the "replacement character";
    // 8 and below are control characters.

    if (code === 65533 || code < 9) {
      return true;
    }
  }

  return false;
}

function descriptorsFromProto(proto) {
  return Object.getOwnPropertyNames(proto).reduce((acc, name) => {
    Object.assign(acc, {
      [name]: Reflect.getOwnPropertyDescriptor(proto, name)
    });
    return acc;
  }, {});
}
/**
 * Takes an array of targets and returns a proxy. The proxy intercepts property accessor calls and
 * returns the value of that property on the first object in `targets` where the target implements that property.
 */


function firstImplementer(...targets) {
  return new Proxy({
    __implementations: targets
  }, {
    get(target, name) {
      if (name === 'getImplementers') {
        return () => targets;
      }

      if (Reflect.has(target, name)) {
        return target[name];
      }

      const firstValidTarget = targets.find(t => Reflect.has(t, name));

      if (firstValidTarget) {
        return firstValidTarget[name];
      } else {
        return undefined;
      }
    },

    set(target, name, value) {
      const firstValidTarget = targets.find(t => Reflect.has(t, name));

      if (firstValidTarget) {
        // eslint-disable-next-line no-return-assign
        return firstValidTarget[name] = value;
      } else {
        // eslint-disable-next-line no-return-assign
        return target[name] = value;
      }
    },

    // Used by sinon
    has(target, name) {
      if (name === 'getImplementers') {
        return true;
      }

      return targets.some(t => Reflect.has(t, name));
    },

    // Used by sinon
    getOwnPropertyDescriptor(target, name) {
      const firstValidTarget = targets.find(t => Reflect.getOwnPropertyDescriptor(t, name));
      const compositeOwnPropertyDescriptor = Reflect.getOwnPropertyDescriptor(target, name);

      if (firstValidTarget) {
        return Reflect.getOwnPropertyDescriptor(firstValidTarget, name);
      } else if (compositeOwnPropertyDescriptor) {
        return compositeOwnPropertyDescriptor;
      } else {
        return undefined;
      }
    },

    // Used by sinon
    getPrototypeOf(target) {
      return targets.reduceRight((acc, t) => {
        return Object.create(acc, descriptorsFromProto(Object.getPrototypeOf(t)));
      }, Object.prototype);
    }

  });
}

function isRoot(dir) {
  return _path["default"].resolve(dir, '..') === dir;
}

function isValidWorkdir(dir) {
  return dir !== _os["default"].homedir() && !isRoot(dir);
}

async function fileExists(absoluteFilePath) {
  try {
    await _fsExtra["default"].access(absoluteFilePath);
    return true;
  } catch (e) {
    if (e.code === 'ENOENT') {
      return false;
    }

    throw e;
  }
}

function getTempDir(options = {}) {
  _temp["default"].track();

  return new Promise((resolve, reject) => {
    _temp["default"].mkdir(options, (tempError, folder) => {
      if (tempError) {
        reject(tempError);
        return;
      }

      if (options.symlinkOk) {
        resolve(folder);
      } else {
        _fsExtra["default"].realpath(folder, (realError, rpath) => realError ? reject(realError) : resolve(rpath));
      }
    });
  });
}

async function isFileExecutable(absoluteFilePath) {
  const stat = await _fsExtra["default"].stat(absoluteFilePath);
  return stat.mode & _fsExtra["default"].constants.S_IXUSR; // eslint-disable-line no-bitwise
}

async function isFileSymlink(absoluteFilePath) {
  const stat = await _fsExtra["default"].lstat(absoluteFilePath);
  return stat.isSymbolicLink();
}

function shortenSha(sha) {
  return sha.slice(0, 8);
}

const classNameForStatus = {
  added: 'added',
  deleted: 'removed',
  modified: 'modified',
  typechange: 'modified',
  equivalent: 'ignored'
};
/*
 * Apply any platform-specific munging to a path before presenting it as
 * a git environment variable or option.
 *
 * Convert a Windows-style "C:\foo\bar\baz" path to a "/c/foo/bar/baz" UNIX-y
 * path that the sh.exe used to execute git's credential helpers will
 * understand.
 */

exports.classNameForStatus = classNameForStatus;

function normalizeGitHelperPath(inPath) {
  if (process.platform === 'win32') {
    return inPath.replace(/\\/g, '/').replace(/^([^:]+):/, '/$1');
  } else {
    return inPath;
  }
}
/*
 * On Windows, git commands report paths with / delimiters. Convert them to \-delimited paths
 * so that Atom unifromly treats paths with native path separators.
 */


function toNativePathSep(rawPath) {
  if (process.platform !== 'win32') {
    return rawPath;
  } else {
    return rawPath.split('/').join(_path["default"].sep);
  }
}
/*
 * Convert Windows paths back to /-delimited paths to be presented to git.
 */


function toGitPathSep(rawPath) {
  if (process.platform !== 'win32') {
    return rawPath;
  } else {
    return rawPath.split(_path["default"].sep).join('/');
  }
}

function filePathEndsWith(filePath, ...segments) {
  return filePath.endsWith(_path["default"].join(...segments));
}
/**
 * Turns an array of things @kuychaco cannot eat
 * into a sentence containing things @kuychaco cannot eat
 *
 * ['toast'] => 'toast'
 * ['toast', 'eggs'] => 'toast and eggs'
 * ['toast', 'eggs', 'cheese'] => 'toast, eggs, and cheese'
 *
 * Oxford comma included because you're wrong, shut up.
 */


function toSentence(array) {
  const len = array.length;

  if (len === 1) {
    return `${array[0]}`;
  } else if (len === 2) {
    return `${array[0]} and ${array[1]}`;
  }

  return array.reduce((acc, item, idx) => {
    if (idx === 0) {
      return `${item}`;
    } else if (idx === len - 1) {
      return `${acc}, and ${item}`;
    } else {
      return `${acc}, ${item}`;
    }
  }, '');
}

function pushAtKey(map, key, value) {
  let existing = map.get(key);

  if (!existing) {
    existing = [];
    map.set(key, existing);
  }

  existing.push(value);
} // Repository and workspace helpers


function getCommitMessagePath(repository) {
  return _path["default"].join(repository.getGitDirectoryPath(), 'ATOM_COMMIT_EDITMSG');
}

function getCommitMessageEditors(repository, workspace) {
  if (!repository.isPresent()) {
    return [];
  }

  return workspace.getTextEditors().filter(editor => editor.getPath() === getCommitMessagePath(repository));
}

let ChangedFileItem = null;

function getFilePatchPaneItems({
  onlyStaged,
  empty
} = {}, workspace) {
  if (ChangedFileItem === null) {
    ChangedFileItem = require('./items/changed-file-item')["default"];
  }

  return workspace.getPaneItems().filter(item => {
    const isFilePatchItem = item && item.getRealItem && item.getRealItem() instanceof ChangedFileItem;

    if (onlyStaged) {
      return isFilePatchItem && item.stagingStatus === 'staged';
    } else if (empty) {
      return isFilePatchItem ? item.isEmpty() : false;
    } else {
      return isFilePatchItem;
    }
  });
}

function destroyFilePatchPaneItems({
  onlyStaged
} = {}, workspace) {
  const itemsToDestroy = getFilePatchPaneItems({
    onlyStaged
  }, workspace);
  itemsToDestroy.forEach(item => item.destroy());
}

function destroyEmptyFilePatchPaneItems(workspace) {
  const itemsToDestroy = getFilePatchPaneItems({
    empty: true
  }, workspace);
  itemsToDestroy.forEach(item => item.destroy());
}

function extractCoAuthorsAndRawCommitMessage(commitMessage) {
  const messageLines = [];
  const coAuthors = [];

  for (const line of commitMessage.split(LINE_ENDING_REGEX)) {
    const match = line.match(CO_AUTHOR_REGEX);

    if (match) {
      // eslint-disable-next-line no-unused-vars
      const [_, name, email] = match;
      coAuthors.push({
        name,
        email
      });
    } else {
      messageLines.push(line);
    }
  }

  return {
    message: messageLines.join('\n'),
    coAuthors
  };
} // Atom API pane item manipulation


function createItem(node, componentHolder = null, uri = null, extra = {}) {
  const holder = componentHolder || new _refHolder["default"]();

  const override = _objectSpread({
    getElement: () => node,
    getRealItem: () => holder.getOr(null),
    getRealItemPromise: () => holder.getPromise()
  }, extra);

  if (uri) {
    override.getURI = () => uri;
  }

  if (componentHolder) {
    return new Proxy(override, {
      get(target, name) {
        if (Reflect.has(target, name)) {
          return target[name];
        } // The {value: ...} wrapper prevents .map() from flattening a returned RefHolder.
        // If component[name] is a RefHolder, we want to return that RefHolder as-is.


        const {
          value
        } = holder.map(component => ({
          value: component[name]
        })).getOr({
          value: undefined
        });
        return value;
      },

      set(target, name, value) {
        return holder.map(component => {
          component[name] = value;
          return true;
        }).getOr(true);
      },

      has(target, name) {
        return holder.map(component => Reflect.has(component, name)).getOr(false) || Reflect.has(target, name);
      }

    });
  } else {
    return override;
  }
} // Set functions


function equalSets(left, right) {
  if (left.size !== right.size) {
    return false;
  }

  for (const each of left) {
    if (!right.has(each)) {
      return false;
    }
  }

  return true;
} // Constants


const NBSP_CHARACTER = '\u00a0';
exports.NBSP_CHARACTER = NBSP_CHARACTER;

function blankLabel() {
  return NBSP_CHARACTER;
}

const reactionTypeToEmoji = {
  THUMBS_UP: '👍',
  THUMBS_DOWN: '👎',
  LAUGH: '😆',
  HOORAY: '🎉',
  CONFUSED: '😕',
  HEART: '❤️',
  ROCKET: '🚀',
  EYES: '👀'
};
exports.reactionTypeToEmoji = reactionTypeToEmoji;
const GHOST_USER = {
  login: 'ghost',
  avatarUrl: 'https://avatars1.githubusercontent.com/u/10137?v=4',
  url: 'https://github.com/ghost'
};
exports.GHOST_USER = GHOST_USER;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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