/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.cluster.AckedClusterStateUpdateTask;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateApplier;
import org.elasticsearch.cluster.ack.AckedRequest;
import org.elasticsearch.cluster.ack.ClusterStateUpdateRequest;
import org.elasticsearch.cluster.ack.ClusterStateUpdateResponse;
import org.elasticsearch.cluster.metadata.MetaData;
import org.elasticsearch.cluster.metadata.RepositoriesMetaData;
import org.elasticsearch.cluster.metadata.RepositoryMetaData;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.regex.Regex;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.repositories.Repository;
import org.elasticsearch.repositories.RepositoryException;
import org.elasticsearch.repositories.RepositoryMissingException;
import org.elasticsearch.repositories.RepositoryVerificationException;
import org.elasticsearch.repositories.VerificationFailure;
import org.elasticsearch.repositories.VerifyNodeRepositoryAction;
import org.elasticsearch.snapshots.RestoreService;
import org.elasticsearch.snapshots.SnapshotsService;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.transport.TransportService;

public class RepositoriesService
implements ClusterStateApplier {
    private static final Logger logger = LogManager.getLogger(RepositoriesService.class);
    private final Map<String, Repository.Factory> typesRegistry;
    private final Map<String, Repository.Factory> internalTypesRegistry;
    private final ClusterService clusterService;
    private final ThreadPool threadPool;
    private final VerifyNodeRepositoryAction verifyAction;
    private final Map<String, Repository> internalRepositories = ConcurrentCollections.newConcurrentMap();
    private volatile Map<String, Repository> repositories = Collections.emptyMap();

    public RepositoriesService(Settings settings, ClusterService clusterService, TransportService transportService, Map<String, Repository.Factory> typesRegistry, Map<String, Repository.Factory> internalTypesRegistry, ThreadPool threadPool) {
        this.typesRegistry = typesRegistry;
        this.internalTypesRegistry = internalTypesRegistry;
        this.clusterService = clusterService;
        this.threadPool = threadPool;
        if (DiscoveryNode.isDataNode(settings) || DiscoveryNode.isMasterNode(settings)) {
            clusterService.addStateApplier(this);
        }
        this.verifyAction = new VerifyNodeRepositoryAction(transportService, clusterService, this);
    }

    public void registerRepository(final RegisterRepositoryRequest request, ActionListener<ClusterStateUpdateResponse> listener) {
        final RepositoryMetaData newRepositoryMetaData = new RepositoryMetaData(request.name, request.type, request.settings);
        VerifyingRegisterRepositoryListener registrationListener = request.verify ? new VerifyingRegisterRepositoryListener(request.name, listener) : listener;
        try {
            this.closeRepository(this.createRepository(newRepositoryMetaData, this.typesRegistry));
        }
        catch (Exception e) {
            registrationListener.onFailure(e);
            return;
        }
        this.clusterService.submitStateUpdateTask(request.cause, new AckedClusterStateUpdateTask<ClusterStateUpdateResponse>((AckedRequest)request, (ActionListener)registrationListener){

            @Override
            protected ClusterStateUpdateResponse newResponse(boolean acknowledged) {
                return new ClusterStateUpdateResponse(acknowledged);
            }

            @Override
            public ClusterState execute(ClusterState currentState) {
                RepositoriesService.this.ensureRepositoryNotInUse(currentState, request.name);
                MetaData metaData = currentState.metaData();
                MetaData.Builder mdBuilder = MetaData.builder(currentState.metaData());
                RepositoriesMetaData repositories = (RepositoriesMetaData)metaData.custom("repositories");
                if (repositories == null) {
                    logger.info("put repository [{}]", (Object)request.name);
                    repositories = new RepositoriesMetaData(Collections.singletonList(new RepositoryMetaData(request.name, request.type, request.settings)));
                } else {
                    boolean found = false;
                    ArrayList<RepositoryMetaData> repositoriesMetaData = new ArrayList<RepositoryMetaData>(repositories.repositories().size() + 1);
                    for (RepositoryMetaData repositoryMetaData : repositories.repositories()) {
                        if (repositoryMetaData.name().equals(newRepositoryMetaData.name())) {
                            if (newRepositoryMetaData.equals(repositoryMetaData)) {
                                return currentState;
                            }
                            found = true;
                            repositoriesMetaData.add(newRepositoryMetaData);
                            continue;
                        }
                        repositoriesMetaData.add(repositoryMetaData);
                    }
                    if (!found) {
                        logger.info("put repository [{}]", (Object)request.name);
                        repositoriesMetaData.add(new RepositoryMetaData(request.name, request.type, request.settings));
                    } else {
                        logger.info("update repository [{}]", (Object)request.name);
                    }
                    repositories = new RepositoriesMetaData(repositoriesMetaData);
                }
                mdBuilder.putCustom("repositories", repositories);
                return ClusterState.builder(currentState).metaData(mdBuilder).build();
            }

            @Override
            public void onFailure(String source, Exception e) {
                logger.warn(() -> new ParameterizedMessage("failed to create repository [{}]", (Object)request2.name), (Throwable)e);
                super.onFailure(source, e);
            }

            @Override
            public boolean mustAck(DiscoveryNode discoveryNode) {
                return discoveryNode.isMasterNode() || discoveryNode.isDataNode();
            }
        });
    }

    public void unregisterRepository(final UnregisterRepositoryRequest request, ActionListener<ClusterStateUpdateResponse> listener) {
        this.clusterService.submitStateUpdateTask(request.cause, new AckedClusterStateUpdateTask<ClusterStateUpdateResponse>((AckedRequest)request, listener){

            @Override
            protected ClusterStateUpdateResponse newResponse(boolean acknowledged) {
                return new ClusterStateUpdateResponse(acknowledged);
            }

            @Override
            public ClusterState execute(ClusterState currentState) {
                RepositoriesService.this.ensureRepositoryNotInUse(currentState, request.name);
                MetaData metaData = currentState.metaData();
                MetaData.Builder mdBuilder = MetaData.builder(currentState.metaData());
                RepositoriesMetaData repositories = (RepositoriesMetaData)metaData.custom("repositories");
                if (repositories != null && repositories.repositories().size() > 0) {
                    ArrayList<RepositoryMetaData> repositoriesMetaData = new ArrayList<RepositoryMetaData>(repositories.repositories().size());
                    boolean changed = false;
                    for (RepositoryMetaData repositoryMetaData : repositories.repositories()) {
                        if (Regex.simpleMatch(request.name, repositoryMetaData.name())) {
                            logger.info("delete repository [{}]", (Object)repositoryMetaData.name());
                            changed = true;
                            continue;
                        }
                        repositoriesMetaData.add(repositoryMetaData);
                    }
                    if (changed) {
                        repositories = new RepositoriesMetaData(repositoriesMetaData);
                        mdBuilder.putCustom("repositories", repositories);
                        return ClusterState.builder(currentState).metaData(mdBuilder).build();
                    }
                }
                if (Regex.isMatchAllPattern(request.name)) {
                    return currentState;
                }
                throw new RepositoryMissingException(request.name);
            }

            @Override
            public boolean mustAck(DiscoveryNode discoveryNode) {
                return discoveryNode.isMasterNode() || discoveryNode.isDataNode();
            }
        });
    }

    public void verifyRepository(final String repositoryName, final ActionListener<VerifyResponse> listener) {
        final Repository repository = this.repository(repositoryName);
        try {
            this.threadPool.executor("snapshot").execute(() -> {
                try {
                    final String verificationToken = repository.startVerification();
                    if (verificationToken != null) {
                        try {
                            this.verifyAction.verify(repositoryName, verificationToken, new ActionListener<VerifyResponse>(){

                                @Override
                                public void onResponse(VerifyResponse verifyResponse) {
                                    RepositoriesService.this.threadPool.executor("snapshot").execute(() -> {
                                        try {
                                            repository.endVerification(verificationToken);
                                        }
                                        catch (Exception e) {
                                            logger.warn(() -> new ParameterizedMessage("[{}] failed to finish repository verification", (Object)repositoryName), (Throwable)e);
                                            listener.onFailure(e);
                                            return;
                                        }
                                        listener.onResponse(verifyResponse);
                                    });
                                }

                                @Override
                                public void onFailure(Exception e) {
                                    listener.onFailure(e);
                                }
                            });
                        }
                        catch (Exception e) {
                            this.threadPool.executor("snapshot").execute(() -> {
                                try {
                                    repository.endVerification(verificationToken);
                                }
                                catch (Exception inner) {
                                    inner.addSuppressed(e);
                                    logger.warn(() -> new ParameterizedMessage("[{}] failed to finish repository verification", (Object)repositoryName), (Throwable)inner);
                                }
                                listener.onFailure(e);
                            });
                        }
                    } else {
                        listener.onResponse(new VerifyResponse(new DiscoveryNode[0], new VerificationFailure[0]));
                    }
                }
                catch (Exception e) {
                    listener.onFailure(e);
                }
            });
        }
        catch (Exception e) {
            listener.onFailure(e);
        }
    }

    @Override
    public void applyClusterState(ClusterChangedEvent event) {
        try {
            RepositoriesMetaData oldMetaData = (RepositoriesMetaData)event.previousState().getMetaData().custom("repositories");
            RepositoriesMetaData newMetaData = (RepositoriesMetaData)event.state().getMetaData().custom("repositories");
            if (oldMetaData == null && newMetaData == null || oldMetaData != null && oldMetaData.equals(newMetaData)) {
                return;
            }
            logger.trace("processing new index repositories for state version [{}]", (Object)event.state().version());
            HashMap<String, Repository> survivors = new HashMap<String, Repository>();
            for (Map.Entry<String, Repository> entry : this.repositories.entrySet()) {
                if (newMetaData == null || newMetaData.repository(entry.getKey()) == null) {
                    logger.debug("unregistering repository [{}]", (Object)entry.getKey());
                    this.closeRepository(entry.getValue());
                    continue;
                }
                survivors.put(entry.getKey(), entry.getValue());
            }
            HashMap<String, Repository> builder = new HashMap<String, Repository>();
            if (newMetaData != null) {
                for (RepositoryMetaData repositoryMetaData : newMetaData.repositories()) {
                    Repository repository = (Repository)survivors.get(repositoryMetaData.name());
                    if (repository != null) {
                        RepositoryMetaData previousMetadata = repository.getMetadata();
                        if (!previousMetadata.type().equals(repositoryMetaData.type()) || !previousMetadata.settings().equals(repositoryMetaData.settings())) {
                            logger.debug("updating repository [{}]", (Object)repositoryMetaData.name());
                            this.closeRepository(repository);
                            repository = null;
                            try {
                                repository = this.createRepository(repositoryMetaData, this.typesRegistry);
                            }
                            catch (RepositoryException ex) {
                                logger.warn(() -> new ParameterizedMessage("failed to change repository [{}]", (Object)repositoryMetaData.name()), (Throwable)ex);
                            }
                        }
                    } else {
                        try {
                            repository = this.createRepository(repositoryMetaData, this.typesRegistry);
                        }
                        catch (RepositoryException ex) {
                            logger.warn(() -> new ParameterizedMessage("failed to create repository [{}]", (Object)repositoryMetaData.name()), (Throwable)ex);
                        }
                    }
                    if (repository == null) continue;
                    logger.debug("registering repository [{}]", (Object)repositoryMetaData.name());
                    builder.put(repositoryMetaData.name(), repository);
                }
            }
            this.repositories = Collections.unmodifiableMap(builder);
        }
        catch (Exception ex) {
            logger.warn("failure updating cluster state ", (Throwable)ex);
        }
    }

    public Repository repository(String repositoryName) {
        Repository repository = this.repositories.get(repositoryName);
        if (repository != null) {
            return repository;
        }
        repository = this.internalRepositories.get(repositoryName);
        if (repository != null) {
            return repository;
        }
        throw new RepositoryMissingException(repositoryName);
    }

    public void registerInternalRepository(String name, String type) {
        RepositoryMetaData metaData = new RepositoryMetaData(name, type, Settings.EMPTY);
        Repository repository = this.internalRepositories.computeIfAbsent(name, n -> {
            logger.debug("put internal repository [{}][{}]", (Object)name, (Object)type);
            return this.createRepository(metaData, this.internalTypesRegistry);
        });
        if (!type.equals(repository.getMetadata().type())) {
            logger.warn((Message)new ParameterizedMessage("internal repository [{}][{}] already registered. this prevented the registration of internal repository [{}][{}].", new Object[]{name, repository.getMetadata().type(), name, type}));
        } else if (this.repositories.containsKey(name)) {
            logger.warn((Message)new ParameterizedMessage("non-internal repository [{}] already registered. this repository will block the usage of internal repository [{}][{}].", new Object[]{name, metaData.type(), name}));
        }
    }

    public void unregisterInternalRepository(String name) {
        Repository repository = this.internalRepositories.remove(name);
        if (repository != null) {
            RepositoryMetaData metadata = repository.getMetadata();
            logger.debug(() -> new ParameterizedMessage("delete internal repository [{}][{}].", (Object)metadata.type(), (Object)name));
            this.closeRepository(repository);
        }
    }

    private void closeRepository(Repository repository) {
        logger.debug("closing repository [{}][{}]", (Object)repository.getMetadata().type(), (Object)repository.getMetadata().name());
        repository.close();
    }

    private Repository createRepository(RepositoryMetaData repositoryMetaData, Map<String, Repository.Factory> factories) {
        logger.debug("creating repository [{}][{}]", (Object)repositoryMetaData.type(), (Object)repositoryMetaData.name());
        Repository.Factory factory = factories.get(repositoryMetaData.type());
        if (factory == null) {
            throw new RepositoryException(repositoryMetaData.name(), "repository type [" + repositoryMetaData.type() + "] does not exist");
        }
        try {
            Repository repository = factory.create(repositoryMetaData, factories::get);
            repository.start();
            return repository;
        }
        catch (Exception e) {
            logger.warn((Message)new ParameterizedMessage("failed to create repository [{}][{}]", (Object)repositoryMetaData.type(), (Object)repositoryMetaData.name()), (Throwable)e);
            throw new RepositoryException(repositoryMetaData.name(), "failed to create repository", e);
        }
    }

    private void ensureRepositoryNotInUse(ClusterState clusterState, String repository) {
        if (SnapshotsService.isRepositoryInUse(clusterState, repository) || RestoreService.isRepositoryInUse(clusterState, repository)) {
            throw new IllegalStateException("trying to modify or unregister repository that is currently used ");
        }
    }

    public static class VerifyResponse {
        private VerificationFailure[] failures;
        private DiscoveryNode[] nodes;

        public VerifyResponse(DiscoveryNode[] nodes, VerificationFailure[] failures) {
            this.nodes = nodes;
            this.failures = failures;
        }

        public VerificationFailure[] failures() {
            return this.failures;
        }

        public DiscoveryNode[] nodes() {
            return this.nodes;
        }

        public boolean failed() {
            return this.failures.length > 0;
        }

        public String failureDescription() {
            return Arrays.stream(this.failures).map(failure -> failure.toString()).collect(Collectors.joining(", ", "[", "]"));
        }
    }

    public static class UnregisterRepositoryRequest
    extends ClusterStateUpdateRequest<UnregisterRepositoryRequest> {
        final String cause;
        final String name;

        public UnregisterRepositoryRequest(String cause, String name) {
            this.cause = cause;
            this.name = name;
        }
    }

    public static class RegisterRepositoryRequest
    extends ClusterStateUpdateRequest<RegisterRepositoryRequest> {
        final String cause;
        final String name;
        final String type;
        final boolean verify;
        Settings settings = Settings.EMPTY;

        public RegisterRepositoryRequest(String cause, String name, String type, boolean verify) {
            this.cause = cause;
            this.name = name;
            this.type = type;
            this.verify = verify;
        }

        public RegisterRepositoryRequest settings(Settings settings) {
            this.settings = settings;
            return this;
        }
    }

    private class VerifyingRegisterRepositoryListener
    implements ActionListener<ClusterStateUpdateResponse> {
        private final String name;
        private final ActionListener<ClusterStateUpdateResponse> listener;

        VerifyingRegisterRepositoryListener(String name, ActionListener<ClusterStateUpdateResponse> listener) {
            this.name = name;
            this.listener = listener;
        }

        @Override
        public void onResponse(final ClusterStateUpdateResponse clusterStateUpdateResponse) {
            if (clusterStateUpdateResponse.isAcknowledged()) {
                RepositoriesService.this.verifyRepository(this.name, new ActionListener<VerifyResponse>(){

                    @Override
                    public void onResponse(VerifyResponse verifyResponse) {
                        if (verifyResponse.failed()) {
                            VerifyingRegisterRepositoryListener.this.listener.onFailure(new RepositoryVerificationException(VerifyingRegisterRepositoryListener.this.name, verifyResponse.failureDescription()));
                        } else {
                            VerifyingRegisterRepositoryListener.this.listener.onResponse(clusterStateUpdateResponse);
                        }
                    }

                    @Override
                    public void onFailure(Exception e) {
                        VerifyingRegisterRepositoryListener.this.listener.onFailure(e);
                    }
                });
            } else {
                this.listener.onResponse(clusterStateUpdateResponse);
            }
        }

        @Override
        public void onFailure(Exception e) {
            this.listener.onFailure(e);
        }
    }
}

