(function() {
  var TextEditor, cheerio, convertAtomEditorToStandardElement, createDOMPurify, fs, highlightCodeBlocks, highlighter, makeAtomEditorNonInteractive, packagePath, path, render, resolveImagePaths, resourcePath, roaster, scopeForFenceName;

  TextEditor = require('atom').TextEditor;

  path = require('path');

  cheerio = require('cheerio');

  createDOMPurify = require('dompurify');

  fs = require('fs-plus');

  roaster = null;

  scopeForFenceName = require('./extension-helper').scopeForFenceName;

  highlighter = null;

  resourcePath = atom.getLoadSettings().resourcePath;

  packagePath = path.dirname(__dirname);

  exports.toDOMFragment = function(text, filePath, grammar, callback) {
    if (text == null) {
      text = '';
    }
    return render(text, filePath, function(error, domFragment) {
      if (error != null) {
        return callback(error);
      }
      return highlightCodeBlocks(domFragment, grammar, makeAtomEditorNonInteractive).then(function() {
        return callback(null, domFragment);
      });
    });
  };

  exports.toHTML = function(text, filePath, grammar, callback) {
    if (text == null) {
      text = '';
    }
    return render(text, filePath, function(error, domFragment) {
      var div;
      if (error != null) {
        return callback(error);
      }
      div = document.createElement('div');
      div.appendChild(domFragment);
      document.body.appendChild(div);
      return highlightCodeBlocks(div, grammar, convertAtomEditorToStandardElement).then(function() {
        callback(null, div.innerHTML);
        return div.remove();
      });
    });
  };

  render = function(text, filePath, callback) {
    var options;
    if (roaster == null) {
      roaster = require('roaster');
    }
    options = {
      sanitize: false,
      breaks: atom.config.get('markdown-preview.breakOnSingleNewline')
    };
    text = text.replace(/^\s*<!doctype(\s+.*)?>\s*/i, '');
    return roaster(text, options, function(error, html) {
      var fragment, template;
      if (error != null) {
        return callback(error);
      }
      html = createDOMPurify().sanitize(html, {
        ALLOW_UNKNOWN_PROTOCOLS: atom.config.get('markdown-preview.allowUnsafeProtocols')
      });
      template = document.createElement('template');
      template.innerHTML = html.trim();
      fragment = template.content.cloneNode(true);
      resolveImagePaths(fragment, filePath);
      return callback(null, fragment);
    });
  };

  resolveImagePaths = function(element, filePath) {
    var i, img, len, ref, results, rootDirectory, src;
    rootDirectory = atom.project.relativizePath(filePath)[0];
    ref = element.querySelectorAll('img');
    results = [];
    for (i = 0, len = ref.length; i < len; i++) {
      img = ref[i];
      if (src = img.getAttribute('src')) {
        if (src.match(/^(https?|atom):\/\//)) {
          continue;
        }
        if (src.startsWith(process.resourcesPath)) {
          continue;
        }
        if (src.startsWith(resourcePath)) {
          continue;
        }
        if (src.startsWith(packagePath)) {
          continue;
        }
        if (src[0] === '/') {
          if (!fs.isFileSync(src)) {
            if (rootDirectory) {
              results.push(img.src = path.join(rootDirectory, src.substring(1)));
            } else {
              results.push(void 0);
            }
          } else {
            results.push(void 0);
          }
        } else {
          results.push(img.src = path.resolve(path.dirname(filePath), src));
        }
      } else {
        results.push(void 0);
      }
    }
    return results;
  };

  highlightCodeBlocks = function(domFragment, grammar, editorCallback) {
    var codeElement, defaultLanguage, fn, fontFamily, i, j, len, len1, preElement, promises, ref, ref1;
    if ((grammar != null ? grammar.scopeName : void 0) === 'source.litcoffee') {
      defaultLanguage = 'coffee';
    } else {
      defaultLanguage = 'text';
    }
    if (fontFamily = atom.config.get('editor.fontFamily')) {
      ref = domFragment.querySelectorAll('code');
      for (i = 0, len = ref.length; i < len; i++) {
        codeElement = ref[i];
        codeElement.style.fontFamily = fontFamily;
      }
    }
    promises = [];
    ref1 = domFragment.querySelectorAll('pre');
    fn = function(preElement) {
      var codeBlock, editor, editorElement, fenceName, ref2, ref3, ref4;
      codeBlock = (ref2 = preElement.firstElementChild) != null ? ref2 : preElement;
      fenceName = (ref3 = (ref4 = codeBlock.getAttribute('class')) != null ? ref4.replace(/^lang-/, '') : void 0) != null ? ref3 : defaultLanguage;
      preElement.classList.add('editor-colors', "lang-" + fenceName);
      editor = new TextEditor({
        readonly: true,
        keyboardInputEnabled: false
      });
      editorElement = editor.getElement();
      editorElement.setUpdatedSynchronously(true);
      preElement.innerHTML = '';
      preElement.parentNode.insertBefore(editorElement, preElement);
      editor.setText(codeBlock.textContent.replace(/\r?\n$/, ''));
      atom.grammars.assignLanguageMode(editor, scopeForFenceName(fenceName));
      editor.setVisible(true);
      return promises.push(editorCallback(editorElement, preElement));
    };
    for (j = 0, len1 = ref1.length; j < len1; j++) {
      preElement = ref1[j];
      fn(preElement);
    }
    return Promise.all(promises);
  };

  makeAtomEditorNonInteractive = function(editorElement, preElement) {
    var cursorLineDecoration, i, len, ref;
    preElement.remove();
    editorElement.setAttributeNode(document.createAttribute('gutter-hidden'));
    editorElement.removeAttribute('tabindex');
    ref = editorElement.getModel().cursorLineDecorations;
    for (i = 0, len = ref.length; i < len; i++) {
      cursorLineDecoration = ref[i];
      cursorLineDecoration.destroy();
    }
  };

  convertAtomEditorToStandardElement = function(editorElement, preElement) {
    return new Promise(function(resolve) {
      var done, editor, languageMode;
      editor = editorElement.getModel();
      done = function() {
        return editor.component.getNextUpdatePromise().then(function() {
          var i, len, line, line2, ref;
          ref = editorElement.querySelectorAll('.line:not(.dummy)');
          for (i = 0, len = ref.length; i < len; i++) {
            line = ref[i];
            line2 = document.createElement('div');
            line2.className = 'line';
            line2.innerHTML = line.firstChild.innerHTML;
            preElement.appendChild(line2);
          }
          editorElement.remove();
          return resolve();
        });
      };
      languageMode = editor.getBuffer().getLanguageMode();
      if (languageMode.fullyTokenized || languageMode.tree) {
        return done();
      } else {
        return editor.onDidTokenize(done);
      }
    });
  };

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
