"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _eventKit = require("event-kit");

var _file = require("./file");

var _patch = _interopRequireWildcard(require("./patch"));

var _helpers = require("../../helpers");

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj.default = obj; return newObj; } }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class FilePatch {
  static createNull() {
    return new this(_file.nullFile, _file.nullFile, _patch.default.createNull());
  }

  static createHiddenFilePatch(oldFile, newFile, marker, renderStatus, showFn) {
    return new this(oldFile, newFile, _patch.default.createHiddenPatch(marker, renderStatus, showFn));
  }

  constructor(oldFile, newFile, patch) {
    this.oldFile = oldFile;
    this.newFile = newFile;
    this.patch = patch;
    this.emitter = new _eventKit.Emitter();
  }

  isPresent() {
    return this.oldFile.isPresent() || this.newFile.isPresent() || this.patch.isPresent();
  }

  getRenderStatus() {
    return this.patch.getRenderStatus();
  }

  getOldFile() {
    return this.oldFile;
  }

  getNewFile() {
    return this.newFile;
  }

  getPatch() {
    return this.patch;
  }

  getMarker() {
    return this.getPatch().getMarker();
  }

  getStartRange() {
    return this.getPatch().getStartRange();
  }

  getOldPath() {
    return this.getOldFile().getPath();
  }

  getNewPath() {
    return this.getNewFile().getPath();
  }

  getOldMode() {
    return this.getOldFile().getMode();
  }

  getNewMode() {
    return this.getNewFile().getMode();
  }

  getOldSymlink() {
    return this.getOldFile().getSymlink();
  }

  getNewSymlink() {
    return this.getNewFile().getSymlink();
  }

  getFirstChangeRange() {
    return this.getPatch().getFirstChangeRange();
  }

  getMaxLineNumberWidth() {
    return this.getPatch().getMaxLineNumberWidth();
  }

  containsRow(row) {
    return this.getPatch().containsRow(row);
  }

  didChangeExecutableMode() {
    if (!this.oldFile.isPresent() || !this.newFile.isPresent()) {
      return false;
    }

    return this.oldFile.isExecutable() && !this.newFile.isExecutable() || !this.oldFile.isExecutable() && this.newFile.isExecutable();
  }

  hasSymlink() {
    return Boolean(this.getOldFile().getSymlink() || this.getNewFile().getSymlink());
  }

  hasTypechange() {
    if (!this.oldFile.isPresent() || !this.newFile.isPresent()) {
      return false;
    }

    return this.oldFile.isSymlink() && !this.newFile.isSymlink() || !this.oldFile.isSymlink() && this.newFile.isSymlink();
  }

  getPath() {
    return this.getOldPath() || this.getNewPath();
  }

  getStatus() {
    return this.getPatch().getStatus();
  }

  getHunks() {
    return this.getPatch().getHunks();
  }

  updateMarkers(map) {
    return this.patch.updateMarkers(map);
  }

  triggerCollapseIn(patchBuffer, {
    before,
    after
  }) {
    if (!this.patch.getRenderStatus().isVisible()) {
      return false;
    }

    const oldPatch = this.patch;
    const oldRange = oldPatch.getRange().copy();
    const insertionPosition = oldRange.start;
    const exclude = new Set([...before, ...after]);
    const {
      patchBuffer: subPatchBuffer,
      markerMap
    } = patchBuffer.extractPatchBuffer(oldRange, {
      exclude
    });
    oldPatch.destroyMarkers();
    oldPatch.updateMarkers(markerMap); // Delete the separating newline after the collapsing patch, if any.

    if (!oldRange.isEmpty()) {
      patchBuffer.getBuffer().deleteRow(insertionPosition.row);
    }

    const patchMarker = patchBuffer.markPosition(_patch.default.layerName, insertionPosition, {
      invalidate: 'never',
      exclusive: true
    });
    this.patch = _patch.default.createHiddenPatch(patchMarker, _patch.COLLAPSED, () => {
      return {
        patch: oldPatch,
        patchBuffer: subPatchBuffer
      };
    });
    this.didChangeRenderStatus();
    return true;
  }

  triggerExpandIn(patchBuffer, {
    before,
    after
  }) {
    if (this.patch.getRenderStatus().isVisible()) {
      return false;
    }

    const {
      patch: nextPatch,
      patchBuffer: subPatchBuffer
    } = this.patch.show();
    const atStart = this.patch.getInsertionPoint().isEqual([0, 0]);
    const atEnd = this.patch.getInsertionPoint().isEqual(patchBuffer.getBuffer().getEndPosition());
    const willHaveContent = !subPatchBuffer.getBuffer().isEmpty(); // The expanding patch's insertion point is just after the unmarked newline that separates adjacent visible
    // patches:
    // <p0> '\n' * <p1> '\n' <p2>
    //
    // If it's to become the first (visible) patch, its insertion point is at [0, 0]:
    // * <p0> '\n' <p1> '\n' <p2>
    //
    // If it's to become the final (visible) patch, its insertion point is at the buffer end:
    // <p0> '\n' <p1> '\n' <p2> *
    //
    // Insert a newline *before* the expanding patch if we're inserting at the buffer's end, but the buffer is non-empty
    // (so it isn't also the end of the buffer). Insert a newline *after* the expanding patch when inserting anywhere
    // but the buffer's end.

    if (willHaveContent && atEnd && !atStart) {
      const beforeNewline = [];
      const afterNewline = after.slice();

      for (const marker of before) {
        if (marker.getRange().isEmpty()) {
          afterNewline.push(marker);
        } else {
          beforeNewline.push(marker);
        }
      }

      patchBuffer.createInserterAt(this.patch.getInsertionPoint()).keepBefore(beforeNewline).keepAfter(afterNewline).insert('\n').apply();
    }

    patchBuffer.createInserterAt(this.patch.getInsertionPoint()).keepBefore(before).keepAfter(after).insertPatchBuffer(subPatchBuffer, {
      callback: map => nextPatch.updateMarkers(map)
    }).insert(!atEnd ? '\n' : '').apply();
    this.patch.destroyMarkers();
    this.patch = nextPatch;
    this.didChangeRenderStatus();
    return true;
  }

  didChangeRenderStatus() {
    return this.emitter.emit('change-render-status', this);
  }

  onDidChangeRenderStatus(callback) {
    return this.emitter.on('change-render-status', callback);
  }

  clone(opts = {}) {
    return new this.constructor(opts.oldFile !== undefined ? opts.oldFile : this.oldFile, opts.newFile !== undefined ? opts.newFile : this.newFile, opts.patch !== undefined ? opts.patch : this.patch);
  }

  getStartingMarkers() {
    return this.patch.getStartingMarkers();
  }

  getEndingMarkers() {
    return this.patch.getEndingMarkers();
  }

  buildStagePatchForLines(originalBuffer, nextPatchBuffer, selectedLineSet) {
    let newFile = this.getNewFile();

    if (this.getStatus() === 'deleted') {
      if (this.patch.getChangedLineCount() === selectedLineSet.size && Array.from(selectedLineSet, row => this.patch.containsRow(row)).every(Boolean)) {
        // Whole file deletion staged.
        newFile = _file.nullFile;
      } else {
        // Partial file deletion, which becomes a modification.
        newFile = this.getOldFile();
      }
    }

    const patch = this.patch.buildStagePatchForLines(originalBuffer, nextPatchBuffer, selectedLineSet);
    return this.clone({
      newFile,
      patch
    });
  }

  buildUnstagePatchForLines(originalBuffer, nextPatchBuffer, selectedLineSet) {
    const nonNullFile = this.getNewFile().isPresent() ? this.getNewFile() : this.getOldFile();
    let oldFile = this.getNewFile();
    let newFile = nonNullFile;

    if (this.getStatus() === 'added') {
      if (selectedLineSet.size === this.patch.getChangedLineCount() && Array.from(selectedLineSet, row => this.patch.containsRow(row)).every(Boolean)) {
        // Ensure that newFile is null if the patch is an addition because we're deleting the entire file from the
        // index. If a symlink was deleted and replaced by a non-symlink file, we don't want the symlink entry to muck
        // up the patch.
        oldFile = nonNullFile;
        newFile = _file.nullFile;
      }
    } else if (this.getStatus() === 'deleted') {
      if (selectedLineSet.size === this.patch.getChangedLineCount() && Array.from(selectedLineSet, row => this.patch.containsRow(row)).every(Boolean)) {
        oldFile = _file.nullFile;
        newFile = nonNullFile;
      }
    }

    const patch = this.patch.buildUnstagePatchForLines(originalBuffer, nextPatchBuffer, selectedLineSet);
    return this.clone({
      oldFile,
      newFile,
      patch
    });
  }

  toStringIn(buffer) {
    if (!this.isPresent()) {
      return '';
    }

    if (this.hasTypechange()) {
      const left = this.clone({
        newFile: _file.nullFile,
        patch: this.getOldSymlink() ? this.getPatch().clone({
          status: 'deleted'
        }) : this.getPatch()
      });
      const right = this.clone({
        oldFile: _file.nullFile,
        patch: this.getNewSymlink() ? this.getPatch().clone({
          status: 'added'
        }) : this.getPatch()
      });
      return left.toStringIn(buffer) + right.toStringIn(buffer);
    } else if (this.getStatus() === 'added' && this.getNewFile().isSymlink()) {
      const symlinkPath = this.getNewSymlink();
      return this.getHeaderString() + `@@ -0,0 +1 @@\n+${symlinkPath}\n\\ No newline at end of file\n`;
    } else if (this.getStatus() === 'deleted' && this.getOldFile().isSymlink()) {
      const symlinkPath = this.getOldSymlink();
      return this.getHeaderString() + `@@ -1 +0,0 @@\n-${symlinkPath}\n\\ No newline at end of file\n`;
    } else {
      return this.getHeaderString() + this.getPatch().toStringIn(buffer);
    }
  }
  /*
   * Construct a String containing diagnostic information about the internal state of this FilePatch.
   */

  /* istanbul ignore next */


  inspect(opts = {}) {
    const options = _objectSpread({
      indent: 0
    }, opts);

    let indentation = '';

    for (let i = 0; i < options.indent; i++) {
      indentation += ' ';
    }

    let inspectString = `${indentation}(FilePatch `;

    if (this.getOldPath() !== this.getNewPath()) {
      inspectString += `oldPath=${this.getOldPath()} newPath=${this.getNewPath()}`;
    } else {
      inspectString += `path=${this.getPath()}`;
    }

    inspectString += '\n';
    inspectString += this.patch.inspect({
      indent: options.indent + 2
    });
    inspectString += `${indentation})\n`;
    return inspectString;
  }

  getHeaderString() {
    const fromPath = this.getOldPath() || this.getNewPath();
    const toPath = this.getNewPath() || this.getOldPath();
    let header = `diff --git a/${(0, _helpers.toGitPathSep)(fromPath)} b/${(0, _helpers.toGitPathSep)(toPath)}`;
    header += '\n';

    if (this.getStatus() === 'added') {
      header += `new file mode ${this.getNewMode()}`;
      header += '\n';
    } else if (this.getStatus() === 'deleted') {
      header += `deleted file mode ${this.getOldMode()}`;
      header += '\n';
    }

    header += this.getOldPath() ? `--- a/${(0, _helpers.toGitPathSep)(this.getOldPath())}` : '--- /dev/null';
    header += '\n';
    header += this.getNewPath() ? `+++ b/${(0, _helpers.toGitPathSep)(this.getNewPath())}` : '+++ /dev/null';
    header += '\n';
    return header;
  }

}

exports.default = FilePatch;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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